/** =================================================
 **                License Statement
 ** =================================================
 *
 ** Copyright (c) 2014-2022 David Winterburn <info@fwsentry.org>.
 *
 ** This file contains Original Code of the author, and is subject to the license
 ** terms below. In brief, using current terminology, this is a permissive license
 ** modelled on the 3-clause BSD License.
 *
 ** Permission to use, copy, modify, and distribute this software for any purpose
 ** is hereby granted as long as such purposes do not violate the terms of this
 ** license, and provided that the above copyright notice and this permission
 ** notice and license appear unmodified in ALL copies, distributed or stored.
 ** All intellectual property and patent rights remain with the author.
 *
 ** THE ORIGINAL CODE AND ALL SOFTWARE DISTRIBUTED UNDER THIS LICENSE IS PROVIDED
 ** ON AN "AS IS" BASIS WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESS OR IMPLIED,
 ** AND THE AUTHOR DISCLAIMS ALL SUCH WARRANTIES, OR LEGAL CLAIMS, WITH REGARD TO
 ** THIS SOFTWARE INCLUDING WITHOUT LIMITATION ANY WARRANTIES, IMPLIED OR OTHER,
 ** OF MERCHANTABILITY, PERFORMANCE AND FITNESS FOR A PARTICULAR PURPOSE, QUIET
 ** ENJOYMENT OR NON-INFRINGEMENT. UNDER NO CIRCUMSTANCES SHALL THE AUTHOR BE IN
 ** ANY WAY LIABLE FOR ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES, OR
 ** ANY DAMAGES WHATSOEVER, RESULTING FROM THE USE OF THIS SOFTWARE. SUCH DAMAGES
 ** MAY INCLUDE, BUT ARE NOT LIMITED TO, LOSS OF USE, EQUIPMENT, DATA, BILLABLE
 ** TIME, PERSONAL OR PHYSICAL INJURY, AND DIRECT OR INDIRECT PROFITS, WHETHER IN
 ** AN ACTION OF CONTRACT, NEGLIGENCE OR ANY OTHER TORTIOUS ACTION, ARISING OUT
 ** OF, OR IN CONNECTION WITH, THE USE OF, THE FITNESS OR PERFORMANCE OF THIS
 ** SOFTWARE, UNDER ANY CIRCUMSTANCES. THIS SOFTWARE WAS NOT DESIGNED TO BE USED
 ** IN ANY SITUATION WHERE HUMAN LIFE IS AT RISK, AND NO SUCH USE IS PROVIDED FOR
 ** OR PERMITTED.
 *
 ** BY USING THIS SOFTWARE YOU AGREE THAT IT IS THE SOLE RESPONSIBILITY OF THE END
 ** USER TO DECIDE IF THIS SOFTWARE OR PACKAGE PERFORMS, OR IS FIT FOR, ANY
 ** INTENDED USAGE.
 *
 ** This License and the rights granted hereunder will terminate; automatically
 ** without notice from the Author if You fail to comply with any term(s) of this
 ** License and fail to cure such breach within 30 days of becoming aware of such
 ** breach; or automatically without notice from the Author if You, at any time
 ** during the term of this License, commence any legal action against the Author.
 ** Upon termination of the license you are required to remove all copies of this
 ** software from all computer systems and data storage archives under your
 ** jurisdiction or control.
 *
 ** This license is automatically extended, if in the future, the Author exercises
 ** any intellectual property, or patent, rights connected with any of the Original
 ** Code included in this package. ALL end users using this package within the terms
 ** of this license, will automatically licensed to use those exercised or covered
 ** claims free of charge as they apply to this software package.
 **/





/* 		D-list.
 *
 * See http://info.fwsentry.org for more information on this package, or see
 * http://fwsentry.org/dlist/bugs/submission.html to report bugs or request new
 * features or improvements, or contact dlist-support@fwsentry.org. Bugs will be
 * addressed, new feature requests and program improvements will be considered,
 * at the sole discretion of the Author. Updates and new releases may be issued
 * in the future, if the Author considers it practical, enjoyable and desirable.
 */

/** 	Statement of Support.
 *
 ** Bug reports must be accompanied with a reasonable description and reasonable
 ** documentation of the problem, and any code changes you may have made to the
 ** affected module. If the problem turns out to be an issue with code not supplied
 ** by the author, the author may, or may not, choose to fix it.
 *
 ** If implemented, such requests or bug fixes may be added to a later release,
 ** package update, or support patch, at the sole discretion of the author, within
 ** time-frame's suitable to the author. All support that may optionally be
 ** provided by the author, including but not limited to, bug reports, new feature
 ** requests, or general usage support; will be provided on an "as-is" basis, at
 ** the sole discretion of the author. Any and all support, or resulting code,
 ** documentation, or product changes, will be provided with the same license and
 ** legal disclaimers described in the original package (and above) and with
 ** absolutely no warranties, implied, stated or otherwise.
 **/

/** Special note about this module, dlist.c.

	This module is covered under a more general open source license than
	FWSentry. It is entirely new code developed by the author.
		
	Originally i developed FWSentry with a different pre-made list subsystem,
	however i discovered a couple of minor but annoying bugs in the original
	code, some sloppy handling of leaf node storage with a minor but (for me at
	least) unacceptable memory leak, and index handling of large lists (30K+ in
	size) was far too slow. So i decided to roll my own list system with a new
	set of interfaces and the ability to manipulate the lists with element
	references in addition to indexes. I also implemented reverse searches,
	pivot and finite lists, and enough flexibility in the overall design to
	allow far more creative use of the list subsystem with complex code.
	Additionally dlist includes features to assist embedded systems developers,
	more secure handling of discarded elements and list structure, and major
	fine tuning and speed improvements. Lastly i provided an extensive set of
	debugging tools that can be activated or disabled by defining DEBUG which
	can greatly assist developers writing and debugging code that uses dlist.
	
	After developing this list subsystem for FWSentry i decided to make it
	available for general use, as it seems like a useful package for many.
	
																			dbw

**/

/*	D-List version 1.2 -- Copyright (c) 2014-2022 David Winterburn <info@fwsentry.org>	*/

#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <stdio.h>
#include <time.h>
#include <limits.h>

/* due to the requirement that this code must be highly portable, i am not
	including any reference to a config.h file or similar, debugging symbols
	such as DEBUG must be defined manually for dlist.c.
	
	DEBUG is used to control the huge amount of debugging and reporting code.
	Personally is use autoconf to define or un-define DEBUG, using config.h as
	the control mechanism. dlist.c is meant to be an independent item, and you
	set DEBUG levels using the automake variables -DDEBUG and -DDLIST_DEBUG
	instead of (or in addition to) using config.h, so although autoconf may
	define DEBUG for your project, it never defines NDEBUG for dlist.c. The
	autoconf script can define it via a AM_CONDITIONAL or similar mechanism.
	I pass the variable DEBUG to the dlist make file and use this code to
	include it as needed (an example from FWSentry).
	
		if DEBUG
		AM_CFLAGS+= -g
		# the FWS package uses config.h, but dlist does not, must define DEBUG here instead
		AM_CFLAGS+= -DDEBUG
		endif
	
	I use a different make file for dlist, than the other project make files. */

/* Enable internal only debugging and audit code, along with the asserts.
	
	This feature enables code that is only used to test and debug internal
	changes to dlist. It has no practical value to others. Running code with
	this enabled will slow the code down severely, and if there are large
	lists involved, the slow downs will be abundantly obvious.

	It should be very obvious, but
	
	please do not attempt code performance profiling with this enabled...			*/

#ifdef DLIST_DEBUG
#	include <assert.h>
#endif		/* DLIST_DEBUG */

/* Uncomment this define to force dlist to take extra steps to clear and flush
	memory before it is returned to the operating system. In most cases this is
	not useful and will add extra overhead. However on some real memory systems
	or where the threat of leakage or covert data channels is a concern, this
	will help eliminate those problems.												*/
/*
#define DLIST_FLUSH_MEMORY				1
*/	


/* uncomment to -> Disable internal leaf node allocation functions,
	and use malloc() for everything
	Why ? - well some users may prefer to stick with malloc instead of my code,
	some users may have serious memory constraints (embedded systems for example) and
	may wish to restrict memory usage to bare minimums, and some users may want
	to profile the two versions to see what works best for them.
	
	For my purposes and testing i have found that this code runs 50-75% faster
	than using malloc() on lists in the hundreds to thousands, and 50-70% faster
	on lists from tens of thousands to hundreds of millions. For example,
	clearing, copying into a list from another list with iteration and append
	then clearing again, a list of 3500 items, 1000 times takes 30 secs every
	time with malloc() with my code takes 15 secs. And that is still using
	malloc() to create the actual data items. If measuring only allocation
	this code is far faster, but of course it is designed to allocate specific
	items in a specific way, it is not a general purpose allocator.
	
	For profiling turn off DEBUG and DLIST_DEBUG first ... 							*/
/*
#define DLIST_USE_MALLOC				1
*/


/* For Embedded systems.

	Although the code footprint of D-List is very small, if you need to
	reduce the size, first define DLIST_USE_MALLOC, it will cause a
	significant drop in performance but reduce the memory overheads, and you
	can also just remove the various functions you do not use from the code
	base that you compile into that system. If you remove callable functions,
	you can also remove any unused internal functions as well. But remember
	that when D-List is compiled without DEBUG the code footprint drops by
	over 50%. In most cases this should be enough memory reduction to avoid
	function pruning.																*/


#include "dlist.h"


/* List Types */
#define LIST_TYPE_LINEAR				"Linear"
#define LIST_TYPE_FINITE				"Finite"
#define LIST_TYPE_PIVOT					"Pivot"



/* minimum size of a list scope to continue narrowing down the search scope of a sorted list  */
#define MIN_SORTED_SEARCH_SCOPE			20
/* skip percentage of a scope to try and find a new reduced scope point. 			*/
#define MIN_SORTED_SEARCH_SKIP			10


/* minimum number of elements for sorting with quicksort instead of insertion
	24 works best for list_fast_sort().	Not used for list_sort() - quickersort
	does not delegate sub-sorts, performance testing made it clear that it is
	less efficient for that algorithm.												*/
#define DLIST_QUICKSORT_MIN_SIZE		24

/* stuff for the quicker quicksorts */
#define DLIST_QUICKER_SORT_MIN_SIZE		0
typedef struct {
	leaf_node *node;
	void *element;
} quicksort_entry__t;

/* Global variables -- I do not like them, but this is required */

static boolean rand_seeded = FALSE;

/* These strings below, correspond to the defines in dlist.h
		LIST_OK							0
		LIST_ERR						-1
		ELEMENT_NOT_FOUND				-2
		ELEMENT_INVALID					-3
		ELEMENT_NO_DATUM				-4
		LIST_UNSORTED					-5											*/
static char *error_strings[] = {	"Invalid Error Code",
									"No Error",
									"Argument or List Error",
									"List Element Not Found",
									"Invalid List Element"
									"No Datum Provided",
									"List is Unsorted"
								};


/*		Internal Service Functions		*/
static inline leaf_node *list__locate_leaf_node(	const list_object *restrict list,
													const long int index);

static inline leaf_node *list__locate_extremum_leaf_node(	const list_object *restrict list,
															unsigned int *index,
															const boolean order);

static inline leaf_node *list__locate_extremum_leaf_node_pivot(	const list_object *restrict list,
																unsigned int *index,
																const boolean order,
																const int scope);

static inline leaf_node *list__locate_pivot_leaf_node(	const list_object *restrict list,
														long int *index);

static inline int list__insert_leaf_node(	list_object *restrict list,
											leaf_node *restrict reference,
											const void *element,
											const unsigned int index,
											const boolean adjust_mid,
											const int scope);

static inline void list__remove_leaf_node(	leaf_node *restrict old_leaf_node,
											const unsigned int index);

static inline void list__adjust_middle_pointer(	list_object *restrict list,
												const unsigned int index,
												const boolean action);

static inline int list__add_element_extra_list(	list_object *restrict dest_list,
												const leaf_node *ptr);

static int list__leaf_node_evaluate_range(	list_object *restrict list,
											const void *datum_min,
											const boolean order,
											const void *datum_max,
											const boolean compare_elements,
											const boolean move_leaf_node,
											list_object *restrict dest_list);

static inline void link__leaf_node(	leaf_node *restrict new_leaf_node,
									leaf_node *restrict reference);

static inline leaf_node *unlink__leaf_node(leaf_node *restrict old_leaf_node);

#ifdef DEBUG
static inline boolean list__validate_list_object(	const list_object *restrict list,
													const boolean list_write,
													const char *calling_func);
#endif	/* DEBUG */

static inline int list__initialize_list_object(	list_object *restrict new_list,
												const boolean copy_data,
												const element_size size_function,
												const unsigned int leaf_block_size,
												const char *calling_func);

static inline int list__validate_reference(	const list_object *restrict list,
											const element_reference *restrict leaf_node,
											const char* calling_function);

int list__validate_evaluate_range(	list_object *restrict list,
									const void *datum_min,
									const void *datum_max,
									const boolean compare_elements,
									const boolean range,
									list_object *restrict dest_list,
									const char *calling_func);

/* Functions to manage the leaf nodes and leaf node block chains */
static inline leaf_node *get__next_free_leaf_node(list_object *restrict list);

static inline void return__free_leaf_node(leaf_node *restrict old_leaf_node);

static unsigned int lookup__leaf_node_block_size(	const unsigned int hint,
													const char *calling_func);

#ifndef DLIST_USE_MALLOC
static inline leaf_node_array *allocate__leaf_node_block(	list_object *restrict list,
															const int block_count);

static inline int release__leaf_node_blocks(list_object *restrict list);

static inline void list__flush_freep_list(	list_object *restrict list,
											const leaf_node_array *restrict leaf_node_block);

static inline int list__trim_freep_list(	list_object *restrict list,
											const leaf_node_array *restrict leaf_node_block);

static inline unsigned int list__locate_open_leaf_node_count(list_object *restrict list);
#endif	/* ! DLIST_USE_MALLOC */

static inline void list__sort_insertion_sort(	const list_object *restrict list,
												const int direction,
												const leaf_node *elem_start,
												const leaf_node *elem_stop,
												const boolean fast_sort);

static inline void list__sort_quick_sort(	list_object *restrict list,
											const int direction,
											leaf_node *elem_start,
											leaf_node *elem_stop,
											const boolean fast_sort);

static inline void list__sort_sub_quick_sort(	list_object *restrict list,
												const int direction,
												leaf_node *elem_start,
												leaf_node *elem_stop,
												const boolean fast_sort);

static inline leaf_node *list__sort_partition(	list_object *restrict list,
												const int direction,
												leaf_node **elem_start,
												leaf_node **elem_stop,
												const boolean fast_sort,
												unsigned int *left_hand,
												unsigned int *right_hand);

static inline void list__sort_quicker_sort(	list_object *restrict list,
											const int direction,
											leaf_node *elem_start,
											leaf_node *elem_stop,
											const boolean fast_sort);

static inline void list__sort_sub_quicker_sort(	list_object *restrict list,
												quicksort_entry__t *restrict sort_table,
												const int direction,
												const unsigned int start,
												const unsigned int stop,
												const boolean fast_sort);

static inline unsigned int list__quicker_partition(	list_object *restrict list,
													quicksort_entry__t *restrict sort_table,
													const int direction,
													const unsigned int start,
													const unsigned int stop,
													const boolean fast_sort);
/*
static inline void list__sort_selection_sort(	const list_object *restrict list,
												const int direction,
												const leaf_node *elem_start,
												const leaf_node *elem_stop,
												const boolean fast_sort);

static inline void list__sort_selection_index_sort(	const list_object *restrict list,
													quicksort_entry__t *restrict sort_table,
													const int direction,
													const unsigned int start,
													const unsigned int stop,
													const boolean fast_sort);

static inline void list__sort_insertion_index_sort(	const list_object *restrict list,
													quicksort_entry__t *restrict sort_table,
													const int direction,
													const unsigned int start,
													const unsigned int stop,
													const boolean fast_sort);
*/

static leaf_node *list__find_sorted_element(	const list_object *restrict list,
												const void *datum,
												unsigned int *restrict index,
												const leaf_node *restrict start,
												const leaf_node *restrict stop,
												const unsigned int current_index,
												const unsigned int scope);

static leaf_node *list__find_sorted_insert(	const list_object *restrict list,
											const void *datum,
											unsigned int *restrict index,
											const leaf_node *restrict start,
											const leaf_node *restrict stop,
											const unsigned int current_index,
											const unsigned int scope);

static inline int list__shift_leaf_node(	list_object *restrict list,
											unsigned int index,
											const boolean where);

static inline void list__flip_leaf_nodes(	list_object *restrict list,
											leaf_node *start_leaf,
											leaf_node *stop_leaf,
											const boolean fix_middle);

void list__full_optimize_index(	list_object *restrict list, 
								const boolean sorted);


static inline void initialize_random_seed(void);

static inline unsigned int random_number_range(	const unsigned int size_a,
												const unsigned int size_b);

static inline boolean coinFlip(void);

#ifdef DLIST_DEBUG

#ifndef DLIST_USE_MALLOC
static inline int list__audit_freep_list(	const list_object *restrict list,
											const int block_number);
#endif	/* ! DLIST_USE_MALLOC */

static int list__audit(const list_object *list);
#endif		/* DLIST_DEBUG */



/*					List Function groups.
					=====================

	List Object Management				Functions calls that update the list object itself. The list object
										is the user provided memory object that acts as the definitive point
										to identify and work on a list. Each list has its own list object.

	List Function Management			Function calls that set and retrieve pointers to special user
										provided that are used during specific list operations, such as a
										search function to process list_search() types calls. The list
										subsystem has no knowledge of the contents of the users elements,
										only how to organize and retrieve them.

	List Information Functions			Function calls to retrieve specific size information about a list.

	List Transformation Functions		Function calls to provide transformation operations on complete
										lists of elements, such as copying or splitting lists.

	Element Placement Functions			Function calls to permit the insertion or other placement of new
										elements in a list. Will render a sorted list unsorted.

	Element Relocation Functions		Function calls to relocate elements between different lists.
	
	Element Removal Functions			Function calls to remove elements from a list.

	Element Retrieval Functions			Function calls to locate and retrieve element data, from
										presumably unsorted lists.

	Element Search Functions			Function calls to perform searches for elements, from presumably
										unsorted lists.

	Sorted List Element Functions		Function calls to perform various operations including searches
										for elements, from previously sorted lists. The intent of this 
										group of functions is to be able to update and modify the lists 
										while maintaining the original sorted element order. It is also 
										assumed that the same compare function that was used to sort the 
										list is used with these find functions.

	Element Transformation Functions	Function calls to transform the relative order of elements in a
										list. Some functions will render a sorted list unsorted, others
										will sort or re-sort a list.

	Miscellaneous List Functions		Functions to perform technical operations on lists.						*/



/*					List functions, grouped by type.
					================================

	List Object Management
	----------------------
	list_create()						Initialize a list object as linear class list, prior to using a list.
	list_create_finite()				Initialize a list object as finite class list, prior to using a list.
	list_create_pivot()					Initialize a list object as pivot class list, prior to using a list.
	list_set_finite_limit()				Change the max size of a finite class list.		
	list_erase()						Erase all elements of a list and the list object itself.

	List Function Management
	------------------------
	list_set_compare_function()			Set the compare function for a list, used for element sorts and evaluations.
	list_get_compare_function()			Retrieve the compare function for a list.
	list_set_pivot_function()			Set the pivot function for a pivot class list, used by list_store().
	list_set_search_function()			Set the search function for a list, used for element searches.
	list_get_search_function()			Retrieve the search function for a list.
	list_set_size_function()			Set the size function for a list, used for lists that manage element storage.
	list_set_insert_function()			Set the insert function for a list, used upon placement to process an element.
	list_set_remove_function()			Set the remove function for a list, used upon removal to process an element.
	list_set_iterate_function()			Set the iteration function for a list, used by list_callback_iteration().

	List Information Functions
	--------------------------
	list_is_empty()						Check if a list is empty, or not.
	list_size()							Retrieve the number of elements in any list.
	list_size_pivot()					Retrieve the number of elements, by section, in a pivot class list.
	list_is_sorted()					Check if list is currently sorted, or not.

	List Transformation Functions
	-----------------------------
	list_concatenate()					Concatenate 2 lists together, serially, into the destination list.
	list_merge()						See function list_assay()
	list_clone()						Create a clone of a list.
	list_transfer()						Transfer the elements of a list, into another list.
	list_weave()						Merge 2 lists together, element by element, into a destination list
	list_assay()						Analyze & merge 2 lists locating unique elements and those elements in common.
	list_split_pivot()					Split a pivot class list, and create 2 destination lists from its contents.
	list_compare()						Perform a comparison of two lists, indicating if they are the same.
	list_differences()					Perform a comparison of two lists, returning the number of differences.	

	Element Placement Functions
	---------------------------
	list_append()						Append an element to the end of a list.
	list_prepend()						Prepend an element to the front of a list.
	list_insert_index()					Add an element to a list, in the position indicated by an index number.
	list_insert_ref()					Add an element to a list, in the location indicated by an element reference.
	list_store()						Add an element to a pivot class list, as indicated by the pivot function.
	list_push()							See function list_prepend()

	Element Relocation Functions
	----------------------------
	list_copy_index()					Copy an element to another list, from the position indicated by an index number.
	list_move_index()					Move an element to another list, from the position indicated by an index number.
	list_copy_ref()						Copy an element to another list, from the location indicated by an element reference.
	list_move_ref()						Move an element to another list, from the location indicated by an element reference.
	list_search_copy()					Search for, and then copy, all instances of an element to another list.
	list_search_move()					Search for, and then move, all instances of an element to another list.
	list_evaluate_copy()				Compare all elements in a list, and copy elements that meet criteria to another list.
	list_evaluate_move()				Compare all elements in a list, and move elements that meet criteria to another list.
	list_range_copy()					Compare all elements in a list, and copy all that are with a range, to another list.
	list_range_move()					Compare all elements in a list, and move all that are with a range, to another list.

	Element Removal Functions
	-------------------------
	list_clear()						Remove all elements from a list, leave it empty.
	list_fetch()						Extract an element from the front (head), or the tail (end), of a list.
	list_fetch_pivot()					Extract an element from the pivot point, of a pivot class list.
	list_extract_index()				Extract an element, from the location indicated by an index number.
	list_extract_ref()					Extract an element, from the location indicated by an element reference.
	list_delete_index()					Remove an element, from the position indicated by an index number.
	list_delete_group()					Remove a block of elements, between the indicated by index numbers.
	list_delete_ref()					Remove an element, from the location indicated by an element reference.
	list_delete_duplicates()			Remove all elements that are duplicates of other elements in the list.
	list_search_extract()				Search for, and then extract, an element in a list.
	list_search_delete()				Search for, and then delete, an element in a list.
	list_search_expunge()				Search for, and then delete, all instances of an element from a list.
	list_pop()							See function list_fetch()

	Element Retrieval Functions
	---------------------------
	list_get_index()					Retrieve an element, from the position indicated by an index number.
	list_get_index_ref()				Retrieve an element and its reference, using position indicated by an index number.
	list_get_ref()						Retrieve an element, from the location indicated by an element reference.
	list_get_next_ref()					Retrieve the next element in a list, as indicated by an element reference.
	list_locate_index()					Locate an element index in a list, identified by its element datum pointer.
	list_locate_pivot()					Locate an element pivot value in a list, identified by its element datum pointer
	list_locate_ref()					Locate an element reference in a list, identified by its element datum pointer.
	list_valid_ref()					Verify if an element reference is valid for the list.
	list_iteration_start()				Start a sequence of iteration of all the elements, from one end of a list.
	list_iteration_start_pivot()		Start a sequence of iteration of all the elements, from the pivot point of a list.	
	list_iteration_has_more()			Inquire if there are more elements, during an iteration sequence of a list.
	list_iteration_has_more_pivot()		Inquire if there are more elements on this side of the pivot point.	
	list_iteration_get_next()			Retrieve the next element in an iteration sequence of a list.
	list_iteration_stop()				Stop an iteration sequence of a list.

	Element Search Functions
	------------------------
	list_simple_search()				Search for an element in a list, using element datum as the search criteria.
	list_occurrences()					Search for all occurrences of an element, using datum as the search criteria.
	list_search()						Search for an element in a list, using element datum as the search criteria.
	list_search_pivot()					Search a pivot class list, using element datum as both pivot value and search criteria.
	list_search_ref()					Search for an element, backwards or forwards, from a given reference point.
	list_search_shift()					Search for an element, and if found, shift the element to the head of the list				
	list_locate_maximum()				Search for an element in a list, using maximum quantity as the search criteria.
	list_locate_minimum()				Search for an element in a list, using minimum quantity as the search criteria.

	Sorted List Element Functions
	-----------------------------
	list_sorted_find()					Search for an element in a sorted list, using datum as the search criteria.
	list_sorted_insert()				Add an element to a sorted list, in the position indicated by sort order.
	list_sorted_extract()				Extract an element from a sorted list, by the position indicated by sort order.
	list_sorted_delete()				Delete an element from a sorted list, by the position indicated by sort order.
	list_sorted_expunge()				Search for, and then delete, all instances of an element from a sorted list.

	Element Transformation Functions
	--------------------------------
	list_sort()							Sort all the elements in a list, using either ascending or descending order.
	list_safe_sort()					Sort all the elements in a list, by order, in a guaranteed safe order.
	list_fast_sort()					Sort all the elements in a list, by order, voiding element references.
	list_tiny_sort()					Sort all the elements in a list, by order, slower but with no extra resources.
	list_shift_index()					Shift an element, identified by index number, to the end of a list.
	list_shift_ref()					Shift an element, identified by an element reference, to the end of a list.
	list_rotate()						Rotate the elements in a list, either to the left or right.
	list_flip()							Reverse all the elements in a list.
	list_shuffle()						Shuffle the elements in a list, by randomly, and repeatedly relocating them.
	list_process()						Process all elements in list using a user function to operate on element data.
	list_process_range()				Process a range of elements in list using a user function to operate on element data.
	list_callback_iteration()			Start a sequence of callback iteration, to process the elements in a list.

	Miscellaneous List Functions
	----------------------------
	list_optimize_index()				Perform an optimization of the element index in a list.
	list_optimize_memory()				Perform an optimization of the element memory storage in a list.
	list_do_not_seed_random()			Prevent D-List from seeding the underlying random number functions.
	list_information()					Provide details about a list to stdout.
	list_error_string()					Return pointer to string corresponding to D-List error code.			*/
	
	

/* NOTES on usage --
	
	D-List is suitable for multi-threaded environments. However the user must use
	appropriate measures to prevent race conditions or corrupting lists during
	updates. Controls such as mutex locks, or barriers, etc must be employed. The
	author uses, and suggests, pthreads for this purpose.

	Getting Started
	---------------
	The D-List package includes a set of example programs demonstrating the use of
	many of the list function calls. Also included in the D-List package is an
	extensive set of regression tests and interface error tests. All 3 of these
	sources can provide you with examples of suggesting multiple ways of using
	D-List to build, manipulate and extract elements with the available list
	classes. If you want to add element modeling, statistics or transformation
	functions to D-List, you might want to pay particular attention to the
	list_process() functions and the user function process_element. There were
	developed specifically to provide a simple way to extend D-List without
	having to put data type dependent code into D-List itself.

	If you decide to make changes to the internal D-List subsystem itself, make
	sure it passes both sets of regression tests, prior to using it for any
	projects, or distributing it to others.
	
	Sorted versus unsorted lists
	----------------------------
	All list functions except the Sorted List Element Functions, which are a group
	functions specifically designed for sorted lists, assume the lists to be
	operated on are not ordered or sorted. However all of the list_sorted_xx()
	function calls assume the lists are sorted, and are intended to stay sorted.

	Depending on the type of operations you wish to perform on a list, keeping a
	list sorted may dramatically speed up performance. If a lists element set is
	highly unstable, with very frequent insertions and removals, keeping a list
	sorted will provide very little performance improvement, and may actually be
	slightly slower. However for generally stable lists, that were built and then
	often searched, with somme insertions and deletions, keeping the list sorted
	and using the Sorted List Element Functions can provide huge performance
	gains. For example using 2 of the performance testing modules, search and
	sorted_search, you will see on average the unsorted very large lists take
	about 3 times as long to search for non-existent elements and much longer
	by comparison when the element exists. So for projects that a search heavy,
	using sorted lists is a good performance booster. In fact for a list of a
	million elements, the processor savings for even a couple of searches of the
	list is far higher than the time taken to sort the list.

	It is important that the current user compare function set for the lists being
	operated on by the various Sorted List Element Functions, keep the same
	ordering semantics as the user compare function used to originally sort the
	list. Generally you will want to use the same user compare function throughout,
	however the compare function can be replaced but the ordering semantics should
	remain the same, or the element ordering results will be undefined when calling
	any of the Sorted List Element Functions. Additionally, if any of the Sorted
	List Element Functions are called for unsorted lists, it will result in a error
	return code.
	
	All functions are available for unsorted lists, with the restrictions noted in
	the function descriptions. For sorted lists all functions are also available,
	however usage of any of the following functions will render the list unsorted,
	which will prevent any of the Sorted List Element Functions from working on
	them, until the list is sorted again.

		All Element Placement Functions -
			list_append()
			list_prepend()
			list_insert_index()
			list_insert_ref()
		And these Element Transformation Functions
			list_search_shift()
			list_shift_index()
			list_shift_ref()
			list_rotate()
			list_shuffle()
	
	All other functions will operate on sorted lists and maintain the current
	sorted order, this includes pivot class lists.
	
	Any list processed by one of the 4 Element Sort Functions will be marked as
	sorted, and that status is only lost if operated on by a function that would
	destroy the sorted order.

	
	Iteration Sequences
	-------------------
	Starting a list iteration will lock the list from index based updates, but
	will allow parallel searches and scans to be performed. Only 1 iteration at
	a time per list is allowed however. The user can access multiple elements
	using either reference, index or searches, while in an iteration cycle. When
	an iteration is stopped the list returns to normal.
	
	Element References
	------------------
	Accessing, inserting, relocating, or deleting elements using the available
	element references is similar to file operations using SEEK. You can keep and
	reuse the elements references, but you cannot make any assumptions about how
	they relate to each other in any way. Never attempt to modify references, and
	never use them as a reference after an element has been removed from a list.
	Any attempts will return an error in debug mode, or a code failure if running
	in non-debug mode. If a function does not allow NULL elements references,
	then always check an element reference is not NULL before providing it to a
	function. You should never assume that element references are pointers, and
	never use the structure information in dlist_internals.h to use or misuse
	them. All element references are implementation dependent and can change
	between releases. It is only guaranteed that if an element exists, then the
	previously provided element reference can be supplied back to any suitable
	D-List function to access that same element again. Element references survive
	sorting, shuffles, reordering, and being transfered to another list. Once an
	element has been removed from a list, its element reference is rendered void.
	
	List iteration sequences are based on element references, which under most
	circumstances, makes iteration the most efficient way to move through a list.
	
	Indexes
	-------
	Unlike accessing elements by reference, using index access can guarantee that
	an element with a lower index number is prior in the list than one with a
	higher number. Element 4 is 2 back from element 6, etc. It is important to
	remember that an element index always starts with element #0. So that the last
	element at the tail of a list, is always found with list_size(the_list)-1.
	Also it is important to note that indexes do not survive transformations of
	lists or elements, and that they are only a relative index of the elements at
	that moment in time.

	Accessing by index number is slower than accessing by reference. In small
	lists (in the 100's) it is not a major issue, the mean search required to find
	an element by index is n/8, and the worst case is n/6 plus the calculation
	overhead of obtaining a starting point. However for large lists, >100,000 or
	very large lists >1,000,000 this becomes a time consuming overhead. Accessing
	elements by reference, or more specifically the next or prior elements by
	reference to the current one, is vastly superior in time and resources. The
	worst case time being n/list_size. So at 50,000 elements the mean search for
	index would require moving over 6250 leaf nodes to acquire the requested node.
	With element reference access it is only required to move across 1 leaf node
	in all cases.

	Process Functions
	-----------------
	As an alternative to iteration, D-List provides a facility to process either part,
	or all, of list using a user supplied function to act on each element, and
	the entire group of the elements. This feature was added to allow users to easily
	add statistics and data transformation features to the list subsystem, without
	adding data type specific knowledge directly to D-List. For example, to add a
	group of functions to a user program to calculate the mean, or mode of a data set,
	can easily be achieved with a few lines of code using this feature. Unlike using
	callback iteration, the function does not have access to the list, and therefore
	cannot remove or relocate elements. Also the function has no option to cut the
	element processing short. It does however have the ability to share data directly
	with the original caller in a MP safe way.

	
	D-List implements 3 classes of lists.
	-------------------------------------
	
	Standard linear lists -
	unconstrained linked lists running from head to tail
	
	Finite lists -
	linear lists with a maximum capacity of elements and automatic removal of
	excess elements from the tail
	
	Pivot lists -
	semi-organized lists with the elements split using a user defined datum
	separating the elements by a conceptual pivot point. Please refer to the
	documentation for functions list_create_pivot() and list_store() for more
	information on the usage and layout of pivot lists.
	
	All three classes of lists are implemented using a double-linked layout,
	however a pivot list uses a user defined pivot function to decide which side
	of a conceptual pivot point, to either place or look for, an element. Some
	functions are not available to operate on pivot lists, and some not for
	linear or finite lists. Many functions operate slightly differently depending
	on which class of list they are operating on. See the function documentation
	for more details. Pivot lists are mainly useful for code that builds and uses
	huge lists, or wants an easy way to organize and reorganize various list
	elements. The output of list_assay() gives a good example of the uses of
	pivot class lists.

*/



/*				Table of D-List functions by availability for class type

																		if LIST_COPY_DATA
									  Linear	  Finite	   Pivot	frees object used
									  											
	list_create()						yes	
	list_create_finite()							yes	
	list_create_pivot()											yes	
	list_set_finite_limit()							yes				
	list_erase()						yes			yes			yes				yes

	list_set_compare_function()			yes			yes			yes	
	list_get_compare_function()			yes			yes			yes	
	list_set_pivot_function()									yes	
	list_set_search_function()			yes			yes			yes	
	list_get_search_function()			yes			yes			yes	
	list_set_size_function()			yes			yes			yes	
	list_set_insert_function()			yes			yes			yes	
	list_set_remove_function()			yes			yes			yes	
	list_set_iterate_function()			yes			yes			yes	

	list_is_empty()						yes			yes			yes	
	list_size()							yes			yes			yes	
	list_size_pivot()					yes			yes			yes	
	list_is_sorted()					yes			yes			

	list_concatenate()					yes			yes(4)		yes(1)			copy	
	list_clone()						yes			yes(4)		yes				copy
	list_transfer()						yes			yes			yes				copy
	list_weave()						yes			yes(4)		yes(1)			copy	
	list_assay()						yes			yes			yes(1)			copy
	list_split_pivot()											yes				copy
	list_compare()						yes			yes			yes	
	list_differences()					yes			yes			yes	

	list_append()						yes							
	list_prepend()						yes			yes				
	list_insert_index()					yes			yes				
	list_insert_ref()					yes							
	list_store()												yes	

	list_copy_index()					yes			yes(4)		yes				copy
	list_move_index()					yes			yes(4)		yes				yes
	list_copy_ref()						yes			yes(4)		yes				copy
	list_move_ref()						yes			yes(4)		yes				yes
	list_search_copy()					yes			yes(4)		yes				copy
	list_search_move()					yes			yes(4)		yes				yes
	list_evaluate_copy()				yes			yes(4)		yes				copy
	list_evaluate_move()				yes			yes(4)		yes				yes
	list_range_copy()					yes			yes(4)		yes				copy
	list_range_move()					yes			yes(4)		yes				yes

	list_clear()						yes			yes			yes				yes
	list_fetch()						yes			yes			yes				no
	list_fetch_pivot()											yes				no
	list_extract_index()				yes			yes			yes				no
	list_extract_ref()					yes			yes			yes				no
	list_delete_index()					yes			yes			yes				yes
	list_delete_group()					yes			yes			yes				yes
	list_delete_ref()					yes			yes(1)		yes				yes
	list_delete_duplicates()			yes			yes			yes				yes
	list_search_extract()				yes			yes			yes				no
	list_search_delete()				yes			yes			yes				yes
	list_search_expunge()				yes			yes			yes				yes

	list_get_index()					yes			yes			yes	
	list_get_index_ref()				yes			yes			yes	
	list_get_ref()						yes						yes	
	list_get_next_ref()					yes						yes	
	list_locate_index()					yes			yes			yes	
	list_locate_pivot()					yes(2)		yes(2)		yes	
	list_locate_ref()					yes			yes			yes	
	list_valid_ref()					yes			yes			yes	
	list_iteration_start()				yes			yes			yes	
	list_iteration_start_pivot()								yes	
	list_iteration_has_more()			yes			yes			yes	
	list_iteration_has_more_pivot()								yes	
	list_iteration_get_next()			yes			yes			yes	
	list_iteration_stop()				yes			yes			yes	

	list_simple_search()				yes			yes			yes	
	list_occurrences()					yes			yes			yes	
	list_search()						yes			yes			yes	
	list_search_pivot()											yes	
	list_search_ref()					yes			yes			yes	
	list_search_shift()					yes			yes				
	list_locate_maximum()				yes			yes			yes	
	list_locate_minimum()				yes			yes			yes	

	list_sorted_find()					yes			yes			yes	
	list_sorted_insert()				yes			yes			yes				yes
	list_sorted_extract()				yes			yes			yes				yes
	list_sorted_delete()				yes			yes			yes				yes
	list_sorted_expunge()				yes			yes			yes				yes

	list_sort()							yes			yes			yes	
	list_safe_sort()					yes			yes			yes	
	list_fast_sort()					yes			yes			yes	
	list_tiny_sort()					yes			yes			yes	
	list_shift_index()					yes			yes				
	list_shift_ref()					yes			yes				
	list_rotate()						yes			yes
	list_flip()							yes			yes			yes(1)	
	list_shuffle()						yes			yes			yes	
	list_process()						yes			yes			yes	
	list_process_range()				yes			yes			yes	
	list_callback_iteration()			yes			yes			yes	

	list_optimize_index()				yes			yes			yes	
	list_optimize_memory()				yes			yes			yes	
	list_do_not_seed_random()			n/a			n/a			n/a
	list_information()					yes			yes			yes	
	list_error_string()					n/a			n/a			n/a
	
	Notes:
	(1) restrictions apply, refer to function documentation
	(2) will always return ENTIRE_LIST
	(3) may not retain pivot alignment, depending of search criteria,
		refer to function documentation
	(4)	destination list cannot be finite class										*/



/*				Table of D-List calls that use sub functions

									  compare	search	size	pivot	insert	remove	iterate	process
									  											
	list_create()							
	list_create_finite()								
	list_create_pivot()												
	list_set_finite_limit()											
	list_erase()																yes

	list_set_compare_function()			yes							
	list_get_compare_function()			yes							
	list_set_pivot_function()									yes	
	list_set_search_function()					yes				
	list_get_search_function()					yes				
	list_set_size_function()							yes				
	list_set_insert_function()											yes	
	list_set_remove_function()													yes	
	list_set_iterate_function()															yes	

	list_is_empty()													
	list_size()														
	list_size_pivot()												
	list_is_sorted()						

	list_concatenate()									yes(1)	yes(2)	yes(1)		
	list_clone()										yes(1)	yes(2)	yes(1)		
	list_transfer()						
	list_weave()										yes(1)	yes(2)	yes(1)			
	list_assay()										yes(1)			yes(1)		
	list_split_pivot()									yes(1)	yes(2)	yes(1)		
	list_compare()								yes				
	list_differences()							yes							

	list_append()										yes				yes				
	list_prepend()										yes				yes				
	list_insert_index()									yes				yes				
	list_insert_ref()									yes				yes			
	list_store()										yes		yes		yes

	list_copy_index()									yes(1)	yes(2)	yes				
	list_move_index()									yes(1)	yes(2)	yes		yes		
	list_copy_ref()										yes(1)	yes(2)	yes		
	list_move_ref()										yes(1)	yes(2)	yes		yes
	list_search_copy()							yes		yes(1)	yes(2)	yes		
	list_search_move()							yes		yes(1)	yes(2)	yes		yes
	list_evaluate_copy()				yes				yes(1)	yes(2)	yes		
	list_evaluate_move()				yes				yes(1)	yes(2)	yes		yes
	list_range_copy()									yes(1)	yes(2)	yes		
	list_range_move()									yes(1)	yes(2)	yes		yes

	list_clear()																yes
	list_fetch()																yes
	list_fetch_pivot()															yes
	list_extract_index()														yes
	list_extract_ref()															yes
	list_delete_index()															yes
	list_delete_group()															yes
	list_delete_ref()															yes
	list_delete_duplicates()					yes								yes
	list_search_extract()						yes								yes
	list_search_delete()						yes								yes
	list_search_expunge()						yes								yes

	list_get_index()												
	list_get_index_ref()												
	list_get_ref()													
	list_get_next_ref()												
	list_locate_index()												
	list_locate_pivot()										
	list_locate_ref()												
	list_valid_ref()												
	list_iteration_start()											
	list_iteration_start_pivot()									
	list_iteration_has_more()										
	list_iteration_has_more_pivot()									
	list_iteration_get_next()										
	list_iteration_stop()											

	list_simple_search()						yes							
	list_occurrences()							yes							
	list_search()								yes							
	list_search_pivot()							yes				yes			
	list_search_ref()							yes							
	list_search_shift()							yes							
	list_locate_maximum()				yes							
	list_locate_minimum()				yes							

	list_sorted_find()					yes						yes	
	list_sorted_insert()				yes						yes		yes		
	list_sorted_extract()				yes						yes				yes	
	list_sorted_delete()				yes						yes				yes	
	list_sorted_expunge()				yes						yes				yes				

	list_sort()							yes							
	list_safe_sort()					yes							
	list_fast_sort()					yes							
	list_tiny_sort()					yes							
	list_shift_index()												
	list_shift_ref()												
	list_rotate()									
	list_flip()														
	list_shuffle()													
	list_process()																				yes	
	list_process_range()																		yes	
	list_callback_iteration()															yes	

	list_optimize_index()											
	list_optimize_memory()											
	list_do_not_seed_random()	
	list_information()												
	list_error_string()	
	
	Notes:
	(1) only for destination list
	(2)	if destination is a pivot class list										*/



/* Initialize a new list, set up ready to store the elements. All other operations
	on a list_object (a list) must occur after initialization, or they will fail.
	
	This function creates a normal linear class list. All operations are permitted
	on such lists, except for functions that are specific to a different class of
	lists, such as list_split_pivot() or list_store().

	new_list	pointer to caller supplied list_object
	copy_data	LIST_COPY_DATA (TRUE) list will allocate, manage element
				memory, and copy all data to the elements when they are
				created.
				LIST_DO_NOT_COPY (FALSE) user is responsible for managing
				element data, allocating and free(ing) the element resources.
	size_function
				If copy_data is TRUE (LIST_COPY_DATA), a pointer to a user
				function that returns size requests for list elements.
				If copy_data is FALSE (LIST_DO_NOT_COPY) then NULL must be
				passed.
	size_hint	An optional hint to D-List about the expected size of the
				new list, this sizing hint allows D-List to set up internal
				resources that are appropriate for the expected usage, and
				if set correctly allows for significantly better performance
				when searching, traversing and sorting very large lists, the
				differences can be 25-40% faster than just using the defaults.
				Alternatively for very small lists there would be a some
				considerable wasted resources if the hint was set too high.
				Unlike finite class lists, this hint does not set any maximum
				size or limits, it is just a hint to balance resources and
				performance. Unlike finite class lists which allow for the
				adjustment of the maximum element limit, you cannot change the
				size hint once the list is established. If you find the actual
				number of elements is wildly different from the hint, the best
				option is to create a new list with the appropriate hint and
				transfer the elements to it, using list_clone(). The function
				list_transfer() copies all the list data to another list, that
				includes all the internal resources, so it would perpetuate the
				original problem. The size_hint can be any of these:
					LIST_NO_SIZE_HINT	No size hint provided, use defaults
					LIST_SIZE_TINY		< 100 elements expected
					LIST_SIZE_SMALL		< 1,000 elements expected
					LIST_SIZE_MEDIUM	< 25,000 elements expected
					LIST_SIZE_LARGE		< 250,000 elements expected
					LIST_SIZE_HUGE		< 2,500,000 elements expected
					LIST_SIZE_MEGA		> 2,500,000 elements expected
	returns	LIST_OK			success
			LIST_ERR		on error 												*/
int list_create(	list_object *restrict new_list,
					const boolean copy_data,
					const element_size size_function,
					const unsigned int size_hint) {

	if (new_list == NULL) {
#ifdef DEBUG
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with NULL list pointer.\n\n", __func__);
#endif
		return LIST_ERR;
	}
	
	if (list__initialize_list_object(	new_list, 
										copy_data, 
										size_function, 
										lookup__leaf_node_block_size(size_hint, __func__), 
										__func__) != LIST_OK)
		return LIST_ERR;

#ifdef DLIST_DEBUG
	assert(list__audit(new_list) == LIST_OK);
#endif		/* DLIST_DEBUG */
	return LIST_OK;
}

/* Initialize a new list, set up ready to store the elements. All other operations
	on a list_object (a list) must occur after initialization, or they will fail.
	
	This function creates a finite list, wich is a restricted linear class list.
	Most operations are permitted on these lists, except for functions that are
	specific to a different class of lists or reference elements in a way which
	would cause problems, such as list_split_pivot() or list_store().
	
	A finite list is a linear list, with a defined maximum number of elements
	that it is permitted to hold. Elements can be added to it until it reaches its
	maximum capacity, then for each new element added that would exceed the stated
	capacity an element is removed from the tail end of the list. Such lists have
	some limited utility, such as keeping a cache of recently added elements from
	a much larger or tree structure list, which is efficient at finding items, but
	poor at locating recently added items, for example. Also keeping a running
	scroll-back of items, similar to a scroll-back of a terminal window. Any
	situation where items are wanted to be kept, but a defined set of resources is
	to be used, instead of an endless list. Obviously this finite class of list is
	not appropriate for elements that are important or must be kept. Additionally
	as elements can be removed without the user explicitly knowing, no functions
	using input references are allowed to operate on finite class lists.
	
	To maintain the integrity of the list, the following functions are not
	available to operate on finite lists.
		list_append()
		list_set_pivot_function()
		list_split_pivot()
		list_insert_ref()
		list_store()
		list_fetch_pivot()
		list_get_ref()
		list_get_next_ref()
		list_search_ref()
		list_iteration_start_pivot()
		list_iteration_has_more_pivot()

	The following functions are available to operate on finite lists, but with
	specific limitations as noted in the function descriptions.
		list_delete_ref()
		list_search()
	
	For obvious reasons, it is best to add new elements to a finite class list
	via the function list_prepend(), this also leads to more readable code where
	this class of list is used.

	new_list	pointer to caller supplied list_object
	copy_data	LIST_COPY_DATA (TRUE) list will allocate, manage element
				memory, and copy all data to the elements when they are
				created.
				LIST_DO_NOT_COPY (FALSE) user is responsible for managing
				element data, allocating and free(ing) the element resources.
	size_function
				If copy_data is TRUE (LIST_COPY_DATA), a pointer to a user
				function that returns size requests for list elements.
				If copy_data is FALSE (LIST_DO_NOT_COPY) then NULL must be
				passed.
	max			The maximum number of elements this list may hold. Unlike the
				other list classes, this is not a hint, it is an absolute
				maximum limit. This size limit can be adjusted by using the
				function list_set_finite_limit().
	returns	LIST_OK			success
			LIST_ERR		on error 												*/
int list_create_finite(	list_object *restrict new_list,
						const boolean copy_data,
						const element_size size_function,
						const unsigned int max) {

	int block_size = 0;
	int ret;
	
	if (new_list == NULL) {
#ifdef DEBUG
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with NULL list pointer.\n\n", __func__);
#endif
		return LIST_ERR;
	}

	if (max < 1) {
#ifdef DEBUG
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with 0 element size.\n\n", __func__);
#endif
		return LIST_ERR;
	}

#ifdef DEBUG
#define LIST_CREATE_FINITE__MAX_CHECK_POINT							100
	if (max > LIST_CREATE_FINITE__MAX_CHECK_POINT) {
		fprintf(stderr, "\n==> Dlist Debug WARNING ==> %s() - "\
				"called with max element count of (#%u) which is greater than %d, did you really "\
				"mean to do this ?.\n\n", __func__, max, LIST_CREATE_FINITE__MAX_CHECK_POINT);	
	}
#endif

#ifndef DLIST_USE_MALLOC
	/* plus 2 for the guard nodes, and
		1 for the insertion overflow at max capacity, so a new element
		insertion does not trigger a new leaf node block to be allocated.			*/
	block_size = max+3;
#endif	/* ! DLIST_USE_MALLOC */

	ret = list__initialize_list_object(new_list, copy_data, size_function, block_size, __func__);
	if (ret == LIST_OK) {
		new_list->max_element_count = max;
		new_list->list_finite_enabled = TRUE;
	}
	
#ifdef DLIST_DEBUG
	assert(list__audit(new_list) == LIST_OK);
#endif		/* DLIST_DEBUG */

	return ret;
}

/* Initialize a new pivot class list, set up ready to store the elements. All
	other operations on a list_object (a list) must occur after initialization,
	or they will fail. Pivot class lists are functionally similar to normal,
	linear, lists, however there are some major differences.
	
	The placement of elements in the list is decided by the response of queries
	to the pivot_function for each element presented. The purpose is to allow
	the list to organize the elements around a pivot point. For large unsorted
	lists this can provide a substantial speed gain for search operations.
	
	To maintain the pivot organization of elements, the following list functions
	are not permitted to operate on pivot class lists.
		list_append()
		list_prepend()
		list_insert_index()
		list_insert_ref()

	Instead the list_store() or list_sorted_insert() function must be used to 
	add any element to a pivot class list. All other functions work on pivot 
	class lists, and any particular issues are described in that functions 
	overview. For a more complete list of what functions are available for which 
	classes of list, please refer to the table at the top of this file.
	
	When using the D-list functions to manage a pivot list, please use the defines
	established above for consistency and readability of the code. A pivot list is
	organized as this diagram shows
	
		elem#1 .. elem#2 .. elem#3 .. elem#4 ... elem .. elem .. elem .. elem .. elem .. elem#n
		  ^									  ^												^
		  |									  |												|
	<HEAD_OF_LIST>	<-- PIVOT_MINOR -->	   <PIVOT>		<-- PIVOT_MAJOR -->			<TAIL_OF_LIST)
	
	A pivot class list must have a pivot function set for the list, prior to any
	update usage. The function can be replaced during the life of the list, but
	one must always be set. Please refer to list_set_pivot_function().

	new_list	pointer to caller supplied list_object
	copy_data	LIST_COPY_DATA (TRUE) list will allocate, manage element
				memory, and copy all data to the elements when they are
				created.
				LIST_DO_NOT_COPY (FALSE) user is responsible for managing
				element data, allocating and free(ing) the element resources.
	size_function
				If copy_data is TRUE (LIST_COPY_DATA), a pointer to a user
				function that returns size requests for list elements.
				If copy_data is FALSE (LIST_DO_NOT_COPY) then NULL must be
				passed.
	pivot_function
				required pointer to user function that will process pivot
				identification requests when new elements are added to the
				list subsystem via list_store().
	size_hint	An optional hint to D-List about the expected size of the
				new list, this sizing hint allows D-List to set up internal
				resources that are appropriate for the expected usage, and
				if set correctly allows for significantly better performance
				when searching, traversing and sorting very large lists, the
				differences can be 25-40% faster than just using the defaults.
				Alternatively for very small lists there would be a some
				considerable wasted resources if the hint was set too high.
				Unlike finite class lists, this hint does not set any maximum
				size or limits, it is just a hint to balance resources and
				performance. Unlike finite class lists which allow for the
				adjustment of the maximum element limit, you cannot change the
				size hint once the list is established. If you find the actual
				number of elements is wildly different from the hint, the best
				option is to create a new list with the appropriate hint and
				transfer the elements to it, using list_clone(). The function
				list_transfer() copies all the list data to another list, that
				includes all the internal resources, so it would perpetuate the
				original problem. The size_hint can be any of these:
					LIST_NO_SIZE_HINT	No size hint provided, use defaults
					LIST_SIZE_TINY		< 100 elements expected
					LIST_SIZE_SMALL		< 1,000 elements expected
					LIST_SIZE_MEDIUM	< 25,000 elements expected
					LIST_SIZE_LARGE		< 250,000 elements expected
					LIST_SIZE_HUGE		< 2,500,000 elements expected
					LIST_SIZE_MEGA		> 2,500,000 elements expected
	returns	LIST_OK			success
			LIST_ERR		on error 												*/
int list_create_pivot(	list_object *restrict new_list,
						const boolean copy_data,
						const element_size size_function,
						const element_pivot pivot_function,
						const unsigned int size_hint) {

	int ret;

	if (new_list == NULL) {
#ifdef DEBUG
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with NULL list pointer.\n\n", __func__);
#endif
		return LIST_ERR;
	}
	if (pivot_function == NULL) {
#ifdef DEBUG
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"required pivot function is NULL.\n\n", __func__);
#endif
		return LIST_ERR;
	}
	
	ret = list__initialize_list_object(	new_list, 
										copy_data, 
										size_function, 
										lookup__leaf_node_block_size(size_hint, __func__), 
										__func__);
	if (ret == LIST_OK) {
		new_list->head_guard->internal->pivot.value = PIVOT_MINOR;
		new_list->tail_guard->internal->pivot.value = PIVOT_MAJOR;
		new_list->list_pivot_enabled = TRUE;
		new_list->functions.pivot = pivot_function;
	}

#ifdef DLIST_DEBUG
	assert(list__audit(new_list) == LIST_OK);
#endif		/* DLIST_DEBUG */
	
	return ret;
}

/* Remove and erase an exiting list.
	
	Upon exit it cannot be used to store elements.

	If the list was initialized with LIST_COPY_DATA set TRUE, all
	elements in the list will be removed, deallocated and returned
	to the system for reuse.
	
	Upon exit the list_object is cleared, but it is not deallocated.
	That is the responsibility of the caller.

	list		list_object that has been previously initialized.
	returns	nothing 																*/
void list_erase(list_object *restrict list) {

#ifdef DEBUG
	if (list__validate_list_object(list, FALSE, __func__) == FALSE) return;
#else
	if (list == NULL) return;
#endif

	list_clear(list);
	
	return__free_leaf_node(list->head_guard);
	return__free_leaf_node(list->tail_guard);

#ifndef DLIST_USE_MALLOC
	list->element_count = 1;						/* force the remaining freep list to be cleared, be clean */
	release__leaf_node_blocks(list);
#endif	/* ! DLIST_USE_MALLOC */

	memset(list, 0x00, sizeof(list_object));
	
	return;
}

/* Traverse a list and remove all elements from the list.
	
	If the list was created with the LIST_COPY_DATA attribute set
	TRUE, this function will also release all the data elements from
	the list. Otherwise it is the users responsibility to manage
	those data objects.

	list		pointer to caller supplied list_object
	returns	The number of elements removed from the list, or
			LIST_ERR		on error.												*/
int list_clear(list_object *restrict list) {

	leaf_node *ptr;
	int i;
	
#ifdef DEBUG
	if (list__validate_list_object(list, FALSE, __func__) == FALSE)
		return LIST_ERR;
	if (list->iteration_enabled) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called on list with iteration enabled.\n\n", __func__);
		return LIST_ERR;
	}
#else
	if ((list == NULL) || (list->iteration_enabled))
		return LIST_ERR;

#endif

	if (list->element_count == 0) return LIST_OK;

	if (list->element_count > INT_MAX) i = INT_MAX;
	else i = (int)list->element_count;
	
	/* i have tried to speed this up, on my platform removing the redundant element counter
		decrement, actually increases cpu overhead by 10% ouch. Splitting the if condition
		outside slows it down by 5%, replacing the for() loop with a while or a do/while
		is even worse adds 15-20% overhead for 10,000,000 elements. Decided this is best overall */
	for (ptr = list->head_guard->next_leaf; ptr->next_leaf != NULL; --list->element_count) {
		if (list->functions.remove) list->functions.remove(ptr->element);
		ptr = ptr->next_leaf;
		return__free_leaf_node(ptr->prior_leaf);
	}

	/* Finish up, clean up the master node. */
	list->head_guard->next_leaf = list->tail_guard;
	list->tail_guard->prior_leaf = list->head_guard;
	list->mid_point = NULL;
	list->list_sorted = FALSE;
	list->pivot_point.minor = list->pivot_point.major = 0;

#ifndef DLIST_USE_MALLOC
	release__leaf_node_blocks(list);
#endif	/* ! DLIST_USE_MALLOC */
	
#ifdef DLIST_DEBUG
	if (list->element_count !=0) {
		fprintf(stderr, "\n==> Dlist INTERNAL ERROR ==> %s() failed - invalid element count "\
				"after clear operation. Should be 0, actual is '%u'.\n\n", __func__,
				(unsigned int)list->element_count);
		 return LIST_ERR;
	}
	assert(list__audit(list) == LIST_OK);
#endif		/* DLIST_DEBUG */
	return i;
}

/* Examine a list for elements, and return a boolean indicator if empty or not.

	list		list_object that has been previously initialized.
	returns	TRUE			the list has no user elements, it is empty.
			FALSE			there are user elements in the list, it is not empty. 	*/
boolean list_is_empty(const list_object *restrict list) {

#ifdef DEBUG
	if (list__validate_list_object(list, FALSE, __func__) == FALSE) return TRUE;
#else
	if (list == NULL) return TRUE;
#endif

	return (list->element_count == 0);
}

/* Examine a list and return the count of the number of elements in the list.

	list		list_object that has been previously initialized.
	returns >0			The number of elements stored in the list,
			0			if it is empty.												*/
unsigned int list_size(const list_object *restrict list) {

#ifdef DEBUG
	if (list__validate_list_object(list, FALSE, __func__) == FALSE) return 0;
#else
	if (list == NULL) return 0;
#endif

	return list->element_count;
}

/* Examine a pivot class list and return the count of the number of elements
	in the list, on both sides of the pivot.
	
	If called for another class of list, will return the current number
	of elements in the list, and generate a warning message in DEBUG mode.

	list		list_object that has been previously initialized.
	returns	minor_count		the number of elements on the left (minor) side
							of the pivot point.
			major_count		the number of elements on the right (major)
							side of the pivot point.
			>0 				The number of elements stored in the list,
			0				if it is empty.											*/
unsigned int list_size_pivot(	const list_object *restrict list,
								unsigned int *minor_count,
								unsigned int *major_count) {

#ifdef DEBUG
	if (list__validate_list_object(list, FALSE, __func__) == FALSE)
		return 0;
	if (minor_count == NULL) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with NULL minor count pointer.\n\n", __func__);
		return 0;
	}
	if (major_count == NULL) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with NULL major count pointer.\n\n", __func__);
		return 0;
	}
	if (! list->list_pivot_enabled) {
		fprintf(stderr, "\n==> Dlist Debug WARNING ==> %s() - "\
				"called on a list other than pivot class.\n\n", __func__);
	}
#else
	if ((list == NULL) || (minor_count == NULL) || (major_count == NULL))
		return 0;
#endif
	
	*minor_count = list->pivot_point.minor;
	*major_count = list->pivot_point.major;

	return list->element_count;
}

/* Examine a list for sorted elements, and return a boolean indicator if
	list is sorted or not.

	list		list_object that has been previously initialized.
	returns	TRUE			the list has sorted elements.
			FALSE			the list is not sorted. 								*/
boolean list_is_sorted(const list_object *restrict list) {

#ifdef DEBUG
	if (list__validate_list_object(list, FALSE, __func__) == FALSE) return FALSE;
#else
	if (list == NULL) return FALSE;
#endif

	return list->list_sorted;
}




/*	A NOTE about inter-list performance.
	------------------------------------

	Due to the nature of the code and design optimizations i made to
	dramatically improve the performance of often used functions,
	especially inserting, iterating, extracting, sorting, and searching,
	some of the less frequently used functions became a little slower.
	This is especially true for list_concatenate(), and list_clone().
	
	The structure and design changes made these 2 functions about 15%
	slower. But that seemed like a fair trade off to make sorts and
	searches 50% faster. In general code use, these high use functions
	are far more important than the rarely used ones.
	
	The performance improvements dramatically helped the other inter-list
	functions, such as list_transfer(), list_weave(), list_assay() and the
	various search_move and search_copy type functions.					-- dbw		*/



/* Concatenate 2 lists together into the destination list.

	This function copies and merges the 2 input lists, list_1 first, then
	list_2. The destination list must have been previously initialized by
	the caller. It may, or may not, contain existing elements. If it does
	the elements will be preserved, with the list_1 and then list_2 elements
	concatenated after them in the list.
	
	The lists may be the same, please see the variations below. Pivot class
	lists have restrictions when used as destinations, but are unrestricted
	when used as input lists. Finite lists can only be used as input lists.

	If the destination list was created with a LIST_COPY_DATA attribute, and
	a size function has been set for the list (see list_set_size_function()),
	the element data is fully copied to the new elements. If not, the element
	is copied as is, with the existing pointer to the original element data,
	unchanged.
	
	linear list usage
	list_concatenate(list_a, list_b, list_c);	add list_a & list_b to list_c
	list_concatenate(list_a, NULL, list_a);		add list_a to itself (double it)
	list_concatenate(list_a, NULL, list_b);		add list_a to list_b
	list_concatenate(list_a, list_a, list_b);	add list_a twice to list_b
	list_concatenate(list_a, list_b, list_b);	add list_a & list_b to list_b
	list_concatenate(list_b, list_a, list_b);	add list_b & list_a to list_b
	list_concatenate(list_a, list_a, list_a);	add list_a to itself twice (triple it)
	
	for destination pivot lists (no self-referencing destinations)
	list_concatenate(list_a, list_b, list_c);	add list_a & list_b to list_c
	list_concatenate(list_a, NULL, list_a);		invalid, returns LIST_ERR
	list_concatenate(list_a, NULL, list_b);		add list_a to list_b
	list_concatenate(list_a, list_a, list_b);	add list_a twice to list_b
	list_concatenate(list_a, list_b, list_b);	invalid, returns LIST_ERR
	list_concatenate(list_b, list_a, list_b);	invalid, returns LIST_ERR
	list_concatenate(list_a, list_a, list_a);	invalid, returns LIST_ERR
	
	
	At first glance example 3 above might look the same as list_clone(),
	however list_concatenate() does not clear the destination list first.
	
	If the destination list is a pivot class list, all elements added to
	it will be correctly positioned using the pivot function for that list
	regardless of either of the input lists types. However all elements
	will be added to the ends of the list, not the pivot of the list. A
	destination list that is pivot class, must not be referenced as one of
	the input lists.
	
	If instead of a straight copy of the 2 input lists, you wish to perform a
	more analytical list merge procedure, and remove any duplicates that may
	exist between the input lists, then please refer to list_assay() instead.
	
	The destination list must not be a finite class list.
	
	List_1 and list_2 remain unchanged, unless either was provided as the
	output destination list. In which case it will be modified as requested
	by this function.
	
	list_1		list_object that has been previously initialized,
				presumably with elements in it.
	list_2		list_object that has been previously initialized,
				presumably with elements in it, or NULL.
	dest_list	list_object that has been previously initialized, and
				may be empty, or have existing elements in it.
	returns	LIST_OK			success
			LIST_ERR		on error 												*/
int list_concatenate(	const list_object *list_1,
						const list_object *list_2,
						list_object *restrict dest_list) {

	register leaf_node *from_ptr;
	leaf_node *second_ptr;
	register long int l;
	long int i, j, k;
	int ret = LIST_OK;
	
#ifdef DEBUG
	if (list__validate_list_object(list_1, FALSE, __func__) == FALSE) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"problems with input list(#1).\n\n", __func__);
		return LIST_ERR;
	}
	if ((list_2) && (list__validate_list_object(list_2, FALSE, __func__) == FALSE)) {
			fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
					"problems with input list(#2).\n\n", __func__);
			return LIST_ERR;
	}
	if (list__validate_list_object(dest_list, LIST_WRITE, __func__) == FALSE) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"problems with the destination list.\n\n", __func__);
		return LIST_ERR;
	}
	if ((dest_list->list_pivot_enabled) && (dest_list == list_1)) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called on destination pivot list that is also input list #1.\n\n", __func__);
		return LIST_ERR;
	}
	if ((list_2) && (dest_list->list_pivot_enabled) && (dest_list == list_2)) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called on destination pivot list that is also input list #2.\n\n", __func__);
		return LIST_ERR;
	}
	if (dest_list->list_finite_enabled == TRUE) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with a finite class list as the destination.\n\n", __func__);
		return LIST_ERR;
	}
	if (list_1->iteration_enabled) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"input list(#1) has iteration enabled.\n\n", __func__);
		return LIST_ERR;
	}
	if ((list_2) && (list_2->iteration_enabled)) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"input list(#2) has iteration enabled.\n\n", __func__);
		return LIST_ERR;
	}
	if (dest_list->iteration_enabled) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"destination list has iteration enabled.\n\n", __func__);
		return LIST_ERR;
	}
#else
	if ((list_1 == NULL) || (dest_list == NULL))
		return LIST_ERR;
	if ((dest_list->list_pivot_enabled) && (dest_list->functions.pivot == NULL))
		return LIST_ERR;
	if ((dest_list->list_pivot_enabled) && ((dest_list == list_1) || (dest_list == list_2)))
		return LIST_ERR;
	if ((list_1->iteration_enabled)
			|| ((list_2) && (list_2->iteration_enabled))
			|| (dest_list->list_finite_enabled == TRUE)
			|| (dest_list->iteration_enabled))
		return LIST_ERR;
#endif

	i = list_1->element_count;
	if (list_2) i+= list_2->element_count;
	if (i < 1) return LIST_OK;
	if (i > UINT_MAX) {
#ifdef DEBUG
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"the two input lists combined, exceed the possible capacity of the "\
				"destination list (%u elements).\n\n", __func__, UINT_MAX);
#endif	
		return LIST_ERR;
	}
	
	/* cache the element counts now, in case the lists are the same ones. */
	i = list_1->element_count;
	if (list_2) {
		second_ptr = list_2->head_guard->next_leaf;
		j = list_2->element_count;
	} else {
		second_ptr = NULL;
		j = -1;
	}
	dest_list->list_sorted = FALSE;

	/* Copy each input list in turn, to the destination list.
	
		The outer loop runs twice, the inner loop runs until each input list
		is exhausted, copying every element in the list.
		The mid point for the destination list, is set automatically in the
		bowels of the code below. 													*/
	for (k = i, from_ptr = list_1->head_guard->next_leaf;
			k != -1;
			k = (k == i) ? j : -1, i = -1, from_ptr = second_ptr) {
		for (l = k; (l > 0) && (ret == LIST_OK); --l, from_ptr = from_ptr->next_leaf) {
			ret = list__add_element_extra_list(dest_list, from_ptr);			
		}
	}

#ifdef DLIST_DEBUG
	assert(list__audit(dest_list) == LIST_OK);
#endif		/* DLIST_DEBUG */
	return ((ret == LIST_OK) ? LIST_OK : ret);
}

/* Create a clone of a list, into the destination list.

	This function copies the input list into the destination list, which must
	have been previously initialized by the caller. The destination list should
	be empty first, before calling this function, however if the destination list
	contains existing elements, they will be deleted first, before the cloning
	operation.
	
	The lists must not be the same. If you wish the perform such merge operations,
	please refer to the list_concatenate() function instead.
	
	If the destination list was created with a LIST_COPY_DATA attribute, and
	a size function has been set for the list (see list_set_size_function()),
	the element data is fully copied to the new elements. If not, the element
	is copied as is, with the existing pointer to the original element data,
	unchanged.
	
	The input and destination lists must be of the same class, either linear or
	pivot. If you wish to make a copy of a list in a different class, please
	use the list_concatenate() function instead.
	
	The destination list must not be a finite class list. However the input list
	may be a finite list, as it is treated as a linear list for this function, as
	long as the destination list is linear class.
		
	The input list remains unchanged, the destination list will be modified as
	requested by this function.
	
	list		list_object that has been previously initialized,
				presumably with elements in it.
	dest_list	list_object that has been previously initialized, and
				may be empty, or have existing elements in it that will
				be removed first.
	returns	LIST_OK			success
			LIST_ERR		on error 												*/
int list_clone(	const list_object *list,
				list_object *restrict dest_list) {

	leaf_node *from_ptr;
	unsigned int total_elements;
	unsigned int m, middle;
	unsigned int i;
	int ret = LIST_OK;
	
#ifdef DEBUG
	if (list__validate_list_object(list, FALSE, __func__) == FALSE) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"problems with the input list.\n\n", __func__);
		return LIST_ERR;
	}
	if (list__validate_list_object(dest_list, LIST_WRITE, __func__) == FALSE) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"problems with the destination list.\n\n", __func__);
		return LIST_ERR;
	}
	if (dest_list == list) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with the same list for input and output.\n\n", __func__);
		return LIST_ERR;
	}
	if ((list->list_pivot_enabled == TRUE) && (dest_list->list_pivot_enabled == FALSE)) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with mismatching list types, input list is pivot, "\
				"output list is linear.\n\n", __func__);
		return LIST_ERR;
	}
	if ((list->list_pivot_enabled == FALSE) && (dest_list->list_pivot_enabled == TRUE)) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with mismatching list types, input list is linear, "\
				"output list is pivot.\n\n", __func__);
		return LIST_ERR;
	}
	if (dest_list->list_finite_enabled == TRUE) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with a finite class list as the destination.\n\n", __func__);
		return LIST_ERR;
	}
	if ((list->element_count) == 0) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with empty input list.\n\n", __func__);
		return LIST_ERR;
	}
	if (list->iteration_enabled) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"input list has iteration enabled.\n\n", __func__);
		return LIST_ERR;
	}
	if (dest_list->iteration_enabled) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"destination list has iteration enabled.\n\n", __func__);
		return LIST_ERR;
	}
#else
	if ((list == NULL) || (dest_list == NULL))
		return LIST_ERR;
	if (dest_list == list)
		return LIST_ERR;
	if (((list->list_pivot_enabled == TRUE) && (dest_list->list_pivot_enabled == FALSE))
			|| ((list->list_pivot_enabled == FALSE) && (dest_list->list_pivot_enabled == TRUE)))
		return LIST_ERR;
	if ((list->iteration_enabled) || (dest_list->iteration_enabled))
		return LIST_ERR;

	/* do not clear the output list, if no input */
	if ((list->element_count == 0) || (dest_list->list_finite_enabled == TRUE))
		return LIST_ERR;
#endif

	ret = list_clear(dest_list);
	if (ret < LIST_OK) {
#ifdef DEBUG
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"internal call to empty destination list failed.\n\n", __func__);
#endif
	return LIST_ERR;
	}

	dest_list->list_sorted = list->list_sorted;
	
	total_elements = list->element_count;
	if (total_elements%2) middle = (total_elements+1)/2;
	else middle = total_elements/2;
	if (list->list_sorted) dest_list->sorted_order = list->sorted_order;
	
	/* Copy each input list, to the destination list. Set the mid point for the completed list, when it is found. */
	for (i = m = 0, ret = LIST_OK, from_ptr = list->head_guard->next_leaf;
			(i < total_elements) && (ret == LIST_OK);
			from_ptr = from_ptr->next_leaf, ++i, ++m) {
		if (m == middle) dest_list->mid_point = dest_list->tail_guard->prior_leaf;

		ret = list__insert_leaf_node(dest_list, dest_list->tail_guard->prior_leaf, from_ptr->element, 0, FALSE, from_ptr->internal->pivot.value);

		if ((ret == LIST_OK) && (dest_list->functions.insert))
			dest_list->functions.insert(dest_list->tail_guard->prior_leaf->element);
	}
	
#ifdef DLIST_DEBUG
	assert(list__audit(dest_list) == LIST_OK);
#endif		/* DLIST_DEBUG */
	return ((ret == LIST_OK) ? LIST_OK : ret);
}

/* Transfer all the elements of a list, into the destination list.

	This function moves the entire set of elements from the input list into the
	destination list, which must have been previously initialized by the caller.
	The destination list should be empty first, before calling this function,
	however if the destination list contains existing elements, they will be
	deleted first, before the transfer operation.
	
	The lists must not be the same. If you wish the perform such merge
	operations, please refer to the list_concatenate() function instead.
	
	The LIST_COPY_DATA attribute is ignored for both lists. This function moves
	the actual elements from input_list to dest_list, leaving input_list empty
	and not creating any new elements. However the destination list must have
	been correctly set up, if it requires size or pivot functions.
	
	The input and destination lists must be of the same class, either linear,
	pivot or finite. If you wish to transfer elements from a list in a different
	class, please use the list_concatenate() function, or list_fetch() and
	list_append() instead.
	
	If the lists are of the finite class, then dest_list must have an equal
	or greater capacity than the input_list.
		
	The input list is emptied, the destination list will be emptied and then
	modified as requested by this function.
	
	list		list_object that has been previously initialized,
				presumably with elements in it.
	dest_list	list_object that has been previously initialized, and
				may be empty, or have existing elements in it that will
				be removed first.
	returns	LIST_OK			success
			LIST_ERR		on error 												*/
int list_transfer(	list_object *restrict input_list,
					list_object *restrict dest_list) {

	leaf_node *ptr;
	int ret = LIST_OK;
	
#ifdef DEBUG
	if (list__validate_list_object(input_list, FALSE, __func__) == FALSE) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"problems with the input list.\n\n", __func__);
		return LIST_ERR;
	}
	if (list__validate_list_object(dest_list, LIST_WRITE, __func__) == FALSE) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"problems with the destination list.\n\n", __func__);
		return LIST_ERR;
	}
	if (dest_list == input_list) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with the same list for input and output.\n\n", __func__);
		return LIST_ERR;
	}
	if ((input_list->list_pivot_enabled == TRUE) && (dest_list->list_pivot_enabled == FALSE)) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with mismatching list types, input list is pivot, "\
				"output list is linear.\n\n", __func__);
		return LIST_ERR;
	}
	if ((input_list->list_pivot_enabled == FALSE) && (dest_list->list_pivot_enabled == TRUE)) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with mismatching list types, input list is linear, "\
				"output list is pivot.\n\n", __func__);
		return LIST_ERR;
	}
	if ((input_list->list_finite_enabled == TRUE) && (dest_list->list_finite_enabled == FALSE)) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with mismatching list types, input list is finite, "\
				"output list is linear.\n\n", __func__);
		return LIST_ERR;
	}
	if ((input_list->list_finite_enabled == FALSE) && (dest_list->list_finite_enabled == TRUE)) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with mismatching list types, input list is linear, "\
				"output list is finite.\n\n", __func__);
		return LIST_ERR;
	}
	if ((input_list->list_finite_enabled == TRUE) && (dest_list->max_element_count < input_list->max_element_count)) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with finite list types, where the destination list has a smaller "\
				"maximum capacity (%u) than the input list (%u).\n\n", __func__,
				dest_list->max_element_count, input_list->max_element_count);
		return LIST_ERR;
	}
	if ((input_list->element_count) == 0) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with empty input list.\n\n", __func__);
		return LIST_ERR;
	}
	if (input_list->iteration_enabled) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"input list has iteration enabled.\n\n", __func__);
		return LIST_ERR;
	}
	if (dest_list->iteration_enabled) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"destination list has iteration enabled.\n\n", __func__);
		return LIST_ERR;
	}
#else
	if ((input_list == NULL) || (dest_list == NULL))
		return LIST_ERR;
	if (dest_list == input_list)
		return LIST_ERR;
	if (((input_list->list_pivot_enabled == TRUE) && (dest_list->list_pivot_enabled == FALSE))
			|| ((input_list->list_pivot_enabled == FALSE) && (dest_list->list_pivot_enabled == TRUE)))
		return LIST_ERR;
	if (((input_list->list_finite_enabled == TRUE) && (dest_list->list_finite_enabled == FALSE))
			|| ((input_list->list_finite_enabled == FALSE) && (dest_list->list_finite_enabled == TRUE)))
		return LIST_ERR;
	if ((input_list->iteration_enabled) || (dest_list->iteration_enabled))
		return LIST_ERR;
	if ((input_list->list_finite_enabled == FALSE) && (dest_list->max_element_count < input_list->max_element_count))
		return LIST_ERR;

	/* do not clear the output list, if no input */
	if (input_list->element_count == 0) return LIST_ERR;
#endif

	ret = list_clear(dest_list);
	if (ret < LIST_OK) {
#ifdef DEBUG
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"internal call to empty destination list failed with error %d.\n\n", __func__, ret);
#endif
	return LIST_ERR;
	}
	return__free_leaf_node(dest_list->head_guard);
	return__free_leaf_node(dest_list->tail_guard);

#ifndef DLIST_USE_MALLOC
	dest_list->element_count = 1;						/* force the remaining freep list to be cleared, be clean */
	release__leaf_node_blocks(dest_list);
#endif	/* ! DLIST_USE_MALLOC */
	
	/* Move the entire chain of leaf nodes from the input list to the destination list */
	dest_list->head_guard = input_list->head_guard;
	dest_list->tail_guard = input_list->tail_guard;

	/* fix up the dest list master node */
#ifndef DLIST_USE_MALLOC
	leaf_node_array *block_ptr;

	dest_list->leaf_block = input_list->leaf_block;					/* we must move the leaf node block list as well, we will allocate new ones below */
	dest_list->freep = input_list->freep;
	for (block_ptr = dest_list->leaf_block; (block_ptr != NULL); block_ptr->master = dest_list, block_ptr = block_ptr->next_block);
#endif	/* ! DLIST_USE_MALLOC */

	dest_list->pivot_point.minor = input_list->pivot_point.minor;
	dest_list->pivot_point.major = input_list->pivot_point.major;
	dest_list->mid_point = input_list->mid_point;
	dest_list->list_sorted = input_list->list_sorted;
	dest_list->sorted_order = input_list->sorted_order;
	dest_list->iteration_point = NULL;
	dest_list->element_count = input_list->element_count;
	
	/* clean up the (now empty) input list master node */
	input_list->head_guard = NULL;
	input_list->tail_guard = NULL;
#ifndef DLIST_USE_MALLOC
	input_list->leaf_block = NULL;
	input_list->freep = NULL;
	if (allocate__leaf_node_block(input_list, dest_list->leaf_block->nodes_allocated) == NULL) {
#ifdef DEBUG
		fprintf(stderr, "\n==> Dlist INTERNAL ERROR ==> %s() - "\
				"unable to create leaf node block for input list guard nodes.\n\n", __func__);
#endif	
		return LIST_ERR;
	}	
#endif	/* ! DLIST_USE_MALLOC */

	input_list->head_guard = get__next_free_leaf_node(input_list);
	input_list->tail_guard = get__next_free_leaf_node(input_list);

	/* Assign empty list values to the guard leaf nodes */
	input_list->head_guard->internal->master = input_list->tail_guard->internal->master = input_list;
	input_list->head_guard->next_leaf = input_list->tail_guard;
	input_list->head_guard->prior_leaf = NULL;
	input_list->tail_guard->next_leaf = NULL;
	input_list->tail_guard->prior_leaf = input_list->head_guard;

	input_list->mid_point = NULL;
	input_list->head_guard->element = input_list->tail_guard->element = NULL;	/* No data here */

	input_list->pivot_point.minor = 0;
	input_list->pivot_point.major = 0;
	input_list->iteration_point = NULL;
	input_list->element_count = 0;
	if (input_list->list_pivot_enabled) {
		input_list->head_guard->internal->pivot.value = PIVOT_MINOR;
		input_list->tail_guard->internal->pivot.value = PIVOT_MAJOR;
	} else {
		input_list->head_guard->internal->pivot.value = input_list->tail_guard->internal->pivot.value = ENTIRE_LIST;
	}
	
	/* fix up the dest list leaf node master pointers */
	for (ptr = dest_list->head_guard; (ptr != NULL); ptr = ptr->next_leaf) {
		ptr->internal->master = dest_list;
	}
	

#ifdef DLIST_DEBUG
	assert(list__audit(input_list) == LIST_OK);
	assert(list__audit(dest_list) == LIST_OK);
#endif		/* DLIST_DEBUG */

	return LIST_OK;
}

/* Merge 2 lists together by weaving each element together in order, into
	the destination list.

	This function copies and merges the 2 input lists, element by element
	into the destination list, so the elements are weaved together. The
	destination list must have been previously initialized by the caller.
	It may, or may not, contain existing elements. If it does the elements
	will be removed first and the list cleared. Upon return the destination
	list will only contain an ordered set of elements from list_1 and list_2.
	
	This function is similar to list_concatenate() except that the lists are
	merged element by element, not serially, and this function clears the
	destination list first. list_concatenate() also allows the lists to
	overlap and perform doubling operations, this function does not.
	
	The input lists may be of any class and do not have to be the same list
	class, but the destination must be a linear class list.
	
	If the destination list was created with a LIST_COPY_DATA attribute, and
	a size function has been set for the list (see list_set_size_function()),
	the element data is fully copied to the new elements. If not, the element
	is copied as is, with the existing pointer to the original element data,
	unchanged.
	
	The caller sets the argument order to define which direction the elements
	are read from list_2. LIST_ASCENDING means list_2 is read in the same order
	(head to tail) as list_1. LIST_DESCENDING means list_2 is read in the
	opposite order, tail to head. This allows the 2 lists to be weaved in order
	or in opposite directions, as the caller requires.
	
	Demonstrating this functions output, by using the following 2 example input
	lists
	
	list_1 elements
	a1	a2	a3	a4	a5	a6	a7	a8	.	.	.	.	.	a99

	list_2 elements
	b1	b2	b3	b4	b5	b6	b7	b8	.	.	.	.	.	b99
	
	With an order of LIST_ASCENDING the destination list would be
	a1	b1	a2	b2	a3	b3	a4	b4	a5	b5	a6	b6	.	.	.	.	.	a99	b99

	With an order of LIST_DESCENDING the destination list would be
	a1	b99	a2	b98	a3	b97	a4	b96	a5	b95	a6	b94	.	.	.	.	.	a99	b1
	
	Although this function has limited uses, it is very helpful if maintaining 	
	sorted or well ordered very large lists, that you wish to merge and sort.
	Overall this would reduce the overhead of the merge and final sort
	operations considerably.

	If instead of a straight merge of the 2 input lists, you wish to perform a
	more analytical list merge procedure, and remove any duplicates that may
	exist between the input lists, then please refer to list_assay() instead.
		
	List_1 and list_2 remain unchanged.
	
	list_1		list_object that has been previously initialized,
				presumably with elements in it.
	list_2		list_object that has been previously initialized,
				presumably with elements in it, or NULL.
	order		either LIST_ASCENDING or LIST_DESCENDING to denote which
		 		order to read the elements in the list_2.
	dest_list	list_object that has been previously initialized, and
				may be empty, or have existing elements in it which will
				be removed first.
	returns	LIST_OK			success
			LIST_ERR		on error 												*/
int list_weave(	const list_object *list_1,
				const list_object *list_2,
				const boolean order,
				list_object *restrict dest_list) {

	leaf_node *list1_ptr, *list2_ptr, *end_ptr;
	int ret = LIST_OK;
	boolean flip = TRUE;
	
#ifdef DEBUG
	if (list__validate_list_object(list_1, FALSE, __func__) == FALSE) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"problems with input list(#1).\n\n", __func__);
		return LIST_ERR;
	}
	if (list__validate_list_object(list_2, FALSE, __func__) == FALSE) {
			fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
					"problems with input list(#2).\n\n", __func__);
			return LIST_ERR;
	}
	if (list__validate_list_object(dest_list, LIST_WRITE, __func__) == FALSE) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"problems with the destination list.\n\n", __func__);
		return LIST_ERR;
	}
	if (list_1 == list_2) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with the same input list for #1 and #2.\n\n", __func__);
		return LIST_ERR;
	}
	if (dest_list == list_1) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with destination list that is also input list #1.\n\n", __func__);
		return LIST_ERR;
	}
	if (dest_list == list_2) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with destination list that is also input list #2.\n\n", __func__);
		return LIST_ERR;
	}
	if (dest_list->list_pivot_enabled) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with a pivot class list as the destination.\n\n", __func__);
		return LIST_ERR;
	}
	if (dest_list->list_finite_enabled == TRUE) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with a finite class list as the destination.\n\n", __func__);
		return LIST_ERR;
	}
	if (list_1->iteration_enabled) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"input list(#1) has iteration enabled.\n\n", __func__);
		return LIST_ERR;
	}
	if (list_2->iteration_enabled) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"input list(#2) has iteration enabled.\n\n", __func__);
		return LIST_ERR;
	}
	if (dest_list->iteration_enabled) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"destination list has iteration enabled.\n\n", __func__);
		return LIST_ERR;
	}
	if ((list_1->element_count) < 1) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with empty input list(#1).\n\n", __func__);
		return LIST_ERR;
	}
	if ((list_2->element_count) < 1) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with empty input list(#2).\n\n", __func__);
		return LIST_ERR;
	}
#else
	if ((list_1 == NULL) || (list_2 == NULL) || (dest_list == NULL))
		return LIST_ERR;
	if ((list_1 == list_2) || (dest_list == list_1) || (dest_list == list_2))
		return LIST_ERR;
	if ((list_1->iteration_enabled)
			|| (list_2->iteration_enabled)
			|| (dest_list->list_finite_enabled == TRUE)
			|| (dest_list->list_pivot_enabled == TRUE)
			|| (dest_list->iteration_enabled))
		return LIST_ERR;
	if (((list_1->element_count) < 1) || ((list_2->element_count) < 1))
		return LIST_ERR;
#endif

	if ((list_1->element_count + list_2->element_count) > UINT_MAX) {
#ifdef DEBUG
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"the two input lists combined, exceed the possible capacity of the "\
				"destination list (%u elements).\n\n", __func__, UINT_MAX);
#endif	
		return LIST_ERR;
	}

	ret = list_clear(dest_list);
	if (ret < LIST_OK) {
#ifdef DEBUG
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"internal call to empty destination list failed.\n\n", __func__);
#endif
	return LIST_ERR;
	}
	dest_list->list_sorted = FALSE;

	list1_ptr = list_1->head_guard->next_leaf;
	if (order == LIST_ASCENDING) {
		list2_ptr = list_2->head_guard->next_leaf;
		end_ptr = list_2->tail_guard;
	} else {
		list2_ptr = list_2->tail_guard->prior_leaf;
		end_ptr = list_2->head_guard;
	}

	do {
		if (flip) {
			if (list1_ptr) {
				ret = list__add_element_extra_list(dest_list, list1_ptr);
				list1_ptr = list1_ptr->next_leaf;
				if (list1_ptr == list_1->tail_guard) list1_ptr = NULL;
			}
		} else {
			/* flop */
			if (list2_ptr) {
				ret = list__add_element_extra_list(dest_list, list2_ptr);
				if (order == LIST_ASCENDING) list2_ptr = list2_ptr->next_leaf;
				else list2_ptr = list2_ptr->prior_leaf;
				if (list2_ptr == end_ptr) list2_ptr = NULL;
			}
		}
		flip = !flip;
	} while((ret == LIST_OK) && ((list1_ptr) || (list2_ptr)));

#ifdef DLIST_DEBUG
	assert(list__audit(dest_list) == LIST_OK);
#endif		/* DLIST_DEBUG */
	return ((ret == LIST_OK) ? LIST_OK : ret);
}

/* Perform an analysis and merge of 2 inputs lists, dividing elements that are common
	from those that are unique.

	This function will perform an element by element analysis of 2 input
	lists, and create an output pivot class list, with the minor side filled
	with elements common to both lists, and the major side filled with the
	elements unique to one list or the other. The output list will contain a
	single enumeration of all elements present in both input lists, with no
	duplicates, and with the list arranged around a pivot point of a value of
	common or unique.
	
	This list_assay() function can also be considered and used as a list merge
	operation, with additional analysis information provided, which can be
	ignored if only a merge is desired. This function differs considerably
	from list_concatenate() which puts 2 lists together as-is, retaining all
	duplicates.
	
	The 2 input lists can be of any list class, and do not have to be the
	same class. The output list must be pivot class list. If you wish the
	output to be linear or finite class, then use another function such as
	list_split_pivot() to copy or clone the output list sections after this
	function is finished.
	
	Although the destination is pivot class, this function will not use any
	pivot function you may have already set for it. The pivot datum will be
	the uniqueness or commonness of the elements, between the 2 input lists,
	and elements will be added to the destination list using that criteria. If
	there is an existing pivot function, it will not be called, or disturbed.
	
	If a search function was previously set for this list (by calling
	list_set_search_function() ), this function will use that search function
	to compare the element data against each other.
	
	If a search function was not previously set for this list, or it was
	removed, this function will perform a straight compare between the data
	pointers in both elements. If the element pointers are the same, they
	are equal, if the pointers point to different element data areas, they
	are not equal.
	
	If the lists were equal the minor side of the destination list will have
	a set of elements in it, and major side will be empty. The output list is
	required, and must not be the same as either input list. The input lists
	must be unique. Like all the list transformation functions, if the output
	list was initialized with the attribute LIST_COPY_DATA, the new elements
	will copied in whole, otherwise only a copy of the element pointer will
	be added to the destination list, with ownership of the original element
	data item with the user, with an additional pointer to it, that must be
	considered prior to any resource removal actions.
	
	The number of unique elements in list_1 is returned in l1_unique, and the
	number of unique elements in list_2 is returned in l2_unique. Using this
	the user can easily locate which elements are which, from the pivot point
	forward to the tail of the list. The destination pivot list is organized
	as this diagram shows
	
		<------   elements in common   ------>	<------------    unique elements    ----------->
												<----  l1_unique  ----><----	l2_unique	--->
		elem#1 .. elem#2 .. elem#3 .. elem#4 ... elem .. elem .. elem .. elem .. elem .. elem#n
		  ^									  ^												^
		  |									  |												|
	<HEAD_OF_LIST>	<-- PIVOT_MINOR -->	   <PIVOT>		<-- PIVOT_MAJOR -->			<TAIL_OF_LIST)
	
	
	The definitive test, on return, to verify if the lists are equal, or
	different, or how many elements are common or unique, is to call the
	function list_size_pivot() and evaluates the results. Use pivot based list
	iteration to retrieve the elements as you need them.

	This function only returns elements in the destination list, it makes no
	changes to the input lists.
	
	list_1		list_object that has been previously initialized,
				presumably with elements in it.
	list_2		list_object that has been previously initialized,
				presumably with elements in it.
	dest_list	list_object that has been previously initialized as a
				pivot class list, may be empty, or have existing elements
				in it that will be removed first. This list will hold all
				elements in common between list_1 and list_2 on the
				PIVOT_MINOR side of the list, and all elements that are
				unique from list_1 and list_2 on the PIVOT_MAJOR side.
	l1_unique	count of the elements returned that were unique to list_1
				these will be the first elements from the pivot point
				towards the tail of the list.
	l2_unique	count of the elements returned that were unique to list_2
				these will be the last elements from the pivot point
				towards the tail of the list.
				l1_unique + l2_unique will equal the MAJOR side count
				returned by a call to list_size_pivot().
	returns	LIST_OK			the evaluation of the lists succeeded
			LIST_ERR		on error 												*/
int list_assay(	const list_object *list_1,
				const list_object *list_2,
				list_object *restrict dest_list,
				unsigned int *l1_unique,
				unsigned int *l2_unique) {

	leaf_node *from_ptr, *search_ptr, *new_ptr;
	long int i, j, k;
	unsigned int u1 = 0, u2 = 0;
	int ret = LIST_OK;
	boolean first_pass = TRUE;
	boolean element_unique;
	
#ifdef DEBUG
	if (list__validate_list_object(list_1, FALSE, __func__) == FALSE) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"problems with the input list(#1).\n\n", __func__);
		return LIST_ERR;
	}
	if (list__validate_list_object(list_2, FALSE, __func__) == FALSE) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"problems with input list(#2).\n\n", __func__);
		return LIST_ERR;
	}
	if (list__validate_list_object(dest_list, LIST_WRITE, __func__) == FALSE) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"problems with destination list.\n\n", __func__);
		return LIST_ERR;
	}
	if (! dest_list->list_pivot_enabled) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with a destination list other than pivot class.\n\n", __func__);
		return LIST_ERR;
	}
	if (l1_unique == NULL) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with NULL l1_unique argument.\n\n", __func__);
		return LIST_ERR;
	}
	if (l2_unique == NULL) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with NULL l2_unique argument.\n\n", __func__);
		return LIST_ERR;
	}
	if (list_1 == dest_list) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with same pointers for input list(#1) and destination list.\n\n", __func__);
		return LIST_ERR;
	}
	if (list_2 == dest_list) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with same pointers for input list(#2) and destination list.\n\n", __func__);
		return LIST_ERR;
	}
	if (list_1 == list_2) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with same pointers for input list(#1) and input list(#2), they "\
				"will always evaluate the same.\n\n", __func__);
		return LIST_ERR;
	}
	if (list_1->iteration_enabled) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"input list(#1) has iteration enabled.\n\n", __func__);
		return LIST_ERR;
	}
	if (list_2->iteration_enabled) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"input list(#2) has iteration enabled.\n\n", __func__);
		return LIST_ERR;
	}
	if (dest_list->iteration_enabled) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"destination list has iteration enabled.\n\n", __func__);
		return LIST_ERR;
	}
	if (list_1->functions.search != list_2->functions.search) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with mismatching search functions for the input lists.\n\n", __func__);
		return LIST_ERR;
	}
	if ((list_1->element_count) < 1) {
		fprintf(stderr, "\n==> Dlist Debug WARNING ==> %s() - "\
				"called with empty input list(#1).\n\n", __func__);
	}
	if ((list_2->element_count) < 1) {
		fprintf(stderr, "\n==> Dlist Debug WARNING ==> %s() - "\
				"called with empty input list(#2).\n\n", __func__);
	}
#else
	if ((list_1 == NULL) || (list_2 == NULL) || (dest_list == NULL)
			|| (l1_unique == NULL) || (l2_unique == NULL))
		return LIST_ERR;
	
	if ((list_1 == dest_list) || (list_2 == dest_list) || (list_1 == list_2))
		return LIST_ERR;

	if ((list_1->iteration_enabled)
			|| (list_2->iteration_enabled)
			|| (dest_list->iteration_enabled)
			|| (! dest_list->list_pivot_enabled))
		return LIST_ERR;
	if (list_1->functions.search != list_2->functions.search)
		return LIST_ERR;
#endif

	ret = list_clear(dest_list);
	if (ret < LIST_OK) {
#ifdef DEBUG
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"internal call to clear destination list failed.\n\n", __func__);
#endif
	return LIST_ERR;
	}

	i = list_1->element_count;
	j = list_2->element_count;

	/* Copy each input list, to the destination list. Run the outer loop twice, the inner loop
		for each element in each list. Set the mid point for the completed list, after we are finished. */
	for (k = i, ret = LIST_OK, from_ptr = list_1->head_guard->next_leaf;
				k != -1; k = (k == i) ? j : -1, i = -1, from_ptr = list_2->head_guard->next_leaf, first_pass = FALSE) {
		for (; (from_ptr) && (from_ptr->next_leaf != NULL) && (ret == LIST_OK); from_ptr = from_ptr->next_leaf) {

			/* first search the destination to see if this element is there already */
			for (search_ptr = dest_list->head_guard->next_leaf;
					(search_ptr) && (search_ptr != dest_list->tail_guard);
					search_ptr = search_ptr->next_leaf) {
				if (list_2->functions.search) {
					if (list_2->functions.search(search_ptr->element, from_ptr->element)) break;
				} else {
					if (search_ptr->element == from_ptr->element) break;
				}
			}
			if (search_ptr == dest_list->tail_guard) {
				/* not in destination yet, lets evaluate it now */
				element_unique = TRUE;
				if (first_pass) {
					for (search_ptr = list_2->head_guard->next_leaf;
							(search_ptr) && (search_ptr != list_2->tail_guard);
							search_ptr = search_ptr->next_leaf) {
						if (list_2->functions.search) {
							if (list_2->functions.search(search_ptr->element, from_ptr->element)) break;
						} else {
							if (search_ptr->element == from_ptr->element) break;
						}
					}
					if (search_ptr != list_2->tail_guard) element_unique = FALSE;
					else element_unique = TRUE;
				}
				if (element_unique) {
					ret = list__insert_leaf_node(dest_list, dest_list->tail_guard->prior_leaf,
													from_ptr->element, 0, FALSE, PIVOT_MAJOR);
					new_ptr = dest_list->tail_guard->prior_leaf;
					if (first_pass) ++u1;
					else ++u2;
				} else {
					ret = list__insert_leaf_node(dest_list, dest_list->head_guard,
													from_ptr->element, 0, FALSE, PIVOT_MINOR);
					new_ptr = dest_list->head_guard->next_leaf;
				}
				if ((ret == LIST_OK) && (dest_list->functions.insert)) dest_list->functions.insert(new_ptr->element);
			}
			/* this element is already in the destination, so skip it */
		}
	}

	list__full_optimize_index(dest_list, FALSE);
	*l1_unique = u1;
	*l2_unique = u2;

#ifdef DLIST_DEBUG
	assert(list__audit(dest_list) == LIST_OK);
#endif		/* DLIST_DEBUG */
	return LIST_OK;
}

/* Split a pivot class list, and create 2 destination lists from its contents.

	This function copies the input list into 2 destination lists, which must
	have been previously initialized by the caller. The destination lists should
	be empty first, before calling this function, however if the destination list
	contains existing elements, they will be deleted first, before the split and
	copy operation.
	
	The lists must not be the same. If you wish the perform such merge operations,
	please refer to the list_concatenate() function instead.
	
	If the destination list was created with a LIST_COPY_DATA attribute, and
	a size function has been set for the list (see list_set_size_function()),
	the element data is fully copied to the new elements. If not, the element
	is copied as is, with the existing pointer to the original element data,
	unchanged.
	
	The input list must be a pivot class list. The two destination lists may be
	of either type, and do not have to be the same class lists. The minor or left
	hand side of the input list is copied to destination list #1, and the major
	or right hand side of the input list, is copied to destination list #2. The
	split happens at the user defined pivot point, which was established by the
	pivot function in operation on the input list.
	
	The input list remains unchanged, the destination list will be modified as
	requested by this function.
	
	list		list_object that has been previously initialized, with
				elements in it.
	dest_list1	list_object that has been previously initialized, and may
				be empty, or have existing elements in it that will be
				removed first.
	dest_list2	list_object that has been previously initialized, and may
				be empty, or have existing elements in it that will be
				removed first.
	returns	LIST_OK			success
			LIST_ERR		on error 												*/
int list_split_pivot(	const list_object *list,
						list_object *restrict dest_list1,
						list_object *restrict dest_list2) {

	leaf_node *from_ptr;
	list_object *next_list;
	long int i, j, k, l;
	int ret = LIST_OK;
	
#ifdef DEBUG
	if (list__validate_list_object(list, FALSE, __func__) == FALSE) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"problems with the input list.\n\n", __func__);
		return LIST_ERR;
	}
	if (list__validate_list_object(dest_list1, LIST_WRITE, __func__) == FALSE) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"problems with destination list(#1).\n\n", __func__);
		return LIST_ERR;
	}
	if (list__validate_list_object(dest_list2, LIST_WRITE, __func__) == FALSE) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"problems with destination list(#2).\n\n", __func__);
		return LIST_ERR;
	}
	if ((list->element_count) < 1) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with empty input list.\n\n", __func__);
		return LIST_ERR;
	}
	if (! list->list_pivot_enabled) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called on a list other than pivot class.\n\n", __func__);
		return LIST_ERR;
	}
	if (list == dest_list1) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with same pointers for input list and destination list(#1).\n\n", __func__);
		return LIST_ERR;
	}
	if (list == dest_list2) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with same pointers for input list and destination list(#2).\n\n", __func__);
		return LIST_ERR;
	}
	if (dest_list1 == dest_list2) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with same pointers for destination list(#1) and destination list(#2).\n\n",
						__func__);
		return LIST_ERR;
	}
	if (list->iteration_enabled) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"input list has iteration enabled.\n\n", __func__);
		return LIST_ERR;
	}
	if (dest_list1->iteration_enabled) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"destination list(#1) has iteration enabled.\n\n", __func__);
		return LIST_ERR;
	}
	if (dest_list2->iteration_enabled) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"destination list(#2) has iteration enabled.\n\n", __func__);
		return LIST_ERR;
	}
	if (dest_list1->list_finite_enabled) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called on destination finite class list(#1).\n\n", __func__);
		return LIST_ERR;
	}
	if (dest_list2->list_finite_enabled) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called on destination finite class list(#2).\n\n", __func__);
		return LIST_ERR;
	}
#else
	if ((list == NULL) || (dest_list1 == NULL) || (dest_list2 == NULL))
		return LIST_ERR;

	if ((list->element_count < 1) || (! list->list_pivot_enabled))
		return LIST_ERR;
	
	if ((list == dest_list1) || (list == dest_list2) || (dest_list1 == dest_list2))
		return LIST_ERR;

	if (((dest_list1->list_pivot_enabled) && (dest_list1->functions.pivot == NULL))
			|| ((dest_list2->list_pivot_enabled) && (dest_list2->functions.pivot == NULL)))
		return LIST_ERR;
	if ((list->iteration_enabled)
			|| (dest_list1->iteration_enabled)
			|| (dest_list2->iteration_enabled))
		return LIST_ERR;
	if ((dest_list1->list_finite_enabled) || (dest_list2->list_finite_enabled))
		return LIST_ERR;
#endif

	ret = list_clear(dest_list1);
	if (ret < LIST_OK) {
#ifdef DEBUG
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"internal call to clear destination list(#1) failed.\n\n", __func__);
#endif
	return LIST_ERR;
	}
	ret = list_clear(dest_list2);
	if (ret < LIST_OK) {
#ifdef DEBUG
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"internal call to clear destination list(#2) failed.\n\n", __func__);
#endif
	return LIST_ERR;
	}
	dest_list1->list_sorted = FALSE;
	dest_list2->list_sorted = FALSE;

	i = list->pivot_point.minor;
	j = list->pivot_point.major;

	/* Copy each input list, to the destination list. Run the outer loop twice, the inner loop
		for each element in each list. Set the mid point for the completed list, when it is found. */
	for (k = i, ret = LIST_OK, from_ptr = list->head_guard->next_leaf, next_list = dest_list1;
				k != -1; k = (k == i) ? j : -1, i = -1, next_list = dest_list2) {
		for (l = 0; (l < k) && (ret == LIST_OK); ++l) {
			ret = list__add_element_extra_list(next_list, from_ptr);
			from_ptr = from_ptr->next_leaf;
		}
	}

#ifdef DLIST_DEBUG
	assert(list__audit(dest_list1) == LIST_OK);
	assert(list__audit(dest_list2) == LIST_OK);
#endif		/* DLIST_DEBUG */

	return ((ret == LIST_OK) ? LIST_OK : ret);
}

/* Perform an element by element comparison of 2 input lists, searching for
	the first element which is not equal between them.
	
	This function traverses the 2 input lists, element by element in strict
	forward order. Each element is compared in sequence and if a non-equal
	result is obtained, the function will return with the index number in
	the lists of the non-matching element. It will only find the first non-
	matching element, and return.
	
	If a search function was previously set for this list (by calling
	list_set_search_function() ), this function will use that search function
	to compare the element datam's against each other.
	
	If a search function was not previously set for this list, or it was
	removed, this function will perform a straight compare between the data
	pointers in both elements. If the element pointers are the same, they
	are equal, if the pointers point to different element data areas, they
	are not equal.
	
	The definitive test, on return, to verify if the lists are equal, is to
	check the return code.
	
	This function only returns an index to any non-matching element in the
	lists, it makes no changes to the lists.
	
	list_1		list_object that has been previously initialized,
				presumably with elements in it.
	list_2		list_object that has been previously initialized,
				presumably with elements in it.
	returns	index	pointer to variable used to store the element index, if
		 			a non-matching element is found in both lists.
			LIST_OK			success, the 2 lists match
			ELEMENT_INVALID	if a non-matching pair of elements was found
			LIST_ERR		on error 												*/
int list_compare(	const list_object *list_1,
					const list_object *list_2,
					unsigned int *index) {

	leaf_node *ptr1, *ptr2 = NULL;
	unsigned int i;
	
#ifdef DEBUG
	if (list__validate_list_object(list_1, FALSE, __func__) == FALSE) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"problems with input list(#1).\n\n", __func__);
		return LIST_ERR;
	}
	if (list__validate_list_object(list_2, FALSE, __func__) == FALSE) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"problems with input list(#2).\n\n", __func__);
		return LIST_ERR;
	}
	if (list_1 == list_2) {
		fprintf(stderr, "\n==> Dlist Debug WARNING ==> %s() - "\
				"called with the same input lists.\n\n", __func__);
		return LIST_OK;
	}
	if (index == NULL) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with a NULL pointer for the index argument.\n\n", __func__);
		return LIST_ERR;
	}
	if (list_1->functions.search != list_2->functions.search) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with mismatching search functions for the input lists.\n\n", __func__);
		return LIST_ERR;
	}
#else
	if ((list_1 == NULL) || (list_2 == NULL) || (index == NULL)) return LIST_ERR;
	if (list_1 == list_2) return LIST_OK;
	if (list_1->functions.search != list_2->functions.search) return LIST_ERR;
#endif
	
	/* set default return state in case of errors, there is to be no ambiguity of success */
	*index = 0;

	for (i = 0, ptr1 = list_1->head_guard->next_leaf, ptr2 = list_2->head_guard->next_leaf;
			(ptr1) && (ptr2) && (ptr1 != list_1->tail_guard) && (ptr2 != list_2->tail_guard);
			++i, ptr1 = ptr1->next_leaf, ptr2 = ptr2->next_leaf) {
		if (list_1->functions.search) {
			if (! list_1->functions.search(ptr1->element, ptr2->element)) break;
		} else {
			if (ptr1->element != ptr2->element) break;
		}
	}
	
	/* Do the lists match ? - if yes, return to caller */
	if ((ptr1 == list_1->tail_guard) && (ptr2 == list_2->tail_guard)) return LIST_OK;	/* Yes */
	
	/* No, lists don't match. return identity of found non-matching element */
	*index = i;
	return ELEMENT_INVALID;
}

/* Perform an element by element comparison of 2 inputs lists, searching for
	all the elements which is are equal between them.
	
	This function traverses the 2 input lists, element by element in strict
	forward order. Each element is compared in sequence and if a non-equal
	result is obtained, the function will count the number of such differing,
	non-matching, elements. It will find all non-matching elements, and then
	return. This function makes no attempt to semantically analyse the lists,
	for example 2 lists of 100 elements, that are the same with the exception
	that list_2 has an extra element in index position 2, will return a
	result of 97 differences, as all elements after element 1 will be
	different. To perform a more detail semantic comparison of two lists,
	please refer to the function list_assay() instead.
	
	If a search function was previously set for this list (by calling
	list_set_search_function() ), this function will use that search function
	to compare the data of both elements against each other.
	
	If a search function was not previously set for this list, or it was
	removed, this function will perform a straight compare between the data
	pointers in both elements. If the element pointers are the same, they
	are equal, if the pointers point to different element data areas, they
	are not equal.
	
	The definitive test, on return, to verify if the lists are equal, is to
	check the return code.
	
	This function only returns an index to any non-matching element in the
	lists, it makes no changes to the lists.
	
	list_1		list_object that has been previously initialized,
				presumably with elements in it.
	list_2		list_object that has been previously initialized,
				presumably with elements in it.
	returns	0				the two input lists match
			>0				the number of non-matching elements found
			LIST_ERR		on error 												*/
int list_differences(	const list_object *list_1,
						const list_object *list_2) {

	leaf_node *ptr1, *ptr2 = NULL;
	unsigned int i;
	
#ifdef DEBUG
	if (list__validate_list_object(list_1, FALSE, __func__) == FALSE) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"problems with input list(#1).\n\n", __func__);
		return LIST_ERR;
	}
	if (list__validate_list_object(list_1, FALSE, __func__) == FALSE) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"problems with input list(#2).\n\n", __func__);
		return LIST_ERR;
	}
	if (list_1 == list_2) {
		fprintf(stderr, "\n==> Dlist Debug WARNING ==> %s() - "\
				"called with the same input lists.\n\n", __func__);
		return 0;
	}
	if (list_1->functions.search != list_2->functions.search) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with mismatching search functions for the input lists.\n\n", __func__);
		return LIST_ERR;
	}
#else
	if ((list_1 == NULL) || (list_2 == NULL)) return LIST_ERR;
	if (list_1 == list_2) return 0;
	if (list_1->functions.search != list_2->functions.search) return LIST_ERR;
#endif
	
	for (i = 0, ptr1 = list_1->head_guard->next_leaf, ptr2 = list_2->head_guard->next_leaf;
			(ptr1) && (ptr2) && (ptr1 != list_1->tail_guard) && (ptr2 != list_2->tail_guard);
			ptr1 = ptr1->next_leaf, ptr2 = ptr2->next_leaf) {
		if (list_1->functions.search) {
			if (! list_1->functions.search(ptr1->element, ptr2->element)) ++i;
		} else {
			if (ptr1->element != ptr2->element) ++i;
		}
	}
	
	/* Do the lists have same length ? - if yes, return to caller */
	if ((ptr1 == list_1->tail_guard) && (ptr2 == list_2->tail_guard))
		return i;
		
	while ((ptr1) && (ptr1 != list_1->tail_guard)) {
		++i;
		ptr1 = ptr1->next_leaf;
	}
	while ((ptr2) && (ptr2 != list_2->tail_guard)) {
		++i;
		ptr2 = ptr2->next_leaf;
	}
	if (i > INT_MAX) i = INT_MAX;								/* deal with unlikely overflow */
	
	return (int)i;
}

/* Append an element to the end of a list.

	Among its many uses, this function is useful for adding elements
	to FIFO lists and queues. It also allows elements to be added to
	list being managed by index, without disturbing the index
	optimizations that making modifications by reference causes.
	
	This function is not available for pivot class lists, or finite
	class lists. If the list is currently sorted, the sorted indexes
	will be voided, and the list_sorted_xx() functions are no longer
	available for the list, until it is resorted.

	list		list_object that has been previously initialized.
	element		pointer to caller supplied datum object to be stored
		 		as the new element in list.
	returns	LIST_OK				success, new element added to the list
			ELEMENT_NO_DATUM	element data not supplied
			LIST_ERR			on error	 										*/
int list_append(	list_object *restrict list,
					const void *element) {

	int ret;

#ifdef DEBUG
	if (list__validate_list_object(list, LIST_WRITE, __func__) == FALSE)
		return LIST_ERR;
	if (element == NULL) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with NULL element pointer.\n\n", __func__);
		return ELEMENT_NO_DATUM;
	}
	if (list->iteration_enabled) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"list has iteration enabled.\n\n", __func__);
		return LIST_ERR;
	}
	if (list->list_finite_enabled == TRUE) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called on a finite class list.\n\n", __func__);
		return LIST_ERR;
	}
	if (list->list_pivot_enabled == TRUE) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called on a pivot class list.\n\n", __func__);
		return LIST_ERR;
	}
#else
	if ((list == NULL)
			|| (list->iteration_enabled)
			|| (list->list_finite_enabled == TRUE)
			|| (list->list_pivot_enabled == TRUE))
		return LIST_ERR;
	if (element == NULL)
		return ELEMENT_NO_DATUM;
#endif

	ret = list__insert_leaf_node(list, list->tail_guard->prior_leaf, element, list->element_count, TRUE, 0);
	if ((ret == LIST_OK) && (list->functions.insert)) list->functions.insert(list->tail_guard->prior_leaf->element);
	list->list_sorted = FALSE;

	return ret;
}

/* Prepend an element to the front of a list.

	Among its many uses, this function is useful for adding elements
	to LIFO lists and stacks. It also allows elements to be added to
	list being managed by index, without disturbing the index
	optimizations that making modifications by reference causes.
	
	This function is not available for pivot class lists. If the list
	is currently sorted, the sorted indexes will be voided, and the
	list_sorted_xx() functions are no longer available for the list,
	until it is resorted.

	list		list_object that has been previously initialized.
	element		pointer to caller supplied datum object to be stored
		 		as the new element in the list.
	returns	LIST_OK				success, new element added to the list
			ELEMENT_NO_DATUM	element data not supplied
			LIST_ERR			on error	 										*/
int list_prepend(	list_object *restrict list,
					const void *restrict element) {

	int ret;

#ifdef DEBUG
	if (list__validate_list_object(list, LIST_WRITE, __func__) == FALSE)
		return LIST_ERR;
	if (element == NULL) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with NULL element pointer.\n\n", __func__);
		return ELEMENT_NO_DATUM;
	}
	if (list->iteration_enabled) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"list has iteration enabled.\n\n", __func__);
		return LIST_ERR;
	}
	if (list->list_pivot_enabled == TRUE) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called on a pivot class list.\n\n", __func__);
		return LIST_ERR;
	}
#else
	if ((list == NULL)
			|| (list->iteration_enabled)
			|| (list->list_pivot_enabled == TRUE))
		return LIST_ERR;
	if (element == NULL)
		return ELEMENT_NO_DATUM;
#endif

	ret = list__insert_leaf_node(list, list->head_guard, element, 0, TRUE, 0);
	if ((ret == LIST_OK) && (list->functions.insert)) list->functions.insert(list->head_guard->next_leaf->element);
	list->list_sorted = FALSE;

	return ret;
}

/* Add an element to a list, in the position indicated by the index.
	
	This function is not available for pivot class lists. If the list
	is currently sorted, the sorted indexes will be voided, and the
	list_sorted_xx() functions are no longer available for the list,
	until it is resorted.

	list		list_object that has been previously initialized.
	element		pointer to caller supplied datum object to be stored
		 		as the new element in the list.
	index		the list index number of the location where to add the
				new element.
	returns	LIST_OK				success, new element added to the list
			ELEMENT_NO_DATUM	element data not supplied
			ELEMENT_NOT_FOUND	supplied index was not a valid in the list
			LIST_ERR			on error	 										*/
int list_insert_index(	list_object *restrict list,
						const void *element,
						const unsigned int index) {

	leaf_node *ptr;
	int ret;

#ifdef DEBUG
	if (list__validate_list_object(list, LIST_WRITE, __func__) == FALSE)
		return LIST_ERR;
	if (element == NULL) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with NULL element pointer.\n\n", __func__);
		return ELEMENT_NO_DATUM;
	}
	if (list->iteration_enabled) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"list has iteration enabled.\n\n", __func__);
		return LIST_ERR;
	}
	if (list->list_pivot_enabled == TRUE) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called on a pivot class list.\n\n", __func__);
		return LIST_ERR;
	}
	if (index > list->element_count) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with out-of-range index. Index '%u', max element index is '%u'.\n\n",
				__func__, index, (unsigned int)list->element_count);
		return ELEMENT_NOT_FOUND;
	}
#else
	if ((list == NULL)
			|| (list->iteration_enabled)
			|| (list->list_pivot_enabled == TRUE))
		return LIST_ERR;
	if (index > list->element_count)
		return ELEMENT_NOT_FOUND;
	if (element == NULL)
		return ELEMENT_NO_DATUM;
#endif

	ptr = list__locate_leaf_node(list, (long int)index-1);

	ret = list__insert_leaf_node(list, ptr, element, index, TRUE, 0);
	if ((ret == LIST_OK) && (list->functions.insert)) list->functions.insert(ptr->next_leaf->element);
	list->list_sorted = FALSE;
	
	return ret;
}

/* Add an element to a list, in the position indicated by the element reference.

	The new element will be inserted in the list at a position higher than the
	reference. If stated from head to tail, the new element will be on the tail
	side of the reference. If stated from left to right, it will be on the right
	side of the referenced element.
	
	It is not possible to add a new element to the head of the list with this
	function. For that you should use the function list_prepend() instead. It
	is possible to add an element to the tail of the list, assuming you already
	have the reference to the last element available. You can add the element
	as the #2 in the list, by using the #1 as the reference.
	
	This function can be used while the list has iteration enabled. It is the
	only function available to add elements during an iteration.
	
	This function is not available for pivot class lists, or finite class lists.
	If the list is currently sorted, the sorted indexes will be voided, and the
	list_sorted_xx() functions are no longer available for the list, until it
	is resorted.

	list		list_object that has been previously initialized.
	element		pointer to caller supplied datum object to be stored
		 		as the list element.
	reference	caller supplied element reference, where to insert the
				new element in the list.
	returns	LIST_OK				success, additionally reference will be updated
								to the reference of the newly added list element.
			ELEMENT_NO_DATUM	element data not supplied
			ELEMENT_INVALID		supplied reference was not a valid element in list
			LIST_ERR			on error											*/
int list_insert_ref(	list_object *restrict list,
						const void *element,
						element_reference **restrict reference) {

	leaf_node *ptr;
	int ret;
	
	if (reference == NULL) {
#ifdef DEBUG
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with NULL reference argument.\n\n", __func__);
#endif
		return LIST_ERR;
	}

	ptr = *reference;

#ifdef DEBUG
	if (list__validate_list_object(list, LIST_WRITE, __func__) == FALSE)
		return LIST_ERR;
	if (element == NULL) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with NULL element pointer.\n\n", __func__);
		return ELEMENT_NO_DATUM;
	}
	if (list->list_pivot_enabled == TRUE) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called on a pivot class list.\n\n", __func__);
		return LIST_ERR;
	}
	if (list->list_finite_enabled == TRUE) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called on a finite class list.\n\n", __func__);
		return LIST_ERR;
	}
	if (ptr == NULL) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with NULL reference pointer.\n\n", __func__);
		return LIST_ERR;
	}
	ret = list__validate_reference(list, ptr, __func__);
	if (ret != LIST_OK) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with invalid reference pointer, error code %d.\n\n", __func__, ret);
		return ret;
	}
#else
	if ((list == NULL)
			|| (list->list_pivot_enabled == TRUE)
			|| (ptr == NULL)
			|| (list->list_finite_enabled == TRUE))
		return LIST_ERR;
	if (list__validate_reference(list, ptr, __func__) != LIST_OK)
		return ELEMENT_INVALID;
	if (element == NULL)
		return ELEMENT_NO_DATUM;
#endif

	/* using the reference based updates destroys the integrity of the
		optimized index */
	list->mid_point = NULL;
	ret = list__insert_leaf_node(list, ptr, element, 0, FALSE, 0);
	
	if (ret) *reference = NULL;
	else *reference = ptr->next_leaf;

	if ((ret == LIST_OK) && (list->functions.insert)) list->functions.insert(ptr->next_leaf->element);
	list->list_sorted = FALSE;
	
#ifdef DLIST_DEBUG
	assert(list__audit(list) == LIST_OK);
#endif		/* DLIST_DEBUG */
	
	return ((ret) ? LIST_ERR : LIST_OK);
}

/* Add an element to a list, in the position computed by the list pivot function.

	The new element will be inserted in the list at a position that is determined
	by the result of calling the pivot function set for the list, with the
	element as input. The element will then be placed either the minor or major
	side of the pivot point, depending on the computed result.
	
	Additionally the position of the new element can be adjusted by the contents
	of the where argument. If HEAD_OF_LIST is provided, the new element is placed
	at the end of the list, the same as a list_append() or list_prepend() would
	do. If PIVOT_OF_LIST is provided, the new element is instead placed on the
	major or minor side of the pivot point.
	
	Using the following example, if this sorted list is added to a new pivot list
	
		a  b  c  d  e  f  g  h  i  j  k  l  m  n
	
	By using PIVOT_OF_LIST as the location, after all elements were added, the new
	list would look like :-
	
		a  b  c  d  e  f  g  n  m  l  k  j  i  h
					<------  ------>
	if instead it was added using END_OF_LIST, and the pivot point was <=g, the
	list would now look like :-
	
		g  f  e  d  c  b  a  h  i  j  k  l  m  n
		 ------>						<------
	The choice depends on the users needs, in particular END_OF_LIST will always
	cause the most recently added elements to be near the ends of the list, which
	can dramatically speed up searches of huge lists, if the likely candidate for
	a match is often more recent, than older.
		
	It is important to remember that pivot lists are not meant to retain ordering
	of elements, or be a naturally sorted list, it is a means of organizing new
	elements around a common datum used as a pivot point.
		
	This function is not available for linear or finite class lists. Use any of
	the other available insertion functions for those lists. A pivot function must
	have been set for the list, prior to calling list_store().

	list		list_object that has been previously initialized.
	element		pointer to caller supplied datum object to be stored
		 		as the new element in list.
	where		denote where to insert the new element, insertion will always
				be on the major or minor side of the pivot point, as decided
				by the pivot function.
				PIVOT_OF_LIST insert at the pivot of the list, or
				END_OF_LIST insert at the end of the list.
	returns	LIST_OK				success, new element added to the list
			ELEMENT_NO_DATUM	element data not supplied
			LIST_ERR			on error	 										*/
int list_store(	list_object *restrict list,
				const void *element,
				const boolean where) {

	leaf_node *ptr;
	long int index = 0;
	int scope, ret;

#ifdef DEBUG
	if (list__validate_list_object(list, LIST_WRITE, __func__) == FALSE)
		return LIST_ERR;
	if (element == NULL) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with NULL element pointer.\n\n", __func__);
		return ELEMENT_NO_DATUM;
	}
	if (list->iteration_enabled) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"list has iteration enabled.\n\n", __func__);
		return LIST_ERR;
	}
	if (list->list_pivot_enabled == FALSE) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called on a non pivot class list.\n\n", __func__);
		return LIST_ERR;
	}
	/* this is redundant, but put here just in case of future dlist changes */
	if (list->list_finite_enabled == TRUE) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called on a finite class list.\n\n", __func__);
		return LIST_ERR;
	}
#else
	if ((list == NULL)
			|| (list->iteration_enabled)
			|| (list->list_pivot_enabled == FALSE))
		return LIST_ERR;
	if ((list->functions.pivot == NULL) || (list->list_finite_enabled == TRUE))
		return LIST_ERR;
	if (element == NULL)
		return ELEMENT_NO_DATUM;
#endif

	scope = list->functions.pivot(element);

	if (where == END_OF_LIST) {
		if (scope < 0) {
			index = HEAD_OF_LIST;
			ptr = list->head_guard;
		} else {
			index = list->element_count;
			ptr = list->tail_guard->prior_leaf;
		}
	} else {
		/* where == PIVOT_OF_LIST */
		ptr = list__locate_pivot_leaf_node(list, &index);
	}

	ret = list__insert_leaf_node(list, ptr, element, index, TRUE, scope);
	if ((ret == LIST_OK) && (list->functions.insert)) list->functions.insert(ptr->next_leaf->element);
	list->list_sorted = FALSE;
	
	return ret;
}

/* Copy an element from one list to another, given the index position.
	
	This function locates an element from the input list, and copies it to
	the destination list. Element is identified by index position in the
	input list, and either appended to a linear class destination list, or
	stored by pivot value to a pivot class list.
	
	If the destination list was initialized with a LIST_COPY_DATA attribute,
	the new element will copied in whole. If the list has no LIST_COPY_DATA
	attribute, only a copy of the element pointer will be added to the
	destination list. The user should remember there is now an additional
	pointer to it, that must be considered prior to any resource removal
	actions.
	
	The destination list must not be a Finite class list.
	
	This function only copies the element and makes no changes to the input
	list.

	list		list_object that has been previously initialized.
	index		denotes where in the input list the element is to be found.
	dest_list	list_object that has been previously initialized, where the
				copied element is to be added.
	returns	LIST_OK				success, the new element was added to the
								destination list.
			ELEMENT_NOT_FOUND	no element was found matching the index.
			LIST_ERR			on error 											*/
int list_copy_index(	list_object *restrict list,
						const unsigned int index,
						list_object *restrict dest_list) {

	leaf_node *ptr;
	int ret;
	
#ifdef DEBUG
	if (list__validate_list_object(list, FALSE, __func__) == FALSE)
		return LIST_ERR;
	if (list__validate_list_object(dest_list, LIST_WRITE, __func__) == FALSE)
		return LIST_ERR;
	if (dest_list == list) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with the same list for input and output.\n\n", __func__);
		return LIST_ERR;
	}
	if (list->element_count < 1) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with empty input list.\n\n", __func__);
		return ELEMENT_NOT_FOUND;
	}
	if (dest_list->list_finite_enabled == TRUE) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called on a finite class list.\n\n", __func__);
		return LIST_ERR;
	}
	if (list->iteration_enabled) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"input list has iteration enabled.\n\n", __func__);
		return LIST_ERR;
	}
	if (dest_list->iteration_enabled) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"destination list has iteration enabled.\n\n", __func__);
		return LIST_ERR;
	}
	if (index >= list->element_count) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with out-of-range index. Index '%u', max element index is '%u'.\n\n",
				__func__, index, list->element_count-1);
		return ELEMENT_NOT_FOUND;
	}
#else	
	if ((list == NULL)
			|| (dest_list == NULL)
			|| (list->iteration_enabled)
			|| (dest_list == list)
			|| (dest_list->list_finite_enabled == TRUE))
		return LIST_ERR;
	if ((list->element_count < 1) || (index >= list->element_count))
		return ELEMENT_NOT_FOUND;
#endif

	ptr = list__locate_leaf_node(list, (long int)index);

	ret = list__add_element_extra_list(dest_list, ptr);
	dest_list->list_sorted = FALSE;
	
#ifdef DLIST_DEBUG
	if (ret == LIST_OK) assert(list__audit(dest_list) == LIST_OK);
#endif		/* DLIST_DEBUG */
	
	return ret;
}

/* Move an element from one list to another, given the index position.
	
	This function extracts an element from the input list, and inserts it to
	the destination list. Element is identified by index position in the
	input list, and either appended to a linear class destination list, or
	stored by pivot value to a pivot class list.
	
	This function is similar to list_copy_index() except the original
	element is deleted from the input list.
	
	If the destination list was initialized with a LIST_COPY_DATA attribute,
	the new element will copied in whole. If the list has no LIST_COPY_DATA
	attribute, only a copy of the element pointer will be added to the
	destination list. However there will not be an additional pointer to it,
	as the original element from the input list has been deleted. Element
	references are tied to the original element and list, and will not be
	valid after this move operation.
	
	The destination list must not be a Finite class list.
	
	This function removes the element from the input list, and releases its
	internal resources, along with the data resources if the input list was
	initialized with the LIST_COPY_DATA attribute.

	list		list_object that has been previously initialized.
	index		denotes where in the input list the element is to be found.
	dest_list	list_object that has been previously initialized, where the
				moved element is to be added.
	returns	LIST_OK				success, the new element was added to the
								destination list and then removed from the
								input list. If the list was created with the
								LIST_COPY_DATA attribute, the data that was
								allocated internally by D-List has now been
								released.
			ELEMENT_NOT_FOUND	no element was found matching the index.
			LIST_ERR			on error 											*/
int list_move_index(	list_object *restrict list,
						const unsigned int index,
						list_object *restrict dest_list) {

	leaf_node *ptr;
	int ret;
	
	/* list_copy_index() performs all the validation we need */
	ret = list_copy_index(list, index, dest_list);
	if (ret == LIST_OK) {
		/* element was copied, now delete the original */
		ptr = list__locate_leaf_node(list, (long int)index);
		if (list->functions.remove) list->functions.remove(ptr->element);
		list__remove_leaf_node(ptr, index);
	}
	
#ifdef DLIST_DEBUG
	if (ret == LIST_OK) assert(list__audit(list) == LIST_OK);
#endif		/* DLIST_DEBUG */
	
	return ret;
}

/* Copy an element from one list to another, given the element reference.
	
	This function copies the element from the list identified by element
	reference, and adds it to the destination list. The new element is either
	appended to a linear class destination list, or stored by pivot value to
	a pivot class list.
	
	It is functionally similar to list_copy_index(), except that the element
	is identified by reference, not by index. A reference must have
	been passed to the caller previously by D-List from another call, such as
	list_search(). Upon return the original element reference is still valid.
	
	If the destination list was initialized with a LIST_COPY_DATA attribute,
	the new element will copied in whole. If the list has no LIST_COPY_DATA
	attribute, only a copy of the element pointer will be added to the
	destination list. The user should remember there is now an additional
	pointer to it, that must be considered prior to any resource removal
	actions.

	The destination list must not be a Finite class list.
	
	This function only copies the element and makes no changes to the input
	list.

	list		list_object that has been previously initialized.
	reference	an abstract locator in list, of the element to be copied.
	dest_list	list_object that has been previously initialized, where the
				copied element is to be added.
	returns	LIST_OK				success, the new element was added to the
								destination list.
			ELEMENT_INVALID		if the element reference was invalid
			LIST_ERR			on error 											*/
int list_copy_ref(	list_object *restrict list,
					element_reference *restrict reference,
					list_object *restrict dest_list) {

	int ret;	

#ifdef DEBUG
	if (list__validate_list_object(list, FALSE, __func__) == FALSE)
		return LIST_ERR;
	if (list__validate_list_object(dest_list, LIST_WRITE, __func__) == FALSE)
		return LIST_ERR;
	if (dest_list == list) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with the same list for input and output.\n\n", __func__);
		return LIST_ERR;
	}
	if (list->element_count < 1) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with empty input list.\n\n", __func__);
		return ELEMENT_INVALID;
	}
	if (dest_list->list_finite_enabled == TRUE) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called on a finite class list.\n\n", __func__);
		return LIST_ERR;
	}
	ret = list__validate_reference(list, reference, __func__);
	if (ret != LIST_OK) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with invalid reference pointer, error code %d.\n\n", __func__, ret);
		return ELEMENT_INVALID;
	}
#else
	if ((list == NULL)
			|| (dest_list == NULL)
			|| (dest_list == list)
			|| (dest_list->list_finite_enabled == TRUE))
		return LIST_ERR;
	if (list->element_count < 1)
		return ELEMENT_INVALID;
	if (list__validate_reference(list, reference, __func__) != LIST_OK)
		return ELEMENT_INVALID;
#endif

	ret = list__add_element_extra_list(dest_list, reference);
	dest_list->list_sorted = FALSE;

#ifdef DLIST_DEBUG
	if (ret == LIST_OK) assert(list__audit(dest_list) == LIST_OK);
#endif		/* DLIST_DEBUG */

	return ret;
}

/* Move an element from one list to another, given the element reference.
	
	This function extracts an element from the input list identified by its
	element reference, and then adds it to the destination list. The new
	element is either appended to a linear class destination list, or stored
	by pivot value to a pivot class list.
	
	It is functionally similar to list_copy_ref(), except the original
	element is deleted from the input list.
	
	A reference must have been passed to the caller previously by D-List from
	another call, such as list_search(). Upon return the original element
	reference is no longer valid and is changed to NULL.
	
	If the destination list was initialized with a LIST_COPY_DATA attribute,
	the new element will copied in whole. If the list has no LIST_COPY_DATA
	attribute, only a copy of the element pointer will be added to the
	destination list. However there will not be an additional pointer to it,
	as the original element from the input list has been deleted.
	
	The destination list must not be a Finite class list.
	
	This function removes the element from the input list, and releases its
	internal resources, along with the data resources if the input list was
	initialized with the LIST_COPY_DATA attribute.

	list		list_object that has been previously initialized.
	reference	an abstract locator in list, of the element to be moved.
	dest_list	list_object that has been previously initialized, where the
				moved element is to be added.
	returns	LIST_OK				success, the new element was added to the
								destination list and then removed from the
								input list. If the list was created with the
								LIST_COPY_DATA attribute, the data that was
								allocated internally by D-List has now been
								released.
			ELEMENT_INVALID		if the element reference was invalid
			LIST_ERR			on error 											*/
int list_move_ref(	list_object *restrict list,
					element_reference **restrict reference,
					list_object *restrict dest_list) {

	leaf_node *ptr;
	int ret;

	if (reference == NULL) {
#ifdef DEBUG
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with NULL reference argument.\n\n", __func__);
#endif
		return LIST_ERR;
	}
	
	ptr = *reference;
	
	ret = list_copy_ref(list, ptr, dest_list);
	
	if (ret == LIST_OK) {
		/* using the reference based updates destroys the integrity of the
			optimized index */
		list->mid_point = NULL;

		if (list->functions.remove) list->functions.remove(ptr->element);
		list__remove_leaf_node(ptr, 0);
	}
	
	*reference = NULL;	

#ifdef DLIST_DEBUG
	if (ret == LIST_OK) assert(list__audit(list) == LIST_OK);
#endif		/* DLIST_DEBUG */

	return ret;
}

/* Search for, and copy, all elements that match the supplied datum, from
	one list to another.
	
	This function will traverse the input list from head to tail, searching
	for all elements with the matching datum supplied. For each element
	found this function copies the element from the input list, and adds it
	to the destination list. The new element is either appended to a linear
	class destination list, or stored by pivot value to a pivot class list.

	This is a simple version of search, the entire input list is traversed,
	no consideration of pivots, or direction are provided.
	
	If a search function was previously set for this input list (by calling
	list_set_search_function() ), this function will use that search function
	to compare the element data against the supplied datum.
	
	If a search function was not previously set for this list, or it was
	removed, this function will perform a straight compare between the data
	pointer in the elements and the pointer provided in datum. If the element
	pointers are the same, they are equal, if the pointers point to different
	element data areas, they are not equal.
	
	In most cases, the user will want to set a search function first. But this
	can be used to find a data element by a known pointer reference.
	
	If the destination list was initialized with a LIST_COPY_DATA attribute,
	the new element will copied in whole. If the list has no LIST_COPY_DATA
	attribute, only a copy of the element pointer will be added to the
	destination list. The user should remember there is now an additional
	pointer to it, that must be considered prior to any resource removal
	actions.

	The destination list must not be a Finite class list.
	
	This function only copies the elements and makes no changes to the input
	list.

	list		list_object that has been previously initialized.
	datum 		data to be used for the search criteria.
	dest_list	list_object that has been previously initialized, where the
				copied element is to be added.
	returns	>0					the number of elements found in the input
								list, that were added to the destination list.
			0					if no elements matched the datum
			ELEMENT_NO_DATUM	element data not supplied
			LIST_ERR			on error 											*/
int list_search_copy(	list_object *restrict list,
						const void *datum,
						list_object *restrict dest_list) {

	int ret;

	ret = list__validate_evaluate_range(list, datum, NULL, FALSE, FALSE, dest_list, __func__);
	if (ret!= LIST_OK)
		return ret;

	return(list__leaf_node_evaluate_range(list, datum, 0, NULL, FALSE, FALSE, dest_list));
}

/* Search for, and move, all elements that match the supplied datum, from
	one list to another.
	
	This function will traverse the input list from head to tail, searching
	for all elements with the matching datum supplied. For each element
	found this function copies the element from the input list, and adds it
	to the destination list, then deletes it from the input list. The new
	element is either appended to a linear class destination list, or stored
	by pivot value to a pivot class list.

	This is a simple version of search, the entire input list is traversed,
	no consideration of pivots, or direction are provided.
	
	If a search function was previously set for this input list (by calling
	list_set_search_function() ), this function will use that search function
	to compare the element data against the supplied datum.
	
	If a search function was not previously set for this list, or it was
	removed, this function will perform a straight compare between the data
	pointer in the elements and the pointer provided in datum. If the element
	pointers are the same, they are equal, if the pointers point to different
	element data areas, they are not equal.
	
	In most cases, the user will want to set a search function first. But this
	can be used to find a data element by a known pointer reference.
	
	If the destination list was initialized with a LIST_COPY_DATA attribute,
	the new element will copied in whole. If the list has no LIST_COPY_DATA
	attribute, only a copy of the element pointer will be added to the
	destination list. However there will not be an additional pointer to it,
	as the original element from the input list has been deleted. Element
	references are tied to the original element and list, and will not be
	valid after this move operation.
	
	The destination list must not be a Finite class list.
	
	This function removes all the elements from the input list, and releases
	their internal resources, along with the data resources if the input list
	was initialized with the LIST_COPY_DATA attribute.

	list		list_object that has been previously initialized.
	datum 		data to be used for the search criteria.
	dest_list	list_object that has been previously initialized, where the
				copied element is to be added.
	returns	>0					the number of elements found in the input
								list, that were added to the destination list.
			0					if no elements matched the datum
			ELEMENT_NO_DATUM	element data not supplied
			LIST_ERR			on error 											*/
int list_search_move(	list_object *restrict list,
						const void *datum,
						list_object *restrict dest_list) {

	int ret;

	ret = list__validate_evaluate_range(list, datum, NULL, FALSE, FALSE, dest_list, __func__);
	if (ret!= LIST_OK)
		return ret;
	
	return(list__leaf_node_evaluate_range(list, datum, 0, NULL, FALSE, TRUE, dest_list));
}

/* Search for, and copy, all elements that evaluate true against the supplied
	datum, from one list to another.
	
	This function will traverse the input list from head to tail, evaluating
	all elements against the matching datum supplied. For each element in the
	list that evaluates true, this function copies the element from the input
	list, and adds it to the destination list. The new element is either
	appended to a linear class destination list, or stored by pivot value to
	a pivot class list.

	This is a simple version of search / compare, the entire input list is
	traversed, no consideration of pivots, or direction are provided.
	
	A compare function must have been previously set for the input list, by
	calling list_set_search_function(). Each element of the input list is
	presented to that compare function, along with the user supplied datum.
	If the result of the evaluation is that the element is comparatively
	greater than or equal to the datum, and LIST_ASCENDING is supplied for
	the argument order, then the evaluation is considered TRUE and that
	element is then copied to the destination list. Conversely if the element
	is comparatively lesser than the datum, and LIST_DESCENDING is supplied
	for the argument order, then the evaluation is considered TRUE and that
	element is then copied to the destination list. Otherwise the element is
	ignored and the next element evaluated.
	
	If the destination list was initialized with a LIST_COPY_DATA attribute,
	the new element will copied in whole. If the list has no LIST_COPY_DATA
	attribute, only a copy of the element pointer will be added to the
	destination list. The user should remember there is now an additional
	pointer to it, that must be considered prior to any resource removal
	actions.

 	The destination list must not be a Finite class list.
	
	This function only copies the elements and makes no changes to the input
	list.

	list		list_object that has been previously initialized.
	datum 		data to be used for the search criteria.
	order		either LIST_ASCENDING or LIST_DESCENDING to denote which order
				to evaluate the elements in the list against the datum.
	dest_list	list_object that has been previously initialized, where the
				copied element is to be added.
	returns	>0					the number of elements found in the input
								list, that were added to the destination list.
			0					if no elements matched the datum
			ELEMENT_NO_DATUM	element data not supplied
			LIST_ERR			on error 											*/
int list_evaluate_copy(	list_object *restrict list,
						const void *datum,
						const boolean order,
						list_object *restrict dest_list) {

	int ret;

	ret = list__validate_evaluate_range(list, datum, NULL, TRUE, FALSE, dest_list, __func__);
	if (ret!= LIST_OK)
		return ret;

	return(list__leaf_node_evaluate_range(list, datum, order, NULL, TRUE, FALSE, dest_list));
}

/* Search for, and move, all elements that evaluate true against the supplied
	datum, from one list to another.
	
	This function will traverse the input list from head to tail, evaluating
	all elements against the matching datum supplied. For each element in the
	list that evaluates true, this function copies the element from the input
	list, and adds it to the destination list and then deletes it from the
	input list. The new element will be either appended to a linear class
	destination list, or stored by pivot value to a pivot class list.

	This is a simple version of search / compare, the entire input list is
	traversed, no consideration of pivots, or direction are provided.
	
	A compare function must have been previously set for the input list, by
	calling list_set_compare_function(). Each element of the input list is
	presented to that compare function, along with the user supplied datum.
	If the result of the evaluation is that the element is comparatively
	greater than or equal to the datum, and LIST_ASCENDING is supplied for
	the argument order, then the evaluation is considered TRUE and that
	element is then copied to the destination list. Conversely if the element
	is comparatively lesser than the datum, and LIST_DESCENDING is supplied
	for the argument order, then the evaluation is considered TRUE and that
	element is then copied to the destination list. Otherwise the element is
	ignored and the next element evaluated. All elements that evaluate TRUE
	will be removed from the input list.
	
	If the destination list was initialized with a LIST_COPY_DATA attribute,
	the new element will copied in whole. If the list has no LIST_COPY_DATA
	attribute, only a copy of the element pointer will be added to the
	destination list. However there will not be an additional pointer to it,
	as the original element from the input list has been deleted. Element
	references are tied to the original element and list, and will not be
	valid after this move operation.
	
	The destination list must not be a Finite class list.
	
	This function removes all the elements from the input list, and releases
	their internal resources, along with the data resources if the input list
	was initialized with the LIST_COPY_DATA attribute.

	list		list_object that has been previously initialized.
	datum 		data to be used for the search criteria.
	order		either LIST_ASCENDING or LIST_DESCENDING to denote which order
				to evaluate the elements in the list against the datum.
	dest_list	list_object that has been previously initialized, where the
				moved element is to be added.
	returns	>0					the number of elements removed from the input
								list, and were added to the destination list.
			0					if no elements matched the datum
			ELEMENT_NO_DATUM	element data not supplied
			LIST_ERR			on error 											*/
int list_evaluate_move(	list_object *restrict list,
						const void *datum,
						const boolean order,
						list_object *restrict dest_list) {

	int ret;	

	ret = list__validate_evaluate_range(list, datum, NULL, TRUE, FALSE, dest_list, __func__);
	if (ret!= LIST_OK)
		return ret;

	return(list__leaf_node_evaluate_range(list, datum, order, NULL, TRUE, TRUE, dest_list));
}

/* Search for, and copy, all elements that evaluate true within a minimum and
	maximum range of user datums, from one list to another.
	
	This function will traverse the input list from head to tail, evaluating
	all elements to check if they evaluate within a minimum and maximum range
	defined by comparing with the supplied datums. For each element in the
	list that evaluates within the datum range, this function copies the
	element from the input list, and adds it to the destination list. The new
	element is either appended to a linear class destination list, or stored
	by pivot value to a pivot class list.

	This is a simple version of search / compare, the entire input list is
	traversed, no consideration of pivots, or direction are provided.
	
	A compare function must have been previously set for the input list, by
	calling list_set_compare_function(). Each element of the input list is
	presented twice to that compare function, along with each of the user
	supplied datums.

	If the result of the evaluation is that the element is comparatively
	greater than or equal to the datum_min, and is also comparatively lesser
	than or equal the datum_max, then the evaluation is considered TRUE and
	that element is then copied to the destination list. Otherwise the element
	is ignored and the next element evaluated.
	
	If the destination list was initialized with a LIST_COPY_DATA attribute,
	the new element will copied in whole. If the list has no LIST_COPY_DATA
	attribute, only a copy of the element pointer will be added to the
	destination list. The user should remember there is now an additional
	pointer to it, that must be considered prior to any resource removal
	actions.

	The destination list must not be a Finite class list.
	
	This function only copies the elements and makes no changes to the input
	list.

	list		list_object that has been previously initialized.
	datum_min	data to be used for the minimum value of the search criteria.
	datum_max	data to be used for the maximum value of the search criteria.
	dest_list	list_object that has been previously initialized, where the
				copied element is to be added.
	returns	>0					the number of elements found in the input
								list, that were added to the destination list.
			0					if no elements matched the datum
			ELEMENT_NO_DATUM	element data not supplied
			LIST_ERR			on error 											*/
int list_range_copy(	list_object *restrict list,
						const void *datum_min,
						const void *datum_max,
						list_object *restrict dest_list) {

	int ret;

	ret = list__validate_evaluate_range(list, datum_min, datum_max, TRUE, TRUE, dest_list, __func__);
	if (ret!= LIST_OK)
		return ret;

	return(list__leaf_node_evaluate_range(list, datum_min, LIST_ASCENDING, datum_max, TRUE, FALSE, dest_list));
}

/* Search for, and move, all elements that evaluate true within a minimum and
	maximum range of user datums, from one list to another.
	
	This function will traverse the input list from head to tail, evaluating
	all elements to check if they evaluate within a minimum and maximum range
	defined by comparing with the supplied datums. For each element in the
	list that evaluates within the datum range, this function removes the
	element from the input list, and adds it to the destination list. The new
	element is either appended to a linear class destination list, or stored
	by pivot value to a pivot class list.

	This is a simple version of search / compare, the entire input list is
	traversed, no consideration of pivots, or direction are provided.
	
	A compare function must have been previously set for the input list, by
	calling list_set_compare_function(). Each element of the input list is
	presented twice to that compare function, along with each of the user
	supplied datums.

	If the result of the evaluation is that the element is comparatively
	greater than or equal to the datum_min, and is also comparatively lesser
	than or equal the datum_max, then the evaluation is considered TRUE and
	that element is then moved to the destination list. Otherwise the element
	is ignored and the next element evaluated.
	
	If the destination list was initialized with a LIST_COPY_DATA attribute,
	the new element will copied in whole. If the list has no LIST_COPY_DATA
	attribute, only a copy of the element pointer will be added to the
	destination list. However there will not be an additional pointer to it,
	as the original element from the input list has been deleted. Element
	references are tied to the original element and list, and will not be
	valid after this move operation.
	
	The destination list must not be a Finite class list.
	
	This function removes the element from the input list, and releases its
	internal resources, along with the data resources if the input list was
	initialized with the LIST_COPY_DATA attribute.


	list		list_object that has been previously initialized.
	datum_min	data to be used for the minimum value of the search criteria.
	datum_max	data to be used for the maximum value of the search criteria.
	dest_list	list_object that has been previously initialized, where the
				moved element is to be added.
	returns	>0					the number of elements found in the input
								list, that were moved to the destination list.
			0					if no elements matched the datum
			ELEMENT_NO_DATUM	element data not supplied
			LIST_ERR			on error 											*/
int list_range_move(	list_object *restrict list,
						const void *datum_min,
						const void *datum_max,
						list_object *restrict dest_list) {

	int ret;

	ret = list__validate_evaluate_range(list, datum_min, datum_max, TRUE, TRUE, dest_list, __func__);
	if (ret!= LIST_OK)
		return ret;

	return(list__leaf_node_evaluate_range(list, datum_min, LIST_ASCENDING, datum_max, TRUE, TRUE, dest_list));
}

/* Retrieve an element from a list, given the index position.
	
	This function only returns a pointer to the element in the list,
	it makes no changes to the list.

	list		list_object that has been previously initialized.
	index		denotes where in the list the element is to be retrieved from.
	returns	NULL			no element was available in list
			PTR				to element data. 										*/
void *list_get_index(	const list_object *restrict list,
						const unsigned int index) {

	leaf_node *ptr;

#ifdef DEBUG
	if (list__validate_list_object(list, FALSE, __func__) == FALSE)
		return NULL;
	if ((list->element_count) == 0) {
		fprintf(stderr, "\n==> Dlist Debug WARNING ==> %s() - "\
				"called with empty input list.\n\n", __func__);
		return NULL;
	}
	if (index >= list->element_count) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with out-of-range index. Index '%u', max element index is '%u'.\n\n",
				__func__, index, list->element_count-1);
		return NULL;
	}
#else	
	if ((list == NULL)
			|| (list->element_count < 1)
			|| (index >= list->element_count))
		return NULL;
#endif

	ptr = list__locate_leaf_node(list, (long int)index);

	return ((ptr) ? ptr->element : NULL);
}

/* Retrieve an element from a list, given the index position, and also
	return the element reference.
	
	This function retrieves an element from the list, using the supplied
	index, and is functionally identical to list_get_index(), except that the
	element reference is returned as well.

	The element reference is an abstract identifier for an element in a list,
	that does not change if the list is modified. An index position is only
	valid until the list changes, as an elements relative position will be
	changed by insertions or deletions. The element reference is not related
	to position in a list.

	The reference remains valid until the element itself is removed from the
	list. Reference is a required argument, and will be modified by this
	function.
	
	This function returns a pointer to the element in the list, along with its
	element reference. It makes no changes to the list.

	list		list_object that has been previously initialized.
	index		denotes where in the list the element is to be retrieved from.
	reference 	the abstract locator of the element in the list, which is 
				provided in this argument when the element is located. If the
				element cannot be found, NULL is returned.
	returns	NULL			no element was available in list
			PTR				to element data. 										*/
void *list_get_index_ref(	const list_object *restrict list,
							const unsigned int index,
							element_reference **restrict reference) {

	leaf_node *ptr;

#ifdef DEBUG
	if (list__validate_list_object(list, FALSE, __func__) == FALSE)
		return NULL;
	if ((list->element_count) == 0) {
		fprintf(stderr, "\n==> Dlist Debug WARNING ==> %s() - "\
				"called with empty input list.\n\n", __func__);
		return NULL;
	}
	if (index >= list->element_count) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with out-of-range index. Index '%u', max element index is '%u'.\n\n",
				__func__, index, list->element_count-1);
		return NULL;
	}
	if (reference == NULL) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with NULL reference argument.\n\n", __func__);
		return NULL;
	}
#else	
	if ((list == NULL)
			|| (list->element_count < 1)
			|| (index >= list->element_count))
		return NULL;
	if (reference == NULL) 
		return NULL;
#endif

	*reference = NULL;								/* redundant but clean */
	ptr = list__locate_leaf_node(list, (long int)index);
	*reference = ptr;

	return ((ptr) ? ptr->element : NULL);
}

/* Retrieve an element from a list, given the element reference.
	
	This function retrieves an element from the list, using the supplied
	element reference. It is functionally similar to list_get_index(), except
	that the element is identified by reference, and not index. A reference
	must have been passed to the caller previously by dlist from another call,
	such as list_search().
	
	This function is not available for finite class lists.

	This function only returns a pointer to the element in the list,
	it makes no changes to the list.

	list		list_object that has been previously initialized.
	reference	an abstract locator in list, of the element to be retrieved.
	returns	NULL			no element was available in list (an error)
			PTR				to element data. 										*/
void *list_get_ref(	const list_object *restrict list,
					const element_reference *restrict reference) {

#ifdef DEBUG
	int ret;
	if (list__validate_list_object(list, FALSE, __func__) == FALSE)
		return NULL;
	if (reference == NULL) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with NULL reference argument.\n\n", __func__);
		return NULL;
	}
	ret = list__validate_reference(list, reference, __func__);
	if (ret != LIST_OK) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with invalid reference pointer, error code %d.\n\n", __func__, ret);
		return NULL;
	}
	if (list->list_finite_enabled == TRUE) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called on a finite class list.\n\n", __func__);
		return NULL;
	}
#else
	if ((list == NULL)
			|| (reference == NULL)
			|| (list->list_finite_enabled == TRUE))
		return NULL;
	if (list__validate_reference(list, reference, __func__) != LIST_OK)
		return NULL;
#endif

	return reference->element;
}

/* Retrieve the next element from a list, given the element reference.
	
	This function retrieves an element from the list, using the supplied
	element reference. It is functionally similar to list_get_index(), except
	that the element is identified by reference, and not index. A reference
	must have been passed to the caller previously by D-list from another call,
	such as list_search() or list_get_next_ref(). Or a reference of NULL can
	be provided to start at the end of the list and traverse in the direction
	specified. If the list is a pivot class list, the state of the output
	argument pivot_point is set to TRUE if this new record comes from the
	other side of the the conceptual pivot point (as enforced by the set
	pivot function for the list). Otherwise it is set to FALSE on return.

	Upon exit the reference, and the pivot_point arguments are always updated,
	and the original contents replaced.
	
	This function can also be used in a similar way to list_iteration_next(),
	and is vastly more efficient than performing an index loop using
	list_get_index() which is very inefficient when dealing with large lists.
	Unlike an iteration sequence, it does not lock the list from modifications,
	and allows you to traverse portions of the list starting at an arbitrary
	point, and allows you to change direction at will.
	
	This function is not available for finite class lists.
	
	This function only returns a pointer to the element in the list,
	it makes no changes to the list.

	list		list_object that has been previously initialized.
	reference 	an abstract locator in list, of the element next to the
		 		element to be retrieved. NULL means start from the head or
		 		tail of the list, depending on direction
	direction	HEAD_OF_LIST towards the head of the list, or TAIL_OF_LIST
				the opposite direction.
	returns	NULL			no element was available in list, end of list,
							or error. reference will also be set to NULL
			PTR				to element data. Reference will be updated to
							this element
			pivot_point		if this next record crosses the pivot point,
							this return argument is set to TRUE.					*/
void *list_get_next_ref(	const list_object *restrict list,
							element_reference **restrict reference,
							const boolean direction,
							boolean *restrict pivot_point) {

	leaf_node *ptr, *cur_ptr;
	
#ifdef DEBUG
	if (list__validate_list_object(list, FALSE, __func__) == FALSE)
		return NULL;
	if (reference == NULL) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with NULL reference argument. The reference variable pointer can be NULL,"\
				"but the argument must still have a valid pointer to the reference variable.\n\n", __func__);
		return NULL;
	}
	if (pivot_point == NULL) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with NULL pivot_point argument.\n\n", __func__);
		return NULL;
	}
	if (list->list_finite_enabled == TRUE) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called on a finite class list.\n\n", __func__);
		return NULL;
	}
#else
	if ((list == NULL)
			|| (reference == NULL)
			|| (pivot_point == NULL)
			|| (list->list_finite_enabled == TRUE))
		return NULL;
#endif
	

	cur_ptr = NULL;
	ptr = *reference;

#ifdef DEBUG
	int ret;
	if (ptr) {
		ret = list__validate_reference(list, ptr, __func__);
		if (ret != LIST_OK) {
			fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with invalid reference pointer, error code %d.\n\n", __func__, ret);
			return NULL;
		}
	}
#else
	if ((ptr) && (list__validate_reference(list, ptr, __func__) != LIST_OK))
		return NULL;
#endif

	*pivot_point = FALSE;
	if (ptr == NULL) {
		if (direction == TAIL_OF_LIST) ptr = list->head_guard->next_leaf;
		else ptr = list->tail_guard->prior_leaf;	
	} else {	
		cur_ptr = ptr;
		if (direction == TAIL_OF_LIST) ptr = ptr->next_leaf;
		else ptr = ptr->prior_leaf;
		if (cur_ptr->internal->pivot.value != ptr->internal->pivot.value) *pivot_point = TRUE;
	}
	
	if ((ptr == list->head_guard) || (ptr == list->tail_guard)) *reference = NULL;
	else *reference = ptr;

	return ((*reference == NULL) ? NULL : ptr->element);
}

/* Extract an element from the front (head), or the tail (end), of a list.

	Among its many uses, this function is useful for managing LIFO lists,
	heaps, and stacks, and FIFO's and queues. It also allows elements to
	be removed from a list being managed by index, without disturbing the
	index optimizations that making modifications by reference causes.
	
	This function removes the element from the list, and releases its
	internal resources.

	list		list_object that has been previously initialized.
	where		either HEAD_OF_LIST or TAIL_OF_LIST, denotes where in the
		 		list the element is to be extracted from.
	returns	NULL	no elements available in list
			PTR		to element data. It is the users responsibility
					to manage this data, and free it when finished with
					it. If the user supplied it originally it is the
					same allocated memory originally given to D-list.
					If the list was created with the LIST_COPY_DATA
					attribute, the data was allocated internally by
					D-list. In both cases it now belongs to the caller.				*/
void *list_fetch(	list_object *restrict list,
					const boolean where) {

	leaf_node *ptr;
	void *element;
	
#ifdef DEBUG
	if (list__validate_list_object(list, LIST_WRITE, __func__) == FALSE) return NULL;
	if (list->iteration_enabled) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"list has iteration enabled.\n\n", __func__);
		return NULL;
	}
#else
	if ((list == NULL) || (list->iteration_enabled))
		return NULL;
#endif

	if (list->element_count == 0) return NULL;
	
	if (where == HEAD_OF_LIST) ptr = list->head_guard->next_leaf;
	else ptr = list->tail_guard->prior_leaf;
	
	element = ptr->element;
	if (list->functions.remove) list->functions.remove(ptr->element);
	ptr->element = NULL;					/* prevent underlying primitives from freeing the resource */

	list__remove_leaf_node(ptr, ((where == HEAD_OF_LIST) ? HEAD_OF_LIST : list->element_count-1));

#ifdef DLIST_DEBUG
	assert(list__audit(list) == LIST_OK);
#endif		/* DLIST_DEBUG */
	
	return element;
}

/* Extract an element from the pivot point, of a pivot class list.

	This function is very similar to list_fetch() except that instead
	of extracting the element from the end of any class of list,
	list_fetch_pivot() will only extract an element from the pivot
	point of a pivot list. The value of the argument where, dictates
	Which side of the pivot point the element is removed from.
	
	This function is only valid for pivot class lists.
	
	Among its many uses, this function is useful for managing complex
	double LIFO lists, heaps, and stacks, and FIFO's and folding queues.
	It also allows elements to be removed from a list being managed by
	index, without disturbing the index optimizations that making
	modifications by reference causes. This function is also helpful in
	reading the results of a list_assay() call.
	
	This function removes the element from the list, and releases its
	internal resources.

	list		list_object that has been previously initialized.
	where		either PIVOT_MINOR or PIVOT_MAJOR denotes which side of
				the pivot point, of the list, the element is to be
				extracted from.
	returns	NULL	no elements available in list on the pivot side
					requested
			PTR		to element data. It is the users responsibility
					to manage this data, and free it when finished with
					it. If the user supplied it originally it is the
					same allocated memory originally given to D-list.
					If the list was created with the LIST_COPY_DATA
					attribute, the data was allocated internally by
					D-list. In both cases it now belongs to the caller.				*/
void *list_fetch_pivot(	list_object *restrict list,
						const int where) {

	leaf_node *ptr;
	void *element;
	long int index = 0;
	
#ifdef DEBUG
	if (list__validate_list_object(list, LIST_WRITE, __func__) == FALSE) return NULL;
	if (list->iteration_enabled) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"list has iteration enabled.\n\n", __func__);
		return NULL;
	}
	if (list->list_pivot_enabled == FALSE) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called on a non pivot class list.\n\n", __func__);
		return NULL;
	}
	if ((where != PIVOT_MINOR) && (where != PIVOT_MAJOR)) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with invalid 'where' argument, must be either PIVOT_MINOR (%d) or "\
				"PIVOT_MAJOR (%d), but is %d.\n\n", __func__, PIVOT_MINOR, PIVOT_MAJOR, where);
		return NULL;
	}
#else
	if ((list == NULL)
			|| (list->list_pivot_enabled == FALSE)
			|| (list->iteration_enabled)
			|| ((where != PIVOT_MINOR) && (where != PIVOT_MAJOR)))
		return NULL;
#endif

	if (list->element_count == 0) return NULL;

	if (where == PIVOT_MINOR) {
		if (list->pivot_point.minor == 0) return NULL;
		ptr = list__locate_pivot_leaf_node(list, &index);
	} else {
		if (list->pivot_point.major == 0) return NULL;
		ptr = list__locate_pivot_leaf_node(list, &index);
		++index;
		ptr = ptr->next_leaf;
	}
	
	element = ptr->element;
	if (list->functions.remove) list->functions.remove(ptr->element);
	ptr->element = NULL;					/* prevent underlying primitives from freeing the resource */

	list__remove_leaf_node(ptr, index-1);

#ifdef DLIST_DEBUG
	assert(list__audit(list) == LIST_OK);
#endif		/* DLIST_DEBUG */
	
	return element;
}

/* Extract an element from a list, given the index position.
	
	This function removes the element from the list, and releases its
	internal resources.

	list		list_object that has been previously initialized.
	index		denotes where in the list the element is to be extracted from.
	returns		NULL no element was available in list
				PTR to element data. It is the users responsibility to manage
					this data, and free it when finished with it. If the user
					supplied it originally it is the same allocated memory
					originally given to D-List. If the list was created with
					the LIST_COPY_DATA attribute, the data was allocated
					internally by D-List. In both cases it now belongs to the
					caller.															*/
void *list_extract_index(	list_object *restrict list,
							const unsigned int index) {

	leaf_node *ptr;
	void *element;
	
#ifdef DEBUG
	if (list__validate_list_object(list, LIST_WRITE, __func__) == FALSE)
		return NULL;
	if (list->element_count < 1) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with empty input list.\n\n", __func__);
		return NULL;
	}
	if (list->iteration_enabled) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"list has iteration enabled.\n\n", __func__);
		return NULL;
	}
	if (index >= list->element_count) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with out-of-range index. Index '%u', max element index is '%u'.\n\n",
				__func__, index, list->element_count-1);
		return NULL;
	}
#else	
	if ((list == NULL)
			|| (list->element_count < 1)
			|| (list->iteration_enabled)
			|| (index >= list->element_count))
		return NULL;
#endif

	ptr = list__locate_leaf_node(list, (long int)index);
	element = ptr->element;
	if (list->functions.remove) list->functions.remove(ptr->element);
	ptr->element = NULL;					/* prevent underlying primitives from freeing the resource */

	list__remove_leaf_node(ptr, index);

#ifdef DLIST_DEBUG
	assert(list__audit(list) == LIST_OK);
#endif		/* DLIST_DEBUG */
	
	return element;
}

/* Extract an element from a list, given the element reference.
	
	This function removes the element from the list, and releases its internal
	resources. It is functionally similar to list_extract_index(), except that
	the element is identified by reference, not by index. A reference must have
	been passed to the caller previously by D-List from another call, such as
	list_search().
	
	This function is only available for finite class lists during an active
	iteration sequence, otherwise calling this function on a finite class list
	will return an error code. If used the caller must only use references that
	were provided by list_iteration_get_next() during this active iteration
	sequence.

	list		list_object that has been previously initialized.
	reference	an abstract locator in list, of the element to be removed.
	returns		NULL the element reference was not valid, list unchanged
				PTR to element data. It is the users responsibility to manage
					this data, and free it when finished with it. If the user
					supplied it originally it is the same allocated memory
					originally given to D-List. If the list was created with
					the LIST_COPY_DATA attribute, the data was allocated
					internally by D-List. In both cases it now belongs to the
					caller.															*/
void *list_extract_ref(	list_object *restrict list,
						element_reference **restrict reference) {

	leaf_node *ptr;
	void *element;

	if (reference == NULL) {
#ifdef DEBUG
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with NULL reference argument.\n\n", __func__);
#endif
		return NULL;
	}
	
	ptr = *reference;

#ifdef DEBUG
	int ret;
	if (list__validate_list_object(list, LIST_WRITE, __func__) == FALSE)
		return NULL;
	if ((list->list_finite_enabled == TRUE) && (! list->iteration_enabled)) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called on a finite class list with no active iteration enabled.\n\n", __func__);
		return NULL;
	}
	if (ptr == NULL) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with NULL reference pointer.\n\n", __func__);
		return NULL;
	}
	ret = list__validate_reference(list, ptr, __func__);
	if (ret != LIST_OK) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with invalid reference pointer, error code %d.\n\n", __func__, ret);
		return NULL;
	}
#else
	if ((list == NULL) || (ptr == NULL))
		return NULL;
	if ((list->list_finite_enabled == TRUE) && (! list->iteration_enabled))
		return NULL;
	if (list__validate_reference(list, ptr, __func__) != LIST_OK)
		return NULL;
#endif
	
	/* using the reference based updates destroys the integrity of the
		optimized index */
	list->mid_point = NULL;
	element = ptr->element;
	if (list->functions.remove) list->functions.remove(ptr->element);
	ptr->element = NULL;					/* prevent underlying primitives from freeing the resource */
	list__remove_leaf_node(ptr, 0);
	
	*reference = NULL;

#ifdef DLIST_DEBUG
	assert(list__audit(list) == LIST_OK);
#endif		/* DLIST_DEBUG */

	return element;
}

/* Delete an element from a list, given the index position.
	
	This function removes the element from the list, and releases its
	internal resources. It is functionally the same to list_extract_index(),
	except that no element is passed back to the caller, and if the list was
	created with the LIST_COPY_DATA attribute, the data is released as well.

	list		list_object that has been previously initialized.
	index		denotes the index number in the list, where the element
				is to be removed.
	returns	LIST_OK				the element was delete as requested
			ELEMENT_NOT_FOUND	no element was found matching the index.
			LIST_ERR			if there was an error.								*/
int list_delete_index(	list_object *restrict list,
						const unsigned int index) {

	leaf_node *ptr;
	
#ifdef DEBUG
	if (list__validate_list_object(list, LIST_WRITE, __func__) == FALSE)
		return LIST_ERR;
	if (list->element_count < 1) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with empty input list.\n\n", __func__);
		return LIST_ERR;
	}
	if (list->iteration_enabled) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"list has iteration enabled.\n\n", __func__);
		return LIST_ERR;
	}
	if (index >= (unsigned int)list->element_count) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with out-of-range index. Index '%u', current max element index "\
				"is '%u'.\n\n", __func__, index, list->element_count-1);
		return ELEMENT_NOT_FOUND;
	}
#else	
	if ((list == NULL)
			|| (list->element_count < 1)
			|| (list->iteration_enabled))
		return LIST_ERR;
	if (index >= list->element_count)
		return ELEMENT_NOT_FOUND;
#endif
	
	ptr = list__locate_leaf_node(list, (long int)index);
	if (list->functions.remove) list->functions.remove(ptr->element);

	list__remove_leaf_node(ptr, index);

#ifdef DLIST_DEBUG
	assert(list__audit(list) == LIST_OK);
#endif		/* DLIST_DEBUG */
	
	return LIST_OK;
}

/* Delete a block of elements from a list, between the indicated index positions.
	
	This function removes the elements in the index range from the list,
	and releases their resources. It is functionally the same to
	list_delete_index(), except that more than one element is deleted.

	list		list_object that has been previously initialized.
	index_start	denotes the first element to be removed.
	index_stop	denotes the last element to be removed, including all
				elements in-between.
	returns	LIST_OK				the element was delete as requested
			ELEMENT_NOT_FOUND	no element was found matching the index.
			LIST_ERR			if there was an error.								*/
int list_delete_group(	list_object *restrict list,
						const unsigned int index_start,
						const unsigned int index_stop) {

	leaf_node *ptr, *next_ptr;
	unsigned int i, range;

#ifdef DEBUG
	if (list__validate_list_object(list, LIST_WRITE, __func__) == FALSE)
		return LIST_ERR;
	if (list->element_count < 1) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with empty input list.\n\n", __func__);
		return LIST_ERR;
	}
	if (list->iteration_enabled) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"list has iteration enabled.\n\n", __func__);
		return LIST_ERR;
	}
	if (index_stop >= list->element_count) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with out-of-range index_stop. Index '%u', max element index is '%u'.\n\n",
				__func__, index_stop, list->element_count-1);
		return ELEMENT_NOT_FOUND;
	}
	if (index_start >= index_stop) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with invalid index range index_start > index_stop. index_start '%u' "\
				"index_stop '%u', current max element index is '%u'.\n\n",
				__func__, index_start, index_stop, list->element_count-1);
		return ELEMENT_NOT_FOUND;
	}
#else	
	if ((list == NULL)
			|| (list->element_count < 1)
			|| (list->iteration_enabled))
		return LIST_ERR;
	if ((index_start >= index_stop) || (index_stop >= list->element_count))
		return ELEMENT_NOT_FOUND;
#endif
	
	range = index_stop - index_start;
	ptr = list__locate_leaf_node(list, (long int)index_start);
	if (ptr == NULL) {
#ifdef DEBUG
		fprintf(stderr, "\n==> Dlist INTERNAL ERROR ==> %s() failed - null leaf node pointer "\
				"returned for element index '%u' list elements '%u'.\n\n", __func__, index_start,
				list->element_count);
#endif
		return LIST_ERR;
	}
	
	for (i = 0; (i <= range) && (ptr->next_leaf != NULL); ++i, ptr = next_ptr) {
		next_ptr = ptr->next_leaf;
		if (list->functions.remove) list->functions.remove(ptr->element);
		list__remove_leaf_node(ptr, index_start);
	}

#ifdef DLIST_DEBUG
	assert(list__audit(list) == LIST_OK);
#endif		/* DLIST_DEBUG */
	
	return LIST_OK;
}

/* Delete an element from a list, given the element reference.
	
	This function removes the element from the list, and releases its internal
	resources. It is functionally similar to list_delete_index(), except that
	the element is identified by reference, not by index. A reference must have
	been passed to the caller previously by dlist from another call, such as
	list_search().
	
	This function is only available for finite class lists during an active
	iteration sequence, otherwise calling this function on a finite class list
	will return an error code. If used the caller must only use references that
	were provided by list_iteration_get_next() during this active iteration
	sequence.

	list		list_object that has been previously initialized.
	reference	an abstract locator in list, of the element to be removed.
	returns	reference		on success, reference is set to NULL
			LIST_OK			the element was delete as requested,
			ELEMENT_INVALID	an invalid element reference was provided
			LIST_ERR		if there was an error.									*/
int list_delete_ref(	list_object *restrict list,
						element_reference **restrict reference) {

	leaf_node *ptr;

	if (reference == NULL) {
#ifdef DEBUG
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with NULL reference argument.\n\n", __func__);
#endif
		return LIST_ERR;
	}
	
	ptr = *reference;

#ifdef DEBUG
	int ret;
	if (list__validate_list_object(list, LIST_WRITE, __func__) == FALSE)
		return LIST_ERR;
	if ((list->list_finite_enabled == TRUE) && (! list->iteration_enabled)) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called on a finite class list with no active iteration enabled.\n\n", __func__);
		return LIST_ERR;
	}
	if (ptr == NULL) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with NULL reference pointer.\n\n", __func__);
		return LIST_ERR;
	}
	ret = list__validate_reference(list, ptr, __func__);
	if (ret != LIST_OK) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with invalid reference pointer, error code %d.\n\n", __func__, ret);
		return ret;
	}
#else
	if ((list == NULL) || (ptr == NULL))
		return LIST_ERR;
	if ((list->list_finite_enabled == TRUE) && (! list->iteration_enabled))
		return LIST_ERR;
	if (list__validate_reference(list, ptr, __func__) != LIST_OK)
		return ELEMENT_INVALID;
#endif
	
	/* using the reference based updates destroys the integrity of the
		optimized index */
	list->mid_point = NULL;
	if (list->functions.remove) list->functions.remove(ptr->element);
	list__remove_leaf_node(ptr, 0);
	
	*reference = NULL;

#ifdef DLIST_DEBUG
	assert(list__audit(list) == LIST_OK);
#endif		/* DLIST_DEBUG */

	return LIST_OK;
}

/* Delete all elements from a list, that are duplicates of other elements in the list.
	
	This function will traverse a list from head to tail, and for each element
	found will search the rest of the list looking for and removing all duplicate
	elements that are found. This function does not change the order of any of
	the elements in the list. It does not perform any sorting of the elements.
	
	It will return a count of the elements removed during this process.

	If a search function was previously set for this list (by calling
	list_set_search_function() ), this function will use that search function
	to compare the element data against each other.
	
	If a search function was not previously set for this list, or it was
	removed, this function will perform a straight compare between the data
	pointer in the elements examined. If the element pointers are the same,
	they are equal, if the pointers point to different element data areas,
	they are not equal.
	
	In most cases, the user will want to set a search function first. But this
	can be used to eliminate duplicate data elements by pointer reference, very
	quickly.
	
	The definitive test, on return, to verify if any elements were removed, is 
	to check the return count for either 0 or a non-zero number.
	
	This function makes extensive changes to the list.

	list		list_object that has been previously initialized.
	returns	0				no duplicate elements were found
			>0				the number of duplicate elements removed,
			LIST_ERR		if there was an error.									*/
int list_delete_duplicates(list_object *restrict list) {

	leaf_node *ptr, *fwd_ptr, *next_ptr;
	unsigned int count = 0;
	unsigned int index, fwd_index;
	boolean remove_element;
	boolean (*element_search)(const void *element, const void *key);

#ifdef DEBUG
	if (list__validate_list_object(list, FALSE, __func__) == FALSE)
		return LIST_ERR;
	if (list->iteration_enabled) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"list has iteration enabled.\n\n", __func__);
		return LIST_ERR;
	}
	if (list->functions.search == NULL) {
		fprintf(stderr, "\n==> Dlist Debug WARNING ==> %s() - "\
				"NULL search function, removing duplicate elements by pointer match only.\n\n", __func__);
	}
#else
	if ((list == NULL) || (list->iteration_enabled))
		return LIST_ERR;
#endif

	if (list->element_count < 2) return 0;

	element_search = list->functions.search;

	/* this has the same general performance characteristics of a selection sort,
		except the performance improves as more elements are deleted.				*/
	for (index = 0, ptr = list->head_guard->next_leaf; (ptr) && (ptr != list->tail_guard); ++index, ptr = ptr->next_leaf) {
		for (fwd_index = index + 1, remove_element = FALSE, fwd_ptr = ptr->next_leaf;
				(fwd_ptr) && (fwd_ptr != list->tail_guard);
				++fwd_index, remove_element = FALSE, fwd_ptr = next_ptr) {
			if (element_search) {
				if ((*element_search)(fwd_ptr->element, ptr->element)) remove_element = TRUE;
			} else {
				if (fwd_ptr->element == ptr->element) remove_element = TRUE;
			}
			next_ptr = fwd_ptr->next_leaf;
			if (remove_element) {
					if (list->functions.remove) list->functions.remove(fwd_ptr->element);
					list__remove_leaf_node(fwd_ptr, fwd_index);
					--fwd_index;
					++count;
			}
		}
	}
	
	if (ptr == NULL) return LIST_ERR;							/* major oops ! */

	if (count > INT_MAX) count = INT_MAX;
		
	return (int)count;
}

/* Shift an element in a list, identified by its index, and relocate the
	element to the head, or tail, of the list.

	This function will relocate the element from its current position in the
	list, to either the head or tail of the list, based on the arguments
	provided. It is functionally similar to list_shift_ref(), except that
	the element is identified by index, not by reference.
	
	This function is not available for pivot class lists. This function is
	useful if used with a finite class list, that is being used as a recent
	history cache, or a linear list which the user wants to keep partially
	sorted without the overhead of constant sort operations. This function
	performs a sequence of operations that are equivalent to list_delete_index()
	and list_append() or list_prepend(), but far more efficient for this more
	specific task, and without destroying the reference to the original data
	element.

	If the list is currently sorted, the sorted indexes will be voided, and the
	list_sorted_xx() functions are no longer available for the list, until it
	is resorted.
		
	This function makes changes to the list.

	list		list_object that has been previously initialized.
	index 		denotes which element in the list, is to be extracted and
				shifted to a new position.
	where 		either HEAD_OF_LIST or TAIL_OF_LIST, denotes where in
				the list to shift the element to.
	returns	LIST_OK				the element was relocated as requested
			ELEMENT_NOT_FOUND	supplied index was not a valid in the list
			LIST_ERR			if there was an error.								*/
int list_shift_index(	list_object *restrict list,
						const unsigned int index,
						const boolean where) {

#ifdef DEBUG
	if (list__validate_list_object(list, LIST_WRITE, __func__) == FALSE)
		return LIST_ERR;
	if (list->element_count < 1) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with empty input list.\n\n", __func__);
		return ELEMENT_NOT_FOUND;
	}
	if (list->iteration_enabled) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"list has iteration enabled.\n\n", __func__);
		return LIST_ERR;
	}
	if (index >= list->element_count) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with out-of-range index. Index '%u', current max element index "\
				"is '%u'.\n\n", __func__, index, list->element_count-1);
		return ELEMENT_NOT_FOUND;
	}
	if (list->list_pivot_enabled == TRUE) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called on a pivot class list.\n\n", __func__);
		return LIST_ERR;
	}
#else
	if ((list == NULL)
			|| (list->iteration_enabled)
			|| (list->list_pivot_enabled == TRUE))
		return LIST_ERR;
	if ((index >= list->element_count) || (list->element_count < 1))
		return ELEMENT_NOT_FOUND;
#endif
	list->list_sorted = FALSE;

	return (list__shift_leaf_node(list, index, where));
}

/* Shift an element in a list, identified by its reference, and relocate the
	element to the head, or tail, of the list.

	This function will relocate the element from its current position in the
	list, to either the head or tail of the list, based on the arguments
	provided. It is functionally similar to list_shift_index(), except that
	the element is identified by reference, not by index. A reference must have
	been passed to the caller previously by dlist from another call, such as
	list_search() or list_locate_ref().
	
	This function is not available for pivot class lists. This function is
	useful if used with a finite class list, that is being used as a recent
	history cache, or a linear list which the user wants to keep partially
	sorted without the overhead of constant sort operations. This function
	performs a sequence of operations that are equivalent to list_delete_ref()
	and list_append() or list_prepend(), but far more efficient for this more
	specific task, and without destroying the reference to the original data
	element.

	If the list is currently sorted, the sorted indexes will be voided, and the
	list_sorted_xx() functions are no longer available for the list, until it
	is resorted.
		
	This function makes changes to the list.

	list		list_object that has been previously initialized.
	reference 	an abstract locator in list, of the element to be removed
				and shifted to a new location.
	where 		either HEAD_OF_LIST or TAIL_OF_LIST, denotes where in the
				list to shift the element to.
	returns	LIST_OK			the element was relocated as requested
			ELEMENT_INVALID	an invalid element reference was provided
			LIST_ERR		if there was an error.									*/
int list_shift_ref(	list_object *restrict list,
					element_reference *restrict reference,
					const boolean where) {

#ifdef DEBUG
	int ret;
	if (list__validate_list_object(list, LIST_WRITE, __func__) == FALSE)
		return LIST_ERR;
	if (list->element_count < 1) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with empty input list.\n\n", __func__);
		return LIST_ERR;
	}
	if (reference == NULL) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with NULL reference argument.\n\n", __func__);
		return LIST_ERR;
	}
	if (list->list_pivot_enabled == TRUE) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called on a pivot class list.\n\n", __func__);
		return LIST_ERR;
	}
	ret = list__validate_reference(list, reference, __func__);
	if (ret != LIST_OK) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with invalid reference pointer, error code %d.\n\n", __func__, ret);
		return ret;
	}
#else
	if ((list == NULL)
			|| (reference == NULL)
			|| (list->list_pivot_enabled == TRUE)
			|| (list->element_count < 1))
		return LIST_ERR;
	if (list__validate_reference(list, reference, __func__) != LIST_OK)
		return ELEMENT_INVALID;
#endif

	/* using the reference based updates destroys the integrity of the
		optimized index */
	list->mid_point = NULL;
	list->list_sorted = FALSE;
	
	/* move the element, (we do not use the normal primitives as we want
		to retain all list data including the original leaf node and its reference) */
	unlink__leaf_node(reference);
	if (where == HEAD_OF_LIST) link__leaf_node(reference, list->head_guard);
	else link__leaf_node(reference, list->tail_guard->prior_leaf);

#ifdef DLIST_DEBUG
	assert(list__audit(list) == LIST_OK);
#endif		/* DLIST_DEBUG */
	
	return LIST_OK;
}

/* Locate and return an element index in a list, identified by its element
	datum pointer.

	This function will locate the element in this list by searching for a
	match using the datum pointer.
	
	This function ignores any set search function., this function will only
	perform a straight compare between the data pointer in the elements and
	the pointer provided in datum. If the element pointers are the same,
	they are equal, if the pointers point to different element data areas,
	they are not equal.

	If you wish to perform a real data search of the list, please use one of
	the many list_search_xx() function calls. This function is only to get the
	index of an element for which you already know the data pointer, such as
	was returned from a list_simple_search(), or retained before the element
	was added to the list originally.

	This function returns an index and makes no changes to the list.

	list		list_object that has been previously initialized.
	datum 		data pointer to be used for the pointer search.
	returns	>=0				element index for the element datum requested
			LIST_ERR			if there was an error or
			ELEMENT_NO_DATUM	element data not supplied
			ELEMENT_NOT_FOUND	no match found.										*/
long int list_locate_index(	const list_object *restrict list,
							const void *datum) {

	leaf_node *ptr;
	long int index;

#ifdef DEBUG
	if (list__validate_list_object(list, FALSE, __func__) == FALSE)
		return LIST_ERR;
	if (datum == NULL) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with NULL datum pointer.\n\n", __func__);
		return ELEMENT_NO_DATUM;
	}
#else
	if (list == NULL)
		return LIST_ERR;
	if (datum == NULL)
		return ELEMENT_NO_DATUM;
#endif

	if (list->element_count < 1) return ELEMENT_NOT_FOUND;

	for (ptr = list->head_guard->next_leaf, index = 0;
			(ptr) && (ptr != list->tail_guard) && (ptr->element != datum);
			ptr = ptr->next_leaf, index++);
	
	if (ptr == NULL) return LIST_ERR;						/* major oops ! */
	
	if (ptr == list->tail_guard) return ELEMENT_NOT_FOUND;	/* No match found */
		
	return index;
}

/* Locate and return an element pivot value in a list, identified by its element
	datum pointer.

	This function will locate the element in this list by searching for a
	match using the datum pointer.
	
	This function ignores any set search function., this function will only
	perform a straight compare between the data pointer in the elements and
	the pointer provided in datum. If the element pointers are the same,
	they are equal, if the pointers point to different element data areas,
	they are not equal.

	If you wish to perform a real data search of the list, please use one of
	the many list_search_xx() function calls. This function is only to get the
	pivot value of an element for which you already know the data pointer, such
	as was returned from a list_simple_search(), or retained before the element
	was added to the list originally.

	This function returns a pivot value and makes no changes to the list.

	list		list_object that has been previously initialized.
	datum 		data pointer to be used for the pointer search.
	returns	PIVOT_MINOR		element datum requested is on the minor side
							of the pivot point in the list
			PIVOT_MAJOR		element datum requested is on the major side
							of the pivot point in the list
			ENTIRE_LIST		either this is not a pivot list, or the
							element was not found in the list.						*/
int list_locate_pivot(	const list_object *restrict list,
						const void *datum) {

	leaf_node *ptr;
	long int index;

#ifdef DEBUG
	if (list__validate_list_object(list, FALSE, __func__) == FALSE)
		return ENTIRE_LIST;
	if (datum == NULL) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with NULL datum pointer.\n\n", __func__);
		return ENTIRE_LIST;
	}
	if (! list->list_pivot_enabled) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called on a list other than pivot class.\n\n", __func__);
		return ENTIRE_LIST;
	}
#else
	if ((list == NULL)
			|| (datum == NULL)
			|| (! list->list_pivot_enabled))
		return ENTIRE_LIST;
#endif

	if (list->element_count < 1) return ENTIRE_LIST;

	for (ptr = list->head_guard->next_leaf, index = 0;
			(ptr) && (ptr != list->tail_guard) && (ptr->element != datum);
			ptr = ptr->next_leaf, index++);
	
	if (ptr == NULL) return ENTIRE_LIST;						/* major oops ! */
	
	if (ptr == list->tail_guard) return ENTIRE_LIST;	/* No match found */
		
	return ptr->internal->pivot.value;
}

/* Locate and return an element reference in a list, identified by its
	element datum pointer.

	This function will locate the element in this list by searching for a
	match using the datum pointer.
	
	This function ignores any set search function., this function will only
	perform a straight compare between the data pointer in the elements and
	the pointer provided in datum. If the element pointers are the same,
	they are equal, if the pointers point to different element data areas,
	they are not equal.

	If you wish to perform a real data search of the list, please use one of
	the many list_search_xx() function calls. This function is only to get the
	reference of an element for which you already know the data pointer, such
	as was returned from a list_simple_search(), or retained before the element
	was added to the list originally.
		
	This function returns a reference and makes no changes to the list.

	list		list_object that has been previously initialized.
	datum 		data pointer to be used for the pointer search.
	returns	reference		for the element datum requested
			NULL			if there was an error or no match found.				*/
element_reference *list_locate_ref(	const list_object *restrict list,
									const void *datum) {

	leaf_node *ptr;

#ifdef DEBUG
	if (list__validate_list_object(list, FALSE, __func__) == FALSE)
		return NULL;
	if (datum == NULL) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with NULL datum pointer.\n\n", __func__);
		return NULL;
	}
#else
	if ((list == NULL) || (datum == NULL))			
		return NULL;
#endif

	if (list->element_count < 1) return NULL;

	for (ptr = list->head_guard->next_leaf;
			(ptr) && (ptr != list->tail_guard) && (ptr->element != datum);
			ptr = ptr->next_leaf);
	
	if (ptr == NULL) return NULL;							/* major oops ! */
	
	if (ptr == list->tail_guard) return NULL;				/* No match found */
	
	return ptr;
}

/* Verify if an element reference is valid for the list.

	This function will return a TRUE or FALSE depending on whether the list
	element reference provided is a valid reference to an element in the
	list or not. This function call is particularly useful for finite lists
	to verify that a known element reference is still in the list, and has
	not been automatically dropped. This function is also useful for other
	list classes, when using some advanced list manipulation techniques.
	
	In general there is no reason to call this function, unless you have very
	specific reasons to feel the element reference could have become invalid
	since you last used it.
	
	If you call this function during development phase of your project coding
	with invalid element references, you will get various errors written to
	stderr, however the results of these calls will be valid. Once DEBUG is
	turned off, the error messages will no longer be present in the code, so
	the logic and outcome are the same, without the error log messages. However
	more validity checks are performed with DEBUG turned on, so for development
	purposes the error messages can be very helpful.

	This function only returns a boolean variable about the validity of the
	supplied element reference. It makes no changes to the list.

	list		list_object that has been previously initialized.
	reference	an abstract element reference to verify.
	returns	TRUE			if the element reference is valid for this list
			FALSE			if the element reference is not valid					*/
boolean list_valid_ref(	const list_object *restrict list,
						const element_reference *restrict reference) {

#ifdef DEBUG
	if (list__validate_list_object(list, FALSE, __func__) == FALSE)
		return FALSE;
	if (reference == NULL) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with NULL reference argument.\n\n", __func__);
		return FALSE;
	}
#else
	if ((list == NULL) || (reference == NULL))
		return FALSE;
#endif
	/* this will produce error messages with DEBUG turned on, they will
		not be compiled into the code once DEBUG is turned off. */
	return (list__validate_reference(list, reference, __func__) == LIST_OK);
}

/* Search for an element in a list, using datum as the search criteria.

	This function will traverse a list from tail to head, searching for an
	element with the matching datum supplied. It will only find the first
	search match, and if found will return a pointer to the element data.
	
	This is a simple version of search, the entire list is traversed, no
	consideration of pivots, or direction are provided.
	
	If a search function was previously set for this list (by calling
	list_set_search_function() ), this function will use that search function
	to compare the element data against the supplied datum.
	
	If a search function was not previously set for this list, or it was
	removed, this function will perform a straight compare between the data
	pointer in the elements and the pointer provided in datum. If the element
	pointers are the same, they are equal, if the pointers point to different
	element data areas, they are not equal.
	
	In most cases, the user will want to set a search function first. But this
	can be used to find a data element by a known pointer reference.
	
	The definitive test, on return, to verify if an element was found, is to
	check the return pointer for either NULL or an address.
	
	This function only returns a pointer to the element data in the list, it
	makes no changes to the list.

	list		list_object that has been previously initialized.
	datum 		data to be used for the search criteria.
	returns	PTR				a pointer to element data, if an element was
							found matching the datum,
			NULL			if no element was found matching the datum				*/
void *list_simple_search(	const list_object *restrict list,
							const void *datum) {

	register const leaf_node *ptr;
	register const leaf_node *end_ptr;
	boolean (*element_search)(const void *element, const void *key);

#ifdef DEBUG
	if (list__validate_list_object(list, FALSE, __func__) == FALSE)
		return NULL;
	if (datum == NULL) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with NULL datum pointer.\n\n", __func__);
		return NULL;
	}
	if (list->functions.search == NULL) {
		fprintf(stderr, "\n==> Dlist Debug WARNING ==> %s() - "\
				"NULL search function, searching for pointer match only.\n\n", __func__);
	}
#else
	if ((list == NULL) || (datum == NULL))			
		return NULL;
#endif

	if (list->element_count < 1) return NULL;

	end_ptr = list->head_guard;
	element_search = list->functions.search;

	/* for whatever strange reason, this benchmarks a few % faster than moving the
		if check outside the loop with 2 loops.  GCC go figure ...								*/
	for (ptr = list->tail_guard->prior_leaf; ptr != end_ptr; ptr = ptr->prior_leaf) {
		if (element_search) {
			if ((*element_search)(ptr->element, datum)) return ptr->element;
		} else {
			if (ptr->element == datum) return ptr->element;
		}
	}
	
	return NULL;
}

/* Search a list for all occurrences of an element, using datum as the search
	criteria.

	This function will traverse a list from tail to head, searching for an
	element with the matching datum supplied. It will find all the elements
	that match the search criteria, and if found will return a count of the
	elements found.
	
	This is a simple version of search, the entire list is traversed, no
	consideration of pivots, or direction are provided.
	
	If a search function was previously set for this list (by calling
	list_set_search_function() ), this function will use that search function
	to compare the element data against the supplied datum.
	
	If a search function was not previously set for this list, or it was
	removed, this function will perform a straight compare between the data
	pointer in the elements and the pointer provided in datum. If the element
	pointers are the same, they are equal, if the pointers point to different
	element data areas, they are not equal.
	
	In most cases, the user will want to set a search function first. But this
	can be used to find a data element by a known pointer reference.
	
	The definitive test, on return, to verify if any element was found, is to
	check the return count for either 0 or a non-zero number.
	
	This function makes no changes to the list.

	list		list_object that has been previously initialized.
	datum 		data to be used for the search criteria.
	returns	>0				number of elements found matching the datum
			0				no element was found matching the datum
			LIST_ERR		if there was an error									*/
int list_occurrences(	const list_object *restrict list,
						const void *datum) {

	register const leaf_node *ptr;
	register const leaf_node *end_ptr;
	register unsigned int i = 0;
	boolean (*element_search)(const void *element, const void *key);

#ifdef DEBUG
	if (list__validate_list_object(list, FALSE, __func__) == FALSE)
		return LIST_ERR;
	if (datum == NULL) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with NULL datum pointer.\n\n", __func__);
		return LIST_ERR;
	}
	if (list->functions.search == NULL) {
		fprintf(stderr, "\n==> Dlist Debug WARNING ==> %s() - "\
				"NULL search function, searching for pointer match only.\n\n", __func__);
	}
#else
	if ((list == NULL) || (datum == NULL))			
		return LIST_ERR;
#endif

	if (list->element_count < 1) return 0;

	end_ptr = list->head_guard;
	element_search = list->functions.search;

	for (ptr = list->tail_guard->prior_leaf; ptr != end_ptr; ptr = ptr->prior_leaf) {
		if (element_search) {
			if ((*element_search)(ptr->element, datum)) ++i;
		} else {
			if (ptr->element == datum) ++i;
		}
	}
	
	if (ptr == NULL) return LIST_ERR;					/* major oops ! */

	if (i > INT_MAX) i = INT_MAX;						/* deal with vastly unlikely boundary issues */
	
	return (int)i;
}

/* Search for an element in a list, using datum as the search criteria, and
	if found, shift the element to the head of the list.

	This function will traverse a list from tail to head, searching for an
	element with the matching datum supplied. It will only find the first
	search match, and if found will relocate the element from its current
	position in the list, to the head of the list, and also will return a
	pointer to the element data.
	
	This is a simple version of search, the entire list is traversed, no
	consideration of pivots, or direction are provided. This function is not
	available for pivot class lists. This function is particularly useful if
	used with a finite class list, that is being used as a recent history
	cache, to keep the most recent elements at the head of the list, allowing
	the oldest elements to be dropped from the list as needed. This function
	performs a sequence of operations that are equivalent to list_search_ref(),
	list_delete_ref() and list_append(), but far more efficient for this more
	specific task, and without destroying the reference to the original data
	element.
	
	If a search function was previously set for this list (by calling
	list_set_search_function() ), this function will use that search function
	to compare the element data against the supplied datum.
	
	If a search function was not previously set for this list, or it was
	removed, this function will perform a straight compare between the data
	pointer in the elements and the pointer provided in datum. If the element
	pointers are the same, they are equal, if the pointers point to different
	element data areas, they are not equal.
	
	In most cases, the user will want to set a search function first. But this
	can be used to find a data element by a known pointer reference.

	If the list is currently sorted, the sorted indexes will be voided, and the
	list_sorted_xx() functions are no longer available for the list, until it
	is resorted.
	
	The definitive test, on return, to verify if an element was found, is to
	check the return pointer for either NULL or an address.
	
	This function not only returns a pointer to the element data in the list,
	it also makes changes to the list.

	list		list_object that has been previously initialized.
	datum 		data to be used for the search criteria.
	returns	PTR			if an element was found matching the datum
			NULL		if an element was not found matching the datum				*/
void *list_search_shift(	list_object *restrict list,
							const void *datum) {

	register leaf_node *ptr;
	register const leaf_node *end_ptr;
	register long int index;
	boolean (*element_search)(const void *element, const void *key);

#ifdef DEBUG
	if (list__validate_list_object(list, FALSE, __func__) == FALSE)
		return NULL;
	if (datum == NULL) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with NULL datum pointer.\n\n", __func__);
		return NULL;
	}
	if (list->list_pivot_enabled == TRUE) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called on a pivot class list.\n\n", __func__);
		return NULL;
	}
	if (list->functions.search == NULL) {
		fprintf(stderr, "\n==> Dlist Debug WARNING ==> %s() - "\
				"NULL search function, searching for pointer match only.\n\n", __func__);
	}
#else
	if ((list == NULL) || (datum == NULL) || (list->list_pivot_enabled == TRUE))			
		return NULL;
#endif

	if (list->element_count < 1) return NULL;

	end_ptr = list->head_guard;
	element_search = list->functions.search;

	for (ptr = list->tail_guard->prior_leaf, index = list->element_count-1; ptr != end_ptr; ptr = ptr->prior_leaf, --index) {
		if (element_search) {
			if ((*element_search)(ptr->element, datum)) break;
		} else {
			if (ptr->element == datum) break;
		}
	}
	
	if (ptr == NULL) return NULL;							/* major oops ! */
	
	if (ptr == end_ptr) return NULL;						/* No match found */
	
	/* Match found, first move the element, (we do not use the normal primitives as we want
		to retain all list data including the original leaf node and its reference) */
	list->list_sorted = FALSE;

	list__adjust_middle_pointer(list, index, LEAF_REMOVED);
	unlink__leaf_node(ptr);
	link__leaf_node(ptr, list->head_guard);
	list__adjust_middle_pointer(list, 0, LEAF_ADDED);

#ifdef DLIST_DEBUG
	assert(list__audit(list) == LIST_OK);
#endif		/* DLIST_DEBUG */
	
	/* then return the pointer to the datum */
	return ptr->element;
}

/* Search for an element in a list, using datum as the search criteria.

	This function will traverse a list constrained by the scope argument, in
	the direction specified, searching for an element with the matching datum
	supplied. It will only find the first search match, and if found will
	return a pointer to the element data, the index number in the list of the
	element, and a list reference to the element. The user may choose to use
	any of these methods to reacquire, remove or manipulate the element, or
	data. Please bear in mind that the index is only valid until the list is
	modified by adding or removing elements. The reference remains valid
	until the element itself is removed from the list.

	The argument scope determines how far along the list this function will
	traverse looking for a matching elek=ment. Providing ENTIRE_LIST as this
	argument will cause the function to traverse the list from head to tail.
	However, if the list is of a pivot class, then providing either PIVOT_MAJOR
	or PIVOT_MINOR will optionally cause this function to confine its search to
	that side of the list pivot point. If the list is of the linear class, then
	the setting of this argument is ignored. This feature is useful if the
	comparison function tests different data points, than the set pivot
	function. It allows for lists to be easily used in a multidimensional way
	with simple code.
	
	The direction argument is directly related to the contents of scope. If
	scope is ENTIRE_LIST, the direction defines from which to start the search.
	If the scope is pivot based, then direction defines whether to start from an
	end of the list, or from the pivot point.
	
	This function is available for finite class lists, with the exception that
	a reference will not be returned regardless of the reference argument.
	
	If a search function was previously set for this list (by calling
	list_set_search_function() ), this function will use that search function
	to compare the element data against the supplied datum.
	
	If a search function was not previously set for this list, or it was
	removed, this function will perform a straight compare between the data
	pointer in the elements and the pointer provided in datum. If the element
	pointers are the same, they are equal, if the pointers point to different
	element data areas, they are not equal.
	
	In most cases, the user will want to set a search function first. But this
	can be used to find a data element by a known pointer reference.
	
	The definitive test, on return, to verify if an element was found, is to
	check the return pointer for either NULL or an address.
	
	This function only returns a pointer, index and reference to the element
	in the list, it makes no changes to the list.

	list		list_object that has been previously initialized.
	datum 		data to be used for the search criteria.
	scope		for pivot lists only, which side of the pivot point to search
				for the datum. Must be either PIVOT_MINOR, PIVOR_MAJOR, or
				ENTIRE_LIST. For linear and finite lists, must be ENTIRE_LIST
	direction	If scope is ENTIRE_LIST, then this argument may contain
				either HEAD_OF_LIST, proceed from the tail of the list towards
				the head of the list, or TAIL_OF_LIST proceed from the head of
				the list towards the tail of the list.
				If this is a pivot class list, and scope is either of the PIVOT
				options, then this argument may contain either PIVOT_OF_LIST,
				search starts at the end of the list towards the pivot, or
				END_OF_LIST in which case the search starts from the pivot
				point and runs to the end of the list, the direction defined by
				the scope argument (PIVOT_MAJOR or PIVOT_MINOR)
	reference	pointer to variable used to store the element reference,
		 		if found. NULL, no element reference is to be returned.
	index		pointer to variable used to store the element index, if
		 		found. Or NULL, no element index is to be returned.
	returns	if an element was found matching the datum, a pointer to element,
				and if requested an index of element in list, and the element
				reference.
			if an element was not found matching the datum,
				NULL pointer, index of 0, and NULL element reference.				*/
void *list_search(	const list_object *restrict list,
					const void *datum,
					const int scope,
					const boolean direction,
					element_reference **restrict reference,
					unsigned int *index) {

	register const leaf_node *ptr;
	register const leaf_node *end_ptr;
	long int temp_index = 0;
	register unsigned int i;
	boolean forwards = TRUE;
	boolean (*element_search)(const void *element, const void *key);
	
	
	/* set default return state in case of errors, there is to be no ambiguity of success */
	if (reference != NULL) *reference = NULL;
	if (index != NULL) *index = 0;

#ifdef DEBUG
	if (list__validate_list_object(list, FALSE, __func__) == FALSE)
		return NULL;
	if ((list->list_finite_enabled == TRUE) && (reference != NULL)) {
		fprintf(stderr, "\n==> Dlist Debug WARNING ==> %s() - "\
				"called on a finite class list with a non NULL reference argument. "\
				"NO reference will be returned.\n\n", __func__);
	}
	if (datum == NULL) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with NULL datum pointer.\n\n", __func__);
		return NULL;
	}
	if (list->functions.search == NULL) {
		fprintf(stderr, "\n==> Dlist Debug WARNING ==> %s() - "\
				"NULL search function, searching for pointer match only.\n\n", __func__);
	}
	if ((list->list_pivot_enabled == FALSE) && (scope != ENTIRE_LIST)) {
		fprintf(stderr, "\n==> Dlist Debug WARNING ==> %s() - "\
				"an ignored scope argument '%d' was provided for a linear class list.\n\n",
				__func__, scope);
	}
	if ((list->list_pivot_enabled == TRUE) && ((scope < PIVOT_MINOR) || (scope > PIVOT_MAJOR))) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"invalid scope argument '%d' was provided for a pivot class list.\n\n",
				__func__, scope);
		return NULL;
	}
#else
	if ((list == NULL)
			|| (datum == NULL)
			|| ((list->list_pivot_enabled == TRUE) && ((scope < PIVOT_MINOR) || (scope > PIVOT_MAJOR)) ))			
		return NULL;
#endif

	if (list->element_count < 1) return NULL;

	element_search = list->functions.search;

	/* Set up the search parameters */
	if ((list->list_pivot_enabled == TRUE) && (scope != ENTIRE_LIST)) {
		ptr = list__locate_pivot_leaf_node(list, &temp_index);
		if (direction == END_OF_LIST) {
			if (scope == PIVOT_MAJOR) {
				forwards = TRUE;
				ptr = ptr->next_leaf;					/* the first leaf node of the major side */
				end_ptr = list->tail_guard;
				i = list->pivot_point.minor;
			} else {
				forwards = FALSE;
				end_ptr = list->head_guard;				/* ptr already points the start leaf node */
				i = list->pivot_point.minor-1;
			}
		} else {
			/* direction == PIVOT_OF_LIST */
			if (scope == PIVOT_MAJOR) {
				forwards = FALSE;
				end_ptr = ptr;							/* the first leaf node of the minor side */
				ptr = list->tail_guard->prior_leaf;
				i = list->element_count - 1;
			} else {
				forwards = TRUE;
				end_ptr = ptr->next_leaf;
				ptr = list->head_guard->next_leaf;
				i = 0;
			}		
		}
	} else {
		/* entire list, no pivot relevant */
		if (direction == TAIL_OF_LIST) {
			forwards = TRUE;
			ptr = list->head_guard->next_leaf;
			end_ptr = list->tail_guard;
			i = 0;
		} else {
			forwards = FALSE;
			ptr = list->tail_guard->prior_leaf;
			end_ptr = list->head_guard;
			i = list->element_count - 1;
		}
	}

	/* Perform the search */
	if (forwards) {
		for (; ptr != end_ptr; ++i, ptr = ptr->next_leaf) {
			if (element_search) {
				if ((*element_search)(ptr->element, datum)) break;
			} else {
				if (ptr->element == datum) break;
			}
		}
	} else {
		for (; ptr != end_ptr; --i, ptr = ptr->prior_leaf) {
			if (element_search) {
				if ((*element_search)(ptr->element, datum)) break;
			} else {
				if (ptr->element == datum) break;
			}
		}
	}
	
	if (ptr == NULL) return NULL;						/* major oops ! */
	
	if (ptr == end_ptr) return NULL;					/* No match found */
	
	/* return identity of found list element */
	if ((reference != NULL) && (! list->list_finite_enabled)) *reference = (leaf_node  *)ptr;
	if (index != NULL) *index = i;
	return ptr->element;
}

/* Search for an element in a pivot class list, using datum to decide which
	side of the pivot to search, and also as the search criteria.

	This function will traverse a list constrained by the scope returned by
	presenting the datum to the lists pivot function. Only that side of the
	pivot list will be searched.
	
	The function will then search for an element with the matching datum
	supplied. It will only find the first search match, and if found will
	return a pointer to the element data, the index number in the list of the
	element, if requested, and an optional list reference to the element. The
	user may choose to use any of these methods to reacquire, remove or later
	manipulate the element, or data. Please bear in mind that the index is
	only valid until the list is modified by adding or removing elements. The
	reference remains valid until the element itself is removed from the list.

	The specific scope of the search, is determined by presenting the datum to
	the current pivot function for the list. The direction argument is directly
	related to the results of obtaining the scope. It defines whether to start
	from one end of the list, or from the pivot point.
	
	This function is only available for pivot class lists. A pivot function must
	be set for this list.
	
	If a search function was previously set for this list (by calling
	list_set_search_function() ), this function will use that search function
	to compare the element data against the supplied datum.
	
	If a search function was not previously set for this list, or it was
	removed, this function will perform a straight compare between the data
	pointer in the elements and the pointer provided in datum. If the element
	pointers are the same, they are equal, if the pointers point to different
	element data areas, they are not equal.
	
	In most cases, the user will want to set a search function first. But this
	can be used to find a data element by a known pointer reference.
	
	The definitive test, on return, to verify if an element was found, is to
	check the return pointer for either NULL or an address.
	
	This function only returns a pointer, index, pivot value, and reference to
	the element in the list, it makes no changes to the list.

	list		list_object that has been previously initialized as a pivot
				class list.
	datum 		data to be used for the search criteria.
	direction	this argument describes which direction to perform the search.
				It may contain either PIVOT_OF_LIST, so the search starts at
				the end of the list towards the pivot, or END_OF_LIST the
				search starts from the pivot point and runs out to the end
				of the list. The specific scope is defined by the result of
				the pivot function on the datum argument (either PIVOT_MAJOR
				or PIVOT_MINOR)
	reference	pointer to variable used to store the element reference,
		 		if found. NULL, no element reference is to be returned.
	index		pointer to variable used to store the element index, if
		 		found. Or NULL, no element index is to be returned.
	pivot_value	pivot value of the element, if one was found. Or NULL, no
				element pivot value is to be returned.
	returns	if an element was found matching the datum, a pointer to element,
				and if requested an index of element in list, the pivot value,
				and the element reference.
			if an element was not found matching the datum, NULL pointer,
				pivot value of 0, index of 0, and NULL element reference.			*/
void *list_search_pivot(	const list_object *restrict list,
							const void *datum,
							const boolean direction,
							element_reference **restrict reference,
							unsigned int *index,
							int *restrict pivot_value) {

	register const leaf_node *ptr;
	register const leaf_node *end_ptr;
	long int temp_index = 0;
	register unsigned int i;
	int scope;
	boolean forwards = TRUE;
	boolean (*element_search)(const void *element, const void *key);
	
	
	/* set default return state in case of errors, there is to be no ambiguity of success */
	if (reference != NULL) *reference = NULL;
	if (index != NULL) *index = 0;
	if (pivot_value != NULL) *pivot_value = 0;

#ifdef DEBUG
	if (list__validate_list_object(list, FALSE, __func__) == FALSE)
		return NULL;
	if (datum == NULL) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with NULL datum pointer.\n\n", __func__);
		return NULL;
	}
	if (list->list_pivot_enabled == FALSE) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called on a non pivot class list.\n\n", __func__);
		return NULL;
	}
	/* this is redundant, but put here just in case of future dlist changes */
	if (list->list_finite_enabled == TRUE) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called on a finite class list.\n\n", __func__);
		return NULL;
	}
	if (list->functions.pivot == NULL) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called for pivot list with NULL pivot function.", __func__);
		return NULL;
	}
	if (list->functions.search == NULL) {
		fprintf(stderr, "\n==> Dlist Debug WARNING ==> %s() - "\
				"NULL search function, searching for pointer match only.\n\n", __func__);
	}
#else
	if ((list == NULL)
			|| (datum == NULL)
			|| (list->functions.pivot == NULL)
			|| (list->list_finite_enabled == TRUE)
			|| (list->list_pivot_enabled == FALSE))
		return NULL;
#endif

	if (list->element_count < 1) return NULL;

	element_search = list->functions.search;

	scope = list->functions.pivot(datum);

	/* Set up the search parameters */
	ptr = list__locate_pivot_leaf_node(list, &temp_index);
	if (direction == END_OF_LIST) {
		if (scope > 0) {
			forwards = TRUE;
			ptr = ptr->next_leaf;					/* the first leaf node of the major side */
			end_ptr = list->tail_guard;
			i = list->pivot_point.minor;
		} else {
			forwards = FALSE;
			end_ptr = list->head_guard;				/* ptr already points the start leaf node */
			i = list->pivot_point.minor-1;
		}
	} else {
		/* direction == PIVOT_OF_LIST */
		if (scope > 0) {
			forwards = FALSE;
			end_ptr = ptr;							/* the first leaf node of the minor side */
			ptr = list->tail_guard->prior_leaf;
			i = list->element_count - 1;
		} else {
			forwards = TRUE;
			end_ptr = ptr->next_leaf;
			ptr = list->head_guard->next_leaf;
			i = 0;
		}		
	}

	/* Perform the search */
	if (forwards) {
		for (; ptr != end_ptr; ++i, ptr = ptr->next_leaf) {
			if (element_search) {
				if ((*element_search)(ptr->element, datum)) break;
			} else {
				if (ptr->element == datum) break;
			}
		}
	} else {
		for (; ptr != end_ptr; --i, ptr = ptr->prior_leaf) {
			if (element_search) {
				if ((*element_search)(ptr->element, datum)) break;
			} else {
				if (ptr->element == datum) break;
			}
		}
	}
	
	if (ptr == NULL) return NULL;						/* major oops ! */
	
	if (ptr == end_ptr) return NULL;					/* No match found */
	
	/* return identity of found list element */
	if ((reference != NULL) && (! list->list_finite_enabled)) *reference = (leaf_node  *)ptr;
	if (index != NULL) *index = i;
	if (pivot_value != NULL) *pivot_value = ptr->internal->pivot.value;
	return ptr->element;

}

/* Search for an element in a list, using datum as the search criteria, backwards
	or forwards, from the reference point indicated.

	This function traverses a list from the supplied reference element to the end,
	searching for an element with the matching datum supplied. It will only find
	the first search match, and if found will return a pointer to the element data,
	and a list reference to the element. The user may choose to use any of these
	methods to reacquire, remove or manipulate the element, or data. The reference
	remains valid until the element itself is removed from the list.
	
	An index cannot be returned from this search, the efficiency gains from using
	element references, would be obviated if the index had to be discovered as well.
	
	This function is not available for finite class lists.
	
	If a search function was previously set for this list (by calling
	list_set_search_function() ), this function will use that search function to
	compare the element data against the supplied datum.
	
	If a search function was not previously set for this list, or it was
	removed, this function will perform a straight compare between the data
	pointer in the elements and the pointer provided in datum. If the element
	pointers are the same, they are equal, if the pointers point to different
	element data areas, they are not equal.
	
	In most cases, the user will want to set a search function first. But this
	can be used to find a data element by a known pointer reference.
	
	The definitive test, on return, to verify if an element was found, is to check
	the return pointer for either NULL or an address. In all cases the argument
	'reference' is modified by calling this function, and the previous contents
	are lost.
	
	This function only returns a pointer and reference to the element
		in the list, it makes no changes to the list.

	list		list_object that has been previously initialized.
	datum 		data to be used for the search criteria.
	reference	caller supplied element reference to begin the reverse
				search. If reference is NULL, the search will be performed
				starting at the end of the list. Reference is always
				returned if an element is found.
	direction	HEAD_OF_LIST from the reference point, towards the head of
				the list, or TAIL_OF_LIST from the reference point towards
				the tail of the list.
	returns	if an element was found matching the datum, the pointer to
					the element and the element reference.
			if an element was not found matching the datum, a NULL pointer,
					a NULL element reference.										
			pivot_point		if this found element was retrieved from the
							other side of the pivot point, this return
							argument is set to TRUE. 								*/
void *list_search_ref(	const list_object *restrict list,
						const void *datum,
						element_reference **restrict reference,
						const boolean direction,
						boolean *restrict pivot_point) {

	leaf_node *ptr, *end_ptr;
	int saved_pivot = 0;
	boolean (*element_search)(const void *element, const void *key);
	
#ifdef DEBUG
	if (list__validate_list_object(list, FALSE, __func__) == FALSE)
		return NULL;
	if (datum == NULL) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with NULL datum pointer.\n\n", __func__);
		return NULL;
	}
	if (list->list_finite_enabled == TRUE) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called on a finite class list.\n\n", __func__);
		return NULL;
	}
	if (list->functions.search == NULL) {
		fprintf(stderr, "\n==> Dlist Debug WARNING ==> %s() - "\
				"NULL search function, searching for pointer match only.\n\n", __func__);
	}
#else
	if ((list == NULL)
			|| (datum == NULL)
			|| (list->list_finite_enabled == TRUE))
		return NULL;
#endif
	
	if (*reference == NULL) {
		if (direction == HEAD_OF_LIST) ptr = list->tail_guard->prior_leaf;
		else ptr = list->head_guard->next_leaf;
	} else {
		ptr = *reference;
		if (list__validate_reference(list, ptr, __func__) != LIST_OK) return NULL;
		saved_pivot = ptr->internal->pivot.value;
	}
	
	if (direction == HEAD_OF_LIST) end_ptr = list->head_guard;
	else end_ptr = list->tail_guard;

	element_search = list->functions.search;

	/* Perform the search */
	if (direction == TAIL_OF_LIST) {
		for (; ptr != end_ptr; ptr = ptr->next_leaf) {
			if (element_search) {
				if ((*element_search)(ptr->element, datum)) break;
			} else {
				if (ptr->element == datum) break;
			}
		}
	} else {
		for (; ptr != end_ptr; ptr = ptr->prior_leaf) {
			if (element_search) {
				if ((*element_search)(ptr->element, datum)) break;
			} else {
				if (ptr->element == datum) break;
			}
		}
	}
	
	if (ptr == NULL) return NULL;
	
	if (ptr == end_ptr) return NULL;				/* No match found */
	
	/* return identity of found list element */
	if (*reference !=NULL) {
		if (ptr->internal->pivot.value != saved_pivot) *pivot_point = TRUE;
		else *pivot_point = FALSE;
	}
	*reference = ptr;
	return ptr->element;
}

/* Search for, and then extract, an element in a list, using datum as the
	search criteria.

	This function traverses a list from head to tail, searching for an element
	with the matching datum supplied. It will only find the first search match,
	and if found will delete the element from the list and return the element
	data to the caller.
	
	If a search function was previously set for this list (by calling
	list_set_search_function() ), this function will use that search function to
	compare the element data against the supplied datum.
	
	If a search function was not previously set for this list, or it was
	removed, this function will perform a straight compare between the data
	pointer in the elements and the pointer provided in datum. If the element
	pointers are the same, they are equal, if the pointers point to different
	element data areas, they are not equal.
	
	In most cases, the user will want to set a search function first. But this
	can be used to find a data element by a known pointer reference.
		
	list		list_object that has been previously initialized.
	datum 		data to be used for the search criteria.
	returns		NULL no element was found in list, matching the datum
				PTR to element data. It is the users responsibility to manage
					this data, and free it when finished with it. If the user
					supplied it originally it is the same allocated memory
					originally given to D-List. If the list was created with
					the LIST_COPY_DATA attribute, the data was allocated
					internally by D-List. In both cases it now belongs to the
					caller.															*/
void *list_search_extract(	list_object *restrict list,
							const void *datum) {

	register leaf_node *ptr;
	register const leaf_node *end_ptr;
	void *element;
	register unsigned int index = 0;
	boolean (*element_search)(const void *element, const void *key);

#ifdef DEBUG
	if (list__validate_list_object(list, FALSE, __func__) == FALSE)
		return NULL;
	if (datum == NULL) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with NULL datum pointer.\n\n", __func__);
		return NULL;
	}
	if (list->functions.search == NULL) {
		fprintf(stderr, "\n==> Dlist Debug WARNING ==> %s() - "\
				"NULL search function, searching for pointer match only.\n\n", __func__);
	}
#else
	if ((list == NULL) || (datum == NULL))
		return NULL;
#endif

	if (list->element_count < 1) return NULL;

	end_ptr = list->tail_guard;
	element_search = list->functions.search;

	for (ptr = list->head_guard->next_leaf; ptr != end_ptr; ++index, ptr = ptr->next_leaf) {
		if (element_search) {
			if ((*element_search)(ptr->element, datum)) break;
		} else {
			if (ptr->element == datum) break;
		}
	}
	
	if (ptr == NULL) return NULL;								/* major oops ! */
	
	if (ptr == end_ptr) return NULL;							/* No match found */
	
	element = ptr->element;
	if (list->functions.remove) list->functions.remove(ptr->element);
	ptr->element = NULL;					/* prevent underlying primitives from freeing the resource */

	list__remove_leaf_node(ptr, index);

#ifdef DLIST_DEBUG
	assert(list__audit(list) == LIST_OK);
#endif		/* DLIST_DEBUG */

	return element;
}

/* Search for, and then delete, an element in a list, using datum as the
	search criteria.

	This function traverses a list from head to tail, searching for an element
	with the matching datum supplied. It will only find the first search match,
	and if found will delete the element from the list.
	
	If a search function was previously set for this list (by calling
	list_set_search_function() ), this function will use that search function to
	compare the element data against the supplied datum.
	
	If a search function was not previously set for this list, or it was
	removed, this function will perform a straight compare between the data
	pointer in the elements and the pointer provided in datum. If the element
	pointers are the same, they are equal, if the pointers point to different
	element data areas, they are not equal.
	
	In most cases, the user will want to set a search function first. But this
	can be used to find a data element by a known pointer reference.
		
	list		list_object that has been previously initialized.
	datum 		data to be used for the search criteria.
	returns	LIST_OK				if an element was found matching the datum,
								and was removed.
			ELEMENT_NO_DATUM	element data not supplied
			ELEMENT_NOT_FOUND	no element was found matching the datum	
			LIST_ERR			if there was an error								*/
int list_search_delete(	list_object *restrict list,
						const void *datum) {

	register leaf_node *ptr;
	register const leaf_node *end_ptr;
	register unsigned int index = 0;
	boolean (*element_search)(const void *element, const void *key);

#ifdef DEBUG
	if (list__validate_list_object(list, FALSE, __func__) == FALSE)
		return LIST_ERR;
	if (datum == NULL) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with NULL datum pointer.\n\n", __func__);
		return ELEMENT_NO_DATUM;
	}
	if (list->functions.search == NULL) {
		fprintf(stderr, "\n==> Dlist Debug WARNING ==> %s() - "\
				"NULL search function, searching for pointer match only.\n\n", __func__);
	}
#else
	if (list == NULL)
		return LIST_ERR;
	if (datum == NULL)
		return ELEMENT_NO_DATUM;
#endif

	if (list->element_count < 1) return ELEMENT_NOT_FOUND;

	end_ptr = list->tail_guard;
	element_search = list->functions.search;

	for (ptr = list->head_guard->next_leaf; ptr != end_ptr; ++index, ptr = ptr->next_leaf) {
		if (element_search) {
			if ((*element_search)(ptr->element, datum)) break;
		} else {
			if (ptr->element == datum) break;
		}
	}
	
	if (ptr == NULL) return LIST_ERR;							/* major oops ! */
	
	if (ptr == end_ptr) return ELEMENT_NOT_FOUND;				/* No match found */

	if (list->functions.remove) list->functions.remove(ptr->element);
	
	list__remove_leaf_node(ptr, index);

#ifdef DLIST_DEBUG
	assert(list__audit(list) == LIST_OK);
#endif		/* DLIST_DEBUG */

	return LIST_OK;
}

/* Search for, and then delete, all occurrences of an element in a list, using
	datum as the search criteria.

	This function traverses a list from head to tail, searching for an element
	with the matching datum supplied. It will find all matching occurrences of
	the supplied datum, and when found will delete all the matching elements
	from the list. This function is similar to list_search_delete() except this
	function removes all instances of the datum from the list, instead one.
	
	If a search function was previously set for this list (by calling
	list_set_search_function() ), this function will use that search function to
	compare the element data against the supplied datum.
	
	If a search function was not previously set for this list, or it was
	removed, this function will perform a straight compare between the data
	pointer in the elements and the pointer provided in datum. If the element
	pointers are the same, they are equal, if the pointers point to different
	element data areas, they are not equal.
	
	In most cases, the user will want to set a search function first. But this
	can be used to find a data element by a known pointer reference.
		
	list		list_object that has been previously initialized.
	datum 		data to be used for the search criteria.
	returns	>0					number of elements found and removed matching
								the datum
			0					no element was found matching the datum
			ELEMENT_NO_DATUM	element data not supplied
			LIST_ERR			if there was an error								*/
int list_search_expunge(	list_object *restrict list,
							const void *datum) {

	register leaf_node *ptr;
	register const leaf_node *end_ptr;
	register unsigned int index = 0;
	unsigned int count = 0;
	boolean remove_leaf_node = FALSE;
	boolean (*element_search)(const void *element, const void *key);

#ifdef DEBUG
	if (list__validate_list_object(list, FALSE, __func__) == FALSE)
		return LIST_ERR;
	if (datum == NULL) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with NULL datum pointer.\n\n", __func__);
		return ELEMENT_NO_DATUM;
	}
	if (list->functions.search == NULL) {
		fprintf(stderr, "\n==> Dlist Debug WARNING ==> %s() - "\
				"NULL search function, searching for pointer match only.\n\n", __func__);
	}
#else
	if (list == NULL)
		return LIST_ERR;
	if (datum == NULL)
		return ELEMENT_NO_DATUM;
#endif

	if (list->element_count < 1) return 0;

	end_ptr = list->tail_guard;
	element_search = list->functions.search;

	for (ptr = list->head_guard->next_leaf; ptr != end_ptr; ++index, ptr = ptr->next_leaf) {
		if (element_search) {
			if ((*element_search)(ptr->element, datum)) remove_leaf_node = TRUE;
		} else {
			if (ptr->element == datum) remove_leaf_node = TRUE;
		}
		if (remove_leaf_node) {
			if (list->functions.remove) list->functions.remove(ptr->element);
			ptr = ptr->prior_leaf;
			list__remove_leaf_node(ptr->next_leaf, index--);
			++count;
			remove_leaf_node = FALSE;
		}
	}
	
	if (ptr == NULL) return LIST_ERR;						/* major oops ! */
	if (count > INT_MAX) count = INT_MAX;

#ifdef DLIST_DEBUG
	assert(list__audit(list) == LIST_OK);
#endif		/* DLIST_DEBUG */

	return (int)count;
}

/* Start a sequence of iteration of all the elements in a list.

	This function initiates an iteration sequence. The iteration sequence will
	continue until the function list_stop_iteration() is called on this list.
	
	An iteration sequence traverses a list from the head to the tail (or the
	opposite direction if that was requested), returning each element in sequence
	using list_iteration_next(). For each element a pointer to the element data
	and a list reference to the element are provided.
	
	During an iteration sequence all index based modifications to the list are
	blocked and denied. However index based gets and searches can occur. Also
	by design, elements can be fetched, inserted, removed and modified using the
	element references returned by various other dlist function calls. It is the
	responsibility of the user to use these features wisely. In particular unwise
	use of inserting an element by reference at a point following the current
	iteration sequence location, has the obvious potential to create an infinite
	loop situation, of adding a new element and iterating to it, and adding
	another, etc etc. Use these features carefully...
	
	A typical use of this iteration sequence is
	
		...
		list_iteration_start(a_list);
		while (list_iteration_has_more(a_list) {
			ptr = list_iteration_get_next(a_list, reference);
			do something with the ptr-> element
			or save or do something with the reference
			...
		}
		list_iteration_stop(a_list);
		...
	
	But there are also many other creative ways to use this, especially with
	nested sub-lists, and cross referencing. The user can operate multiple
	iteration sequences on multiple lists at the same time. But can only
	operate an iteration sequence on a specific list, one at a time.
	
	An iteration sequence can be started on any class of list. For a pivot
	class list the sequence can be controlled either to the other end of the
	list, or to the pivot point, depending on how the caller uses the
	list_iteration_has_more() function calls.
	
	list		list_object that has been previously initialized.
	direction	HEAD_OF_LIST towards the head of the list, or TAIL_OF_LIST
				towards the tail of the list.
	returns	LIST_OK			the iteration sequence has been established
			LIST_ERR		if there was an error.									*/
int list_iteration_start(	list_object *restrict list,
							const boolean direction) {

#ifdef DEBUG
	if (list__validate_list_object(list, FALSE, __func__) == FALSE)
		return LIST_ERR;
	if (list->iteration_enabled) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called on list with iteration already enabled.\n\n", __func__);
		return LIST_ERR;
	}
#else
	if ((list == NULL) || (list->iteration_enabled))
		return LIST_ERR;
#endif

	list->iteration_direction = direction;
	if (direction == TAIL_OF_LIST) list->iteration_point = list->head_guard;
	else list->iteration_point = list->tail_guard;
	list->iteration_enabled = TRUE;

	return LIST_OK;
}

/* Start a sequence of iteration of all the elements in a list on one side of
	a list pivot point.

	This function initiates an iteration sequence. The iteration sequence will
	continue until the function list_stop_iteration() is called on this list.
	The list must be a pivot class list.
	
	An iteration sequence traverses a list from the current pivot point of the
	list, to either the head or the tail depending on the direction that was
	requested. Each element can be returned in sequence using the function
	list_iteration_next(). For each element a pointer to the element data
	and a list reference to the element are provided.
	
	During an iteration sequence all index based modifications to the list are
	blocked and denied. However index based gets and searches can occur. Also
	by design, elements can be fetched, inserted, removed and modified using the
	element references returned by various other D-list function calls. It is the
	responsibility of the user to use these features wisely.
	
	An iteration sequence started from a pivot point, can only be initiated for
	a pivot class list. The caller must use the function list_iteration_has_more()
	to determine if there are additional elements to find in this sequence.
	
	list		list_object that has been previously initialized.
	direction	PIVOT_MINOR from the pivot point towards the head of the list
				(traversing all the minor elements), or PIVOT_MAJOR from the
				pivot point towards the tail of the list (traversing all the
				major elements).
	returns	LIST_OK			the iteration sequence has been established
			LIST_ERR		if there was an error.									*/
int list_iteration_start_pivot(	list_object *restrict list,
								const int direction) {

	long index = 0;

#ifdef DEBUG
	if (list__validate_list_object(list, FALSE, __func__) == FALSE)
		return LIST_ERR;
	if (list->iteration_enabled) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called on list with iteration already enabled.\n\n", __func__);
		return LIST_ERR;
	}
	if (list->list_pivot_enabled == FALSE) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called on a non pivot class list.\n\n", __func__);
		return LIST_ERR;
	}
	if (list->list_finite_enabled == TRUE) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called on a finite class list.\n\n", __func__);
		return LIST_ERR;
	}
	if ((direction != PIVOT_MINOR) && (direction != PIVOT_MAJOR)) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with invalid direction value '%d' Should be either PIVOT_MINOR "\
				"or PIVOT_MAJOR.\n\n", __func__, direction);
		return LIST_ERR;
	}
#else
	if ((list == NULL)
			|| (list->iteration_enabled)
			|| ((direction != PIVOT_MINOR) && (direction != PIVOT_MAJOR)))
		return LIST_ERR;
	if ((list->list_pivot_enabled == FALSE)
			|| (list->list_finite_enabled == TRUE))
		return LIST_ERR;
#endif

	
	if (direction == PIVOT_MINOR) list->iteration_direction = HEAD_OF_LIST;
	else list->iteration_direction = TAIL_OF_LIST;

	if (direction == TAIL_OF_LIST)
		list->iteration_point = list__locate_pivot_leaf_node(list, &index);
	else {
		list->iteration_point = list__locate_pivot_leaf_node(list, &index);
		list->iteration_point = list->iteration_point->next_leaf;
	}
	
	list->iteration_enabled = TRUE;

	return LIST_OK;
}

/* Inquire if there are more elements to be found, during a sequence of
	iteration of all the elements in a list.

	This function requires an active iteration sequence. It returns TRUE if
	there are 1 or more elements left to iterate over in the list. It returns
	FALSE if there are no more left.
	
	For more details about iteration sequences, refer to list_start_iteration().
	
	list		list_object that has been previously initialized.
	returns	TRUE			there are more elements left in the iteration
							sequence
			FALSE			no more elements are remaining to iterate over			*/
boolean list_iteration_has_more(const list_object *restrict list) {

#ifdef DEBUG
	if (list__validate_list_object(list, FALSE, __func__) == FALSE)
		return FALSE;
	if (! list->iteration_enabled) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called on list with no active iteration.\n\n", __func__);
		return FALSE;
	}
#else
	if ((list == NULL) || (! list->iteration_enabled))
		return FALSE;
#endif
	
	if (list->iteration_direction == TAIL_OF_LIST) {
		if (list->iteration_point->next_leaf->next_leaf == NULL) return FALSE;
		else return TRUE;
	} else {
		if (list->iteration_point->prior_leaf->prior_leaf == NULL) return FALSE;
		else return TRUE;
	}
	
	return FALSE;							/* Never touched, but i despise drop through functions */
}

/* Inquire if there are more elements to be found, during a sequence of
	iteration of all the elements in a list on one side of a pivot.

	This function requires an active iteration sequence. It returns TRUE if
	there are 1 or more elements left to iterate over in the list before
	reaching the pivot point of the list. It returns FALSE if there are no
	more elements left before the pivot point. You can still continue to
	call list_iteration_get_next() until list_iteration_has_more() returns
	FALSE. By using both list_iteration_has_more functions you can
	navigate lists in a variety of useful and non-standard ways.
	
	This functions main purpose is to support iteration sequences starting
	from the end of the list, where the user wants to know where the pivot
	point is. However, if the iteration sequence was started using the
	list_iteration_start_pivot() function instead, then this function is not
	needed as the iteration sequence will be moving away from the pivot
	point, not towards it. Under those circumstances it will operate the
	same as a call to list_iteration_has_more() does.
	
	This function can only be called for a pivot class list.
	
	For more details about iteration sequences, refer to list_start_iteration().
	
	list		list_object that has been previously initialized.
	returns	TRUE			there are more elements left in the iteration
							sequence
			FALSE			no more elements are remaining to iterate over			*/
boolean list_iteration_has_more_pivot(const list_object *restrict list) {

#ifdef DEBUG
	if (list__validate_list_object(list, FALSE, __func__) == FALSE)
		return FALSE;
	if (! list->iteration_enabled) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called on list with no active iteration.\n\n", __func__);
		return FALSE;
	}
	if (! list->list_pivot_enabled) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called on list other than pivot class.\n\n", __func__);
		return FALSE;
	}
#else
	if ((list == NULL)
			|| (! list->iteration_enabled)
			|| (! list->list_pivot_enabled))
		return FALSE;
#endif
	
	if (list->iteration_direction == TAIL_OF_LIST) {
		if (list->iteration_point->next_leaf->next_leaf == NULL) {
			return FALSE;
		} else {
			if (list->iteration_point->next_leaf->internal->pivot.value == list->iteration_point->internal->pivot.value)
				return TRUE;
			else
				return FALSE;
		}
	} else {
		if (list->iteration_point->prior_leaf->prior_leaf == NULL) {
			return FALSE;
		} else {
			if (list->iteration_point->prior_leaf->internal->pivot.value == list->iteration_point->internal->pivot.value)
				return TRUE;
			else
				return FALSE;
		}
	}
	
	return FALSE;							/* Never touched, but i despise drop through functions */
}

/* Returns a pointer to the next element, during a sequence of iteration of
	all the elements in a list.

	This function requires an active iteration sequence for this list was
	initiated prior to this call.
	
	For more details about iteration sequences, refer to list_start_iteration().

	This function will return a pointer to the element data, and a list reference
	to the element. The user may choose to use any of these methods to reacquire,
	remove or manipulate the element, or data. The reference remains valid until
	the element itself is removed from the list. Reference is an optional argument
	and if supplied NULL, it will be ignored, otherwise it will be modified.

	list		list_object that has been previously initialized.
	returns	if another element was available in this iteration sequence, the
					pointer to the element and the optional element reference.
			if no more elements are available, a NULL pointer, and a NULL
					element reference		.										*/
void *list_iteration_get_next(	list_object *restrict list,
								element_reference **restrict reference) {

	leaf_node *ptr = NULL;

	if (reference != NULL) *reference = NULL;				/* clear this out first */

#ifdef DEBUG
	if (list__validate_list_object(list, FALSE, __func__) == FALSE)
		return NULL;
	if (! list->iteration_enabled) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called on list with no active iteration.\n\n", __func__);
		return NULL;
	}
#else
	if ((list == NULL) || (! list->iteration_enabled))
		return NULL;
#endif

	
	if (list->iteration_direction == TAIL_OF_LIST) {
		ptr = list->iteration_point->next_leaf;
		if (ptr->next_leaf == NULL) return NULL;		/* this is the tail guard, do not return it */
	} else {
		ptr = list->iteration_point->prior_leaf;
		if (ptr->prior_leaf == NULL) return NULL;		/* this is the head guard, do not return it */
	}

	list->iteration_point = ptr;
	if (reference != NULL) *reference = ptr;	
	
	return ptr->element;
}

/* Stop a sequence of iteration of all the elements in a list.

	This function terminates an iteration sequence. The iteration sequence was
	initiated by the function list_start_iteration().
	
	For more details about iteration sequences, refer to list_start_iteration().
		
	list		list_object that has been previously initialized.
	returns	LIST_OK			the iteration sequence has been stopped
			LIST_ERR		if there was an error.									*/
int list_iteration_stop(list_object *restrict list) {

#ifdef DEBUG
	if (list__validate_list_object(list, FALSE, __func__) == FALSE)
		return LIST_ERR;
	if (! list->iteration_enabled) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called on list with no active iteration.\n\n", __func__);
		return LIST_ERR;
	}
#else
	if ((list == NULL) || (! list->iteration_enabled))
		return LIST_ERR;
#endif
	
	list->iteration_point = NULL;
	list->iteration_enabled = FALSE;
	
	return LIST_OK;
}

/* Start a sequence of callback iteration of all the elements in a list.

	This function initiates an iteration sequence. The iteration sequence will
	continue until the iterate function signals upon return with a non-zero
	return code. This requires an iterate function to have been set prior to
	calling list_callback_iterate()
		
	An iteration sequence traverses a list from the head to the tail (or the
	opposite direction if that was requested), calling the iterate function for
	each element in sequence from the list. For each element a pointer to the
	element data, a list reference to the element, the index of the element,
	and the current pivot value are provided to the iterate function. If the
	list is linear class, pivot value will always be ENTIRE_LIST (which is 0).
	
	The iterate function can do whatever processing it requires, and return with
	either a 0 return code, which means continue with the iteration sequence, or
	a non zero return code, in which case the iteration sequence ends, and this
	function returns to its original caller. The meaning of the return codes is
	entirely user dependent.
	
	During an iteration sequence all index based modifications to the list are
	blocked and denied. However index based gets and searches can occur. Also
	by design, elements can be fetched, inserted, removed and modified using the
	element references returned by various other D-list function calls. It is the
	responsibility of the user to use these features wisely. In particular unwise
	use of inserting an element by reference at a point following the current
	iteration sequence location, has the obvious potential to create an infinite
	loop situation, of adding a new element and iterating to it, and adding
	another, etc. Use these features carefully...
		
	An iteration sequence can be started on any class of list.
	
	This iteration sequence operates in the same way as the regular iteration
	above except that there is only one function call to use, and control is
	not returned to the caller until the iteration sequence is complete.
	
	list		list_object that has been previously initialized.
	direction	HEAD_OF_LIST towards the head of the list, or TAIL_OF_LIST
				towards the tail of the list.
	returns	LIST_OK			the iteration sequence has been established
			LIST_ERR		if there was an error.									*/
int list_callback_iteration(	list_object *restrict list,
								const boolean direction){

	leaf_node *ptr, *end_ptr;
	unsigned int i;
	int ret;

#ifdef DEBUG
	if (list__validate_list_object(list, FALSE, __func__) == FALSE)
		return LIST_ERR;
	if (list->iteration_enabled) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called on list with iteration already enabled.\n\n", __func__);
		return LIST_ERR;
	}
	if (list->functions.iterate == NULL) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called on list with a NULL iterate function.\n\n", __func__);
		return LIST_ERR;
	}
#else
	if ((list == NULL)
			|| (list->iteration_enabled)
			|| (list->functions.iterate == NULL))
		return LIST_ERR;
#endif

	list->iteration_direction = direction;
	list->iteration_point = NULL;					/* to stop a regular iteration from grabbing elements */
	list->iteration_enabled = TRUE;

	if (direction == TAIL_OF_LIST) {
		ptr = list->head_guard->next_leaf;
		end_ptr = list->tail_guard;
		i = 0;
	} else {
		ptr = list->tail_guard->prior_leaf;
		end_ptr = list->head_guard;
		i = list->element_count-1;
	}

	for (ret = 0; (ptr) && (ptr != end_ptr) && (ret == 0); ) {
		ret = list->functions.iterate(ptr->element, ptr, i, ptr->internal->pivot.value, list);
		if (direction == TAIL_OF_LIST) {
			ptr = ptr->next_leaf;
			++i;
		} else {
			ptr = ptr->prior_leaf;
			--i;
		}
	}
	list->iteration_enabled = FALSE;

	return ret;
}

/* Process all the elements in a list, using a user supplied function.

	This function initiates a series of calls to the user supplied process
	function, one for each element in the list, constrained by the scope. The
	function may process the elements however it wants, accumulate data, sum
	values, transform data in some user defined way. The caller may also
	provide the process function with a user data long int field, and a void
	pointer, that can be read, and/or modified by the function. This provides
	a clean MP safe mechanism for the function to communicate and return
	computed values to the original caller of list_process().
	
	This is a similar function to list_callback_iteration(), and creates a
	similar iteration sequence. However there are some major differences, using
	list_process() the function is supplied ad-hoc and can be changed for every
	call. The function has no control over the termination of the sequence, and
	the function has no access to relocate or remove elements. Additionally, in
	this function there is an MP safe mechanism provided to allow the function
	to accumulate data between calls, and safely and cleanly return significant
	amounts of data back to the caller after the element processing is over.
	
	The general purpose of this function is to allow users to easily extend the
	element statistics and transformation operations available, without having
	to either modify D-List, or make D-List aware of what kind of data is being
	stored in the lists. This feature allows users to quickly build extensive
	element computation functions with very little code.
	
	The processing sequence traverses a list from the head to the tail (or the
	opposite direction if that was requested), calling the process function for
	each element in sequence from the list, within the scope defined. For each
	element a pointer to the element data, a list reference to the element, the
	index of the element, and the current pivot value are provided to the
	process function, along with the provided optional user data pointers. If the
	list is linear class, pivot value will always be ENTIRE_LIST (which is 0).
	
	The process function can do whatever processing it requires, and when the
	element processing is completed, return with user dependent return code.
	
	This function can be thought of in different way, rather than just another
	solution to scrolling through elements in a list. It is also a method for
	processing all elements in a list with a specific function, such as if you
	wish to compute a transformation on all elements of a list holding 3D point
	values of objects, list_process() would be faster than setting up your own
	iteration sequence to perform such transformations.
	
	list		list_object that has been previously initialized.
	direction	HEAD_OF_LIST towards the head of the list, or TAIL_OF_LIST
				towards the tail of the list.
	scope		which side of the pivot to process the elements
				must be either PIVOT_MINOR, PIVOR_MAJOR, or ENTIRE_LIST
				For linear and finite class lists, must be ENTIRE_LIST
	process_function
				pointer to user supplied function to process all the elements
				in scope of the list.
	user_value	an optional long int field provided by the original caller of
				list_process() to allow the function use as needed, or NULL.
	user_data	an optional pointer to a field or a struct, that is provided
				by the original caller of list_process() to allow the
				function use as needed, or NULL.
	returns	>= 0				the process function may return a user defined
								code at the end of processing all the elements.
								Any return code >= 0 is user defined.
			ELEMENT_NOT_FOUND	the list is empty
			LIST_ERR			if there was an error.								*/
int list_process(	list_object *restrict list,
					const boolean direction,
					const int scope,
					const element_process process_function,
					long int *user_value,
					void *user_data) {

	register const leaf_node *ptr;
	register const leaf_node *next_ptr;
	register const leaf_node *end_ptr;
	long int temp_index = 0;
	int end_element = FIRST_ELEMENT;
	int ret = LIST_OK;
	register unsigned int i;
	boolean forwards = TRUE;

#ifdef DEBUG
	if (list__validate_list_object(list, FALSE, __func__) == FALSE)
		return LIST_ERR;
	if (process_function == NULL) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with NULL process function.\n\n", __func__);
		return LIST_ERR;
	}
	if (list->iteration_enabled) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"list has iteration enabled.\n\n", __func__);
		return LIST_ERR;
	}
	if (list->element_count < 1) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with empty input list.\n\n", __func__);
		return ELEMENT_NOT_FOUND;
	}
	if ((list->list_pivot_enabled == FALSE) && (scope != ENTIRE_LIST)) {
		fprintf(stderr, "\n==> Dlist Debug WARNING ==> %s() - "\
				"an ignored scope argument '%d' was provided for a linear class list.\n\n",
				__func__, scope);
	}
	if ((list->list_pivot_enabled == TRUE) && ((scope < PIVOT_MINOR) || (scope > PIVOT_MAJOR))) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"invalid scope argument '%d' was provided for a pivot class list.\n\n",
				__func__, scope);
		return LIST_ERR;
	}
#else
	if ((list == NULL)
			|| (process_function == NULL)
			|| (list->iteration_enabled)
			|| ((list->list_pivot_enabled == TRUE) && ((scope < PIVOT_MINOR) || (scope > PIVOT_MAJOR)) ))			
		return LIST_ERR;
	if (list->element_count < 1) 
		return ELEMENT_NOT_FOUND;
#endif


	/* Set up the search parameters */
	if ((list->list_pivot_enabled == TRUE) && (scope != ENTIRE_LIST)) {
		ptr = list__locate_pivot_leaf_node(list, &temp_index);
		if (direction == END_OF_LIST) {
			if (scope == PIVOT_MAJOR) {
				forwards = TRUE;
				ptr = ptr->next_leaf;					/* the first leaf node of the major side */
				end_ptr = list->tail_guard;
				i = list->pivot_point.minor;
			} else {
				forwards = FALSE;
				end_ptr = list->head_guard;				/* ptr already points the start leaf node */
				i = list->pivot_point.minor-1;
			}
		} else {
			/* direction == PIVOT_OF_LIST */
			if (scope == PIVOT_MAJOR) {
				forwards = FALSE;
				end_ptr = ptr;							/* the first leaf node of the minor side */
				ptr = list->tail_guard->prior_leaf;
				i = list->element_count - 1;
			} else {
				forwards = TRUE;
				end_ptr = ptr->next_leaf;
				ptr = list->head_guard->next_leaf;
				i = 0;
			}		
		}
	} else {
		/* entire list, no pivot relevant */
		if (direction == TAIL_OF_LIST) {
			forwards = TRUE;
			ptr = list->head_guard->next_leaf;
			end_ptr = list->tail_guard;
			i = 0;
		} else {
			forwards = FALSE;
			ptr = list->tail_guard->prior_leaf;
			end_ptr = list->head_guard;
			i = list->element_count - 1;
		}
	}

	/* Perform the search */
	if (forwards) {
		for (; ptr != end_ptr; ++i, ptr = next_ptr, end_element = OTHER_ELEMENT) {
			next_ptr = ptr->next_leaf;
			if (next_ptr == end_ptr) end_element = TRUE;
			ret = process_function(ptr->element, end_element, ptr, i, ptr->internal->pivot.value, user_value, user_data);		
		}
	} else {
		for (; ptr != end_ptr; --i, ptr = next_ptr, end_element = OTHER_ELEMENT) {
			next_ptr = ptr->prior_leaf;
			if (next_ptr == end_ptr) end_element = TRUE;
			ret = process_function(ptr->element, end_element, ptr, i, ptr->internal->pivot.value, user_value, user_data);		
		}
	}	

	return ret;
}

/* Process a range of elements in a list, indicated by index number, using a
	user supplied function.

	This function initiates a series of calls to the user supplied process
	function, one for each element in the list, constrained by the index range
	provided. The function may process the elements however it wants, accumulate
	data, sum values, transform data in some user defined way. The caller may also
	provide the process function with a user data long int field, and a void
	pointer, that can be read, and/or modified by the function. This provides
	a clean MP safe mechanism for the function to communicate and return
	computed values to the original caller of list_process_range().

	This is a similar function to list_callback_iteration(), and creates a
	similar iteration sequence. However there are some major differences, using
	list_process() the function is supplied ad-hoc and can be changed for every
	call. The function has no control over the termination of the sequence, and
	the function has no access to relocate or remove elements. Additionally, in
	this function there is an MP safe mechanism provided to allow the function
	to accumulate data between calls, and safely and cleanly return significant
	amounts of data back to the caller after the element processing is over.
	
	This function differs from list_process() by constraining the range of element
	process to start and stop index points.
	
	The general purpose of this function is to allow users to easily extend the
	element statistics and transformation operations available, without having
	to either modify D-List, or make D-List aware of what kind of data is being
	stored in the lists. This feature allows users to quickly build extensive
	element computation functions with very little code.
	
	The processing sequence traverses a list from the element identified by
	index_start to the element identified by index_stop, calling the user process
	function for each element in sequence from the list. For each element a
	pointer to the element data, a list reference to the element, the index of the
	element, and the current pivot value are provided to the process function,
	along with the provided optional user data pointers. If the list is linear
	class, pivot value will always be ENTIRE_LIST (which is 0).
	
	The process function can do whatever processing it requires, and when the
	element processing is completed, return with user dependent return code.
	
	This function can be thought of in different way, rather than just another
	solution to scrolling through elements in a list. It is also a method for
	processing all elements in a list with a specific function, such as if you
	wish to compute a transformation on all elements of a list holding 3D point
	values of objects, list_process_range() would be faster than setting up your
	own iteration sequence to perform such transformations.
	
	list		list_object that has been previously initialized.
	index_start	denotes the first element to be processed.
	index_stop	denotes the last element to be processed, including all
				elements in-between.
	process_function
				pointer to user supplied function to process all the elements
				in scope of the list.
	user_value	an optional long int field provided by the original caller of
				list_process() to allow the function use as needed, or NULL.
	user_data	an optional pointer to a field or a struct, that is provided
				by the original caller of list_process() to allow the
				function use as needed, or NULL.
	returns	>= 0				the process function may return a user defined
								code at the end of processing all the elements.
								Any return code >= 0 is user defined.
			ELEMENT_NOT_FOUND	no element was found matching the index.
			LIST_ERR			if there was an error.								*/
int list_process_range(	list_object *restrict list,
						const unsigned int index_start,
						const unsigned int index_stop,
						const element_process process_function,
						long int *user_value,
						void *user_data) {

	register leaf_node *ptr;
	unsigned int i, range;
	int ret = LIST_OK;
	int end_element = FIRST_ELEMENT;

#ifdef DEBUG
	if (list__validate_list_object(list, LIST_WRITE, __func__) == FALSE)
		return LIST_ERR;
	if (process_function == NULL) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with NULL process function.\n\n", __func__);
		return LIST_ERR;
	}
	if (list->element_count < 1) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with empty input list.\n\n", __func__);
		return ELEMENT_NOT_FOUND;
	}
	if (list->iteration_enabled) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"list has iteration enabled.\n\n", __func__);
		return LIST_ERR;
	}
	if (index_stop >= list->element_count) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with out-of-range index_stop. Index '%u', max element index is '%u'.\n\n",
				__func__, index_stop, list->element_count-1);
		return ELEMENT_NOT_FOUND;
	}
	if (index_start >= index_stop) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with invalid index range index_start > index_stop. index_start '%u' "\
				"index_stop '%u', current max element index is '%u'.\n\n",
				__func__, index_start, index_stop, list->element_count-1);
		return ELEMENT_NOT_FOUND;
	}
#else	
	if ((list == NULL)
			|| (process_function == NULL)
			|| (list->iteration_enabled))
		return LIST_ERR;
	if ((index_start >= index_stop) 
			|| (list->element_count < 1) 
			|| (index_stop >= list->element_count))
		return ELEMENT_NOT_FOUND;
#endif
	
	range = index_stop - index_start;
	ptr = list__locate_leaf_node(list, (long int)index_start);
	if (ptr == NULL) {
#ifdef DEBUG
		fprintf(stderr, "\n==> Dlist INTERNAL ERROR ==> %s() failed - null leaf node pointer "\
				"returned for element index '%u' list elements '%u'.\n\n", __func__, index_start,
				list->element_count);
#endif
		return LIST_ERR;
	}
	
	for (i = 0; i <= range; ++i, ptr = ptr->next_leaf, end_element = OTHER_ELEMENT) {
		if (index_start + i == index_stop) end_element = LAST_ELEMENT;
		ret = process_function(ptr->element, end_element, ptr, index_start+i, ptr->internal->pivot.value, user_value, user_data);		
	}
	
	return ret;
}

/* Search for an element in a list, using maximum quantity as the search criteria.

	This function traverses the list, for the distance specified in scope,
	searching for an element with the matching the notion of maximum. It will
	return a pointer to the element data, the index number in the list of the
	element, and a list reference to the element. The user may choose to use any
	of these methods to reacquire, remove or manipulate the element, or data.
	Please bear in mind that the index is only valid until the list is modified
	by adding or removing elements. The reference remains valid until the element
	itself is removed from the list.
	
	The argument scope determines how far along the list this function will
	traverse looking for a maximum quantity. Providing ENTIRE_LIST as this argument
	will cause the function to traverse the list from head to tail. However, if the
	list is of a pivot class, then providing either PIVOT_MAJOR or PIVOT_MINOR will
	optionally cause this function to confine its search to that side of the list
	pivot point. If the list is of the linear class, then the setting of this
	argument is ignored. This feature is useful if the comparison function tests
	different data points, than the set pivot function. It allows for lists to be
	easily used in a multidimensional way with simple code.
	
	A comparison function must have been previously set for this list (by calling
	list_set_compare_function() ), this function will use that compare function to
	compare the element data looking for the largest item. What constitutes maximum
	is implementation dependent for the compare function and the data stored in the
	list by the user.
	
	The definitive test, on return, to verify if an element was found, is to check
	the return pointer for either NULL or an address.
	
	This function only returns a pointer, index and reference to the element
		in the list, it makes no changes to the list.

	list		list_object that has been previously initialized.
	scope		which side of the pivot to search for the maximum element
				must be either PIVOT_MINOR, PIVOR_MAJOR, or ENTIRE_LIST
				For linear and finite class lists, must be ENTIRE_LIST
	reference	pointer to variable used to store the element reference,
		 		if found. NULL, no element reference is to be returned.
	index		pointer to variable used to store the element index, if
		 		found. Or NULL, no element index is to be returned.
	returns	if an element was found, a pointer to element, and if requested an
					element index in list, and the element reference are returned
			if an element was not found (i.e. the list was empty), a NULL
					pointer, index of 0, and NULL element reference.				*/
void *list_locate_maximum(	const list_object *restrict list,
							const int scope,
							element_reference **restrict reference,
							unsigned int *index) {

	leaf_node *ptr = NULL;
	unsigned int i = 0;

	/* set default return state in case of errors, there is to be no ambiguity of success */
	if (reference != NULL) *reference = NULL;
	if (index != NULL) *index = 0;

#ifdef DEBUG
	if (list__validate_list_object(list, FALSE, __func__) == FALSE)
		return NULL;
	if (list->functions.compare == NULL) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"NULL compare function.\n\n", __func__);
		return NULL;
	}
	if ((list->list_pivot_enabled == FALSE) && (scope != ENTIRE_LIST)) {
		fprintf(stderr, "\n==> Dlist Debug WARNING ==> %s() - "\
				"an ignored scope argument '%d' was provided for a linear class list.\n\n",
				__func__, scope);
	}
	if ((list->list_pivot_enabled == TRUE) && ((scope < PIVOT_MINOR) || (scope > PIVOT_MAJOR))) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"invalid scope argument '%d' was provided for a pivot class list.\n\n",
				__func__, scope);
		return NULL;
	}
#else
	if ((list == NULL)
			|| (list->functions.compare == NULL)
			|| ((list->list_pivot_enabled == TRUE) && ((scope < PIVOT_MINOR) || (scope > PIVOT_MAJOR)) ))			
		return NULL;
#endif

	if ((list->list_pivot_enabled == TRUE) && (scope != ENTIRE_LIST)) {
		ptr = list__locate_extremum_leaf_node_pivot(list, &i, LIST_ASCENDING, scope);	
	} else {
		ptr = list__locate_extremum_leaf_node(list, &i, LIST_ASCENDING);
	}

	if (ptr == NULL) return NULL;						/* empty list */

	/* return identity of found list element */
	if (reference != NULL) *reference = ptr;
	if (index != NULL) *index = i;
	return ptr->element;
}

/* Search for an element in a list, using minimum quantity as the search criteria.

	This function traverses the list, for the distance specified in scope,
	searching for an element with the matching the notion of minimum. It will
	return a pointer to the element data, the index number in the list of the
	element, and a list reference to the element. The user may choose to use any
	of these methods to reacquire, remove or manipulate the element, or data.
	Please bear in mind that the index is only valid until the list is modified
	by adding or removing elements. The reference remains valid until the element
	itself is removed from the list.
	
	The argument scope determines how far along the list this function will
	traverse looking for a minimum quantity. Providing ENTIRE_LIST as this argument
	will cause the function to traverse the list from head to tail. However, if the
	list is of a pivot class, then providing either PIVOT_MAJOR or PIVOT_MINOR will
	optionally cause this function to confine its search to that side of the list
	pivot point. If the list is of the linear class, then the setting of this
	argument is ignored. This feature is useful if the comparison function tests
	different data points, than the set pivot function. It allows for lists to be
	easily used in a multidimensional way with simple code.
	
	The definitive test, on return, to verify if an element was found, is to check
	the return pointer for either NULL or an address.
	
	This function only returns a pointer, index and reference to the element
		in the list, it makes no changes to the list.

	list		list_object that has been previously initialized.
	scope		which side of the pivot to search for the maximum element
				must be either PIVOT_MINOR, PIVOR_MAJOR, or ENTIRE_LIST
				For linear and finite class lists, must be ENTIRE_LIST
	reference	pointer to variable used to store the element reference,
		 		if found. NULL, no element reference is to be returned.
	index		pointer to variable used to store the element index, if
		 		found. Or NULL, no element index is to be returned.
	returns	if an element was found, a pointer to element, and if requested an
					element index in list, and the element reference are returned
			if an element was not found (i.e. the list was empty), a NULL
					pointer, index of 0, and NULL element reference.				*/
void *list_locate_minimum(	const list_object *restrict list,
							const int scope,
							element_reference **restrict reference,
							unsigned int *index) {

	leaf_node *ptr = NULL;
	unsigned int i = 0;

	/* set default return state in case of errors, there is to be no ambiguity of success */
	if (reference != NULL) *reference = NULL;
	if (index != NULL) *index = 0;

#ifdef DEBUG
	if (list__validate_list_object(list, FALSE, __func__) == FALSE)
		return NULL;
	if (list->functions.compare == NULL) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"NULL compare function.\n\n", __func__);
		return NULL;
	}
	if ((list->list_pivot_enabled == FALSE) && (scope != ENTIRE_LIST)) {
		fprintf(stderr, "\n==> Dlist Debug WARNING ==> %s() - "\
				"an ignored scope argument '%d' was provided for a linear class list.\n\n",
				__func__, scope);
	}
	if ((list->list_pivot_enabled == TRUE) && ((scope < PIVOT_MINOR) || (scope > PIVOT_MAJOR))) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"invalid scope argument '%d' was provided for a pivot class list.\n\n",
				__func__, scope);
		return NULL;
	}
#else
	if ((list == NULL)
			|| (list->functions.compare == NULL)
			|| ((list->list_pivot_enabled == TRUE) && ((scope < PIVOT_MINOR) || (scope > PIVOT_MAJOR)) ))			
		return NULL;
#endif
	
	if ((list->list_pivot_enabled == TRUE) && (scope != ENTIRE_LIST)) {
		ptr = list__locate_extremum_leaf_node_pivot(list, &i, LIST_DESCENDING, scope);	
	} else {
		ptr = list__locate_extremum_leaf_node(list, &i, LIST_DESCENDING);
	}

	if (ptr == NULL) return NULL;						/* empty list */

	/* return identity of found list element */
	if (reference != NULL) *reference = ptr;
	if (index != NULL) *index = i;
	return ptr->element;
}

/*	Sorted list function notes

	I was originally going to try and optimize these functions using more
	index hints and randomize the direction from a hint i started looking
	for an element prior to some some action. However after modeling the
	code and its performance, i decided that a simpler approach was largely
	the same in performance (when the datums were of a widely dispersed
	random grouping) and the code was much easier read and maintain. D-List
	was designed to optimize for widely dispersed non-recurring random
	element datums, not for largely coherent, or duplicated grouping datums.
	
	Currently all the only list_sorted_xx() use a binary split of the mid_point.
	The reason is that all the insert and removal primitives under the covers
	need to know the index # of the leaf node being removed or added.
	Currently no facility exists to either maintain or quickly find it under
	these circumstances. It can be easily found by using other list primitives
	but that would require a leaf node chain walk, which would totally negate
	any speed advantage of using sorted lists or optimized sub binary
	searches. The code overhead of maintaining other index counters to
	optimize these functions drops the performance of the other basic high
	usage function calls and is not acceptable to improve lower usage functions.
	
*/


/* Search for an element in an already sorted list, using datum as the search
	criteria.

	This function will perform a binary search of a sorted list, searching for
	an element with the matching datum supplied. It will only find the first
	search match in the sorted order of the list, and if found will return a
	pointer to the element data, and an optional list reference to the found
	element. The reference remains valid until the element itself is removed
	from the list. An optional index to the element can also be returned.

	This function is available for finite class lists, with the exception that
	a reference will not be returned regardless of the reference argument. 
	
	This function is available for pivot class lists, first the pivot function
	set for the list is called using the datum, to determine which side of the
	list to operate on, then the function proceeds constrained to that side of
	the pivot list. If called for a pivot list, a pivot function must have been
	previously set for this list.
	
	A compare function must have been previously set for this list (by calling
	list_set_compare_function() ), this find operation will use the set compare
	function to compare the element data against the supplied datum. It is also
	assumed that the compare function used to sort the list, ordered it using
	the same data points that the current compare function uses. If not, the
	results are undefined.
	
	This function is only available for lists that are currently sorted. The 
	definitive test, on return, to verify if an element was found, is to check 
	the return pointer for either NULL or an address.
	
	This function only returns a pointer, index and reference to the element
	in the list, it makes no changes to the list.

	list		list_object that has been previously initialized.
	datum 		data to be used for the search criteria.
	reference	pointer to variable used to store the element reference,
		 		if found. NULL, no element reference is to be returned.
	index		pointer to variable used to store the element index, if
 		 		found. Or NULL, no element index is to be returned.
	returns	PTR		if an element was found matching the datum, a pointer to
					the element, and if requested the element reference and
					the element index.
			NULL	if an element was not found matching the datum, and NULL
					element reference.												*/
void *list_sorted_find(	const list_object *restrict list,
						const void *datum,
						element_reference **restrict reference,
						unsigned int *ret_index) {

	const leaf_node *ptr, *pivot_ptr;
	long int temp_index = 0;
	int scope;
	unsigned int index;
	
	/* set default return state in case of errors, there is to be no ambiguity of success */
	if (reference != NULL) *reference = NULL;
	if (ret_index != NULL) *ret_index = 0;

#ifdef DEBUG
	if (list__validate_list_object(list, FALSE, __func__) == FALSE)
		return NULL;
	if ((list->list_finite_enabled == TRUE) && (reference != NULL)) {
		fprintf(stderr, "\n==> Dlist Debug WARNING ==> %s() - "\
				"called on a finite class list with a non NULL reference argument. "\
				"NO reference will be returned.\n\n", __func__);
	}
	if (datum == NULL) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with NULL datum pointer.\n\n", __func__);
		return NULL;
	}
	if (list->functions.compare == NULL) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"required compare function is NULL.\n\n", __func__);
		return NULL;
	}
	if (list->list_sorted == FALSE) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"list is not sorted.\n\n", __func__);
		return NULL;
	}
	if ((list->list_pivot_enabled == TRUE) && (list->functions.pivot == NULL)) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called for pivot list with NULL pivot function.", __func__);
		return NULL;
	}
#else
	if ((list == NULL)
			|| (list->functions.compare == NULL)
			|| (list->list_sorted == FALSE)
			|| (datum == NULL)
			||((list->list_pivot_enabled == TRUE) && (list->functions.pivot == NULL)))			
		return NULL;
#endif

	if (list->element_count < 1) return NULL;

	/* Perform the search */
	if (list->list_pivot_enabled) {
		pivot_ptr = list__locate_pivot_leaf_node(list, &temp_index);
		scope = list->functions.pivot(datum);
		if (scope > 0) {
			if (list->pivot_point.major > 0) {
				ptr = list__find_sorted_element(list, 
												datum, 
												&index, 
												pivot_ptr->next_leaf, 
												list->tail_guard->prior_leaf, 
												list->pivot_point.minor, 
												list->pivot_point.major);
			} else {
				return NULL;
			}
		} else { 
			if (list->pivot_point.minor > 0) {
				ptr = list__find_sorted_element(list, 
												datum, 
												&index, 
												list->head_guard->next_leaf, 
												pivot_ptr, 
												0, 
												list->pivot_point.minor);
			} else {
				return NULL;
			}
		}
	} else {
		/* not a pivot class list, search the whole list */
		ptr = list__find_sorted_element(list, 
										datum, 
										&index, 
										list->head_guard->next_leaf, 
										list->tail_guard->prior_leaf, 
										0, list->element_count);
	}

		
	if (ptr == NULL) return NULL;								/* No match found */
	
	/* return identity of found list element */
	if ((reference != NULL) && (! list->list_finite_enabled)) *reference = (leaf_node *)ptr;
	if (ret_index != NULL) *ret_index = index;
	return ptr->element;
}

/* Insert a new element in an already sorted list, in correct sorted order,
	using datum as the search criteria to find the position.

	This function will perform a binary search of a sorted list, searching for
	the correct position to insert the new element, using the supplied datum.
	It will insert the element between other elements that lower and higher in
	sort order. If there are equal elements, the new element will be placed at
	the end of the equivalent elements, and before the first element that is
	higher in sort order.

	This function is available for finite class lists, however it is not
	generally recommended that is is used on them, as there exists the real
	possibility that if the new element happens to be placed in the last element
	position, and the list is currently full, the new element will be immediately
	deleted from the list. Should you choose to use this function on a finite
	class list please pay particular attention to the number of elements that are
	currently in the list, in relation to the maximum capacity of that list.
	
	This function is available for pivot class lists, first the pivot function
	set for the list is called using the datum, to determine which side of the
	list to operate on, then the function proceeds constrained to that side of
	the pivot list. If called for a pivot list, a pivot function must have been
	previously set for this list.
	
	A compare function must have been previously set for this list (by calling
	list_set_compare_function() ), this insert operation will use the set
	compare function to compare the element data against the supplied datum. It
	is also assumed that the compare function used to sort the list, ordered it
	using the same data points that the current compare function uses. If not,
	the results are undefined.
	
	This function makes changes to the list, but retains its sorted order.

	list		list_object that has been previously initialized.
	element 	new element data, also used for the search criteria.
	returns	LIST_OK				success, new element added to the list
			ELEMENT_NO_DATUM	element data not supplied
			LIST_UNSORTED		function was called for an unsorted list
			LIST_ERR			on error	 										*/
int list_sorted_insert(	list_object *restrict list,
						const void *element) {

	leaf_node *ptr;
	leaf_node *pivot_ptr;
	long int temp_index = 0;
	int scope = ENTIRE_LIST;	
	int ret;
	unsigned int index;

#ifdef DEBUG
	if (list__validate_list_object(list, FALSE, __func__) == FALSE)
		return LIST_ERR;
	if (element == NULL) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with NULL datum pointer.\n\n", __func__);
		return ELEMENT_NO_DATUM;
	}
	if (list->functions.compare == NULL) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"required compare function is NULL.\n\n", __func__);
		return LIST_ERR;
	}
	if (list->list_sorted == FALSE) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"list is not sorted.\n\n", __func__);
		return LIST_UNSORTED;
	}
	if ((list->list_pivot_enabled == TRUE) && (list->functions.pivot == NULL)) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called for pivot list with NULL pivot function.", __func__);
		return LIST_ERR;
	}
#else
	if ((list == NULL)
			|| (list->functions.compare == NULL)
			||((list->list_pivot_enabled == TRUE) && (list->functions.pivot == NULL)))			
		return LIST_ERR;
	if (list->list_sorted == FALSE)
		return LIST_UNSORTED;
	if (element == NULL)
		return ELEMENT_NO_DATUM;
#endif

	/* Perform the search */
	if (list->list_pivot_enabled) {
		pivot_ptr = list__locate_pivot_leaf_node(list, &temp_index);
		scope = list->functions.pivot(element);
		if (list->element_count < 1) {
			ptr = list->head_guard;
			index = 0;
		} else {
			if (scope > 0) {
				if (list->pivot_point.major > 0) {
					ptr = list__find_sorted_insert(	list, 
													element, 
													&index, 
													pivot_ptr->next_leaf, 
													list->tail_guard->prior_leaf, 
													list->pivot_point.minor, 
													list->pivot_point.major);
				} else {
					ptr = list->tail_guard->prior_leaf;
					index = list->element_count - 1;
				}
			} else {
				if (list->pivot_point.minor > 0) {
					ptr = list__find_sorted_insert(	list, 
													element, 
													&index, 
													list->head_guard->next_leaf, 
													pivot_ptr, 
													0, 
													list->pivot_point.minor);
				} else {
					ptr = list->head_guard;
					index = 0;
				}
			}
		}


	} else {
		/* not a pivot class list, search the whole list */
		if (list->element_count < 1) {
			ptr = list->head_guard;
			index = 0;
		} else {

			ptr = list__find_sorted_insert(	list, 
											element, 
											&index, 
											list->head_guard->next_leaf, 
											list->tail_guard->prior_leaf, 
											0, list->element_count);
		}
	}

	ret = list__insert_leaf_node(list, ptr, element, index, TRUE, scope);

	if ((ret == LIST_OK) && (list->functions.insert)) list->functions.insert(ptr->prior_leaf->element);

	return LIST_OK;
}

/* Extract an element from an already sorted list, using datum as the search
	criteria.

	This function will perform a binary search of a sorted list, searching for
	an element with the matching datum supplied. It will only find the first
	search match in the sorted order of the list, and if found will delete the
	element, remove its internal resources, and return a pointer to the element
	data.

	This function is available for pivot class lists, first the pivot function
	set for the list is called using the datum, to determine which side of the
	list to operate on, then the function proceeds constrained to that side of
	the pivot list. If called for a pivot list, a pivot function must have been
	previously set for this list.
	
	A compare function must have been previously set for this list (by calling
	list_set_compare_function() ), this find operation will use the set compare
	function to compare the element data against the supplied datum. It is also
	assumed that the compare function used to sort the list, ordered it using
	the same data points that the current compare function uses. If not, the
	results are undefined.
	
	The definitive test, on return, to verify if an element was found, is to
	check the return pointer for either NULL or an address.

	This function is only available for lists that are currently sorted. 

	list		list_object that has been previously initialized.
	datum 		data to be used for the search criteria.
	returns		NULL no element was available in list
				PTR to element data. It is the users responsibility to manage
					this data, and free it when finished with it. If the user
					supplied it originally it is the same allocated memory
					originally given to D-List. If the list was created with
					the LIST_COPY_DATA attribute, the data was allocated
					internally by D-List. In both cases it now belongs to the
					caller.															*/
void *list_sorted_extract(	list_object *restrict list,
							const void *datum) {

	leaf_node *ptr;
	leaf_node *pivot_ptr;
	long int temp_index = 0;
	int scope;	
	unsigned int index;
	void *element;

#ifdef DEBUG
	if (list__validate_list_object(list, FALSE, __func__) == FALSE)
		return NULL;
	if (datum == NULL) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with NULL datum pointer.\n\n", __func__);
		return NULL;
	}
	if (list->functions.compare == NULL) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"required compare function is NULL.\n\n", __func__);
		return NULL;
	}
	if (list->list_sorted == FALSE) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"list is not sorted.\n\n", __func__);
		return NULL;
	}
	if ((list->list_pivot_enabled == TRUE) && (list->functions.pivot == NULL)) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called for pivot list with NULL pivot function.", __func__);
		return NULL;
	}
#else
	if ((list == NULL)
			|| (list->functions.compare == NULL)
			|| (list->list_sorted == FALSE)
			|| (datum == NULL)
			||((list->list_pivot_enabled == TRUE) && (list->functions.pivot == NULL)))			
		return NULL;
#endif

	if (list->element_count < 1) return NULL;

	/* Perform the search */
	if (list->list_pivot_enabled) {
		pivot_ptr = list__locate_pivot_leaf_node(list, &temp_index);
		scope = list->functions.pivot(datum);
		if (scope > 0) {
			if (list->pivot_point.major > 0) {
				ptr = list__find_sorted_element(list, 
												datum, 
												&index, 
												pivot_ptr->next_leaf, 
												list->tail_guard->prior_leaf, 
												list->pivot_point.minor, 
												list->pivot_point.major);
			} else {
				return NULL;
			}
		} else { 
			if (list->pivot_point.minor > 0) {
				ptr = list__find_sorted_element(list, 
												datum, 
												&index, 
												list->head_guard->next_leaf, 
												pivot_ptr, 
												0, 
												list->pivot_point.minor);
			} else {
				return NULL;
			}
		}
	} else {
		/* not a pivot class list, search the whole list */
		ptr = list__find_sorted_element(list, 
										datum, 
										&index, 
										list->head_guard->next_leaf, 
										list->tail_guard->prior_leaf, 
										0, list->element_count);
	}

	if (ptr == NULL) return NULL;							/* No match found */

	/* Remove the element */
	element = ptr->element;

	if (list->functions.remove) list->functions.remove(ptr->element);
	ptr->element = NULL;					/* prevent underlying primitives from freeing the resource */

	list__remove_leaf_node(ptr, index);

#ifdef DLIST_DEBUG
	assert(list__audit(list) == LIST_OK);
#endif		/* DLIST_DEBUG */
	
	return element;
}

/* Delete an element from an already sorted list, using datum as the search
	criteria.

	This function will perform a binary search of a sorted list, searching for
	an element with the matching datum supplied. It will only find the first
	search match in the sorted order of the list, and if found will delete the
	element, remove its internal resources, and delete the element data.

	This function is available for pivot class lists, first the pivot function
	set for the list is called using the datum, to determine which side of the
	list to operate on, then the function proceeds constrained to that side of
	the pivot list. If called for a pivot list, a pivot function must have been
	previously set for this list.

	A compare function must have been previously set for this list (by calling
	list_set_compare_function() ), this find operation will use the set compare
	function to compare the element data against the supplied datum. It is also
	assumed that the compare function used to sort the list, ordered it using
	the same data points that the current compare function uses. If not, the
	results are undefined.

	This function is only available for lists that are currently sorted. 

	list		list_object that has been previously initialized.
	datum 		data to be used for the search criteria.
	returns	LIST_OK				success, element deleted from the list
			ELEMENT_NO_DATUM	element data not supplied
			LIST_UNSORTED		function was called for an unsorted list
			ELEMENT_NOT_FOUND	no element was found matching the datum
			LIST_ERR			on error	 										*/
int list_sorted_delete(	list_object *restrict list,
						const void *datum) {

	leaf_node *ptr;
	leaf_node *pivot_ptr;
	long int temp_index = 0;
	int scope;	
	unsigned int index;

#ifdef DEBUG
	if (list__validate_list_object(list, FALSE, __func__) == FALSE)
		return LIST_ERR;
	if (datum == NULL) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with NULL datum pointer.\n\n", __func__);
		return ELEMENT_NO_DATUM;
	}
	if (list->functions.compare == NULL) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"required compare function is NULL.\n\n", __func__);
		return LIST_ERR;
	}
	if (list->list_sorted == FALSE) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"list is not sorted.\n\n", __func__);
		return LIST_UNSORTED;
	}
	if ((list->list_pivot_enabled == TRUE) && (list->functions.pivot == NULL)) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called for pivot list with NULL pivot function.", __func__);
		return LIST_ERR;
	}
#else
	if ((list == NULL)
			|| (list->functions.compare == NULL)
			||((list->list_pivot_enabled == TRUE) && (list->functions.pivot == NULL)))			
		return LIST_ERR;
	if (list->list_sorted == FALSE)
		return LIST_UNSORTED;
	if (datum == NULL)
		return ELEMENT_NO_DATUM;
#endif

	if (list->element_count < 1) return ELEMENT_NOT_FOUND;

	/* Perform the search */
	if (list->list_pivot_enabled) {
		pivot_ptr = list__locate_pivot_leaf_node(list, &temp_index);
		scope = list->functions.pivot(datum);
		if (scope > 0) {
			if (list->pivot_point.major > 0) {
				ptr = list__find_sorted_element(list, 
												datum, 
												&index, 
												pivot_ptr->next_leaf, 
												list->tail_guard->prior_leaf, 
												list->pivot_point.minor, 
												list->pivot_point.major);
			} else {
				return ELEMENT_NOT_FOUND;
			}
		} else { 
			if (list->pivot_point.minor > 0) {
				ptr = list__find_sorted_element(list, 
												datum, 
												&index, 
												list->head_guard->next_leaf, 
												pivot_ptr, 
												0, 
												list->pivot_point.minor);
			} else {
				return ELEMENT_NOT_FOUND;
			}
		}
	} else {
		/* not a pivot class list, search the whole list */
		ptr = list__find_sorted_element(list, 
										datum, 
										&index, 
										list->head_guard->next_leaf, 
										list->tail_guard->prior_leaf, 
										0, list->element_count);
	}

	if (ptr == NULL) return ELEMENT_NOT_FOUND;							/* No match found */

	/* Delete the element */
	if (list->functions.remove) list->functions.remove(ptr->element);

	list__remove_leaf_node(ptr, index);

#ifdef DLIST_DEBUG
	assert(list__audit(list) == LIST_OK);
#endif		/* DLIST_DEBUG */
	
	return LIST_OK;
}

/* Expunge all elements from an already sorted list, using datum as the search
	criteria.

	This function will perform a binary search of a sorted list, searching for
	all instances of the element with the matching datum supplied. If found it
	will delete all the instances of element, remove their internal resources,
	and delete all the element data.

	This function is available for pivot class lists, first the pivot function
	set for the list is called using the datum, to determine which side of the
	list to operate on, then the function proceeds constrained to that side of
	the pivot list. If called for a pivot list, a pivot function must have been
	previously set for this list.

	A compare function must have been previously set for this list (by calling
	list_set_compare_function() ), this find operation will use the set compare
	function to compare the element data against the supplied datum. It is also
	assumed that the compare function used to sort the list, ordered it using
	the same data points that the current compare function uses. If not, the
	results are undefined.

	This function is only available for lists that are currently sorted. 

	list		list_object that has been previously initialized.
	datum 		data to be used for the search criteria.
	returns	>0					number of elements found and removed matching
								the datum
			0					no element was found matching the datum
			ELEMENT_NO_DATUM	element data not supplied
			LIST_UNSORTED		function was called for an unsorted list
			LIST_ERR			on error	 										*/
int list_sorted_expunge(	list_object *restrict list,
							const void *datum) {

	register leaf_node *ptr;
	register const leaf_node *end_ptr;
	leaf_node *pivot_ptr;
	long int temp_index = 0;
	int scope;	
	unsigned int index, count;
	int direction;
	int (*element_compare)(const void *element, const void *key);

#ifdef DEBUG
	if (list__validate_list_object(list, FALSE, __func__) == FALSE)
		return LIST_ERR;
	if (datum == NULL) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with NULL datum pointer.\n\n", __func__);
		return ELEMENT_NO_DATUM;
	}
	if (list->functions.compare == NULL) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"required compare function is NULL.\n\n", __func__);
		return LIST_ERR;
	}
	if (list->list_sorted == FALSE) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"list is not sorted.\n\n", __func__);
		return LIST_UNSORTED;
	}
	if ((list->list_pivot_enabled == TRUE) && (list->functions.pivot == NULL)) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called for pivot list with NULL pivot function.", __func__);
		return LIST_ERR;
	}
#else
	if ((list == NULL)
			|| (list->functions.compare == NULL)
			||((list->list_pivot_enabled == TRUE) && (list->functions.pivot == NULL)))			
		return LIST_ERR;
	if (list->list_sorted == FALSE)
		return LIST_UNSORTED;
	if (datum == NULL)
		return ELEMENT_NO_DATUM;
#endif

	if (list->element_count < 1) return 0;

	element_compare = list->functions.compare;
	if (list->sorted_order == LIST_ASCENDING) direction = 1;
	else direction = -1;
	count = 0;

	/* Perform the search */
	if (list->list_pivot_enabled) {
		pivot_ptr = list__locate_pivot_leaf_node(list, &temp_index);
		scope = list->functions.pivot(datum);
		if (scope > 0) {
			if (list->pivot_point.major > 0) {
				ptr = list__find_sorted_element(list, 
												datum, 
												&index, 
												pivot_ptr->next_leaf, 
												list->tail_guard->prior_leaf, 
												list->pivot_point.minor, 
												list->pivot_point.major);
			} else {
				return 0;
			}
		} else { 
			if (list->pivot_point.minor > 0) {
				ptr = list__find_sorted_element(list, 
												datum, 
												&index, 
												list->head_guard->next_leaf, 
												pivot_ptr, 
												0, 
												list->pivot_point.minor);
			} else {
				return 0;
			}
		}
	} else {
		/* not a pivot class list, search the whole list */
		ptr = list__find_sorted_element(list, 
										datum, 
										&index, 
										list->head_guard->next_leaf, 
										list->tail_guard->prior_leaf, 
										0, list->element_count);
	}

	if (ptr == NULL) return 0;					/* No match found */

	/* Expunge the elements */
	end_ptr = list->tail_guard;
	do {
		if ((*element_compare)(ptr->element, datum) * direction != 0) break;
		if (list->functions.remove) list->functions.remove(ptr->element);
		ptr = ptr->next_leaf;
		list__remove_leaf_node(ptr->prior_leaf, index);
		++count;
	} while (ptr != end_ptr);


		
#ifdef DLIST_DEBUG
	assert(list__audit(list) == LIST_OK);
#endif		/* DLIST_DEBUG */
	
	return count;
}



/* Sort all the elements in a list, using either ascending or descending order.

	This function traverses a list from head to tail, sorting all the elements
	in the order specified by the argument order, constrained by the argument 
	scope.
	
	The argument scope determines how much of the list this function will
	traverse sorting the elements. Setting this argument to ENTIRE_LIST will
	cause the function to sort the entire list from head to tail. For finite
	and linear class lists, any other setting for scope is ignored. 
	
	For pivot class lists there are 3 possible settings. Providing either 
	PIVOT_MAJOR or PIVOT_MINOR will cause this function to confine its sort to 
	that side of the list pivot point and leave the other side alone. Setting 
	scope to ENTIRE_LIST will cause this function to independently sort both 
	sides of the pivot in the order wished. It would be the same as calling this 
	function twice, once with PIVOT_MINOR, and once with PIVOT_MAJOR, except
	that by doing it in one operation the list is also marked as sorted, so is
	available for any of the Sorted List Element Functions. 
	
	Using ENTIRE_LIST on a pivot class list will sort the minor and major sides 
	of the list in the same order. For example, if the order was LIST_ASCENDING 
	then the minor elements would be ordered ascending from the head of the 
	list to the pivot point, and then the major elements would be ordered in 
	continuing ascending order from the pivot point to the tail of the list. If 
	you wish a different ordering scheme, please call this function twice once 
	for each side of the pivot with the ordering scheme you wish. However this 
	will not leave the list marked as sorted, so it will not be available to any 
	of the Sorted List Element Functions, as they all expect the ordering scheme
	to be consistent across a list.
	
	If you wish to sort a pivot class list while using data points that are 
	different from the pivot arraignment, or totally ignoring the pivot list 
	structure and rearrange the elements differently, first transfer or copy 
	the pivot class list into a linear class list, and then sort the elements 
	in that destination list anyway you wish. An alternative approach is to
	call this function twice once for each side of the pivot, and use a compare
	function that differs in data points from the pivot function currently set 
	for the list.

	A compare function must have been previously set for this list (by calling
	list_set_compare_function() ), this function uses that compare function to
	compare the element data looking for relative order of those elements. What
	constitutes greater or lesser for each element, is implementation dependent
	for the compare function, and the type of data the user stores in the list.

	This function extensively modifies the list. All previous indexes that may
	have been saved by the user are invalid. However all element references are
	still valid and can be used until the element has been removed from the
	list. Additionally if this is a pivot class list, all the internal pivot
	data is retained.
	
	Unlike the other sort functions provided by D-List, this function allocates
	additional memory resources to perform the sort operations, which is then
	released prior to return. For most users this is not a concern, and this
	function should be the sort of general use. However for systems with very
	tight memory requirements, such as low ram real memory embedded systems,
	one of the other sorts may be a better choice, if there is no memory left.
	
	This sort function is the best overall choice for sorting, it sorts lists at
	very high speeds and maintains all existing element references. The faster
	sort function, list_fast_sort() is about 15% faster than this sort, but will
	destroy all existing element references that may have been retained by the
	user. If element references are not being used, or stored, then the best sort
	to use is list_fast_sort(), otherwise for speed and retention this sort is
	the next choice.

	list		list_object that has been previously initialized.
	order		either LIST_ASCENDING or LIST_DESCENDING to denote which
		 		order to sort the elements in the list.
	scope		what section the list to sort the elements. Must be ENTIRE_LIST
				for linear or finite class lists, or may be either PIVOT_MINOR,
				PIVOR_MAJOR, or ENTIRE_LIST for pivot class lists
	returns	LIST_OK			the sort sequence was successful
			LIST_ERR		if there was an error.									*/
int list_sort(	list_object *restrict list,
				const boolean order,
				const int scope) {

	leaf_node *ptr, *end_ptr;
	long index = 0;
	int direction;
	
#ifdef DEBUG
	if (list__validate_list_object(list, FALSE, __func__) == FALSE)
		return LIST_ERR;
	if (list->iteration_enabled) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"list has iteration enabled.\n\n", __func__);
		return LIST_ERR;
	}
	if (list->functions.compare == NULL) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"NULL compare function.\n\n", __func__);
		return LIST_ERR;
	}
	if ((list->list_pivot_enabled == FALSE) && (scope != ENTIRE_LIST)) {
		fprintf(stderr, "\n==> Dlist Debug WARNING ==> %s() - "\
				"an ignored scope argument '%d' was provided for a linear class list.\n\n",
				__func__, scope);
	}
	if ((list->list_pivot_enabled == TRUE) && ((scope < PIVOT_MINOR) || (scope > PIVOT_MAJOR))) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"invalid scope argument '%d' was provided for a pivot class list.\n\n",
				__func__, scope);
		return LIST_ERR;
	}
#else
	if ((list == NULL)
			|| (list->iteration_enabled)
			|| (list->functions.compare == NULL)
			|| ((list->list_pivot_enabled == TRUE) && ((scope < PIVOT_MINOR) || (scope > PIVOT_MAJOR)) ))			
		return LIST_ERR;
#endif

	if (order == LIST_ASCENDING) direction = 1;
	else direction = -1;
	
	if (list->element_count < 2) {
		list->list_sorted = TRUE;
		list->sorted_order = order;
		return LIST_OK;
	}
	
	if (list->list_pivot_enabled == TRUE) {
		if (scope == ENTIRE_LIST) {
			list_sort(list, order, PIVOT_MINOR);
			list_sort(list, order, PIVOT_MAJOR);
			list->list_sorted = TRUE;

#ifdef DLIST_DEBUG
			assert(list__audit(list) == LIST_OK);
#endif		/* DLIST_DEBUG */
			return LIST_OK;
		} else {
			if (scope == PIVOT_MAJOR) {
				if (list->pivot_point.major < 2) return LIST_OK;
			}
			if (scope == PIVOT_MINOR) {
				if (list->pivot_point.minor < 2) return LIST_OK;
			}
		}
	}

	list->mid_point = NULL;

	if ((list->list_pivot_enabled == TRUE) && (scope != ENTIRE_LIST)) {
		ptr = list__locate_pivot_leaf_node(list, &index);
		if (scope == PIVOT_MAJOR) {
			ptr = ptr->next_leaf;							/* the first leaf node of the major side */
			end_ptr = list->tail_guard->prior_leaf;
		} else {
			end_ptr = ptr;
			ptr = list->head_guard->next_leaf;
		}
	} else {
		ptr = list->head_guard->next_leaf;
		end_ptr = list->tail_guard->prior_leaf;
	}

	list__sort_quicker_sort(list, direction, ptr, end_ptr, FALSE);

	list__full_optimize_index(list, TRUE);
	list->sorted_order = order;

#ifdef DLIST_DEBUG
	assert(list__audit(list) == LIST_OK);
#endif		/* DLIST_DEBUG */
	
	return LIST_OK;
}

/* Sort all the elements in a list, using either ascending or descending order
	in a guaranteed safe order.

	This function traverses a list from head to tail, sorting all the elements
	in the order specified by the argument order, using a safe sort algorithm.
	
	A compare function must have been previously set for this list (by calling
	list_set_compare_function() ), this function uses that compare function to
	compare the element data looking for relative order of those elements. What
	constitutes greater or lesser for each element, is implementation dependent
	for the compare function, and the type of data the user stores in the list.
	
	Although this sort guarantees a safe sorting method, it will be much slower
	than the regular list_fast_sort(), list_sort() or list_tiny_sort() functions,
	especially for large lists. The larger the list size, the more dramatic the
	slow down will be. Only use this function if a safe sort if really needed,
	or the significant overhead is acceptable. I had considered naming this
	function list_turtle_sort(), but the safe_sort name is a more accurate
	description of the functions purpose.
	
	This function will sort pivot lists as requested, the list will be sorted in
	2 operations once for each side of the pivot point, using the order requested.
			
	This function extensively modifies the list. All previous indexes that may
	have been saved by the user are invalid. However all element references are
	still valid and can be used until the element has been removed from the
	list. Additionally if this is a pivot class list, all the internal pivot
	data is retained.
	
	This function uses no additional resources to perform the sort operation.
	It is a good choice for embedded systems, and for all users that need safe
	sort ordering and are willing to pay the performance price involved. In
	general this is the worst sort to use, but it depends on your specific
	device or design requirements.

	list		list_object that has been previously initialized.
	order		either LIST_ASCENDING or LIST_DESCENDING to denote which order
				to sort the elements in the list.
	returns	LIST_OK			the sort sequence was successful
			LIST_ERR		if there was an error.									*/
int list_safe_sort(	list_object *restrict list,
					const boolean order) {

	leaf_node *ptr;
	long index = 0;
	int direction;
	
#ifdef DEBUG
	if (list__validate_list_object(list, FALSE, __func__) == FALSE)
		return LIST_ERR;
	if (list->iteration_enabled) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"list has iteration enabled.\n\n", __func__);
		return LIST_ERR;
	}
	if (list->functions.compare == NULL) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"NULL compare function.\n\n", __func__);
		return LIST_ERR;
	}
#else
	if ((list == NULL)
			|| (list->functions.compare == NULL)
			|| (list->iteration_enabled))
		return LIST_ERR;
#endif

	if (order == LIST_ASCENDING) direction = 1;
	else direction = -1;
	
	if (list->element_count < 2) {
		list->list_sorted = TRUE;
		list->sorted_order = order;
		return LIST_OK;
	}

	list->mid_point = NULL;


	if (list->list_pivot_enabled == TRUE) {
		ptr = list__locate_pivot_leaf_node(list, &index);
		list__sort_insertion_sort(list, direction, list->head_guard->next_leaf, ptr, FALSE);
		list__sort_insertion_sort(list, direction, ptr->next_leaf, list->tail_guard->prior_leaf, FALSE);
	} else {
		list__sort_insertion_sort(list, direction, list->head_guard->next_leaf, list->tail_guard->prior_leaf, FALSE);
	}

	list__full_optimize_index(list, TRUE);
	list->sorted_order = order;
	if (list->list_pivot_enabled == TRUE) list->list_sorted = TRUE;

#ifdef DLIST_DEBUG
	assert(list__audit(list) == LIST_OK);
#endif		/* DLIST_DEBUG */
	
	return LIST_OK;
}

/* Sort all the elements in a list, using either ascending or descending order.

	This function traverses a list from head to tail, sorting all the elements
	in the order specified by the argument order. This function is identical to
	list_sort() except that all previous element references are voided. Never
	call list_fast_sort() if you need to retain any previous element references,
	however if you either did not retain element references, or can refresh them
	this is a much faster sort. On average this sort function is 15% faster than
	the list_sort() function.
	
	The argument scope determines how much of the list this function will
	traverse sorting the elements. Setting this argument to ENTIRE_LIST will
	cause the function to sort the entire list from head to tail. For finite
	and linear class lists, any other setting for scope is ignored. 
	
	For pivot class lists there are 3 possible settings. Providing either 
	PIVOT_MAJOR or PIVOT_MINOR will cause this function to confine its sort to 
	that side of the list pivot point and leave the other side alone. Setting 
	scope to ENTIRE_LIST will cause this function to independently sort both 
	sides of the pivot in the order wished. It would be the same as calling this 
	function twice, once with PIVOT_MINOR, and once with PIVOT_MAJOR, except
	that by doing it in one operation the list is also marked as sorted, so is
	available for any of the Sorted List Element Functions. 
	
	Using ENTIRE_LIST on a pivot class list will sort the minor and major sides 
	of the list in the same order. For example, if the order was LIST_ASCENDING 
	then the minor elements would be ordered ascending from the head of the 
	list to the pivot point, and then the major elements would be ordered in 
	continuing ascending order from the pivot point to the tail of the list. If 
	you wish a different ordering scheme, please call this function twice once 
	for each side of the pivot with the ordering scheme you wish. However this 
	will not leave the list marked as sorted, so it will not be available to any 
	of the Sorted List Element Functions, as they all expect the ordering scheme
	to be consistent across a list.
	
	If you wish to sort a pivot class list while using data points that are 
	different from the pivot arraignment, or totally ignoring the pivot list 
	structure and rearrange the elements differently, first transfer or copy 
	the pivot class list into a linear class list, and then sort the elements 
	in that destination list anyway you wish. An alternative approach is to
	call this function twice once for each side of the pivot, and use a compare
	function that differs in data points from the pivot function currently set 
	for the list.

	A compare function must have been previously set for this list (by calling
	list_set_compare_function() ), this function uses that compare function to
	compare the element data looking for relative order of those elements. What
	constitutes greater or lesser for each element, is implementation dependent
	for the compare function, and the type of data the user stores in the list.

	This function extensively modifies the list. All previous indexes and all
	element references that may have been saved by the user are invalid. If this
	is a pivot class list, all the internal pivot data is retained.
	
	This function uses no additional resources to perform the sort operation.
	It is a good choice for embedded systems, and for all users that do not
	need to retain any element references across the sort. In general this is
	the best sort to use, but it depends on your specific device or design
	requirements.

	list		list_object that has been previously initialized.
	order		either LIST_ASCENDING or LIST_DESCENDING to denote which
		 		order to sort the elements in the list.
	scope		what section the list to sort the elements. Must be ENTIRE_LIST
				for linear or finite class lists, or may be either PIVOT_MINOR,
				PIVOR_MAJOR, or ENTIRE_LIST for pivot class lists
	returns	LIST_OK			the sort sequence was successful
			LIST_ERR		if there was an error.									*/
int list_fast_sort(	list_object *restrict list,
					const boolean order,
					const int scope) {

	leaf_node *ptr, *end_ptr;
	long index = 0;
	int direction;
	
#ifdef DEBUG
	if (list__validate_list_object(list, FALSE, __func__) == FALSE)
		return LIST_ERR;
	if (list->iteration_enabled) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"list has iteration enabled.\n\n", __func__);
		return LIST_ERR;
	}
	if (list->functions.compare == NULL) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"NULL compare function.\n\n", __func__);
		return LIST_ERR;
	}
	if ((list->list_pivot_enabled == FALSE) && (scope != ENTIRE_LIST)) {
		fprintf(stderr, "\n==> Dlist Debug WARNING ==> %s() - "\
				"an ignored scope argument '%d' was provided for a linear class list.\n\n",
				__func__, scope);
	}
	if ((list->list_pivot_enabled == TRUE) && ((scope < PIVOT_MINOR) || (scope > PIVOT_MAJOR))) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"invalid scope argument '%d' was provided for a pivot class list.\n\n",
				__func__, scope);
		return LIST_ERR;
	}
#else
	if ((list == NULL)
			|| (list->iteration_enabled)
			|| (list->functions.compare == NULL)
			|| ((list->list_pivot_enabled == TRUE) && ((scope < PIVOT_MINOR) || (scope > PIVOT_MAJOR)) ))			
		return LIST_ERR;
#endif

	if (order == LIST_ASCENDING) direction = 1;
	else direction = -1;
	
	if (list->element_count < 2) {
		list->list_sorted = TRUE;
		list->sorted_order = order;
		return LIST_OK;
	}
	
	if (list->list_pivot_enabled == TRUE) {
		if (scope == ENTIRE_LIST) {
			list_fast_sort(list, order, PIVOT_MINOR);
			list_fast_sort(list, order, PIVOT_MAJOR);
			list->list_sorted = TRUE;

#ifdef DLIST_DEBUG
			assert(list__audit(list) == LIST_OK);
#endif		/* DLIST_DEBUG */
			return LIST_OK;
		} else {
			if (scope == PIVOT_MAJOR) {
				if (list->pivot_point.major < 2) return LIST_OK;
			}
			if (scope == PIVOT_MINOR) {
				if (list->pivot_point.minor < 2) return LIST_OK;
			}
		}
	}

	list->mid_point = NULL;

	if ((list->list_pivot_enabled == TRUE) && (scope != ENTIRE_LIST)) {
		ptr = list__locate_pivot_leaf_node(list, &index);
		if (scope == PIVOT_MAJOR) {
			ptr = ptr->next_leaf;							/* the first leaf node of the major side */
			end_ptr = list->tail_guard->prior_leaf;
		} else {
			end_ptr = ptr;
			ptr = list->head_guard->next_leaf;
		}
	} else {
		ptr = list->head_guard->next_leaf;
		end_ptr = list->tail_guard->prior_leaf;
	}

	if (list->element_count <= DLIST_QUICKSORT_MIN_SIZE) {
		list__sort_insertion_sort(list, direction, ptr, end_ptr, TRUE);
	} else {
		list__sort_quick_sort(list, direction, ptr, end_ptr, TRUE);
	}

	list__full_optimize_index(list, TRUE);
	list->sorted_order = order;

#ifdef DLIST_DEBUG
	assert(list__audit(list) == LIST_OK);
#endif		/* DLIST_DEBUG */
	
	return LIST_OK;
}

/* Sort all the elements in a list, using either ascending or descending order.

	This function traverses a list from head to tail, sorting all the elements
	in the order specified by the argument order. This sort will retain all
	exiting element references.
		
	The argument scope determines how much of the list this function will
	traverse sorting the elements. Setting this argument to ENTIRE_LIST will
	cause the function to sort the entire list from head to tail. For finite
	and linear class lists, any other setting for scope is ignored. 
	
	For pivot class lists there are 3 possible settings. Providing either 
	PIVOT_MAJOR or PIVOT_MINOR will cause this function to confine its sort to 
	that side of the list pivot point and leave the other side alone. Setting 
	scope to ENTIRE_LIST will cause this function to independently sort both 
	sides of the pivot in the order wished. It would be the same as calling this 
	function twice, once with PIVOT_MINOR, and once with PIVOT_MAJOR, except
	that by doing it in one operation the list is also marked as sorted, so is
	available for any of the Sorted List Element Functions. 
	
	Using ENTIRE_LIST on a pivot class list will sort the minor and major sides 
	of the list in the same order. For example, if the order was LIST_ASCENDING 
	then the minor elements would be ordered ascending from the head of the 
	list to the pivot point, and then the major elements would be ordered in 
	continuing ascending order from the pivot point to the tail of the list. If 
	you wish a different ordering scheme, please call this function twice once 
	for each side of the pivot with the ordering scheme you wish. However this 
	will not leave the list marked as sorted, so it will not be available to any 
	of the Sorted List Element Functions, as they all expect the ordering scheme
	to be consistent across a list.
	
	If you wish to sort a pivot class list while using data points that are 
	different from the pivot arraignment, or totally ignoring the pivot list 
	structure and rearrange the elements differently, first transfer or copy 
	the pivot class list into a linear class list, and then sort the elements 
	in that destination list anyway you wish. An alternative approach is to
	call this function twice once for each side of the pivot, and use a compare
	function that differs in data points from the pivot function currently set 
	for the list.

	A compare function must have been previously set for this list (by calling
	list_set_compare_function() ), this function uses that compare function to
	compare the element data looking for relative order of those elements. What
	constitutes greater or lesser for each element, is implementation dependent
	for the compare function, and the type of data the user stores in the list.

	This function extensively modifies the list. All previous indexes that may
	have been saved by the user are invalid. However all element references are
	still valid and can be used until the element has been removed from the
	list. Additionally if this is a pivot class list, all the internal pivot
	data is retained. 
		
	This function uses no additional resources to perform the sort operation.
	It is a good choice for embedded systems, and systems with extreme memory
	constraints. However for all other users either list_fast_sort() or
	list_sort() are better choices. This sort will retain all existing element
	references across the sort. In general this is not a good sort to use, but
	it depends on your specific device or design requirements. The much faster
	function, list_fast_sort() is over 2x faster than this sort, but will
	destroy all existing element references that may have been retained by the
	user. The other fast sort, list_sort() is about 1.75x faster this sort,
	but allocates additional resources to perform the sort operations.

	list		list_object that has been previously initialized.
	order		either LIST_ASCENDING or LIST_DESCENDING to denote which
		 		order to sort the elements in the list.
	scope		what section the list to sort the elements. Must be ENTIRE_LIST
				for linear or finite class lists, or may be either PIVOT_MINOR,
				PIVOR_MAJOR, or ENTIRE_LIST for pivot class lists
	returns	LIST_OK			the sort sequence was successful
			LIST_ERR		if there was an error.									*/
int list_tiny_sort(	list_object *restrict list,
					const boolean order,
					const int scope) {

	leaf_node *ptr, *end_ptr;
	long index = 0;
	int direction;
	
#ifdef DEBUG
	if (list__validate_list_object(list, FALSE, __func__) == FALSE)
		return LIST_ERR;
	if (list->iteration_enabled) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"list has iteration enabled.\n\n", __func__);
		return LIST_ERR;
	}
	if (list->functions.compare == NULL) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"NULL compare function.\n\n", __func__);
		return LIST_ERR;
	}
	if ((list->list_pivot_enabled == FALSE) && (scope != ENTIRE_LIST)) {
		fprintf(stderr, "\n==> Dlist Debug WARNING ==> %s() - "\
				"an ignored scope argument '%d' was provided for a linear class list.\n\n",
				__func__, scope);
	}
	if ((list->list_pivot_enabled == TRUE) && ((scope < PIVOT_MINOR) || (scope > PIVOT_MAJOR))) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"invalid scope argument '%d' was provided for a pivot class list.\n\n",
				__func__, scope);
		return LIST_ERR;
	}
#else
	if ((list == NULL)
			|| (list->iteration_enabled)
			|| (list->functions.compare == NULL)
			|| ((list->list_pivot_enabled == TRUE) && ((scope < PIVOT_MINOR) || (scope > PIVOT_MAJOR)) ))			
		return LIST_ERR;
#endif

	if (order == LIST_ASCENDING) direction = 1;
	else direction = -1;
	
	if (list->element_count < 2) {
		list->list_sorted = TRUE;
		list->sorted_order = order;
		return LIST_OK;
	}
	
	if (list->list_pivot_enabled == TRUE) {
		if (scope == ENTIRE_LIST) {
			list_tiny_sort(list, order, PIVOT_MINOR);
			list_tiny_sort(list, order, PIVOT_MAJOR);
			list->list_sorted = TRUE;

#ifdef DLIST_DEBUG
			assert(list__audit(list) == LIST_OK);
#endif		/* DLIST_DEBUG */
			return LIST_OK;
		} else {
			if (scope == PIVOT_MAJOR) {
				if (list->pivot_point.major < 2) return LIST_OK;
			}
			if (scope == PIVOT_MINOR) {
				if (list->pivot_point.minor < 2) return LIST_OK;
			}
		}
	}

	list->mid_point = NULL;

	if ((list->list_pivot_enabled == TRUE) && (scope != ENTIRE_LIST)) {
		ptr = list__locate_pivot_leaf_node(list, &index);
		if (scope == PIVOT_MAJOR) {
			ptr = ptr->next_leaf;							/* the first leaf node of the major side */
			end_ptr = list->tail_guard->prior_leaf;
		} else {
			end_ptr = ptr;
			ptr = list->head_guard->next_leaf;
		}
	} else {
		ptr = list->head_guard->next_leaf;
		end_ptr = list->tail_guard->prior_leaf;
	}

	if (list->element_count <= DLIST_QUICKSORT_MIN_SIZE) {
		list__sort_insertion_sort(list, direction, ptr, end_ptr, FALSE);
	} else {
		list__sort_quick_sort(list, direction, ptr, end_ptr, FALSE);
	}

	list__full_optimize_index(list, TRUE);
	list->sorted_order = order;

#ifdef DLIST_DEBUG
	assert(list__audit(list) == LIST_OK);
#endif		/* DLIST_DEBUG */
	
	return LIST_OK;
}

/* Rotate all the elements in a list, by the number of positions requested,
	either towards the head, or tail, of the list.

	This function will rotate all the elements from their current positions
	in the list, towards either the head or tail of the list, based on the
	the arguments provided. It is similar to list_shift_index(), except
	that the elements are kept in relative order and all shifted together.

	If the list is currently sorted, the sorted indexes will be voided, and the
	list_sorted_xx() functions are no longer available for the list, until it
	is resorted.
	
	This function is not available for pivot class lists.
	
	This function performs a sequence of operations that are equivalent to
	a repeated sequence of calls to of list_fetch() followed by list_append()
	or list_prepend(), but far more efficient for this more specific task,
	and without destroying the reference to the original data element.
	
	This function operates in a manner consistent with assembly language
	register shifts/rotates. If shifting elements towards the head of the
	list, each element that is shifted off the list, is then placed at the
	tail of the list. The opposite occurs if shifting towards the tail of
	the list.
	
	By using the following example of a sorted linear list
	
	data contents	->	a	b	c	d	e	f	g	h	i	j	k	l	m	n
	index number	->	0	1	2	3	4	5	6	7	8	9	10	11	12	13
	
	if the list is processed by list_rotate(list, 4, HEAD_OF_LIST), the result
	will be

	data contents	->	e	f	g	h	i	j	k	l	m	n	a	b	c	d
	index number	->	0	1	2	3	4	5	6	7	8	9	10	11	12	13

	if the same original list is processed by list_rotate(list, 6, TAIL_OF_LIST),
	the result will be

	data contents	->	i	j	k	l	m	n	a	b	c	d	e	f	g	h
	index number	->	0	1	2	3	4	5	6	7	8	9	10	11	12	13
			
	This function makes changes to the list.

	list		list_object that has been previously initialized.
	count 		denotes how many postions to shift all the elements
				in the list.
	direction 	either HEAD_OF_LIST or TAIL_OF_LIST, denotes which
				direction in the list to shift all the elements.
	returns	LIST_OK			the list elements were rotated as requested
			LIST_ERR		if there was an error.									*/
int list_rotate(	list_object *restrict list,
					const unsigned int count,
					const boolean direction) {

	unsigned int index;

#ifdef DEBUG
	if (list__validate_list_object(list, LIST_WRITE, __func__) == FALSE)
		return LIST_ERR;
	if (list->element_count < 1) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with empty input list.\n\n", __func__);
		return LIST_ERR;
	}
	if (list->iteration_enabled) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"list has iteration enabled.\n\n", __func__);
		return LIST_ERR;
	}
	if (list->list_pivot_enabled == TRUE) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called on a pivot class list.\n\n", __func__);
		return LIST_ERR;
	}
#else
	if ((list == NULL)
			|| (list->iteration_enabled)
			|| (list->list_pivot_enabled == TRUE)
			|| (list->element_count < 1))
		return LIST_ERR;

#endif

	if (count > list->element_count) index = count % list->element_count;
	else index = count;
	
	if (index == 0) {
#ifdef DEBUG
		fprintf(stderr, "\n==> Dlist Debug WARNING ==> %s() - "\
				"called to rotate elements %u times, which places them all back to same "\
				"positions. List contains %u elements.\n\n", __func__, count, list->element_count);
#endif
	return LIST_OK;
	}
		
	/*	there is an inbuilt assumption here that this function is not used
		frequently, and it is only used to shift elements in small numbers.
		Therefore it reuses list__shift_leaf_node(). However if required,
		this function can be sped up by unlinking and relinking leaf nodes
		directly here in a loop and fixing the middle pointer at the end.
		But this seems like an efficiency that is not required, and i prefer
		not to replicate code, unless such copies are really beneficial.
		On the same note, this could be made more efficient in principal,
		if i analysed the direction and shift count, if the shift is greater
		than elements_count/2 it would be faster to switch directions and
		shift the smaller number of times. But again i doubt anyone will
		use this beyond the occasional small number of shifts and that would
		create overhead code for every rotate operation, when most if not all
		of them would not need it. It would be a waste.								*/
	list->list_sorted = FALSE;
	while (index-- > 0) {
		if (direction == HEAD_OF_LIST) list__shift_leaf_node(list, 0, TAIL_OF_LIST);
		else list__shift_leaf_node(list, list->element_count-1, HEAD_OF_LIST);
	}

#ifdef DLIST_DEBUG
	assert(list__audit(list) == LIST_OK);
#endif		/* DLIST_DEBUG */
	
	return LIST_OK;
}

/* Reverse the order of all the elements in a list.
	
	This function will take the chain of elements in a list and reverse them
	so the relative positioning of neighboring elements is the same, but the
	entire list will be reversed, or flipped like rotating a coin. If the list
	is a linear or finite class list, the reversal operation is performed on
	the whole list in one operation. If it is a pivot class list, there is the
	option to reverse the entire list, keeping the pivot semantics, or just a
	portion of the list, the minor or major side of the pivot point.
	
	This function does not sort the list, and retains all consistant list and
	element information without destroying the element references to the
	original data elements. If the list is currently sorted, the sorted data
	and indexes will be left in tact, just reversed.

	By using the following example of a sorted linear list
	
	data contents	->	a	b	c	d	e	f	g	h	i	j	k	l	m	n
	index number	->	0	1	2	3	4	5	6	7	8	9	10	11	12	13
	
	if the list is processed by list_flip(), the result will be

	data contents	->	n	m	l	k	j	i	h	g	f	e	d	c	b	a
	index number	->	0	1	2	3	4	5	6	7	8	9	10	11	12	13
	
	If the original list above was a pivot class list, and the defined pivot
	point for the list was <=d, then a list_flip(list, ENTIRE_LIST) operation
	would result in the following

	data contents	->	d	c	b	a	n	m	l	k	j	i	h	g	f	e
	index number	->	0	1	2	3	4	5	6	7	8	9	10	11	12	13

	By using PIVOT_MINOR, or PIVOT_MAJOR as the scope, the processing of the list
	would have been restricted to that portion only.
	
	For both linear and finite class lists, any scope other than ENTIRE_LIST is
	ignored and the whole list is processed, however a warning message will be
	issued if running in DEBUG mode.
	
	This function makes extensive changes to the list.

	list		list_object that has been previously initialized.
	scope		can be ENTIRE_LIST for all list classes, or PIVOT_MINOR or
				PIVOT_MAJOR for pivot class lists.
	returns	LIST_OK			reversal operation was successful.
			LIST_ERR		if there was an error.									*/
int list_flip(	list_object *restrict list,
				const int scope) {

	leaf_node *ptr, *next_ptr;
	unsigned int total_elements;
	unsigned int middle;
	long int temp_index = 0;
	boolean list_sorted;
	

#ifdef DEBUG
	if (list__validate_list_object(list, FALSE, __func__) == FALSE)
		return LIST_ERR;
	if (list->iteration_enabled) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"list has iteration enabled.\n\n", __func__);
		return LIST_ERR;
	}
	if ((list->list_pivot_enabled == FALSE) && (scope != ENTIRE_LIST)) {
		fprintf(stderr, "\n==> Dlist Debug WARNING ==> %s() - "\
				"an ignored scope argument '%d' was provided for a non pivot class list.\n\n",
				__func__, scope);
	}
	if ((list->list_pivot_enabled == TRUE) && ((scope < PIVOT_MINOR) || (scope > PIVOT_MAJOR))) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"invalid scope argument '%d' was provided for a pivot class list.\n\n",
				__func__, scope);
		return LIST_ERR;
	}
#else
	if ((list == NULL)
			|| (list->iteration_enabled)
			|| ((list->list_pivot_enabled == TRUE) && ((scope < PIVOT_MINOR) || (scope > PIVOT_MAJOR)) ))			
		return LIST_ERR;
#endif

	if (list->element_count < 2) return LIST_OK;

	total_elements = list->element_count;
	if (total_elements%2) middle = (total_elements+1)/2;
	else middle = total_elements/2;

	if (list->list_pivot_enabled == TRUE) {
		ptr = list__locate_pivot_leaf_node(list, &temp_index);
		if (scope == ENTIRE_LIST) {
			list_sorted = list->list_sorted;

			/* pivot list, flip the entire list in 2 passes */
			if (list->pivot_point.minor > 1) {
				next_ptr = ptr->next_leaf;
				list__flip_leaf_nodes(list, list->head_guard->next_leaf,
						ptr->next_leaf, (middle < list->pivot_point.minor) ? TRUE : FALSE);
				ptr = next_ptr->prior_leaf;				/* original ptr was scrambled by flip */
			}
			if (list->pivot_point.major > 1)
				list__flip_leaf_nodes(list, ptr->next_leaf,
						list->tail_guard, (middle >= list->pivot_point.minor) ? TRUE : FALSE);
			list->list_sorted = list_sorted;
			list->sorted_order = !list->sorted_order;
		} else {
			/* pivot list, flip the side of the list requested */
			if (scope == PIVOT_MAJOR) {
				if (list->pivot_point.major > 1)
					list__flip_leaf_nodes(list, ptr->next_leaf,
							list->tail_guard, (middle >= list->pivot_point.minor) ? TRUE : FALSE);
			} else {
				if (list->pivot_point.minor > 1)
					list__flip_leaf_nodes(list, list->head_guard->next_leaf,
							ptr->next_leaf, (middle < list->pivot_point.minor) ? TRUE : FALSE);
			}
		}
	} else {
		/* not pivot list, flip the entire list */
		list_sorted = list->list_sorted;
		
		list__flip_leaf_nodes(list, list->head_guard->next_leaf, list->tail_guard, TRUE);

		list->list_sorted = list_sorted;
		list->sorted_order = !list->sorted_order;
	}

	return LIST_OK;
}

/* Shuffle the elements in a list, by randomly, and repeatedly relocating
	random list elements to the head, or tail, of the list.

	This function will un-sort, or shuffle, all the elements in the list.
	list. It is functionally similar to repeatedly calling list_shift_index(),
	but is more efficient.

	If the list is currently sorted, the sorted indexes will be voided, and the
	list_sorted_xx() functions are no longer available for the list, until it
	is resorted.
	
	This function is available for pivot class lists, it will shuffle the
	elements on both sides of the pivot point, keeping them on the correct
	sides, retaining the meaning of the pivot.
	
	This function is not useful for most business or technical OS purposes,
	but is useful for games, and other software that requires some randomness.
	
	This function retains all consistant list and element information without
	destroying the element references to the original data elements.
	
	This function changes the seed for the random number generator, if your
	code is using a predictable sequence of random numbers (non-seeded rand())
	then you should not use this function.
		
	This function makes changes to the list.

	list		list_object that has been previously initialized.
	returns	LIST_OK			the list was shuffled as requested
			LIST_ERR		if there was an error.									*/
int list_shuffle(list_object *restrict list) {

	long int i;
	unsigned int j;

#ifdef DEBUG
	if (list__validate_list_object(list, FALSE, __func__) == FALSE)
		return LIST_ERR;
	if (list->element_count < 1) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with empty input list.\n\n", __func__);
		return LIST_OK;
	}
	if (list->element_count < 2) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with input list with 1 element.\n\n", __func__);
		return LIST_OK;
	}
	if (list->iteration_enabled) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"list has iteration enabled.\n\n", __func__);
		return LIST_ERR;
	}
#else
	if ((list == NULL) || (list->iteration_enabled))
		return LIST_ERR;
#endif

	if (list->element_count < 2) return LIST_OK;

	if (list->element_count == 2)
		return (list__shift_leaf_node(list, 1, HEAD_OF_LIST));

	initialize_random_seed();
	list->list_sorted = FALSE;

	if (list->list_pivot_enabled == TRUE) {
		if (list->pivot_point.minor > 1) {
			if (list->pivot_point.minor == 2) {
				list__shift_leaf_node(list, 1, HEAD_OF_LIST);
			} else {
				i = list->pivot_point.minor * 2;
				j = list->pivot_point.minor / 2;
				while(i-- > 0) {
					list__shift_leaf_node(list,
							random_number_range(0, j+1),
							HEAD_OF_LIST);
					list__shift_leaf_node(list,
							random_number_range(j-1, list->pivot_point.minor-1),
							HEAD_OF_LIST);
				}
			}
		}
		if (list->pivot_point.major > 1) {
			if (list->pivot_point.major == 2) {
				list__shift_leaf_node(list, 1, TAIL_OF_LIST);
			} else {
				i = list->pivot_point.major * 2;
				j = list->pivot_point.major / 2;
				while(i-- > 0) {
					list__shift_leaf_node(list,
							random_number_range(list->pivot_point.minor,
														j+1),
							TAIL_OF_LIST);
					list__shift_leaf_node(list,
							random_number_range(list->pivot_point.minor+(j-1),
														list->pivot_point.major-1),
							TAIL_OF_LIST);
				}
			}
		}
	} else {
		i = list->element_count * 2;
		j = list->element_count / 2;
		while(i-- > 0) {
			list__shift_leaf_node(list, random_number_range(0, j+1),
					(coinFlip()) ? HEAD_OF_LIST : TAIL_OF_LIST);
			list__shift_leaf_node(list, random_number_range(j-1, list->element_count-1),
					(coinFlip()) ? HEAD_OF_LIST : TAIL_OF_LIST);
		}
	}
	
	return LIST_OK;
}

/* Perform an optimization of the element index in a list.

	This function traverses a list from head to tail, rebuilding the index
	optimizations. These optimizations are lost when a list is modified with
	functions that use references instead of indexes. All other operations on
	a list either do not disturb the index, or adjust it as they operate. But
	insertions and removals by reference prevent this. They are used on very large
	lists to avoid the overhead of repeated operations when using the indexes.
	At the conclusion of a set of such reference operations, the caller can
	use this function to re-establish the index optimizations.
	
	This function is not a requirement to use or reuse indexes, just help
	support greater efficiency. Even if list modifications have been extensively
	made via element references, elements can still be retrieved, or modified
	using indexes. This function just restores the inbuilt index efficiency.
		
	list		list_object that has been previously initialized.
	returns	nothing																	*/
void list_optimize_index(list_object *restrict list) {

#ifdef DEBUG
	if (list__validate_list_object(list, FALSE, __func__) == FALSE) return;
#else
	if (list == NULL) return;
#endif

	list__full_optimize_index(list, FALSE);	

#ifdef DLIST_DEBUG
	assert(list__audit(list) == LIST_OK);
#endif		/* DLIST_DEBUG */
	
	return;	
}

/* Perform an optimization of the element memory storage in a list.

	This function first checks the current status of the list and the memory
	it is using, and if there is suitable potential for space reduction, it
	will traverses the list from head to tail, looking for blocks of memory
	to return to the operating system.
	
	This is a highly technical function, that most users will never need. It
	can be useful on embedded systems, and also if the user built a huge list,
	that is now very small, and wants to reclaim the used heap space. Embedded
	systems developers will want to fine tune the definition of the value of
	LEAF_NODE_ARRAY_SIZE, the settings for block sizes related to list size
	hints, and the vacancy percentage below in this function, to suit their
	particular system and needs. Also the use of finite lists would be
	generally recommended, where such use is practical.
	
	This function is not a requirement to use or reuse lists, or indexes, just
	help support greater efficiency for the small number of developers that may
	need it.
	
	This function does not disturb the actual list elements, all elements,
	data, references and indexes are left intact. From a users perspective, this
	function makes no changes to the list.
		
	list		list_object that has been previously initialized.
	returns	nothing																	*/
void list_optimize_memory(list_object *restrict list) {

#ifdef DEBUG
	if (list__validate_list_object(list, FALSE, __func__) == FALSE) return;
#else
	if (list == NULL) return;
#endif

#ifndef DLIST_USE_MALLOC
	unsigned int j, k;

	k = (list->element_count/10)*6;
	j = list__locate_open_leaf_node_count(list);
	if (j > k) release__leaf_node_blocks(list);	

#endif	/* ! DLIST_USE_MALLOC */

#ifdef DLIST_DEBUG
	assert(list__audit(list) == LIST_OK);
#endif		/* DLIST_DEBUG */
	
	return;	
}

/* Instruct D-List not to seed the underlying random functions.

	This function is called to prevent D-List from calling srand() with its
	own seed. Some D-List functions use random numbers, and to ensure clean
	and safe randomization D-List seeds the random number generator with a
	complicated real-time seed.
	
	However some users may actually want a predictable set of pseudo-random
	numbers for a particular reason. D-List functions would interfere with
	that. By calling this function before calling any other D-List functions
	this situation can be avoided.
	
	This is a highly technical function, that most users will never need. It
	is only useful in very rare circumstances, and otherwise should never be
	called.
	
	This function is not a requirement to use D-List, or use or reuse lists,
	indexes or element references. It is just provided to help support a very
	rare need to allow users to generate predictable pseudo-random numbers
	from rand().
	
	This function does not disturb the actual list elements, all elements,
	data, references and indexes are left intact. From a users perspective,
	this function makes no changes to the lists or the D-List subsystem.
		
	returns	nothing																	*/
void list_do_not_seed_random(void) {

	rand_seeded = TRUE;
	
	return;	
}

/* Provide details about a list to stdout.

	This function first checks the current status of the list and the memory
	it is using, and then outputs to standard out (stdout) a brief report of
	the current status and resources used.
	
	This is a highly technical function, that most users will never need. It
	can be useful during code development on embedded systems, and also for
	users that built a huge lists.
	
	This function is not a requirement to use or reuse lists, or indexes, just
	help support greater efficiency for the small number of developers that may
	need it. It is important to remember that if dlist is compiled to use malloc
	then this function cannot compute the overhead or wastage (slop) in malloc()
	due to whatever algorithm is being used, it can only account for the actual
	space assigned to the list.
	
	This function makes no changes to the list.
		
	list		list_object that has been previously initialized.
	listname	a user provided string to identify the list in the output
	returns	nothing																	*/
void list_information(	const list_object *restrict list,
						const char *listname) {

	long int j;

#ifdef DEBUG
	if (list__validate_list_object(list, FALSE, __func__) == FALSE) return;
#else
	if (list == NULL) return;
#endif

	fprintf(stdout, "Information for list '%s' -- Type: %s %sElems: %u %s", listname,
			(list->list_pivot_enabled) ? LIST_TYPE_PIVOT : ((list->list_finite_enabled) ? LIST_TYPE_FINITE : LIST_TYPE_LINEAR),
			(list->list_manages_elements) ? "(Copy Data) " : "" , list->element_count,
			(list->list_sorted) ? "(Sorted) " : "");
	if (list->list_pivot_enabled)
		fprintf(stdout, "Minor: %u Major: %d ", list->pivot_point.minor, list->pivot_point.major);
	if (list->list_finite_enabled)
		fprintf(stdout, "Max Elems: %u ", list->max_element_count);
	if ((list->functions.compare) || (list->functions.pivot) || (list->functions.search)
			|| (list->functions.size) || (list->functions.insert) || (list->functions.remove)
			|| (list->functions.iterate))
		fprintf(stdout, "Functions Set: %s%s%s%s%s%s%s",
			(list->functions.search) ? "Search " : "",
			(list->functions.compare) ? "Compare " : "",
			(list->functions.size) ? "Size " : "",
			(list->functions.pivot) ? "Pivot " : "",
			(list->functions.iterate) ? "Iterate " : "",
			(list->functions.insert) ? "Insert " : "",
			(list->functions.remove) ? "Remove " : "");

#ifndef DLIST_USE_MALLOC
	leaf_node_array *block_ptr;
	unsigned int i, k;

	j = sizeof(list_object);
	for (i = k = 0, block_ptr = list->leaf_block;
			(block_ptr != NULL);
			j += (sizeof(leaf_node_array) + (sizeof(leaf_node) * block_ptr->nodes_allocated)),
			k += (block_ptr->nodes_allocated - block_ptr->nodes_in_use),
			++i, block_ptr = block_ptr->next_block);
	fprintf(stdout, "Internal Memory: %ld bytes Leaf Block Size: %u Leaf Blocks: %u Free "\
			"Nodes: %u.\n",j , list->allocation_size ,i , k);
#else

	j = sizeof(list_object) + (sizeof(leaf_node) * (list->element_count + 2));
	fprintf(stdout, "Internal Memory: %ld bytes.\n", j);
	
#endif	/* ! DLIST_USE_MALLOC */
	
	return;	
}


/* Return pointer to string corresponding to D-List error code.

	Returns a pointer to a string that describes the error code provided in
	error_code, or the string "Invalid Error Code" if the provided error code
	is invalid.
	
	error_code	a valid D-List error code, documented in this file.
	returns	pointer a null terminated read-only string. The string does not 
			belong to the caller, you must not try to modify or free it.			*/
char *list_error_string(const int error_code) {

	if ((error_code < LIST_UNSORTED) || (error_code > LIST_OK)) return error_strings[0];
	return error_strings[(0 - error_code) + 1];
}

/* Set the compare function for the list provided.

	Compare functions are used while performing sorts, ordering and
	assessments of value ranges on list elements.

	list		list_object that has been previously initialized.
	compare_function
				pointer to function to process compare requests by the list
				subsystem when sorted or comparing list elements. If NULL any
				existing compare function is removed.
	returns	LIST_OK			success
			LIST_ERR		on error	 											*/
int list_set_compare_function(	list_object *restrict list,
								const element_compare compare_function) {
	
#ifdef DEBUG
	if (list__validate_list_object(list, FALSE, __func__) == FALSE) return LIST_ERR;
#else
	if (list == NULL) return LIST_ERR;
#endif

	list->functions.compare = compare_function;
	return LIST_OK;
}

/* Get the pointer to the current compare function for the list provided.

	Compare functions are used while performing sorts, ordering and
	assessments of value ranges on list elements. Please refer to
	list_set_compare_function() above.
	
	Useful for large code bases that need to use a different compare
	function, and then restore the original compare function when done.

	list		list_object that has been previously initialized.
	returns	current pointer to compare function, or NULL if none is set.			*/
element_compare *list_get_compare_function(const list_object *restrict list) {
	
#ifdef DEBUG
	if (list__validate_list_object(list, FALSE, __func__) == FALSE) return NULL;
#else
	if (list == NULL) return NULL;
#endif

	return (element_compare *)list->functions.compare;
}

/* Set the pivot function for the list provided.
	
	Pivot functions are used to decide which side of a pivot list an
	element is placed, or where to search for the element. A pivot
	function must be set before a pivot class list can have any
	elements added to it.

	list		list_object that has been previously initialized.
	pivot_function
				pointer to function to process pivot requests for list
				elements, by the list subsystem. If NULL is passed, any
				existing compare function is removed.
	returns	LIST_OK			success
			LIST_ERR		on error 												*/
int list_set_pivot_function(	list_object *restrict list,
								const element_pivot pivot_function) {
	
#ifdef DEBUG
	if (list__validate_list_object(list, FALSE, __func__) == FALSE)
		return LIST_ERR;
	if (list->list_pivot_enabled == FALSE) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called on a non pivot class list.\n\n", __func__);
		return LIST_ERR;
	}
#else
	if ((list == NULL) || (list->list_pivot_enabled == FALSE)) return LIST_ERR;
#endif

	list->functions.pivot = pivot_function;
	return LIST_OK;
}

/* Set the search function for the list provided.

	Search functions are used while performing searches list elements to
	find one that matches user supplied search criteria.

	list		list_object that has been previously initialized.
	search_function
				pointer to function to process search requests for list
				elements, by the list subsystem. If NULL is passed, any
				existing compare function is removed.
	returns	LIST_OK			success
			LIST_ERR		on error 												*/
int list_set_search_function(	list_object *restrict list,
								const element_search search_function) {

#ifdef DEBUG
	if (list__validate_list_object(list, FALSE, __func__) == FALSE) return LIST_ERR;
#else
	if (list == NULL) return LIST_ERR;
#endif

	list->functions.search = search_function;
	return LIST_OK;
}

/* Get the pointer to the current search function for the list provided.

	Search functions are used while performing searches on list elements.
	Please refer to list_set_search_function() above.
	
	Useful for large code bases that need to use a different search function,
	and then restore the original compare function when done.

	list		list_object that has been previously initialized.
	returns	current pointer to search function, or NULL								*/
element_search *list_get_search_function(const list_object *restrict list) {
	
#ifdef DEBUG
	if (list__validate_list_object(list, FALSE, __func__) == FALSE) return NULL;
#else
	if (list == NULL) return NULL;
#endif

	return (element_search *)list->functions.search;
}

/* Set the size function for the list provided.

	Size functions are used to compute the size required to store the
	element. This is required when the list allocates and manages the
	elements storage.

	list		list_object that has been previously initialized.
	size_function
				pointer to function to return size requests for list elements.
				If NULL is passed, any existing size function is removed.
	returns	LIST_OK			success
			LIST_ERR		on error 												*/
int list_set_size_function(	list_object *restrict list,
							const element_size size_function) {
	
#ifdef DEBUG
	if (list__validate_list_object(list, FALSE, __func__) == FALSE) return LIST_ERR;
#else
	if (list == NULL) return LIST_ERR;
#endif

	list->functions.size = size_function;
	return LIST_OK;
}

/* Set the insert function for the list provided.

	Insert functions are used to perform user actions related to the
	element, or its insertion into the list. It can perform whatever
	actions it wants, such as gather statistics, modify the element,
	signal a thread, make computations, copy data to a cache, etc.
	However this function has no control over the insertion process
	itself.
	
	This function is called after the element is inserted into the
	list, by all list insertion functions (list_append(),
	list_prepend(), list_store(), list_insert_ref(), and
	list_insert_index() ). This function is not called by list_clone(),
	list_transfer(), list_concatenate() or list_split_pivot().

	list		list_object that has been previously initialized.
	insert_function
				pointer to function to process new list elements. If NULL is
				passed, any existing insert function is removed.
	returns	LIST_OK			success
			LIST_ERR		on error 												*/
int list_set_insert_function(	list_object *restrict list,
								const element_insert insert_function) {
	
#ifdef DEBUG
	if (list__validate_list_object(list, FALSE, __func__) == FALSE) return LIST_ERR;
#else
	if (list == NULL) return LIST_ERR;
#endif

	list->functions.insert = insert_function;
	return LIST_OK;
}

/* Set the remove function for the list provided.

	Remove functions are used to perform user actions related to
	the element, or its removal from the list. It can perform whatever
	actions it wants, such as gather statistics, free resources, modify
	the element, signal a thread, make computations, etc. However this
	function has no control over the removal process itself.

	This function is called before the element is removed from the list,
	by all list removal functions (list_fetch(), list_delete_index(),
	list_delete_ref(), list_delete_group(), list_extract_index(),
	list_clear(), list_erase() and list_search_delete() ). This function
	is not called by list_transfer().

	This function is also called when other functions, such as
	list_clone(), clear out a list prior to cloning. If you do not want
	this function called during such implicit operations, first remove
	the function prior to calling them.

	list		list_object that has been previously initialized.
	remove_function
				pointer to function to process list elements that are about to
				be removed. If NULL is passed, any existing remove function is
				removed from the list.
	returns	LIST_OK			success
			LIST_ERR		on error 												*/
int list_set_remove_function(	list_object *restrict list,
								const element_remove remove_function) {
	
#ifdef DEBUG
	if (list__validate_list_object(list, FALSE, __func__) == FALSE) return LIST_ERR;
#else
	if (list == NULL) return LIST_ERR;
#endif

	list->functions.remove = remove_function;
	return LIST_OK;
}

/* Set the iterate function for the list provided.

	Iterate functions are used to process elements during a sequence
	of list iteration started by the function list_callback_iteration().

	list		list_object that has been previously initialized.
	iterate_function
				pointer to function to process list elements. If NULL
				is passed, any existing iterate function is removed.
	returns	LIST_OK			success
			LIST_ERR		on error 												*/
int list_set_iterate_function(	list_object *restrict list,
								const element_iterate iterate_function) {
	
#ifdef DEBUG
	if (list__validate_list_object(list, FALSE, __func__) == FALSE) return LIST_ERR;
#else
	if (list == NULL) return LIST_ERR;
#endif

	list->functions.iterate = iterate_function;
	return LIST_OK;
}

/* Change the maximum number of elements a finite class list may hold.
	
	If the new maximum number is greater than the current one, no changes will
	occur in the list, it will just accept new elements without removing old
	elements from the tail, until the new limit is reached. If the new limit is
	less than the old one, then the next time an element is added, the list will
	be reduced to the new limit set here, by sequentially removing elements from
	the tail end until the new limit is reached.

	list		list_object that has been previously initialized.
	max			The new maximum number of elements this list may hold.
	returns	LIST_OK			success
			LIST_ERR		on error 												*/
int list_set_finite_limit(	list_object *restrict list,
							const unsigned int max) {

#ifdef DEBUG
	if (list__validate_list_object(list, FALSE, __func__) == FALSE)
		return LIST_ERR;
	if (list->list_finite_enabled == FALSE) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called on a non finite class list.\n\n", __func__);
		return LIST_ERR;
	}
	if (max < 1) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with a 0 element count.\n\n", __func__);
		return LIST_ERR;
	}
	if (max > LIST_CREATE_FINITE__MAX_CHECK_POINT) {
		fprintf(stderr, "\n==> Dlist Debug WARNING ==> %s() - "\
				"called with max element count of (#%u) which is greater than %d, did you really "\
				"mean to do this ?.\n\n", __func__, max, LIST_CREATE_FINITE__MAX_CHECK_POINT);	
	}
#else
	if ((list == NULL) || (list->list_finite_enabled == FALSE) || (max < 1))			
		return LIST_ERR;
#endif

#ifndef DLIST_USE_MALLOC
	int leafs_needed;
	if (max > list->max_element_count) {
		leafs_needed = max - list->max_element_count+1;
		if (allocate__leaf_node_block(list, leafs_needed) == NULL) {
#ifdef DEBUG
			fprintf(stderr, "\n==> Dlist Debug INTERNAL ERROR ==> %s() - "\
					"unable to create new leaf node block.\n\n", __func__);
#endif	
			return LIST_ERR;
		}	

	}
#endif	/* ! DLIST_USE_MALLOC */

	list->max_element_count = max;

	return LIST_OK;
}


/*	=====================================================================
		Internal Service Functions -- NOT exposed to external callers
	=====================================================================	*/


/*	=====================================================================
			Functions to link, unlink and find key leaf nodes
	=====================================================================	*/
/* Return the leaf node requested by list index pos */
static inline leaf_node *list__locate_leaf_node(	const list_object *restrict list,
													const long int  index) {

	register leaf_node *ptr;
	float hint;
	register unsigned int i;

	if (index < 0) {
		ptr = list->head_guard;
	} else {
		if (index > list->element_count - 2) {
			/* this covers 2 cases, all cases locating the last node will present a max index
				except list_insert_index() adding an element to the end of the list, could
				present the index # after the last one.										*/
			ptr = list->tail_guard->prior_leaf;
		} else {
			if (list->mid_point) {
				hint = (float)(index+1) / (float)list->element_count;
				/* locate the leaf node from head guard forward, or the middle, back or forward
					or backwards from the tail guard, depending on where the index is */
				if (hint <= 0.25) {
					for (i = 0, ptr = list->head_guard->next_leaf; i < index; ptr = ptr->next_leaf, i++);
				} else {
					if (hint < 0.5) {
						for (i = (list->element_count-1)/2, ptr = list->mid_point;
								i > index; ptr = ptr->prior_leaf, i--);
					} else {
						if (hint <= 0.75) {
							for (i = (list->element_count-1)/2, ptr = list->mid_point;
									i < index; ptr = ptr->next_leaf, i++);
						} else {
							for (i = list->element_count, ptr = list->tail_guard;
									i > index; ptr = ptr->prior_leaf, i--);
						}
					}
				}
			} else {
				/* list has no mid point, so the user has been using reference based
					updates. Decide which end to start counting from */
				if (index+1 > (list->element_count-1)/2) {
					for (i = 0, ptr = list->head_guard->next_leaf; i < index; ptr = ptr->next_leaf, i++);
				} else {
					for (i = list->element_count, ptr = list->tail_guard;
							i > index; ptr = ptr->prior_leaf, i--);		
				}
			}
		}
	}
	
	return ptr;
}

/* Return the leaf node with the extreme value, from the entire list, by order, as requested by caller */
static inline leaf_node *list__locate_extremum_leaf_node(	const list_object *restrict list,
															unsigned int *index,
															const boolean order) {

	leaf_node *ptr;
	leaf_node *extreme_ptr;
	unsigned int i = 0;
	unsigned int extreme_index = 0;
	int vector;
	int (*element_compare)(const void *element_a, const void *element_b);

	if ((list->element_count == 0) || (list->functions.compare == NULL)) return NULL;
	
	if (order == LIST_ASCENDING) vector = 1;
	else vector = -1;

	element_compare = list->functions.compare;

	for (extreme_ptr = list->head_guard->next_leaf, ptr = list->head_guard->next_leaf->next_leaf;
			(i < list->element_count) && (ptr->next_leaf != NULL);
			++i, ptr = ptr->next_leaf) {
		if ((*element_compare)(extreme_ptr->element, ptr->element) * vector < 0) {
			extreme_ptr = ptr;
			extreme_index = i+1;
		}
	}
	
	if (index != NULL) *index = extreme_index;
	return extreme_ptr;
}

/* Return the leaf node with the extreme value, from a pivot side of the list, and
	by the order, as requested by caller */
static inline leaf_node *list__locate_extremum_leaf_node_pivot(	const list_object *restrict list,
																unsigned int *index,
																const boolean order,
																const int scope) {

	leaf_node *ptr, *end_ptr, *extreme_ptr;
	long pivot_index = 0;
	unsigned int i = 0;
	unsigned int j;
	unsigned int extreme_index = 0;
	int vector;
	int (*element_compare)(const void *element_a, const void *element_b);

	if ((list->element_count == 0) || (list->functions.compare == NULL)) return NULL;
	
	if (order == LIST_ASCENDING) vector = 1;
	else vector = -1;

	ptr = list__locate_pivot_leaf_node(list, &pivot_index);

	element_compare = list->functions.compare;
	
	if (scope == PIVOT_MINOR) {
		extreme_ptr = list->head_guard->next_leaf;
		end_ptr = ptr->next_leaf;
		j = list->pivot_point.minor;
	} else {
		extreme_ptr = ptr->next_leaf;
		end_ptr = list->tail_guard;
		j = list->pivot_point.major;
	}

	for (ptr = extreme_ptr->next_leaf; (i < j) && (ptr != end_ptr) && (ptr->next_leaf != NULL);
			++i, ptr = ptr->next_leaf) {
		if ((*element_compare)(extreme_ptr->element, ptr->element) * vector < 0) {
			extreme_ptr = ptr;
			extreme_index = i+1;
		}
	}

	if (index != NULL) *index = extreme_index;
	return extreme_ptr;
}

/* Return the leaf node at the list pivot point. Only valid for pivot class lists.

	Will return the leaf node that is on the minor side (left side) of the conceptual pivot point.
	To insert an element on either side of the pivot, insert as normal to right hand (-> next) side.
	if it is a new minor leaf node it will be on the minor side, and shift the pivot point 1 to the
	right, if it is a major leaf node, it will be on the major side, and shift the pivot point one
	to the left.
	If there are no minor leaf nodes, or no leaf nodes at all, the head_guard is returned. If there
	are no major elements, the leaf node prior to the tail_guard is returned. Otherwise the pivot
	point leaf node is returned. In all cases insertions must be done at the right hand side.
	
	To traverse a list from this point, from the minor pivot to the end, start at the leaf node
	returned, moving to prior leaf nodes. From the major pivot to the end, start at the next leaf
	node and move forward (->next_leaf) */
static inline leaf_node *list__locate_pivot_leaf_node(	const list_object *restrict list,
														long int *index) {

	leaf_node *ptr;
	long int pivot_index;

	if (! list->list_pivot_enabled) return NULL;

	pivot_index = (long int)list->pivot_point.minor-1;

	if ((list->element_count == 0) || (list->pivot_point.minor == 0)) pivot_index = -1;
	if (list->pivot_point.major == 0) pivot_index = (long int)list->element_count-1;

	ptr = list__locate_leaf_node(list, pivot_index);

#ifdef DLIST_DEBUG
/* dbw -- i will remove this before release most likely, it is just a double
		protection safety measure and audit code, and has never been hit yet */
	if (ptr->internal->pivot.value == PIVOT_MAJOR) {
		fprintf(stderr, "\n==> Dlist INTERNAL ERROR ==> %s() - "\
				"found incorrect pivot leaf node, should be leaf node %ld, was %ld.\n\n",
				__func__, pivot_index-1, pivot_index);	
		pivot_index -= 1;
		ptr = list__locate_leaf_node(list, pivot_index);
	}
#endif		/* DLIST_DEBUG */

	*index = pivot_index+1;
	return ptr;
}

/* Inserts a leaf node in a list, to the right of the reference leaf node.
	Index is only used for adjusting the index optimizations. Index is not used to place
	the new leaf node, that is always done by reference. */
static inline int list__insert_leaf_node(	register list_object *restrict list,
											leaf_node *restrict reference,
											const void *element,
											const unsigned int index,
											const boolean adjust_mid,
											const int scope) {

	register leaf_node *ptr;
	size_t element_size;

	ptr = get__next_free_leaf_node(list);
	
	if (list->list_manages_elements == TRUE) {
		/* In NON DEBUG mode, this will cause a segmentation fault (fetch from NULL)
			if you failed to set a size function before adding elements to a
			list with the LIST_COPY_DATA attribute.									*/
		element_size = list->functions.size(element);
		ptr->element = (void *)malloc(element_size);
		if (! ptr->element) {
			fprintf(stderr, "\n==> Dlist INTERNAL ERROR ==> %s() unable to allocate memory ."\
					"- malloc() returned a NULL pointer when trying allocate "
					"memory for element, with error: %s.\n", __func__, strerror(errno));
			/* the users list will be left in an inconsistent state, with a damaged
				freep list. But as malloc() has failed, we are going to crash anyway */
			return ELEMENT_NOT_FOUND;
		} else {
#ifdef DLIST_FLUSH_MEMORY
			memset(ptr->element, 0x00, element_size);
#endif	/* DLIST_FLUSH_MEMORY */
			memcpy(ptr->element, element, element_size);
		}
	} else {
		ptr->element = (void *)element;
	}

	if (scope !=0) {
		/* enforce consistency, to prevent badly written pivot functions from making a mess of things */
		if (scope < 0) {
			ptr->internal->pivot.value = PIVOT_MINOR;
			list->pivot_point.minor++;
		} else {
			ptr->internal->pivot.value = PIVOT_MAJOR;
			list->pivot_point.major++;
		}
	} else {
		ptr->internal->pivot.value = ENTIRE_LIST;
	}

	link__leaf_node(ptr, reference);

	++list->element_count;

	if (adjust_mid) list__adjust_middle_pointer(list, index, LEAF_ADDED);

	if (list->list_finite_enabled) {
		while (list->element_count > list->max_element_count) {
			ptr = list->tail_guard->prior_leaf;
			list__remove_leaf_node(ptr, list->element_count-1);
		}
	}

#ifdef DLIST_DEBUG
	/* don't run the audit if the caller does not want the mid point adjusted. They will
		call for an audit when they are finished whatever they are doing. */
	if (adjust_mid) assert(list__audit(list) == LIST_OK);
#endif		/* DLIST_DEBUG */

	return LIST_OK;
}

/* Removes a leaf node from a list using its reference.
	Index is only used for adjusting the index optimizations. */
static inline void list__remove_leaf_node(	leaf_node *restrict old_leaf_node,
											const unsigned int index) {
	list_object *ptr;

	
	ptr = (list_object *)old_leaf_node->internal->master;

#ifdef DLIST_DEBUG
/* dbw -- i will remove this before release most likely, it is just a
		triple protection safety measure, and has never been hit yet */
if (index != 0) {
	leaf_node *temp_ptr;
	temp_ptr = list__locate_leaf_node(ptr, (long int)index);
	if (temp_ptr != old_leaf_node) {
		fprintf(stderr, "\n==> Dlist INTERNAL ERROR ==> %s() - "\
				"called with index not indexing the leaf node.\n\n", __func__);	
	}
}
#endif		/* DLIST_DEBUG */

	list__adjust_middle_pointer(ptr, index, LEAF_REMOVED);

	unlink__leaf_node(old_leaf_node);

	if (ptr->list_pivot_enabled == TRUE) {
		if (old_leaf_node->internal->pivot.value < ENTIRE_LIST) --ptr->pivot_point.minor;
		else --ptr->pivot_point.major;
	}

	return__free_leaf_node(old_leaf_node);
	
	--ptr->element_count;

	return;
}

/* Adjust the middle pointer of a list to compensate for a change in the list at a known position.

	If called for a leaf insertion, must be called after the leaf is added, and after
		list->element_count is incremented.
	if called for a leaf removal, must be called before the leaf is removed, before the
		leaf pointers are changed, and before the list->element_count is decremented. */
static inline void list__adjust_middle_pointer(	register list_object *restrict list,
												register const unsigned int index,
												const boolean action) {

	register leaf_node *mid_ptr;
	register unsigned int count;
	
	count = list->element_count;
	mid_ptr = list->mid_point;

	if (action == LEAF_ADDED) {
		/* if this is the first leaf added to the list, then set the mid point and exit */
		if ((count == 1) && (mid_ptr == NULL)) {
			list->mid_point = list->head_guard->next_leaf;
			return;
		}
		/* if there is no mid point, then the list has been updated by reference,
			mid point was lost */
		if (! mid_ptr) return;

		if (count % 2) {	/* now odd */
			if (index >= (count-1)/2) list->mid_point = mid_ptr->next_leaf;
		} else {										/* now even */
			if (index <= (count-1)/2) list->mid_point = mid_ptr->prior_leaf;
		}
	} else {
		/* if there is no mid point, then the list has been updated by reference,
			mid point was lost */
		if (! mid_ptr) return;
	
		if (count == 1) {
			list->mid_point = NULL;
			return;
		}	
		if (count % 2) {	/* now odd */
			if (index >= count/2) list->mid_point = mid_ptr->prior_leaf;
		} else {										/* now even */
			if (index < count/2) list->mid_point = mid_ptr->next_leaf;
		}
	}

	return;
}

/* Add a new leaf node to an external list, from an existing element.

	Called while processing elements between 2 lists, such as list_concatenate()
	list_move_index() list_split_pivot() etc
	
	DO NOT use for general purpose leaf node adding, it is too rigid for most
	applications. Use the proper procedure outlined in list_insert_index()
	or list_insert_ref() as examples. If you short cut the general process
	you stand a good chance of messing the list up. 									*/
static inline int list__add_element_extra_list(	list_object *restrict dest_list,
												const leaf_node *ptr) {

	leaf_node *new_ptr;
	int scope, ret;

	if (dest_list->list_pivot_enabled) {
		scope = dest_list->functions.pivot(ptr->element);
		if (scope < ENTIRE_LIST) {
			ret = list__insert_leaf_node(dest_list, dest_list->head_guard,
											ptr->element, 0, TRUE, scope);
		} else {
			ret = list__insert_leaf_node(dest_list, dest_list->tail_guard->prior_leaf,
											ptr->element, dest_list->element_count, TRUE, scope);
		}
	} else {
		ret = list__insert_leaf_node(dest_list, dest_list->tail_guard->prior_leaf,
										ptr->element, dest_list->element_count, TRUE, 0);
	}
	if ((ret == LIST_OK) && (dest_list->functions.insert)) {
		new_ptr = dest_list->tail_guard->prior_leaf;
		dest_list->functions.insert(new_ptr->element);
	}

	return ret;
}

/* called by list_search_copy() list search_move() list_evaluate_copy() list_evaluate_move()
	list_range_copy() list_range_move() to perform the actual operation.
	
	See those functions for how to set up the arguments, this is not a general
	purpose function, and no description is needed. It only services those user called
	functions and has no other purpose.
	
	Order will always be LIST_ASCENDING for RANGE type operations
	datum_max will always be NULL for non-range type operations
	compare_elements == FALSE we do a search (and ignore order, and datum_max),
						otherwise TRUE we call compare function
	move_leaf_node should be obvious, copy or move it instead
																								*/
static int list__leaf_node_evaluate_range(	list_object *restrict list,
											const void *datum_min,
											const boolean order,
											const void *datum_max,
											const boolean compare_elements,
											const boolean move_leaf_node,
											list_object *restrict dest_list) {

	leaf_node *ptr;
	int ret = LIST_OK;
	int direction;
	unsigned int count = 0;
	unsigned int index = 0;
	boolean copy_leaf_node = FALSE;
	boolean (*element_search)(const void *element, const void *key);
	int (*element_compare)(const void *element_a, const void *element_b);


	if (order == LIST_ASCENDING) direction = 1;
	else direction = -1;
	
	element_search = list->functions.search;
	element_compare = list->functions.compare;
	dest_list->list_sorted = FALSE;
	
	for (ptr = list->head_guard->next_leaf;
			(ret == LIST_OK) && (ptr) && (ptr != list->tail_guard);
			++index, ptr = ptr->next_leaf) {
		if (compare_elements) {
			if (order == LIST_ASCENDING) {
				if ((*element_compare)(datum_min, ptr->element) * direction <= 0) copy_leaf_node = TRUE;
			} else {
				if ((*element_compare)(datum_min, ptr->element) * direction < 0) copy_leaf_node = TRUE;
			}
		} else {
			if (element_search) {
				if ((*element_search)(ptr->element, datum_min)) copy_leaf_node = TRUE;
			} else {
				if (ptr->element == datum_min) copy_leaf_node = TRUE;
			}
		}
		if ((copy_leaf_node) && (datum_max) && (compare_elements)) {
			if ((*element_compare)(datum_max, ptr->element) * direction >= 0) copy_leaf_node = TRUE;
			else copy_leaf_node = FALSE;
		}		
		if (copy_leaf_node) {
			/* first copy the element to the destination list */
			ret = list__add_element_extra_list(dest_list, ptr);
			/* now remove the element in the input list */
			if ((ret == LIST_OK) && (move_leaf_node)) {
				if (list->functions.remove) list->functions.remove(ptr->element);
				ptr = ptr->prior_leaf;
				list__remove_leaf_node(ptr->next_leaf, index--);
			}
			++count;
			copy_leaf_node = FALSE;
		}
	}
	
	if (ptr == NULL) return LIST_ERR;						/* major oops ! */
	if (count > INT_MAX) count = INT_MAX;

#ifdef DLIST_DEBUG
	assert(list__audit(dest_list) == LIST_OK);
#endif		/* DLIST_DEBUG */

	return((ret == LIST_OK) ? count : ret);
}

/* add a new leaf node to a list, after the reference node provided. */
static inline void link__leaf_node(	register leaf_node *restrict new_leaf_node,
									register leaf_node *restrict reference) {

	/* Link the new leaf into the list */
	new_leaf_node->next_leaf = reference->next_leaf;
	reference->next_leaf = reference->next_leaf->prior_leaf = new_leaf_node;
	
	/* finish the new leaf pointers */
	new_leaf_node->internal->master = reference->internal->master;
	new_leaf_node->prior_leaf = reference;

	return;
}

static inline leaf_node *unlink__leaf_node(register leaf_node *restrict old_leaf_node) {

	register leaf_node *ptr;
	register list_object *master_node;
	
	ptr = old_leaf_node;
	/* are we removing the current iteration leaf node ?, of so adjust the sequence */
	master_node = old_leaf_node->internal->master;
	if (master_node->iteration_point == old_leaf_node) {
		if (master_node->iteration_direction == TAIL_OF_LIST) master_node->iteration_point = ptr->prior_leaf;
		else master_node->iteration_point = ptr->next_leaf;
	}
	
	/* unlink the leaf node */
	ptr->prior_leaf->next_leaf = ptr->next_leaf;
	ptr->next_leaf->prior_leaf = ptr->prior_leaf;
		
	return ptr;
}

/* initialize a list_object provided, used for list_create_xx() */
static inline int list__initialize_list_object(	list_object *restrict new_list,
												const boolean copy_data,
												const element_size size_function,
												const unsigned int leaf_block_size,
												const char *calling_func) {

	new_list->element_count = 0;
	
	/* allocate the resources needed */
#ifndef DLIST_USE_MALLOC
	new_list->leaf_block = NULL;
	new_list->freep = NULL;
	new_list->allocation_size = leaf_block_size;
	if (allocate__leaf_node_block(new_list, leaf_block_size) == NULL) {
#ifdef DEBUG
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"unable to create leaf node block for new list.\n\n", calling_func);
#endif	
		return LIST_ERR;
	}	
#endif	/* ! DLIST_USE_MALLOC */
	
	if (copy_data) {
		new_list->list_manages_elements = TRUE;
		if (! size_function) {
#ifdef DEBUG
			fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
					"NULL size function provided for new list, with LIST_COPY_DATA "\
					"attribute.\n\n", calling_func);
#endif		/* DEBUG */
			return LIST_ERR;
		} else {
			new_list->functions.size = size_function;
		}
	} else {
		new_list->list_manages_elements = FALSE;
		new_list->functions.size = NULL;
#ifdef DEBUG
		if (size_function) {
			fprintf(stderr, "\n==> Dlist Debug WARNING ==> %s() - "\
					"size function provided for new list, with LIST_DO_NOT_COPY "\
					"attribute.\n\n", calling_func);
		}
#endif		/* DEBUG */
	}

	new_list->head_guard = get__next_free_leaf_node(new_list);
	new_list->tail_guard = get__next_free_leaf_node(new_list);

	/* Assign the pointers back to the master node */
	new_list->head_guard->internal->master = new_list->tail_guard->internal->master = new_list;
	/* Assign the pointers to each other, and to the fake mid point */
	new_list->head_guard->next_leaf = new_list->tail_guard;
	new_list->head_guard->prior_leaf = NULL;
	new_list->tail_guard->next_leaf = NULL;
	new_list->tail_guard->prior_leaf = new_list->head_guard;

	new_list->mid_point = NULL;
	new_list->list_sorted = FALSE;
	new_list->head_guard->element = new_list->tail_guard->element = NULL;	/* No data here */
	new_list->head_guard->internal->pivot.value = ENTIRE_LIST;
	new_list->tail_guard->internal->pivot.value = ENTIRE_LIST;
	
	/* Set defaults for other list functions */
	new_list->functions.compare = NULL;
	new_list->functions.pivot = NULL;
	new_list->functions.search = NULL;
	new_list->functions.insert = NULL;
	new_list->functions.remove = NULL;
	new_list->functions.iterate = NULL;
	
	/* Set defaults for iterations */
	new_list->iteration_point = NULL;
	new_list->iteration_enabled = FALSE;
	new_list->iteration_direction = FALSE;

	/* set defaults for non-pivot class lists */
	new_list->list_pivot_enabled = FALSE;
	new_list->pivot_point.minor = 0;
	new_list->pivot_point.major = 0;

	/* set defaults for non-finite class lists */
	new_list->list_finite_enabled = FALSE;
	new_list->max_element_count = 0;

	return LIST_OK;
}

#ifdef DEBUG

/* validate a list_object provided in a function call
	
	provide the list, obviously,
	the type of function request, LIST_WRITE or 0, and
	the function name to display on error messages.									*/
static inline boolean list__validate_list_object(	const list_object *restrict list,
													const boolean list_write,
													const char *calling_func) {

	int error_type;
	boolean object_ok;
	
	if (list == NULL) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with NULL list pointer.", calling_func);
		error_type = 1;
		object_ok = FALSE;
	} else {
		object_ok = TRUE;	
	}
	
	if (object_ok) {
		if ((! list->head_guard) || (! list->tail_guard)) {
			fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
					"called with pointer %p to an uninitialized list object. You must call "\
					"list_create() first.", calling_func, list);
			error_type = 2;
			object_ok = FALSE;
		} else {
			object_ok = TRUE;	
		}
	}

	if (object_ok) {
		if ((list->head_guard) && (list->head_guard->next_leaf) && (list->head_guard->next_leaf->prior_leaf)) {
			if (list->head_guard->next_leaf->prior_leaf == list->head_guard) object_ok = TRUE;
			else { error_type = 3; object_ok = FALSE; }
		}
	}
	if (object_ok) {
		if (list->head_guard->internal->master == list) object_ok = TRUE;
		else { error_type = 4; object_ok = FALSE; }
	}
	if (object_ok) {
		if (list->head_guard->next_leaf->internal->master == list) object_ok = TRUE;
		else { error_type = 5; object_ok = FALSE; }
	}
	if (object_ok) {
		if ((list->tail_guard->prior_leaf) && (list->tail_guard->prior_leaf->next_leaf)) {
			if (list->tail_guard->prior_leaf->next_leaf == list->tail_guard) object_ok = TRUE;
			else { error_type = 6; object_ok = FALSE; }
		}
	}
	if (object_ok) {
		if (list->tail_guard->internal->master == list) object_ok = TRUE;
		else { error_type = 7; object_ok = FALSE; }
	}
	if (object_ok) {
		if (list->tail_guard->prior_leaf->internal->master == list) object_ok = TRUE;
		else { error_type = 8; object_ok = FALSE; }
	}
	if (object_ok) {
		if (list->list_pivot_enabled) {
			if ((list->head_guard->internal->pivot.value == PIVOT_MINOR) && (list->tail_guard->internal->pivot.value == PIVOT_MAJOR)) object_ok = TRUE;
				else { error_type = 9; object_ok = FALSE; }
		} else {
			if ((list->head_guard->internal->pivot.value == ENTIRE_LIST) && (list->tail_guard->internal->pivot.value == ENTIRE_LIST)) object_ok = TRUE;
				else { error_type = 10; object_ok = FALSE; }
		}
	}
	if (object_ok) {
		if (list->mid_point) {
			if (list->mid_point->internal->master == list) object_ok = TRUE;
			else { error_type = 11; object_ok = FALSE; }
		}
	}
	if (object_ok) {
		if (list->list_finite_enabled) {
			if (list->max_element_count != 0) object_ok = TRUE;
			else { error_type = 12; object_ok = FALSE; }
		}
	}
	if ((object_ok) && (list_write)) {
		if ((list->list_pivot_enabled) && (list->functions.pivot == NULL)) {
			fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
					"called for pivot list with NULL pivot function. You must call "\
					"list_set_pivot_function() before adding elements to the list.", calling_func);
			error_type = 13;
			object_ok = FALSE;
		} else {
			object_ok = TRUE;	
		}
	}
	if ((object_ok) && (list_write)) {
		if ((list->list_manages_elements == TRUE) && (list->functions.size == NULL)) {
			fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
					"called for list with LIST_COPY_DATA attribute, and a NULL size function. "\
					"You must call list_set_size_function() before adding elements to the "\
					"list.", calling_func);
			error_type = 14;
			object_ok = FALSE;
		} else {
			object_ok = TRUE;	
		}
	}

	if (! object_ok) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with pointer %p to invalid list_object, (error type %d).\n\n", calling_func, list, error_type);
	}
	
	return object_ok;
}
#endif		/* DEBUG */

/* validate an element reference provided in a function call */
static inline int list__validate_reference(	const list_object *restrict list,
											const element_reference *restrict leaf_node,
											const char* calling_function) {

#ifdef DEBUG
	if (leaf_node->internal == NULL) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with invalid reference pointer, element deformed, NULL internal pointer.\n", calling_function);
		return ELEMENT_INVALID;
	}
	if (leaf_node->internal->master != list) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with invalid reference pointer, element not part of list.\n", calling_function);
		return ELEMENT_INVALID;
	}
	if (leaf_node->prior_leaf == NULL) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with invalid reference pointer, pointing to head_guard leaf "\
				"node.\n", calling_function);
		return ELEMENT_INVALID;
	}
	if (leaf_node->prior_leaf->internal->master != list) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with invalid reference pointer, element linked into list with "\
				"invalid prior leaf node.\n", calling_function);
		return ELEMENT_INVALID;
	}
	if (leaf_node->next_leaf == NULL) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with invalid reference pointer, pointing to tail_guard leaf "\
				"node.\n", calling_function);
		return ELEMENT_INVALID;
	}
	if (leaf_node->next_leaf->internal->master != list) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with invalid reference pointer, element linked into list with "\
				"invalid next leaf node.\n", calling_function);
		return ELEMENT_INVALID;
	}
	/* this was my original check for a guard node, but decided to make more specific checks
		above. Left this in just in case of someone's weird programming error. */
	if (leaf_node->element == NULL) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with invalid reference pointer, referenced element has NULL datum "\
				"pointer.\n", calling_function);
		return ELEMENT_NO_DATUM;
	}
#ifndef DLIST_USE_MALLOC
	leaf_node_array *block_ptr;
	for (block_ptr = list->leaf_block;
			(block_ptr != NULL) && (block_ptr != leaf_node->internal->leaf_block);
			block_ptr = block_ptr->next_block);
	if (block_ptr == NULL) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with invalid reference pointer, referenced element resides in a "\
				"leaf node block, that does not belong to this list.\n\n", calling_function);
		return ELEMENT_NO_DATUM;
	}
#endif	/* ! DLIST_USE_MALLOC */

#else	/* DEBUG */
	/* Debug off, we assume the user debugged their code properly, and only do a basic check */
	if (leaf_node->internal->master != list) return ELEMENT_INVALID;
#endif	/* DEBUG */

	return LIST_OK;
}

/* called by list_search_copy() list search_move() list_evaluate_copy() list_evaluate_move()
	list_range_copy() list_range_move() to validate arguments prior to the actual operation.
	
	See those functions for how to set up the arguments, this is not a general
	purpose function, and no description is needed. It only services those user
	called functions and has no other purpose.													*/
int list__validate_evaluate_range(	list_object *restrict list,
									const void *datum_min,
									const void *datum_max,
									const boolean compare_elements,
									const boolean range,
									list_object *restrict dest_list,
									const char *calling_func) {

#ifdef DEBUG
	if (list__validate_list_object(list, FALSE, calling_func) == FALSE)
		return LIST_ERR;
	if (list__validate_list_object(dest_list, LIST_WRITE, calling_func) == FALSE)
		return LIST_ERR;
	if (datum_min == NULL) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with NULL datum%s pointer.\n\n",
				calling_func, (range) ? "_min" : "");
		return ELEMENT_NO_DATUM;
	}
	if (dest_list == list) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with the same list for input and output.\n\n", calling_func);
		return LIST_ERR;
	}
	if (list->element_count < 1) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called with empty input list.\n\n", calling_func);
		return 0;
	}
	if (dest_list->list_finite_enabled) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"called on a finite class list.\n\n", calling_func);
		return LIST_ERR;
	}
	if (list->iteration_enabled == TRUE) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"input list has iteration enabled.\n\n", __func__);
		return LIST_ERR;
	}
	if (dest_list->iteration_enabled) {
		fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
				"destination list has iteration enabled.\n\n", __func__);
		return LIST_ERR;
	}
	if (compare_elements) {
		if (list->functions.compare == NULL) {
			fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
					"NULL compare function.\n\n", calling_func);
			return LIST_ERR;
		}
		if (range) {
			if (datum_max == NULL) {
				fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
						"called with NULL datum_max pointer.\n\n", calling_func);
				return ELEMENT_NO_DATUM;
			}
			if (datum_max == datum_min) {
				fprintf(stderr, "\n==> Dlist Debug ERROR ==> %s() - "\
						"called with identical dataum_min & datum_max pointers.\n\n",
						calling_func);
				return LIST_ERR;
			}
		}
	}
#else
	if ((list == NULL)
			|| (dest_list == NULL)
			|| (dest_list == list)
			|| (list->iteration_enabled)
			|| (dest_list->iteration_enabled)
			|| (dest_list->list_finite_enabled))
		return LIST_ERR;
	if (list->element_count < 1)
		return 0;
	if (datum_min == NULL)
		return ELEMENT_NO_DATUM;
	if (compare_elements) {
		if (list->functions.compare == NULL)
			return LIST_ERR;
		if (range) {
			if (datum_max == NULL)
				return ELEMENT_NO_DATUM;
			if (datum_max == datum_min)
				return LIST_ERR;
		}
	}
#endif

	return LIST_OK;
}


/*	=====================================================================
		Functions to manage the leaf nodes and leaf node block chains
	=====================================================================	*/


static inline leaf_node *get__next_free_leaf_node(list_object *restrict list) {

	leaf_node *ptr;

#ifndef DLIST_USE_MALLOC

	/* Did we run out of leaf nodes ? if so we must allocate another block */
	if (list->freep == NULL) {
		if (allocate__leaf_node_block(list, list->allocation_size) == NULL) {
			fprintf(stderr, "\n==> Dlist Debug INTERNAL ERROR ==> %s() - "\
					"unable to create new leaf node block.\n\n", __func__);
			return NULL;
		}
	}

	ptr = list->freep;
	list->freep = ptr->internal->next_freep;
	ptr->internal->next_freep = NULL;
	++((leaf_node_array *)ptr->internal->leaf_block)->nodes_in_use;

#else
	/* No leaf node blocks, just malloc() it */
	
	ptr = malloc(sizeof(leaf_node) + sizeof(leaf_node_data));
	if (! ptr) {
		fprintf(stderr, "\n==> Dlist INTERNAL ERROR ==> %s() unable to allocate memory ."\
				"- malloc() returned a NULL pointer when trying allocate "
				"memory for new leaf node, with error: %s.\n", __func__, strerror(errno));
		return NULL;
	}
	ptr->internal = (void *)ptr + sizeof(leaf_node);
#endif	/* ! DLIST_USE_MALLOC */
	
	return ptr;
}

static inline void return__free_leaf_node(leaf_node *restrict leaf_node) {

	list_object *ptr;
	
	if (leaf_node->element) {
		ptr = (list_object *)leaf_node->internal->master;
		if (ptr->list_manages_elements) {
#ifdef DLIST_FLUSH_MEMORY
			int element_size;
			if (ptr->functions.size) {
				element_size = ptr->functions.size(leaf_node->element);
				if (element_size > 0) memset(leaf_node->element, 0x00, element_size);
			}
#endif	/* DLIST_FLUSH_MEMORY */
			if (leaf_node->element) free(leaf_node->element);
		}
		leaf_node->element = NULL;
	}

#ifndef DLIST_USE_MALLOC
	/* Update the free leaf_node block and add the leaf node to the free list */
	--((leaf_node_array *)leaf_node->internal->leaf_block)->nodes_in_use;
	ptr = (list_object *)leaf_node->internal->master;
	leaf_node->internal->next_freep = ptr->freep;
	ptr->freep = leaf_node;
#endif	/* ! DLIST_USE_MALLOC */
	
	/* clean out the leaf node, in case someone kept a reference to it, and to be tidy */
	leaf_node->internal->master = leaf_node->next_leaf = leaf_node->prior_leaf = NULL;
	leaf_node->internal->pivot.value = 0;

#ifdef DLIST_USE_MALLOC
	/* No leaf node blocks, we just free() it. */
	
	if (leaf_node) {
#ifdef DLIST_FLUSH_MEMORY
		memset(leaf_node, 0x00, sizeof(leaf_node__t));
#endif	/* DLIST_FLUSH_MEMORY */
		if (leaf_node) free(leaf_node);
	}
#endif	/* DLIST_USE_MALLOC */

	return;
}

/* Return the number of leaf nodes to allocate per block, based on the caller
	supplied list size hint.
	
	Callers may provide this hint, or leave it at 0.
	The setting of the hint allows a more efficient trade-off between resources
	consumed and operation speeds of various functions. Setting the hint to an
	appropriate value can reduce processing times for searches and sorts etc,
	for very large lists, by 25% or more. Setting this hint too high will
	consume unneeded resources, setting it too low will increase processor
	overhead.
	
	Getting the balance right during code testing and beta-testing will
	produce better results with less resources consumed. If you compile D-List
	forcing it to use malloc() instead, these hints are ignored.
	
	These hints can be particularly helpful to embedded systems developers.
	Setting the hint lower may cause the lists to process some functions slower,
	that is assuming the lists are very large, but will use resources far more
	efficiently, and reduce the empty space to a very low number. Embedded
	systems developers may also wish to experiment with the default setting
	(LEAF_NODE_ARRAY_SIZE) as well. Others should probably leave it alone, as a
	lot of testing and experimentation went into setting it originally.
	
	If you are running test code with DEBUG_DLIST defined, or developing D-List
	itself, use either the Tiny or Small hint only. Due to the excessive audit
	that debugging D-List creates, especially the double audit of the freep list,
	having very large and very empty allocation blocks will cause enormous
	slow-downs. For example, running the regression test suite for 1000 cycles
	with a default or larger allocation block size with DEBUG_DLIST defined will
	cause it to take 45 mins or more to complete. Without defining DEBUG_DLIST it
	takes about 2 mins for a 1000 cycle run. Of course the whole point of the
	regression tests is to stress and audit the D-List code itself. But there are
	also times when running huge numbers of cycles with debugging off, helps
	isolate random or other creeping errors in the regression test code itself.		*/
static inline unsigned int lookup__leaf_node_block_size(	const unsigned int hint,
															const char *calling_func) {

	unsigned int block_sizes[LIST_SIZE_MEGA+1] = { 	LEAF_NODE_ARRAY_SIZE,
													110,
													200,
													2000,
													5000,
													15000,
													30000 };

	unsigned int size = block_sizes[LIST_NO_SIZE_HINT];
#ifndef DLIST_USE_MALLOC


	if (hint > LIST_SIZE_MEGA) {
		size = block_sizes[LIST_NO_SIZE_HINT];
#ifdef DEBUG
		fprintf(stderr, "\n==> Dlist Debug WARNING ==> %s() - "\
				"invalid size hint (%u) provided for new list, must be between "\
				"%u and %u inclusive. Default hint size used instead.\n\n",
				calling_func, hint, LIST_NO_SIZE_HINT, LIST_SIZE_MEGA);
#endif	
	} else {
		size = block_sizes[hint];
	}

#endif	/* ! DLIST_USE_MALLOC */
	return size;
}

#ifndef DLIST_USE_MALLOC
/* This function will add a new leaf node block to the front of the existing chain.
	It is called by list_create() to place the initial leaf node block, and then
	as required to add more, as the existing ones fill up. 							*/
static inline leaf_node_array *allocate__leaf_node_block(	list_object *restrict list,
															const int block_count) {

	leaf_node_array *ptr_next;
	leaf_node_internals *ptr_internals;
	int i, j;
	int alloc_size, internal_size;
	
/* IMPORTANT NOTE -  To dramatically improve performance i split the leaf node,
	into the base form and an extension called internals. Each leaf node is set
	up with a pointer to its corresponding internal block, and added to the
	free list. On x86 systems there is a serious performance reduction if you
	put the structs together again. It creates prefetch conditions that exceed
	the size of a RAM fetch and requires multiple fetches. For search, sort and
	other leaf node traversal operations it will introduce unacceptable overheads.	*/

	if (block_count == 0) j = LEAF_NODE_ARRAY_SIZE;
	else j = block_count;
	alloc_size = sizeof(leaf_node_array) + (j * sizeof(leaf_node));
	internal_size = sizeof(leaf_node_internals) + (j * sizeof(leaf_node_data));
	ptr_next = malloc(alloc_size);
	ptr_internals = malloc(internal_size);
	if ((! ptr_next) || (! ptr_internals)){
#ifdef DLIST_DEBUG
		fprintf(stderr, "\n==> Dlist INTERNAL ERROR ==> %s() unable to allocate memory ."\
				"- malloc() returned a NULL pointer when trying allocate "
				"memory for new leaf node block, with error: %s.\n", __func__, strerror(errno));
#endif
		return NULL;
	}
	memset(ptr_next, 0x00, alloc_size);
	memset(ptr_internals, 0x00, internal_size);

	ptr_next->master = list;
	ptr_internals->master = list;
	if (list->leaf_block) ptr_next->block_number = list->leaf_block->block_number + 1;
	else ptr_next->block_number = 1;

	ptr_next->next_block = list->leaf_block;
	list->leaf_block = ptr_next;
	ptr_next->nodes_in_use = 0;
	ptr_next->nodes_allocated = j;
	ptr_next->internals = ptr_internals;
	
	/* initialize each of the allocated leaf nodes, and add them all to the free list */
	for (i = 0; i < j; ++i) {
		ptr_next->leaf_nodes[i].internal = &ptr_internals->leaf_nodes_data[i];
		ptr_next->leaf_nodes[i].internal->master = NULL;
		ptr_next->leaf_nodes[i].internal->leaf_block = ptr_next;
		ptr_next->leaf_nodes[i].internal->next_freep = list->freep;
		list->freep = &ptr_next->leaf_nodes[i];
	}
	
	return ptr_next;
}
#endif	/* ! DLIST_USE_MALLOC */

#ifndef DLIST_USE_MALLOC
/* removes and frees the complete chain of existing blocks with no allocated leaf nodes,
	any remaining leaf node blocks are left linked to the list.
	
	This function is designed to be called by list_clear() after a list is emptied, and
	list_erase() when the list is to be destroyed. */
static inline int release__leaf_node_blocks(list_object *restrict list) {

	leaf_node_array *ptr, *prior_ptr, *next_ptr;
	int j;

	if (list->element_count == 0) {
		list__flush_freep_list(list, list->head_guard->internal->leaf_block);
		for (ptr = list->leaf_block, prior_ptr = NULL; (ptr != NULL); ptr = next_ptr) {
			if (ptr->nodes_in_use == 0) {
				if (prior_ptr) prior_ptr->next_block = ptr->next_block;
				else list->leaf_block = ptr->next_block;
				next_ptr = ptr->next_block;

#ifdef DLIST_FLUSH_MEMORY
				memset(ptr, 0x00, (sizeof(leaf_node_array) + (sizeof(leaf_node__t) * ptr->nodes_allocated)));
#endif	/* DLIST_FLUSH_MEMORY */
			
				free(ptr->internals);				/* if -> internals is NULL i want it to crash, somebody destroyed our space */
				free(ptr);
			} else {
				prior_ptr = ptr;
				next_ptr = ptr->next_block;
			}
		}
	} else {
		/* not a flush, a memory compression */
		for (ptr = list->leaf_block, prior_ptr = NULL; (ptr != NULL); ptr = next_ptr) {
			if (ptr->nodes_in_use == 0) {
#ifdef DLIST_DEBUG
				int count;
				count = list__audit_freep_list(list, ptr->block_number);
				if (count != ptr->nodes_allocated) {
					fprintf(stderr, "\n==> Dlist INTERNAL ERROR ==> %s() internal audit failed "\
							"for leaf node block #%d. nodes_in_use = 0, nodes_allocated = %d "\
							"nodes found in free list = %d.\n", __func__, ptr->block_number, ptr->nodes_allocated, count);
					return LIST_ERR;
				}
#endif		/* DLIST_DEBUG */

				j = list__trim_freep_list(list, ptr);
#ifdef DLIST_DEBUG
				if (count != j) {
					fprintf(stderr, "\n==> Dlist INTERNAL ERROR ==> %s() internal audit failed "\
							"for leaf node block #%d. trim failed to equal my count -- nodes_in_use = 0, nodes_allocated = %d "\
							"nodes found in free list = %d, nodes trimmed %d.\n", __func__, ptr->block_number, ptr->nodes_allocated, count, j);
#else
				if (j != ptr->nodes_allocated) {
#endif		/* ! DLIST_DEBUG */
					return LIST_ERR;
				}
				if (prior_ptr) prior_ptr->next_block = ptr->next_block;
				else list->leaf_block = ptr->next_block;
				next_ptr = ptr->next_block;
	
#ifdef DLIST_FLUSH_MEMORY
				memset(ptr, 0x00, (sizeof(leaf_node_array) + (sizeof(leaf_node__t) * ptr->nodes_allocated)));
#endif	/* DLIST_FLUSH_MEMORY */
				
				free(ptr->internals);
				free(ptr);
			} else {
				prior_ptr = ptr;
				next_ptr = ptr->next_block;
			}
		}
	}	
	return LIST_OK;
}
#endif	/* ! DLIST_USE_MALLOC */

#ifndef DLIST_USE_MALLOC

/* remove all leaf nodes in the free leaf node list that belong all leaf
	node blocks except the block containing the guard nodes (#0),
	which is passed as an argument. 	
	
	This is called during the clean up after a list_clear or similar 							*/
static inline void list__flush_freep_list(	list_object *restrict list,
											const leaf_node_array *restrict leaf_node_block) {

	leaf_node *ptr, *prior_ptr;

	/* search the free list, looking for, and removing, all leaf nodes */
	for (ptr = list->freep, prior_ptr = NULL; (ptr != NULL); ptr = ptr->internal->next_freep) {
		if (ptr->internal->leaf_block != leaf_node_block) {
			if (prior_ptr) {
				prior_ptr->internal->next_freep = ptr->internal->next_freep;
			} else {
				list->freep = ptr->internal->next_freep;
			}
		} else {
			prior_ptr = ptr;
		}
	}

	return;
}

/* remove all leaf nodes in the free leaf node list that belong to the leaf
	node block.

	returns the number of free leaf nodes found that match the block pointer
			and were removed from the free list.
			
	Called during a memory compression / optimization 											*/
static inline int list__trim_freep_list(	list_object *restrict list,
											const leaf_node_array *restrict leaf_node_block) {

	leaf_node *ptr, *prior_ptr;
	int count = 0;

	/* search the free list, looking for, and removing, leaf nodes from specified block */
	for (ptr = list->freep, prior_ptr = NULL;
			(count < leaf_node_block->nodes_allocated) && (ptr != NULL);
			ptr = ptr->internal->next_freep) {
		if (ptr->internal->leaf_block == leaf_node_block) {
			++count;
			if (prior_ptr) {
				prior_ptr->internal->next_freep = ptr->internal->next_freep;
			} else {
				list->freep = ptr->internal->next_freep;
			}
		} else {
			prior_ptr = ptr;
		}
	}

	return count;
}
#endif	/* ! DLIST_USE_MALLOC */


#ifndef DLIST_USE_MALLOC
/* returns a count of the available leaf nodes in the lists leaf node blocks */
static inline unsigned int list__locate_open_leaf_node_count(list_object *restrict list) {

	leaf_node_array *ptr;
	unsigned int count;

	for (count = 0, ptr = (leaf_node_array *)list->leaf_block;
			(ptr != NULL);
			count += (ptr->nodes_allocated - ptr->nodes_in_use), ptr = ptr->next_block);

	return count;
}
#endif	/* ! DLIST_USE_MALLOC */

/*	=====================================================================
					Functions to sort leaf nodes in a list
	=====================================================================	*/
	
/*	There are many different sort algoritms provided here in D-List.
	
	list__sort_quicker_sort() is the fastest general usage sort, it uses an
		optimized array reduction algorithm, and maintains complete integrity
		of element references. Its only drawback is the need to allocate extra
		memory to process the sort. Some limited memory platforms may not be
		able to use it. Otherwise all other callers should use this unless they
		need a safe sort, in which case use the function list_safe_sort().
	list__sort_insertion_sort() is called by list_safe_sort() uses an insertion
		sort optimized for linked lists, which does not shuffle all the elements
		during each pass. It will process all the elements in a safe order manner.
		It is very slow compared to either of the quicksorts provided, and suffers
		from the N-squared problem of all insertion sorts. It is however much
		faster than a selection sort. It is also used by the list_sort_quick_sort()
		subsystem to finish small batches which reduce to small ranges. on x86 (64
		or 32 bit) the optimal setting of DLIST_QUICKSORT_MIN_SIZE is 24
	list__sort_insertion_index_sort() is a more normal version of an insertion sort
		for sorting elements in arrays. It is too slow to use, and is not called
		by any D-List function. It is left for users to test if they have an
		unusual platform or requirements. It is only useful if called by the
		list__sort_quicker_sort() subsystem, but in my benchmarking it is too slow.
	list__sort_selection_sort() is an optimized version of a selection sort for
		sorting elements in linked lists. It is too slow to use, and is not called
		by any D-List function. It is left for users to test if they have an
		unusual platform or requirements. It is only useful if called by the
		list__sort_quick_sort() subsystem, but in my benchmarking it is far too
		slow, running on average 50% slower than list__sort_insertion_sort().
	list__sort_selection_index_sort() is a more normal version of a selection sort
		for sorting elements in arrays. It is far too slow to use, and is not called
		by any D-List function. It is left for users to test if they have an
		unusual platform or requirements. It is only useful if called by the
		list__sort_quicker_sort() subsystem, but in my benchmarking it is so slow
		as to be totally useless.
	list_sort_quick_sort() is a specialized quicksort algoritm i developed to sort
		linked lists extremely efficiently. It has subversions, which are invoked
		by the use of the fast_sort argument set true or false. If set false it
		produces a slower sort that is about 25% slower than the best competitor
		i can find in the market, but sorts while keeping element reference
		integrity and not using additional memory resources. It is the best choice
		for limited memory/embedded systems. In fast mode, this sort is blazingly
		fast, it sorts large lists in half the time, an average savings of 55-60%
		It is also 10% faster than list__sort_quicker_sort() but does not maintain
		the integrity of element references. list__sort_quicker_sort()  is the best
		general usage choice and runs about 25% faster than the best competitor i
		can find on very large lists.
		
		ALL sort subsystems that can be called from user callable functions have the
		same function interface. The other sorts that are callable from within  the
		list__sort_quicker_sort() subsystem have an incompatible interface, and must
		be called only from a lower level primitive. This was a deliberate design
		decision, so the functions cannot be mixed up by accident. And anyway the
		index based insertion and selection sorts are so inefficient, they are not
		worth calling directly by the users.										*/


static inline void list__sort_insertion_sort(	const list_object *restrict list,
												const int direction,
												const leaf_node *elem_start,
												const leaf_node *elem_stop,
												const boolean fast_sort) {

	register leaf_node *current;
	register leaf_node *candidate;
	register leaf_node *next_ptr;
	register const leaf_node *start_ptr = elem_start->prior_leaf;;
	register const leaf_node *end_ptr = elem_stop->next_leaf;
	int (*element_compare)(const void *element_a, const void *element_b);
	
	/* these safety checks are here due to the quicksort having an issue with
		overlapping element pointers - Do not remove them */
	if (elem_start == elem_stop) return;		/* one element, sorted already */
	if (elem_start->element == NULL) return;	/* called by a mistake, is a head guard or overlap tail guard */
	if (elem_start == elem_stop->next_leaf) return;
	element_compare = list->functions.compare;

	for (current = elem_start->next_leaf, candidate = (leaf_node *)elem_start;
			current != end_ptr; current = next_ptr, candidate = next_ptr->prior_leaf) {
		next_ptr = current->next_leaf;
		while ((candidate != start_ptr) && ((*element_compare)(current->element, candidate->element) * direction < 0))
				candidate = candidate->prior_leaf;
		if (candidate != current->prior_leaf) {
			/* unlink the current leaf node */
			current->prior_leaf->next_leaf = current->next_leaf;
			current->next_leaf->prior_leaf = current->prior_leaf;
			/* Link the current leaf node back into the list, in its new location to the right of candidate */
			current->next_leaf = candidate->next_leaf;
			candidate->next_leaf->prior_leaf = current;
			candidate->next_leaf = current;
			current->prior_leaf = candidate;
		}
	}
	
	return;
}

/* this was written to compare against my insertion sort, just to verify my hypothesis
	that this would be slower for a linked list sort. It is, by about 20% on average.
	Even the fast sort version (just swapping element pointers) is actually 10% slower
	than the unmodified insertion sort running for slow sort. That is because the classic
	insertion sort suffers from having to shift elements across an array, which is very
	inefficient. But with a linked list, there is no excess shifting required.
	
	So i am not using selection, i am just leaving here for reference purposes.					*/
/*
static inline void list__sort_selection_sort(	const list_object *restrict list,
												const int direction,
												const leaf_node *elem_start,
												const leaf_node *elem_stop,
												const boolean fast_sort) {

	register leaf_node *cursor;
	register leaf_node *current;
	register leaf_node *candidate;
	register const leaf_node *end_ptr = elem_stop->next_leaf;
    void *swap_data;
	int (*element_compare)(const void *element_a, const void *element_b);
	
	if (elem_start == elem_stop) return;	
	if (elem_start->element == NULL) return;	
	if (elem_start == elem_stop->next_leaf) return;
	element_compare = list->functions.compare;
												
	for (current = (leaf_node *)elem_start; current != end_ptr; current = current->next_leaf) {
		for (cursor = current, candidate = current->next_leaf; candidate != end_ptr; candidate = candidate->next_leaf)
			if ((*element_compare)(cursor->element, candidate->element) * direction > 0) cursor = candidate;
        if (cursor != current) {
			if (! fast_sort) {
				cursor->prior_leaf->next_leaf = cursor->next_leaf;
				cursor->next_leaf->prior_leaf = cursor->prior_leaf;
				cursor->next_leaf = current;
				cursor->prior_leaf = current->prior_leaf;
				current->prior_leaf->next_leaf = cursor;
				current->prior_leaf = cursor;
				current = current->prior_leaf;
			} else {
				swap_data = current->element;
				current->element = cursor->element;
				cursor->element = swap_data;
			}
    	}
	}												
	return;												
}
*/

/*
	IMPORTANT NOTE:
	
	Unlike most implementations of quicksort, this one is NOT mp safe. Although at
	first glance it may appear to be MP safe, it is not. Do not try and use it that
	way. Due to the need to maintain integrity of both data and element references
	without using extra resources, the sort cannot be organized by arbitrary index
	numbers. Instead it must rely on leaf boundaries on each outside arms of the
	current sort range. Meaning in short that the problem cannot be safely tiled,
	due to the requirement of relinking the range to the outer boundary leaf nodes.
	
	This implementation is however very efficient for single threaded environments,
	given the design constraints, the comparisons operations are very fast, but the
	swap operations are slow.
	
	The insertion sorts of sub ranges also uses the outside leafs as boundaries, again
	you cannot pass those sub-sorts to separate threads.
	
	The other version of quicksort provided in D-List is MP safe, and it very fast,
	even single threaded, but requires the use of extra memory resources, that some
	embedded systems may not be able to spare.
*/
static inline void list__sort_quick_sort(	list_object *restrict list,
											const int direction,
											leaf_node *elem_start,
											leaf_node *elem_stop,
											const boolean fast_sort) {

	list__sort_sub_quick_sort(list, direction, elem_start, elem_stop, fast_sort);

	return;
}

static inline void list__sort_sub_quick_sort(	list_object *restrict list,
												const int direction,
												leaf_node *elem_start,
												leaf_node *elem_stop,
												const boolean fast_sort) {

	leaf_node *pivot, *start_ptr, *end_ptr;
	unsigned int left_hand = 0, right_hand = 0;
	
	if (elem_start != elem_stop) {
		start_ptr = elem_start;
		end_ptr = elem_stop;
		pivot = list__sort_partition(list, direction, &start_ptr, &end_ptr, fast_sort, &left_hand, &right_hand);
		if (pivot) {
			if (start_ptr != pivot) {
				if (left_hand < DLIST_QUICKSORT_MIN_SIZE)
					list__sort_insertion_sort(list, direction, start_ptr, pivot, fast_sort);
				else
					list__sort_sub_quick_sort(list, direction, start_ptr, pivot, fast_sort);
			}
			if ((pivot != end_ptr)|| (pivot->next_leaf != end_ptr)) {
				if (right_hand < DLIST_QUICKSORT_MIN_SIZE)
					list__sort_insertion_sort(list, direction, pivot->next_leaf, end_ptr, fast_sort);
				else
					list__sort_sub_quick_sort(list, direction, pivot->next_leaf, end_ptr, fast_sort);
			}
		}
	}
	return;
}

static inline leaf_node *list__sort_partition(	list_object *restrict list,
												const int direction,
												leaf_node **elem_start,
												leaf_node **elem_stop,
												const boolean fast_sort,
												unsigned int *left_hand,
												unsigned int *right_hand) {

	register leaf_node *pivot;
	register leaf_node *reference;
	register leaf_node *fwd_ptr;
	leaf_node *update_ptr = NULL;
	leaf_node *temp_ptr1;
	register leaf_node *temp_ptr2;
	register leaf_node *temp_ptr3;
	leaf_node *start_ptr;
	register leaf_node *end_ptr;
	leaf_node *exit_ptr;
	register unsigned int i = 0;
	register unsigned int j = 0;
	boolean two_elements = FALSE;
	boolean list_is_pivot = FALSE;
	int temp_pivot_value;
	int (*element_compare)(const void *element_a, const void *element_b);

	if (*elem_start == *elem_stop) return NULL;		/* one element, sorted already */

	/* Store our true boundary points, and setup the partitioning vector pointers */
	exit_ptr = fwd_ptr = reference = start_ptr = *elem_start;
	if (start_ptr->element == NULL) return NULL;	/* called by a mistake, is a head guard or overlap tail guard */
	start_ptr = start_ptr->prior_leaf;
	end_ptr = *elem_stop;
	pivot = end_ptr = end_ptr->next_leaf;
	if (*elem_start == end_ptr) return NULL;
	element_compare = list->functions.compare;
	if ((fast_sort) && (list->list_pivot_enabled)) list_is_pivot = TRUE;

	/* are there only 2 elements ? - if so, optimize and cut out needless re-entrant calls - check if they are equal or in order */
	if (reference->next_leaf == pivot->prior_leaf) {
		/* Equal ? or already in correct order ? - if so return and don't come back for these elements */
		if ((*element_compare)(reference->element, pivot->prior_leaf->element) * direction <= 0) return NULL;
		two_elements = TRUE;
	}
	
	do {
		do {
			++i;
			pivot = pivot->prior_leaf;
		} while ((*element_compare)(reference->element, pivot->element) * direction < 0);
		
		do {
			if (fwd_ptr == pivot) break;
			++j;
			fwd_ptr = fwd_ptr->next_leaf;
		} while ((*element_compare)(reference->element, fwd_ptr->element) * direction > 0);
		
		if (fwd_ptr == pivot) {
			if ((pivot == end_ptr->prior_leaf) && ((*element_compare)(reference->element, pivot->element) == 0)) {
				exit_ptr = pivot->prior_leaf;		/* the reference and the end elements are the same, pivot didn't move, force pivot back one element */
				update_ptr = NULL;					/* this situation wil get corrected on the next pass for this pivot set, with the pivot moved back one */
			} else {
				if (pivot != reference) {
					update_ptr = reference;
					exit_ptr = reference;
				} else {
						exit_ptr = reference;
						update_ptr = NULL;
				}
			}
		}
		else update_ptr = fwd_ptr;

		if (! fast_sort) {
			/* Not quite so fast, sort */
			if (update_ptr) {
/*	This was the original code for swapping the leaf nodes, it is left here
	as a comment to allow readers to understand the code. After squeezing
	out all the performance i could, it left the code much harder to read.
	The performance improved about 2.5% on average for a large sort, so it
	was worth it.
	
	The optimized code is logically identical to the original below.
	
	** swap the opposing leaf nodes - this must be done in safe mode, as they could be neighbors **
	** retain reference points to re-insert the nodes **
	temp_ptr1 = update_ptr->prior_leaf;		** to the node, left of fwd_ptr - the left hand marker **
	temp_ptr2 = pivot->next_leaf;			** to the node, right of pivot - the right hand marker **
	** unlink the current fwd_ptr node **
	update_ptr->prior_leaf->next_leaf = update_ptr->next_leaf;
	update_ptr->next_leaf->prior_leaf = update_ptr->prior_leaf;
	** unlink the current pivot node **
	pivot->prior_leaf->next_leaf = pivot->next_leaf;
	pivot->next_leaf->prior_leaf = pivot->prior_leaf;
	** Link the pivot leaf node back into the list, in its new location
		to the right of the left hand marker **
	pivot->next_leaf = update_ptr->prior_leaf->next_leaf;
	update_ptr->prior_leaf->next_leaf->prior_leaf = pivot;
	update_ptr->prior_leaf->next_leaf = pivot;
	pivot->prior_leaf = update_ptr->prior_leaf;
	** Link the fwd_ptr leaf node back into the list, in its new
		location to the left of the right hand marker **
	update_ptr->prior_leaf = pivot->next_leaf->prior_leaf;
	pivot->next_leaf->prior_leaf->next_leaf = update_ptr;
	pivot->next_leaf->prior_leaf = update_ptr;
	update_ptr->next_leaf = pivot->next_leaf;												*/
				/* swap the opposing leaf nodes - this must be done in safe mode, as they could be neighbors */
				/* retain reference points to re-insert the swapped nodes */
				temp_ptr1 = update_ptr->prior_leaf;		/* to the node, left of fwd_ptr - the left hand marker */
				temp_ptr2 = pivot->next_leaf;			/* to the node, right of pivot - the right hand marker */
				/* unlink the current fwd_ptr node */
				temp_ptr3 = update_ptr->next_leaf;
				temp_ptr1->next_leaf = temp_ptr3;
				temp_ptr3->prior_leaf = temp_ptr1;
				/* unlink the current pivot node */
				temp_ptr3 = pivot->prior_leaf;
				temp_ptr3->next_leaf = temp_ptr2;
				temp_ptr2->prior_leaf = temp_ptr3;
				/* Link the fwd_ptr (or reference) leaf node back into the list,
					in its new location to the left of the right hand marker */
				update_ptr->prior_leaf = temp_ptr3;
				temp_ptr2->prior_leaf = temp_ptr3->next_leaf = update_ptr;
				update_ptr->next_leaf = temp_ptr2;
				/* Link the pivot leaf node back into the list, in its new location
					to the right of the left hand marker */
				pivot->next_leaf = temp_ptr1->next_leaf;
				temp_ptr1->next_leaf = temp_ptr1->next_leaf->prior_leaf = pivot;
				pivot->prior_leaf = temp_ptr1;
				if (fwd_ptr != pivot) {
					/* restore the correct pivot and fwd pointers, as the leaf nodes have been swapped */
					temp_ptr1 = pivot;
					pivot = fwd_ptr;
					fwd_ptr = temp_ptr1;
				}
			}
		} else {
			/* Fast Sort only */
			if (update_ptr) {
				if (list_is_pivot) {
					/* splitting the 2 if checks, benchmarks faster, weird */
					if (pivot->internal->pivot.value != update_ptr->internal->pivot.value) {
						temp_pivot_value = pivot->internal->pivot.value;
						pivot->internal->pivot.value = update_ptr->internal->pivot.value;
						update_ptr->internal->pivot.value = temp_pivot_value;
					}
				}
				temp_ptr1 = pivot->element;
				pivot->element = update_ptr->element;
				update_ptr->element = temp_ptr1;
			}
			if ((fwd_ptr == pivot) && (update_ptr)) exit_ptr = fwd_ptr;
			/* If update_ptr == NULL then the exit_ptr is already set to the reference */
		}
	} while (fwd_ptr != pivot);	

	/* restore our corrected boundary points */
	*elem_start = start_ptr->next_leaf;
	*elem_stop = end_ptr->prior_leaf;
	/* give the caller some data points, to be able to decide what to do next */
	*left_hand = j;
	*right_hand = i;

	return ((two_elements) ? NULL : exit_ptr);
}

static inline void list__sort_quicker_sort(	list_object *restrict list,
											const int direction,
											leaf_node *elem_start,
											leaf_node *elem_stop,
											const boolean fast_sort) {

	register leaf_node *ptr;
	register leaf_node *next_ptr;
	leaf_node *left_ptr, *right_ptr;
	register const leaf_node *end_ptr = elem_stop->next_leaf;
	quicksort_entry__t *sort_table;
	unsigned int count;
	unsigned int i = 0;

	left_ptr = elem_start->prior_leaf;
	right_ptr = elem_stop->next_leaf;
	
	/* for speed we do not scan the elements first to count the actual number. */
	sort_table = (quicksort_entry__t  *)malloc(sizeof(quicksort_entry__t ) * list->element_count);
	if (! sort_table) {
#ifdef DLIST_DEBUG
		fprintf(stderr, "\n==> Dlist INTERNAL ERROR ==> %s() unable to allocate memory ."\
				"- malloc() returned a NULL pointer when trying allocate "
				"memory for quicksort array block, with error: %s.\n", __func__, strerror(errno));
#endif
		return;
	}

	/* Set up the sort table to process with quicksort */
	for (ptr = elem_start, count = 0; ptr != end_ptr; ptr = ptr->next_leaf, ++count) {
		sort_table[count].node = ptr;
		sort_table[count].element = ptr->element;
	}
	
	list__sort_sub_quicker_sort(list, sort_table, direction, 0, count-1, fast_sort);

	/* rebuild the list with the now sorted leaf nodes */
	ptr = left_ptr;
	while (i < count) {
		next_ptr = sort_table[i++].node;
		ptr->next_leaf = next_ptr;
		next_ptr->prior_leaf = ptr;
		ptr = next_ptr;		
	}
	ptr->next_leaf = right_ptr;
	right_ptr->prior_leaf = ptr;
	
	if (sort_table) free(sort_table);


	return;
}

static inline void list__sort_sub_quicker_sort(	list_object *restrict list,
												quicksort_entry__t *restrict sort_table,
												const int direction,
												const unsigned int start,
												const unsigned int stop,
												const boolean fast_sort) {

	unsigned int pivot;
	
	if (start < stop) {
		pivot = list__quicker_partition(list, sort_table, direction, start, stop, fast_sort);
		if (pivot < stop + 1) {
			if (start != pivot) {
					list__sort_sub_quicker_sort(list, sort_table, direction, start, pivot, fast_sort);
			}
			if (pivot+1 < stop) {
					list__sort_sub_quicker_sort(list, sort_table, direction, pivot+1, stop, fast_sort);
			}

/*	IF you wish to experiment with either insertion or selection sort, to finish the quicksort '
	when it reduces to a small number, you can uncomment this below and comment the section above
	
	In my extensive testing i found that letting this version of quicksort finish is faster
	than any other solution. However on other platforms things may benchmark differently.
	For x64 64 or 32 bit it runs 5-15% slower no matter what the setting of DLIST_QUICKER_SORT_MIN_SIZE
	up to about 20, and then it gets much slower.
	
			if (start != pivot) {
				if (pivot < DLIST_QUICKER_SORT_MIN_SIZE)
					list__sort_insertion_index_sort(list, sort_table, direction, start, pivot, fast_sort);
				else
					list__sort_sub_quicker_sort(list, sort_table, direction, start, pivot, fast_sort);
			}
			if (pivot+1 < stop) {
				if ((stop - pivot) < DLIST_QUICKER_SORT_MIN_SIZE)
					list__sort_insertion_index_sort(list, sort_table, direction, pivot+1, stop, fast_sort);
				else
					list__sort_sub_quicker_sort(list, sort_table, direction, pivot+1, stop, fast_sort);
			}																									*/
		}
	}
	return;
}

static inline unsigned int list__quicker_partition(	list_object *restrict list,
													quicksort_entry__t *restrict sort_table,
													const int direction,
													const unsigned int start,
													const unsigned int stop,
													const boolean fast_sort) {

	register unsigned int pivot = stop+1;
	register unsigned int reference = start;
	register unsigned int fwd_ptr = start;
	register unsigned int update_ptr;
	leaf_node *temp_ptr;
	void *temp_elem;
	int (*element_compare)(const void *element_a, const void *element_b);
	boolean two_elements = FALSE;


	if (start == stop) return stop+100;		/* one element, sorted already */

	element_compare = list->functions.compare;

	/* are there only 2 elements ? - if so, optimize and cut out needless re-entrant calls - check if they are equal or in order */
	if (reference+1 == stop) {
		/* Equal ? or already in correct order ? - if so return and don't come back for these elements */
		if ((*element_compare)(sort_table[reference].element, sort_table[stop].element) * direction <= 0) return stop+100;
		two_elements = TRUE;
	}
	
	do {
		do {
			--pivot;
		} while ((*element_compare)(sort_table[reference].element, sort_table[pivot].element) * direction < 0);
		
		do {
			++fwd_ptr;
		} while ((fwd_ptr < pivot) && ((*element_compare)(sort_table[reference].element, sort_table[fwd_ptr].element) * direction > 0));
		
		if (fwd_ptr >= pivot) {
			if ((pivot == stop) && ((*element_compare)(sort_table[reference].element, sort_table[pivot].element) == 0)) {
				--pivot;								/* the reference and the end elements are the same, pivot didn't move, force pivot back one element */
				update_ptr = stop+100;					/* this situation wil get corrected on the next pass for this pivot set, with the pivot moved back one */
			} else {
				if (pivot != reference) {
					update_ptr = reference;
				} else {
						update_ptr = stop+100;
				}
			}
		}
		else update_ptr = fwd_ptr;

		if (update_ptr < stop) {
			/* swap the table entries for pivot and update */
			temp_ptr = sort_table[pivot].node;
			temp_elem = sort_table[pivot].element;
			sort_table[pivot].node = sort_table[update_ptr].node;
			sort_table[pivot].element = sort_table[update_ptr].element;
			sort_table[update_ptr].node = temp_ptr;
			sort_table[update_ptr].element = temp_elem;
		}
	} while (fwd_ptr < pivot);	

	return ((two_elements) ? stop+100 : pivot);
}
/* this was written to benchmark the quicker sort algorithm and see if it could
	help with the tiny reduced sections. It does not. I decided to stick with the
	pure quicker sort instead, it benchmarked 10-15% faster even in the best cases.
	The problem lies with the insertion sort needing to shuffle the elements, far
	too inefficient in this scenario.													*/
/*
static inline void list__sort_insertion_index_sort(	const list_object *restrict list,
													quicksort_entry__t *restrict sort_table,
													const int direction,
													const unsigned int start,
													const unsigned int stop,
													const boolean fast_sort) {

	register unsigned int current = start +1;
	register unsigned int cursor;
	register leaf_node *temp_ptr;
	register void *temp_elem;
	leaf_node *candidate_node;
	void *candidate_elem;
	int (*element_compare)(const void *element_a, const void *element_b);

	if (start == stop) return;
	element_compare = list->functions.compare;

	do {
		candidate_node = sort_table[current].node;
		candidate_elem = sort_table[current].element;
		cursor = current;
		while ((cursor > start) && ((*element_compare)(sort_table[cursor-1].element, sort_table[cursor].element) * direction > 0)) {
			temp_ptr = sort_table[cursor].node;
			temp_elem = sort_table[cursor].element;
			sort_table[cursor].node = sort_table[cursor-1].node;
			sort_table[cursor].element = sort_table[cursor-1].element;
			sort_table[cursor-1].node = temp_ptr;
			sort_table[cursor-1].element = temp_elem;
			--cursor;
		}
		sort_table[cursor].node = candidate_node;
		sort_table[cursor].element = candidate_elem;
		++current;
	} while (current <= stop);

	return;
}
*/

/* this was written to compare against my index based insertion sort, just to verify
	my hypothesis that it would be chronically slower. It is, by about 50% on average.
	So i am not using it, just leaving here for reference purposes.								*/
/*
static inline void list__sort_selection_index_sort(	const list_object *restrict list,
													quicksort_entry__t *restrict sort_table,
													const int direction,
													const unsigned int start,
													const unsigned int stop,
													const boolean fast_sort) {

	leaf_node *temp_ptr;
	void *temp_elem;
	register unsigned int cursor;
	register unsigned int current = start;
	register unsigned int candidate;
	int (*element_compare)(const void *element_a, const void *element_b);

	if (start == stop) return;
	element_compare = list->functions.compare;
												
	for (current = start; current <= stop; ++current) {
		for (cursor = current, candidate = current+1; candidate <= stop; ++candidate)
			if ((*element_compare)(sort_table[cursor].element, sort_table[candidate].element) * direction > 0) cursor = candidate;
        if (cursor != current) {
			temp_ptr = sort_table[cursor].node;
			temp_elem = sort_table[cursor].element;
			sort_table[cursor].node = sort_table[current].node;
			sort_table[cursor].element = sort_table[current].element;
			sort_table[current].node = temp_ptr;
			sort_table[current].element = temp_elem;
    	}
	}												
	return;												
}
*/

/* find element in sorted range of elements. Recursively call to narrow down the
	scope of the search until it hits a minimum size, and then perform an element
	by element search of the narrowed down scope.									*/
static leaf_node *list__find_sorted_element(	const list_object *restrict list,
												register const void *datum,
												unsigned int *restrict index,
												const leaf_node *restrict start,
												register const leaf_node *restrict stop,
												const unsigned int current_index,
												const unsigned int scope) {

	register const leaf_node *back_ptr;
	register const leaf_node *cursor;
	register const leaf_node *end_ptr;
	register int evaluation;
	int direction;
	unsigned long int long_skip;
	unsigned int i = 0;
	unsigned int j = 0;
	unsigned int skip, back_index;
	int (*element_compare)(const void *element, const void *key);

	element_compare = list->functions.compare;
	if (list->sorted_order == LIST_ASCENDING) direction = 1;
	else direction = -1;
	
	if ((scope == list->element_count) && (list->element_count > 1)) {
		i = (list->element_count-1) / 2;
		if ((*element_compare)(list->mid_point->element, datum) * direction < 0) {
			/* element is somewhere forward of the mid-point */
			return list__find_sorted_element(list, datum, index, list->mid_point->next_leaf, list->tail_guard->prior_leaf, i+1, list->element_count - (i+1));
		} else {
			/* element is somewhere behind the mid-point */
			return list__find_sorted_element(list, datum, index, list->head_guard->next_leaf, list->mid_point->next_leaf, 0, i+1);
		}
	}

	cursor = back_ptr = start;
	end_ptr = stop->next_leaf;
	back_index = current_index;

	if (scope < MIN_SORTED_SEARCH_SCOPE) {
		/* perform a regular search to find the element */
		do {
			evaluation = (*element_compare)(cursor->element, datum) * direction;
			if (evaluation >= 0) break;
			++i;
			cursor = cursor->next_leaf;
		} while (cursor != end_ptr);
		if ((cursor == end_ptr) || (evaluation > 0)) return NULL;		/* No match found */

		/* return identity of found list element */
		*index = current_index + i;
		return (leaf_node *)cursor;
	} else {
		/* perform a scope narrowing operation */
		long_skip = scope * MIN_SORTED_SEARCH_SKIP;						/* working with int's, not doubles */
		skip = long_skip / 100;					
		if (skip < 2) skip = 2;
		if (skip > 5000) skip = 5000;									/* based on performance testing x86 64bit */
		++back_index;
		for (; (i <= scope) && (cursor != end_ptr); i += j) {
			evaluation = (*element_compare)(cursor->element, datum) * direction;
			if (evaluation >= 0) {
				if ((evaluation == 0) && (i == 0)) --back_index;
				return list__find_sorted_element(list, datum, index, back_ptr, cursor, back_index, j);
			}
			if (cursor == stop) break;
			for (back_ptr = cursor->next_leaf, back_index +=j, j = 0; (j < skip) && (cursor != stop); ++j, cursor = cursor->next_leaf);
		}
		return NULL;
	}

	return NULL;							/* we can never get here, but i don't like drop through functions */
}


/* find the place in sorted range of elements to insert a new element. Recursively
	call to narrow down the scope of the search until it hits a minimum size, and
	then perform an element by element search of the narrowed down scope, to locate
	the correct place to insert the new element.	
	
	This is almost the same as list__find_sorted_element (but there are some subtle
	differences between them, and i wanted to keep them separate in case i want to
	make any extensions to one or the other in the future.							*/
static leaf_node *list__find_sorted_insert(	const list_object *restrict list,
											register const void *datum,
											unsigned int *restrict index,
											const leaf_node *restrict start,
											register const leaf_node *restrict stop,
											const unsigned int current_index,
											const unsigned int scope) {

	register const leaf_node *back_ptr;
	register const leaf_node *cursor;
	register const leaf_node *end_ptr;
	int direction;
	unsigned long int long_skip;
	unsigned int i = 0;
	unsigned int j = 0;
	unsigned int skip, back_index;
	int (*element_compare)(const void *element, const void *key);

	element_compare = list->functions.compare;
	if (list->sorted_order == LIST_ASCENDING) direction = 1;
	else direction = -1;

	if ((scope == list->element_count) && (list->element_count > 1)) {
		/* first time in, split the list, and only search the correct half */
		i = (list->element_count-1) / 2;
		if ((*element_compare)(list->mid_point->element, datum) * direction < 0) {
			/* element is somewhere forward of the mid-point */
			return list__find_sorted_insert(list, datum, index, list->mid_point, list->tail_guard->prior_leaf, i, list->element_count - (i+1));
		} else {
			/* element is somewhere behind the mid-point */
			return list__find_sorted_insert(list, datum, index, list->head_guard->next_leaf, list->mid_point, 0, i+1);
		}
	}
	
	cursor = back_ptr = start;
	end_ptr = stop->next_leaf;
	back_index = current_index;

	if (scope < MIN_SORTED_SEARCH_SCOPE) {
		/* perform a regular search to find the element */
		do {
			if ((*element_compare)(cursor->element, datum) * direction > 0) break;
			++i;
			cursor = cursor->next_leaf;
		} while (cursor != end_ptr);
		/* return identity of found list location */
		*index = current_index + i;
		return (leaf_node *)cursor->prior_leaf;
	} else {
		/* perform a scope narrowing operation */
		long_skip = scope * MIN_SORTED_SEARCH_SKIP;						/* working with int's, not doubles */
		skip = long_skip / 100;					
		if (skip < 2) skip = 2;
		if (skip > 5000) skip = 5000;									/* based on performance testing x86 64bit */
		++back_index;
		for (; (i <= scope) && (cursor != end_ptr); i += j) {
			if ((*element_compare)(cursor->element, datum) * direction > 0) {
				if (i == 0) --back_index;
				return list__find_sorted_insert(list, datum, index, back_ptr, cursor, back_index, j);
			}
			if (cursor == stop) break;
			for (back_ptr = cursor->next_leaf, back_index +=j, j = 0; (j < skip) && (cursor != stop); ++j, cursor = cursor->next_leaf);
		}
		*index = current_index + i + 1;
		return (leaf_node *)cursor;
	}

	return NULL;							/* we can never get here, but i don't like drop through functions */
}

static inline int list__shift_leaf_node(	list_object *restrict list,
											unsigned int index,
											const boolean where) {

	leaf_node *ptr;

	ptr = list__locate_leaf_node(list, (long int)index);
	
	/* move the element, (we do not use the normal primitives as we want
		to retain all list data including the original leaf node and its reference) */
	list__adjust_middle_pointer(list, (long int)index, LEAF_REMOVED);
	unlink__leaf_node(ptr);
	if (where == HEAD_OF_LIST) {
		link__leaf_node(ptr, list->head_guard);
		list__adjust_middle_pointer(list, 0, LEAF_ADDED);
	} else {
		link__leaf_node(ptr, list->tail_guard->prior_leaf);
		list__adjust_middle_pointer(list, ((long int)list->element_count)-1, LEAF_ADDED);
	}

#ifdef DLIST_DEBUG
	assert(list__audit(list) == LIST_OK);
#endif		/* DLIST_DEBUG */

	return LIST_OK;
}

/* reverse the order of a range of leaf nodes from start_leaf, ending at
	the leaf node prior to stop_leaf (stop_leaf being the barrier).
	If requested, update the middle pointer in the master node.	
	
	This must not be called for a range of leaf nodes which is less than 2			*/
static inline void list__flip_leaf_nodes(	list_object *restrict list,
											leaf_node *start_leaf,
											leaf_node *stop_leaf,
											const boolean fix_middle) {

	leaf_node *ptr, *fixup_ptr, *new_next_ptr;
	
	ptr = start_leaf;
	
	if (fix_middle) list->mid_point = NULL;

	fixup_ptr = ptr->prior_leaf;
	new_next_ptr = stop_leaf;
	new_next_ptr->prior_leaf = ptr;
	ptr = ptr->next_leaf;
	while (ptr->prior_leaf != stop_leaf) {
		ptr = ptr->prior_leaf;
		ptr->prior_leaf = ptr->next_leaf;
		ptr->next_leaf = new_next_ptr;
		new_next_ptr = ptr;
	}
	ptr->prior_leaf = fixup_ptr;
	fixup_ptr->next_leaf = ptr;

	if (fix_middle) list__full_optimize_index(list, FALSE);

#ifdef DLIST_DEBUG
	assert(list__audit(list) == LIST_OK);
#endif		/* DLIST_DEBUG */

	return;
}

void list__full_optimize_index(	list_object *restrict list,
								const boolean sorted) {

	leaf_node *ptr;
	unsigned int middle;
	long int i;

#ifdef DEBUG
	if (list__validate_list_object(list, FALSE, __func__) == FALSE) return;
#else
	if (list == NULL) return;
#endif

	if (list->mid_point) return;				/* mid point is already set, no need to update */

	middle = (list->element_count-1) / 2;

	for (i = -1, ptr = list->head_guard; (ptr->next_leaf != NULL); ++i, ptr = ptr->next_leaf) {
		if (i == middle) break;
	}
	list->mid_point = ptr;

	if (! list->list_pivot_enabled) list->list_sorted = sorted;

#ifdef DLIST_DEBUG
	assert(list__audit(list) == LIST_OK);
#endif		/* DLIST_DEBUG */
	
	return;	
}

/* initialize the seed for random functions */
static inline void initialize_random_seed(void) {

	if (! rand_seeded) {
		srand((time(NULL) >> random_number_range(3,6)) ^ (clock() << 4));
		rand_seeded = TRUE;
	}
	return;
}


/* return a pseudo random positive number within a defined range
	the result will be between size_a - size_b inclusive.										*/
static inline unsigned int random_number_range(	const unsigned int size_a,
												const unsigned int size_b) {

#ifdef DLIST_DEBUG	
	assert(size_b > size_a);
#endif		/* DLIST_DEBUG */
	
	if (size_a == size_b) return size_a;

	return((rand() % ((size_b+1) - size_a)) + size_a);
}

/* Return a coin flip,
	 at least on my platform, this comes back at 50% probability +- 0.2% on average
	 i have tested for over 100 million flips in 100000 chunks, and the range never
	 changes more than +- 0.5% and is typically +-0.15% or less. This is well within
	 the acceptable range of natural chance with a real coin. However your milage may
	 vary on other platforms. It is actually not important, it just has to be a
	 reasonable way of making sort of random choices.								*/
static inline boolean coinFlip(void) {
	return ((rand() % 2) == 0);
}



/*	=====================================================================
		Audit the list, the leaf nodes and the leaf node block chains
	=====================================================================	*/
#ifdef DLIST_DEBUG

#ifndef DLIST_USE_MALLOC

/* verify how many leaf nodes in the free leaf node list, belong to this block
	number. If the leak node block has a zero in use count, then the number
	returned should be equal to the number allocated in that block. If the block
	is fully assigned (all in use) then the number returned should be 0.

	returns the number of free leaf nodes found that match the block number.		*/
static inline int list__audit_freep_list(	const list_object *restrict list,
											const int block_number) {

	leaf_node_array *block_ptr;
	leaf_node *ptr;
	int count = 0;

	/* find the correct block address */
	for (block_ptr = list->leaf_block;
			(block_ptr != NULL) && (block_ptr->block_number != block_number);
			block_ptr = block_ptr->next_block);
	if (block_ptr) {
		/* search the free list, looking for leaf nodes from that block */
		for (ptr = list->freep; (ptr != NULL); ptr = ptr->internal->next_freep) {
			if (ptr->internal->leaf_block == block_ptr) ++count;
		}
	} else {
		count = LIST_ERR;						/* did not find the block number */
	}
	
	return count;
}
#endif	/* ! DLIST_USE_MALLOC */

static int list__audit(const list_object *restrict list) {
	leaf_node *ptr;
	long int i;
	unsigned int middle = 0;
	unsigned int pivot = 0;
	unsigned int middle_found = 0;
	int pivot_found = -1;
	
	/* Audit the master node */
	if ((list == NULL) ||
			(list->head_guard == NULL) ||
			(list->tail_guard == NULL) ||
			(list->tail_guard == list->head_guard) ||
			(list->head_guard->prior_leaf != NULL) ||
			(list->tail_guard->next_leaf != NULL)) {
		fprintf(stderr, "\n==> Dlist INTERNAL ERROR ==> %s() failed - list master node corrupt.\n"
				, __func__);
		return LIST_ERR;
	}
		
	if (list->element_count > 0) {
		if ((list->head_guard->next_leaf == list->tail_guard) ||
				(list->tail_guard->prior_leaf == list->head_guard)) {
			fprintf(stderr, "\n==> Dlist INTERNAL ERROR ==> %s() failed - occupied list guard "\
					"nodes corrupt.\n", __func__);
			return LIST_ERR;
		}
		if (list->list_pivot_enabled) {
			if (list->element_count != (list->pivot_point.minor +
					(unsigned int)list->pivot_point.major)) {
				fprintf(stderr, "\n==> Dlist INTERNAL ERROR ==> %s() failed - occupied list pivot "\
						"element counts incorrect - total %u minor %u major %u.\n", __func__,
						list->element_count, list->pivot_point.minor, list->pivot_point.major);
				return LIST_ERR;
			}
		}
		if (list->list_finite_enabled) {
			if (list->element_count > list->max_element_count) {
				fprintf(stderr, "\n==> Dlist INTERNAL ERROR ==> %s() failed - occupied finite list "\
						"element count exceeds permitted maximum - maximum allowed %u actual %u.\n",
						__func__, list->max_element_count, list->element_count);
				return LIST_ERR;
			}
		}
	} else {
		if ((list->head_guard->next_leaf != list->tail_guard) ||
				(list->tail_guard->prior_leaf != list->head_guard)) {
			fprintf(stderr, "\n==> Dlist INTERNAL ERROR ==> %s() failed - empty list guard nodes "\
					"corrupt.\n", __func__);
			return LIST_ERR;
		}
	}

	if ((list->head_guard->element != NULL) ||
			(list->tail_guard->element != NULL) ||
			(list->head_guard->internal->master != list) ||
			(list->tail_guard->internal->master != list)) {
		fprintf(stderr, "\n==> Dlist INTERNAL ERROR ==> %s() failed - header guard nodes data "\
				"corrupt.\n", __func__);
		return LIST_ERR;
	}

#ifndef DLIST_USE_MALLOC
	leaf_node_array *block_ptr;
	leaf_node *freep_ptr;
	int j, k, l, m;
	int count_total = 0;
	int count_in_use = -2;
	int count_freep = 0;
	boolean last_free_node_found = FALSE;
	boolean last_block_found = FALSE;

	/* Audit the free leaf node block chain */
	if (list->leaf_block == NULL) {
		fprintf(stderr, "\n==> Dlist INTERNAL ERROR ==> %s() failed - list master node "\
				"corrupt, NULL leaf block pointer.\n", __func__);
		return LIST_ERR;
	}

	for (i = 1, block_ptr = list->leaf_block; (block_ptr != NULL); ++i, block_ptr = block_ptr->next_block) {
		if (block_ptr->master != list) {
			fprintf(stderr, "\n==> Dlist INTERNAL ERROR ==> %s() failed - leaf node block list "\
					"block in list position #%ld, with number (%d) has an "\
					"invalid master pointer. It is %p, but should be %p.\n", __func__, i, block_ptr->block_number, block_ptr->master, list);
		
			return LIST_ERR;
		}

		if (block_ptr->next_block) {		
			if (block_ptr->block_number <= block_ptr->next_block->block_number) {
				fprintf(stderr, "\n==> Dlist INTERNAL ERROR ==> %s() failed - leaf node block list "\
						"block in list position #%ld, with number (%d) should be larger than next in "\
						"chain (%d).\n", __func__, i, block_ptr->block_number, block_ptr->next_block->block_number);
				return LIST_ERR;
			}
		} else {
			if (block_ptr->block_number != 1) {
				fprintf(stderr, "\n==> Dlist INTERNAL ERROR ==> %s() failed - leaf node block list "\
						"block in list position #%ld, with number (%d) has NULL next pointer, and "\
						"should have number of #1.\n", __func__, i, block_ptr->block_number);
				return LIST_ERR;
			}
		}
		if (block_ptr->next_block == NULL) {
			if (last_block_found) {
				fprintf(stderr, "\n==> Dlist INTERNAL ERROR ==> %s() failed - leaf node block list "\
						"block in list position #%ld, with number (%d) has NULL next_block pointer, "\
						"but last block already found.\n", __func__, i, block_ptr->block_number);
				return LIST_ERR;
			} else {
				last_block_found = TRUE;
			}
		}
		j = block_ptr->nodes_allocated;			/* we just cannot know what finite blocks should be */
		if (block_ptr->nodes_allocated != j) {
			fprintf(stderr, "\n==> Dlist INTERNAL ERROR ==> %s() failed - leaf node block list "\
					"block in list position #%ld, with number (%d) allocated nodes should be %d, but is %d.\n",
					 __func__, i, block_ptr->block_number, j, block_ptr->nodes_allocated);
			return LIST_ERR;
		}
		if ((block_ptr->nodes_in_use < 0) || (block_ptr->nodes_in_use > j)) {
			fprintf(stderr, "\n==> Dlist INTERNAL ERROR ==> %s() failed - leaf node block list "\
					"block in list position #%ld, with number (%d) nodes in use is in error should be between 0 and %d, but is %d.\n",
					 __func__, i, block_ptr->block_number, j, block_ptr->nodes_in_use);
			return LIST_ERR;
		}
		count_total += block_ptr->nodes_allocated;
		count_in_use += block_ptr->nodes_in_use;
		/* now check the leaf nodes inside this block */
		for (j = k = l = 0; j < block_ptr->nodes_allocated; ++j) {
			if (block_ptr->leaf_nodes[j].internal->leaf_block != block_ptr) {
				fprintf(stderr, "\n==> Dlist INTERNAL ERROR ==> %s() failed - leaf node block list "\
						"block in list position #%ld, with number (%d) leaf node #%d has invalid leaf_block pointer, should be %p, "\
						"but is %p.\n", __func__, i, block_ptr->block_number, j, block_ptr, block_ptr->leaf_nodes[j].internal->leaf_block);
				return LIST_ERR;
			}
			if (block_ptr->leaf_nodes[j].internal->master == list) {
				++k;
				if (block_ptr->leaf_nodes[j].internal->next_freep != NULL) {
					fprintf(stderr, "\n==> Dlist INTERNAL ERROR ==> %s() failed - leaf node block list "\
							"block in list position #%ld, with number (%d) leaf node #%d is allocated, but has invalid freep pointer, should be NULL, "\
							"but is %p.\n", __func__, i, block_ptr->block_number, j, block_ptr->leaf_nodes[j].internal->master);
					return LIST_ERR;
				}
			} else {
				if (block_ptr->leaf_nodes[j].internal->master == NULL) {
					++l;
					if (block_ptr->leaf_nodes[j].internal->next_freep == NULL) {
						if (last_free_node_found) {
							fprintf(stderr, "\n==> Dlist INTERNAL ERROR ==> %s() failed - leaf node block list "\
									"block in list position #%ld, with number (%d) leaf node #%d is not allocated, but has invalid freep pointer, should not be NULL, "\
									"but is %p. End of free chain already found.\n", __func__, i, block_ptr->block_number, j, block_ptr->leaf_nodes[j].internal->next_freep);
							return LIST_ERR;
						} else last_free_node_found = TRUE;
					}
					/* make sure this free leaf node is actually in the free list */
					for (freep_ptr = list->freep;
							(freep_ptr != NULL) && (freep_ptr != &block_ptr->leaf_nodes[j]);
							freep_ptr = freep_ptr->internal->next_freep);
					if (freep_ptr == NULL) {
						fprintf(stderr, "\n==> Dlist INTERNAL ERROR ==> %s() failed - leaf node block list "\
								"block in list position #%ld, with number (%d) leaf node #%d is not allocated, but is not in the free leaf "\
								"node list pointer.\n", __func__, i, block_ptr->block_number, j);
						return LIST_ERR;
					}
				} else {
					fprintf(stderr, "\n==> Dlist INTERNAL ERROR ==> %s() failed - leaf node block list "\
							"block in list position #%ld, with number (%d) leaf node #%d has invalid master pointer, should be %p or NULL, "\
							"but is %p.\n", __func__, i, block_ptr->block_number, j, list, block_ptr->leaf_nodes[j].internal->master);
					if (block_ptr->leaf_nodes[j].internal->master == block_ptr) {
						fprintf(stderr, "\n==> Dlist INTERNAL ERROR ==> %s() failed - leaf node block list "\
								"the invalid master pointer, points to the leaf node block.\n", __func__);					
					}
					return LIST_ERR;
				}
			}
		}			/* end of for loop */
		
		if (block_ptr->nodes_in_use != k) {
			fprintf(stderr, "\n==> Dlist INTERNAL ERROR ==> %s() failed - leaf node block list "\
					"block in list position #%ld, with number (%d) nodes in use does not match leaf node block audit. Block record states %d, but is %d.\n",
					 __func__, i, block_ptr->block_number, block_ptr->nodes_in_use, k);
			return LIST_ERR;
		}
		if ((block_ptr->nodes_allocated - block_ptr->nodes_in_use) != l) {
			fprintf(stderr, "\n==> Dlist INTERNAL ERROR ==> %s() failed - leaf node block list "\
					"block in list position #%ld, with number (%d) free nodes does not match leaf node block audit. Block record states %d, but is %d.\n",
					 __func__, i, block_ptr->block_number, (block_ptr->nodes_allocated - block_ptr->nodes_in_use), l);
			return LIST_ERR;
		}
		
		m = list__audit_freep_list(list, block_ptr->block_number);
		if (m != l) {
			fprintf(stderr, "\n==> Dlist INTERNAL ERROR ==> %s() failed - leaf node block list "\
					"block in list position #%ld, with number (%d) free nodes does not match leaf node free list audit. Free nodes "\
					"found in Block audit %d, free nodes in this block found by free list audit %d. "\
					"Total Elements in list %d, for this block - nodes in use %d nodes allocated %d.\n",
					 __func__, i, block_ptr->block_number, l, m, list->element_count, block_ptr->nodes_in_use, block_ptr->nodes_allocated);
			return LIST_ERR;
		}
		count_freep += m;		
	}
	if (count_in_use != list->element_count) {
		fprintf(stderr, "\n==> Dlist INTERNAL ERROR ==> %s() failed - leaf node block list "\
				"total nodes in use (%d) does not match list element count (%d).\n",
				 __func__, count_in_use, list->element_count);
		return LIST_ERR;
	}
	if ((count_in_use + count_freep + HEAD_GUARD + TAIL_GUARD) != count_total) {
		fprintf(stderr, "\n==> Dlist INTERNAL ERROR ==> %s() failed - leaf node block list "\
				"total of nodes in use (%d) plus free nodes (%d) [= %d] does not match allocated leaf node count (%d).\n",
				 __func__, count_in_use, count_freep, (count_in_use + count_freep), count_total);
		return LIST_ERR;
	}
	for (i = 0, freep_ptr = list->freep; (freep_ptr != NULL); ++i, freep_ptr = freep_ptr->internal->next_freep);
	if (i != count_freep) {
		fprintf(stderr, "\n==> Dlist INTERNAL ERROR ==> %s() failed - leaf node block list "\
				"total of leaf block free nodes (%d) does not match the contents of the free leaf node list which has (%ld) elements.\n",
				 __func__, count_freep, i);
		return LIST_ERR;
	}
#endif	/* ! DLIST_USE_MALLOC */

	/* audit the list, look for link errors, structure problems and lost elements. */
	if (list->element_count > 0) {
		if (list->mid_point != NULL) middle = (list->element_count-1)/2;
		
		if (list->list_pivot_enabled) {
			if (list->pivot_point.minor == 0) pivot = 0;
			else pivot = list->pivot_point.minor-1;
		}
		for (i = -1, ptr = list->head_guard; (i < (list->element_count +1))
				&& (ptr->next_leaf != NULL); ++i, ptr = ptr->next_leaf) {
			if (ptr->next_leaf->prior_leaf != ptr) {
				fprintf(stderr, "\n==> Dlist INTERNAL ERROR ==> %s() failed - linked list "\
						"next->prior_leaf ptr not reflecting current leaf. index %ld current %p "\
						"next %p.\n", __func__, i, ptr, ptr->next_leaf);
				return LIST_ERR;
			}
			if (ptr->internal->master != list) {
				fprintf(stderr, "\n==> Dlist INTERNAL ERROR ==> %s() failed - linked list "\
						"leaf node pointing to different list master.\n", __func__);
				return LIST_ERR;
			}
			if ((ptr->element == NULL) && (i > -1)) {
				fprintf(stderr, "\n==> Dlist INTERNAL ERROR ==> %s() failed - linked list "\
						"leaf node lost element.\n", __func__);
				return LIST_ERR;
			}
			if (ptr == list->mid_point) middle_found = i;
			if ((middle) && (i == middle)) {
				if (ptr != list->mid_point) {
					fprintf(stderr, "\n==> Dlist INTERNAL ERROR ==> %s() failed - middle leaf "\
							"node incorrect.\n", __func__);
					if (! middle_found) {
						for (; (i < (list->element_count +1)) && (ptr->next_leaf != NULL);
								++i, ptr = ptr->next_leaf) {
							if (ptr == list->mid_point) middle_found = i;
							break;
						}
					}
					fprintf(stderr, "\n==> Dlist INTERNAL ERROR ==> %s() failed - The correct "\
							"middle is at leaf node #%u, instead was pointing at leaf node #%u.\n",
							__func__, middle, middle_found);
					return LIST_ERR;
				}
			}
			if (list->list_pivot_enabled) {
				if ((ptr == list->head_guard) || (ptr == list->tail_guard)) {
					if ((ptr == list->head_guard) && (ptr->internal->pivot.value != PIVOT_MINOR)) {
						fprintf(stderr, "\n==> Dlist INTERNAL ERROR ==> %s() failed - pivot list "\
								"head guard leaf node #%ld has invalid pivot data %d.\n", __func__,
								i, ptr->internal->pivot.value);
						return LIST_ERR;
					}
					if ((ptr == list->tail_guard) && (ptr->internal->pivot.value != PIVOT_MAJOR)) {
						fprintf(stderr, "\n==> Dlist INTERNAL ERROR ==> %s() failed - pivot list "\
								"tail guard leaf node #%ld has invalid pivot data %d.\n", __func__,
								i, ptr->internal->pivot.value);
						return LIST_ERR;
					}
				} else {

					if (ptr->internal->pivot.value == 0) {
						fprintf(stderr, "\n==> Dlist INTERNAL ERROR ==> %s() failed - pivot list "\
								"leaf node #%ld has invalid pivot data %d.\n", __func__, i, ptr->internal->pivot.value);
						return LIST_ERR;
					}
					if ((pivot_found >= 0) && (ptr->internal->pivot.value == PIVOT_MINOR)) {
						fprintf(stderr, "\n==> Dlist INTERNAL ERROR ==> %s() failed - pivot list "\
								"minor leaf node #%ld found after pivot point %d.\n",
								__func__, i, pivot_found);
						return LIST_ERR;
					}
					if ((list->pivot_point.minor > 0) && (pivot_found < 0)
							&& (ptr->internal->pivot.value == PIVOT_MAJOR)) {
						fprintf(stderr, "\n==> Dlist INTERNAL ERROR ==> %s() failed - pivot list "\
								"major leaf node #%ld found before calculated pivot point %d.\n",
								__func__, i, pivot);
						return LIST_ERR;
					}
					if (list->pivot_point.minor > 0){
						if (i <= pivot) {
							if (ptr->internal->pivot.value != PIVOT_MINOR) {
								fprintf(stderr, "\n==> Dlist INTERNAL ERROR ==> %s() failed - "\
										"pivot list minor leaf node #%ld has major node pivot "\
										"data %d.\n", __func__, i, ptr->internal->pivot.value);
								return LIST_ERR;
							}					
						}
					}
					
					if (list->pivot_point.major > 0){
						if (i > pivot) {
							if (ptr->internal->pivot.value != PIVOT_MAJOR) {
								fprintf(stderr, "\n==> Dlist INTERNAL ERROR ==> %s() failed - "\
										"pivot list major leaf node #%ld has minor node pivot "\
										"data %d.\n", __func__, i, ptr->internal->pivot.value);
								return LIST_ERR;
							}
						}
					}

					if (i == pivot) {
						if (ptr->next_leaf->internal->pivot.value == PIVOT_MAJOR) {
							pivot_found = i;
						} else {
							fprintf(stderr, "\n==> Dlist INTERNAL ERROR ==> %s() failed - "\
									"pivot list major leaf node NOT found after calculated "\
									"pivot point %d.\n", __func__, pivot);
							return LIST_ERR;
						}
					}

				}
				
			} else {
				/* NOT a pivot list */
				if (ptr->internal->pivot.value != 0) {
					fprintf(stderr, "\n==> Dlist INTERNAL ERROR ==> %s() failed - normal list "\
							"leaf node has invalid pivot data %d.\n", __func__, ptr->internal->pivot.value);
					return LIST_ERR;
				}
			}
		}
		if (ptr != list->tail_guard) {
			fprintf(stderr, "\n==> Dlist INTERNAL ERROR ==> %s() failed - invalid list, no tail "\
					"guard node found.\n", __func__);
			return LIST_ERR;
		}
		if (i != list->element_count) {
			fprintf(stderr, "\n==> Dlist INTERNAL ERROR ==> %s() failed - invalid leaf node count. "\
					"Found %ld, but there should be %d.\n", __func__, i,
					list->element_count);
			return LIST_ERR;
		}
	}

	return LIST_OK;
}

#endif		/* DLIST_DEBUG */

/* A set of pre-built basic functions to use directly, or as templates for your own.
		This was an excellent idea i borrowed from simclist, thanks ! */

/* Pre-built compare functions */
#define DLIST_COMPARE_FUNCTION(type)	int list_default_compare_##type(const void *a, const void *b) { return( *(type *)a > *(type *)b) - (*(type *)a < *(type *)b); }

DLIST_COMPARE_FUNCTION(int8_t)
DLIST_COMPARE_FUNCTION(int16_t)
DLIST_COMPARE_FUNCTION(int32_t)
DLIST_COMPARE_FUNCTION(int64_t)

DLIST_COMPARE_FUNCTION(uint8_t)
DLIST_COMPARE_FUNCTION(uint16_t)
DLIST_COMPARE_FUNCTION(uint32_t)
DLIST_COMPARE_FUNCTION(uint64_t)

DLIST_COMPARE_FUNCTION(short)
DLIST_COMPARE_FUNCTION(int)
DLIST_COMPARE_FUNCTION(long)
DLIST_COMPARE_FUNCTION(float)
DLIST_COMPARE_FUNCTION(double)

#ifdef time_t
DLIST_COMPARE_FUNCTION(time_t)
#endif

int list_default_compare_pointer(const void *a, const void *b) { return(a > b) - (a < b); }

int list_default_compare_string(const void *a, const void *b) { return strcmp((const char *)b, (const char *)a); }

/* Pre-built search functions */
#define DLIST_SEARCH_FUNCTION(type)		boolean list_default_search_##type(const void *a, const void *b) { return( *(type *)a == *(type *)b); }

DLIST_SEARCH_FUNCTION(int8_t)
DLIST_SEARCH_FUNCTION(int16_t)
DLIST_SEARCH_FUNCTION(int32_t)
DLIST_SEARCH_FUNCTION(int64_t)

DLIST_SEARCH_FUNCTION(uint8_t)
DLIST_SEARCH_FUNCTION(uint16_t)
DLIST_SEARCH_FUNCTION(uint32_t)
DLIST_SEARCH_FUNCTION(uint64_t)

DLIST_SEARCH_FUNCTION(short)
DLIST_SEARCH_FUNCTION(int)
DLIST_SEARCH_FUNCTION(long)
DLIST_SEARCH_FUNCTION(float)
DLIST_SEARCH_FUNCTION(double)

#ifdef time_t
DLIST_SEARCH_FUNCTION(time_t)
#endif

boolean list_default_search_pointer(const void *a, const void *b) { return(a == b); }

boolean list_default_search_string(const void *a, const void *b) { return(strcmp((const char *)b, (const char *)a) == 0); }

/* Pre-built size functions */
#define DCLIST_SIZE_FUNCTION(type)	size_t list_default_size_##type(const void *element) { if (element) { /* stop compilers whining */ } return sizeof(type); }

DCLIST_SIZE_FUNCTION(int8_t)
DCLIST_SIZE_FUNCTION(int16_t)
DCLIST_SIZE_FUNCTION(int32_t)
DCLIST_SIZE_FUNCTION(int64_t)

DCLIST_SIZE_FUNCTION(uint8_t)
DCLIST_SIZE_FUNCTION(uint16_t)
DCLIST_SIZE_FUNCTION(uint32_t)
DCLIST_SIZE_FUNCTION(uint64_t)

DCLIST_SIZE_FUNCTION(short)
DCLIST_SIZE_FUNCTION(int)
DCLIST_SIZE_FUNCTION(long)
DCLIST_SIZE_FUNCTION(float)
DCLIST_SIZE_FUNCTION(double)

#ifdef time_t
DLIST_SIZE_FUNCTION(time_t)
#endif

size_t list_default_size_pointer(const void *element) { return(sizeof((void *)element)); }

size_t list_default_size_string(const void *element) { return strlen((const char *)element) + 1; }

/* example pivot function, not that useful, but a template. */
int list_pivot_zero_int(const void *element) { return(( *(int *)element <= 0) ? PIVOT_MINOR : PIVOT_MAJOR); }

/* example (and pointless) iterate callback function will end the iteration sequence if it
	find negative numbers, a pivot reference or element exceeding 1000. */
int list_callback_element_filter(	const void *element,
									const element_reference *reference,
	const unsigned int index, const int pivot, const list_object *list) {
	const int *data_to_filter = (const void *)element;
	
	if (*data_to_filter < 0) return -10;
	if (pivot != ENTIRE_LIST) return 100;
	if (*data_to_filter >1000) return 1000;
	return 0;
	
}

