/** =================================================
 **                License Statement
 ** =================================================
 *
 ** Copyright (c) 2014-2022 David Winterburn <info@fwsentry.org>.
 *
 ** This file contains Original Code of the author, and is subject to the license
 ** terms below. In brief, using current terminology, this is a permissive license
 ** modelled on the 3-clause BSD License.
 *
 ** Permission to use, copy, modify, and distribute this software for any purpose
 ** is hereby granted as long as such purposes do not violate the terms of this
 ** license, and provided that the above copyright notice and this permission
 ** notice and license appear unmodified in ALL copies, distributed or stored.
 ** All intellectual property and patent rights remain with the author.
 *
 ** THE ORIGINAL CODE AND ALL SOFTWARE DISTRIBUTED UNDER THIS LICENSE IS PROVIDED
 ** ON AN "AS IS" BASIS WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESS OR IMPLIED,
 ** AND THE AUTHOR DISCLAIMS ALL SUCH WARRANTIES, OR LEGAL CLAIMS, WITH REGARD TO
 ** THIS SOFTWARE INCLUDING WITHOUT LIMITATION ANY WARRANTIES, IMPLIED OR OTHER,
 ** OF MERCHANTABILITY, PERFORMANCE AND FITNESS FOR A PARTICULAR PURPOSE, QUIET
 ** ENJOYMENT OR NON-INFRINGEMENT. UNDER NO CIRCUMSTANCES SHALL THE AUTHOR BE IN
 ** ANY WAY LIABLE FOR ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES, OR
 ** ANY DAMAGES WHATSOEVER, RESULTING FROM THE USE OF THIS SOFTWARE. SUCH DAMAGES
 ** MAY INCLUDE, BUT ARE NOT LIMITED TO, LOSS OF USE, EQUIPMENT, DATA, BILLABLE
 ** TIME, PERSONAL OR PHYSICAL INJURY, AND DIRECT OR INDIRECT PROFITS, WHETHER IN
 ** AN ACTION OF CONTRACT, NEGLIGENCE OR ANY OTHER TORTIOUS ACTION, ARISING OUT
 ** OF, OR IN CONNECTION WITH, THE USE OF, THE FITNESS OR PERFORMANCE OF THIS
 ** SOFTWARE, UNDER ANY CIRCUMSTANCES. THIS SOFTWARE WAS NOT DESIGNED TO BE USED
 ** IN ANY SITUATION WHERE HUMAN LIFE IS AT RISK, AND NO SUCH USE IS PROVIDED FOR
 ** OR PERMITTED.
 *
 ** BY USING THIS SOFTWARE YOU AGREE THAT IT IS THE SOLE RESPONSIBILITY OF THE END
 ** USER TO DECIDE IF THIS SOFTWARE OR PACKAGE PERFORMS, OR IS FIT FOR, ANY
 ** INTENDED USAGE.
 *
 ** This License and the rights granted hereunder will terminate; automatically
 ** without notice from the Author if You fail to comply with any term(s) of this
 ** License and fail to cure such breach within 30 days of becoming aware of such
 ** breach; or automatically without notice from the Author if You, at any time
 ** during the term of this License, commence any legal action against the Author.
 ** Upon termination of the license you are required to remove all copies of this
 ** software from all computer systems and data storage archives under your
 ** jurisdiction or control.
 *
 ** This license is automatically extended, if in the future, the Author exercises
 ** any intellectual property, or patent, rights connected with any of the Original
 ** Code included in this package. ALL end users using this package within the terms
 ** of this license, will automatically licensed to use those exercised or covered
 ** claims free of charge as they apply to this software package.
 **/



/*	D-List version 1.2 -- Copyright (c) 2014-2022 David Winterburn <info@fwsentry.org>	*/


#ifndef DLIST_H
#define DLIST_H


/*					Introduction
					============

	D-List implements a comprehensive set of linked list primitives that provides
	multiple forms of both sequential and random element access, bi-directional
	movement within a list, and an extensive set of list and element manipulation
	and reordering features. It also provides for a expansive set of user
	functions that can be added to lists and called automatically, to efficiently
	implement a broad range of services using the list primitives as underlying
	technology.

	D-List is a production quality, high performance list package written in C
	to handle huge lists very efficiently. I developed this package to support
	some of my other projects, and felt it was so useful and comprehensive, that
	others might find it helpful a well.

	So i decided to release it via sourceforge, and here.

	If you have any issues with this package, please create a support ticket at
	sourceforge, or email dlist-support@fwsentry.org

	This software is released with a 3 clause totally open BSD based license.

	Included in the package is a set of regression tests, and a set of examples
	on how to use D-list.																	*/



/*
	NOTE - This list package was only written and tested for UNIX and Linux type systems.
	There was no consideration or support of Windows in its design or testing, although
	i did provide support for the required includes. I know of no reason why dlist
	should not compile and run under Windows, but Windows support is up to the user.
	If a user has to make changes to support Windows, i will gladly incorporate such
	modifications into the next release, if they choose to provide them to me.				*/

/* If you wish to activate the internal dlist debugging code and asserts, please define
	
	DLIST_DEBUG
	
	in your environment at compile time. gcc -DDLIST_DEBUG is the simplest way.
	
	Do not leave this compiled in for deployment, it will slow down the code
	noticeably. This feature set is only useful for making internal changes to
	dlist, and serves no other useful purpose.											 	*/
	
/* If you wish to activate the general interface debugging tools, error messages etc
	please define DEBUG in your compilation environment. With DEBUG defined,
	interfaces will provide more complete checking of arguments, and provide suitable
	error messages to stderr when problems are discovered. With DEBUG not defined, the
	code is smaller and faster, but your calling code will fail ungracefully if the
	function calls are not correctly issued, or you have left NULL pointers hanging
	around etc...																			*/
	
	
	
#ifdef __cplusplus
extern "C" {
#endif

/* inttypes.h support is broken MS Visual Studio compilers */
#if !defined(_WIN32) || !defined(_MSC_VER)
#include <inttypes.h>
#else
#include <basetsd.h>
typedef		UINT8		uint8_t;
typedef		INT8		int8_t;
typedef		UINT16		uint16_t;
typedef		INT16		int16_t;
typedef		ULONG32		uint32_t;
typedef		LONG32		int32_t;
typedef		UINT64		uint64_t;
typedef		INT64		int64_t;
#endif

#include <errno.h>
#include <sys/types.h>

/* Support for old compilers */
#ifdef __STDC_VERSION__
#	if __STDC_VERSION__ >= 199901L
	/* "inline" and "restrict" are already builtin keywords */
#else
	/* Stop compiler errors */
#define inline
#define restrict
#ifndef __func__
#define		__func__	"unknown_function"
#endif
#endif
#endif		/* (__STDC_VERSION__) */


#if __STDC_VERSION__ >- 199901L
#include <stdbool.h>
#endif		/* __STDC_VERSION__ >- 199901L */

#ifdef __bool_true_false_are_defined
#define TRUE										true
#define FALSE										false
#define boolean										_Bool

#else

/* for environments with no stdbool.h */
typedef	int	bool
#define	false										(bool)0
#define	true										(bool)1

#ifndef TRUE
#define TRUE										(bool)1
#endif		/* !TRUE */
#ifndef FALSE
#define FALSE										(bool)0
#endif		/* !FALSE */

#endif		/* __bool_true_false_are_defined */


/* List Subsystem return codes */
#define LIST_OK							0
#define LIST_ERR						-1
#define ELEMENT_NOT_FOUND				-2
#define ELEMENT_INVALID					-3
#define ELEMENT_NO_DATUM				-4
#define LIST_UNSORTED					-5

/* For more complex Element Compare Functions */
#define ELEMENTS_EQUAL					0
#define ELEMENT_A_LESSTHAN_B			-1
#define ELEMENT_A_MORETHAN_B			1

/* Function call defines */
/* LIST_HEAD was my original define, but it seems that sys/queue.h defined it already,
	ouch so i am stuck with something much less elegant */
/* starting or ending points, iteration direction, placement points */
#define HEAD_OF_LIST					0
#define	TAIL_OF_LIST					1

/* direction, or order, of movement, or of a sort, or order of evaluation result */
#define LIST_ASCENDING					1
#define	LIST_DESCENDING					0

/* which element of a sequence is this one ? */
#define FIRST_ELEMENT					-1
#define OTHER_ELEMENT					0
#define LAST_ELEMENT					+1

/* what part of a pivot list to place, or look for, an element */
#define PIVOT_OF_LIST					1
#define END_OF_LIST						0

/* where to confine a search or define a scope for an operation
	or return a result from a pivot function */
#define PIVOT_MINOR						-1
#define ENTIRE_LIST						0
#define PIVOT_MAJOR						+1

/* list attributes */
#define LIST_COPY_DATA					1
#define LIST_DO_NOT_COPY				0

/* List size hints */
#define LIST_NO_SIZE_HINT				0
#define LIST_SIZE_TINY					1
#define LIST_SIZE_SMALL					2
#define LIST_SIZE_MEDIUM				3
#define LIST_SIZE_LARGE					4
#define LIST_SIZE_HUGE					5
#define LIST_SIZE_MEGA					6

/*					List Function groups.
					=====================

	List Object Management				Functions calls that update the list object itself. The list object
										is the user provided memory object that acts as the definitive point
										to identify and work on a list. Each list has its own list object.

	List Function Management			Function calls that set and retrieve pointers to special user
										provided that are used during specific list operations, such as a
										search function to process list_search() types calls. The list
										subsystem has no knowledge of the contents of the users elements,
										only how to organize and retrieve them.

	List Information Functions			Function calls to retrieve specific size information about a list.

	List Transformation Functions		Function calls to provide transformation operations on complete
										lists of elements, such as copying or splitting lists.

	Element Placement Functions			Function calls to permit the insertion or other placement of new
										elements in a list. Will render a sorted list unsorted.

	Element Relocation Functions		Function calls to relocate elements between different lists.
	
	Element Removal Functions			Function calls to remove elements from a list.

	Element Retrieval Functions			Function calls to locate and retrieve element data, from
										presumably unsorted lists.

	Element Search Functions			Function calls to perform searches for elements, from presumably
										unsorted lists.

	Sorted List Element Functions		Function calls to perform various operations including searches
										for elements, from previously sorted lists. The intent of this 
										group of functions is to be able to update and modify the lists 
										while maintaining the original sorted element order. It is also 
										assumed that the same compare function that was used to sort the 
										list is used with these find functions.

	Element Transformation Functions	Function calls to transform the relative order of elements in a
										list. Some functions will render a sorted list unsorted, others
										will sort or re-sort a list.

	Miscellaneous List Functions		Functions to perform technical operations on lists.						*/


/*					List functions, grouped by type.
					================================

	List Object Management
	----------------------
	list_create()						Initialize a list object as linear class list, prior to using a list.
	list_create_finite()				Initialize a list object as finite class list, prior to using a list.
	list_create_pivot()					Initialize a list object as pivot class list, prior to using a list.
	list_set_finite_limit()				Change the max size of a finite class list.		
	list_erase()						Erase all elements of a list and the list object itself.

	List Function Management
	------------------------
	list_set_compare_function()			Set the compare function for a list, used for element sorts and evaluations.
	list_get_compare_function()			Retrieve the compare function for a list.
	list_set_pivot_function()			Set the pivot function for a pivot class list, used by list_store().
	list_set_search_function()			Set the search function for a list, used for element searches.
	list_get_search_function()			Retrieve the search function for a list.
	list_set_size_function()			Set the size function for a list, used for lists that manage element storage.
	list_set_insert_function()			Set the insert function for a list, used upon placement to process an element.
	list_set_remove_function()			Set the remove function for a list, used upon removal to process an element.
	list_set_iterate_function()			Set the iteration function for a list, used by list_callback_iteration().

	List Information Functions
	--------------------------
	list_is_empty()						Check if a list is empty, or not.
	list_size()							Retrieve the number of elements in any list.
	list_size_pivot()					Retrieve the number of elements, by section, in a pivot class list.
	list_is_sorted()					Check if list is currently sorted, or not.

	List Transformation Functions
	-----------------------------
	list_concatenate()					Concatenate 2 lists together, serially, into the destination list.
	list_merge()						See function list_assay()
	list_clone()						Create a clone of a list.
	list_transfer()						Transfer the elements of a list, into another list.
	list_weave()						Merge 2 lists together, element by element, into a destination list
	list_assay()						Analyze & merge 2 lists locating unique elements and those elements in common.
	list_split_pivot()					Split a pivot class list, and create 2 destination lists from its contents.
	list_compare()						Perform a comparison of two lists, indicating if they are the same.
	list_differences()					Perform a comparison of two lists, returning the number of differences.	

	Element Placement Functions
	---------------------------
	list_append()						Append an element to the end of a list.
	list_prepend()						Prepend an element to the front of a list.
	list_insert_index()					Add an element to a list, in the position indicated by an index number.
	list_insert_ref()					Add an element to a list, in the location indicated by an element reference.
	list_store()						Add an element to a pivot class list, as indicated by the pivot function.
	list_push()							See function list_prepend()

	Element Relocation Functions
	----------------------------
	list_copy_index()					Copy an element to another list, from the position indicated by an index number.
	list_move_index()					Move an element to another list, from the position indicated by an index number.
	list_copy_ref()						Copy an element to another list, from the location indicated by an element reference.
	list_move_ref()						Move an element to another list, from the location indicated by an element reference.
	list_search_copy()					Search for, and then copy, all instances of an element to another list.
	list_search_move()					Search for, and then move, all instances of an element to another list.
	list_evaluate_copy()				Compare all elements in a list, and copy elements that meet criteria to another list.
	list_evaluate_move()				Compare all elements in a list, and move elements that meet criteria to another list.
	list_range_copy()					Compare all elements in a list, and copy all that are with a range, to another list.
	list_range_move()					Compare all elements in a list, and move all that are with a range, to another list.

	Element Removal Functions
	-------------------------
	list_clear()						Remove all elements from a list, leave it empty.
	list_fetch()						Extract an element from the front (head), or the tail (end), of a list.
	list_fetch_pivot()					Extract an element from the pivot point, of a pivot class list.
	list_extract_index()				Extract an element, from the position indicated by an index number.
	list_extract_ref()					Extract an element, from the location indicated by an element reference.
	list_delete_index()					Remove an element, from the position indicated by an index number.
	list_delete_group()					Remove a block of elements, between the indicated by index numbers.
	list_delete_ref()					Remove an element, from the location indicated by an element reference.
	list_delete_duplicates()			Remove all elements that are duplicates of other elements in the list.
	list_search_extract()				Search for, and then extract, an element in a list.
	list_search_delete()				Search for, and then delete, an element in a list.
	list_search_expunge()				Search for, and then delete, all instances of an element from a list.
	list_pop()							See function list_fetch()

	Element Retrieval Functions
	---------------------------
	list_get_index()					Retrieve an element, from the position indicated by an index number.
	list_get_index_ref()				Retrieve an element and its reference, using position indicated by an index number.
	list_get_ref()						Retrieve an element, from the location indicated by an element reference.
	list_get_next_ref()					Retrieve the next element in a list, as indicated by an element reference.
	list_locate_index()					Locate an element index in a list, identified by its element datum pointer.
	list_locate_pivot()					Locate an element pivot value in a list, identified by its element datum pointer
	list_locate_ref()					Locate an element reference in a list, identified by its element datum pointer.
	list_valid_ref()					Verify if an element reference is valid for the list.
	list_iteration_start()				Start a sequence of iteration of all the elements, from one end of a list.
	list_iteration_start_pivot()		Start a sequence of iteration of all the elements, from the pivot point of a list.	
	list_iteration_has_more()			Inquire if there are more elements, during an iteration sequence of a list.
	list_iteration_has_more_pivot()		Inquire if there are more elements on this side of the pivot point.	
	list_iteration_get_next()			Retrieve the next element in an iteration sequence of a list.
	list_iteration_stop()				Stop an iteration sequence of a list.

	Element Search Functions
	------------------------
	list_simple_search()				Search for an element in a list, using element datum as the search criteria.
	list_occurrences()					Search for all occurrences of an element, using datum as the search criteria.
	list_search()						Search for an element in a list, using element datum as the search criteria.
	list_search_pivot()					Search a pivot class list, using element datum as both pivot value and search criteria.
	list_search_ref()					Search for an element, backwards or forwards, from a given reference point.
	list_search_shift()					Search for an element, and if found, shift the element to the head of the list				
	list_locate_maximum()				Search for an element in a list, using maximum quantity as the search criteria.
	list_locate_minimum()				Search for an element in a list, using minimum quantity as the search criteria.

	Sorted List Element Functions
	-----------------------------
	list_sorted_find()					Search for an element in a sorted list, using datum as the search criteria.
	list_sorted_insert()				Add an element to a sorted list, in the position indicated by sort order.
	list_sorted_extract()				Extract an element from a sorted list, by the position indicated by sort order.
	list_sorted_delete()				Delete an element from a sorted list, by the position indicated by sort order.
	list_sorted_expunge()				Search for, and then delete, all instances of an element from a sorted list.

	Element Transformation Functions
	--------------------------------
	list_sort()							Sort all the elements in a list, using either ascending or descending order.
	list_safe_sort()					Sort all the elements in a list, by order, in a guaranteed safe order.
	list_fast_sort()					Sort all the elements in a list, by order, voiding element references.
	list_tiny_sort()					Sort all the elements in a list, by order, slower but with no extra resources.
	list_shift_index()					Shift an element, identified by index number, to the end of a list.
	list_shift_ref()					Shift an element, identified by an element reference, to the end of a list.
	list_rotate()						Rotate the elements in a list, either to the left or right.
	list_flip()							Reverse all the elements in a list.
	list_shuffle()						Shuffle the elements in a list, by randomly, and repeatedly relocating them.
	list_process()						Process all elements in list using a user function to operate on element data.
	list_process_range()				Process a range of elements in list using a user function to operate on element data.
	list_callback_iteration()			Start a sequence of callback iteration, to process the elements in a list.

	Miscellaneous List Functions
	----------------------------
	list_optimize_index()				Perform an optimization of the element index in a list.
	list_optimize_memory()				Perform an optimization of the element memory storage in a list.
	list_do_not_seed_random()			Prevent D-List from seeding the underlying random number functions.
	list_information()					Provide details about a list to stdout.
	list_error_string()					Return pointer to string corresponding to D-List error code.			*/
	
	



/* NOTES on usage --
	
	D-List is suitable for multi-threaded environments. However the user must use
	appropriate measures to prevent race conditions or corrupting lists during
	updates. Controls such as mutex locks, or barriers, etc must be employed. The
	author uses, and suggests, pthreads for this purpose.

	Getting Started
	---------------
	The D-List package includes a set of example programs demonstrating the use of
	many of the list function calls. Also included in the D-List package is an
	extensive set of regression tests and interface error tests. All 3 of these
	sources can provide you with examples of suggesting multiple ways of using
	D-List to build, manipulate and extract elements with the available list
	classes. If you want to add element modeling, statistics or transformation
	functions to D-List, you might want to pay particular attention to the
	list_process() functions and the user function process_element. There were
	developed specifically to provide a simple way to extend D-List without
	having to put data type dependent code into D-List itself.

	If you decide to make changes to the internal D-List subsystem itself, make
	sure it passes both sets of regression tests, prior to using it for any
	projects, or distributing it to others.
	
	Sorted versus unsorted lists
	----------------------------
	All list functions except the Sorted List Element Functions, which are a group
	functions specifically designed for sorted lists, assume the lists to be
	operated on are not ordered or sorted. However all of the list_sorted_xx()
	function calls assume the lists are sorted, and are intended to stay sorted.

	Depending on the type of operations you wish to perform on a list, keeping a
	list sorted may dramatically speed up performance. If a lists element set is
	highly unstable, with very frequent insertions and removals, keeping a list
	sorted will provide very little performance improvement, and may actually be
	slightly slower. However for generally stable lists, that were built and then
	often searched, with somme insertions and deletions, keeping the list sorted
	and using the Sorted List Element Functions can provide huge performance
	gains. For example using 2 of the performance testing modules, search and
	sorted_search, you will see on average the unsorted very large lists take
	about 3 times as long to search for non-existent elements and much longer
	by comparison when the element exists. So for projects that a search heavy,
	using sorted lists is a good performance booster. In fact for a list of a
	million elements, the processor savings for even a couple of searches of the
	list is far higher than the time taken to sort the list.

	It is important that the current user compare function set for the lists being
	operated on by the various Sorted List Element Functions, keep the same
	ordering semantics as the user compare function used to originally sort the
	list. Generally you will want to use the same user compare function throughout,
	however the compare function can be replaced but the ordering semantics should
	remain the same, or the element ordering results will be undefined when calling
	any of the Sorted List Element Functions. Additionally, if any of the Sorted
	List Element Functions are called for unsorted lists, it will result in a error
	return code.
	
	All functions are available for unsorted lists, with the restrictions noted in
	the function descriptions. For sorted lists all functions are also available,
	however usage of any of the following functions will render the list unsorted,
	which will prevent any of the Sorted List Element Functions from working on
	them, until the list is sorted again.

		All Element Placement Functions -
			list_append()
			list_prepend()
			list_insert_index()
			list_insert_ref()
		And these Element Transformation Functions
			list_search_shift()
			list_shift_index()
			list_shift_ref()
			list_rotate()
			list_shuffle()
	
	All other functions will operate on sorted lists and maintain the current
	sorted order, this includes pivot class lists.
	
	Any list processed by one of the 4 Element Sort Functions will be marked as
	sorted, and that status is only lost if operated on by a function that would
	destroy the sorted order. Even empty lists may be sorted and available for 
	all of the Sorted List Element Functions.
 
	
	Iteration Sequences
	-------------------
	Starting a list iteration will lock the list from index based updates, but
	will allow parallel searches and scans to be performed. Only 1 iteration at
	a time per list is allowed however. The user can access multiple elements
	using either reference, index or searches, while in an iteration cycle. When
	an iteration is stopped the list returns to normal.
	
	Element References
	------------------
	Accessing, inserting, relocating, or deleting elements using the available
	element references is similar to file operations using SEEK. You can keep and
	reuse the elements references, but you cannot make any assumptions about how
	they relate to each other in any way. Never attempt to modify references, and
	never use them as a reference after an element has been removed from a list.
	Any attempts will return an error in debug mode, or a code failure if running
	in non-debug mode. If a function does not allow NULL elements references,
	then always check an element reference is not NULL before providing it to a
	function. You should never assume that element references are pointers, and
	never use the structure information in dlist_internals.h to use or misuse
	them. All element references are implementation dependent and can change
	between releases. It is only guaranteed that if an element exists, then the
	previously provided element reference can be supplied back to any suitable
	D-List function to access that same element again. Element references survive
	sorting, shuffles, reordering, and being transfered to another list. Once an
	element has been removed from a list, its element reference is rendered void.
	
	List iteration sequences are based on element references, which under most
	circumstances, makes iteration the most efficient way to move through a list.
	
	Indexes
	-------
	Unlike accessing elements by reference, using index access can guarantee that
	an element with a lower index number is prior in the list than one with a
	higher number. Element 4 is 2 back from element 6, etc. It is important to
	remember that an element index always starts with element #0. So that the last
	element at the tail of a list, is always found with list_size(the_list)-1.
	Also it is important to note that indexes do not survive transformations of
	lists or elements, and that they are only a relative index of the elements at
	that moment in time.

	Accessing by index number is slower than accessing by reference. In small
	lists (in the 100's) it is not a major issue, the mean search required to find
	an element by index is n/8, and the worst case is n/6 plus the calculation
	overhead of obtaining a starting point. However for large lists, >100,000 or
	very large lists >1,000,000 this becomes a time consuming overhead. Accessing
	elements by reference, or more specifically the next or prior elements by
	reference to the current one, is vastly superior in time and resources. The
	worst case time being n/list_size. So at 50,000 elements the mean search for
	index would require moving over 6250 leaf nodes to acquire the requested node.
	With element reference access it is only required to move across 1 leaf node
	in all cases.

	Process Functions
	-----------------
	As an alternative to iteration, D-List provides a facility to process either part,
	or all, of list using a user supplied function to act on each element, and
	the entire group of the elements. This feature was added to allow users to easily
	add statistics and data transformation features to the list subsystem, without
	adding data type specific knowledge directly to D-List. For example, to add a
	group of functions to a user program to calculate the mean, or mode of a data set,
	can easily be achieved with a few lines of code using this feature. Unlike using
	callback iteration, the function does not have access to the list, and therefore
	cannot remove or relocate elements. Also the function has no option to cut the
	element processing short. It does however have the ability to share data directly
	with the original caller in a MP safe way.

	
	D-List implements 3 classes of lists.
	-------------------------------------
	
	Standard linear lists -
	unconstrained linked lists running from head to tail
	
	Finite lists -
	linear lists with a maximum capacity of elements and automatic removal of
	excess elements from the tail
	
	Pivot lists -
	semi-organized lists with the elements split using a user defined datum
	separating the elements by a conceptual pivot point. Please refer to the
	documentation for functions list_create_pivot() and list_store() for more
	information on the usage and layout of pivot lists.
	
	All three classes of lists are implemented using a double-linked layout,
	however a pivot list uses a user defined pivot function to decide which side
	of a conceptual pivot point, to either place or look for, an element. Some
	functions are not available to operate on pivot lists, and some not for
	linear or finite lists. Many functions operate slightly differently depending
	on which class of list they are operating on. See the function documentation
	for more details. Pivot lists are mainly useful for code that builds and uses
	huge lists, or wants an easy way to organize and reorganize various list
	elements. The output of list_assay() gives a good example of the uses of
	pivot class lists.

*/


/*				Table of D-List functions by availability for class type

																		if LIST_COPY_DATA
									  Linear	  Finite	   Pivot	frees object used
									  											
	list_create()						yes	
	list_create_finite()							yes	
	list_create_pivot()											yes	
	list_set_finite_limit()							yes				
	list_erase()						yes			yes			yes				yes

	list_set_compare_function()			yes			yes			yes	
	list_get_compare_function()			yes			yes			yes	
	list_set_pivot_function()									yes	
	list_set_search_function()			yes			yes			yes	
	list_get_search_function()			yes			yes			yes	
	list_set_size_function()			yes			yes			yes	
	list_set_insert_function()			yes			yes			yes	
	list_set_remove_function()			yes			yes			yes	
	list_set_iterate_function()			yes			yes			yes	

	list_is_empty()						yes			yes			yes	
	list_size()							yes			yes			yes	
	list_size_pivot()					yes			yes			yes	
	list_is_sorted()					yes			yes			

	list_concatenate()					yes			yes(4)		yes(1)			copy	
	list_clone()						yes			yes(4)		yes				copy
	list_transfer()						yes			yes			yes				copy
	list_weave()						yes			yes(4)		yes(1)			copy	
	list_assay()						yes			yes			yes(1)			copy
	list_split_pivot()											yes				copy
	list_compare()						yes			yes			yes	
	list_differences()					yes			yes			yes	

	list_append()						yes							
	list_prepend()						yes			yes				
	list_insert_index()					yes			yes				
	list_insert_ref()					yes							
	list_store()												yes	

	list_copy_index()					yes			yes(4)		yes				copy
	list_move_index()					yes			yes(4)		yes				yes
	list_copy_ref()						yes			yes(4)		yes				copy
	list_move_ref()						yes			yes(4)		yes				yes
	list_search_copy()					yes			yes(4)		yes				copy
	list_search_move()					yes			yes(4)		yes				yes
	list_evaluate_copy()				yes			yes(4)		yes				copy
	list_evaluate_move()				yes			yes(4)		yes				yes
	list_range_copy()					yes			yes(4)		yes				copy
	list_range_move()					yes			yes(4)		yes				yes

	list_clear()						yes			yes			yes				yes
	list_fetch()						yes			yes			yes				no
	list_fetch_pivot()											yes				no
	list_extract_index()				yes			yes			yes				no
	list_extract_ref()					yes			yes			yes				no
	list_delete_index()					yes			yes			yes				yes
	list_delete_group()					yes			yes			yes				yes
	list_delete_ref()					yes			yes(1)		yes				yes
	list_delete_duplicates()			yes			yes			yes				yes
	list_search_extract()				yes			yes			yes				no
	list_search_delete()				yes			yes			yes				yes
	list_search_expunge()				yes			yes			yes				yes

	list_get_index()					yes			yes			yes	
	list_get_index_ref()				yes			yes			yes	
	list_get_ref()						yes						yes	
	list_get_next_ref()					yes						yes	
	list_locate_index()					yes			yes			yes	
	list_locate_pivot()					yes(2)		yes(2)		yes	
	list_locate_ref()					yes			yes			yes	
	list_valid_ref()					yes			yes			yes	
	list_iteration_start()				yes			yes			yes	
	list_iteration_start_pivot()								yes	
	list_iteration_has_more()			yes			yes			yes	
	list_iteration_has_more_pivot()								yes	
	list_iteration_get_next()			yes			yes			yes	
	list_iteration_stop()				yes			yes			yes	

	list_simple_search()				yes			yes			yes	
	list_occurrences()					yes			yes			yes	
	list_search()						yes			yes			yes	
	list_search_pivot()											yes	
	list_search_ref()					yes			yes			yes	
	list_search_shift()					yes			yes				
	list_locate_maximum()				yes			yes			yes	
	list_locate_minimum()				yes			yes			yes	

	list_sorted_find()					yes			yes			yes	
	list_sorted_insert()				yes			yes			yes				yes
	list_sorted_extract()				yes			yes			yes				yes
	list_sorted_delete()				yes			yes			yes				yes
	list_sorted_expunge()				yes			yes			yes				yes

	list_sort()							yes			yes			yes	
	list_safe_sort()					yes			yes			yes	
	list_fast_sort()					yes			yes			yes	
	list_tiny_sort()					yes			yes			yes	
	list_shift_index()					yes			yes				
	list_shift_ref()					yes			yes				
	list_rotate()						yes			yes
	list_flip()							yes			yes			yes(1)	
	list_shuffle()						yes			yes			yes	
	list_process()						yes			yes			yes	
	list_process_range()				yes			yes			yes	
	list_callback_iteration()			yes			yes			yes	

	list_optimize_index()				yes			yes			yes	
	list_optimize_memory()				yes			yes			yes	
	list_do_not_seed_random()			n/a			n/a			n/a
	list_information()					yes			yes			yes	
	list_error_string()					n/a			n/a			n/a
	
	Notes:
	(1) restrictions apply, refer to function documentation
	(2) will always return ENTIRE_LIST
	(3) may not retain pivot alignment, depending of search criteria,
		refer to function documentation
	(4)	destination list cannot be finite class										*/

/*	A Note about User Functions.
	============================
	
	D-List provides for many user provided functions to either support or extend
	the list subsystem. This is a brief overview of them, and how they can be used.
	
	D-List provides a large set of pre-built functions for basic C data types,
	that are documented in this header file. The pre-built basic functions are
	named with the data type at the end of the function name. The generally
	provided data types are
	
		int8_t, int16_t, int32_t, int64_t,
		uint8_t, uint16_t, uint32_t, uint64_t,
		short, int, long, float, double,
		time_t, pointer (void*), string (char*)
		
	so for example the pre-built compare function for int's is
	
		list_default_compare_int()
	
	All the others are named with the same template form. Please refer to the
	table in the section below, which outlines which user functions are used by
	which D-List operations.
	
	The pre-build functions provided can serve as a basis for users developing
	new one tailored to their specific needs.

	compare
	-------
	There is a full set of pre-built compare functions provided in the D-List
	distribution.
	
	This function facility is provided to allow D-List to perform value based
	comparisons between elements, without the underlying list subsystem
	needing to know or understand data types or user defined data structures.
	A compare function is provided with 2 element datums and is expected to
	make an evaluation to determine if they are equal or if one is "greater"
	than the other. For numbers this is obvious, for user data this can be
	more complicated. For example, if the datums being compared are CIDR
	form IP addresses, the compare function will need to consider a greater
	range of semantics than just a plain number
		
	Compare functions are used for sorting lists, and also for functions that
	perform evaluations of list elements, such as the various Sorted List
	Element Functions, and the list_evaluate_copy() and list_evaluate_move()
	functions.

	search
	------
	There is a full set of pre-built search functions provided in the D-List
	distribution.
	
	This function facility is provided to allow D-List to perform equality
	based comparisons between elements, without the underlying list subsystem
	needing to know or understand data types or user defined data structures.
	A search function is provided with 2 element datums and is expected to
	make an evaluation to determine if the are equal or not.
	
	Search functions are used for searching for elements and for comparing
	lists. For a basic search operation the search function will want to
	make a specific and accurate equality comparison. However for some D-List
	operations such a list_search_copy() a partial matching search function
	will produce a broader set of results.
	
	size
	----
	There is a full set of pre-built search functions provided in the D-List
	distribution.
	
	This function facility is provided to allow D-List to accurately allocate
	memory for new list elements when the user has created a list with the
	LIST_COPY_DATA attribute set. If this attribute was set, then a size
	function is mandatory for all D-List operations that insert new elements
	into that list.
	
	Size functions must return the size of a given element in bytes.
	
	pivot
	-----
	Pivot functions are provided to allow elements to be stored in a pivot
	class list. The function is called to determine which side of the
	conceptual pivot point of the list, to store the new element.

	As pivot functions are totally user data and semantics dependent,
	there are no pre-built ones in the D-List distribution. There is an
	example one that arranges numbers if they are prime or not, in the
	D-List source code, and some variants of it are used in the example
	code and regression testing code provided.
		
	insert
	------
	Insert functions are used to allow the user to process new elements
	being added to a list. This can have a variety of uses, such as
	resource allocation, garbage collection housekeeping, automatic element
	backup, etc. Insert functions cannot by themselves prevent an element
	from being added.
	
	remove
	------
	Remove functions are used to allow the user to process elements that are
	about to be removed from a list. This can have a variety of uses, such
	as resource deallocation, garbage collection housekeeping, automatic
	element backup, etc. Remove functions cannot by themselves prevent an
	element from being removed.

	iterate
	-------
	Iterate functions are used by list_callback_iteration() to process a
	group of elements in a list. This is an alternative way of iterating a
	list than that provided by list_iteration_start().
	
	process
	-------
	Process functions are used by list_process() and list_process_range()
	to allow the user to extend the base functions of D-List. It is
	especially useful if you want to add extensive statistical or
	transformational functions to D-List.
	
	
	
	The use of user set function is covered in the detailed descriptions,
	which are provided with each of the functions.									*/


/*				Table of D-List calls that use sub functions

									  compare	search	size	pivot	insert	remove	iterate	process
									  											
	list_create()							
	list_create_finite()								
	list_create_pivot()												
	list_set_finite_limit()											
	list_erase()																yes

	list_set_compare_function()			yes							
	list_get_compare_function()			yes							
	list_set_pivot_function()									yes	
	list_set_search_function()					yes				
	list_get_search_function()					yes				
	list_set_size_function()							yes				
	list_set_insert_function()											yes	
	list_set_remove_function()													yes	
	list_set_iterate_function()															yes	

	list_is_empty()													
	list_size()														
	list_size_pivot()												
	list_is_sorted()						

	list_concatenate()									yes(1)	yes(2)	yes(1)		
	list_clone()										yes(1)	yes(2)	yes(1)		
	list_transfer()						
	list_weave()										yes(1)	yes(2)	yes(1)			
	list_assay()										yes(1)			yes(1)		
	list_split_pivot()									yes(1)	yes(2)	yes(1)		
	list_compare()								yes				
	list_differences()							yes							

	list_append()										yes				yes				
	list_prepend()										yes				yes				
	list_insert_index()									yes				yes				
	list_insert_ref()									yes				yes			
	list_store()										yes		yes		yes

	list_copy_index()									yes(1)	yes(2)	yes				
	list_move_index()									yes(1)	yes(2)	yes		yes		
	list_copy_ref()										yes(1)	yes(2)	yes		
	list_move_ref()										yes(1)	yes(2)	yes		yes
	list_search_copy()							yes		yes(1)	yes(2)	yes		
	list_search_move()							yes		yes(1)	yes(2)	yes		yes
	list_evaluate_copy()				yes				yes(1)	yes(2)	yes		
	list_evaluate_move()				yes				yes(1)	yes(2)	yes		yes
	list_range_copy()									yes(1)	yes(2)	yes		
	list_range_move()									yes(1)	yes(2)	yes		yes

	list_clear()																yes
	list_fetch()																yes
	list_fetch_pivot()															yes
	list_extract_index()														yes
	list_extract_ref()															yes
	list_delete_index()															yes
	list_delete_group()															yes
	list_delete_ref()															yes
	list_delete_duplicates()					yes								yes
	list_search_extract()						yes								yes
	list_search_delete()						yes								yes
	list_search_expunge()						yes								yes

	list_get_index()												
	list_get_index_ref()												
	list_get_ref()													
	list_get_next_ref()												
	list_locate_index()												
	list_locate_pivot()										
	list_locate_ref()												
	list_valid_ref()												
	list_iteration_start()											
	list_iteration_start_pivot()									
	list_iteration_has_more()										
	list_iteration_has_more_pivot()									
	list_iteration_get_next()										
	list_iteration_stop()											

	list_simple_search()						yes							
	list_occurrences()							yes							
	list_search()								yes							
	list_search_pivot()							yes				yes			
	list_search_ref()							yes							
	list_search_shift()							yes							
	list_locate_maximum()				yes							
	list_locate_minimum()				yes							

	list_sorted_find()					yes						yes	
	list_sorted_insert()				yes						yes		yes	
	list_sorted_extract()				yes						yes				yes	
	list_sorted_delete()				yes						yes				yes	
	list_sorted_expunge()				yes						yes				yes				

	list_sort()							yes							
	list_safe_sort()					yes							
	list_fast_sort()					yes							
	list_tiny_sort()					yes							
	list_shift_index()												
	list_shift_ref()												
	list_rotate()									
	list_flip()														
	list_shuffle()													
	list_process()																				yes	
	list_process_range()																		yes	
	list_callback_iteration()															yes	

	list_optimize_index()											
	list_optimize_memory()											
	list_do_not_seed_random()
	list_information()												
	list_error_string()	
	
	Notes:
	(1) only for destination list
	(2)	if destination is a pivot class list										*/


/*	A Note about Sorts.
	===================
	
	D-List provides 4 sort functions, which might seem excessive, however each
	is designed to meet certain design criteria. This is a brief overview of
	them, however detailed descriptions are provided with each of the functions.
	
																 Retains	  Uses
												Relative		 Element	  Extra		Safe
		Name 				Sort Type 			 Speed			References 	Resources Ordering
		----				---------			--------		----------	--------- --------
		list_fast_sort()	Quicksort (list)	Fastest 100%		No		  No		 No
		list_sort()			Quicksort (array)	Fast	85%			Yes		  Yes		 No
		list_tiny_sort()	Quicksort (list)	Slower 50%	 		Yes		  No		 No
		list_safe_sort()	Insertion (list)	Very Slow <25%		Yes		  No		 Yes


	In general it is recommended that users use the function list_fast_sort() if
	there is no requirement to retain existing element references, or the function
	list_sort() if element references are to be retained across the sort. The other
	sort functions are tailored to more specific requirements.
	
	Additionally the dlist.c source code includes a few other sub sort algorithms
	that are not used, as they are too inefficient, but were used as benchmarking
	experiments for finishing the small blocks of quicksort array sorts. On the
	x86 platform the quicksort array algorithm sorts fastest without passing the
	small blocks to another sort function. These unused sub-functions are included
	for users who may wish to experiment on other platforms. The quicksort list
	algorithm performs better when small blocks are passed to the customized
	insertion sort instead of recursive calling to finish them off.						*/





/*	A Note about list sizes and list size hints.
	============================================

	The list_create() and list_create_pivot() function calls allow the user to
	provide an optional hint as to the expected size (in elements) of the list.
	
	Callers may provide this hint, or leave it at LIST_NO_SIZE_HINT. The setting
	of the hint allows a more efficient trade-off between resources consumed and
	operation speeds of various functions. Setting the hint to an appropriate
	value can reduce processing times for searches and sorts etc, for very large
	lists, by 25% or more. Setting this hint too high will consume unneeded
	resources, setting it too low will increase processor overhead.
	
	The list size hints, defined in this file are expected to be used as follows:
	
		Define name			Intended sizes					Allocation Size
		LIST_NO_SIZE_HINT	No size hint provided, 			 1000 nodes per block
								use defaults
		LIST_SIZE_TINY		< 100 elements expected			  110 nodes per block
		LIST_SIZE_SMALL		< 1,000 elements expected		  200 nodes per block
		LIST_SIZE_MEDIUM	< 25,000 elements expected		 2000 nodes per block
		LIST_SIZE_LARGE		< 250,000 elements expected		 5000 nodes per block
		LIST_SIZE_HUGE		< 2,500,000 elements expected	15000 nodes per block
		LIST_SIZE_MEGA		> 2,500,000 elements expected	30000 nodes per block
	
	Getting the balance right during code testing and beta-testing will
	produce better performance results with less resources consumed.
	
	These size hints can be particularly helpful to embedded systems developers.
	Setting the hint lower may cause the lists to process some functions slower,
	that is assuming the lists are very large, but will use resources far more
	efficiently, and reduce the empty space to a very low number. Embedded
	systems developers may also wish to experiment with the default setting
	(LEAF_NODE_ARRAY_SIZE) as well. Others should probably leave it alone, as a
	lot of testing and experimentation went into setting it originally. The
	same is true for the block settings for each available hint, they were
	chosen as a good balance between various function speeds and resources
	used for the lists of a given size. You may choose to experiment with the
	performance testing modules, and changing the list size hints, and see how
	that effects the performance of the functions you need most in your project.
	
		List size hints are ignored and have no practical effect, if D-List was
		compiled with DLIST_USE_MALLOC defined (forcing D-List to use malloc()
		instead), such an option will render the operation of the lists much
		slower anyway, in most cases by 60% or more. But there will be a very
		minor memory savings, which for a few platforms may be important, but
		that also depends on how much slop is lost by that particular malloc()
		implementation.
	
	If you are running the regression test suite with DEBUG_DLIST defined, or
	developing D-List internal code, use either the TINY or SMALL hint only on
	test code list_create() calls. Due to the excessive audit that debugging
	D-List creates, especially the double cross-referencing audit of the freep
	list, having very large AND very empty allocation blocks will cause enormous
	slow-downs. For example, running the regression test suite for 1000 cycles
	with a default or larger allocation block size with DEBUG_DLIST defined will
	cause it to take 45 mins or more to complete. Without defining DEBUG_DLIST it
	takes about 2 mins for a 1000 cycle run. Of course the whole point of the
	regression tests is to stress and audit the D-List code itself. But there are
	also times when running huge numbers of cycles with debugging off, helps
	isolate random or other creeping errors in the regression test code itself,
	or in the D-List code that compiles when DEBUG is not defined.

	In general the only time to compile D-List with DEBUG_DLIST defined is for
	core development of D-List code and nothing else. It serves no useful purpose
	in any other scenario.

																					*/


/*	=====================================================================
		Section - User Function Prototypes
	=====================================================================	*/


/* A function to compare the value of elements. 
	
	Used while performing sorts, ordering and assessments of value ranges on
	list elements.
	
	input	pointers to elements a and b,
			The function must handle NULL pointers and element data errors
			internally. No checking is done prior to calling.
	returns	<0	if (a > b)			ELEMENT_A_LESSTHAN_B
			 0	if (a == b)			ELEMENTS_EQUAL
			>0	if (a < b)			ELEMENT_A_MORETHAN_B							 */
typedef int (*element_compare)(const void *element_a, const void *element_b);

/* A function to decide the pivot value of elements. 

	Used while performing inserts, searches and locates. The returned value 
	direct decides the ordering of the element in relation to the pivot point,
	and the head and tail of the list.
	
	input	pointer to element to calculate its pivot value.
			The function must handle NULL pointers and element data errors
			internally. No checking is done prior to calling.
	returns PIVOT_MINOR if (element < pivot point),
			PIVOT_MAJOR if (element > pivot point)									*/
typedef int (*element_pivot)(const void *element);

/* A function to locate elements based on an arbitrary input value. 
	
	Used for comparison during locate, search and list evaluation
	operations.

	input 	pointer to the element to search
			pointer to data to use for the search, key
			The function must handle NULL pointers and element data errors
			internally. No checking is done prior to calling.
	returns	TRUE if element matches the provided key,
			FALSE if not															 */
typedef boolean (*element_search)(const void *element, const void *key);

/* A function to compute the size, in bytes, required to store the element

	input 	pointer to the element to measure
			The function must handle NULL pointers and element data errors
			internally. No checking is done prior to calling.
	returns	Size of element in bytes												 */
typedef size_t (*element_size)(const void *element);

/* A function to process every element during the sequence of
	list_callback_iteration().
	
	Upon each call to this function, the element reference of the element,
	the index and the pivot value are provided to the function with the
	element data.

	input 		pointer to the element to process
				The function must handle NULL pointers and element data
				errors internally. No checking is done prior to calling.
	reference	the element reference of this element.
	index		the index of this element.
	pivot		the pivot value for this element, will be ENTIRE_LIST if
				list is not pivot class.
	list		the list being iterated on, in case the function needs to
				move or delete any elements during the iteration sequence.
	returns 0	continue iterating.
			!0	stop iterating and return control and return code back to
				original caller. Meaning of return code is totally user
				dependent.															*/
typedef int (*element_iterate)(	const void *element,
								const void *reference,
								const unsigned int index,
								const int pivot,
								const void *list);

/* A function to process every element in a list during the operations
	of list_process() function call.
	
	Unlike an element_iterate function this function is called for every
	list element in the range of the list_process() call without exception.
	It is typically used to build user or customized statistical or data
	transformation functions to extend D-List.

	Upon each call to this function, the element reference of the current
	element, the index and the pivot value are provided to the function with
	the element data. Also provided are the optional user data, a long int
	field and a void pointer, that can be read, and/or modified by the
	function. This provides a clean MP safe mechanism for the function to
	communicate and return computed values to the original caller of
	list_process().
	
	Upon calling this function for the first or the last element in the range
	of elements that must be processed, the argument end_element is set to
	TRUE. Otherwise for all other elements it is set to FALSE. So for example
	if the original call is processing the entire list, then it is set TRUE
	for the first element (index will be 0 as well) and then again for the
	last element in the list (index will be > 0). It will be set to FALSE for
	all other list elements. This provides a signal to the function, if it
	needs to compute such items as totals, distributions, mean values, or any
	other statistical functions.
	
	Return code is only valid and used when returning from the last
	element call for a list.

	input 		pointer to the element to process
				The function must handle NULL pointers and element data errors
				internally. No checking is done prior to calling.
	end_element	<0 this current element is the first element to be processed
				>0 this current element is the last element to be processed
				0 (FALSE) any other list element.
	reference	the element reference of this element.
	index		the index of this element.
	pivot		the pivot value for this element, will be ENTIRE_LIST if
				list is not pivot class.
	user_value	an optional long int field provided by the original caller of
				list_process() to allow the function use as needed.
	user_data	an optional pointer to a field or a struct, that is provided
				by the original caller of list_process() to allow the
				function use as needed.
	returns	any int value. Meaning of return code is totally user dependent.
			Can also return any mutually agreed upon values through the args
			user_value and user_data.												*/
typedef int (*element_process)(	const void *element,
								const int end_element,
								const void *reference,
								const unsigned int index,
								const int pivot,
								long int *user_value,
								void *user_data);

/* A user function to process a new element when it is inserted in a list

	input 	pointer to the element to process after insertion
			The function must handle NULL pointers and element data errors
			internally. No checking is done prior to calling.
	returns None																 	*/
typedef void (*element_insert)(const void *element);

/* A user function to process an element when it is removed from a list

	input 	pointer to the element to process before removal
			The function must handle NULL pointers and element data errors
			internally. No checking is done prior to calling.
	returns None																 	*/
typedef void (*element_remove)(const void *element);

#ifdef DLIST_STREAMS

/* all functions and types related to save and restore are only described.
	They may or may not be implemented at a later date
		
	Curent plans for D-List are to never implement this set of features.	dbw		*/

/* A function to process elements during the sequence of saving the list
	to an output stream. Please see list_save_out() for more information.
	
	The function must process the element into an output buffer, which the
	function is responsible for allocating and sizing. The output buffer
	must be suitable for a text stream, not a binary stream, and the data
	elements separated by the '|' character, which is uniquely reserved
	for field separation. The representation of the element data must be
	both byte ordering and type agnostic, so it can be read used on a
	different platform as needed. If you need to use the '|' character for
	the element, you must use a substitute and convert upon restore.
	
	The buffer provided as output from this function will be free(d) after
	the element data is written out to the stream. The buffer must be at
	least 2 characters larger than the size of the data being output. The
	output data will have a newline '\n' character added to the end of the
	buffer for output. The buffer can contain many lines as needed, but
	will always be terminated by the D-List subsystem prior to writing to
	the output stream.

	input 	element - pointer to the element to process for saving to a stream
	 		The function must handle NULL pointers and element data errors
			internally. No checking is done prior to calling.
	returns a pointer to a buffer allocated by the function containing the
			element data fields in text form.
			Function writes to data_size the length of the data, in bytes,
			in the buffer returned. The buffer must be at least 2 bytes longer.
			Function writes to hash a function dependent hash for this element
			data. If the hash is 0 then no hash (checksum) will be applied
			to this data when read back by the element_restore function.			*/
typedef void (*element_save_out)(const void *element, size_t *data_size, uint32_t *hash);

/* A function to process elements during the sequence of restoring the list
	from an input stream. Please see list_restore() for more information.
	
	The function must process the element from the input buffer, which the
	function is responsible for allocating and sizing. The output element
	must be allocated and sized correctly by the function. The input data
	elements are separated by the '|' character, which is uniquely reserved
	for field separation. The representation of the element data and fields
	was established by the element_save_out() function.
	
	The element provided as input to this function will be free(d) after
	the element is provided upon return. The input buffer will not include the
	final newline '\n' character at the end stream. The input size denotes all
	bytes up to, but not including it.

	input 	data_buffer - pointer to the data buffer to process into a new
			data element.
	 		The function must handle NULL pointers and element data errors
			internally. No checking is done prior to calling.
			data_size	the size of the data in the buffer.
			hash		the hash previously computed by the save_out function
	returns a pointer to an element allocated by the function containing the
			element data fields in final list form. This element will be
			appended to the list, if the list is linear class, prepended to
			the list if finite class, or placed on the correct side of the
			pivot point, if the list is pivot class.								*/
typedef void (*element_restore)(const void *element, const size_t data_size, const uint32_t hash);

#endif	/* DLIST_STREAMS */

/* A function to compute the hash of an element. Only used, and required for, 
	storing or accessing elements in all hashed lists.
	
	The result of this function call will determine which list bucket of
	a hashed list, this element will be in. Required for all element 
	operations, insertions, access, deletions etc.
	
	ONLY used for hashed lists. Not used for normal D-List list types.

	input 	pointer to the element to calculate the hash for
			The function must handle NULL pointers and element data errors
			internally. No checking is done prior to calling.
	returns	The hash of the element in the hashed_list_object						*/
typedef unsigned int (*element_hash)(const void *element);


/*	=====================================================================
		Section - Pre-built User Functions supplied by D-List
	=====================================================================	*/


/* Pre-built compare functions that are included in D-List

	You can use these for basic data types, or use them as a template
	to write your own more complex ones. You must write you own compare
	functions if you wish to operate the lists with elements that are
	arrays or structures.
	
	Choose the compare function you need based on the type name embedded
	in the function name.
	
		please refer to list_set_compare_function() for more information			*/

int list_default_compare_int8_t(const void *element_a, const void *element_b);
int list_default_compare_int16_t(const void *element_a, const void *element_b);
int list_default_compare_int32_t(const void *element_a, const void *element_b);
int list_default_compare_int64_t(const void *element_a, const void *element_b);

int list_default_compare_uint8_t(const void *element_a, const void *element_b);
int list_default_compare_uint16_t(const void *element_a, const void *element_b);
int list_default_compare_uint32_t(const void *element_a, const void *element_b);
int list_default_compare_uint64_t(const void *element_a, const void *element_b);

int list_default_compare_short(const void *element_a, const void *element_b);
int list_default_compare_int(const void *element_a, const void *element_b);
int list_default_compare_long(const void *element_a, const void *element_b);
int list_default_compare_float(const void *element_a, const void *element_b);
int list_default_compare_double(const void *element_a, const void *element_b);

#ifdef time_t
int list_default_compare_time_t(const void *element_a, const void *element_b);
#endif

int list_default_compare_pointer(const void *element_a, const void *element_b);
int list_default_compare_string(const void *element_a, const void *element_b);

/* Pre-built search functions that are included in D-List

	You can use these for basic data types, or use them as a template
	to write your own more complex ones. You must write you own search
	functions if you wish to operate the lists with elements that are
	arrays or structures.
	
	Choose the search function you need based on the type name embedded
	in the function name.
	
		please refer to list_set_search_function() for more information				*/

boolean list_default_search_int8_t(const void *element_a, const void *element_b);
boolean list_default_search_int16_t(const void *element_a, const void *element_b);
boolean list_default_search_int32_t(const void *element_a, const void *element_b);
boolean list_default_search_int64_t(const void *element_a, const void *element_b);

boolean list_default_search_uint8_t(const void *element_a, const void *element_b);
boolean list_default_search_uint16_t(const void *element_a, const void *element_b);
boolean list_default_search_uint32_t(const void *element_a, const void *element_b);
boolean list_default_search_uint64_t(const void *element_a, const void *element_b);

boolean list_default_search_short(const void *element_a, const void *element_b);
boolean list_default_search_int(const void *element_a, const void *element_b);
boolean list_default_search_long(const void *element_a, const void *element_b);
boolean list_default_search_float(const void *element_a, const void *element_b);
boolean list_default_search_double(const void *element_a, const void *element_b);

#ifdef time_t
boolean list_default_search_time_t(const void *element_a, const void *element_b);
#endif

boolean list_default_search_pointer(const void *element_a, const void *element_b);
boolean list_default_search_string(const void *element_a, const void *element_b);

/* Pre-built size functions that are included in D-List

	You can use these for basic data types, or use them as a template
	to write your own more complex ones. You must write you own size
	functions if you wish to operate the lists with elements that are
	arrays or structures.
	
	Choose the size function you need based on the type name embedded
	in the function name.
	
		please refer to list_set_size_function() for more information				*/

size_t list_default_size_int8_t(const void *element);
size_t list_default_size_int16_t(const void *element);
size_t list_default_size_int32_t(const void *element);
size_t list_default_size_int64_t(const void *element);

size_t list_default_size_uint8_t(const void *element);
size_t list_default_size_uint16_t(const void *element);
size_t list_default_size_uint32_t(const void *element);
size_t list_default_size_uint64_t(const void *element);

size_t list_default_size_short(const void *element);
size_t list_default_size_int(const void *element);
size_t list_default_size_long(const void *element);
size_t list_default_size_float(const void *element);
size_t list_default_size_double(const void *element);

#ifdef time_t
size_t list_default_size_time_t(const void *element);
#endif

/* use this pointer size function with care, if on a platform with multiple pointer sizes */
size_t list_default_size_pointer(const void *element);

size_t list_default_size_string(const void *element);

/* example pivot function, not that useful, but a template.
	An element <0 will be put on the minor side. */
int list_pivot_zero_int(const void *element);


#include "dlist_internal.h"


/*	=====================================================================
		Section - Function like MACRO's
	=====================================================================	*/

/* Macro to achieve the same as list_is_sorted(), but without the function
	call, stack shift etc. x MUST be a pointer to a list_object.
	Used often by FWS address subsystem, which uses dynamic list objects.			*/
#define LIST_SIZE(x) (x->element_count)

/* Macro to achieve the same as list_is_sorted(), but without the function
	call, stack shift etc. x must be a local list_object variable.					*/
#define LIST_SIZE_VAR(x) (x.element_count)

/* Macro to achieve the same as list_is_sorted(), but without the function
	call, stack shift etc. x MUST be a pointer to a list_object.
	Used often by FWS address subsystem, which uses dynamic list objects.			*/
#define LIST_IS_SORTED(x) (x->list_sorted == TRUE)

/* Macro to achieve the same as list_is_sorted(), but without the function
	call, stack shift etc. x must be a local list_object variable.					*/
#define LIST_IS_SORTED_VAR(x) (x.list_sorted == TRUE)




/*	=====================================================================
		Section - List Object Management
	=====================================================================	*/

/* Initialize a new list, set up ready to store the elements. All other operations
	on a list_object (a list) must occur after initialization, or they will fail.
	
	This function creates a normal linear class list. All operations are permitted
	on such lists, except for functions that are specific to a different class of
	lists, such as list_split_pivot() or list_store().

	new_list	pointer to caller supplied list_object
	copy_data	LIST_COPY_DATA (TRUE) list will allocate, manage element
				memory, and copy all data to the elements when they are
				created.
				LIST_DO_NOT_COPY (FALSE) user is responsible for managing
				element data, allocating and free(ing) the element resources.
	size_function
				If copy_data is TRUE (LIST_COPY_DATA), a pointer to a user
				function that returns size requests for list elements.
				If copy_data is FALSE (LIST_DO_NOT_COPY) then NULL must be
				passed.
	size_hint	An optional hint to D-List about the expected size of the
				new list, this sizing hint allows D-List to set up internal
				resources that are appropriate for the expected usage, and
				if set correctly allows for significantly better performance
				when searching, traversing and sorting very large lists, the
				differences can be 25-40% faster than just using the defaults.
				Alternatively for very small lists there would be a some
				considerable wasted resources if the hint was set too high.
				Unlike finite class lists, this hint does not set any maximum
				size or limits, it is just a hint to balance resources and
				performance. Unlike finite class lists which allow for the
				adjustment of the maximum element limit, you cannot change the
				size hint once the list is established. If you find the actual
				number of elements is wildly different from the hint, the best
				option is to create a new list with the appropriate hint and
				transfer the elements to it, using list_clone(). The function
				list_transfer() copies all the list data to another list, that
				includes all the internal resources, so it would perpetuate the
				original problem. The size_hint can be any of these:
					LIST_NO_SIZE_HINT	No size hint provided, use defaults
					LIST_SIZE_TINY		< 100 elements expected
					LIST_SIZE_SMALL		< 1,000 elements expected
					LIST_SIZE_MEDIUM	< 25,000 elements expected
					LIST_SIZE_LARGE		< 250,000 elements expected
					LIST_SIZE_HUGE		< 2,500,000 elements expected
					LIST_SIZE_MEGA		> 2,500,000 elements expected
	returns	LIST_OK			success
			LIST_ERR		on error 												*/
int list_create(	list_object *restrict new_list,
					const boolean copy_data,
					const element_size size_function,
					const unsigned int size_hint);

/* Initialize a new list, set up ready to store the elements. All other operations
	on a list_object (a list) must occur after initialization, or they will fail.
	
	This function creates a finite list, wich is a restricted linear class list.
	Most operations are permitted on these lists, except for functions that are
	specific to a different class of lists or reference elements in a way which
	would cause problems, such as list_split_pivot() or list_store().
	
	A finite list is a linear list, with a defined maximum number of elements
	that it is permitted to hold. Elements can be added to it until it reaches its
	maximum capacity, then for each new element added that would exceed the stated
	capacity an element is removed from the tail end of the list. Such lists have
	some limited utility, such as keeping a cache of recently added elements from
	a much larger or tree structure list, which is efficient at finding items, but
	poor at locating recently added items, for example. Also keeping a running
	scroll-back of items, similar to a scroll-back of a terminal window. Any
	situation where items are wanted to be kept, but a defined set of resources is
	to be used, instead of an endless list. Obviously this finite class of list is
	not appropriate for elements that are important or must be kept. Additionally
	as elements can be removed without the user explicitly knowing, no functions
	using input references are allowed to operate on finite class lists.
	
	To maintain the integrity of the list, the following functions are not
	available to operate on finite lists.
		list_append()
		list_size_pivot()
		list_set_pivot_function()
		list_split_pivot()
		list_insert_ref()
		list_store()
		list_fetch_pivot()
		list_get_ref()
		list_get_next_ref()
		list_search_ref()
		list_iteration_start_pivot()
		list_iteration_has_more_pivot()

	The following functions are available to operate on finite lists, but with
	specific limitations as noted in the function descriptions.
		list_concatenate()
		list_clone()
		list_delete_ref()
		list_search()
	
	For obvious reasons, it is best to add new elements to a finite class list
	via the function list_prepend(), this also leads to more readable code where
	this class of list is used.

	new_list	pointer to caller supplied list_object
	copy_data	LIST_COPY_DATA (TRUE) list will allocate, manage element
				memory, and copy all data to the elements when they are
				created.
				LIST_DO_NOT_COPY (FALSE) user is responsible for managing
				element data, allocating and free(ing) the element resources.
	size_function
				If copy_data is TRUE (LIST_COPY_DATA), a pointer to a user
				function that returns size requests for list elements.
				If copy_data is FALSE (LIST_DO_NOT_COPY) then NULL must be
				passed.
	max			The maximum number of elements this list may hold. Unlike the
				other list classes, this is not a hint, it is an absolute
				maximum limit. This size limit can be adjusted by using the
				function list_set_finite_limit().
	returns	LIST_OK			success
			LIST_ERR		on error 												*/
int list_create_finite(	list_object *restrict new_list,
						const boolean copy_data,
						const element_size size_function,
						const unsigned int max);

/* Initialize a new pivot class list, set up ready to store the elements. All
	other operations on a list_object (a list) must occur after initialization,
	or they will fail. Pivot class lists are functionally similar to normal,
	linear, lists, however there are some major differences.
	
	The placement of elements in the list is decided by the response of queries
	to the pivot_function for each element presented. The purpose is to allow
	the list to organize the elements around a pivot point. For large unsorted
	lists this can provide a substantial speed gain for search operations.
	
	To maintain the pivot organization of elements, the following list functions
	are not permitted to operate on pivot class lists.
		list_append()
		list_prepend()
		list_insert_index()
		list_insert_ref()

	Instead the list_store() or list_sorted_insert() function must be used to 
	add any element to a pivot class list. All other functions work on pivot 
	class lists, and any particular issues are described in that functions 
	overview. For a more complete list of what functions are available for which 
	classes of list, please refer to the table at the top of this file.
	
	When using the D-List functions to manage a pivot list, please use the defines
	established above for consistency and readability of the code. A pivot list is
	organized as this diagram shows
	
		elem#1 .. elem#2 .. elem#3 .. elem#4 ... elem .. elem .. elem .. elem .. elem .. elem#n
		  ^									  ^												^
		  |									  |												|
	<HEAD_OF_LIST>	<-- PIVOT_MINOR -->	   <PIVOT>		<-- PIVOT_MAJOR -->			<TAIL_OF_LIST)
	
	A pivot class list must have a pivot function set for the list, prior to any
	update usage. The function can be replaced during the life of the list, but
	one must always be set. Please refer to list_set_pivot_function().

	new_list	pointer to caller supplied list_object
	copy_data	LIST_COPY_DATA (TRUE) list will allocate, manage element
				memory, and copy all data to the elements when they are
				created.
				LIST_DO_NOT_COPY (FALSE) user is responsible for managing
				element data, allocating and free(ing) the element resources.
	size_function
				If copy_data is TRUE (LIST_COPY_DATA), a pointer to a user
				function that returns size requests for list elements.
				If copy_data is FALSE (LIST_DO_NOT_COPY) then NULL must be
				passed.
	pivot_function
				required pointer to user function that will process pivot
				identification requests when new elements are added to the
				list subsystem via list_store().
	size_hint	An optional hint to D-List about the expected size of the
				new list, this sizing hint allows D-List to set up internal
				resources that are appropriate for the expected usage, and
				if set correctly allows for significantly better performance
				when searching, traversing and sorting very large lists, the
				differences can be 25-40% faster than just using the defaults.
				Alternatively for very small lists there would be a some
				considerable wasted resources if the hint was set too high.
				Unlike finite class lists, this hint does not set any maximum
				size or limits, it is just a hint to balance resources and
				performance. Unlike finite class lists which allow for the
				adjustment of the maximum element limit, you cannot change the
				size hint once the list is established. If you find the actual
				number of elements is wildly different from the hint, the best
				option is to create a new list with the appropriate hint and
				transfer the elements to it, using list_clone(). The function
				list_transfer() copies all the list data to another list, that
				includes all the internal resources, so it would perpetuate the
				original problem. The size_hint can be any of these:
					LIST_NO_SIZE_HINT	No size hint provided, use defaults
					LIST_SIZE_TINY		< 100 elements expected
					LIST_SIZE_SMALL		< 1,000 elements expected
					LIST_SIZE_MEDIUM	< 25,000 elements expected
					LIST_SIZE_LARGE		< 250,000 elements expected
					LIST_SIZE_HUGE		< 2,500,000 elements expected
					LIST_SIZE_MEGA		> 2,500,000 elements expected
	returns	LIST_OK			success
			LIST_ERR		on error 												*/
int list_create_pivot(	list_object *restrict new_list,
						const boolean copy_data,
						const element_size size_function,
						const element_pivot pivot_function,
						const unsigned int size_hint);

/* Change the maximum number of elements a finite class list may hold.
	
	If the new maximum number is greater than the current one, no changes will
	occur in the list, it will just accept new elements without removing old
	elements from the tail, until the new limit is reached. If the new limit is
	less than the old one, then the next time an element is added, the list will
	be reduced to the new limit set here, by sequentially removing elements from
	the tail end until the new limit is reached.

	list		list_object that has been previously initialized.
	max			The new maximum number of elements this list may hold.
	returns	LIST_OK			success
			LIST_ERR		on error 												*/
int list_set_finite_limit(	list_object *restrict list,
							const unsigned int max);

/* Remove and erase an exiting list.

	Upon exit it cannot be used to store elements.
	
	If the list was initialized with LIST_COPY_DATA set TRUE, all
	elements in the list will be removed, deallocated and returned
	to the system for reuse.
	
	Upon exit the list_object is cleared, but it is not deallocated.
	That is the responsibility of the caller.

	list		list_object that has been previously initialized.
	returns	nothing 																*/
void list_erase(list_object *restrict list);


/*	=====================================================================
		Section - List Function Management
	=====================================================================	*/


/* Set the compare function for the list provided.

	Compare functions are used while performing sorts, ordering and
	assessments of value ranges on list elements.

	list		list_object that has been previously initialized.
	compare_function
				pointer to function to process compare requests by the list
				subsystem when sorted or comparing list elements. If NULL any
				existing compare function is removed.
	returns	LIST_OK			success
			LIST_ERR		on error	 											*/
int list_set_compare_function(	list_object *restrict list,
								const element_compare compare_function);

/* Get the pointer to the current compare function for the list provided.

	Compare functions are used while performing sorts, ordering and
	assessments of value ranges on list elements. Please refer to
	list_set_compare_function() above.
	
	Useful for large code bases that need to use a different compare
	function, and then restore the original compare function when done.

	list		list_object that has been previously initialized.
	returns	current pointer to compare function, or NULL if none is set.			*/
element_compare *list_get_compare_function(const list_object *restrict list);

/* Set the pivot function for the list provided.
	
	Pivot functions are used to decide which side of a pivot list an
	element is placed, or where to search for the element. A pivot
	function must be set before a pivot class list can have any
	elements added to it. The pivot function was set when the pivot
	list was created, this function allows you to change that function.

	list		list_object that has been previously initialized.
	pivot_function
				pointer to function to process pivot requests for list
				elements, by the list subsystem. If NULL is passed, any
				existing compare function is removed.
	returns	LIST_OK			success
			LIST_ERR		on error 												*/
int list_set_pivot_function(	list_object *restrict list,
								const element_pivot pivot_function);

/* Set the search function for the list provided.

	Search functions are used while performing searches list elements to
	find one that matches user supplied search criteria.

	list		list_object that has been previously initialized.
	search_function
				pointer to function to process search requests for list
				elements, by the list subsystem. If NULL is passed, any
				existing compare function is removed.
	returns	LIST_OK			success
			LIST_ERR		on error 												*/
int list_set_search_function(	list_object *restrict list,
								const element_search search_function);

/* Get the pointer to the current search function for the list provided.

	Search functions are used while performing searches on list elements.
	Please refer to list_set_search_function() above.
	
	Useful for large code bases that need to use a different search function,
	and then restore the original compare function when done.

	list		list_object that has been previously initialized.
	returns	current pointer to search function, or NULL								*/
element_search *list_get_search_function(const list_object *restrict list);

/* Set the size function for the list provided.

	Size functions are used to compute the size required to store the
	element. This function was required when the list was created with
	the LIST_COPY_DATA attribute, which makes the list allocate and
	manages the storage for elements. The size function was set when
	the list was created, however this function allows you to change
	that function, if needed.
	
	It was a D-List design choice to prevent a list that was created
	originally with LIST_COPY_DATA, to later remove that attribute.
	If such a procedure is wished, first create a new list with the
	LIST_DO_NOT_COPY attribute, and then clone the elements to that
	new list.

	list		list_object that has been previously initialized.
	size_function
				pointer to function to return size requests for list elements.
				If NULL is passed, any existing size function is removed.
	returns	LIST_OK			success
			LIST_ERR		on error 												*/
int list_set_size_function(	list_object *restrict list,
							const element_size size_function);

/* Set the insert function for the list provided.

	Insert functions are used to perform user actions related to the
	element, or its insertion into the list. It can perform whatever
	actions it wants, such as gather statistics, modify the element,
	signal a thread, make computations, copy data to a cache, etc.
	However this function has no control over the insertion process
	itself.
	
	This function is called after the element is inserted into the
	list, by all list insertion functions (list_append(),
	list_prepend(), list_store(), list_insert_ref(), and
	list_insert_index() ). This function is not called by list_clone(),
	list_transfer(), list_concatenate() or list_split_pivot().

	list		list_object that has been previously initialized.
	insert_function
				pointer to function to process new list elements. If NULL is
				passed, any existing insert function is removed.
	returns	LIST_OK			success
			LIST_ERR		on error 												*/
int list_set_insert_function(	list_object *restrict list,
								const element_insert insert_function);

/* Set the remove function for the list provided.

	Remove functions are used to perform user actions related to
	the element, or its removal from the list. It can perform whatever
	actions it wants, such as gather statistics, free resources, modify
	the element, signal a thread, make computations, etc. However this
	function has no control over the removal process itself.

	This function is called before the element is removed from the list,
	by all list removal functions (list_fetch(), list_delete_index(),
	list_delete_ref(), list_delete_group(), list_extract_index(),
	list_clear(), list_erase() and list_search_delete() ). This function
	is not called by list_transfer().

	This function is also called when other functions, such as
	list_clone(), clear out a list prior to cloning. If you do not want
	this function called during such implicit operations, first remove
	the function prior to calling them.

	list		list_object that has been previously initialized.
	remove_function
				pointer to function to process list elements that are about to
				be removed. If NULL is passed, any existing remove function is
				removed from the list.
	returns	LIST_OK			success
			LIST_ERR		on error 												*/
int list_set_remove_function(	list_object *restrict list,
								const element_remove remove_function);

/* Set the iterate function for the list provided.

	Iterate functions are used to process elements during a sequence
	of list iteration started by the function list_callback_iteration().

	list		list_object that has been previously initialized.
	iterate_function
				pointer to function to process list elements. If NULL
				is passed, any existing iterate function is removed.
	returns	LIST_OK			success
			LIST_ERR		on error 												*/
int list_set_iterate_function(	list_object *restrict list,
								const element_iterate iterate_function);


/*	=====================================================================
		Section - List Information Functions
	=====================================================================	*/

/* Examine a list for elements, and return a boolean indicator if empty or not.

	list		list_object that has been previously initialized.
	returns	TRUE			the list has no user elements, it is empty.
			FALSE			there are user elements in the list, it is not empty. 	*/
boolean list_is_empty(const list_object *restrict list);

/* Examine a list and return the count of the number of elements in the list.

	list		list_object that has been previously initialized.
	returns >0			The number of elements stored in the list,
			0			if it is empty.												*/
unsigned int list_size(const list_object *restrict list);

/* Examine a pivot class list and return the count of the number of elements
	in the list, on both sides of the pivot.
	
	If called for another class of list, will return the current number
	of elements in the list, and generate a warning message in DEBUG mode.

	list		list_object that has been previously initialized.
	returns	minor_count		the number of elements on the left (minor) side
							of the pivot point.
			major_count		the number of elements on the right (major)
							side of the pivot point.
			>0 				The number of elements stored in the list,
			0				if it is empty.											*/
unsigned int list_size_pivot(	const list_object *restrict list,
								unsigned int *minor_count,
								unsigned int *major_count);

/* Examine a list for sorted elements, and return a boolean indicator if
	list is sorted or not.

	list		list_object that has been previously initialized.
	returns	TRUE			the list has sorted elements.
			FALSE			the list is not sorted. 								*/
boolean list_is_sorted(const list_object *restrict list);


/*	=====================================================================
		Section - List Transformation Functions
	=====================================================================	*/

/* Concatenate 2 lists together into the destination list.

	This function copies and merges the 2 input lists, list_1 first, then
	list_2. The destination list must have been previously initialized by
	the caller. It may, or may not, contain existing elements. If it does
	the elements will be preserved, with the list_1 and then list_2 elements
	concatenated after them in the list.
	
	The lists may be the same, please see the variations below. Pivot class
	lists have restrictions when used as destinations, but are unrestricted
	when used as input lists. Finite lists can only be used as input lists.

	If the destination list was created with a LIST_COPY_DATA attribute, and
	a size function has been set for the list (see list_set_size_function()),
	the element data is fully copied to the new elements. If not, the element
	is copied as is, with the existing pointer to the original element data,
	unchanged.
	
	linear list usage
	list_concatenate(list_a, list_b, list_c);	add list_a & list_b to list_c
	list_concatenate(list_a, NULL, list_a);		add list_a to itself (double it)
	list_concatenate(list_a, NULL, list_b);		add list_a to list_b
	list_concatenate(list_a, list_a, list_b);	add list_a twice to list_b
	list_concatenate(list_a, list_b, list_b);	add list_a & list_b to list_b
	list_concatenate(list_b, list_a, list_b);	add list_b & list_a to list_b
	list_concatenate(list_a, list_a, list_a);	add list_a to itself twice (triple it)
	
	for destination pivot lists (no self-referencing destinations)
	list_concatenate(list_a, list_b, list_c);	add list_a & list_b to list_c
	list_concatenate(list_a, NULL, list_a);		invalid, returns LIST_ERR
	list_concatenate(list_a, NULL, list_b);		add list_a to list_b
	list_concatenate(list_a, list_a, list_b);	add list_a twice to list_b
	list_concatenate(list_a, list_b, list_b);	invalid, returns LIST_ERR
	list_concatenate(list_b, list_a, list_b);	invalid, returns LIST_ERR
	list_concatenate(list_a, list_a, list_a);	invalid, returns LIST_ERR
	
	
	At first glance example 3 above might look the same as list_clone(),
	however list_concatenate() does not clear the destination list first.
	
	If the destination list is a pivot class list, all elements added to
	it will be correctly positioned using the pivot function for that list
	regardless of either of the input lists types. However all elements
	will be added to the ends of the list, not the pivot of the list. A
	destination list that is pivot class, must not be referenced as one of
	the input lists.
	
	If instead of a straight copy of the 2 input lists, you wish to perform a
	more analytical list merge procedure, and remove any duplicates that may
	exist between the input lists, then please refer to list_assay() instead.
	
	The destination list must not be a finite class list.
	
	List_1 and list_2 remain unchanged, unless either was provided as the
	output destination list. In which case it will be modified as requested
	by this function.
	
	list_1		list_object that has been previously initialized,
				presumably with elements in it.
	list_2		list_object that has been previously initialized,
				presumably with elements in it, or NULL.
	dest_list	list_object that has been previously initialized, and
				may be empty, or have existing elements in it.
	returns	LIST_OK			success
			LIST_ERR		on error 												*/
int list_concatenate(	const list_object *list_1,
						const list_object *list_2,
						list_object *restrict dest_list);

/* Create a clone of a list, into the destination list.

	This function copies the input list into the destination list, which must
	have been previously initialized by the caller. The destination list should
	be empty first, before calling this function, however if the destination list
	contains existing elements, they will be deleted first, before the cloning
	operation.
	
	The lists must not be the same. If you wish the perform such merge operations,
	please refer to the list_concatenate() function instead.
	
	If the destination list was created with a LIST_COPY_DATA attribute, and
	a size function has been set for the list (see list_set_size_function()),
	the element data is fully copied to the new elements. If not, the element
	is copied as is, with the existing pointer to the original element data,
	unchanged.
	
	The input and destination lists must be of the same class, either linear or
	pivot. If you wish to make a copy of a list in a different class, please
	use the list_concatenate() function instead.
	
	The destination list must not be a finite class list. However the input list
	may be a finite list, as it is treated as a linear list for this function, as
	long as the destination list is linear class.
		
	The input list remains unchanged, the destination list will be modified as
	requested by this function.
	
	list		list_object that has been previously initialized,
				presumably with elements in it.
	dest_list	list_object that has been previously initialized, and
				may be empty, or have existing elements in it that will
				be removed first.
	returns	LIST_OK			success
			LIST_ERR		on error 												*/
int list_clone(	const list_object *list,
				list_object *restrict dest_list);

/* Transfer all the elements of a list, into the destination list.

	This function moves the entire set of elements from the input list into the
	destination list, which must have been previously initialized by the caller.
	The destination list should be empty first, before calling this function,
	however if the destination list contains existing elements, they will be
	deleted first, before the transfer operation.
	
	The lists must not be the same. If you wish the perform such merge
	operations, please refer to the list_concatenate() function instead.
	
	The LIST_COPY_DATA attribute is ignored for both lists. This function moves
	the actual elements from input_list to dest_list, leaving input_list empty
	and not creating any new elements. However the destination list must have
	been correctly set up, if it requires size or pivot functions.
	
	The input and destination lists must be of the same class, either linear,
	pivot or finite. If you wish to transfer elements from a list in a different
	class, please use the list_concatenate() function, or list_fetch() and
	list_append() instead.
	
	If the lists are of the finite class, then dest_list must have an equal
	or greater capacity than the input_list.
		
	The input list is emptied, the destination list will be emptied and then
	modified as requested by this function.
	
	list		list_object that has been previously initialized,
				presumably with elements in it.
	dest_list	list_object that has been previously initialized, and
				may be empty, or have existing elements in it that will
				be removed first.
	returns	LIST_OK			success
			LIST_ERR		on error 												*/
int list_transfer(	list_object *restrict list,
					list_object *restrict dest_list);

/* Merge 2 lists together by weaving each element together in order, into
	the destination list.

	This function copies and merges the 2 input lists, element by element
	into the destination list, so the elements are weaved together. The
	destination list must have been previously initialized by the caller.
	It may, or may not, contain existing elements. If it does the elements
	will be removed first and the list cleared. Upon return the destination
	list will only contain an ordered set of elements from list_1 and list_2.
	
	This function is similar to list_concatenate() except that the lists are
	merged element by element, not serially, and this function clears the
	destination list first. list_concatenate() also allows the lists to
	overlap and perform doubling operations, this function does not.
	
	The input lists may be of any class and do not have to be the same list
	class, but the destination must be a linear class list.
	
	If the destination list was created with a LIST_COPY_DATA attribute, and
	a size function has been set for the list (see list_set_size_function()),
	the element data is fully copied to the new elements. If not, the element
	is copied as is, with the existing pointer to the original element data,
	unchanged.
	
	The caller sets the argument order to define which direction the elements
	are read from list_2. LIST_ASCENDING means list_2 is read in the same order
	(head to tail) as list_1. LIST_DESCENDING means list_2 is read in the
	opposite order, tail to head. This allows the 2 lists to be weaved in order
	or in opposite directions, as the caller requires.
	
	Demonstrating this functions output, by using the following 2 example input
	lists
	
	list_1 elements
	a1	a2	a3	a4	a5	a6	a7	a8	.	.	.	.	.	a99

	list_2 elements
	b1	b2	b3	b4	b5	b6	b7	b8	.	.	.	.	.	b99
	
	With an order of LIST_ASCENDING the destination list would be
	a1	b1	a2	b2	a3	b3	a4	b4	a5	b5	a6	b6	.	.	.	.	.	a99	b99

	With an order of LIST_DESCENDING the destination list would be
	a1	b99	a2	b98	a3	b97	a4	b96	a5	b95	a6	b94	.	.	.	.	.	a99	b1
	
	Although this function has limited uses, it is very helpful if maintaining 	
	sorted or well ordered very large lists, that you wish to merge and sort.
	Overall this would reduce the overhead of the merge and final sort
	operations considerably.

	If instead of a straight merge of the 2 input lists, you wish to perform a
	more analytical list merge procedure, and remove any duplicates that may
	exist between the input lists, then please refer to list_assay() instead.
		
	List_1 and list_2 remain unchanged.
	
	list_1		list_object that has been previously initialized,
				presumably with elements in it.
	list_2		list_object that has been previously initialized,
				presumably with elements in it, or NULL.
	order		either LIST_ASCENDING or LIST_DESCENDING to denote which
		 		order to read the elements in the list_2.
	dest_list	list_object that has been previously initialized, and
				may be empty, or have existing elements in it which will
				be removed first.
	returns	LIST_OK			success
			LIST_ERR		on error 												*/
int list_weave(	const list_object *list_1,
				const list_object *list_2,
				const boolean order,
				list_object *restrict dest_list);

/* Perform an analysis and merge of 2 inputs lists, dividing elements that are common
	from those that are unique.

	This function will perform an element by element analysis of 2 input
	lists, and create an output pivot class list, with the minor side filled
	with elements common to both lists, and the major side filled with the
	elements unique to one list or the other. The output list will contain a
	single enumeration of all elements present in both input lists, with no
	duplicates, and with the list arranged around a pivot point of a value of
	common or unique.
	
	This list_assay() function can also be considered and used as a list merge
	operation, with additional analysis information provided, which can be
	ignored if only a merge is desired. This function differs considerably
	from list_concatenate() which puts 2 lists together as-is, retaining all
	duplicates.
	
	The 2 input lists can be of any list class, and do not have to be the
	same class. The output list must be pivot class list. If you wish the
	output to be linear or finite class, then use another function such as
	list_split_pivot() to copy or clone the output list sections after this
	function is finished.
	
	Although the destination is pivot class, this function will not use any
	pivot function you may have already set for it. The pivot datum will be
	the uniqueness or commonness of the elements, between the 2 input lists,
	and elements will be added to the destination list using that criteria. If
	there is an existing pivot function, it will not be called, or disturbed.
	
	If a search function was previously set for this list (by calling
	list_set_search_function() ), this function will use that search function
	to compare the element data against each other.
	
	If a search function was not previously set for this list, or it was
	removed, this function will perform a straight compare between the data
	pointers in both elements. If the element pointers are the same, they
	are equal, if the pointers point to different element data areas, they
	are not equal.
	
	If the lists were equal the minor side of the destination list will have
	a set of elements in it, and major side will be empty. The output list is
	required, and must not be the same as either input list. The input lists
	must be unique. Like all the list transformation functions, if the output
	list was initialized with the attribute LIST_COPY_DATA, the new elements
	will copied in whole, otherwise only a copy of the element pointer will
	be added to the destination list, with ownership of the original element
	data item with the user, with an additional pointer to it, that must be
	considered prior to any resource removal actions.
	
	The number of unique elements in list_1 is returned in l1_unique, and the
	number of unique elements in list_2 is returned in l2_unique. Using this
	the user can easily locate which elements are which, from the pivot point
	forward to the tail of the list. The destination pivot list is organized
	as this diagram shows
	
		<------   elements in common   ------>	<------------    unique elements    ----------->
												<----  l1_unique  ----><----	l2_unique	--->
		elem#1 .. elem#2 .. elem#3 .. elem#4 ... elem .. elem .. elem .. elem .. elem .. elem#n
		  ^									  ^												^
		  |									  |												|
	<HEAD_OF_LIST>	<-- PIVOT_MINOR -->	   <PIVOT>		<-- PIVOT_MAJOR -->			<TAIL_OF_LIST)
	
	
	The definitive test, on return, to verify if the lists are equal, or
	different, or how many elements are common or unique, is to call the
	function list_size_pivot() and evaluates the results. Use pivot based list
	iteration to retrieve the elements as you need them.

	This function only returns elements in the destination list, it makes no
	changes to the input lists.
	
	list_1		list_object that has been previously initialized,
				presumably with elements in it.
	list_2		list_object that has been previously initialized,
				presumably with elements in it.
	dest_list	list_object that has been previously initialized as a
				pivot class list, may be empty, or have existing elements
				in it that will be removed first. This list will hold all
				elements in common between list_1 and list_2 on the
				PIVOT_MINOR side of the list, and all elements that are
				unique from list_1 and list_2 on the PIVOT_MAJOR side.
	l1_unique	count of the elements returned that were unique to list_1
				these will be the first elements from the pivot point
				towards the tail of the list.
	l2_unique	count of the elements returned that were unique to list_2
				these will be the last elements from the pivot point
				towards the tail of the list.
				l1_unique + l2_unique will equal the MAJOR side count
				returned by a call to list_size_pivot().
	returns	LIST_OK			the evaluation of the lists succeeded
			LIST_ERR		on error 												*/
int list_assay(	const list_object *list_1,
				const list_object *list_2,
				list_object *restrict dest_list,
				unsigned int *l1_unique,
				unsigned int *l2_unique);

/* Split a pivot class list, and create 2 destination lists from its contents.

	This function copies the input list into 2 destination lists, which must
	have been previously initialized by the caller. The destination lists should
	be empty first, before calling this function, however if the destination list
	contains existing elements, they will be deleted first, before the split and
	copy operation.
	
	The lists must not be the same. If you wish the perform such merge operations,
	please refer to the list_concatenate() function instead.
	
	If the destination list was created with a LIST_COPY_DATA attribute, and
	a size function has been set for the list (see list_set_size_function()),
	the element data is fully copied to the new elements. If not, the element
	is copied as is, with the existing pointer to the original element data,
	unchanged.
	
	The input list must be a pivot class list. The two destination lists may be
	of either type, and do not have to be the same class lists. The minor or left
	hand side of the input list is copied to destination list #1, and the major
	or right hand side of the input list, is copied to destination list #2. The
	split happens at the user defined pivot point, which was established by the
	pivot function in operation on the input list.
	
	The input list remains unchanged, the destination list will be modified as
	requested by this function.
	
	list		list_object that has been previously initialized, with
				elements in it.
	dest_list1	list_object that has been previously initialized, and may
				be empty, or have existing elements in it that will be
				removed first.
	dest_list2	list_object that has been previously initialized, and may
				be empty, or have existing elements in it that will be
				removed first.
	returns	LIST_OK			success
			LIST_ERR		on error 												*/
int list_split_pivot(	const list_object *list,
						list_object *restrict dest_list1,
						list_object *restrict dest_list2);

/* Perform an element by element comparison of 2 input lists, searching for
	the first element which is not equal between them.
	
	This function traverses the 2 input lists, element by element in strict
	forward order. Each element is compared in sequence and if a non-equal
	result is obtained, the function will return with the index number in
	the lists of the non-matching element. It will only find the first non-
	matching element, and return.
	
	If a search function was previously set for this list (by calling
	list_set_search_function() ), this function will use that search function
	to compare the element datam's against each other.
	
	If a search function was not previously set for this list, or it was
	removed, this function will perform a straight compare between the data
	pointers in both elements. If the element pointers are the same, they
	are equal, if the pointers point to different element data areas, they
	are not equal.
	
	The definitive test, on return, to verify if the lists are equal, is to
	check the return code.
	
	This function only returns an index to any non-matching element in the
	lists, it makes no changes to the lists.
	
	list_1		list_object that has been previously initialized,
				presumably with elements in it.
	list_2		list_object that has been previously initialized,
				presumably with elements in it.
	returns	index	pointer to variable used to store the element index, if
		 			a non-matching element is found in both lists.
			LIST_OK			success, the 2 lists match
			ELEMENT_INVALID	if a non-matching pair of elements was found
			LIST_ERR		on error 												*/
int list_compare(	const list_object *list_1,
					const list_object *list_2,
					unsigned int *index);

/* Perform an element by element comparison of 2 inputs lists, searching for
	all the elements which is are equal between them.
	
	This function traverses the 2 input lists, element by element in strict
	forward order. Each element is compared in sequence and if a non-equal
	result is obtained, the function will count the number of such differing,
	non-matching, elements. It will find all non-matching elements, and then
	return. This function makes no attempt to semantically analyze the lists,
	for example 2 lists of 100 elements, that are the same with the exception
	that list_2 has an extra element in index position 2, will return a
	result of 97 differences, as all elements after element 1 will be
	different. To perform a more detail semantic comparison of two lists,
	please refer to the function list_assay() instead.
	
	If a search function was previously set for this list (by calling
	list_set_search_function() ), this function will use that search function
	to compare the data of both elements against each other.
	
	If a search function was not previously set for this list, or it was
	removed, this function will perform a straight compare between the data
	pointers in both elements. If the element pointers are the same, they
	are equal, if the pointers point to different element data areas, they
	are not equal.
	
	The definitive test, on return, to verify if the lists are equal, is to
	check the return code.
	
	This function only returns an index to any non-matching element in the
	lists, it makes no changes to the lists.
	
	list_1		list_object that has been previously initialized,
				presumably with elements in it.
	list_2		list_object that has been previously initialized,
				presumably with elements in it.
	returns	0				the two input lists match
			>0				the number of non-matching elements found
			LIST_ERR		on error 												*/
int list_differences(	const list_object *list_1,
						const list_object *list_2);


/*	=====================================================================
		Section - Element Placement Functions
	=====================================================================	*/


/* Append an element to the end of a list.

	Among its many uses, this function is useful for adding elements
	to FIFO lists and queues. It also allows elements to be added to
	list being managed by index, without disturbing the index
	optimizations that making modifications by reference causes.
	
	This function is not available for pivot class lists, or finite
	class lists. If the list is currently sorted, the sorted indexes
	will be voided, and the list_sorted_xx() functions are no longer
	available for the list, until it is resorted.

	list		list_object that has been previously initialized.
	element		pointer to caller supplied datum object to be stored
		 		as the new element in list.
	returns	LIST_OK				success, new element added to the list
			ELEMENT_NO_DATUM	element data not supplied
			LIST_ERR			on error	 										*/
int list_append(	list_object *restrict list,
					const void *element);

/* Prepend an element to the front of a list.

	Among its many uses, this function is useful for adding elements
	to LIFO lists and stacks. It also allows elements to be added to
	list being managed by index, without disturbing the index
	optimizations that making modifications by reference causes.
	
	This function is not available for pivot class lists. If the list
	is currently sorted, the sorted indexes will be voided, and the
	list_sorted_xx() functions are no longer available for the list,
	until it is resorted.

	list		list_object that has been previously initialized.
	element		pointer to caller supplied datum object to be stored
		 		as the new element in the list.
	returns	LIST_OK				success, new element added to the list
			ELEMENT_NO_DATUM	element data not supplied
			LIST_ERR			on error	 										*/
int list_prepend(	list_object *restrict list,
					const void *restrict element);

/* Add an element to a list, in the position indicated by the index.
	
	This function is not available for pivot class lists. If the list
	is currently sorted, the sorted indexes will be voided, and the
	list_sorted_xx() functions are no longer available for the list,
	until it is resorted.

	list		list_object that has been previously initialized.
	element		pointer to caller supplied datum object to be stored
		 		as the new element in the list.
	index		the list index number of the location where to add the
				new element.
	returns	LIST_OK				success, new element added to the list
			ELEMENT_NO_DATUM	element data not supplied
			ELEMENT_NOT_FOUND	supplied index was not a valid in the list
			LIST_ERR			on error	 										*/
int list_insert_index(	list_object *restrict list,
						const void *element,
						const unsigned int index);

/* Add an element to a list, in the position indicated by the element reference.

	The new element will be inserted in the list at a position higher than the
	reference. If stated from head to tail, the new element will be on the tail
	side of the reference. If stated from left to right, it will be on the right
	side of the referenced element.
	
	It is not possible to add a new element to the head of the list with this
	function. For that you should use the function list_prepend() instead. It
	is possible to add an element to the tail of the list, assuming you already
	have the reference to the last element available. You can add the element
	as the #2 in the list, by using the #1 as the reference.
	
	This function can be used while the list has iteration enabled. It is the
	only function available to add elements during an iteration.
	
	This function is not available for pivot class lists, or finite class lists.
	If the list is currently sorted, the sorted indexes will be voided, and the
	list_sorted_xx() functions are no longer available for the list, until it
	is resorted.

	list		list_object that has been previously initialized.
	element		pointer to caller supplied datum object to be stored
		 		as the list element.
	reference	caller supplied element reference, where to insert the
				new element in the list.
	returns	LIST_OK				success, additionally reference will be updated
								to the reference of the newly added list element.
			ELEMENT_NO_DATUM	element data not supplied
			ELEMENT_INVALID		supplied reference was not a valid element in list
			LIST_ERR			on error											*/
int list_insert_ref(	list_object *restrict list,
						const void *element,
						element_reference **restrict reference);

/* Add an element to a list, in the position computed by the list pivot function.

	The new element will be inserted in the list at a position that is determined
	by the result of calling the pivot function set for the list, with the
	element as input. The element will then be placed either the minor or major
	side of the pivot point, depending on the computed result.
	
	Additionally the position of the new element can be adjusted by the contents
	of the where argument. If HEAD_OF_LIST is provided, the new element is placed
	at the end of the list, the same as a list_append() or list_prepend() would
	do. If PIVOT_OF_LIST is provided, the new element is instead placed on the
	major or minor side of the pivot point.
	
	Using the following example, if this sorted list is added to a new pivot list
	
		a  b  c  d  e  f  g  h  i  j  k  l  m  n
	
	By using PIVOT_OF_LIST as the location, after all elements were added, the new
	list would look like :-
	
		a  b  c  d  e  f  g  n  m  l  k  j  i  h
					<------  ------>
	if instead it was added using END_OF_LIST, and the pivot point was <=g, the
	list would now look like :-
	
		g  f  e  d  c  b  a  h  i  j  k  l  m  n
		 ------>						<------
	The choice depends on the users needs, in particular END_OF_LIST will always
	cause the most recently added elements to be near the ends of the list, which
	can dramatically speed up searches of huge lists, if the likely candidate for
	a match is often more recent, than older.
		
	It is important to remember that pivot lists are not meant to retain ordering
	of elements, or be a naturally sorted list, it is a means of organizing new
	elements around a common datum used as a pivot point.
		
	This function is not available for linear or finite class lists. Use any of
	the other available insertion functions for those lists. A pivot function must
	have been set for the list, prior to calling list_store().

	list		list_object that has been previously initialized.
	element		pointer to caller supplied datum object to be stored
		 		as the new element in list.
	where		denote where to insert the new element, insertion will always
				be on the major or minor side of the pivot point, as decided
				by the pivot function.
				PIVOT_OF_LIST insert at the pivot of the list, or
				END_OF_LIST insert at the end of the list.
	returns	LIST_OK				success, new element added to the list
			ELEMENT_NO_DATUM	element data not supplied
			LIST_ERR			on error	 										*/
int list_store(	list_object *restrict list,
				const void *element,
				const boolean where);



/*	=====================================================================
		Section - Element Relocation Functions
	=====================================================================	*/

/* Copy an element from one list to another, given the index position.
	
	This function locates an element from the input list, and copies it to
	the destination list. Element is identified by index position in the
	input list, and either appended to a linear class destination list, or
	stored by pivot value to a pivot class list.
	
	If the destination list was initialized with a LIST_COPY_DATA attribute,
	the new element will copied in whole. If the list has no LIST_COPY_DATA
	attribute, only a copy of the element pointer will be added to the
	destination list. The user should remember there is now an additional
	pointer to it, that must be considered prior to any resource removal
	actions.
	
	The destination list must not be a Finite class list.

	list		list_object that has been previously initialized.
	index		denotes where in the input list the element is to be found.
	dest_list	list_object that has been previously initialized, where the
				copied element is to be added.
	returns	LIST_OK				success, the new element was added to the
								destination list.
			ELEMENT_NOT_FOUND	no element was found matching the index.
			LIST_ERR			on error 											*/
int list_copy_index(	list_object *restrict list,
						const unsigned int index,
						list_object *restrict dest_list);

/* Move an element from one list to another, given the index position.
	
	This function extracts an element from the input list, and inserts it to
	the destination list. Element is identified by index position in the
	input list, and either appended to a linear class destination list, or
	stored by pivot value to a pivot class list.
	
	This function is similar to list_copy_index() except the original
	element is deleted from the input list.
	
	If the destination list was initialized with a LIST_COPY_DATA attribute,
	the new element will copied in whole. If the list has no LIST_COPY_DATA
	attribute, only a copy of the element pointer will be added to the
	destination list. However there will not be an additional pointer to it,
	as the original element from the input list has been deleted. Element
	references are tied to the original element and list, and will not be
	valid after this move operation.
	
	The destination list must not be a Finite class list.
	
	This function removes the element from the input list, and releases its
	internal resources, along with the data resources if the input list was
	initialized with the LIST_COPY_DATA attribute.

	list		list_object that has been previously initialized.
	index		denotes where in the input list the element is to be found.
	dest_list	list_object that has been previously initialized, where the
				moved element is to be added.
	returns	LIST_OK				success, the new element was added to the
								destination list and then removed from the
								input list.
			ELEMENT_NOT_FOUND	no element was found matching the index.
			LIST_ERR			on error 											*/
int list_move_index(	list_object *restrict list,
						const unsigned int index,
						list_object *restrict dest_list);

/* Copy an element from one list to another, given the element reference.
	
	This function copies the element from the list identified by element
	reference, and adds it to the destination list. The new element is either
	appended to a linear class destination list, or stored by pivot value to
	a pivot class list.
	
	It is functionally similar to list_copy_index(), except that the element
	is identified by reference, not by index. A reference must have
	been passed to the caller previously by D-List from another call, such as
	list_search(). Upon return the original element reference is still valid.
	
	If the destination list was initialized with a LIST_COPY_DATA attribute,
	the new element will copied in whole. If the list has no LIST_COPY_DATA
	attribute, only a copy of the element pointer will be added to the
	destination list. The user should remember there is now an additional
	pointer to it, that must be considered prior to any resource removal
	actions.

	The destination list must not be a Finite class list.
	
	This function only copies the element and makes no changes to the input
	list.

	list		list_object that has been previously initialized.
	reference	an abstract locator in list, of the element to be copied.
	dest_list	list_object that has been previously initialized, where the
				copied element is to be added.
	returns	LIST_OK				success, the new element was added to the
								destination list.
			ELEMENT_INVALID		if the element reference was invalid
			LIST_ERR			on error 											*/
int list_copy_ref(	list_object *restrict list,
					element_reference *restrict reference,
					list_object *restrict dest_list);

/* Move an element from one list to another, given the element reference.
	
	This function extracts an element from the input list identified by its
	element reference, and then adds it to the destination list. The new
	element is either appended to a linear class destination list, or stored
	by pivot value to a pivot class list.
	
	It is functionally similar to list_copy_ref(), except the original
	element is deleted from the input list.
	
	A reference must have been passed to the caller previously by D-List from
	another call, such as list_search(). Upon return the original element
	reference is no longer valid and is changed to NULL.
	
	If the destination list was initialized with a LIST_COPY_DATA attribute,
	the new element will copied in whole. If the list has no LIST_COPY_DATA
	attribute, only a copy of the element pointer will be added to the
	destination list. However there will not be an additional pointer to it,
	as the original element from the input list has been deleted. Element
	references are tied to the original element and list, and will not be
	valid after this move operation.
	
	The destination list must not be a Finite class list.
	
	This function removes the element from the input list, and releases its
	internal resources, along with the data resources if the input list was
	initialized with the LIST_COPY_DATA attribute.

	list		list_object that has been previously initialized.
	reference	an abstract locator in list, of the element to be moved.
	dest_list	list_object that has been previously initialized, where the
				moved element is to be added.
	returns	LIST_OK				success, the new element was added to the
								destination list and then removed from the
								input list. If the list was created with the
								LIST_COPY_DATA attribute, the data that was
								allocated internally by D-List has now been
								released.
			ELEMENT_INVALID		if the element reference was invalid
			LIST_ERR			on error 											*/
int list_move_ref(	list_object *restrict list,
					element_reference **restrict reference,
					list_object *restrict dest_list);

/* Search for, and copy, all elements that match the supplied datum, from
	one list to another.
	
	This function will traverse the input list from head to tail, searching
	for all elements with the matching datum supplied. For each element
	found this function copies the element from the input list, and adds it
	to the destination list. The new element is either appended to a linear
	class destination list, or stored by pivot value to a pivot class list.

	This is a simple version of search, the entire input list is traversed,
	no consideration of pivots, or direction are provided.
	
	If a search function was previously set for this input list (by calling
	list_set_search_function() ), this function will use that search function
	to compare the element data against the supplied datum.
	
	If a search function was not previously set for this list, or it was
	removed, this function will perform a straight compare between the data
	pointer in the elements and the pointer provided in datum. If the element
	pointers are the same, they are equal, if the pointers point to different
	element data areas, they are not equal.
	
	In most cases, the user will want to set a search function first. But this
	can be used to find a data element by a known pointer reference.
	
	If the destination list was initialized with a LIST_COPY_DATA attribute,
	the new element will copied in whole. If the list has no LIST_COPY_DATA
	attribute, only a copy of the element pointer will be added to the
	destination list. The user should remember there is now an additional
	pointer to it, that must be considered prior to any resource removal
	actions.

	The destination list must not be a Finite class list.
	
	This function only copies the elements and makes no changes to the input
	list.

	list		list_object that has been previously initialized.
	datum 		data to be used for the search criteria.
	dest_list	list_object that has been previously initialized, where the
				copied element is to be added.
	returns	>0					the number of elements found in the input
								list, that were added to the destination list.
			0					if no elements matched the datum
			ELEMENT_NO_DATUM	element data not supplied
			LIST_ERR			on error 											*/
int list_search_copy(	list_object *restrict list,
						const void *datum,
						list_object *restrict dest_list);

/* Search for, and move, all elements that match the supplied datum, from
	one list to another.
	
	This function will traverse the input list from head to tail, searching
	for all elements with the matching datum supplied. For each element
	found this function copies the element from the input list, and adds it
	to the destination list, then deletes it from the input list. The new
	element is either appended to a linear class destination list, or stored
	by pivot value to a pivot class list.

	This is a simple version of search, the entire input list is traversed,
	no consideration of pivots, or direction are provided.
	
	If a search function was previously set for this input list (by calling
	list_set_search_function() ), this function will use that search function
	to compare the element data against the supplied datum.
	
	If a search function was not previously set for this list, or it was
	removed, this function will perform a straight compare between the data
	pointer in the elements and the pointer provided in datum. If the element
	pointers are the same, they are equal, if the pointers point to different
	element data areas, they are not equal.
	
	In most cases, the user will want to set a search function first. But this
	can be used to find a data element by a known pointer reference.
	
	If the destination list was initialized with a LIST_COPY_DATA attribute,
	the new element will copied in whole. If the list has no LIST_COPY_DATA
	attribute, only a copy of the element pointer will be added to the
	destination list. However there will not be an additional pointer to it,
	as the original element from the input list has been deleted. Element
	references are tied to the original element and list, and will not be
	valid after this move operation.
	
	The destination list must not be a Finite class list.
	
	This function removes all the elements from the input list, and releases
	their internal resources, along with the data resources if the input list
	was initialized with the LIST_COPY_DATA attribute.

	list		list_object that has been previously initialized.
	datum 		data to be used for the search criteria.
	dest_list	list_object that has been previously initialized, where the
				copied element is to be added.
	returns	>0					the number of elements found in the input
								list, that were added to the destination list.
			0					if no elements matched the datum
			ELEMENT_NO_DATUM	element data not supplied
			LIST_ERR			on error 											*/
int list_search_move(	list_object *restrict list,
						const void *datum,
						list_object *restrict dest_list);

/* Search for, and copy, all elements that evaluate true against the supplied
	datum, from one list to another.
	
	This function will traverse the input list from head to tail, evaluating
	all elements against the matching datum supplied. For each element in the
	list that evaluates true, this function copies the element from the input
	list, and adds it to the destination list. The new element is either
	appended to a linear class destination list, or stored by pivot value to
	a pivot class list.

	This is a simple version of search / compare, the entire input list is
	traversed, no consideration of pivots, or direction are provided.
	
	A compare function must have been previously set for the input list, by
	calling list_set_compare_function(). Each element of the input list is
	presented to that compare function, along with the user supplied datum.
	If the result of the evaluation is that the element is comparatively
	greater than or equal to the datum, and LIST_ASCENDING is supplied for
	the argument order, then the evaluation is considered TRUE and that
	element is then copied to the destination list. Conversely if the element
	is comparatively lesser than the datum, and LIST_DESCENDING is supplied
	for the argument order, then the evaluation is considered TRUE and that
	element is then copied to the destination list. Otherwise the element is
	ignored and the next element evaluated.
	
	If the destination list was initialized with a LIST_COPY_DATA attribute,
	the new element will copied in whole. If the list has no LIST_COPY_DATA
	attribute, only a copy of the element pointer will be added to the
	destination list. The user should remember there is now an additional
	pointer to it, that must be considered prior to any resource removal
	actions.

	The destination list must not be a Finite class list.
	
	This function only copies the elements and makes no changes to the input
	list.

	list		list_object that has been previously initialized.
	datum 		data to be used for the search criteria.
	order		either LIST_ASCENDING or LIST_DESCENDING to denote which order
				to evaluate the elements in the list against the datum.
	dest_list	list_object that has been previously initialized, where the
				copied element is to be added.
	returns	>0					the number of elements found in the input
								list, that were added to the destination list.
			0					if no elements matched the datum
			ELEMENT_NO_DATUM	element data not supplied
			LIST_ERR			on error 											*/
int list_evaluate_copy(	list_object *restrict list,
						const void *datum,
						const boolean order,
						list_object *restrict dest_list);

/* Search for, and move, all elements that evaluate true against the supplied
	datum, from one list to another.
	
	This function will traverse the input list from head to tail, evaluating
	all elements against the matching datum supplied. For each element in the
	list that evaluates true, this function copies the element from the input
	list, and adds it to the destination list and then deletes it from the
	input list. The new element will be either appended to a linear class
	destination list, or stored by pivot value to a pivot class list.

	This is a simple version of search / compare, the entire input list is
	traversed, no consideration of pivots, or direction are provided.
	
	A compare function must have been previously set for the input list, by
	calling list_set_compare_function(). Each element of the input list is
	presented to that compare function, along with the user supplied datum.
	If the result of the evaluation is that the element is comparatively
	greater than or equal to the datum, and LIST_ASCENDING is supplied for
	the argument order, then the evaluation is considered TRUE and that
	element is then copied to the destination list. Conversely if the element
	is comparatively lesser than the datum, and LIST_DESCENDING is supplied
	for the argument order, then the evaluation is considered TRUE and that
	element is then copied to the destination list. Otherwise the element is
	ignored and the next element evaluated. All elements that evaluate TRUE
	will be removed from the input list.
	
	If the destination list was initialized with a LIST_COPY_DATA attribute,
	the new element will copied in whole. If the list has no LIST_COPY_DATA
	attribute, only a copy of the element pointer will be added to the
	destination list. However there will not be an additional pointer to it,
	as the original element from the input list has been deleted. Element
	references are tied to the original element and list, and will not be
	valid after this move operation.
	
	The destination list must not be a Finite class list.
	
	This function removes all the elements from the input list, and releases
	their internal resources, along with the data resources if the input list
	was initialized with the LIST_COPY_DATA attribute.

	list		list_object that has been previously initialized.
	datum 		data to be used for the search criteria.
	order		either LIST_ASCENDING or LIST_DESCENDING to denote which order
				to evaluate the elements in the list against the datum.
	dest_list	list_object that has been previously initialized, where the
				moved element is to be added.
	returns	>0					the number of elements removed from the input
								list, and were added to the destination list.
			0					if no elements matched the datum
			ELEMENT_NO_DATUM	element data not supplied
			LIST_ERR			on error 											*/
int list_evaluate_move(	list_object *restrict list,
						const void *datum,
						const boolean order,
						list_object *restrict dest_list);

/* Search for, and copy, all elements that evaluate true within a minimum and
	maximum range of user datums, from one list to another.
	
	This function will traverse the input list from head to tail, evaluating
	all elements to check if they evaluate within a minimum and maximum range
	defined by comparing with the supplied datums. For each element in the
	list that evaluates within the datum range, this function copies the
	element from the input list, and adds it to the destination list. The new
	element is either appended to a linear class destination list, or stored
	by pivot value to a pivot class list.

	This is a simple version of search / compare, the entire input list is
	traversed, no consideration of pivots, or direction are provided.
	
	A compare function must have been previously set for the input list, by
	calling list_set_compare_function(). Each element of the input list is
	presented twice to that compare function, along with each of the user
	supplied datums.

	If the result of the evaluation is that the element is comparatively
	greater than or equal to the datum_min, and is also comparatively lesser
	than or equal the datum_max, then the evaluation is considered TRUE and
	that element is then copied to the destination list. Otherwise the element
	is ignored and the next element evaluated.
	
	If the destination list was initialized with a LIST_COPY_DATA attribute,
	the new element will copied in whole. If the list has no LIST_COPY_DATA
	attribute, only a copy of the element pointer will be added to the
	destination list. The user should remember there is now an additional
	pointer to it, that must be considered prior to any resource removal
	actions.

	The destination list must not be a Finite class list.
	
	This function only copies the elements and makes no changes to the input
	list.

	list		list_object that has been previously initialized.
	datum_min	data to be used for the minimum value of the search criteria.
	datum_max	data to be used for the maximum value of the search criteria.
	dest_list	list_object that has been previously initialized, where the
				copied element is to be added.
	returns	>0					the number of elements found in the input
								list, that were added to the destination list.
			0					if no elements matched the datum
			ELEMENT_NO_DATUM	element data not supplied
			LIST_ERR			on error 											*/
int list_range_copy(	list_object *restrict list,
						const void *datum_min,
						const void *datum_max,
						list_object *restrict dest_list);

/* Search for, and move, all elements that evaluate true within a minimum and
	maximum range of user datums, from one list to another.
	
	This function will traverse the input list from head to tail, evaluating
	all elements to check if they evaluate within a minimum and maximum range
	defined by comparing with the supplied datums. For each element in the
	list that evaluates within the datum range, this function removes the
	element from the input list, and adds it to the destination list. The new
	element is either appended to a linear class destination list, or stored
	by pivot value to a pivot class list.

	This is a simple version of search / compare, the entire input list is
	traversed, no consideration of pivots, or direction are provided.
	
	A compare function must have been previously set for the input list, by
	calling list_set_compare_function(). Each element of the input list is
	presented twice to that compare function, along with each of the user
	supplied datums.

	If the result of the evaluation is that the element is comparatively
	greater than or equal to the datum_min, and is also comparatively lesser
	than or equal the datum_max, then the evaluation is considered TRUE and
	that element is then moved to the destination list. Otherwise the element
	is ignored and the next element evaluated.
	
	If the destination list was initialized with a LIST_COPY_DATA attribute,
	the new element will copied in whole. If the list has no LIST_COPY_DATA
	attribute, only a copy of the element pointer will be added to the
	destination list. However there will not be an additional pointer to it,
	as the original element from the input list has been deleted. Element
	references are tied to the original element and list, and will not be
	valid after this move operation.
	
	The destination list must not be a Finite class list.
	
	This function removes the element from the input list, and releases its
	internal resources, along with the data resources if the input list was
	initialized with the LIST_COPY_DATA attribute.


	list		list_object that has been previously initialized.
	datum_min	data to be used for the minimum value of the search criteria.
	datum_max	data to be used for the maximum value of the search criteria.
	dest_list	list_object that has been previously initialized, where the
				moved element is to be added.
	returns	>0					the number of elements found in the input
								list, that were moved to the destination list.
			0					if no elements matched the datum
			ELEMENT_NO_DATUM	element data not supplied
			LIST_ERR			on error 											*/
int list_range_move(	list_object *restrict list,
						const void *datum_min,
						const void *datum_max,
						list_object *restrict dest_list);



/*	=====================================================================
		Section - Element Removal Functions
	=====================================================================	*/

/* Traverse a list and remove all elements from the list.

	If the list was created with the LIST_COPY_DATA attribute set
	TRUE, this function will also release all the data elements from
	the list. Otherwise it is the users responsibility to manage
	those data objects.

	list		pointer to caller supplied list_object
	returns	The number of elements removed from the list, or
			LIST_ERR		on error.												*/
int list_clear(list_object *restrict list);

/* Extract an element from the front (head), or the tail (end), of a list.

	Among its many uses, this function is useful for managing LIFO lists,
	heaps, and stacks, and FIFO's and queues. It also allows elements to
	be removed from a list being managed by index, without disturbing the
	index optimizations that making modifications by reference causes.
	
	This function removes the element from the list, and releases its
	internal resources.

	list		list_object that has been previously initialized.
	where		either HEAD_OF_LIST or TAIL_OF_LIST, denotes where in the
		 		list the element is to be extracted from.
	returns	NULL	no elements available in list
			PTR		to element data. It is the users responsibility
					to manage this data, and free it when finished with
					it. If the user supplied it originally it is the
					same allocated memory originally given to D-list.
					If the list was created with the LIST_COPY_DATA
					attribute, the data was allocated internally by
					D-list. In both cases it now belongs to the caller.				*/
void *list_fetch(	list_object *restrict list,
					const boolean where);

/* Extract an element from the pivot point, of a pivot class list.

	This function is very similar to list_fetch() except that instead
	of extracting the element from the end of any class of list,
	list_fetch_pivot() will only extract an element from the pivot
	point of a pivot list. The value of the argument where, dictates
	Which side of the pivot point the element is removed from.
	
	This function is only valid for pivot class lists.
	
	Among its many uses, this function is useful for managing complex
	double LIFO lists, heaps, and stacks, and FIFO's and folding queues.
	It also allows elements to be removed from a list being managed by
	index, without disturbing the index optimizations that making
	modifications by reference causes. This function is also helpful in
	reading the results of a list_assay() call.
	
	This function removes the element from the list, and releases its
	internal resources.

	list		list_object that has been previously initialized.
	where		either PIVOT_MINOR or PIVOT_MAJOR denotes which side of
				the pivot point, of the list, the element is to be
				extracted from.
	returns	NULL	no elements available in list on the pivot side
					requested
			PTR		to element data. It is the users responsibility
					to manage this data, and free it when finished with
					it. If the user supplied it originally it is the
					same allocated memory originally given to D-list.
					If the list was created with the LIST_COPY_DATA
					attribute, the data was allocated internally by
					D-list. In both cases it now belongs to the caller.				*/
void *list_fetch_pivot(	list_object *restrict list,
						const int where);

/* Extract an element from a list, given the index position.
	
	This function removes the element from the list, and releases its
	internal resources.

	list		list_object that has been previously initialized.
	index		denotes where in the list the element is to be extracted from.
	returns		NULL no element was available in list
				PTR to element data. It is the users responsibility to manage
					this data, and free it when finished with it. If the user
					supplied it originally it is the same allocated memory
					originally given to D-List. If the list was created with
					the LIST_COPY_DATA attribute, the data was allocated
					internally by D-List. In both cases it now belongs to the
					caller.															*/
void *list_extract_index(	list_object *restrict list,
							const unsigned int index);

/* Extract an element from a list, given the element reference.
	
	This function removes the element from the list, and releases its internal
	resources. It is functionally similar to list_extract_index(), except that
	the element is identified by reference, not by index. A reference must have
	been passed to the caller previously by D-List from another call, such as
	list_search().
	
	This function is only available for finite class lists during an active
	iteration sequence, otherwise calling this function on a finite class list
	will return an error code. If used the caller must only use references that
	were provided by list_iteration_get_next() during this active iteration
	sequence.

	list		list_object that has been previously initialized.
	reference	an abstract locator in list, of the element to be removed.
	returns		NULL the element reference was not valid, list unchanged
				PTR to element data. It is the users responsibility to manage
					this data, and free it when finished with it. If the user
					supplied it originally it is the same allocated memory
					originally given to D-List. If the list was created with
					the LIST_COPY_DATA attribute, the data was allocated
					internally by D-List. In both cases it now belongs to the
					caller.															*/
void *list_extract_ref(	list_object *restrict list,
						element_reference **restrict reference);

/* Delete an element from a list, given the index position.
	
	This function removes the element from the list, and releases its
	internal resources. It is functionally the same to list_extract_index(),
	except that no element is passed back to the caller, and if the list was
	created with the LIST_COPY_DATA attribute, the data is released as well.

	list		list_object that has been previously initialized.
	index		denotes the index number in the list, where the element
				is to be removed.
	returns	LIST_OK				the element was delete as requested
			ELEMENT_NOT_FOUND	no element was found matching the index.
			LIST_ERR			if there was an error.								*/
int list_delete_index(	list_object *restrict list,
						const unsigned int index);

/* Delete a block of elements from a list, between the indicated index positions.
	
	This function removes the elements in the index range from the list,
	and releases their resources. It is functionally the same to
	list_delete_index(), except that more than one element is deleted.

	list		list_object that has been previously initialized.
	index_start	denotes the first element to be removed.
	index_stop	denotes the last element to be removed, including all
				elements in-between.
	returns	LIST_OK				the element was delete as requested
			ELEMENT_NOT_FOUND	no element was found matching the index.
			LIST_ERR			if there was an error.								*/
int list_delete_group(	list_object *restrict list,
						const unsigned int index_start,
						const unsigned int index_stop);

/* Delete an element from a list, given the element reference.
	
	This function removes the element from the list, and releases its internal
	resources. It is functionally similar to list_delete_index(), except that
	the element is identified by reference, not by index. A reference must have
	been passed to the caller previously by D-List from another call, such as
	list_search().
	
	This function is only available for finite class lists during an active
	iteration sequence, otherwise calling this function on a finite class list
	will return an error code. If used the caller must only use references that
	were provided by list_iteration_get_next() during this active iteration
	sequence.

	list		list_object that has been previously initialized.
	reference	an abstract locator in list, of the element to be removed.
	returns	reference		on success, reference is set to NULL
			LIST_OK			the element was delete as requested,
			ELEMENT_INVALID	an invalid element reference was provided
			LIST_ERR		if there was an error.									*/
int list_delete_ref(	list_object *restrict list,
						element_reference **restrict reference);

/* Delete all elements from a list, that are duplicates of other elements in the list.
	
	This function will traverse a list from head to tail, and for each element
	found will search the rest of the list looking for and removing all duplicate
	elements that are found. This function does not change the order of any of
	the elements in the list. It does not perform any sorting of the elements.
	
	It will return a count of the elements removed during this process.

	If a search function was previously set for this list (by calling
	list_set_search_function() ), this function will use that search function
	to compare the element data against each other.
	
	If a search function was not previously set for this list, or it was
	removed, this function will perform a straight compare between the data
	pointer in the elements examined. If the element pointers are the same,
	they are equal, if the pointers point to different element data areas,
	they are not equal.
	
	In most cases, the user will want to set a search function first. But this
	can be used to eliminate duplicate data elements by pointer reference, very
	quickly.
	
	The definitive test, on return, to verify if any elements were removed, is 
	to check the return count for either 0 or a non-zero number.
	
	This function makes extensive changes to the list.

	list		list_object that has been previously initialized.
	returns	0				no duplicate elements were found
			>0				the number of duplicate elements removed,
			LIST_ERR		if there was an error.									*/
int list_delete_duplicates(list_object *restrict list);

/* Search for, and then extract, an element in a list, using datum as the
	search criteria.

	This function traverses a list from head to tail, searching for an element
	with the matching datum supplied. It will only find the first search match,
	and if found will delete the element from the list and return the element
	data to the caller.
	
	If a search function was previously set for this list (by calling
	list_set_search_function() ), this function will use that search function to
	compare the element data against the supplied datum.
	
	If a search function was not previously set for this list, or it was
	removed, this function will perform a straight compare between the data
	pointer in the elements and the pointer provided in datum. If the element
	pointers are the same, they are equal, if the pointers point to different
	element data areas, they are not equal.
	
	In most cases, the user will want to set a search function first. But this
	can be used to find a data element by a known pointer reference.
		
	list		list_object that has been previously initialized.
	datum 		data to be used for the search criteria.
	returns		NULL no element was found in list, matching the datum
				PTR to element data. It is the users responsibility to manage
					this data, and free it when finished with it. If the user
					supplied it originally it is the same allocated memory
					originally given to D-List. If the list was created with
					the LIST_COPY_DATA attribute, the data was allocated
					internally by D-List. In both cases it now belongs to the
					caller.															*/
void *list_search_extract(	list_object *restrict list,
							const void *datum);

/* Search for, and then delete, an element in a list, using datum as the
	search criteria.

	This function traverses a list from head to tail, searching for an element
	with the matching datum supplied. It will only find the first search match,
	and if found will delete the element from the list.
	
	If a search function was previously set for this list (by calling
	list_set_search_function() ), this function will use that search function to
	compare the element data against the supplied datum.
	
	If a search function was not previously set for this list, or it was
	removed, this function will perform a straight compare between the data
	pointer in the elements and the pointer provided in datum. If the element
	pointers are the same, they are equal, if the pointers point to different
	element data areas, they are not equal.
	
	In most cases, the user will want to set a search function first. But this
	can be used to find a data element by a known pointer reference.
		
	list		list_object that has been previously initialized.
	datum 		data to be used for the search criteria.
	returns	LIST_OK				if an element was found matching the datum,
								and was removed.
			ELEMENT_NO_DATUM	element data not supplied
			ELEMENT_NOT_FOUND	no element was found matching the datum
			LIST_ERR			if there was an error								*/
int list_search_delete(	list_object *restrict list,
						const void *datum);

/* Search for, and then delete, all occurrences of an element in a list, using
	datum as the search criteria.

	This function traverses a list from head to tail, searching for an element
	with the matching datum supplied. It will find all matching occurrences of
	the supplied datum, and when found will delete all the matching elements
	from the list. This function is similar to list_search_delete() except this
	function removes all instances of the datum from the list, instead one.
	
	If a search function was previously set for this list (by calling
	list_set_search_function() ), this function will use that search function to
	compare the element data against the supplied datum.
	
	If a search function was not previously set for this list, or it was
	removed, this function will perform a straight compare between the data
	pointer in the elements and the pointer provided in datum. If the element
	pointers are the same, they are equal, if the pointers point to different
	element data areas, they are not equal.
	
	In most cases, the user will want to set a search function first. But this
	can be used to find a data element by a known pointer reference.
		
	list		list_object that has been previously initialized.
	datum 		data to be used for the search criteria.
	returns	>0					number of elements found and removed matching
								the datum
			0					no element was found matching the datum
			ELEMENT_NO_DATUM	element data not supplied
			LIST_ERR			if there was an error								*/
int list_search_expunge(	list_object *restrict list,
							const void *datum);



/*	=====================================================================
		Section - Element Retrieval Functions
	=====================================================================	*/

/* Retrieve an element from a list, given the index position.
	
	This function only returns a pointer to the element in the list,
	it makes no changes to the list.

	list		list_object that has been previously initialized.
	index		denotes where in the list the element is to be retrieved from.
	returns	NULL			no element was available in list
			PTR				to element data. 										*/
void *list_get_index(	const list_object *restrict list,
						const unsigned int index);

/* Retrieve an element from a list, given the index position, and also
	return the element reference.
	
	This function retrieves an element from the list, using the supplied
	index, and is functionally identical to list_get_index(), except that the
	element reference is returned as well.

	The element reference is an abstract identifier for an element in a list,
	that does not change if the list is modified. An index position is only
	valid until the list changes, as an elements relative position will be
	changed by insertions or deletions. The element reference is not related
	to position in a list.

	The reference remains valid until the element itself is removed from the
	list. Reference is a required argument, and will be modified by this
	function.
	
	This function returns a pointer to the element in the list, along with its
	element reference. It makes no changes to the list.

	list		list_object that has been previously initialized.
	index		denotes where in the list the element is to be retrieved from.
	reference 	the abstract locator of the element in the list, which is 
				provided in this argument when the element is located. If the
				element cannot be found, NULL is returned.
	returns	NULL			no element was available in list
			PTR				to element data. 										*/
void *list_get_index_ref(	const list_object *restrict list,
							const unsigned int index,
							element_reference **restrict reference);


/* Retrieve an element from a list, given the element reference.
	
	This function retrieves an element from the list, using the supplied
	element reference. It is functionally similar to list_get_index(), except
	that the element is identified by reference, and not index. A reference
	must have been passed to the caller previously by D-List from another call,
	such as list_search().
	
	This function is not available for finite class lists.

	This function only returns a pointer to the element in the list,
	it makes no changes to the list.

	list		list_object that has been previously initialized.
	reference	an abstract locator in list, of the element to be retrieved.
	returns	NULL			no element was available in list (an error)
			PTR				to element data. 										*/
void *list_get_ref(	const list_object *restrict list,
					const element_reference *restrict reference);

/* Retrieve the next element from a list, given the element reference.
	
	This function retrieves an element from the list, using the supplied
	element reference. It is functionally similar to list_get_index(), except
	that the element is identified by reference, and not index. A reference
	must have been passed to the caller previously by D-List from another call,
	such as list_search() or list_get_next_ref(). Or a reference of NULL can
	be provided to start at the end of the list and traverse in the direction
	specified. If the list is a pivot class list, the state of the output
	argument pivot_point is set to TRUE if this new record comes from the
	other side of the the conceptual pivot point (as enforced by the set
	pivot function for the list). Otherwise it is set to FALSE on return.

	Upon exit the reference, and the pivot_point arguments are always updated,
	and the original contents replaced.
	
	This function can also be used in a similar way to list_iteration_next(),
	and is vastly more efficient than performing an index loop using
	list_get_index() which is very inefficient when dealing with large lists.
	Unlike an iteration sequence, it does not lock the list from modifications,
	and allows you to traverse portions of the list starting at an arbitrary
	point, and allows you to change direction at will.
	
	This function is not available for finite class lists.
	
	This function only returns a pointer to the element in the list,
	it makes no changes to the list.

	list		list_object that has been previously initialized.
	reference 	an abstract locator in list, of the element next to the
		 		element to be retrieved. NULL means start from the head or
		 		tail of the list, depending on direction
	direction	HEAD_OF_LIST towards the head of the list, or TAIL_OF_LIST
				the opposite direction.
	returns	NULL			no element was available in list, end of list,
							or error. reference will also be set to NULL
			PTR				to element data. Reference will be updated to
							this element
			pivot_point		if this next record crosses the pivot point,
							this return argument is set to TRUE.					*/
void *list_get_next_ref(	const list_object *restrict list,
							element_reference **restrict reference,
							const boolean direction,
							boolean *restrict pivot_point);

/* Locate and return an element index in a list, identified by its element
	datum pointer.

	This function will locate the element in this list by searching for a
	match using the datum pointer.
	
	This function ignores any set search function., this function will only
	perform a straight compare between the data pointer in the elements and
	the pointer provided in datum. If the element pointers are the same,
	they are equal, if the pointers point to different element data areas,
	they are not equal.

	If you wish to perform a real data search of the list, please use one of
	the many list_search_xx() function calls. This function is only to get the
	index of an element for which you already know the data pointer, such as
	was returned from a list_simple_search(), or retained before the element
	was added to the list originally.

	This function returns an index and makes no changes to the list.

	list		list_object that has been previously initialized.
	datum 		data pointer to be used for the pointer search.
	returns	>=0				element index for the element datum requested
			LIST_ERR			if there was an error or
			ELEMENT_NO_DATUM	element data not supplied
			ELEMENT_NOT_FOUND	no match found.										*/
long int list_locate_index(	const list_object *restrict list,
							const void *datum);

/* Locate and return an element pivot value in a list, identified by its element
	datum pointer.

	This function will locate the element in this list by searching for a
	match using the datum pointer.
	
	This function ignores any set search function., this function will only
	perform a straight compare between the data pointer in the elements and
	the pointer provided in datum. If the element pointers are the same,
	they are equal, if the pointers point to different element data areas,
	they are not equal.

	If you wish to perform a real data search of the list, please use one of
	the many list_search_xx() function calls. This function is only to get the
	pivot value of an element for which you already know the data pointer, such
	as was returned from a list_simple_search(), or retained before the element
	was added to the list originally.

	This function returns a pivot value and makes no changes to the list.

	list		list_object that has been previously initialized.
	datum 		data pointer to be used for the pointer search.
	returns	PIVOT_MINOR		element datum requested is on the minor side
							of the pivot point in the list
			PIVOT_MAJOR		element datum requested is on the major side
							of the pivot point in the list
			ENTIRE_LIST		either this is not a pivot list, or the
							element was not found in the list.						*/
int list_locate_pivot(	const list_object *restrict list,
						const void *datum);

/* Locate and return an element reference in a list, identified by its
	element datum pointer.

	This function will locate the element in this list by searching for a
	match using the datum pointer.
	
	This function ignores any set search function., this function will only
	perform a straight compare between the data pointer in the elements and
	the pointer provided in datum. If the element pointers are the same,
	they are equal, if the pointers point to different element data areas,
	they are not equal.

	If you wish to perform a real data search of the list, please use one of
	the many list_search_xx() function calls. This function is only to get the
	reference of an element for which you already know the data pointer, such
	as was returned from a list_simple_search(), or retained before the element
	was added to the list originally.
		
	This function returns a reference and makes no changes to the list.

	list		list_object that has been previously initialized.
	datum 		data pointer to be used for the pointer search.
	returns	reference		for the element datum requested
			NULL			if there was an error or no match found.				*/
element_reference *list_locate_ref(	const list_object *restrict list,
									const void *datum);

/* Verify if an element reference is valid for the list.

	This function will return a TRUE or FALSE depending on whether the list
	element reference provided is a valid reference to an element in the
	list or not. This function call is particularly useful for finite lists
	to verify that a known element reference is still in the list, and has
	not been automatically dropped. This function is also useful for other
	list classes, when using some advanced list manipulation techniques.
	
	In general there is no reason to call this function, unless you have very
	specific reasons to feel the element reference could have become invalid
	since you last used it.
	
	If you call this function during development phase of your project coding
	with invalid element references, you will get various errors written to
	stderr, however the results of these calls will be valid. Once DEBUG is
	turned off, the error messages will no longer be present in the code, so
	the logic and outcome are the same, without the error log messages. However
	more validity checks are performed with DEBUG turned on, so for development
	purposes the error messages can be very helpful.

	This function only returns a boolean variable about the validity of the
	supplied element reference. It makes no changes to the list.

	list		list_object that has been previously initialized.
	reference	an abstract element reference to verify.
	returns	TRUE			if the element reference is valid for this list
			FALSE			if the element reference is not valid					*/
boolean list_valid_ref(	const list_object *restrict list,
						const element_reference *restrict reference);

/* Start a sequence of iteration of all the elements in a list.

	This function initiates an iteration sequence. The iteration sequence will
	continue until the function list_stop_iteration() is called on this list.
	
	An iteration sequence traverses a list from the head to the tail (or the
	opposite direction if that was requested), returning each element in sequence
	using list_iteration_next(). For each element a pointer to the element data
	and a list reference to the element are provided.
	
	During an iteration sequence all index based modifications to the list are
	blocked and denied. However index based gets and searches can occur. Also
	by design, elements can be fetched, inserted, removed and modified using the
	element references returned by various other D-List function calls. It is the
	responsibility of the user to use these features wisely. In particular unwise
	use of inserting an element by reference at a point following the current
	iteration sequence location, has the obvious potential to create an infinite
	loop situation, of adding a new element and iterating to it, and adding
	another, etc etc. Use these features carefully...
	
	A typical use of this iteration sequence is
	
		...
		list_iteration_start(a_list);
		while (list_iteration_has_more(a_list) {
			ptr = list_iteration_get_next(a_list, reference);
			do something with the ptr-> element
			or save or do something with the reference
			...
		}
		list_iteration_stop(a_list);
		...
	
	But there are also many other creative ways to use this, especially with
	nested sub-lists, and cross referencing. The user can operate multiple
	iteration sequences on multiple lists at the same time. But can only
	operate an iteration sequence on a specific list, one at a time.
	
	An iteration sequence can be started on any class of list. For a pivot
	class list the sequence can be controlled either to the other end of the
	list, or to the pivot point, depending on how the caller uses the
	list_iteration_has_more() function calls.
	
	list		list_object that has been previously initialized.
	direction	HEAD_OF_LIST towards the head of the list, or TAIL_OF_LIST
				towards the tail of the list.
	returns	LIST_OK			the iteration sequence has been established
			LIST_ERR		if there was an error.									*/
int list_iteration_start(	list_object *restrict list,
							const boolean direction);

/* Start a sequence of iteration of all the elements in a list on one side of
	a list pivot point.

	This function initiates an iteration sequence. The iteration sequence will
	continue until the function list_stop_iteration() is called on this list.
	The list must be a pivot class list.
	
	An iteration sequence traverses a list from the current pivot point of the
	list, to either the head or the tail depending on the direction that was
	requested. Each element can be returned in sequence using the function
	list_iteration_next(). For each element a pointer to the element data
	and a list reference to the element are provided.
	
	During an iteration sequence all index based modifications to the list are
	blocked and denied. However index based gets and searches can occur. Also
	by design, elements can be fetched, inserted, removed and modified using the
	element references returned by various other D-List function calls. It is the
	responsibility of the user to use these features wisely.
	
	An iteration sequence started from a pivot point, can only be initiated for
	a pivot class list. The caller must use the function list_iteration_has_more()
	to determine if there are additional elements to find in this sequence.
	
	list		list_object that has been previously initialized.
	direction	PIVOT_MINOR from the pivot point towards the head of the list
				(traversing all the minor elements), or PIVOT_MAJOR from the
				pivot point towards the tail of the list (traversing all the
				major elements).
	returns	LIST_OK			the iteration sequence has been established
			LIST_ERR		if there was an error.									*/
int list_iteration_start_pivot(	list_object *restrict list,
								const int direction);

/* Inquire if there are more elements to be found, during a sequence of
	iteration of all the elements in a list.

	This function requires an active iteration sequence. It returns TRUE if
	there are 1 or more elements left to iterate over in the list. It returns
	FALSE if there are no more left.
	
	For more details about iteration sequences, refer to list_start_iteration().
	
	list		list_object that has been previously initialized.
	returns	TRUE			there are more elements left in the iteration
							sequence
			FALSE			no more elements are remaining to iterate over			*/
boolean list_iteration_has_more(const list_object *restrict list);

/* Inquire if there are more elements to be found, during a sequence of
	iteration of all the elements in a list on one side of a pivot.

	This function requires an active iteration sequence. It returns TRUE if
	there are 1 or more elements left to iterate over in the list before
	reaching the pivot point of the list. It returns FALSE if there are no
	more elements left before the pivot point. You can still continue to
	call list_iteration_get_next() until list_iteration_has_more() returns
	FALSE. By using both list_iteration_has_more functions you can
	navigate lists in a variety of useful and non-standard ways.
	
	This functions main purpose is to support iteration sequences starting
	from the end of the list, where the user wants to know where the pivot
	point is. However, if the iteration sequence was started using the
	list_iteration_start_pivot() function instead, then this function is not
	needed as the iteration sequence will be moving away from the pivot
	point, not towards it. Under those circumstances it will operate the
	same as a call to list_iteration_has_more() does.
	
	This function can only be called for a pivot class list.
	
	For more details about iteration sequences, refer to list_start_iteration().
	
	list		list_object that has been previously initialized.
	returns	TRUE			there are more elements left in the iteration
							sequence
			FALSE			no more elements are remaining to iterate over			*/
boolean list_iteration_has_more_pivot(const list_object *restrict list);

/* Returns a pointer to the next element, during a sequence of iteration of
	all the elements in a list.

	This function requires an active iteration sequence for this list was
	initiated prior to this call.
	
	For more details about iteration sequences, refer to list_start_iteration().

	This function will return a pointer to the element data, and a list reference
	to the element. The user may choose to use any of these methods to reacquire,
	remove or manipulate the element, or data. The reference remains valid until
	the element itself is removed from the list. Reference is an optional argument
	and if supplied NULL, it will be ignored, otherwise it will be modified.

	list		list_object that has been previously initialized.
	returns	if another element was available in this iteration sequence, the
					pointer to the element and the optional element reference.
			if no more elements are available, a NULL pointer, and a NULL
					element reference		.										*/
void *list_iteration_get_next(	list_object *restrict list,
								element_reference **restrict reference);

/* Stop a sequence of iteration of all the elements in a list.

	This function terminates an iteration sequence. The iteration sequence was
	initiated by the function list_start_iteration().
	
	For more details about iteration sequences, refer to list_start_iteration().
		
	list		list_object that has been previously initialized.
	returns	LIST_OK			the iteration sequence has been stopped
			LIST_ERR		if there was an error.									*/
int list_iteration_stop(list_object *restrict list);



/*	=====================================================================
		Section - Element Search Functions
	=====================================================================	*/

/* Search for an element in a list, using datum as the search criteria.

	This function will traverse a list from tail to head, searching for an
	element with the matching datum supplied. It will only find the first
	search match, and if found will return a pointer to the element data.
	
	This is a simple version of search, the entire list is traversed, no
	consideration of pivots, or direction are provided.
	
	If a search function was previously set for this list (by calling
	list_set_search_function() ), this function will use that search function
	to compare the element data against the supplied datum.
	
	If a search function was not previously set for this list, or it was
	removed, this function will perform a straight compare between the data
	pointer in the elements and the pointer provided in datum. If the element
	pointers are the same, they are equal, if the pointers point to different
	element data areas, they are not equal.
	
	In most cases, the user will want to set a search function first. But this
	can be used to find a data element by a known pointer reference.
	
	The definitive test, on return, to verify if an element was found, is to
	check the return pointer for either NULL or an address.
	
	This function only returns a pointer to the element data in the list, it
	makes no changes to the list.

	list		list_object that has been previously initialized.
	datum 		data to be used for the search criteria.
	returns	PTR				a pointer to element data, if an element was
							found matching the datum,
			NULL			if no element was found matching the datum				*/
void *list_simple_search(	const list_object *restrict list,
							const void *datum);

/* Search a list for all occurrences of an element, using datum as the search
	criteria.

	This function will traverse a list from tail to head, searching for an
	element with the matching datum supplied. It will find all the elements
	that match the search criteria, and if found will return a count of the
	elements found.
	
	This is a simple version of search, the entire list is traversed, no
	consideration of pivots, or direction are provided.
	
	If a search function was previously set for this list (by calling
	list_set_search_function() ), this function will use that search function
	to compare the element data against the supplied datum.
	
	If a search function was not previously set for this list, or it was
	removed, this function will perform a straight compare between the data
	pointer in the elements and the pointer provided in datum. If the element
	pointers are the same, they are equal, if the pointers point to different
	element data areas, they are not equal.
	
	In most cases, the user will want to set a search function first. But this
	can be used to find a data element by a known pointer reference.
	
	The definitive test, on return, to verify if any element was found, is to
	check the return count for either 0 or a non-zero number.
	
	This function makes no changes to the list.

	list		list_object that has been previously initialized.
	datum 		data to be used for the search criteria.
	returns	>0				number of elements found matching the datum
			0				no element was found matching the datum
			LIST_ERR		if there was an error									*/
int list_occurrences(	const list_object *restrict list,
						const void *datum);

/* Search for an element in a list, using datum as the search criteria.

	This function will traverse a list constrained by the scope argument, in
	the direction specified, searching for an element with the matching datum
	supplied. It will only find the first search match, and if found will
	return a pointer to the element data, the index number in the list of the
	element, and a list reference to the element. The user may choose to use
	any of these methods to reacquire, remove or manipulate the element, or
	data. Please bear in mind that the index is only valid until the list is
	modified by adding or removing elements. The reference remains valid
	until the element itself is removed from the list.

	The argument scope determines how far along the list this function will
	traverse looking for a matching elek=ment. Providing ENTIRE_LIST as this
	argument will cause the function to traverse the list from head to tail.
	However, if the list is of a pivot class, then providing either PIVOT_MAJOR
	or PIVOT_MINOR will optionally cause this function to confine its search to
	that side of the list pivot point. If the list is of the linear class, then
	the setting of this argument is ignored. This feature is useful if the
	comparison function tests different data points, than the set pivot
	function. It allows for lists to be easily used in a multidimensional way
	with simple code.
	
	The direction argument is directly related to the contents of scope. If
	scope is ENTIRE_LIST, the direction defines from which to start the search.
	If the scope is pivot based, then direction defines whether to start from an
	end of the list, or from the pivot point.
	
	This function is available for finite class lists, with the exception that
	a reference will not be returned regardless of the reference argument.
	
	If a search function was previously set for this list (by calling
	list_set_search_function() ), this function will use that search function
	to compare the element data against the supplied datum.
	
	If a search function was not previously set for this list, or it was
	removed, this function will perform a straight compare between the data
	pointer in the elements and the pointer provided in datum. If the element
	pointers are the same, they are equal, if the pointers point to different
	element data areas, they are not equal.
	
	In most cases, the user will want to set a search function first. But this
	can be used to find a data element by a known pointer reference.
	
	The definitive test, on return, to verify if an element was found, is to
	check the return pointer for either NULL or an address.
	
	This function only returns a pointer, index and reference to the element
	in the list, it makes no changes to the list.

	list		list_object that has been previously initialized.
	datum 		data to be used for the search criteria.
	scope		for pivot lists only, which side of the pivot point to search
				for the datum. Must be either PIVOT_MINOR, PIVOR_MAJOR, or
				ENTIRE_LIST. For linear and finite lists, must be ENTIRE_LIST
	direction	If scope is ENTIRE_LIST, then this argument may contain
				either HEAD_OF_LIST, proceed from the tail of the list towards
				the head of the list, or TAIL_OF_LIST proceed from the head of
				the list towards the tail of the list.
				If this is a pivot class list, and scope is either of the PIVOT
				options, then this argument may contain either PIVOT_OF_LIST,
				search starts at the end of the list towards the pivot, or
				END_OF_LIST in which case the search starts from the pivot
				point and runs to the end of the list, the direction defined by
				the scope argument (PIVOT_MAJOR or PIVOT_MINOR)
	reference	pointer to variable used to store the element reference,
		 		if found. NULL, no element reference is to be returned.
	index		pointer to variable used to store the element index, if
		 		found. Or NULL, no element index is to be returned.
	returns	if an element was found matching the datum, a pointer to element,
				and if requested an index of element in list, and the element
				reference.
			if an element was not found matching the datum,
				NULL pointer, index of 0, and NULL element reference.				*/
void *list_search(	const list_object *restrict list,
					const void *datum,
					const int scope,
					const boolean direction,
					element_reference **restrict reference,
					unsigned int *index);

/* Search for an element in a pivot class list, using datum to decide which
	side of the pivot to search, and also as the search criteria.

	This function will traverse a list constrained by the scope returned by
	presenting the datum to the lists pivot function. Only that side of the
	pivot list will be searched.
	
	The function will then search for an element with the matching datum
	supplied. It will only find the first search match, and if found will
	return a pointer to the element data, the index number in the list of the
	element, if requested, and an optional list reference to the element. The
	user may choose to use any of these methods to reacquire, remove or later
	manipulate the element, or data. Please bear in mind that the index is
	only valid until the list is modified by adding or removing elements. The
	reference remains valid until the element itself is removed from the list.

	The specific scope of the search, is determined by presenting the datum to
	the current pivot function for the list. The direction argument is directly
	related to the results of obtaining the scope. It defines whether to start
	from one end of the list, or from the pivot point.
	
	This function is only available for pivot class lists. A pivot function must
	be set for this list.
	
	If a search function was previously set for this list (by calling
	list_set_search_function() ), this function will use that search function
	to compare the element data against the supplied datum.
	
	If a search function was not previously set for this list, or it was
	removed, this function will perform a straight compare between the data
	pointer in the elements and the pointer provided in datum. If the element
	pointers are the same, they are equal, if the pointers point to different
	element data areas, they are not equal.
	
	In most cases, the user will want to set a search function first. But this
	can be used to find a data element by a known pointer reference.
	
	The definitive test, on return, to verify if an element was found, is to
	check the return pointer for either NULL or an address.
	
	This function only returns a pointer, index, pivot value, and reference to
	the element in the list, it makes no changes to the list.

	list		list_object that has been previously initialized as a pivot
				class list.
	datum 		data to be used for the search criteria.
	direction	this argument describes which direction to perform the search.
				It may contain either PIVOT_OF_LIST, so the search starts at
				the end of the list towards the pivot, or END_OF_LIST the
				search starts from the pivot point and runs out to the end
				of the list. The specific scope is defined by the result of
				the pivot function on the datum argument (either PIVOT_MAJOR
				or PIVOT_MINOR)
	reference	pointer to variable used to store the element reference,
		 		if found. NULL, no element reference is to be returned.
	index		pointer to variable used to store the element index, if
		 		found. Or NULL, no element index is to be returned.
	pivot_value	pivot value of the element, if one was found. Or NULL, no
				element pivot value is to be returned.
	returns	if an element was found matching the datum, a pointer to element,
				and if requested an index of element in list, the pivot value,
				and the element reference.
			if an element was not found matching the datum, NULL pointer,
				pivot value of 0, index of 0, and NULL element reference.			*/
void *list_search_pivot(	const list_object *restrict list,
							const void *datum,
							const boolean direction,
							element_reference **restrict reference,
							unsigned int *index,
							int *restrict pivot_value);

/* Search for an element in a list, using datum as the search criteria, backwards
	or forwards, from the reference point indicated.

	This function traverses a list from the supplied reference element to the end,
	searching for an element with the matching datum supplied. It will only find
	the first search match, and if found will return a pointer to the element data,
	and a list reference to the element. The user may choose to use any of these
	methods to reacquire, remove or manipulate the element, or data. The reference
	remains valid until the element itself is removed from the list.
	
	An index cannot be returned from this search, the efficiency gains from using
	element references, would be obviated if the index had to be discovered as well.

	This function is not available for finite class lists.
	
	If a search function was previously set for this list (by calling
	list_set_search_function() ), this function will use that search function to
	compare the element data against the supplied datum.
	
	If a search function was not previously set for this list, or it was
	removed, this function will perform a straight compare between the data
	pointer in the elements and the pointer provided in datum. If the element
	pointers are the same, they are equal, if the pointers point to different
	element data areas, they are not equal.
	
	In most cases, the user will want to set a search function first. But this
	can be used to find a data element by a known pointer reference.
	
	The definitive test, on return, to verify if an element was found, is to check
	the return pointer for either NULL or an address. In all cases the argument
	'reference' is modified by calling this function, and the previous contents
	are lost.
	
	This function only returns a pointer and reference to the element
	in the list, it makes no changes to the list.

	list		list_object that has been previously initialized.
	datum 		data to be used for the search criteria.
	reference	caller supplied element reference to begin the reverse
				search. If reference is NULL, the search will be performed
				starting at the end of the list. Reference is always
				returned if an element is found.
	direction	HEAD_OF_LIST from the reference point, towards the head of
				the list, or TAIL_OF_LIST from the reference point towards
				the tail of the list.
	returns	if an element was found matching the datum, the pointer to
					the element and the element reference.
			if an element was not found matching the datum, a NULL pointer,
					a NULL element reference.										
			pivot_point		if this found element was retrieved from the
							other side of the pivot point, this return
							argument is set to TRUE. 								*/
void *list_search_ref(	const list_object *restrict list,
						const void *datum,
						element_reference **restrict reference,
						const boolean direction,
						boolean *restrict pivot_point);

/* Search for an element in a list, using datum as the search criteria, and
	if found, shift the element to the head of the list.

	This function will traverse a list from tail to head, searching for an
	element with the matching datum supplied. It will only find the first
	search match, and if found will relocate the element from its current
	position in the list, to the head of the list, and also will return a
	pointer to the element data.
	
	This is a simple version of search, the entire list is traversed, no
	consideration of pivots, or direction are provided. This function is not
	available for pivot class lists. This function is particularly useful if
	used with a finite class list, that is being used as a recent history
	cache, to keep the most recent elements at the head of the list, allowing
	the oldest elements to be dropped from the list as needed. This function
	performs a sequence of operations that are equivalent to list_search_ref(),
	list_delete_ref() and list_append(), but far more efficient for this more
	specific task, and without destroying the reference to the original data
	element.
	
	If a search function was previously set for this list (by calling
	list_set_search_function() ), this function will use that search function
	to compare the element data against the supplied datum.
	
	If a search function was not previously set for this list, or it was
	removed, this function will perform a straight compare between the data
	pointer in the elements and the pointer provided in datum. If the element
	pointers are the same, they are equal, if the pointers point to different
	element data areas, they are not equal.
	
	In most cases, the user will want to set a search function first. But this
	can be used to find a data element by a known pointer reference.

	If the list is currently sorted, the sorted indexes will be voided, and the
	list_sorted_xx() functions are no longer available for the list, until it
	is resorted.
	
	The definitive test, on return, to verify if an element was found, is to
	check the return pointer for either NULL or an address.
	
	This function not only returns a pointer to the element data in the list,
	it also makes changes to the list.
	
	list		list_object that has been previously initialized.
	datum 		data to be used for the search criteria.
	returns	PTR			if an element was found matching the datum
			NULL		if an element was not found matching the datum				*/
void *list_search_shift(	list_object *restrict list,
							const void *datum);

/* Search for an element in a list, using maximum quantity as the search criteria.

	This function traverses the list, for the distance specified in scope,
	searching for an element with the matching the notion of maximum. It will
	return a pointer to the element data, the index number in the list of the
	element, and a list reference to the element. The user may choose to use any
	of these methods to reacquire, remove or manipulate the element, or data.
	Please bear in mind that the index is only valid until the list is modified
	by adding or removing elements. The reference remains valid until the element
	itself is removed from the list.
	
	The argument scope determines how far along the list this function will
	traverse looking for a maximum quantity. Providing ENTIRE_LIST as this argument
	will cause the function to traverse the list from head to tail. However, if the
	list is of a pivot class, then providing either PIVOT_MAJOR or PIVOT_MINOR will
	optionally cause this function to confine its search to that side of the list
	pivot point. If the list is of the linear class, then the setting of this
	argument is ignored. This feature is useful if the comparison function tests
	different data points, than the set pivot function. It allows for lists to be
	easily used in a multidimensional way with simple code.
	
	A comparison function must have been previously set for this list (by calling
	list_set_compare_function() ), this function will use that compare function to
	compare the element data looking for the largest item. What constitutes maximum
	is implementation dependent for the compare function and the data stored in the
	list by the user.
	
	The definitive test, on return, to verify if an element was found, is to check
	the return pointer for either NULL or an address.
	
	If you wish to only find the maximum value of the list then this function is
	the most efficient method. However if you wish to find both the minimum and the
	maximum values of the list it is more efficient to first sort the list using
	either list_fast_sort() or list_sort() and then calling list_get_index(list, 0)
	to obtain the minimum value, and list_get_index(list, list_size(list)-1) to get
	the maximum value. That is assuming that it is not important if the list is
	reordered from its current state. Please refer to the max and max_alt modules
	in the performance section of the D-List distribution for examples.

	This function only returns a pointer, index and reference to the element
	in the list, it makes no changes to the list.

	list		list_object that has been previously initialized.
	scope		which side of the pivot to search for the maximum element
				must be either PIVOT_MINOR, PIVOR_MAJOR, or ENTIRE_LIST
				For linear and finite class lists, must be ENTIRE_LIST
	reference	pointer to variable used to store the element reference,
		 		if found. NULL, no element reference is to be returned.
	index		pointer to variable used to store the element index, if
		 		found. Or NULL, no element index is to be returned.
	returns	if an element was found, a pointer to element, and if requested an
					element index in list, and the element reference are returned
			if an element was not found (i.e. the list was empty), a NULL
					pointer, index of 0, and NULL element reference.				*/
void *list_locate_maximum(	const list_object *restrict list,
							const int scope,
							element_reference **restrict reference,
							unsigned int *index);

/* Search for an element in a list, using minimum quantity as the search criteria.

	This function traverses the list, for the distance specified in scope,
	searching for an element with the matching the notion of minimum. It will
	return a pointer to the element data, the index number in the list of the
	element, and a list reference to the element. The user may choose to use any
	of these methods to reacquire, remove or manipulate the element, or data.
	Please bear in mind that the index is only valid until the list is modified
	by adding or removing elements. The reference remains valid until the element
	itself is removed from the list.
	
	The argument scope determines how far along the list this function will
	traverse looking for a minimum quantity. Providing ENTIRE_LIST as this argument
	will cause the function to traverse the list from head to tail. However, if the
	list is of a pivot class, then providing either PIVOT_MAJOR or PIVOT_MINOR will
	optionally cause this function to confine its search to that side of the list
	pivot point. If the list is of the linear class, then the setting of this
	argument is ignored. This feature is useful if the comparison function tests
	different data points, than the set pivot function. It allows for lists to be
	easily used in a multidimensional way with simple code.
	
	The definitive test, on return, to verify if an element was found, is to check
	the return pointer for either NULL or an address.
	
	If you wish to only find the minimum value of the list then this function is
	the most efficient method. However if you wish to find both the minimum and the
	maximum values of the list it is more efficient to first sort the list using
	either list_fast_sort() or list_sort() and then calling list_get_index(list, 0)
	to obtain the minimum value, and list_get_index(list, list_size(list)-1) to get
	the maximum value. That is assuming that it is not important if the list is
	reordered from its current state. Please refer to the max and max_alt modules
	in the performance section of the D-List distribution for examples.

	This function only returns a pointer, index and reference to the element
	in the list, it makes no changes to the list.

	list		list_object that has been previously initialized.
	scope		which side of the pivot to search for the maximum element
				must be either PIVOT_MINOR, PIVOR_MAJOR, or ENTIRE_LIST
				For linear and finite class lists, must be ENTIRE_LIST
	reference	pointer to variable used to store the element reference,
		 		if found. NULL, no element reference is to be returned.
	index		pointer to variable used to store the element index, if
		 		found. Or NULL, no element index is to be returned.
	returns	if an element was found, a pointer to element, and if requested an
					element index in list, and the element reference are returned
			if an element was not found (i.e. the list was empty), a NULL
					pointer, index of 0, and NULL element reference.				*/
void *list_locate_minimum(	const list_object *restrict list,
							const int scope,
							element_reference **restrict reference,
							unsigned int *index);



/*	=====================================================================
		Section - Sorted List Element Functions
	=====================================================================	*/

/* Search for an element in an already sorted list, using datum as the search
	criteria.

	This function will perform a binary search of a sorted list, searching for
	an element with the matching datum supplied. It will only find the first
	search match in the sorted order of the list, and if found will return a
	pointer to the element data, and an optional list reference to the found
	element. The reference remains valid until the element itself is removed
	from the list. An optional index to the element can also be returned.

	This function is available for finite class lists, with the exception that
	a reference will not be returned regardless of the reference argument. 
	
	This function is available for pivot class lists, first the pivot function
	set for the list is called using the datum, to determine which side of the
	list to operate on, then the function proceeds constrained to that side of
	the pivot list. If called for a pivot list, a pivot function must have been
	previously set for this list.
	
	A compare function must have been previously set for this list (by calling
	list_set_compare_function() ), this find operation will use the set compare
	function to compare the element data against the supplied datum. It is also
	assumed that the compare function used to sort the list, ordered it using
	the same data points that the current compare function uses. If not, the
	results are undefined.
	
	This function is only available for lists that are currently sorted. The 
	definitive test, on return, to verify if an element was found, is to check 
	the return pointer for either NULL or an address.
	
	This function only returns a pointer, index and reference to the element
	in the list, it makes no changes to the list.

	list		list_object that has been previously initialized.
	datum 		data to be used for the search criteria.
	reference	pointer to variable used to store the element reference,
		 		if found. NULL, no element reference is to be returned.
	index		pointer to variable used to store the element index, if
 		 		found. Or NULL, no element index is to be returned.
	returns	PTR		if an element was found matching the datum, a pointer to
					the element, and if requested the element reference and
					the element index.
			NULL	if an element was not found matching the datum, and NULL
					element reference.												*/
void *list_sorted_find(	const list_object *restrict list,
						const void *datum,
						element_reference **restrict reference,
						unsigned int *index);


/* Insert a new element in an already sorted list, in correct sorted order,
	using datum as the search criteria to find the position.

	This function will perform a binary search of a sorted list, searching for
	the correct position to insert the new element, using the supplied datum.
	It will insert the element between other elements that lower and higher in
	sort order. If there are equal elements, the new element will be placed at
	the end of the equivalent elements, and before the first element that is
	higher in sort order.

	This function is available for finite class lists, however it is not
	generally recommended that is is used on them, as there exists the real
	possibility that if the new element happens to be placed in the last element
	position, and the list is currently full, the new element will be immediately
	deleted from the list. Should you choose to use this function on a finite
	class list please pay particular attention to the number of elements that are
	currently in the list, in relation to the maximum capacity of that list.
	
	This function is available for pivot class lists, first the pivot function
	set for the list is called using the datum, to determine which side of the
	list to operate on, then the function proceeds constrained to that side of
	the pivot list. If called for a pivot list, a pivot function must have been
	previously set for this list.
	
	A compare function must have been previously set for this list (by calling
	list_set_compare_function() ), this insert operation will use the set
	compare function to compare the element data against the supplied datum. It
	is also assumed that the compare function used to sort the list, ordered it
	using the same data points that the current compare function uses. If not,
	the results are undefined.
	
	This function is only available for lists that are currently sorted. This 
	function makes changes to the list, but retains its sorted order.

	list		list_object that has been previously initialized.
	element 	new element data, also used for the search criteria.
	returns	LIST_OK				success, new element added to the list
			ELEMENT_NO_DATUM	element data not supplied
			LIST_UNSORTED		function was called for an unsorted list
			LIST_ERR			on error	 										*/
int list_sorted_insert(	list_object *restrict list,
						const void *element);

/* Extract an element from an already sorted list, using datum as the search
	criteria.

	This function will perform a binary search of a sorted list, searching for
	an element with the matching datum supplied. It will only find the first
	search match in the sorted order of the list, and if found will delete the
	element, remove its internal resources, and return a pointer to the element
	data.

	This function is available for pivot class lists, first the pivot function
	set for the list is called using the datum, to determine which side of the
	list to operate on, then the function proceeds constrained to that side of
	the pivot list. If called for a pivot list, a pivot function must have been
	previously set for this list.
	
	A compare function must have been previously set for this list (by calling
	list_set_compare_function() ), this find operation will use the set compare
	function to compare the element data against the supplied datum. It is also
	assumed that the compare function used to sort the list, ordered it using
	the same data points that the current compare function uses. If not, the
	results are undefined.
	
	This function is only available for lists that are currently sorted. The 
	definitive test, on return, to verify if an element was found, is to check 
	the return pointer for either NULL or an address.

	list		list_object that has been previously initialized.
	datum 		data to be used for the search criteria.
	returns		NULL no element was available in list
				PTR to element data. It is the users responsibility to manage
					this data, and free it when finished with it. If the user
					supplied it originally it is the same allocated memory
					originally given to D-List. If the list was created with
					the LIST_COPY_DATA attribute, the data was allocated
					internally by D-List. In both cases it now belongs to the
					caller.															*/
void *list_sorted_extract(	list_object *restrict list,
							const void *datum);

/* Delete an element from an already sorted list, using datum as the search
	criteria.

	This function will perform a binary search of a sorted list, searching for
	an element with the matching datum supplied. It will only find the first
	search match in the sorted order of the list, and if found will delete the
	element, remove its internal resources, and delete the element data.

	This function is available for pivot class lists, first the pivot function
	set for the list is called using the datum, to determine which side of the
	list to operate on, then the function proceeds constrained to that side of
	the pivot list. If called for a pivot list, a pivot function must have been
	previously set for this list.

	A compare function must have been previously set for this list (by calling
	list_set_compare_function() ), this find operation will use the set compare
	function to compare the element data against the supplied datum. It is also
	assumed that the compare function used to sort the list, ordered it using
	the same data points that the current compare function uses. If not, the
	results are undefined.

	This function is only available for lists that are currently sorted. 

	list		list_object that has been previously initialized.
	datum 		data to be used for the search criteria.
	returns	LIST_OK				success, element deleted from the list
			ELEMENT_NO_DATUM	element data not supplied
			LIST_UNSORTED		function was called for an unsorted list
			ELEMENT_NOT_FOUND	no element was found matching the datum
			LIST_ERR			on error	 										*/
int list_sorted_delete(	list_object *restrict list,
						const void *datum);

/* Expunge all elements from an already sorted list, using datum as the search
	criteria.

	This function will perform a binary search of a sorted list, searching for
	all instances of the element with the matching datum supplied. If found it
	will delete all the instances of element, remove their internal resources,
	and delete all the element data.

	This function is available for pivot class lists, first the pivot function
	set for the list is called using the datum, to determine which side of the
	list to operate on, then the function proceeds constrained to that side of
	the pivot list. If called for a pivot list, a pivot function must have been
	previously set for this list.

	A compare function must have been previously set for this list (by calling
	list_set_compare_function() ), this find operation will use the set compare
	function to compare the element data against the supplied datum. It is also
	assumed that the compare function used to sort the list, ordered it using
	the same data points that the current compare function uses. If not, the
	results are undefined.

	This function is only available for lists that are currently sorted. 

	list		list_object that has been previously initialized.
	datum 		data to be used for the search criteria.
	returns	>0					number of elements found and removed matching
								the datum
			0					no element was found matching the datum
			ELEMENT_NO_DATUM	element data not supplied
			LIST_UNSORTED		function was called for an unsorted list
			LIST_ERR			on error	 										*/
int list_sorted_expunge(	list_object *restrict list,
							const void *datum);








/*	=====================================================================
		Section - Element Transformation Functions
	=====================================================================	*/

/* Sort all the elements in a list, using either ascending or descending order.

	This function traverses a list from head to tail, sorting all the elements
	in the order specified by the argument order, constrained by the argument 
	scope.
	
	The argument scope determines how much of the list this function will
	traverse sorting the elements. Setting this argument to ENTIRE_LIST will
	cause the function to sort the entire list from head to tail. For finite
	and linear class lists, any other setting for scope is ignored. 
	
	For pivot class lists there are 3 possible settings. Providing either 
	PIVOT_MAJOR or PIVOT_MINOR will cause this function to confine its sort to 
	that side of the list pivot point and leave the other side alone. Setting 
	scope to ENTIRE_LIST will cause this function to independently sort both 
	sides of the pivot in the order wished. It would be the same as calling this 
	function twice, once with PIVOT_MINOR, and once with PIVOT_MAJOR, except
	that by doing it in one operation the list is also marked as sorted, so is
	available for any of the Sorted List Element Functions. 
	
	Using ENTIRE_LIST on a pivot class list will sort the minor and major sides 
	of the list in the same order. For example, if the order was LIST_ASCENDING 
	then the minor elements would be ordered ascending from the head of the 
	list to the pivot point, and then the major elements would be ordered in 
	continuing ascending order from the pivot point to the tail of the list. If 
	you wish a different ordering scheme, please call this function twice once 
	for each side of the pivot with the ordering scheme you wish. However this 
	will not leave the list marked as sorted, so it will not be available to any 
	of the Sorted List Element Functions, as they all expect the ordering scheme
	to be consistent across a list.
	
	If you wish to sort a pivot class list while using data points that are 
	different from the pivot arraignment, or totally ignoring the pivot list 
	structure and rearrange the elements differently, first transfer or copy 
	the pivot class list into a linear class list, and then sort the elements 
	in that destination list anyway you wish. An alternative approach is to
	call this function twice once for each side of the pivot, and use a compare
	function that differs in data points from the pivot function currently set 
	for the list.

	A compare function must have been previously set for this list (by calling
	list_set_compare_function() ), this function uses that compare function to
	compare the element data looking for relative order of those elements. What
	constitutes greater or lesser for each element, is implementation dependent
	for the compare function, and the type of data the user stores in the list.

	This function extensively modifies the list. All previous indexes that may
	have been saved by the user are invalid. However all element references are
	still valid and can be used until the element has been removed from the
	list. Additionally if this is a pivot class list, all the internal pivot
	data is retained. However if the set compare function orders the elements
	using different data points than the pivot function for the list, then the
	pivot ordering of the elements is undefined. To retain the correct pivot
	arrangement while using a different compare data point, please use the
	order argument, and call the function twice, once with PIVOT_MINOR, once
	with PIVOT_MAJOR.
	
	Unlike the other sort functions provided by D-List, this function allocates
	additional memory resources to perform the sort operations, which is then
	released prior to return. For most users this is not a concern, and this
	function should be the sort of general use. However for systems with very
	tight memory requirements, such as low ram real memory embedded systems,
	one of the other sorts may be a better choice, if there is no memory left.
	
	This sort function is the best overall choice for sorting, it sorts lists at
	very high speeds and maintains all existing element references. The faster
	sort function, list_fast_sort() is about 15% faster than this sort, but will
	destroy all existing element references that may have been retained by the
	user. If element references are not being used, or stored, then the best sort
	to use is list_fast_sort(), otherwise for speed and retention this sort is
	the next choice.

	list		list_object that has been previously initialized.
	order		either LIST_ASCENDING or LIST_DESCENDING to denote which
		 		order to sort the elements in the list.
	scope		what section the list to sort the elements. Must be ENTIRE_LIST
				for linear or finite class lists, or may be either PIVOT_MINOR,
				PIVOR_MAJOR, or ENTIRE_LIST for pivot class lists
	returns	LIST_OK			the sort sequence was successful
			LIST_ERR		if there was an error.									*/
int list_sort(	list_object *restrict list,
				const boolean order,
				const int scope);

/* Sort all the elements in a list, using either ascending or descending order
	in a guaranteed safe order.

	This function traverses a list from head to tail, sorting all the elements
	in the order specified by the argument order, using a safe sort algorithm.
	
	A compare function must have been previously set for this list (by calling
	list_set_compare_function() ), this function uses that compare function to
	compare the element data looking for relative order of those elements. What
	constitutes greater or lesser for each element, is implementation dependent
	for the compare function, and the type of data the user stores in the list.
	
	Although this sort guarantees a safe sorting method, it will be much slower
	than the regular list_fast_sort(), list_sort() or list_tiny_sort() functions,
	especially for large lists. The larger the list size, the more dramatic the
	slow down will be. Only use this function if a safe sort if really needed,
	or the significant overhead is acceptable. I had considered naming this
	function list_turtle_sort(), but the safe_sort name is a more accurate
	description of the functions purpose.
	
	This function will sort pivot lists as requested, the list will be sorted in
	2 operations once for each side of the pivot point, using the order requested.
			
	This function extensively modifies the list. All previous indexes that may
	have been saved by the user are invalid. However all element references are
	still valid and can be used until the element has been removed from the
	list. Additionally if this is a pivot class list, all the internal pivot
	data is retained.
	
	This function uses no additional resources to perform the sort operation.
	It is a good choice for embedded systems, and for all users that need safe
	sort ordering and are willing to pay the performance price involved. In
	general this is the worst sort to use, but it depends on your specific
	device or design requirements.

	list		list_object that has been previously initialized.
	order		either LIST_ASCENDING or LIST_DESCENDING to denote which order
				to sort the elements in the list.
	returns	LIST_OK			the sort sequence was successful
			LIST_ERR		if there was an error.									*/
int list_safe_sort(	list_object *restrict list,
					const boolean order);

/* Sort all the elements in a list, using either ascending or descending order.

	This function traverses a list from head to tail, sorting all the elements
	in the order specified by the argument order. This function is identical to
	list_sort() except that all previous element references are voided. Never
	call list_fast_sort() if you need to retain any previous element references,
	however if you either did not retain element references, or can refresh them
	this is a much faster sort. On average this sort function is 15% faster than
	the list_sort() function.
	
	The argument scope determines how much of the list this function will
	traverse sorting the elements. Setting this argument to ENTIRE_LIST will
	cause the function to sort the entire list from head to tail. For finite
	and linear class lists, any other setting for scope is ignored. 
	
	For pivot class lists there are 3 possible settings. Providing either 
	PIVOT_MAJOR or PIVOT_MINOR will cause this function to confine its sort to 
	that side of the list pivot point and leave the other side alone. Setting 
	scope to ENTIRE_LIST will cause this function to independently sort both 
	sides of the pivot in the order wished. It would be the same as calling this 
	function twice, once with PIVOT_MINOR, and once with PIVOT_MAJOR, except
	that by doing it in one operation the list is also marked as sorted, so is
	available for any of the Sorted List Element Functions. 
	
	Using ENTIRE_LIST on a pivot class list will sort the minor and major sides 
	of the list in the same order. For example, if the order was LIST_ASCENDING 
	then the minor elements would be ordered ascending from the head of the 
	list to the pivot point, and then the major elements would be ordered in 
	continuing ascending order from the pivot point to the tail of the list. If 
	you wish a different ordering scheme, please call this function twice once 
	for each side of the pivot with the ordering scheme you wish. However this 
	will not leave the list marked as sorted, so it will not be available to any 
	of the Sorted List Element Functions, as they all expect the ordering scheme
	to be consistent across a list.
	
	If you wish to sort a pivot class list while using data points that are 
	different from the pivot arraignment, or totally ignoring the pivot list 
	structure and rearrange the elements differently, first transfer or copy 
	the pivot class list into a linear class list, and then sort the elements 
	in that destination list anyway you wish. An alternative approach is to
	call this function twice once for each side of the pivot, and use a compare
	function that differs in data points from the pivot function currently set 
	for the list.

	A compare function must have been previously set for this list (by calling
	list_set_compare_function() ), this function uses that compare function to
	compare the element data looking for relative order of those elements. What
	constitutes greater or lesser for each element, is implementation dependent
	for the compare function, and the type of data the user stores in the list.

	This function extensively modifies the list. All previous indexes and all
	element references that may have been saved by the user are invalid. If this
	is a pivot class list, all the internal pivot data is retained.
	
	This function uses no additional resources to perform the sort operation.
	It is a good choice for embedded systems, and for all users that do not
	need to retain any element references across the sort. In general this is
	the best sort to use, but it depends on your specific device or design
	requirements.

	list		list_object that has been previously initialized.
	order		either LIST_ASCENDING or LIST_DESCENDING to denote which
		 		order to sort the elements in the list.
	scope		what section the list to sort the elements. Must be ENTIRE_LIST
				for linear or finite class lists, or may be either PIVOT_MINOR,
				PIVOR_MAJOR, or ENTIRE_LIST for pivot class lists
	returns	LIST_OK			the sort sequence was successful
			LIST_ERR		if there was an error.									*/
int list_fast_sort(	list_object *restrict list,
					const boolean order,
					const int scope);

/* Sort all the elements in a list, using either ascending or descending order.

	This function traverses a list from head to tail, sorting all the elements
	in the order specified by the argument order. This sort will retain all
	exiting element references.
	
	The argument scope determines how much of the list this function will
	traverse sorting the elements. Setting this argument to ENTIRE_LIST will
	cause the function to sort the entire list from head to tail. For finite
	and linear class lists, any other setting for scope is ignored. 
	
	For pivot class lists there are 3 possible settings. Providing either 
	PIVOT_MAJOR or PIVOT_MINOR will cause this function to confine its sort to 
	that side of the list pivot point and leave the other side alone. Setting 
	scope to ENTIRE_LIST will cause this function to independently sort both 
	sides of the pivot in the order wished. It would be the same as calling this 
	function twice, once with PIVOT_MINOR, and once with PIVOT_MAJOR, except
	that by doing it in one operation the list is also marked as sorted, so is
	available for any of the Sorted List Element Functions. 
	
	Using ENTIRE_LIST on a pivot class list will sort the minor and major sides 
	of the list in the same order. For example, if the order was LIST_ASCENDING 
	then the minor elements would be ordered ascending from the head of the 
	list to the pivot point, and then the major elements would be ordered in 
	continuing ascending order from the pivot point to the tail of the list. If 
	you wish a different ordering scheme, please call this function twice once 
	for each side of the pivot with the ordering scheme you wish. However this 
	will not leave the list marked as sorted, so it will not be available to any 
	of the Sorted List Element Functions, as they all expect the ordering scheme
	to be consistent across a list.
	
	If you wish to sort a pivot class list while using data points that are 
	different from the pivot arraignment, or totally ignoring the pivot list 
	structure and rearrange the elements differently, first transfer or copy 
	the pivot class list into a linear class list, and then sort the elements 
	in that destination list anyway you wish. An alternative approach is to
	call this function twice once for each side of the pivot, and use a compare
	function that differs in data points from the pivot function currently set 
	for the list.

	A compare function must have been previously set for this list (by calling
	list_set_compare_function() ), this function uses that compare function to
	compare the element data looking for relative order of those elements. What
	constitutes greater or lesser for each element, is implementation dependent
	for the compare function, and the type of data the user stores in the list.

	This function extensively modifies the list. All previous indexes that may
	have been saved by the user are invalid. However all element references are
	still valid and can be used until the element has been removed from the
	list. Additionally if this is a pivot class list, all the internal pivot
	data is retained.
		
	This function uses no additional resources to perform the sort operation.
	It is a good choice for embedded systems, and systems with extreme memory
	constraints. However for all other users either list_fast_sort() or
	list_sort() are better choices. This sort will retain all existing element
	references across the sort. In general this is not a good sort to use, but
	it depends on your specific device or design requirements. The much faster
	function, list_fast_sort() is over 2x faster than this sort, but will
	destroy all existing element references that may have been retained by the
	user. The other fast sort, list_sort() is about 1.75x faster this sort,
	but allocates additional resources to perform the sort operations.

	list		list_object that has been previously initialized.
	order		either LIST_ASCENDING or LIST_DESCENDING to denote which
		 		order to sort the elements in the list.
	scope		what section the list to sort the elements. Must be ENTIRE_LIST
				for linear or finite class lists, or may be either PIVOT_MINOR,
				PIVOR_MAJOR, or ENTIRE_LIST for pivot class lists
	returns	LIST_OK			the sort sequence was successful
			LIST_ERR		if there was an error.									*/
int list_tiny_sort(	list_object *restrict list,
					const boolean order,
					const int scope);

/* Shift an element in a list, identified by its index, and relocate the
	element to the head, or tail, of the list.

	This function will relocate the element from its current position in the
	list, to either the head or tail of the list, based on the arguments
	provided. It is functionally similar to list_shift_ref(), except that
	the element is identified by index, not by reference.
	
	This function is not available for pivot class lists. This function is
	useful if used with a finite class list, that is being used as a recent
	history cache, or a linear list which the user wants to keep partially
	sorted without the overhead of constant sort operations. This function
	performs a sequence of operations that are equivalent to list_delete_index()
	and list_append() or list_prepend(), but far more efficient for this more
	specific task, and without destroying the reference to the original data
	element.

	If the list is currently sorted, the sorted indexes will be voided, and the
	list_sorted_xx() functions are no longer available for the list, until it
	is resorted.
		
	This function makes changes to the list.

	list		list_object that has been previously initialized.
	index 		denotes which element in the list, is to be extracted and
				shifted to a new position.
	where 		either HEAD_OF_LIST or TAIL_OF_LIST, denotes where in
				the list to shift the element to.
	returns	LIST_OK				the element was relocated as requested
			ELEMENT_NOT_FOUND	supplied index was not a valid in the list
			LIST_ERR			if there was an error.								*/
int list_shift_index(	list_object *restrict list,
						const unsigned int index,
						const boolean where);

/* Shift an element in a list, identified by its reference, and relocate the
	element to the head, or tail, of the list.

	This function will relocate the element from its current position in the
	list, to either the head or tail of the list, based on the arguments
	provided. It is functionally similar to list_shift_index(), except that
	the element is identified by reference, not by index. A reference must have
	been passed to the caller previously by dlist from another call, such as
	list_search() or list_locate_ref().
	
	This function is not available for pivot class lists. This function is
	useful if used with a finite class list, that is being used as a recent
	history cache, or a linear list which the user wants to keep partially
	sorted without the overhead of constant sort operations. This function
	performs a sequence of operations that are equivalent to list_delete_ref()
	and list_append() or list_prepend(), but far more efficient for this more
	specific task, and without destroying the reference to the original data
	element.

	If the list is currently sorted, the sorted indexes will be voided, and the
	list_sorted_xx() functions are no longer available for the list, until it
	is resorted.
		
	This function makes changes to the list.

	list		list_object that has been previously initialized.
	reference 	an abstract locator in list, of the element to be removed
				and shifted to a new location.
	where 		either HEAD_OF_LIST or TAIL_OF_LIST, denotes where in the
				list to shift the element to.
	returns	LIST_OK			the element was relocated as requested
			ELEMENT_INVALID	an invalid element reference was provided
			LIST_ERR		if there was an error.									*/
int list_shift_ref(	list_object *restrict list,
					element_reference *restrict reference,
					const boolean where);

/* Rotate all the elements in a list, by the number of positions requested,
	either towards the head, or tail, of the list.

	This function will rotate all the elements from their current positions
	in the list, towards either the head or tail of the list, based on the
	the arguments provided. It is similar to list_shift_index(), except
	that the elements are kept in relative order and all shifted together.

	If the list is currently sorted, the sorted indexes will be voided, and the
	list_sorted_xx() functions are no longer available for the list, until it
	is resorted.
	
	This function is not available for pivot class lists.
	
	This function performs a sequence of operations that are equivalent to
	a repeated sequence of calls to of list_fetch() followed by list_append()
	or list_prepend(), but far more efficient for this more specific task,
	and without destroying the reference to the original data element.
	
	This function operates in a manner consistent with assembly language
	register shifts/rotates. If shifting elements towards the head of the
	list, each element that is shifted off the list, is then placed at the
	tail of the list. The opposite occurs if shifting towards the tail of
	the list.
	
	By using the following example of a sorted linear list
	
	data contents	->	a	b	c	d	e	f	g	h	i	j	k	l	m	n
	index number	->	0	1	2	3	4	5	6	7	8	9	10	11	12	13
	
	if the list is processed by list_rotate(list, 4, HEAD_OF_LIST), the result
	will be

	data contents	->	e	f	g	h	i	j	k	l	m	n	a	b	c	d
	index number	->	0	1	2	3	4	5	6	7	8	9	10	11	12	13

	if the same original list is processed by list_rotate(list, 6, TAIL_OF_LIST),
	the result will be

	data contents	->	i	j	k	l	m	n	a	b	c	d	e	f	g	h
	index number	->	0	1	2	3	4	5	6	7	8	9	10	11	12	13
			
	This function makes changes to the list.

	list		list_object that has been previously initialized.
	count 		denotes how many postions to shift all the elements
				in the list.
	direction	either HEAD_OF_LIST or TAIL_OF_LIST, denotes which
				direction in the list to shift all the elements.
	returns	LIST_OK			the list elements were rotated as requested
			LIST_ERR		if there was an error.									*/
int list_rotate(	list_object *restrict list,
					const unsigned int count,
					const boolean direction);

/* Reverse the order of all the elements in a list.
	
	This function will take the chain of elements in a list and reverse them
	so the relative positioning of neighboring elements is the same, but the
	entire list will be reversed, or flipped like rotating a coin. If the list
	is a linear or finite class list, the reversal operation is performed on
	the whole list in one operation. If it is a pivot class list, there is the
	option to reverse the entire list, keeping the pivot semantics, or just a
	portion of the list, the minor or major side of the pivot point.
	
	This function does not sort the list, and retains all consistant list and
	element information without destroying the element references to the
	original data elements. If the list is currently sorted, the sorted data
	and indexes will be left in tact, just reversed.

	By using the following example of a sorted linear list
	
	data contents	->	a	b	c	d	e	f	g	h	i	j	k	l	m	n
	index number	->	0	1	2	3	4	5	6	7	8	9	10	11	12	13
	
	if the list is processed by list_flip(), the result will be

	data contents	->	n	m	l	k	j	i	h	g	f	e	d	c	b	a
	index number	->	0	1	2	3	4	5	6	7	8	9	10	11	12	13
	
	If the original list above was a pivot class list, and the defined pivot
	point for the list was <=d, then a list_flip(list, ENTIRE_LIST) operation
	would result in the following

	data contents	->	d	c	b	a	n	m	l	k	j	i	h	g	f	e
	index number	->	0	1	2	3	4	5	6	7	8	9	10	11	12	13

	By using PIVOT_MINOR, or PIVOT_MAJOR as the scope, the processing of the list
	would have been restricted to that portion only.
	
	For both linear and finite class lists, any scope other than ENTIRE_LIST is
	ignored and the whole list is processed, however a warning message will be
	issued if running in DEBUG mode.
	
	This function makes extensive changes to the list.

	list		list_object that has been previously initialized.
	scope		can be ENTIRE_LIST for all list classes, or PIVOT_MINOR or
				PIVOT_MAJOR for pivot class lists.
	returns	LIST_OK			reversal operation was successful.
			LIST_ERR		if there was an error.									*/
int list_flip(	list_object *restrict list,
				const int scope);

/* Shuffle the elements in a list, by randomly, and repeatedly relocating
	random list elements to the head, or tail, of the list.

	This function will un-sort, or shuffle, all the elements in the list.
	list. It is functionally similar to repeatedly calling list_shift_index(),
	but is more efficient.

	If the list is currently sorted, the sorted indexes will be voided, and the
	list_sorted_xx() functions are no longer available for the list, until it
	is resorted.
	
	This function is available for pivot class lists, it will shuffle the
	elements on both sides of the pivot point, keeping them on the correct
	sides, retaining the meaning of the pivot.
	
	This function is not useful for most business or technical OS purposes,
	but is useful for games, and other software that requires some randomness.
	
	This function retains all consistant list and element information without
	destroying the element references to the original data elements.
	
	This function changes the seed for the random number generator, if your
	code is using a predictable sequence of random numbers (non-seeded rand())
	then you should not use this function.
		
	This function makes changes to the list.

	list		list_object that has been previously initialized.
	returns	LIST_OK			the list was shuffled as requested
			LIST_ERR		if there was an error.									*/
int list_shuffle(list_object *restrict list);

/* Process all the elements in a list, using a user supplied function.

	This function initiates a series of calls to the user supplied process
	function, one for each element in the list, constrained by the scope. The
	function may process the elements however it wants, accumulate data, sum
	values, transform data in some user defined way. The caller may also
	provide the process function with a user data long int field, and a void
	pointer, that can be read, and/or modified by the function. This provides
	a clean MP safe mechanism for the function to communicate and return
	computed values to the original caller of list_process().
	
	This is a similar function to list_callback_iteration(), and creates a
	similar iteration sequence. However there are some major differences, using
	list_process() the function is supplied ad-hoc and can be changed for every
	call. The function has no control over the termination of the sequence, and
	the function has no access to relocate or remove elements. Additionally, in
	this function there is an MP safe mechanism provided to allow the function
	to accumulate data between calls, and safely and cleanly return significant
	amounts of data back to the caller after the element processing is over.
	
	The general purpose of this function is to allow users to easily extend the
	element statistics and transformation operations available, without having
	to either modify D-List, or make D-List aware of what kind of data is being
	stored in the lists. This feature allows users to quickly build extensive
	element computation functions with very little code.
	
	The processing sequence traverses a list from the head to the tail (or the
	opposite direction if that was requested), calling the process function for
	each element in sequence from the list, within the scope defined. For each
	element a pointer to the element data, a list reference to the element, the
	index of the element, and the current pivot value are provided to the
	process function, along with the provided optional user data pointers. If the
	list is linear class, pivot value will always be ENTIRE_LIST (which is 0).
	
	The process function can do whatever processing it requires, and when the
	element processing is completed, return with user dependent return code.
	
	This function can be thought of in different way, rather than just another
	solution to scrolling through elements in a list. It is also a method for
	processing all elements in a list with a specific function, such as if you
	wish to compute a transformation on all elements of a list holding 3D point
	values of objects, list_process() would be faster than setting up your own
	iteration sequence to perform such transformations.
	
	list		list_object that has been previously initialized.
	direction	HEAD_OF_LIST towards the head of the list, or TAIL_OF_LIST
				towards the tail of the list.
	scope		which side of the pivot to process the elements
				must be either PIVOT_MINOR, PIVOR_MAJOR, or ENTIRE_LIST
				For linear and finite class lists, must be ENTIRE_LIST
	process_function
				pointer to user supplied function to process all the elements
				in scope of the list.
	user_value	an optional long int field provided by the original caller of
				list_process() to allow the function use as needed, or NULL.
	user_data	an optional pointer to a field or a struct, that is provided
				by the original caller of list_process() to allow the
				function use as needed, or NULL.
	returns	>= 0				the process function may return a user defined
								code at the end of processing all the elements.
								Any return code >= 0 is user defined.
			ELEMENT_NOT_FOUND	the list is empty
			LIST_ERR			if there was an error.								*/
int list_process(	list_object *restrict list,
					const boolean direction,
					const int scope,
					const element_process process_function,
					long int *user_value,
					void *user_data);

/* Process a range of elements in a list, indicated by index number, using a
	user supplied function.

	This function initiates a series of calls to the user supplied process
	function, one for each element in the list, constrained by the index range
	provided. The function may process the elements however it wants, accumulate
	data, sum values, transform data in some user defined way. The caller may also
	provide the process function with a user data long int field, and a void
	pointer, that can be read, and/or modified by the function. This provides
	a clean MP safe mechanism for the function to communicate and return
	computed values to the original caller of list_process_range().

	This is a similar function to list_callback_iteration(), and creates a
	similar iteration sequence. However there are some major differences, using
	list_process() the function is supplied ad-hoc and can be changed for every
	call. The function has no control over the termination of the sequence, and
	the function has no access to relocate or remove elements. Additionally, in
	this function there is an MP safe mechanism provided to allow the function
	to accumulate data between calls, and safely and cleanly return significant
	amounts of data back to the caller after the element processing is over.
	
	This function differs from list_process() by constraining the range of element
	process to start and stop index points.
	
	The general purpose of this function is to allow users to easily extend the
	element statistics and transformation operations available, without having
	to either modify D-List, or make D-List aware of what kind of data is being
	stored in the lists. This feature allows users to quickly build extensive
	element computation functions with very little code.
	
	The processing sequence traverses a list from the element identified by
	index_start to the element identified by index_stop, calling the user process
	function for each element in sequence from the list. For each element a
	pointer to the element data, a list reference to the element, the index of the
	element, and the current pivot value are provided to the process function,
	along with the provided optional user data pointers. If the list is linear
	class, pivot value will always be ENTIRE_LIST (which is 0).
	
	The process function can do whatever processing it requires, and when the
	element processing is completed, return with user dependent return code.
	
	This function can be thought of in different way, rather than just another
	solution to scrolling through elements in a list. It is also a method for
	processing all elements in a list with a specific function, such as if you
	wish to compute a transformation on all elements of a list holding 3D point
	values of objects, list_process_range() would be faster than setting up your
	own iteration sequence to perform such transformations.
	
	list		list_object that has been previously initialized.
	index_start	denotes the first element to be processed.
	index_stop	denotes the last element to be processed, including all
				elements in-between.
	process_function
				pointer to user supplied function to process all the elements
				in scope of the list.
	user_value	an optional long int field provided by the original caller of
				list_process() to allow the function use as needed, or NULL.
	user_data	an optional pointer to a field or a struct, that is provided
				by the original caller of list_process() to allow the
				function use as needed, or NULL.
	returns	>= 0				the process function may return a user defined
								code at the end of processing all the elements.
								Any return code >= 0 is user defined.
			ELEMENT_NOT_FOUND	no element was found matching the index.
			LIST_ERR			if there was an error.								*/
int list_process_range(	list_object *restrict list,
						const unsigned int index_start,
						const unsigned int index_stop,
						const element_process process_function,
						long int *user_value,
						void *user_data);

/* Start a sequence of callback iteration of all the elements in a list.

	This function initiates an iteration sequence. The iteration sequence will
	continue until the iterate function signals upon return with a non-zero
	return code. This requires an iterate function to have been set prior to
	calling list_callback_iterate()
	
	An iteration sequence traverses a list from the head to the tail (or the
	opposite direction if that was requested), calling the iterate function for
	each element in sequence from the list. For each element a pointer to the
	element data, a list reference to the element, the index of the element,
	and the current pivot value are provided to the iterate function. If the
	list is linear class, pivot value will always be ENTIRE_LIST (which is 0).
	
	The iterate function can do whatever processing it requires, and return with
	either a 0 return code, which means continue with the iteration sequence, or
	a non zero return code, in which case the iteration sequence ends, and this
	function returns to its original caller. The meaning of the return codes is
	entirely user dependent.
	
	During an iteration sequence all index based modifications to the list are
	blocked and denied. However index based gets and searches can occur. Also
	by design, elements can be fetched, inserted, removed and modified using the
	element references returned by various other D-List function calls. It is the
	responsibility of the user to use these features wisely. In particular unwise
	use of inserting an element by reference at a point following the current
	iteration sequence location, has the obvious potential to create an infinite
	loop situation, of adding a new element and iterating to it, and adding
	another, etc. Use these features carefully...
		
	An iteration sequence can be started on any class of list.
	
	This iteration sequence operates in the same way as the regular iteration
	above except that there is only one function call to use, and control is
	not returned to the caller until the iteration sequence is complete.
	
	list		list_object that has been previously initialized.
	direction	HEAD_OF_LIST towards the head of the list, or TAIL_OF_LIST
				towards the tail of the list.
	returns	!0				the function returned a user defined code
			LIST_OK			the iteration sequence returned a zero code
			LIST_ERR		if there was an error.									*/
int list_callback_iteration(	list_object *restrict list,
								const boolean direction);



/*	=====================================================================
		Section - Miscellaneous List Functions
	=====================================================================	*/

/* Perform an optimization of the element index in a list.

	This function traverses a list from head to tail, rebuilding the index
	optimizations. These optimizations are lost when a list is modified with
	functions that use references instead of indexes. All other operations on
	a list either do not disturb the index, or adjust it as they operate. But
	insertions and removals by reference prevent this. They are used on very large
	lists to avoid the overhead of repeated operations when using the indexes.
	At the conclusion of a set of such reference operations, the caller can
	use this function to re-establish the index optimizations.
	
	This function is not a requirement to use or reuse indexes, just help
	support greater efficiency. Even if list modifications have been extensively
	made via element references, elements can still be retrieved, or modified
	using indexes. This function just restores the inbuilt index efficiency.
		
	list		list_object that has been previously initialized.
	returns	nothing																	*/
void list_optimize_index(list_object *restrict list);

/* Perform an optimization of the element memory storage in a list.

	This function first checks the current status of the list and the memory
	it is using, and if there is suitable potential for space reduction, it
	will traverses the list from head to tail, looking for blocks of memory
	to return to the operating system.
	
	This is a highly technical function, that most users will never need. It
	can be useful on embedded systems, and also if the user built a huge list,
	that is now very small, and wants to reclaim the used heap space. Embedded
	systems developers will want to fine tune the definition of the value of
	LEAF_NODE_ARRAY_SIZE, the settings for block sizes related to list size
	hints, and the vacancy percentage below in this function, to suit their
	particular system and needs. Also the use of finite lists would be
	generally recommended, where such use is practical.
	
	This function is not a requirement to use or reuse lists, or indexes, just
	help support greater efficiency for the small number of developers that may
	need it.
	
	This function does not disturb the actual list elements, all elements,
	data, references and indexes are left intact. From a users perspective, this
	function makes no changes to the list.
		
	list		list_object that has been previously initialized.
	returns	nothing																	*/
void list_optimize_memory(list_object *restrict list);

/* Instruct D-List not to seed the underlying random functions.

	This function is called to prevent D-List from calling srand() with its
	own seed. Some D-List functions use random numbers, and to ensure clean
	and safe randomization D-List seeds the random number generator with a
	complicated real-time seed.
	
	However some users may actually want a predictable set of pseudo-random
	numbers for a particular reason. D-List functions would interfere with
	that. By calling this function before calling any other D-List functions
	this situation can be avoided.
	
	This is a highly technical function, that most users will never need. It
	is only useful in very rare circumstances, and otherwise should never be
	called.
	
	This function is not a requirement to use D-List, or use or reuse lists,
	indexes or element references. It is just provided to help support a very
	rare need to allow users to generate predictable pseudo-random numbers
	from rand().
	
	This function does not disturb the actual list elements, all elements,
	data, references and indexes are left intact. From a users perspective,
	this function makes no changes to the lists or the D-List subsystem.
		
	returns	nothing																	*/
void list_do_not_seed_random(void);

/* Provide details about a list to stdout.

	This function first checks the current status of the list and the memory
	it is using, and then outputs to standard out (stdout) a brief report of
	the current status and resources used.
	
	This is a highly technical function, that most users will never need. It
	can be useful during code development on embedded systems, and also for
	users that built a huge lists.
	
	This function is not a requirement to use or reuse lists, or indexes, just
	help support greater efficiency for the small number of developers that may
	need it. It is important to remember that if dlist is compiled to use malloc
	then this function cannot compute the overhead or wastage (slop) in malloc()
	due to whatever algorithm is being used, it can only account for the actual
	space assigned to the list.
	
	This function makes no changes to the list.
		
	list		list_object that has been previously initialized.
	listname	a user provided string to identify the list in the output
	returns	nothing																	*/
void list_information(	const list_object *restrict list,
						const char *listname);

/* Return pointer to string corresponding to D-List error code.

	Returns a pointer to a string that describes the error code provided in
	error_code, or the string "Invalid Error Code" if the provided error code
	is invalid.
	
	error_code	a valid D-List error code, documented in this file.
	returns	pointer a null terminated read-only string. The string does not 
			belong to the caller, you must not try to modify or free it.			*/
char *list_error_string(const int error_code);



#ifdef __cplusplus
}
#endif		/* __cplusplus */

#endif		/* DLIST_H */