/** =================================================
 **                License Statement
 ** =================================================
 * 
 ** Copyright (c) 2014,2015 David Winterburn <info@fwsentry.org>. All commercial rights reserved.
 * 
 ** This file contains Original Code and/or Modifications of Original Code, and is
 ** subject to the license terms below. If this file contains portions of other
 ** code, it is so noted at the end of this license, with copyright credit as
 ** required.
 * 
 ** The intent of the Author with regards to this software package and license, is
 ** to allow all end users (personal, corporate or other) to be able to freely use
 ** this software to protect their computers, data and assets; but nobody should
 ** be free to further profit from the Original Code or Original Work(s) of the
 ** Author, by incorporating any portion of Original Code in commercial software,
 ** packages or offerings, without explicit written authorization from the Author.
 ** The Author generally encourages Open Source works, but in this case wishes to
 ** ensure that all users may continue to use this package free of charge, as
 ** philosophically the Author is opposed to people and companies making profits
 ** from legitimate users trying to protect their valuable resources, due to the
 ** inherent security weakness in the design of the Internet and/or traditional
 ** UNIX based software, which have led to the current explosion of abuse of
 ** systems and servers, by criminals, hackers, bulk mailers etc.
 * 
 ** Permission to use, copy, modify, and distribute this software for any purpose
 ** WITHOUT any fee is hereby granted as long as such purposes do not violate the
 ** terms of this license, and provided that the above copyright notice and this
 ** permission notice and license appear unmodified in ALL copies, distributed or
 ** stored. Permission is expressly denied and withheld for any commercial uses of
 ** this Original Code, either in whole or part, or for any purpose with a direct,
 ** or any indirect, fee. You may not include Original Code of the Author in any
 ** other work or package that will be sold, rented, or leased commercially. All
 ** commercial rights reserved, and all intellectual property and patent rights
 ** remain with the author.
 * 
 ** THE ORIGINAL CODE AND ALL SOFTWARE DISTRIBUTED UNDER THIS LICENSE IS PROVIDED
 ** ON AN "AS IS" BASIS WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESS OR IMPLIED,
 ** AND THE AUTHOR DISCLAIMS ALL SUCH WARRANTIES, OR LEGAL CLAIMS, WITH REGARD TO
 ** THIS SOFTWARE INCLUDING WITHOUT LIMITATION ANY WARRANTIES, IMPLIED OR OTHER,
 ** OF MERCHANTABILITY, PERFORMANCE AND FITNESS FOR A PARTICULAR PURPOSE, QUIET
 ** ENJOYMENT OR NON-INFRINGEMENT. UNDER NO CIRCUMSTANCES SHALL THE AUTHOR BE IN
 ** ANY WAY LIABLE FOR ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES, OR
 ** ANY DAMAGES WHATSOEVER, RESULTING FROM THE USE OF THIS SOFTWARE. SUCH DAMAGES
 ** MAY INCLUDE, BUT ARE NOT LIMITED TO, LOSS OF USE, EQUIPMENT, DATA, BILLABLE
 ** TIME, PERSONAL OR PHYSICAL INJURY, AND DIRECT OR INDIRECT PROFITS, WHETHER IN
 ** AN ACTION OF CONTRACT, NEGLIGENCE OR ANY OTHER TORTIOUS ACTION, ARISING OUT
 ** OF, OR IN CONNECTION WITH, THE USE OF, THE FITNESS OR PERFORMANCE OF THIS
 ** SOFTWARE, UNDER ANY CIRCUMSTANCES. THIS SOFTWARE WAS NOT DESIGNED TO BE USED
 ** IN ANY SITUATION WHERE HUMAN LIFE IS AT RISK, AND NO SUCH USE IS PROVIDED FOR
 ** OR PERMITTED.
 * 
 ** By using this software you agree not to violate any of the commercial use
 ** rights of the author with regard to the original code, and that any such
 ** violations of commercial use rights, either knowingly on unknowingly, will
 ** result in a) without notification from the author, the immediate and
 ** automatic termination of this license, and the subsequent requirement to
 ** remove and delete any copies of this software in your possession, b)
 ** forfeiture of all commercial use proceeds to the author and c) additional
 ** punitive damages, recovery and legal costs payable to the author. Under no
 ** circumstances whatsoever are others authorized to profit from the work(s) of
 ** the author without express written permission and consent.
 * 
 ** BY USING THIS SOFTWARE YOU AGREE THAT IT IS THE SOLE RESPONSIBILITY OF THE END
 ** USER TO DECIDE IF THIS SOFTWARE OR PACKAGE PERFORMS, OR IS FIT FOR, ANY
 ** INTENDED USAGE.
 * 
 ** This License and the rights granted hereunder will terminate; automatically
 ** without notice from the Author if You fail to comply with any term(s) of this
 ** License and fail to cure such breach within 30 days of becoming aware of such
 ** breach; or automatically without notice from the Author if You, at any time
 ** during the term of this License, commence any legal action against the Author
 ** or engage in any unauthorized commercial use activity with any of the Original
 ** Code. Upon termination of the license you are required to remove all copies of
 ** this software from all computer systems and data storage archives under your
 ** jurisdiction or control.
 * 
 ** This license is automatically extended, if in the future, the Author exercises
 ** any intellectual property, or patent, rights connected with any of the
 ** Original Code included in this package. ALL end users using this package
 ** within the terms of this license, will automatically licensed to use those
 ** exercised or covered claims free of charge as they apply to this software
 ** package.
 **/

/* 		FWSentry.
 * 
 * See http://info.fwsentry.org for more information on this package, or see
 * http://fwsentry.org/bugs/submission.html to report bugs or request new
 * features or improvements, or contact support@fwsentry.org. Bugs will be
 * addressed, new feature requests and program improvements will be considered,
 * at the sole discretion of the Author. Updates and new releases may be issued
 * in the future, if the Author considers it practical, enjoyable and desirable.
 */

/** 	Statement of Support.
 * 
 ** Bug reports must be accompanied with a reasonable description and reasonable
 ** documentation of the problem, and any code changes you may have made to the 
 ** affected module. If the problem turns out to be an issue with code not 
 ** supplied by the author, the author may, or may not, choose to fix it.
 * 
 ** If implemented, such requests or bug fixes may be added to a later release,
 ** package update, or support patch, at the sole discretion of the author, within
 ** time-frame's suitable to the author. All support that may optionally be
 ** provided by the author, including but not limited to, responding to requests
 ** for new attack signatures, bug reports, new feature requests, or general usage
 ** support; will be provided on an "as-is" basis, at the sole discretion of the
 ** author. Any and all support, or resulting code, documentation, or product
 ** changes, will be provided with the same license and legal disclaimers
 ** described in the original package (and above) and with absolutely no
 ** warranties, implied, stated or otherwise.
 **/ 

/*	JSON Parser version 1.0 -- Copyright (c) 2014-2022 David Winterburn <info@fwsentry.org>	*/


#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <time.h>
#include <fcntl.h>
#include <pthread.h>
#include <signal.h>
#include <assert.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <math.h>


#include "config.h"

#include "json-parser-test.h"
#include "dlist/dlist.h"
#include "json-parser/json-parser.h"

/*	There is no need to make this complicated, we will use a fixed directory name,
	and keep the test file names here. This test program must be executed with the
	user in the same directory they were in when they ran the make install.				*/

#define JSON_TEST_DIRECTORY				"test-data/"
char test_file_names[][64] = {
	"v-test-1.json", 
	"valid-0000.json",
	"valid-0001.json",
	"valid-0002.json",
	"valid-0003.json", 
	"valid-0004.json",
	"valid-0005.json",
	"valid-0006.json",
	"valid-0007.json",
	"valid-0008.json",
	"valid-0009.json",
	"valid-0010.json",
	"valid-0011.json",
	"valid-0012.json",
	"valid-0013.json",
	"valid-0014.json",
	"valid-0015.json",
	"i-test-1.json", 
	"invalid-0001.json",
	"invalid-0002.json",
	"invalid-0003.json",
	"invalid-0004.json",
	"invalid-0005.json",
	"invalid-0006.json",
	"invalid-0007.json",
	"invalid-0008.json",
	"invalid-0009.json",
	"invalid-0010.json",
	"extra-invalid-0000.json",
	"extra-invalid-0001.json",
	"extra-invalid-0002.json",
	"extra-invalid-0003.json",
	"extra-invalid-0004.json",
	"extra-invalid-0005.json",
	"extra-invalid-0006.json", 
	""
};

#define LIST_TYPE_ROOT				0
#define LIST_TYPE_OBJECT			JSON_object
#define LIST_TYPE_ARRAY				JSON_array
#define LIST_TYPE_VALUE				JSON_value

static char *load_json_file(const char *filename); 
static long int get_file_size(FILE *test_file, const char *filename);
void test_default_functions_numbers(list_object* json_object, const char *list_name, const int list_type);
void test_default_functions_strings(list_object* json_object, const char *list_name, const int list_type);
boolean test_find_members_searches(list_object* json_object, const char *list_name, const int list_type);
void display_found_members(list_object* list_object);

static boolean compare_json_objects(	list_object *original,
										list_object *generated,
										const char *list_name,
										const int list_type);
boolean test_json_function_set(void);


extern int elements_removed;
extern int lists_removed;

/* This code regression tests the JSON cose module in 3 passes.

	First it feeds all the known valid and invalid JSON files to the 
	JSON Validator, and analyses the results.
	
	Next it runs all the known valid and invalid JSON files through 
	the JSON Parser, and analyses the results.

	Lastly it runs all the known valid JSON files through the JSON
	Generator, then parses the results of those JSON Generator runs
	and finally compares the parsed results with those of pass #2.

	This code is not useful for any production purposes, but if you
	are unfamiliar with D-List this will serve as a brief primer on
	how to use the results of the JSON parser, and how to easily 
	walk the root list object returned by JSON Parsing operations.						*/


int main(int argc, char *argv[]) {
	
	list_object *json_object;
	list_object *generated_object;
	json_member_list_t *parsed_result;

	char *json_object_buffer;
	char *json_generated_data_block;
	JSON_value_types json_data_type;
	unsigned int generated_data_type;

	char next_test_file[128];
	int i, ret;
	boolean testing_for_valid;
	boolean all_validator_tests_passed = TRUE;
	boolean all_parser_tests_passed = TRUE;
	boolean all_generator_tests_passed = TRUE;
	boolean all_find_member_tests_passed = FALSE;


	fprintf(stdout, "\n\n");

	/* first run the JSON Validator test sequence */

	for (i = 0; test_file_names[i][0] != '\0'; ++i) {
		/* build file name for this test file */
		if (test_file_names[i][0] == 'v') testing_for_valid = TRUE;
		else testing_for_valid = FALSE;
		strcpy(next_test_file, JSON_TEST_DIRECTORY);
		strcat(next_test_file, test_file_names[i]);
		
		/* Open the file, and read the contents */
		json_object_buffer = load_json_file(next_test_file);
		if (json_object_buffer) {
			/* Parse the JSON object read in from the file, and analyse the results */
			fprintf(stdout, "Validating Test File '%s'. \n", next_test_file);
			if (json_valid_data_block(json_object_buffer, NULL, NULL, TRUE)) {
				if (testing_for_valid) {
					fprintf(stdout, "File '%s' was valid, and was correctly validated.\n", test_file_names[i]);
				} else {
					all_validator_tests_passed = FALSE;
					fprintf(stdout, "File '%s' was invalid, and should not have been passed by the validated.\n    Error wil show here .......\n", test_file_names[i]);
				}
			} else {
				if (testing_for_valid) {
					all_validator_tests_passed = FALSE;
					fprintf(stdout, "File '%s' was valid, and should have been validated correctly.\n    Error wil show here .......\n", test_file_names[i]);
				} else {
					fprintf(stdout, "File '%s' was invalid, and was correctly rejected by the validator.\n", test_file_names[i]);
				}
			}
			if (json_object_buffer) free(json_object_buffer);
		} else {
			all_validator_tests_passed = FALSE;
			fprintf(stdout, "Unable to open and read Test File '%s'. \n", next_test_file);
		}
	}

	/* Now run the JSON Parser test sequence */

	for (i = 0; test_file_names[i][0] != '\0'; ++i) {
		/* build file name for this test file */
		if (test_file_names[i][0] == 'v') testing_for_valid = TRUE;
		else testing_for_valid = FALSE;
		strcpy(next_test_file, JSON_TEST_DIRECTORY);
		strcat(next_test_file, test_file_names[i]);
		
		/* Open the file, and read the contents */
		json_object_buffer = load_json_file(next_test_file);
		if (json_object_buffer) {
			/* Parse the JSON object read in from the file, and analyse the results */
			fprintf(stdout, "Parsing Test File '%s'. \n", next_test_file);
			json_object = json_parse_data_block(json_object_buffer, &json_data_type, NULL, NULL, NULL, NULL, TRUE);
			if (json_object) {
				if (testing_for_valid) {
					fprintf(stdout, "File '%s' was valid, and parsed correctly. It returned a JSON ", test_file_names[i]);
					if (json_data_type == JSON_object) {
						fprintf(stdout, " Object.\n");
					} else {
						if (json_data_type == JSON_array) fprintf(stdout, " Array.\n");
						else fprintf(stdout, " Value.\n");
					}
					ret = json_report_list_object_type(json_object);
					if (ret != json_data_type) {
						all_parser_tests_passed = FALSE;
						fprintf(stdout, "File '%s' was of type %u, but json_report_list_object_type() reported it as type %u.\n\n", test_file_names[i], json_data_type, ret);
					}
					/* First display all the JSON objects at the root level */
					if (json_parser_debugging) display_parsed_json_object(json_object, test_file_names[i], json_data_type, FALSE);
					
					/* Next display everything, hierarchically */
					if (json_parser_debugging) display_parsed_json_object(json_object, test_file_names[i], json_data_type, TRUE);
					
					/* Now we special case some valid files, to test searches and sorts 
						this code is only used in debugging mode, as it only tests the various
						default search and compare functions */
					if (json_parser_debugging) {
						if (strcmp(test_file_names[i], "valid-0005.json") == 0) {
							test_default_functions_numbers(json_object, test_file_names[i], json_data_type);
						}
						if (strcmp(test_file_names[i], "valid-0000.json") == 0) {
							test_default_functions_strings(json_object, test_file_names[i], json_data_type);
						}
					}
					/* this exercises the find_members_value_xx functions */
					if (strcmp(test_file_names[i], "valid-0000.json") == 0) {
						all_find_member_tests_passed = test_find_members_searches(json_object, test_file_names[i], json_data_type);
						ret = json_report_list_object_type(json_object);
						if (ret != JSON_object) {
							all_parser_tests_passed = FALSE;
							fprintf(stdout, "File '%s' was of JSON_object, but json_report_list_object_type() reported it as %u.\n\n", test_file_names[i], ret);
						}
						parsed_result = json_parse_dotted_path("web-app.servlet[0].init-param.useJSP");
						if (! parsed_result) {
							fprintf(stdout, "%s() error parsing (web-app.servlet[0].init-param.useJSP) failed.\n", __func__);
							all_parser_tests_passed = FALSE;
						} else {
							if (json_path_locate(json_object, parsed_result, TRUE, FALSE) == NULL) {
								fprintf(stdout, "%s() error could not find Path (web-app.servlet[0].init-param.useJSP).\n", __func__);
							} else {
								if (parsed_result->json_member->value_type != JSON_boolean) {
									fprintf(stdout, "%s() error Member should be (web-app.servlet[0].init-param.useJSP) \"%s\": false.\n", __func__, parsed_result->json_member->member_name);
									all_parser_tests_passed = FALSE;
								}
								fprintf(stdout, "Found (web-app.servlet[0].init-param.useJSP) \"%s\": %s.\n", parsed_result->json_member->member_name,  (parsed_result->json_member->value.boolean_value) ? "true" : "false");
							}
							if (parsed_result) free(parsed_result);
						}
						parsed_result = json_parse_dotted_path("web-app.servlet[0].init-param.\"testing -value\"[1]");
						if (! parsed_result) {
							fprintf(stdout, "%s() error parsing (web-app.servlet[0].init-param.\"testing -value\"[1]) failed.\n", __func__);
							all_parser_tests_passed = FALSE;
						} else {
							if (json_path_locate(json_object, parsed_result, TRUE, FALSE) == NULL) {
								fprintf(stdout, "%s() error could not find Path (web-app.servlet[0].init-param.\"testing -value\"[1]).\n", __func__);
							} else {
								if (parsed_result->json_member->value_type != JSON_string) {
									fprintf(stdout, "%s() error Member should be (web-app.servlet[0].init-param.\"testing -value\"[1]) \"%s\": \"Steam Engine\".\n", __func__, parsed_result->json_member->member_name);
									all_parser_tests_passed = FALSE;
								}
								fprintf(stdout, "Found (web-app.servlet[0].init-param.\"testing -value\"[1]) \"%s\": \"%s\".\n", parsed_result->json_member->member_name,  parsed_result->json_member->value.string);
							}
							if (parsed_result) free(parsed_result);
						}
						/* now let's try a fuzzy search */
						parsed_result = json_parse_dotted_path("init-param.\"testing -value\"[1]");
						if (! parsed_result) {
							fprintf(stdout, "%s() error parsing (web-app.servlet[0].init-param.\"testing -value\"[1]) failed.\n", __func__);
							all_parser_tests_passed = FALSE;
						} else {
							if (json_path_locate(json_object, parsed_result, FALSE, FALSE) == NULL) {
								fprintf(stdout, "%s() error could not find Path (web-app.servlet[0].init-param.\"testing -value\"[1]).\n", __func__);
							} else {
								if (parsed_result->json_member->value_type != JSON_string) {
									fprintf(stdout, "%s() error Member should be (web-app.servlet[0].init-param.\"testing -value\"[1]) \"%s\": \"Steam Engine\".\n", __func__, parsed_result->json_member->member_name);
									all_parser_tests_passed = FALSE;
								}
								fprintf(stdout, "Found fuzzy (web-app.servlet[0].init-param.\"testing -value\"[1]) \"%s\": \"%s\".\n", parsed_result->json_member->member_name,  parsed_result->json_member->value.string);
							}
							if (parsed_result) free(parsed_result);
						}
					}

					/* these various sections below exercise the JSON Support Functions beyond find_members_value_xx */
					if (strcmp(test_file_names[i], "valid-0001.json") == 0) {
						ret = json_report_list_object_type(json_object);
						if (ret != JSON_value) {
							all_parser_tests_passed = FALSE;
							fprintf(stdout, "File '%s' was of JSON_value, but json_report_list_object_type() reported it as %u.\n\n", test_file_names[i], ret);
						}
					}
					if (strcmp(test_file_names[i], "valid-0002.json") == 0) {
						ret = json_report_list_object_type(json_object);
						if (ret != JSON_array) {
							all_parser_tests_passed = FALSE;
							fprintf(stdout, "File '%s' was of JSON_array, but json_report_list_object_type() reported it as %u.\n\n", test_file_names[i], ret);
						}
					}
					if (strcmp(test_file_names[i], "valid-0004.json") == 0) {
						ret = json_report_list_object_type(json_object);
						if (ret != JSON_object) {
							all_parser_tests_passed = FALSE;
							fprintf(stdout, "File '%s' was of JSON_object, but json_report_list_object_type() reported it as %u.\n\n", test_file_names[i], ret);
						}
						parsed_result = json_parse_dotted_path("results[0].id");
						if (! parsed_result) {
							fprintf(stdout, "%s() error parsing (results[0].id) failed.\n", __func__);
							all_parser_tests_passed = FALSE;
						} else {
							if (json_path_locate(json_object, parsed_result, TRUE, FALSE) == NULL) {
								fprintf(stdout, "%s() error could not find Path (results[0].id).\n", __func__);
							} else {
								if (strcmp(parsed_result->json_member->value.string, "45047cb1-3d3f-477e-a3dc-f14e8254e78d") != 0) {
									fprintf(stdout, "%s() error Member should be(results[0].id) \"%s\": \"45047cb1-3d3f-477e-a3dc-f14e8254e78d\".\n", __func__, parsed_result->json_member->member_name);
									all_parser_tests_passed = FALSE;
								}
								fprintf(stdout, "Found (results[0].id) \"%s\": \"%s\".\n", parsed_result->json_member->member_name,  parsed_result->json_member->value.string);
							}
							if (parsed_result) free(parsed_result);
						}
						parsed_result = json_parse_dotted_path("results[0].recordings[0].id");
						if (! parsed_result) {
							fprintf(stdout, "%s() error parsing (results[0].recordings[0].id) failed.\n", __func__);
							all_parser_tests_passed = FALSE;
						} else {
							if (json_path_locate(json_object, parsed_result, TRUE, FALSE) == NULL) {
								fprintf(stdout, "%s() error could not find Path (results[0].recordings[0].id).\n", __func__);
							} else {
								if (strcmp(parsed_result->json_member->value.string, "889ec8e0-b8a6-4ff1-a104-5512ea49fe87") != 0) {
									fprintf(stdout, "%s() error Member should be(results[0].recordings[0].id) \"%s\": \"889ec8e0-b8a6-4ff1-a104-5512ea49fe87\".\n", __func__, parsed_result->json_member->member_name);
									all_parser_tests_passed = FALSE;
								}
								fprintf(stdout, "Found (results[0].recordings[0].id) \"%s\": \"%s\".\n", parsed_result->json_member->member_name,  parsed_result->json_member->value.string);
							}
							if (parsed_result) free(parsed_result);
						}
					}
					list_erase(json_object);
					if (json_object) free(json_object);
				} else {
					all_parser_tests_passed = FALSE;
					fprintf(stdout, "File '%s' was invalid, and should not have been accepted by the parser.\n\n", test_file_names[i]);
				}
			} else {
				if (testing_for_valid) {
					all_parser_tests_passed = FALSE;
					fprintf(stdout, "File '%s' was valid, and should have been parsed correctly.\n\n", test_file_names[i]);
				} else {
					fprintf(stdout, "File '%s' was invalid, and was correctly rejected by the parser.\n", test_file_names[i]);
				}
			}
			if (json_object_buffer) free(json_object_buffer);
		} else {
			all_parser_tests_passed = FALSE;
			fprintf(stdout, "Unable to open and read Test File '%s'. \n", next_test_file);
		}
	}

	/* now run the JSON Generator test sequence */

	for (i = 0; ((test_file_names[i][0] != '\0') && (all_parser_tests_passed)); ++i) {
		/* build file name for this test file */
		if (test_file_names[i][0] != 'v') 
			/* we can only generate what we can parse */
			continue;

		strcpy(next_test_file, JSON_TEST_DIRECTORY);
		strcat(next_test_file, test_file_names[i]);
		
		/* Open the file, and read the contents */
		json_object_buffer = load_json_file(next_test_file);
		if (json_object_buffer) {
			/* Parse the JSON object read in from the file, and analyse the results */
			fprintf(stdout, "Re-generating Test File '%s'. \n", next_test_file);
			json_object = json_parse_data_block(json_object_buffer, &json_data_type, NULL, NULL, NULL, NULL, TRUE);
			if (json_object) {
					if (json_parser_debugging) {
						fprintf(stdout, "File '%s' was parsed and returned a JSON", test_file_names[i]);
						if (json_data_type == JSON_object) {
							fprintf(stdout, " Object.\n");
						} else {
							if (json_data_type == JSON_array) fprintf(stdout, " Array.\n");
							else fprintf(stdout, " Value.\n");
						}
					}
					ret = json_generate_data_block(json_object, json_data_type, &json_generated_data_block);
					if (ret < 0) {
						fprintf(stdout, "File '%s' was valid, and should have been re-generated correctly (ret %d).\n\n", test_file_names[i], ret);
						all_generator_tests_passed = FALSE;
						continue;
					}
					 if ((json_parser_debugging) && (json_generated_data_block)) {
						 fprintf(stdout, "Re-generating File '%s' produced this output :\n%s\n\n", test_file_names[i], json_generated_data_block);
					}
				
					generated_object = json_parse_data_block(json_generated_data_block, &generated_data_type, NULL, NULL, NULL, NULL, TRUE);
					if (! generated_object) {
						fprintf(stdout, "File '%s' was valid, and should have been generated and re-parsed correctly.\n\n", test_file_names[i]);
					 	all_generator_tests_passed = FALSE;
					 	continue;
					}
					if (json_data_type != generated_data_type) {
						fprintf(stdout, "File '%s' was valid, and should have been generated and re-parsed correctly, different JSON type returned.\n\n", test_file_names[i]);
					 	all_generator_tests_passed = FALSE;
					 	continue;
					}

					/* We have a generated JSON object data, lets validate it against the original list object */
					if (compare_json_objects(json_object, generated_object, test_file_names[i], json_data_type) == FALSE) {
						fprintf(stdout, "File '%s' was valid, and should have been re-generated correctly result compare failed.\n\n", test_file_names[i]);
					 	all_generator_tests_passed = FALSE;
					 	continue;
					 }
				
					list_erase(json_object);
					if (json_object) free(json_object);
					list_erase(generated_object);
					if (generated_object) free(generated_object);
					if (json_generated_data_block) free(json_generated_data_block);
			} else {
				fprintf(stdout, "File '%s' was valid, and should have been parsed correctly.\n\n", test_file_names[i]);
				all_parser_tests_passed = FALSE;
			}
			if (json_object_buffer) free(json_object_buffer);
		} else {
			all_parser_tests_passed = FALSE;
			fprintf(stdout, "Unable to open and read Test File '%s'. \n", next_test_file);
		}
	}


	fprintf(stdout, "\n\n==================================================================================================");
	if (all_validator_tests_passed) {
		fprintf(stdout, "\n\nAll JSON files were correctly validated.\n");
	} else {
		fprintf(stdout, "\n\nErrors Occurred - Some JSON files were NOT correctly validated.\n");
	}
	if (all_parser_tests_passed) {
		fprintf(stdout, "\n\nAll JSON files were correctly parsed.\n");
	} else {
		fprintf(stdout, "\n\nErrors Occurred - Some JSON files were NOT correctly parsed.\n");
	}
	if (all_generator_tests_passed) {
		fprintf(stdout, "\n\nAll JSON files were then correctly re-generated, parsed again and compared to each other.\n");
	} else {
		fprintf(stdout, "\n\nErrors Occurred - Some JSON files were NOT correctly re-generated and compared to each other.\n");
	}
	if (all_find_member_tests_passed) {
		fprintf(stdout, "\n\nAll JSON Support Member Find functions were then correctly tested against the original JSON file.\n");
	} else {
		fprintf(stdout, "\n\nErrors Occurred - Some JSON Support functions were NOT correctly regression tested.\n");
	}
	fprintf(stdout, "\n\n");
	
	fprintf(stdout, "\n\n==================================================================================================");
	if (test_json_function_set()) {
		fprintf(stdout, "\n\nAll JSON Support Data functions were correctly tested.\n");
	} else {
		fprintf(stdout, "\n\nErrors Occurred - Some JSON Support Data functions were NOT correctly regression tested.\n");
	}
	fprintf(stdout, "\n\n");
	
	
	/* namaste - je suis fini - もう おだぶつです - i estoy muerto - tôi chết - ฉันตาย - 我死 - io sono morto - Ich bin tot - Я мертв */
	/* So Long, and Thanks for All the Fish! */
	exit(0);
}

/* Open a test file, allocate required storage, and read the file in.

	returns		Pointer to memory block containing the JSON objects 
				read in from the file.
				NULL unable to load file.												*/

static char *load_json_file(const char *filename) {


	FILE *test_file;
	char *input_file_buffer = NULL;

	long int file_size_result;
	size_t file_read_size;


	test_file = fopen(filename, "rb");
	if(! test_file) {
		fprintf(stderr, "Unable to open test file '%s': %s\n", filename, strerror(errno));
	} else {
		file_size_result = get_file_size(test_file, filename);
		if (file_size_result > 0) {
			input_file_buffer = (char *)malloc(file_size_result+1);
			if (! input_file_buffer) {
				fprintf(stderr, "Unable to allocate %ld bytes to load test file '%s': %s\n", file_size_result, filename, strerror(errno));
			} else {
				file_read_size = fread(input_file_buffer, 1, file_size_result, test_file);
				if(file_read_size < (size_t)file_size_result) {
					fprintf(stderr, "Read error trying to load test file '%s': %ld\n", filename, file_read_size);
					free(input_file_buffer);
					input_file_buffer = NULL;
				} else {
					input_file_buffer[file_read_size] = '\0';
				}
			}
		}
		fclose(test_file);
	}

	return input_file_buffer;
}

/* Obtain the size in bytes of the file contents, and reset the file
	seek pointer to the start of the file, ready for subsequent 
	reading file operations.
	
	returns		size of file in bytes
				0 error occurred, or file is empty										*/
static long int get_file_size(FILE *test_file, const char *filename) {

	long int file_size_result = 0;
	int seek_result;

	seek_result = fseek(test_file, 0, SEEK_END);
	if(seek_result != 0) {
		fprintf(stderr, "Unable to locate EOF of test file '%s': %d %s\n", filename, seek_result, strerror(seek_result));
	} else {
		file_size_result = ftell(test_file);
		if(file_size_result < 0) {
			fprintf(stderr, "Unable to get size of test file '%s'.\n", filename);
		} else {
			seek_result = fseek(test_file, 0, SEEK_SET);
			if(seek_result != 0) {
				fprintf(stderr, "Unable to locate start of test file '%s': %d %s\n", filename, seek_result, strerror(seek_result));
				file_size_result = 0;
			}
		}
	}		

	return 	file_size_result;

}

/* This function performs various searches and sorts with a specific
	test JSON data file, to exercise the default search and compare
	function inside the module json-parser.c
	
	The code in main() will only call this is debugging is enabled in the
	environment prior to execution. Otherwise it serves as example code.				*/
void test_default_functions_numbers(list_object* json_object, const char *list_name, const int list_type) {

	json_element_t *element;
	json_element_t search_criteria;
	
	char search_float_str[] = "-1.0e-28";
	json_integer_t search_integer = 1001;
	
	search_criteria.value_type = JSON_integer;
	search_criteria.value.integer = -42;
	element = list_simple_search(json_object, &search_criteria);		/* Yes i know it is supposed to be there */
	if ((element) && (element->value_type == JSON_integer)) {
		fprintf(stdout, "\nJSON Data Block %s default function search for #%Ld yielded this result -> %Ld.\n", list_name, search_criteria.value.integer, element->value.integer);
	} else {
		fprintf(stdout, "\n*** ERROR ***    JSON Data Block %s search for #%Ld failed.\n", list_name, search_criteria.value.integer);
	}
	search_criteria.value.integer = -142;
	element = list_simple_search(json_object, &search_criteria);		/* Yes i know it is supposed to be there */
	if (! element) {
		fprintf(stdout, "\nJSON Data Block %s default function search for non-existent #%Ld yielded NO result.\n", list_name, search_criteria.value.integer);
	} else {
		fprintf(stdout, "\n*** ERROR ***    JSON Data Block %s search for #%Ld should not have succeeded.\n", list_name, search_criteria.value.integer);
	}
	
	search_criteria.value_type = JSON_double;
	search_criteria.value.double_number = 1.7734L;
	element = list_simple_search(json_object, &search_criteria);		/* Yes i know it is supposed to be there */
	if ((element) && (element->value_type == JSON_double)) {
		fprintf(stdout, "\nJSON Data Block %s default function search for #%Lf yielded this result -> %Lf.\n", list_name, search_criteria.value.double_number, element->value.double_number);
	} else {
		fprintf(stdout, "\n*** ERROR ***    JSON Data Block %s search for #%Lf failed.\n", list_name, search_criteria.value.double_number);
	}
	
	search_criteria.value_type = JSON_float_str;
	search_criteria.value.string = search_float_str;
	element = list_simple_search(json_object, &search_criteria);		/* Yes i know it is supposed to be there */
	if ((element) && (element->value_type == JSON_float_str)) {
		fprintf(stdout, "\nJSON Data Block %s default function search for #%s yielded this result -> %s.\n", list_name, search_criteria.value.string, element->value.string);
	} else {
		fprintf(stdout, "\n*** ERROR ***    JSON Data Block %s search for #%s failed.\n", list_name, search_criteria.value.string);
	}
	
	
	list_set_search_function(json_object, default_search_function_array_integer_value);
	element = list_simple_search(json_object, &search_integer);		/* Yes i know it is supposed to be there */
	if ((element) && (element->value_type == JSON_integer)) {
		fprintf(stdout, "\nJSON Data Block %s integer function search for #%Ld yielded this result -> %Ld.\n", list_name, search_integer, element->value.integer);
	} else {
		fprintf(stdout, "\n*** ERROR ***    JSON Data Block %s integer function search for #%Ld failed.\n", list_name, search_integer);
	}
	
	list_sort(json_object, LIST_ASCENDING, ENTIRE_LIST);
	fprintf(stdout, "\nJSON Data Block %s now sorted.\n", list_name);
	display_parsed_json_object(json_object, list_name, list_type, FALSE);

}

/* This function test the various default D-List functions provided with
	the JSON Parser release. This code knows it is working with the file
	valid-0000.json and is familiar with its contents.
	
	It is only used by development debugging, and has no other value other
	than as a set of examples of using D-List with the JSON parser output.
	
	There is no need to run these functions unless you modify the code
	of the default search or compare functions in json-parser.c							*/
void test_default_functions_strings(list_object* json_object, const char *list_name, const int list_type) {

	list_object *param_list;
	json_element_t *element_ptr, *param_element_ptr;
	
	json_element_t search_values;
	
	fprintf(stdout, "\nJSON Data Block %s starting string tests.\n", list_name);
	element_ptr = list_simple_search(json_object, "web-app");
	if ((element_ptr) && (element_ptr->value_type == JSON_object)) {
		element_ptr = list_simple_search(element_ptr->value.object, "servlet");
		if ((element_ptr) && (element_ptr->value_type == JSON_array)) {
			element_ptr = list_get_index(element_ptr->value.object, 0);
			if ((element_ptr) && (element_ptr->value_type == JSON_object)) {
				element_ptr = list_simple_search(element_ptr->value.object, "init-param");
				if ((element_ptr) && (element_ptr->value_type == JSON_object)) {
					param_list = element_ptr->value.object;
					/* we have to correct list or JSON objects, lets do something with them */
					param_element_ptr = list_simple_search(element_ptr->value.object, "dataStoreUrl");
					if ((param_element_ptr) && (param_element_ptr->value_type == JSON_string)){
						fprintf(stdout, "The search for 'dataStoreUrl' yielded %s : %s.\n", 
								param_element_ptr->member_name, 
								param_element_ptr->value.string);
						list_set_search_function(element_ptr->value.object, default_search_function_partial_member_name);
						param_element_ptr = list_simple_search(element_ptr->value.object, "Refresh");
						if (param_element_ptr) {
							if (param_element_ptr->value_type == JSON_string)
								fprintf(stdout, "The search for 'Refresh' partial name yielded %s : %s.\n", 
										param_element_ptr->member_name, 
										param_element_ptr->value.string);
							else
								fprintf(stdout, "The search for 'Refresh' partial name yielded %s : %Ld.\n", 
										param_element_ptr->member_name, 
										param_element_ptr->value.integer);
						}						
						
						list_set_search_function(element_ptr->value.object, default_search_function_array_values);
						search_values.value_type = JSON_boolean;
						search_values.value.boolean_value = TRUE;
						param_element_ptr = list_simple_search(element_ptr->value.object, &search_values);
						if ((param_element_ptr) && (param_element_ptr->value_type == JSON_boolean)){
							fprintf(stdout, "The search for boolean TRUE yielded %s : %s.\n", 
									param_element_ptr->member_name, 
									(param_element_ptr->value.boolean_value) ? "true" : "false");
						} else {
							fprintf(stdout, "\n*** ERROR ***    JSON Data Block %s boolean function search for true failed.\n", list_name);
						}

						fprintf(stdout, "\nJSON Data Block %s un-sorted.\n", list_name);
						display_parsed_json_object(element_ptr->value.object, list_name, list_type, FALSE);
						list_sort(element_ptr->value.object, LIST_ASCENDING, ENTIRE_LIST);
						fprintf(stdout, "\nJSON Data Block %s now sorted by object name.\n", list_name);
						display_parsed_json_object(element_ptr->value.object, list_name, list_type, FALSE);
						list_set_compare_function(element_ptr->value.object, default_compare_function_array_values);
						list_sort(element_ptr->value.object, LIST_ASCENDING, ENTIRE_LIST);
						fprintf(stdout, "\nJSON Data Block %s now sorted by value instead.\n", list_name);
						display_parsed_json_object(element_ptr->value.object, list_name, list_type, FALSE);
					}
				}
			}
		}
	}

	return;
}




/* This function test the various default D-List functions provided with
	the JSON Parser release. This code knows it is working with the file
	valid-0000.json and is familiar with its contents.
	
	It is only used by development debugging, and has no other value other
	than as a set of examples of using D-List with the JSON parser output.
	
	There is no need to run these functions unless you modify the code
	of the default search or compare functions in json-parser.c							*/
boolean test_find_members_searches(list_object* json_object, const char *list_name, const int list_type) {

	list_object *param_list;
	
	json_member_list_t search_values;
	
	int ret;
	boolean testing_success = TRUE;
	
	fprintf(stdout, "\nJSON Data Block %s starting FInd Object Members tests.\n", list_name);

	fprintf(stdout, " -- Testing full name 'blahblah-class' single return value should not result in a match.\n");
	ret = json_find_member_value(&search_values, json_object, "blahblah-class", FALSE, NULL, FALSE);
	if (ret == 0) {
		fprintf(stdout, "No Match found which is correct.\n");
	} else {
		fprintf(stdout, "\n*** ERROR ***    JSON Data Block %s blahblah-class Member Name search failed (%d) should have returned 0.\n", list_name, ret);
		testing_success = FALSE;
	}

	fprintf(stdout, " -- Testing full name 'servlet-class' single return value.\n");
	ret = json_find_member_value(&search_values, json_object, "servlet-class", FALSE, NULL, FALSE);
	if (ret == 5) {
		fprintf(stdout, "Match #1 of %d found was \"%s\" : \"%s\" type %d ptr %p\n\n", ret, search_values.json_member->member_name, search_values.json_member->value.string, search_values.value_type, search_values.json_owner_object);
	} else {
		fprintf(stdout, "\n*** ERROR ***    JSON Data Block %s servlet-class Member Name search failed.\n", list_name);
		testing_success = FALSE;
	}
	
	fprintf(stdout, " -- Testing full name 'servlet-class'.\n");
	param_list = json_find_members_value_string(json_object, "servlet-class", FALSE, NULL, FALSE);
	if ((param_list) && (list_size(param_list) == 5)) {
		display_found_members(param_list);
	} else {
		fprintf(stdout, "\n*** ERROR ***    JSON Data Block %s servlet-class Member Name search failed.\n", list_name);
		testing_success = FALSE;
	}

	fprintf(stdout, " -- Testing full name 'servlet-class' string value 'Email'.\n");
	param_list = json_find_members_value_string(json_object, "servlet-class", FALSE, "Email", TRUE);
	if ((param_list) && (list_size(param_list) == 1)) {
		display_found_members(param_list);
	} else {
		fprintf(stdout, "\n*** ERROR ***    JSON Data Block %s servlet-class Member Name search failed.\n", list_name);
		testing_success = FALSE;
	}

	fprintf(stdout, " -- Testing partial name 'servlet'.\n");
	param_list = json_find_members_value_string(json_object, "servlet", TRUE, NULL, FALSE);
	if ((param_list) && (list_size(param_list) == 13)) {
		display_found_members(param_list);
	} else {
		fprintf(stdout, "\n*** ERROR ***    JSON Data Block %s servlet Member Name search failed.\n", list_name);
		testing_success = FALSE;
	}

	fprintf(stdout, " -- Testing partial name 'ai' partial string value 'mail'.\n");
	param_list = json_find_members_value_string(json_object, "ai", TRUE, "mail", TRUE);
	if ((param_list) && (list_size(param_list) == 3)) {
		display_found_members(param_list);
	} else {
		fprintf(stdout, "\n*** ERROR ***    JSON Data Block %s ai Member Name search failed.\n", list_name);
		testing_success = FALSE;
	}

	fprintf(stdout, " -- Testing partial name 'te' partial string value 'Engine'.\n");
	param_list = json_find_members_value_string(json_object, "te", TRUE, "Engine", TRUE);
	if ((param_list) && (list_size(param_list) == 3)) {
		display_found_members(param_list);
	} else {
		fprintf(stdout, "\n*** ERROR ***    JSON Data Block %s te Member Name search failed.\n", list_name);
		testing_success = FALSE;
	}

	fprintf(stdout, " -- Testing partial name 'cache' integer value 10.\n");
	param_list = json_find_members_value_integer(json_object, "cache", TRUE, 10, 0, FALSE);
	if ((param_list) && (list_size(param_list) == 2)) {
		display_found_members(param_list);
	} else {
		fprintf(stdout, "\n*** ERROR ***    JSON Data Block %s cache Member Name search failed.\n", list_name);
		testing_success = FALSE;
	}

	fprintf(stdout, " -- Testing partial name 'age' integer range 10 - 150.\n");
	param_list = json_find_members_value_integer(json_object, "age", TRUE, 10, 150, TRUE);
	if ((param_list) && (list_size(param_list) == 5)) {
		display_found_members(param_list);
	} else {
		fprintf(stdout, "\n*** ERROR ***    JSON Data Block %s age Member Name search failed.\n", list_name);
		testing_success = FALSE;
	}

	fprintf(stdout, " -- Testing partial name 'a' integer range 11 - 200.\n");
	param_list = json_find_members_value_integer(json_object, "a", TRUE, 11, 200, TRUE);
	if ((param_list) && (list_size(param_list) == 11)) {
		display_found_members(param_list);
	} else {
		fprintf(stdout, "\n*** ERROR ***    JSON Data Block %s a Member Name search failed.\n", list_name);
		testing_success = FALSE;
	}

	fprintf(stdout, " -- Testing partial name 'value' double value 101.79.\n");
	param_list = json_find_members_value_double(json_object, "value", TRUE, 101.79L, 0, FALSE);
	if ((param_list) && (list_size(param_list) == 1)) {
		display_found_members(param_list);
	} else {
		fprintf(stdout, "\n*** ERROR ***    JSON Data Block %s value Member Name search failed.\n", list_name);
		testing_success = FALSE;
	}

	fprintf(stdout, " -- Testing partial name 'value' double range 200 - 20000.\n");
	param_list = json_find_members_value_double(json_object, "value", TRUE, 200, 20000, TRUE);
	if ((param_list) && (list_size(param_list) == 2)) {
		display_found_members(param_list);
	} else {
		fprintf(stdout, "\n*** ERROR ***    JSON Data Block %s value Member Name search failed.\n", list_name);
		testing_success = FALSE;
	}

	fprintf(stdout, " -- Testing partial name 'beta' boolean.\n");
	param_list = json_find_members_value_type(json_object, "beta", TRUE, JSON_boolean);
	if ((param_list) && (list_size(param_list) == 1)) {
		display_found_members(param_list);
	} else {
		fprintf(stdout, "\n*** ERROR ***    JSON Data Block %s beta Member Name search failed.\n", list_name);
		testing_success = FALSE;
	}

	fprintf(stdout, " -- Testing partial name 'data' integer.\n");
	param_list = json_find_members_value_type(json_object, "data", TRUE, JSON_integer);
	if ((param_list) && (list_size(param_list) == 4)) {
		display_found_members(param_list);
	} else {
		fprintf(stdout, "\n*** ERROR ***    JSON Data Block %s data Member Name search failed.\n", list_name);
		testing_success = FALSE;
	}

	fprintf(stdout, " -- Testing partial name 'test' double.\n");
	param_list = json_find_members_value_type(json_object, "test", TRUE, JSON_double);
	if ((param_list) && (list_size(param_list) == 2)) {
		display_found_members(param_list);
	} else {
		fprintf(stdout, "\n*** ERROR ***    JSON Data Block %s test Member Name search failed.\n", list_name);
		testing_success = FALSE;
	}

	fprintf(stdout, " -- Testing partial name 'Class' string.\n");
	param_list = json_find_members_value_type(json_object, "Class", TRUE, JSON_string);
	if ((param_list) && (list_size(param_list) == 4)) {
		display_found_members(param_list);
	} else {
		fprintf(stdout, "\n*** ERROR ***    JSON Data Block %s Class Member Name search failed.\n", list_name);
		testing_success = FALSE;
	}

	fprintf(stdout, " -- Testing partial name 'pa' object.\n");
	param_list = json_find_members_value_type(json_object, "pa", TRUE, JSON_object);
	if ((param_list) && (list_size(param_list) == 3)) {
		display_found_members(param_list);
	} else {
		fprintf(stdout, "\n*** ERROR ***    JSON Data Block %s pa Member Name search failed.\n", list_name);
		testing_success = FALSE;
	}

	fprintf(stdout, " -- Testing partial name 'value' array.\n");
	param_list = json_find_members_value_type(json_object, "value", TRUE, JSON_array);
	if ((param_list) && (list_size(param_list) == 1)) {
		display_found_members(param_list);
	} else {
		fprintf(stdout, "\n*** ERROR ***    JSON Data Block %s value Member Name search failed.\n", list_name);
		testing_success = FALSE;
	}



	return testing_success;
}

void display_found_members(list_object *list_object) {

	json_member_list_t *found_element;
	json_element_t *element_ptr;
	char *output_buffer;
	json_member_list_t *parsed_result;
	int i;

		list_iteration_start(list_object, TAIL_OF_LIST);
		i = 1;
		while (list_iteration_has_more(list_object)) {
			found_element = list_iteration_get_next(list_object, NULL);
			fprintf(stdout, "Match #%d of %d found was \"%s\" : ", i++,
					list_size(list_object), found_element->json_member->member_name);
			if ((found_element->value_type == JSON_string) ||
					(found_element->value_type == JSON_float_str)) {
				fprintf(stdout, "\"%s\"\n", found_element->json_member->value.string);
			}
			if (found_element->value_type == JSON_integer) {
				fprintf(stdout, "%Ld\n", found_element->json_member->value.integer);
			}
			if (found_element->value_type == JSON_double) {
				fprintf(stdout, "%Lf\n", found_element->json_member->value.double_number);
			}
			if (found_element->value_type == JSON_boolean) {
				fprintf(stdout, "%s\n", (found_element->json_member->value.boolean_value) ? "true" : "false");
			}
			if (found_element->value_type == JSON_null) {
				fprintf(stdout, "null\n");
			}
			if (found_element->value_type == JSON_object) {
				fprintf(stdout, "{ ... }\n");
			}
			if (found_element->value_type == JSON_array) {
				fprintf(stdout, "[ ");
				list_iteration_start(found_element->json_member->value.object, TAIL_OF_LIST);
				while (list_iteration_has_more(found_element->json_member->value.object)) {
					element_ptr = list_iteration_get_next(found_element->json_member->value.object, NULL);
					if ((element_ptr->value_type == JSON_string) || 
							(found_element->value_type == JSON_float_str))
						fprintf(stdout, "\"%s\"", element_ptr->value.string);
					if (element_ptr->value_type == JSON_integer) 
						fprintf(stdout, "%Ld", element_ptr->value.integer);
					if (element_ptr->value_type == JSON_double) 
						fprintf(stdout, "%Lf", element_ptr->value.double_number);
					if (found_element->value_type == JSON_boolean)
						fprintf(stdout, "%s", (found_element->json_member->value.boolean_value) ? "true" : "false");
					if (found_element->value_type == JSON_null)
						fprintf(stdout, "null");
					if (found_element->value_type == JSON_object)
						fprintf(stdout, "{ ... }");
					fprintf(stdout, "%s", (list_iteration_has_more(found_element->json_member->value.object)) ? ", " : " ");
				}
				list_iteration_stop(found_element->json_member->value.object);
				fprintf(stdout, "]\n");
			}
			fprintf(stdout, "Path to match ->");
			json_print_path_from_links(found_element->path, found_element->path_segments);
			fprintf(stdout, "\n");

			/* Test the json dotted path parser */
			output_buffer = json_generate_path_from_links(found_element->path, found_element->path_segments);
			fprintf(stdout, "Match TxtPath ->%s\n", output_buffer);
			parsed_result = json_parse_dotted_path(output_buffer);
			if (! parsed_result) {
				fprintf(stdout, "%s() error Re-Path failed.\n", __func__);
			} else {
				fprintf(stdout, "Match Re-Path ->");
				json_print_path_from_links(parsed_result->path, parsed_result->path_segments);
				fprintf(stdout, "\n\n");
				if (parsed_result) free(parsed_result);
			}


			if (output_buffer) free(output_buffer);

		}
		list_iteration_stop(list_object);
		list_erase(list_object);
		if (list_object) free(list_object);
	return;
}








/* this function compares 2 JSON data structures that are represented in
	D-List list objects, and verifies they are logically identical with
	identical element values.															*/
static boolean compare_json_objects(	list_object *original,
										list_object *generated,
										const char *list_name,
										const int list_type) {
									
	json_element_t *original_element;
	json_element_t *generated_element;

	if (json_parser_debugging) fprintf(stdout, "\nComparing the list objects of JSON Data Block %s.\n", list_name);

	if (list_size(original) != list_size(generated)) return FALSE;

	list_iteration_start(original, TAIL_OF_LIST);
	list_iteration_start(generated, TAIL_OF_LIST);
	while (list_iteration_has_more(original)) {
		original_element = list_iteration_get_next(original, NULL);
		generated_element = list_iteration_get_next(generated, NULL);
		if (! generated_element) return FALSE;
		if (original_element->value_type != generated_element->value_type) return FALSE;

		switch (original_element->value_type) {
			case JSON_object:
				if (json_parser_debugging) fprintf(stdout, "    New JSON Object Structure %s.\n", original_element->member_name);
				if (compare_json_objects(original_element->value.object, generated_element->value.object, original_element->member_name, LIST_TYPE_OBJECT) == FALSE) {
					return FALSE;
				}
				break;
			case JSON_array:
				if (json_parser_debugging) fprintf(stdout, "    New JSON Array Structure %s.\n", (original_element->member_name != NULL) ? original_element->member_name : "");
				if (compare_json_objects(original_element->value.object, generated_element->value.object, "An Array", LIST_TYPE_ARRAY) == FALSE) {
					return FALSE;
				}
				break;
			case JSON_string:
				if (json_parser_debugging) fprintf(stdout, "    String Element \"%s%s%s\"\n", 
						(original_element->member_name != NULL) ? original_element->member_name : "",
						(list_type == LIST_TYPE_OBJECT) ? " : " : "",
						original_element->value.string);
				if (strcmp(original_element->value.string, generated_element->value.string) != 0) return FALSE;
				break;
			case JSON_integer:
				if (json_parser_debugging) fprintf(stdout, "    Number (Integer) Element %s%s%Ld\n", 
						(original_element->member_name != NULL) ? original_element->member_name : "",
						(list_type == LIST_TYPE_OBJECT) ? " : " : "",
						original_element->value.integer);
				if (original_element->value.integer != generated_element->value.integer) return FALSE;
				break;
			case JSON_double:
				if (json_parser_debugging) fprintf(stdout, "    Number (Double) Element %s%s%Lf\n", 
						(original_element->member_name != NULL) ? original_element->member_name : "",
						(list_type == LIST_TYPE_OBJECT) ? " : " : "",
						original_element->value.double_number);
				/* quick non-authoritative if() check first to save time and processing */
				if (original_element->value.double_number != generated_element->value.double_number) {
					/* this is most likely a precision error, rather than an element mistake
						The value of MIN error in this check is much higher than LDBL_MIN 
						because the precision errors introduced by parsing, then writing
						out to JSON and parsing again seem to creep ever higher, we are
						not conducting maths here, just verifying known test files.*/
					if ((fabsl(original_element->value.double_number - generated_element->value.double_number)) > 0.00001L) {
						return FALSE;
					}
				}
				break;
			case JSON_float_str:
				if (json_parser_debugging) fprintf(stdout, "    Number String Element %s%s%s\n", 
						(original_element->member_name != NULL) ? original_element->member_name : "",
						(list_type == LIST_TYPE_OBJECT) ? " : " : "",
						original_element->value.string);
				if (strcmp(original_element->value.string, generated_element->value.string) != 0) return FALSE;
				break;
			case JSON_boolean:
				if (json_parser_debugging) fprintf(stdout, "    Boolean Element %s%s%s\n", 
						(original_element->member_name != NULL) ? original_element->member_name : "",
						(list_type == LIST_TYPE_OBJECT) ? " : " : "",
						(original_element->value.boolean_value) ? "true" : "false");
				if (original_element->value.boolean_value != generated_element->value.boolean_value) return FALSE;
				break;
			case JSON_null:
				if (json_parser_debugging) fprintf(stdout, "    NULL Element %s%s\"null\"\n",
						(original_element->member_name != NULL) ? original_element->member_name : "",
						(list_type == LIST_TYPE_OBJECT) ? " : " : "");
				break;
		
			default:
				fprintf(stdout, "** Unknown element %s : \"%s\"\n", original_element->member_name, original_element->value.string);
		}
	}
	list_iteration_stop(original);
	if (list_iteration_has_more(generated)) return FALSE;
	list_iteration_stop(generated);
	if (json_parser_debugging) fprintf(stdout, "\nCompare of JSON Data Blocks %s was successful.\n", list_name);

	return TRUE;
}

/* This function performs minor regression testing (the main testing is done in 
	other functions.
	
	Its main purpose is to provide the work used in the tutorial document
	that describes how to use the JSON Support Functions.								*/

boolean test_json_function_set(void) {

	list_object *json_object;
	list_object *films_json_object;
	list_object *cities_json_array;
	list_object *array_list;
	list_object *museum_json_object;
	list_object *search_result_list;
	json_element_t *array_element;

	json_element_t *temp_element;
	json_member_list_t search_result;
	json_member_list_t *search_match;

	json_element_t search_criteria;
	json_member_list_t *parsed_result;

	char *json_object_buffer;
	char *json_generated_data_block;
	JSON_value_types json_data_type;
	json_integer_t new_date = 2022;
	json_integer_t dates[] = { 1988, 1992, 1998, 2002, 2008, 2014 };
	char new_object_name[] = "museums";
	char museums_to_add[][10] = { "Louvre", "Tate", "MOMA", "Getty", "" };
	char location[][14] = { "Paris", "London", "New York", "Santa Monica","" };

	int i, ret;

	fprintf(stdout, "\n\n");

	json_object = json_create_new_list_object(JSON_object, NULL);

	ret = json_list_object_member_add_number(json_object, "Hayao", 72, 0);
	ret -= json_list_object_member_add_string(json_object, "Miyazaki", "Director");
	ret -= json_list_object_member_add_number(json_object, "IMDB", 0, 9.2);
	ret -= json_list_object_member_add_boolean(json_object, "IMDB active", TRUE);
	ret -= json_list_object_member_add_null(json_object, "Showing in SF");

	array_list = json_create_new_list_object(JSON_array, NULL);
	ret += json_list_object_member_add_list(json_object, "films", array_list, JSON_array);
	ret += json_list_object_array_value_add_number(array_list, &new_date, NULL, 1);
	ret += json_list_object_array_value_add_number(array_list, dates, NULL, 6);
	list_fast_sort(array_list, LIST_ASCENDING, ENTIRE_LIST);

	films_json_object = json_create_new_list_object(JSON_object, NULL);
	ret += json_list_object_array_value_add_list(array_list, films_json_object, JSON_object);
	list_fast_sort(array_list, LIST_ASCENDING, ENTIRE_LIST);

	for (i = 0; (museums_to_add[i][0] != '\0'); ++i) {
		ret -= json_list_object_member_add_string(films_json_object, museums_to_add[i], location[i]);
	}

	museum_json_object = json_create_new_list_object(JSON_object, NULL);
	ret += json_list_object_member_add_list(json_object, new_object_name, museum_json_object, JSON_object);
	
	/* yes we already have it, but this is to demonstrate how to re-acuire the element */
	array_element = list_get_index(array_list, 0);				/* get the first Array value */
	if ((array_element) && (array_element->value_type == JSON_object)) {
		films_json_object = array_element->value.object;
	} else {
		/* this is for demonstration purposes only, you should have checked that array_element is not NULL as well */
		fprintf(stdout, "Error occurred, we should have found the correct array element, type found was %u.\n", array_element->value_type);
	}
	
	list_transfer(films_json_object, museum_json_object);		/* move all the elements to the new list_object */
	/* now we delete the array value #0, also because of the default remove function, 
		this will also clean up the underlying list_object by calling list_erase() */
	list_delete_index(array_list, 0);							/* Now we can remove the (now empty) Array value */
	if (films_json_object) free(films_json_object);				/* release the storage we used for the list_object */
	films_json_object = NULL;								

	cities_json_array = json_create_new_list_object(JSON_array, NULL);
	ret += json_list_object_member_add_list(json_object, "cities", cities_json_array, JSON_array);
	for (i = 0; (location[i][0] != '\0'); ++i) {
		ret += json_list_object_array_value_add_string(cities_json_array, location[i]);
	}

	
	/* for proper production code, you should check ret on each return. */
	if (ret != LIST_OK) {
		fprintf(stdout, "ERROR  ==>  %s() json_list_object_member_add() failed (%d).\n", __func__, ret);
	}

	/* the find members section code snippets. */
	array_element = list_simple_search(json_object, "IMDB");
	if ((array_element) && (array_element->value_type == JSON_double)) { 
		fprintf(stdout, "Search #1 for IMDB found %Lf.\n", array_element->value.double_number);
	} else {
		fprintf(stdout, "Search for IMDB failed, or found the wrong thing !!!\n");
	}

	ret = json_find_member_value(&search_result, json_object, "IMDB", FALSE, NULL, FALSE);
	if (ret < 0) {
		fprintf(stdout, "Search for IMDB failed !!!\n");
	} else {
		if ((ret > 0) && (search_result.value_type == JSON_double)) {
			fprintf(stdout, "Search #2 for IMDB found %Lf.\n", search_result.json_member->value.double_number);
		} else {
			fprintf(stdout, "Search for IMDB found the wrong thing !!!\n");
		}
	}
	
	ret = json_find_member_value(&search_result, json_object, "Tate", FALSE, NULL, FALSE);
	if (ret < 0) {
		fprintf(stdout, "Search for Tate failed !!!\n");
	} else {
		if ((ret > 0) && (search_result.value_type == JSON_string)) {
			fprintf(stdout, "Search for Tate found %d matches, first one is -> \"%s\": \"%s\"\n", 
					ret, search_result.json_member->member_name, search_result.json_member->value.string);
		} else {
			fprintf(stdout, "Search for Tate found the wrong thing !!!\n");
		}
	}

	list_iteration_start(search_result.json_owner_object, TAIL_OF_LIST);
	fprintf(stdout, "The contents of the JSON Object are :\n{\n");
	while (list_iteration_has_more(search_result.json_owner_object)) {
		array_element = list_iteration_get_next(search_result.json_owner_object, NULL);
		fprintf(stdout, "\t\"%s\": \"%s\"%s\n", 
				array_element->member_name, 
				array_element->value.string, 
				(list_iteration_has_more(search_result.json_owner_object)) ? "," : "");
	}
	list_iteration_stop(search_result.json_owner_object);
	fprintf(stdout, "}\n"); 

	/* This time use the normal JSON Member search that returns all results in a list */
	search_result_list = json_find_members_value_string(json_object, "Tate", FALSE, NULL, FALSE);
	if (! list_is_empty(search_result_list)) {
		/* get the first match */
		search_match = list_get_index(search_result_list, 0);
		/* display the JSON path to the found Member */
		json_object_buffer = json_generate_path_from_links(search_match->path, search_match->path_segments);
		fprintf(stdout, "Search for Tate found it with the JSON Path -> %s\n", json_object_buffer);
		list_erase(search_result_list);
	}

	/* search an array for specific values. */
	array_element = list_simple_search(json_object, "films");
	if ((array_element) && (array_element->value_type == JSON_array)) {
		search_criteria.value_type = JSON_integer;					/* tell the search what you are looking for */
		search_criteria.value.integer = 1998;
		if (list_simple_search(array_element->value.object, &search_criteria)) 
			fprintf(stdout, "1998 was found in the Array.\n");
		search_criteria.value.integer = 2000;
		if (list_simple_search(array_element->value.object, &search_criteria) == NULL) 
			fprintf(stdout, "2000 was NOT found in the Array, which is correct.\n");
	} else {
		fprintf(stdout, "Search for films failed, or found the wrong thing !!!\n");
	}

	list_iteration_start(array_element->value.object, TAIL_OF_LIST);
	fprintf(stdout, "The contents of the JSON Array are -> films: [ ");
	while (list_iteration_has_more(array_element->value.object)) {
		temp_element = list_iteration_get_next(array_element->value.object, NULL);
		fprintf(stdout, "%Ld%s", 
				temp_element->value.integer, 
				(list_iteration_has_more(array_element->value.object)) ? ", " : " ");
	}
	list_iteration_stop(array_element->value.object);
	fprintf(stdout, "]\n"); 

	/* to find something from a JSON dotted notation path */
	parsed_result = json_parse_dotted_path("museums.Getty");
	if (! parsed_result) {
		fprintf(stdout, "%s() error Path parsing failed.\n", __func__);
	} else {
		if (json_path_locate(json_object, parsed_result, TRUE, FALSE) == NULL) {
			fprintf(stdout, "%s() error could not find Path (museums.Getty).\n", __func__);
		} else {
			fprintf(stdout, "Found (museums.Getty) %s: %s.\n", parsed_result->json_member->member_name,  parsed_result->json_member->value.string);
		}
		if (parsed_result) free(parsed_result);
	}

	parsed_result = json_parse_dotted_path("cities[2]");
	if (! parsed_result) {
		fprintf(stdout, "%s() error Path parsing failed.\n", __func__);
	} else {
		if (json_path_locate(json_object, parsed_result, TRUE, FALSE) == NULL) {
			fprintf(stdout, "%s() error could not find Path (museums.Getty).\n", __func__);
		} else {
			fprintf(stdout, "Found (cities[2]) [ %s ].\n",  parsed_result->json_member->value.string);

			/* Now display the entire contents of the array we found */
			list_iteration_start(parsed_result->json_owner_object, TAIL_OF_LIST);
			fprintf(stdout, "The contents of the JSON Array are -> cities: [ ");
			while (list_iteration_has_more(parsed_result->json_owner_object)) {
				temp_element = list_iteration_get_next(parsed_result->json_owner_object, NULL);
				fprintf(stdout, "\"%s\"%s", 
						temp_element->value.string, 
						(list_iteration_has_more(parsed_result->json_owner_object)) ? ", " : " ");
			}
			list_iteration_stop(parsed_result->json_owner_object);
			fprintf(stdout, "]\n");
			
			/* you can also manipulate the found element using its element_reference 
				that was returned by json_path_locate() */
			temp_element = list_get_ref(parsed_result->json_owner_object, parsed_result->reference);
			fprintf(stdout, "Found by reference (cities[2]) [ %s ].\n",  temp_element->value.string);
			/* you can use the reference for many uses, without 
				having to search first the get the element */
			list_delete_ref(parsed_result->json_owner_object, &parsed_result->reference);

			/* Now display the contents of the array we found, minus that element */
			list_iteration_start(parsed_result->json_owner_object, TAIL_OF_LIST);
			fprintf(stdout, "The contents of the JSON Array are -> cities: [ ");
			while (list_iteration_has_more(parsed_result->json_owner_object)) {
				temp_element = list_iteration_get_next(parsed_result->json_owner_object, NULL);
				fprintf(stdout, "\"%s\"%s", 
						temp_element->value.string, 
						(list_iteration_has_more(parsed_result->json_owner_object)) ? ", " : " ");
			}
			list_iteration_stop(parsed_result->json_owner_object);
			fprintf(stdout, "]\n");
			
		}
		
		if (parsed_result) free(parsed_result);
	}

	/* adding an Object Member using path instead of list_object */
	parsed_result = json_parse_dotted_path("museums");
	if (! parsed_result) {
		fprintf(stdout, "%s() error Path parsing failed.\n", __func__);
	} else {
		if (json_path_locate(json_object, parsed_result, TRUE, FALSE) == NULL) {
			fprintf(stdout, "%s() error could not find Path (museums.Getty).\n", __func__);
		}
		if ((parsed_result->json_member) && (parsed_result->value_type == JSON_object)) {
			museum_json_object = parsed_result->json_member->value.object;		/* if the path_locate failed, this is NULL */
		} else {
			museum_json_object = NULL;
		}
		if (parsed_result) free(parsed_result);								/* we don't need this any longer */
	}
	
	if (museum_json_object) {
		ret -= json_list_object_member_add_string(museum_json_object, "Musée d'Orsay", "Paris");
	}

	/* now doing the same thing, but an easier way, however you would need to keep 
		the list_object if you wanted to do more work on that JSON Object Museums		*/
	ret -= json_list_object_member_add_string(json_path_to_list_object(json_object, "museums"), "SF MOMA", "San Francisco");
	ret -= json_list_object_array_value_add_string(json_path_to_list_object(json_object, "cities"), "San Francisco");




	/* you can copy this block of code to any point above, and display the state
		of the sample JSON block at that point, to replicate what is in the tutorial.	*/

	json_data_type = JSON_object;
	ret = json_generate_data_block(json_object, json_data_type, &json_generated_data_block);
	if (ret < 0) {
		fprintf(stdout, "Test Block generator error (ret %d).\n\n", ret);
	} else {
		if (json_generated_data_block) {
			fprintf(stdout, "\n\nGenerating test block produced this output :\n%s\n\n", json_generated_data_block);
		}
	}



	/* Clean up and exit */	
	list_erase(json_object);

	return (ret == 0);
}
