/** =================================================
 **                License Statement
 ** =================================================
 * 
 ** Copyright (c) 2014-2022 David Winterburn <info@fwsentry.org>. All commercial
 *	rights reserved.
 * 
 ** This file contains Original Code and/or Modifications of Original Code, and is
 ** subject to the license terms below. If this file contains portions of other
 ** code, it is so noted at the end of this license, with copyright credit as
 ** required.
 * 
 ** The intent of the Author with regards to this software package and license, is
 ** to allow all end users (personal, corporate or other) to be able to freely use
 ** this software to protect their computers, data and assets; but nobody should
 ** be free to further profit from the Original Code or Original Work(s) of the
 ** Author, by incorporating any portion of Original Code in commercial software,
 ** packages or offerings, without explicit written authorization from the Author.
 ** The Author generally encourages Open Source works, but in this case wishes to
 ** ensure that all users may continue to use this package free of charge, as
 ** philosophically the Author is opposed to people and companies making profits
 ** from legitimate users trying to protect their valuable resources, due to the
 ** inherent security weakness in the design of the Internet and/or traditional
 ** UNIX based software, which have led to the current explosion of abuse of
 ** systems and servers, by criminals, hackers, bulk mailers etc.
 * 
 ** Permission to use, copy, modify, and distribute this software for any purpose
 ** WITHOUT any fee is hereby granted as long as such purposes do not violate the
 ** terms of this license, and provided that the above copyright notice and this
 ** permission notice and license appear unmodified in ALL copies, distributed or
 ** stored. Permission is expressly denied and withheld for any commercial uses of
 ** this Original Code, either in whole or part, or for any purpose with a direct,
 ** or any indirect, fee. You may not include Original Code of the Author in any
 ** other work or package that will be sold, rented, or leased commercially. All
 ** commercial rights reserved, and all intellectual property and patent rights
 ** remain with the author.
 * 
 ** THE ORIGINAL CODE AND ALL SOFTWARE DISTRIBUTED UNDER THIS LICENSE IS PROVIDED
 ** ON AN "AS IS" BASIS WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESS OR IMPLIED,
 ** AND THE AUTHOR DISCLAIMS ALL SUCH WARRANTIES, OR LEGAL CLAIMS, WITH REGARD TO
 ** THIS SOFTWARE INCLUDING WITHOUT LIMITATION ANY WARRANTIES, IMPLIED OR OTHER,
 ** OF MERCHANTABILITY, PERFORMANCE AND FITNESS FOR A PARTICULAR PURPOSE, QUIET
 ** ENJOYMENT OR NON-INFRINGEMENT. UNDER NO CIRCUMSTANCES SHALL THE AUTHOR BE IN
 ** ANY WAY LIABLE FOR ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES, OR
 ** ANY DAMAGES WHATSOEVER, RESULTING FROM THE USE OF THIS SOFTWARE. SUCH DAMAGES
 ** MAY INCLUDE, BUT ARE NOT LIMITED TO, LOSS OF USE, EQUIPMENT, DATA, BILLABLE
 ** TIME, PERSONAL OR PHYSICAL INJURY, AND DIRECT OR INDIRECT PROFITS, WHETHER IN
 ** AN ACTION OF CONTRACT, NEGLIGENCE OR ANY OTHER TORTIOUS ACTION, ARISING OUT
 ** OF, OR IN CONNECTION WITH, THE USE OF, THE FITNESS OR PERFORMANCE OF THIS
 ** SOFTWARE, UNDER ANY CIRCUMSTANCES. THIS SOFTWARE WAS NOT DESIGNED TO BE USED
 ** IN ANY SITUATION WHERE HUMAN LIFE IS AT RISK, AND NO SUCH USE IS PROVIDED FOR
 ** OR PERMITTED.
 * 
 ** By using this software you agree not to violate any of the commercial use
 ** rights of the author with regard to the original code, and that any such
 ** violations of commercial use rights, either knowingly on unknowingly, will
 ** result in a) without notification from the author, the immediate and
 ** automatic termination of this license, and the subsequent requirement to
 ** remove and delete any copies of this software in your possession, b)
 ** forfeiture of all commercial use proceeds to the author and c) additional
 ** punitive damages, recovery and legal costs payable to the author. Under no
 ** circumstances whatsoever are others authorized to profit from the work(s) of
 ** the author without express written permission and consent.
 * 
 ** BY USING THIS SOFTWARE YOU AGREE THAT IT IS THE SOLE RESPONSIBILITY OF THE END
 ** USER TO DECIDE IF THIS SOFTWARE OR PACKAGE PERFORMS, OR IS FIT FOR, ANY
 ** INTENDED USAGE.
 * 
 ** This License and the rights granted hereunder will terminate; automatically
 ** without notice from the Author if You fail to comply with any term(s) of this
 ** License and fail to cure such breach within 30 days of becoming aware of such
 ** breach; or automatically without notice from the Author if You, at any time
 ** during the term of this License, commence any legal action against the Author
 ** or engage in any unauthorized commercial use activity with any of the Original
 ** Code. Upon termination of the license you are required to remove all copies of
 ** this software from all computer systems and data storage archives under your
 ** jurisdiction or control.
 * 
 ** This license is automatically extended, if in the future, the Author exercises
 ** any intellectual property, or patent, rights connected with any of the
 ** Original Code included in this package. ALL end users using this package
 ** within the terms of this license, will automatically licensed to use those
 ** exercised or covered claims free of charge as they apply to this software
 ** package.
 **/

/* 		FWSentry.
 * 
 * See http://info.fwsentry.org for more information on this package, or see
 * http://fwsentry.org/bugs/submission.html to report bugs or request new
 * features or improvements, or contact support@fwsentry.org. Bugs will be
 * addressed, new feature requests and program improvements will be considered,
 * at the sole discretion of the Author. Updates and new releases may be issued
 * in the future, if the Author considers it practical, enjoyable and desirable.
 */

/** 	Statement of Support.
 * 
 ** Bug reports must be accompanied with a reasonable description and reasonable
 ** documentation of the problem, and any code changes you may have made to the 
 ** affected module. If the problem turns out to be an issue with code not 
 ** supplied by the author, the author may, or may not, choose to fix it.
 * 
 ** If implemented, such requests or bug fixes may be added to a later release,
 ** package update, or support patch, at the sole discretion of the author, within
 ** time-frame's suitable to the author. All support that may optionally be
 ** provided by the author, including but not limited to, responding to requests
 ** for new attack signatures, bug reports, new feature requests, or general usage
 ** support; will be provided on an "as-is" basis, at the sole discretion of the
 ** author. Any and all support, or resulting code, documentation, or product
 ** changes, will be provided with the same license and legal disclaimers
 ** described in the original package (and above) and with absolutely no
 ** warranties, implied, stated or otherwise.
 **/ 

/*	JSON Parser version 1.0 -- Copyright (c) 2014-2022 David Winterburn <info@fwsentry.org>	*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <time.h>
#include <fcntl.h>
#include <pthread.h>
#include <signal.h>
#include <assert.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <math.h>
#include <float.h>

/* I have tried to make FWSentry suitable for wide variety of systems, not tying it to a C99 minimum
	standard. For now i am not going force users to use a C99 compiler and libraries, and so i define
	these here, so i can migrate my booleans to the C99 format, but keep the computability. I also
	personally prefer the TRUE and FALSE in upper case, it is easier quickly understand the code. */
#if __STDC_VERSION__ >- 199901L
#include <stdbool.h>
#endif		/* __STDC_VERSION__ >- 199901L */

#ifdef __bool_true_false_are_defined
#define TRUE										true
#define FALSE										false
#define boolean										_Bool

#else

/* for environments with no stdbool.h */
typedef	int	bool
#define	false										(bool)0
#define	true										(bool)1

#ifndef TRUE
#define TRUE										(bool)1
#endif		/* !TRUE */
#ifndef FALSE
#define FALSE										(bool)0
#endif		/* !FALSE */

#endif		/* __bool_true_false_are_defined */


#include "config.h"

#include "dlist/dlist.h"

#define _json_free_(x)		x = _json_free(x, __func__)

/* global debugging variables referenced here */
#include "json-parser.h"


/*	Material specific to the JSON Support Functions
	===============================================		*/
#define MEMBER_SEARCH_NAME_ONLY						1
#define MEMBER_SEARCH_STRING_VALUE					2
#define MEMBER_SEARCH_INTEGER_VALUE					3
#define MEMBER_SEARCH_INTEGER_RANGE					4
#define MEMBER_SEARCH_DOUBLE_VALUE					5
#define MEMBER_SEARCH_DOUBLE_RANGE					6
#define MEMBER_SEARCH_VALUE_TYPE					7
#define MEMBER_SEARCH_NUMBER_RANGE					8
#define MEMBER_SEARCH_COMLETE						-1
#define MEMBER_SEARCH_PASS_2						0
#define MEMBER_SEARCH_PASS_1						1
#define MAX_ARGUMENT_STRING_LENGTH					256




static int _json_find_members_full_iterate(	list_object *results,
											list_object *path_scratchpad,
											list_object *json_data, 
											const int search_depth,
											const char *member_name, 
											const boolean partial_name,
											const char *string_value,
											const json_integer_t minimum_value,
											const json_integer_t maximum_value,
											const json_double_t minimum_double_value,
											const json_double_t maximum_double_value,
											const boolean partial_or_range,
											const JSON_value_types value_type);

static int _json_list_object_element_add(	list_object *list_to_add, 
											const char *member_name,
											const JSON_value_types value_type,
											const char *string_value,
											const json_integer_t integer_number,
											const json_double_t double_number,
											const boolean boolean_value,
											list_object *new_list);

static int _json_test_add_member_to_array(	list_object *list_to_add, 
											const char *calling_function,
											const char *member_name,
											const boolean suppress_errors);

static int _json_test_add_value_to_object(	list_object *list_to_add, 
											const char *calling_function,
											const boolean suppress_errors);

char *_json_create_string_copy(		const char *input_string);

char *_json_create_string_n_copy(	const char *input_string,
											const size_t num_bytes);

void *_json_malloc(	const size_t size, 
					const char *caller);

void *_json_free(	void *pointer,
					const char *caller);


/*		Internal Only JSON Data Hierarchy D-List Functions		*/
static size_t json_member_find_size_function(const void *element) { 
	json_member_list_t *derivative_elem = (json_member_list_t *)element;
	if (element) {
		return (sizeof(json_member_list_t) + (sizeof(json_path_t) * derivative_elem->path_segments)); 
	} else {
		fprintf(stderr, "%s() called with NULL value    ==>  ERROR  <==\n", __func__);
		return 0;					/* Safety condition, but cannot happen unless someone has messed with the dlist.c code */
	}
}
static size_t json_path_size_function(const void *element) { 
	if (element) {
		return sizeof(json_path_t); 
	} else {
		fprintf(stderr, "%s() called with NULL value    ==>  ERROR  <==\n", __func__);
		return 0;					/* Safety condition, but cannot happen unless someone has messed with the dlist.c code */
	}
}


static void json_member_list_remove_function(const void *element) {
	json_member_list_t *extremis_elem = (json_member_list_t *)element;

	int i = 0;
	
	/* This remove function is added to all JSON Member lists (used to
		describe search matches, JSON data syntax paths etc) each element
		that is to be removed will be checked to see if it has any char *
		pointers inside it. If it does, those strings will be erased 
		before this element is erased. The strings used by these elements
		are always copies of the  originals, never unique.								*/
	if (extremis_elem) {
		while (i < extremis_elem->path_segments) {
			if (extremis_elem->path[i].value_type == JSON_object) {
				_json_free_(extremis_elem->path[i].link.member_name);
			}
			++i;
		}
	} else {
		fprintf(stderr, "%s() called with NULL value    ==>  ERROR  <==\n", __func__);
		return;						/* Safety condition, but cannot happen unless someone has messed with the dlist.c code */
	}
		
	return;
}


static void json_path_list_remove_function(const void *element) {
	json_path_t *extremis_elem = (json_path_t *)element;
	
	/* This remove function is ONLY added to JSON json_path_t lists (used
		to describe JSON Paths in link form) when that list is going to be
		prematurely erased due to some processing problems. 
		
		Before Such as list if handed off to someone else, it holds original
		pointers to allocated memory. After the list is handed off, that 
		hand-off list will have a remove function attached to deal with
		those floating strings etc											*/
	if (extremis_elem) {
		if (extremis_elem->value_type == JSON_object) {
			_json_free_(extremis_elem->link.member_name);
		}
	} else {
		fprintf(stderr, "%s() called with NULL value    ==>  ERROR  <==\n", __func__);
		return;						/* Safety condition, but cannot happen unless 
										someone has messed with the dlist.c code */
	}
		
	return;
}






/*	=========================================================================

		Section for the JSON Support Functions.
		
		All functions below here are related to the group of JSON Parser
		D-List type support functions. For example function to search
		Parser results for JSON Object Members by name and value.
		
		If you want to save footprint on a small or embedded system, and
		only use the parser, you can eliminated these functions below here.

		Please note:	All these functions are fully thread / MP safe.
		------------	However, it is still the callers responsibility
						to ensure that multiple threads do not attempt 
						to simultaneously modify and access the same data
						structures or list_object(s) used by these JSON
						functions. This should be done by using mutex 
						locks, semaphores, gates, barriers etc, or 
						whatever mechanisms you deem fit.

	=========================================================================	*/

				

/* Using the provided search criteria, search the JSON data and locate the first
	JSON object member in the list hierarchy that match the criteria, returning
	it a pointer to it to the caller.
	
	This function can be called to search only JSON Object Member names, or it
	can search for Member names and a string value criteria as well. For example, 
	it can search the entire hierarchy for a JSON Object Members with the name 
	"handle" or it could search for an Object Members with the name "roles" and
	the value "technical", which would return the first Object Member named 
	"roles" that also has a string value of "technical". The value of course 
	could be a data value in an JSON Array, depending on what the found Object
	Member had as a data value.
	
	If you wish to get a list of all the JSON Object Members that matched the
	search criteria, you must use the function json_find_members_value_string()
	instead, which returns a list of Object Members instead.
	
	This function may be called with an option to perform partial JSON Object Name
	and/or string value matches, so for the above example, a partial Object Name
	match for "role" would return the first Object Member which could be "role",
	roles" etc.
	
	This function DOES NOT return the internal representation of the JSON Path
	with the match. This is because the json_member_list_t is supplied by the
	caller, and they could not have allocated it to cover unknown length of path.

	Upon return if an Object Members was found that matched the search criteria,
	a pointer to the element and a pointer to the owning list_object will be
	returned in a  json_member_list_t structure the caller already provided. 
	
	result			A json_member_list_t struct which will be used to write the 
					information about any JSON Object Member found that matched the
					search criteria. Only one Member will be returned, even if there
					are more that matched the criteria.
	json_data		A D-List list_object that describes the JSON data to be searched.
					This can be a root list_object, or it can be any subordinate 
					list_object within that hierarchy. It can be an Array or an Object
					list. The search is confined to the list(s) covered by this top 
					level list_object.
	member_name		Pointer to a nul terminated string to use as a match against the
					JSON Object Member names in the JSON data list.
	partial_name	TRUE	perform the search using member_name as a partial name
							match,
					FALSE 	perform the search looking only for a complete match with
							member_name.
	string_value	Optional pointer to a nul terminated string to use as a match 
					against the JSON Object Member values in the JSON data list, that
					have already matched with member_name, and have the value type of
					JSON_string.
					NULL to perform a search for only JSON OObject Member names, and
					not involving any search of Object Member values. 
	partial_value	TRUE	perform the search using string_value as a partial string
							value match, only value if string_value is not NULL.
					FALSE 	perform the search looking only for a complete string 
							value match with string_value.
	
	returns		0			No JSON Object Members were found that matched the
							search criteria.
				>0			The number of JSON Object Members found that matched the
							search criteria. Only the first match found was returned
							in the struct results, no JSON path is included. 	
				<0			An error occurred											*/
int json_find_member_value(	json_member_list_t *result,
							list_object *json_data, 
							const char *member_name, 
							const boolean partial_name,
							const char *string_value,
							const boolean partial_value) {

	list_object *results;
	json_member_list_t *element;
	int ret = 0;
	
	/* Verify Arguments */
	if (! result) {
		fprintf(stderr, "ERROR  ==>  %s() argument #1 (result) is NULL.\n", __func__);
		return -1;
	}
	if (! json_data) {
		fprintf(stderr, "ERROR  ==>  %s() argument #2 (json_data) is NULL.\n", __func__);
		return -1;
	}
	if (list_is_empty(json_data)) {
		fprintf(stderr, "Warning  ==>  %s() argument #2 (json_data) is an empty list.\n", __func__);
		return -1;
	}
	if (! member_name) {
		fprintf(stderr, "ERROR  ==>  %s() argument #3 (member_name) is NULL.\n", __func__);
		return -1;
	}
	if (strlen(member_name) > MAX_ARGUMENT_STRING_LENGTH) {
		fprintf(stderr, "Warning  ==>  %s() excessive length argument #3 (%ld bytes)\n"\
						"%s\nDid you forget to terminate the string with a nul ?.\n",
						__func__, strlen(member_name), member_name);
		return -1;
	}
	if ((string_value) && (strlen(string_value) > MAX_ARGUMENT_STRING_LENGTH)) {
		fprintf(stderr, "Warning  ==>  %s() excessive length argument #5 (%ld bytes)\n"\
						"%s\nDid you forget to terminate the string with a nul ?.\n",
						__func__, strlen(string_value), string_value);
		return -1;
	}

	/* Set up this search group */
	result->json_owner_object = NULL;
	result->json_member = NULL;
	result->reference = NULL;
	result->value_type = invalid_empty;
	result->path_segments = 0;

	if (! (results = (list_object *)_json_malloc(sizeof(list_object), __func__))) return -1;

	list_create(results, LIST_COPY_DATA, json_member_find_size_function, LIST_SIZE_TINY);
	list_set_remove_function(results, json_member_list_remove_function);

	ret = _json_find_members_full_iterate(	results,
											NULL,
											json_data,
											-1,
											member_name,
											partial_name,
											string_value,
											0,
											0,
											0,
											0,
											partial_value,
											invalid_empty);
	
	/* examine the results */
	if (ret) {
		fprintf(stderr, "ERROR  ==>  %s() *** internal error occurred *** (%d)\n.", __func__, ret);
		ret = -1;
	} else {
		if (! list_is_empty(results)) {
			element = list_get_index(results, 0);
			result->json_owner_object = element->json_owner_object;
			result->json_member = element->json_member;
			result->reference = element->reference;
			result->value_type = element->value_type;
		}
		ret = list_size(results);
	}
	/* clean up */
	list_erase(results);
	 _json_free_(results);
					
	return ret;
}
				


/* Using the provided search criteria, search the JSON data and locate all object
	members in the list hierarchy that match the criteria, returning them in a
	new list_object to the caller.
	
	This function can be called to search only JSON Object Member names, or it
	can search for Member names and a string value criteria as well. For example, 
	it can search the entire hierarchy for all JSON Object Members with the name 
	"handle" and return a list of all Object Members found. Alternatively it could
	search for all Object Members with the name "roles" and the value "technical",
	which would return all the Object Members named "roles" that also has a string
	value of "technical". The value of course could be a data value in an JSON 
	Array, depending on what the found Object Member had as a data value.
	
	This function may be called with an option to perform partial JSON Object Name
	and/or string value matches, so for the above example, a partial Object Name
	match for "role" would return all Object Members including, "role", roles" etc.

	Upon return if Object Members were found that matched the search criteria,
	a list containing elements of the type json_member_list_t would be provided
	one element in the list for each match found. When the caller has finished
	with the list, it must erased with list_erase() and then freed. 
	
	json_data		A D-List list_object that describes the JSON data to be searched.
					This can be a root list_object, or it can be any subordinate 
					list_object within that hierarchy. It can be an Array or an Object
					list. The search is confined to the list(s) covered by this top 
					level list_object.
	member_name		Pointer to a nul terminated string to use as a match against the
					JSON Object Member names in the JSON data list.
	partial_name	TRUE	perform the search using member_name as a partial name
							match,
					FALSE 	perform the search looking only for a complete match with
							member_name.
	string_value	Optional pointer to a nul terminated string to use as a match 
					against the JSON Object Member values in the JSON data list, that
					have already matched with member_name, and have the value type of
					JSON_string.
					NULL to perform a search for only JSON OObject Member names, and
					not involving any search of Object Member values. 
	partial_value	TRUE	perform the search using string_value as a partial string
							value match, only value if string_value is not NULL.
					FALSE 	perform the search looking only for a complete string 
							value match with string_value.
	
	returns		list_object	contains the JSON Object Members found that matched the
							search criteria. This list was created with LIST_COPY_DATA
							option, so element's storage will be managed as described
							by D-List depending on what functions you use to access or
							manage this list. list_erase() will release ay internal and
							element storage remaining when called.
				NULL		No JSON Object Members were found that matched the search 
							criteria. No list_object was created.						*/
list_object *json_find_members_value_string(	list_object *json_data, 
												const char *member_name, 
												const boolean partial_name,
												const char *string_value,
												const boolean partial_value) {

	list_object *results;
	int ret = 0;
	
	/* Verify Arguments */
	if (! json_data) {
		fprintf(stderr, "ERROR  ==>  %s() argument #1 (json_data) is NULL.\n", __func__);
		return NULL;
	}
	if (list_is_empty(json_data)) {
		fprintf(stderr, "Warning  ==>  %s() argument #1 (json_data) is an empty list.\n", __func__);
		return NULL;
	}
	if (! member_name) {
		fprintf(stderr, "ERROR  ==>  %s() argument #2 (member_name) is NULL.\n", __func__);
		return NULL;
	}
	if (strlen(member_name) > MAX_ARGUMENT_STRING_LENGTH) {
		fprintf(stderr, "Warning  ==>  %s() excessive length argument #2 (%ld bytes)\n"\
						"%s\nDid you forget to terminate the string with a nul ?.\n",
						__func__, strlen(member_name), member_name);
		return NULL;
	}
	if ((string_value) && (strlen(string_value) > MAX_ARGUMENT_STRING_LENGTH)) {
		fprintf(stderr, "Warning  ==>  %s() excessive length argument #4 (%ld bytes)\n"\
						"%s\nDid you forget to terminate the string with a nul ?.\n",
						__func__, strlen(string_value), string_value);
		return NULL;
	}

	/* Set up this search group */
	if (! (results = (list_object *)_json_malloc(sizeof(list_object), __func__))) return NULL;

	list_create(results, LIST_COPY_DATA, json_member_find_size_function, LIST_SIZE_TINY);
	list_set_remove_function(results, json_member_list_remove_function);

	ret = _json_find_members_full_iterate(	results,
											NULL,
											json_data,
											-1,
											member_name,
											partial_name,
											string_value,
											0,
											0,
											0,
											0,
											partial_value,
											invalid_empty);
	
	/* examine the results */
	if (ret) {
		fprintf(stderr, "ERROR  ==>  %s() *** internal error occurred *** (%d)\n.", __func__, ret);
	}
	if ((ret) || list_is_empty(results)) {
		list_erase(results);
		_json_free_(results);
		return NULL;	
	}
					
	return results;
}




/* Using the provided search criteria, search the JSON data and locate all object
	members in the list hierarchy that match the criteria, returning them in a
	new list_object to the caller.
	
	This function can only be called to search JSON Object Member names, and a 
	number value criteria as well. To search for JSON Object Member names alone 
	please refer to the function json_find_members_value_string() above.
	
	For example, to search for all Object Members in the list hierarchy for a 
	match with the partial name "PackageTags" and the number values between 0 
	and 100, could return such Members as 	"cachePackageTagsRefresh": 60,
	but not return		"cachePackageTagsStore": 200,
	
	This function may be called with an option to perform JSON Object Name 
	number value matches on a single number or an inclusive range of the
	minimum and maximum numbers provided.
	
	This function will only compare all JSON_integer value types. This function
	cannot evaluate numbers of different classes (non-integer and integer), if
	you require this or other more complicated searches of JSON data list 
	hierarchies, you will have to write your own code to iterate through those
	list_objects and perform whatever evaluations you need.
	
	JSON data elements with the value type JSON_float_str, may be evaluated by 
	string comparison using json_find_members_value_string() above.

	Upon return if Object Members were found that matched the search criteria,
	a list containing elements of the type json_member_list_t would be provided
	one element in the list for each match found. When the caller has finished
	with the list, it must erased with list_erase() and then freed. 
	
	json_data		A D-List list_object that describes the JSON data to be searched.
					This can be a root list_object, or it can be any subordinate 
					list_object within that hierarchy. It can be an Array or an Object
					list. The search is confined to the list(s) covered by this top 
					level list_object.
	member_name		Pointer to a nul terminated string to use as a match against the
					JSON Object Member names in the JSON data list.
	partial_name	TRUE	perform the search using member_name as a partial name
							match,
					FALSE 	perform the search looking only for a complete match with
							member_name.
	minimum_value	A number to be used either as a direct match against the JSON 
					Object Member number values in the JSON data list, or as a minimum
					for a range match. Only JSON Object Members that have already 
					matched with member_name, and have the value type of JSON_integer 
					will be evaluated against either the number or the range.
	maximum_value	Optional number to to be used as a maximum number for a range match.
					Only JSON Object Members that have already matched with member_name,
					and have the value type of JSON_integer will be evaluated against 
					the range provided.
	value_range		TRUE	perform the search using minimum_value and maximum_value as
							an inclusive range of numbers to value match.
					FALSE 	perform the search looking only for a match with the number
							in minimum_value.
	
	returns		list_object	contains the JSON Object Members found that matched the
							search criteria. This list was created with LIST_COPY_DATA
							option, so element's storage will be managed as described
							by D-List depending on what functions you use to access or
							manage this list. list_erase() will release ay internal and
							element storage remaining when called.
				NULL		No JSON Object Members were found that matched the search 
							criteria. No list_object was created.						*/
list_object *json_find_members_value_integer(	list_object *json_data, 
												char *member_name, 
												const boolean partial_name,
												const json_integer_t minimum_value,
												const json_integer_t maximum_value,
												const boolean value_range) {

	list_object *results;
	int ret = 0;
	boolean range_value = value_range;
	
	/* Verify Arguments */
	if (! json_data) {
		fprintf(stderr, "ERROR  ==>  %s() argument #1 (json_data) is NULL.\n", __func__);
		return NULL;
	}
	if (list_is_empty(json_data)) {
		fprintf(stderr, "Warning  ==>  %s() argument #1 (json_data is an empty list).\n", __func__);
		return NULL;
	}
	if (! member_name) {
		fprintf(stderr, "ERROR  ==>  %s() argument #2 (member_name) is NULL.\n", __func__);
		return NULL;
	}
	if (strlen(member_name) > MAX_ARGUMENT_STRING_LENGTH) {
		fprintf(stderr, "Warning  ==>  %s() excessive length argument #2 (%ld bytes)\n"\
						"%s\nDid you forget to terminate the string with a nul ?.\n", 
						__func__, strlen(member_name), member_name);
		return NULL;
	}
	if ((minimum_value == 0) && (maximum_value == 0)) {
		fprintf(stderr, "ERROR  ==>  %s() called with zero in arguments #4 & 5.\n", __func__);
		return NULL;
	}
	if ((value_range) && (minimum_value >= maximum_value)) {
		fprintf(stderr, "ERROR  ==>  %s() arguments #4 (%" PRId64 ") must not be larger than argument "\
						"#5 (%" PRId64 ") in a range search.\n", 
						__func__, minimum_value, maximum_value);
		return NULL;
	}
	if ((value_range) && (minimum_value == maximum_value)) {
		fprintf(stderr, "Warning  ==>  %s() arguments #4 & 5 (%" PRId64 ") are equal in a range search, "\
						"range ignored.\n", __func__, minimum_value);
		range_value = FALSE;
	}

	/* Set up this search group */
	if (! (results = (list_object *)_json_malloc(sizeof(list_object), __func__))) return NULL;

	list_create(results, LIST_COPY_DATA, json_member_find_size_function, LIST_SIZE_TINY);
	list_set_remove_function(results, json_member_list_remove_function);

	ret = _json_find_members_full_iterate(	results,
											NULL,
											json_data,
											-1,
											member_name,
											partial_name,
											NULL,
											minimum_value,
											maximum_value,
											0,
											0,
											range_value,
											invalid_empty);
	
	/* examine the results */
	if (ret) {
		fprintf(stderr, "ERROR  ==>  %s() *** internal error occurred *** (%d)\n.", __func__, ret);
	}
	if ((ret) || list_is_empty(results)) {
		list_erase(results);
		_json_free_(results);
		return NULL;	
	}
					
	return results;
}

/* Using the provided search criteria, search the JSON data and locate all object
	members in the list hierarchy that match the criteria, returning them in a
	new list_object to the caller.
	
	This function can only be called to search JSON Object Member names, and a 
	number value criteria as well. To search for JSON Object Member names alone 
	please refer to the function json_find_members_value_string() above.
	
	For example, to search for all Object Members in the list hierarchy for a 
	match with the partial name "PackageTags" and the number values between 0 
	and 100, could return such Members as 	"cachePackageTagsRefresh": 52.5,
	but not return		"cachePackageTagsStore": 200,
	
	This function may be called with an option to perform JSON Object Name 
	number value matches on a single number or an inclusive range of the
	minimum and maximum numbers provided.
	
	This function will only compare all JSON_double value types. This function
	cannot evaluate numbers of different classes (non-integer and integer), if
	you require this or other more complicated searches of JSON data list 
	hierarchies, you will have to write your own code to iterate through those
	list_objects and perform whatever evaluations you need.
	
	JSON data elements with the value type JSON_float_str, may be evaluated by 
	string comparison using json_find_members_value_string() above.

	Upon return if Object Members were found that matched the search criteria,
	a list containing elements of the type json_member_list_t would be provided
	one element in the list for each match found. When the caller has finished
	with the list, it must erased with list_erase() and then freed. 
	
	json_data		A D-List list_object that describes the JSON data to be searched.
					This can be a root list_object, or it can be any subordinate 
					list_object within that hierarchy. It can be an Array or an Object
					list. The search is confined to the list(s) covered by this top 
					level list_object.
	member_name		Pointer to a nul terminated string to use as a match against the
					JSON Object Member names in the JSON data list.
	partial_name	TRUE	perform the search using member_name as a partial name
							match,
					FALSE 	perform the search looking only for a complete match with
							member_name.
	minimum_value	A number to be used either as a direct match against the JSON 
					Object Member number values in the JSON data list, or as a minimum
					for a range match. Only JSON Object Members that have already 
					matched with member_name, and have the value type of JSON_double 
					will be evaluated against either the number or the range.
	maximum_value	Optional number to to be used as a maximum number for a range match.
					Only JSON Object Members that have already matched with member_name,
					and have the value type of JSON_double will be  evaluated against
					the range provided.
	value_range		TRUE	perform the search using minimum_value and maximum_value as
							an inclusive range of numbers to value match.
					FALSE 	perform the search looking only for a match with the number
							in minimum_value.
	
	returns		list_object	contains the JSON Object Members found that matched the
							search criteria. This list was created with LIST_COPY_DATA
							option, so element's storage will be managed as described
							by D-List depending on what functions you use to access or
							manage this list. list_erase() will release ay internal and
							element storage remaining when called.
				NULL		No JSON Object Members were found that matched the search 
							criteria. No list_object was created.						*/
list_object *json_find_members_value_double(	list_object *json_data, 
												char *member_name, 
												const boolean partial_name,
												const json_double_t minimum_value,
												const json_double_t maximum_value,
												const boolean value_range) {

	list_object *results;
	int ret = 0;
	boolean range_value = value_range;
	
	/* Verify Arguments */
	if (! json_data) {
		fprintf(stderr, "ERROR  ==>  %s() argument #1 (json_data) is NULL.\n", __func__);
		return NULL;
	}
	if (list_is_empty(json_data)) {
		fprintf(stderr, "Warning  ==>  %s() argument #1 (json_data) is an empty list.\n", __func__);
		return NULL;
	}
	if (! member_name) {
		fprintf(stderr, "ERROR  ==>  %s() argument #2 (member_name) is NULL.\n", __func__);
		return NULL;
	}
	if (strlen(member_name) > MAX_ARGUMENT_STRING_LENGTH) {
		fprintf(stderr, "Warning  ==>  %s() excessive length argument #2 (%ld bytes)\n"\
						"%s\nDid you forget to terminate the string with a nul ?.\n",
						__func__, strlen(member_name), member_name);
		return NULL;
	}
	if ((minimum_value == 0L) && (maximum_value == 0L)) {
		fprintf(stderr, "ERROR  ==>  %s() called with zero in arguments #4 & 5.\n", __func__);
		return NULL;
	}
	if ((value_range) && (minimum_value >= maximum_value)) {
		fprintf(stderr, "ERROR  ==>  %s() arguments #4 (%Lf) must not be larger than argument "\
						"#5 (%Lf) in a range search.\n", __func__, minimum_value, maximum_value);
		return NULL;
	}
	if ((value_range) && (minimum_value == maximum_value)) {
		/* I know this is not an exact floating point equality evaluation, but if they are not exact, i want to do a range search */
		fprintf(stderr, "Warning  ==>  %s() arguments #4 & 5 (%Lf) are equal in a range "\
						"search, range ignored.\n", __func__, minimum_value);
		range_value = FALSE;
	}

	/* Set up this search group */
	if (! (results = (list_object *)_json_malloc(sizeof(list_object), __func__))) return NULL;

	list_create(results, LIST_COPY_DATA, json_member_find_size_function, LIST_SIZE_TINY);
	list_set_remove_function(results, json_member_list_remove_function);

	ret = _json_find_members_full_iterate(	results,
											NULL,
											json_data,
											-1,
											member_name,
											partial_name,
											NULL,
											0,
											0,
											minimum_value,
											maximum_value,
											range_value,
											invalid_empty);
	
	/* examine the results */
	if (ret) {
		fprintf(stderr, "ERROR  ==>  %s() *** internal error occurred *** (%d)\n.", __func__, ret);
	}
	if ((ret) || list_is_empty(results)) {
		list_erase(results);
		_json_free_(results);
		return NULL;	
	}
					
	return results;
}

/* Using the provided search criteria, search the JSON data and locate all object
	members in the list hierarchy that match the criteria, returning them in a
	new list_object to the caller.
	
	This function can only be called to search JSON Object Member names, and a 
	value type criteria as well. To search for JSON Object Member names alone 
	please refer to the function json_find_members_value_string() above.
	
	For example, to search for all Object Members in the list hierarchy for a 
	match with the name "init-param" and the value type JSON_object could return
	such Members as 	"init-param": {

	Upon return if Object Members were found that matched the search criteria,
	a list containing elements of the type json_member_list_t would be provided
	one element in the list for each match found. When the caller has finished
	with the list, it must erased with list_erase() and then freed. 
	
	json_data		A D-List list_object that describes the JSON data to be searched.
					This can be a root list_object, or it can be any subordinate 
					list_object within that hierarchy. It can be an Array or an Object
					list. The search is confined to the list(s) covered by this top 
					level list_object.
	member_name		Pointer to a nul terminated string to use as a match against the
					JSON Object Member names in the JSON data list.
	partial_name	TRUE	perform the search using member_name as a partial name
							match,
					FALSE 	perform the search looking only for a complete match with
							member_name.
	value_type		A JSON value type to be used as a direct match against the JSON 
					Object Member number value types in the JSON data list. Only JSON
					Object Members that have already matched with member_name, will be
					evaluated against supplied JSON value type. This can be ay valid
					JSON value type as defined in this header file above.
	
	returns		list_object	contains the JSON Object Members found that matched the
							search criteria. This list was created with LIST_COPY_DATA
							option, so element's storage will be managed as described
							by D-List depending on what functions you use to access or
							manage this list. list_erase() will release ay internal and
							element storage remaining when called.
				NULL		No JSON Object Members were found that matched the search 
							criteria. No list_object was created.						*/
list_object *json_find_members_value_type(	list_object *json_data, 
											const char *member_name, 
											const boolean partial_name,
											const JSON_value_types value_type) {

	list_object *results;
	int ret = 0;
	
	/* Verify Arguments */
	if (! json_data) {
		fprintf(stderr, "ERROR  ==>  %s() argument #1 (json_data) is NULL.\n", __func__);
		return NULL;
	}
	if (list_is_empty(json_data)) {
		fprintf(stderr, "Warning  ==>  %s() argument #1 (json_data) is an empty list.\n", __func__);
		return NULL;
	}
	if (! member_name) {
		fprintf(stderr, "ERROR  ==>  %s() argument #2 (member_name) is NULL.\n", __func__);
		return NULL;
	}
	if (strlen(member_name) > MAX_ARGUMENT_STRING_LENGTH) {
		fprintf(stderr, "Warning  ==>  %s() excessive length argument #2 (%ld bytes)\n"\
						"%s\nDid you forget to terminate the string with a nul ?.\n",
						__func__, strlen(member_name), member_name);
		return NULL;
	}
	if ((value_type < JSON_object) || (value_type > JSON_string)) {
		fprintf(stderr, "ERROR  ==>  %s() called with invalid argument #4.\n", __func__);
		return NULL;
	}

	/* Set up this search group */
	if (! (results = (list_object *)_json_malloc(sizeof(list_object), __func__))) return NULL;

	list_create(results, LIST_COPY_DATA, json_member_find_size_function, LIST_SIZE_TINY);
	list_set_remove_function(results, json_member_list_remove_function);

	ret = _json_find_members_full_iterate(	results,
											NULL,
											json_data,
											-1,
											member_name,
											partial_name,
											NULL,
											0,
											0,
											0,
											0,
											FALSE,
											value_type);
	
	/* examine the results */
	if (ret) {
		fprintf(stderr, "ERROR  ==>  %s() *** internal error occurred *** (%d)\n.", __func__, ret);
	}
	if ((ret) || list_is_empty(results)) {
		list_erase(results);
		_json_free_(results);
		return NULL;	
	}
					
	return results;
}





/* Given an unknown list_object, provide the JSON value type this list_object
	represents. The result can either be JSON_object, JSON_array or JSON_value.
	
	See the JSON Parser documentation for the meanings of these types and how
	to apply them to your work managing JSON data hierarchy structures.
	
	This function cannot tell the difference between a JSON Array with one item
	in it, and a JSON data block with one value in it, unless the list_object
	presented is one level higher. So if the JSON data block presented by the 
	list_object pointer is either:	[ 72 ]   or   72	this function will
	return the same result JSON_value. If the list_object pointer is a level
	higher and has a pointer that Array, then the correct result will be 
	returned. In all cases where this function may be used, this issue has
	no consequences, but the caller should be aware of it. 
	
	If you were originally provided with a JSON value type for a list_object
	you now hold, you should always keep track of what it was.
	
	list_to_check	The list_object you need to find out the JSON type of
	
	returns		A valid JSON type, or invalid_empty if the list_object was not
				of a JSON_type															*/	
JSON_value_types json_report_list_object_type(list_object *list_to_check) {

	json_element_t *first_element;

	if (list_is_empty(list_to_check)) {
		return invalid_empty;
	}

	first_element = list_get_index(list_to_check, 0);
	
	if (! first_element) {
		return invalid_empty;								/* this is actually an error */
	}
	if (first_element->member_name) {
		return JSON_object;
	}	
	if ((list_size(list_to_check) == 1) && ((first_element->value_type != JSON_object) && (first_element->value_type != JSON_array))) {
		return JSON_value;
	}
	
	return JSON_array;
}

/* Add a new Member with a string value to a JSON Object, defined by
	the list_object provided.

	This function will add a new JSON Member to the provided list_object
	at the last position of that list. 
	
	If there are already elements in this list_object, then they are used to
	confirm that the list_object requested, is actually a JSON Object. If this
	is the first element to be added, then as no check can be made, it is 
	assumed the caller is providing a Member to a valid JSON list type. If the
	list type is incorrect, an error will be thrown up when the second element
	is added.
	
	list_to_add		the list_object of the JSON data hierarchy to add the new
					Member to.
	member_name		pointer to a nul terminated string to use as the JSON Object 
					Member Name, this argument is required both by this function, 
					and by JSON Standards RFC 8259.
	string_value	pointer the a nul terminated string to use as the Member
					Value
	
	returns		LIST_OK				success, new element added to the list
				LIST_ERR			on error	 										*/
int json_list_object_member_add_string(	list_object *list_to_add, 
										const char *member_name,
										const char *string_value) {
	
	/* Verify Arguments */
	if (! list_to_add) {
		fprintf(stderr, "ERROR  ==>  %s() argument #1 (list_to_add) is NULL.\n", __func__);
		return LIST_ERR;
	}
	if (! member_name) {
		fprintf(stderr, "ERROR  ==>  %s() argument #2 (member_name) is NULL.\n", __func__);
		return LIST_ERR;
	}
	if (! string_value) {
		fprintf(stderr, "ERROR  ==>  %s() argument #3 (string_value) is NULL.\n", __func__);
		return LIST_ERR;
	}

	/* verify this list_object is really a JSON Object */
	if (_json_test_add_member_to_array(list_to_add, __func__, member_name, FALSE) != LIST_OK) {
		fprintf(stderr, "ERROR  ==>  %s() was called to add a JSON Member with a string value "\
						"\"%s\": \"%s\"  to a JSON Array, instead of a JSON Object.\n",
						__func__, member_name, string_value);
		return LIST_ERR;
	}

	/* Do the actual work */
	return (_json_list_object_element_add(	list_to_add,
											member_name,
											JSON_string,
											string_value,
											0,
											0,
											0,
											NULL));
}	

/* Add a new Member with a number value to a JSON Object, defined by
	the list_object provided.

	This function will add a new JSON Member to the provided list_object
	at the last position of that list. 
	
	Only one value item can be provided, either an integer or a
	fractional number. The arguments take the following precedence,
		
		If double_number == 0, integer_number is used
		else double_number is used.
	
	which means if the new Member has a value of zero, it will always be added
	as an integer value.
	
	If there are already elements in this list_object, then they are used to
	confirm that the list_object requested, is actually a JSON Object. If this
	is the first element to be added, then as no check can be made, it is 
	assumed the caller is providing a Member to a valid JSON list type. If the
	list type is incorrect, an error will be thrown up when the second element
	is added.
	
	list_to_add		the list_object of the JSON data hierarchy to add the new
					Member to.
	member_name		pointer to a nul terminated string to use as the JSON Object 
					Member Name, this argument is required both by this function, 
					and by JSON Standards RFC 8259.
	integer_number	used as the Member value only if a) string_value is NULL, and
					b) double_number == 0, otherwise it is ignored
	double_number	used as the Member value only if a) string_value is NULL, and
					b) this argument is non-zero, otherwise it is ignored
	
	returns		LIST_OK				success, new element added to the list
				LIST_ERR			on error	 										*/
int json_list_object_member_add_number(	list_object *list_to_add, 
										const char *member_name,
										const json_integer_t integer_number,
										const json_double_t double_number) {

	/* Verify Arguments */
	if (! list_to_add) {
		fprintf(stderr, "ERROR  ==>  %s() argument #1 (list_to_add) is NULL.\n", __func__);
		return LIST_ERR;
	}
	if (! member_name) {
		fprintf(stderr, "ERROR  ==>  %s() argument #2 (member_name) is NULL.\n", __func__);
		return LIST_ERR;
	}

	/* verify this list_object is really a JSON Object */
	if (_json_test_add_member_to_array(list_to_add, __func__, member_name, FALSE) != LIST_OK) {
		if (double_number == 0) {
			fprintf(stderr, "ERROR  ==>  %s() was called to add a JSON Member with a number value "\
							"\"%s\": \"%" PRId64 "\"  to a JSON Array, instead of a JSON Object.\n",
							__func__, member_name, integer_number);
		} else {
			fprintf(stderr, "ERROR  ==>  %s() was called to add a JSON Member with a number value "\
							"\"%s\": \"%Lf\"  to a JSON Array, instead of a JSON Object.\n",
							__func__, member_name, double_number);
		}
		return LIST_ERR;
	}

	/* Do the actual work */
	return (_json_list_object_element_add(	list_to_add,
											member_name,
											(double_number == 0) ? JSON_integer : JSON_double,
											NULL,
											integer_number,
											double_number,
											0,
											NULL));
}

/* Add a new Member with a exponential number represented as a string value 
	(instead of a binary floating number) to a JSON Object, defined by
	the list_object provided.

	This function will add a new JSON Member to the provided list_object
	at the last position of that list. 
	
	As the basic C language has limited ability to properly represent very high
	precision exponential numbers, and this JSON package was designed to be 
	highly portable, only requiring a minimum of C99 to compile and run, it uses
	the base forms of the language for numbers, and does not rely on any external
	or optional libraries to handle these numbers. For some users either the 
	limited precision of these forms, or their inability to hold huge numbers 
	will be an issue. So this JSON package provides an option for both the JSON
	Parser and the JSON Generator to handle these numbers in a textural form 
	instead. This functions allows you to add such a number form to a JSON data 
	structure, for later output into a JSON data block using the JSON Generator.
	
	If there are already elements in this list_object, then they are used to
	confirm that the list_object requested, is actually a JSON Object. If this
	is the first element to be added, then as no check can be made, it is 
	assumed the caller is providing a Member to a valid JSON list type. If the
	list type is incorrect, an error will be thrown up when the second element
	is added.
	
	list_to_add		the list_object of the JSON data hierarchy to add the new
					Member to.
	member_name		pointer to a nul terminated string to use as the JSON Object 
					Member Name, this argument is required both by this function, 
					and by JSON Standards RFC 8259.
	float_str		pointer the a nul terminated string to use as the Member Value
	
	returns		LIST_OK				success, new element added to the list
				LIST_ERR			on error	 										*/
int json_list_object_member_add_float_str(	list_object *list_to_add, 
											const char *member_name,
											const char *float_str) {
	
	/* Verify Arguments */
	if (! list_to_add) {
		fprintf(stderr, "ERROR  ==>  %s() argument #1 (list_to_add) is NULL.\n", __func__);
		return LIST_ERR;
	}
	if (! member_name) {
		fprintf(stderr, "ERROR  ==>  %s() argument #2 (member_name) is NULL.\n", __func__);
		return LIST_ERR;
	}
	if (! float_str) {
		fprintf(stderr, "ERROR  ==>  %s() argument #3 (float_str) is NULL.\n", __func__);
		return LIST_ERR;
	}

	/* verify this list_object is really a JSON Object */
	if (_json_test_add_member_to_array(list_to_add, __func__, member_name, FALSE) != LIST_OK) {
		fprintf(stderr, "ERROR  ==>  %s() was called to add a JSON Member with a float_str value "\
						"\"%s\": \"%s\"  to a JSON Array, instead of a JSON Object.\n",
						__func__, member_name, float_str);
		return LIST_ERR;
	}

	/* Do the actual work */
	return (_json_list_object_element_add(	list_to_add,
											member_name,
											JSON_float_str,
											float_str,
											0,
											0,
											0,
											NULL));
}	

/* Add a new Member with a boolean value to a JSON Object, defined by 
	the list_object provided.

	This function will add a new JSON Member to the provided list_object
	at the last position of that list. 
	
	This function only supports adding Members with the value type boolean. 
	
	If there are already elements in this list_object, then they are used to
	confirm that the list_object requested, is actually a JSON Object. If this
	is the first element to be added, then as no check can be made, it is 
	assumed the caller is providing a Member to a valid JSON list type. If the
	list type is incorrect, an error will be thrown up when the second element
	is added.
	
	list_to_add		the list_object of the JSON data hierarchy to add the new
					Member to.
	member_name		pointer to a nul terminated string to use as the JSON Object 
					Member Name, this argument is required both by this function, 
					and by JSON Standards RFC 8259.
	boolean_value	TRUE or FALSE, to use as the Member Value
	
	returns		LIST_OK				success, new element added to the list
				LIST_ERR			on error	 										*/
int json_list_object_member_add_boolean(	list_object *list_to_add, 
											const char *member_name,
											const boolean boolean_value) {

	/* Verify Arguments */
	if (! list_to_add) {
		fprintf(stderr, "ERROR  ==>  %s() argument #1 (list_to_add) is NULL.\n", __func__);
		return LIST_ERR;
	}
	if (! member_name) {
		fprintf(stderr, "ERROR  ==>  %s() argument #2 (member_name) is NULL.\n", __func__);
		return LIST_ERR;
	}

	/* verify this list_object is really a JSON Object */
	if (_json_test_add_member_to_array(list_to_add, __func__, member_name, FALSE) != LIST_OK) {
		fprintf(stderr, "ERROR  ==>  %s() was called to add a JSON Member with a boolean value "\
						"\"%s\": \"%s\"  to a JSON Array, instead of a JSON Object.\n",
						__func__, member_name, (boolean_value) ? "true" : "false");
		return LIST_ERR;
	}
	
	/* Do the actual work */
	return (_json_list_object_element_add(	list_to_add,
											member_name,
											JSON_boolean,
											NULL,
											0,
											0,
											boolean_value,
											NULL));
}

/* Add a new Member with a null value to a JSON Object, defined by 
	the list_object provided.

	This function will add a new JSON Member to the provided list_object
	at the last position of that list. 
	
	This function only supports adding Members with the value type null.
	
	If there are already elements in this list_object, then they are used to
	confirm that the list_object requested, is actually a JSON Object. If this
	is the first element to be added, then as no check can be made, it is 
	assumed the caller is providing a Member to a valid JSON list type. If the
	list type is incorrect, an error will be thrown up when the second element
	is added.
	
	list_to_add		the list_object of the JSON data hierarchy to add the new
					Member to.
	member_name		pointer to a nul terminated string to use as the JSON Object 
					Member Name, this argument is required both by this function, 
					and by JSON Standards RFC 8259.
	
	returns		LIST_OK				success, new element added to the list
				LIST_ERR			on error	 										*/
int json_list_object_member_add_null(	list_object *list_to_add, 
										const char *member_name) {

	/* Verify Arguments */
	if (! list_to_add) {
		fprintf(stderr, "ERROR  ==>  %s() argument #1 (list_to_add) is NULL.\n", __func__);
		return LIST_ERR;
	}
	if (! member_name) {
		fprintf(stderr, "ERROR  ==>  %s() argument #2 (member_name) is NULL.\n", __func__);
		return LIST_ERR;
	}

	/* verify this list_object is really a JSON Object */
	if (_json_test_add_member_to_array(list_to_add, __func__, member_name, FALSE) != LIST_OK) {
		fprintf(stderr, "ERROR  ==>  %s() was called to add a JSON Member with a null value "\
						"\"%s\": null  to a JSON Array, instead of a JSON Object.\n",
						__func__, member_name);
		return LIST_ERR;
	}

	/* Do the actual work */
	return (_json_list_object_element_add(	list_to_add,
											member_name,
											JSON_null,
											NULL,
											0,
											0,
											0,
											NULL));
}

/* Add a new Member with a JSON Array or JSON Object value to an existing
	JSON Object, defined by the list_object provided.

	This function will add a new JSON Member to the provided list_object
	at the last position of that list, this Member will not have a data
	value, but will instead open either a new Array of values, or a new
	JSON Object, with the Member Name attached. 
	
	So either { "member_name" : { } }  or  { "member_name" : [ ] } 
	
	The caller must have previously created the new list_object to add as a
	Member to this JSON Object, and provide it in argument #3 (new_list).
	
	If there are already elements in this list_object, then they are used to
	confirm that the list_object requested, is actually a JSON Object. If this
	is the first element to be added, then as no check can be made, it is 
	assumed the caller is providing a Member to a valid JSON list type. If the
	list type is incorrect, an error will be thrown up when the second element
	is added.
	
	list_to_add		the list_object of the JSON data hierarchy to add the new
					Member to.
	member_name		pointer to a nul terminated string to use as the JSON Object 
					Member Name, this argument is required both by this function, 
					and by JSON Standards RFC 8259.
	new_list		the list_object of the new JSON Array or Object to add to
					the existing JSON Object as a new Member.
	list_type		must be either JSON_object or JSON_array.
	
	returns		LIST_OK				success, new element added to the list
				LIST_ERR			on error	 										*/
int json_list_object_member_add_list(	list_object *list_to_add, 
										const char *member_name,
										list_object *new_list,
										const JSON_value_types list_type) {

	/* Verify Arguments */
	if (! list_to_add) {
		fprintf(stderr, "ERROR  ==>  %s() argument #1 (list_to_add) is NULL.\n", __func__);
		return LIST_ERR;
	}
	if (! member_name) {
		fprintf(stderr, "ERROR  ==>  %s() argument #2 (member_name) is NULL.\n", __func__);
		return LIST_ERR;
	}
	if (! new_list) {
		fprintf(stderr, "ERROR  ==>  %s() argument #3 (new_list) is NULL.\n", __func__);
		return LIST_ERR;
	}
	
	if ((list_type != JSON_object) && (list_type != JSON_array)) {
		fprintf(stderr, "ERROR  ==>  %s() invalid list_type value (%u) must be either %u "\
						"or %u.\n", __func__, list_type, JSON_object, JSON_array);
		return LIST_ERR;
	}

	/* verify this list_object is really a JSON Object */
	if (_json_test_add_member_to_array(list_to_add, __func__, member_name, FALSE) != LIST_OK) {
		fprintf(stderr, "ERROR  ==>  %s() was called to add a JSON Member with an %s value "\
						"\"%s\": \"%s\"  to a JSON Array, instead of a JSON Object.\n",
						__func__, member_name, 
						(list_type == JSON_object) ? "Object" : "Array",
						(list_type == JSON_object) ? "{ ... }" : "[ ... ]");
		return LIST_ERR;
	}

	/* Do the actual work */
	return (_json_list_object_element_add(	list_to_add,
											member_name,
											list_type,
											NULL,
											0,
											0,
											0,
											new_list));
}


/* Add a new String Value to a JSON Array, defined by the list_object provided.

	This function will add a new JSON Value to the provided list_object
	at the last position of that list. 
	
	If there are already elements in this list_object, then they are used to
	confirm that the list_object requested, is actually a JSON Array. If this
	is the first element to be added, then as no check can be made, it is 
	assumed the caller is providing a Member to a valid JSON list type. If the
	list type is incorrect, an error will be thrown up when the second element
	is added.
	
	list_to_add		the list_object of the JSON data hierarchy to add the new
					Member to.
	string_value	pointer the a nul terminated string to use as the Array 
					Value
	
	returns		LIST_OK				success, new element added to the list
				LIST_ERR			on error	 										*/
int json_list_object_array_value_add_string(	list_object *list_to_add, 
												const char *string_value) {
	
	/* Verify Arguments */
	if (! list_to_add) {
		fprintf(stderr, "ERROR  ==>  %s() argument #1 (list_to_add) is NULL.\n", __func__);
		return LIST_ERR;
	}
	if (! string_value) {
		fprintf(stderr, "ERROR  ==>  %s() argument #2 (string_value) is NULL.\n", __func__);
		return LIST_ERR;
	}

	/* verify this list_object is really a JSON Array */
	if (_json_test_add_value_to_object(list_to_add, __func__, FALSE) != LIST_OK) {
		fprintf(stderr, "ERROR  ==>  %s() was called to add a JSON string value \"%s\" to "\
						"a JSON Object, instead of a JSON Array.\n", __func__, string_value);
		return LIST_ERR;
	}

	/* Do the actual work */
	return (_json_list_object_element_add(	list_to_add,
											NULL,
											JSON_string,
											string_value,
											0,
											0,
											0,
											NULL));
}	

/* Add a new Number Value to a JSON Array, defined by the list_object provided.

	This function will add a new Array Value to the provided list_object
	at the last position of that list. 
	
	One or more values can be provided, either an integer or a
	fractional number. The arguments take the following precedence,
		
		If double_number[0] == 0, integer_number is used
		else double_number is used.
	
	which means if the new Number Value has is zero, it will always be added
	as an integer value. Only the first entry of the input array is checked.
	
	If there are already elements in this list_object, then they are used to
	confirm that the list_object requested, is actually a JSON Array. If this
	is the first element to be added, then as no check can be made, it is 
	assumed the caller is providing a Value to a valid JSON list type. If the
	list type is incorrect, an error will be thrown up when the second element
	is added.
	
	list_to_add		the list_object of the JSON data hierarchy to add the new
					Number Value to.
	integer_number	used as the Value only if  double_number[0] == 0, 
					otherwise it is ignored
	double_number	used as the Value only if adouble_number[0] != 0,
					otherwise it is ignored
	item_count		The number of these items to add to the JSON Array. It is
					up to the caller to ensure the count and actual number of
					values in the input array are consistent.
	
	returns		LIST_OK				success, new element(s) added to the list
				LIST_ERR			on error	 										*/
int json_list_object_array_value_add_number(	list_object *list_to_add, 
												const json_integer_t *integer_number,
												const json_double_t *double_number,
												const unsigned int item_count) {
	int i = 0;
	int ret = LIST_OK;

	/* Verify Arguments */
	if (! list_to_add) {
		fprintf(stderr, "ERROR  ==>  %s() argument #1 (list_to_add) is NULL.\n", __func__);
		return LIST_ERR;
	}
	if ((integer_number) && (double_number)) {
		fprintf(stderr, "ERROR  ==>  %s() both integer and fractional number values provided.\n", __func__);
		return LIST_ERR;
	}
	if ((! integer_number) && (! double_number)) {
		fprintf(stderr, "ERROR  ==>  %s() neither integer and fractional number values provided, "\
						"one is required.\n", __func__);
		return LIST_ERR;
	}
	
	/* verify this list_object is really a JSON Array */
	if (_json_test_add_value_to_object(list_to_add, __func__, FALSE) != LIST_OK) {
		if (integer_number) {
			fprintf(stderr, "ERROR  ==>  %s() was called to add %u JSON number values (first %" PRId64 ") "\
							"to a JSON Object, instead of a JSON Array.\n", __func__, item_count, *integer_number);
		} else {
			fprintf(stderr, "ERROR  ==>  %s() was called to add %u JSON number values (first %Lf) "\
							"to a JSON Object, instead of a JSON Array.\n", __func__, item_count, *double_number);
		}
		return LIST_ERR;
	}
	if ((item_count < 1) || (item_count > MAX_ARGUMENT_STRING_LENGTH)) {
		fprintf(stderr, "ERROR  ==>  %s() was called to add %u JSON number values to a JSON Array, "\
						"must be between 1 and %u.\n", __func__, item_count, MAX_ARGUMENT_STRING_LENGTH);
		return LIST_ERR;
	}

	/* Do the actual work */
	do {
		_json_list_object_element_add(	list_to_add,
										NULL,
										(double_number == 0) ? JSON_integer : JSON_double,
										NULL,
										(integer_number) ? integer_number[i] : 0,
										(double_number) ? double_number[i] : 0,
										0,
										NULL);
	} while ((ret == LIST_OK) && (++i < item_count));

	return ret;
}

/* Add a new value with a exponential number represented as a string, 
	(instead of a binary floating number) to a JSON Object, defined by
	the list_object provided.

	This function will add a new JSON Value to the provided list_object
	at the last position of that list. 
	
	As the basic C language has limited ability to properly represent very high
	precision exponential numbers, and this JSON package was designed to be 
	highly portable, only requiring a minimum of C99 to compile and run, it uses
	the base forms of the language for numbers, and does not rely on any external
	or optional libraries to handle these numbers. For some users either the 
	limited precision of these forms, or their inability to hold huge numbers 
	will be an issue. So this JSON package provides an option for both the JSON
	Parser and the JSON Generator to handle these numbers in a textural form 
	instead. This functions allows you to add such a number form to a JSON data 
	structure, for later output into a JSON data block using the JSON Generator.
	
	If there are already elements in this list_object, then they are used to
	confirm that the list_object requested, is actually a JSON Array. If this
	is the first element to be added, then as no check can be made, it is 
	assumed the caller is providing a Member to a valid JSON list type. If the
	list type is incorrect, an error will be thrown up when the second element
	is added.
	
	list_to_add		the list_object of the JSON data hierarchy to add the new
					Member to.
	float_str		pointer the a nul terminated string to use as the Array 
					Value
	
	returns		LIST_OK				success, new element added to the list
				LIST_ERR			on error	 										*/
int json_list_object_array_value_add_float_str(	list_object *list_to_add, 
												const char *float_str) {
	
	/* Verify Arguments */
	if (! list_to_add) {
		fprintf(stderr, "ERROR  ==>  %s() argument #1 (list_to_add) is NULL.\n", __func__);
		return LIST_ERR;
	}
	if (! float_str) {
		fprintf(stderr, "ERROR  ==>  %s() argument #2 (float_str) is NULL.\n", __func__);
		return LIST_ERR;
	}

	/* verify this list_object is really a JSON Array */
	if (_json_test_add_value_to_object(list_to_add, __func__, FALSE) != LIST_OK) {
		fprintf(stderr, "ERROR  ==>  %s() was called to add a JSON string value \"%s\" to "\
						"a JSON Object, instead of a JSON Array.\n", __func__, float_str);
		return LIST_ERR;
	}

	/* Do the actual work */
	return (_json_list_object_element_add(	list_to_add,
											NULL,
											JSON_float_str,
											float_str,
											0,
											0,
											0,
											NULL));
}	

/* Add a new Boolean Value to a JSON Array, defined by the list_object provided.

	This function will add a new Value element to the provided list_object
	at the last position of that list. 
	
	This function only supports adding value of type boolean. 
	
	If there are already elements in this list_object, then they are used to
	confirm that the list_object requested, is actually a JSON Array. If this
	is the first element to be added, then as no check can be made, it is 
	assumed the caller is providing a Value to a valid JSON list type. If the
	list type is incorrect, an error will be thrown up when the second element
	is added.
	
	list_to_add		the list_object of the JSON data hierarchy to add the new
					value to.
	boolean_value	TRUE or FALSE, to use as the new Value
	
	returns		LIST_OK				success, new element added to the list
				LIST_ERR			on error	 										*/
int json_list_object_array_value_add_boolean(	list_object *list_to_add, 
												const boolean boolean_value) {

	/* Verify Arguments */
	if (! list_to_add) {
		fprintf(stderr, "ERROR  ==>  %s() argument #1 (list_to_add) is NULL.\n", __func__);
		return LIST_ERR;
	}

	/* verify this list_object is really a JSON Array */
	if (_json_test_add_value_to_object(list_to_add, __func__, FALSE) != LIST_OK) {
		fprintf(stderr, "ERROR  ==>  %s() was called to add a JSON boolean %s value to a "\
						"JSON Object, instead of a JSON Array.\n", __func__,
						(boolean_value) ? "true" : "false");
		return LIST_ERR;
	}
	
	/* Do the actual work */
	return (_json_list_object_element_add(	list_to_add,
											NULL,
											JSON_boolean,
											NULL,
											0,
											0,
											boolean_value,
											NULL));
}

/* Add a new null value to a JSON Array, defined by the list_object provided.

	This function will add a new value element to the provided list_object
	at the last position of that list. 
	
	This function only supports adding value type null.
	
	If there are already elements in this list_object, then they are used to
	confirm that the list_object requested, is actually a JSON Array. If this
	is the first element to be added, then as no check can be made, it is 
	assumed the caller is providing a value to a valid JSON list type. If the
	list type is incorrect, an error will be thrown up when the second element
	is added.
	
	list_to_add		the list_object of the JSON data hierarchy to add the new
					value to.
	
	returns		LIST_OK				success, new element added to the list
				LIST_ERR			on error	 										*/
int json_list_object_array_value_add_null(	list_object *list_to_add) {

	/* Verify Arguments */
	if (! list_to_add) {
		fprintf(stderr, "ERROR  ==>  %s() argument #1 (list_to_add) is NULL.\n", __func__);
		return LIST_ERR;
	}

	/* verify this list_object is really a JSON Array */
	if (_json_test_add_value_to_object(list_to_add, __func__, FALSE) != LIST_OK) {
		fprintf(stderr, "ERROR  ==>  %s() was called to add a JSON null value to a JSON "\
						"Object, instead of a JSON Array.\n", __func__);
		return LIST_ERR;
	}

	/* Do the actual work */
	return (_json_list_object_element_add(	list_to_add,
											NULL,
											JSON_null,
											NULL,
											0,
											0,
											0,
											NULL));
}








/* Add a new data value to an existing JSON Array, that has as a value type of 
	either a JSON Array or JSON Object, defined by the list_object provided.

	This function will add a new JSON value to the provided Array list_object
	at the last position of that list, this value will not have a data
	value, but will instead open either a new Array of values, or a new
	JSON Object. 
	
	So either [ ..., {  } ]  or  [ ..,  [ ] ] 
	
	The caller must have previously created the new list_object to add as the
	value element to this JSON Array, and provide it in argument #2 (new_list).
	
	If there are already elements in this list_object, then they are used to
	confirm that the list_object requested, is actually a JSON Array. If this
	is the first element to be added, then as no check can be made, it is 
	assumed the caller is providing a Member to a valid JSON list type. If the
	list type is incorrect, an error will be thrown up when the second element
	is added.
	
	list_to_add		the list_object of the JSON data hierarchy to add the new
					Member to.
	new_list		the list_object of the new JSON Array or Object to add to
					the existing JSON Object as a new Member.
	list_type		must be either JSON_object or JSON_array.
	
	returns		LIST_OK				success, new element added to the list
				LIST_ERR			on error	 										*/
int json_list_object_array_value_add_list(	list_object *list_to_add, 
											list_object *new_list,
											const JSON_value_types list_type) {

	/* Verify Arguments */
	if (! list_to_add) {
		fprintf(stderr, "ERROR  ==>  %s() argument #1 (list_to_add) is NULL.\n", __func__);
		return LIST_ERR;
	}
	if (! new_list) {
		fprintf(stderr, "ERROR  ==>  %s() argument #2 (new_list) is NULL.\n", __func__);
		return LIST_ERR;
	}
	if ((list_type != JSON_object) && (list_type != JSON_array)) {
		fprintf(stderr, "ERROR  ==>  %s() was called to with invalid list_type value (%u)"\
						"must be either %u or %u.\n", __func__, list_type, JSON_object, JSON_array);
		return LIST_ERR;
	}

	/* verify this list_object is really a JSON Array */
	if (_json_test_add_value_to_object(list_to_add, __func__, FALSE) != LIST_OK) {
		fprintf(stderr, "ERROR  ==>  %s() was called to add a JSON %s as an Array value, "\
						"to a JSON Object instead.\n", __func__, 
						(list_type == JSON_object) ? "Object": "Array");
		return LIST_ERR;
	}

	/* Do the actual work */
	return (_json_list_object_element_add(	list_to_add,
											NULL,
											list_type,
											NULL,
											0,
											0,
											0,
											new_list));
}






/*	======================================================================

	JSON Support Functions for Reporting hierarchical lists and JSON data.

		create character strings from data for display, convert paths, etc

	=======================================================================*/





/* Convert a JSON dotted notation path to a list_object for that path.
	
	This function performs the work of converting a textural JSON dotted 
	notation path, into first a json_member_list_t set of path segments
	and then returning the specific list_object needed to access the original
	item the JSON dotted notation path was meant to describe.
	
	If the path is invalid, or describes a path that does not exist, various
	error messages will be output to sdterr, and a NULL pointer is returned 
	to the caller.
	
	This function will only return a result for a path that ends with either
	a JSON Object or a JSON Array. If you wish to locate an individual Object
	Member that is a normal data value, or a normal value of an Array, you 
	must use either json or D-List functions on the owner list_object to do 
	that. This function is one of those that will return the list_object you
	would need to do that.
	
	The path is always followed strictly, if a segment of the path is not 
	found an error is produced and NULL returned. If you wish to have the path 
	converted by using the fuzzy search option available in the lower level 
	functions, you will have to write the calls to those functions yourself. It
	was felt that making the fuzzy search option available in a general purpose
	conversion tools was too error prone for users.
	
	If the JSON path you wish to locate is an absolute path, then you must also
	provide the root list_object for that JSON data block, or the search will 
	fail, as absolute paths must originate from the top of the JSON tree. This
	function makes no distinction between absolute and relative paths, but the
	starting point will always dictate how successful a search is going to be.

	The returned list object pointer can to be used as an argument in any of the 
	json_list_object_xxx() or json_find_xxx() functions, as well as any D-List 
	functions that you wish to use to access, manage, or update the data within
	that JSON Object or Array at the end of the original path.
	
	This function performs the same sequence a user would have to call, it 
	just removed the work from their code.
	
	json_data		A D-List list_object that describes the JSON data to be searched.
					This can be a root list_object, or it can be any subordinate 
					list_object within that hierarchy. It can be an Array or an Object
					list. The search is confined to the list(s) covered by this top 
					level list_object.
	path			pointer to a nul terminated character array that contains the
					JSON dotted notation path to be converted.
	returns		pointer to the list object that represents that JSON path object
				or NULL of there was an error, or the path was invalid.					*/
list_object *json_path_to_list_object(	list_object *json_data,
										const char *path) {

	json_member_list_t *parsed_result;
	list_object *result = NULL;

	if (! path) {
		fprintf(stderr, "ERROR  ==>  %s() NULL Path pointer.\n", __func__);
		return NULL;
	}

	parsed_result = json_parse_dotted_path(path);
	if (! parsed_result) {
		fprintf(stdout, "ERROR  ==>  %s() Unable to parse JSON Path (%s).\n", __func__, path);
	} else {
		/* Strict path, do not suppress warnings */
		if (json_path_locate(json_data, parsed_result, TRUE, FALSE) == NULL) {
			fprintf(stdout, "ERROR  ==>  %s() could not locate JSON Path (%s).\n", __func__, path);
		}
		if (parsed_result->json_member) {
			if ((parsed_result->value_type == JSON_object) || 
					(parsed_result->value_type == JSON_array)) {
				result = parsed_result->json_member->value.object;		/* if the path_locate failed, this is NULL */
			}
		}
		/* Clean up our garbage */
		_json_free_(parsed_result);
	}

	return result;
}
				

/* Using the internal representation of a JSON dotted notation path, trace the
	provided path and return the list_object or element found.
	
	You can specify that the JSON path be followed precisely to find the JSON
	element, or you can ask for the path to be following in a more fuzzy method.
	For example if you are not sure if the top level JSON Object at the
	top of your path is actually at the top, or inside a top level array, then
	choosing a fuzzy approach to following the path will allow this function to
	try a few options to find the top segment of the specified path. After the
	first segment of the specified path, it is always followed strictly.
	
	Following paths with a fuzzy approach, is the default.
	
	If the JSON path you wish to locate is an absolute path, then you must also
	provide the root list_object for that JSON data block, or the search will 
	fail, as absolute paths must originate from the top of the JSON tree. This
	function makes no distinction between absolute and relative paths, but the
	starting point will always dictate how successful a search is going to be.
	
	json_data		A D-List list_object that describes the JSON data to be searched.
					This can be a root list_object, or it can be any subordinate 
					list_object within that hierarchy. It can be an Array or an Object
					list. The search is confined to the list(s) covered by this top 
					level list_object.
	path_to_find	a json_member_list_t struct that contains the path to follow,
					it is required that both .path_segments and .path[] fields be
					correctly filled in prior to calling this function. All the 
					other fields of the json_member_list_t struct are ignored upon
					entry, and upon return will be filled in with the information
					about the new element, and the parent list_object that new
					element now belongs to.
	strict_path		TRUE follow the provided path strictly, do not perform 
					additional searches if the top segment of the path is not
					found at the first level of the starting point.
	suppress_errors	TRUE do not output errors to stderr. FALSE output path and
					structure errors to stderr. Argument or coding errors will
					always be output to stderr.
	
	returns		Pointer to the updated json_member_list_t that was provided as an
				input parameter. If the element was not found, the path did not
				resolve to a findable element, then NULL is returned.					*/
json_member_list_t *json_path_locate(	list_object *json_data,
										json_member_list_t *path_to_find, 
										const boolean strict_path,
										const boolean suppress_errors) {

	list_object *search_list;
	list_object *next_search_list;
	list_object *results;
	json_element_t *segment_element;
	json_member_list_t *temp_element;
	element_reference *last_found_ref;
	char *path_string;
	
	int i = 0;								/* index the caller provided path segments */
	int ret;


	if (! json_data) {
		fprintf(stderr, "ERROR  ==>  %s() NULL list pointer.\n", __func__);
		return NULL;
	}
	if (! path_to_find) {
		fprintf(stderr, "ERROR  ==>  %s() NULL Path pointer.\n", __func__);
		return NULL;
	}
	if (path_to_find->path_segments == 0) {
		fprintf(stderr, "ERROR  ==>  %s() called with a zero segment count.\n", __func__);
		return NULL;
	}

	/* set up this search */
	path_string = json_generate_path_from_links(path_to_find->path, path_to_find->path_segments);
	search_list = next_search_list = json_data;
	segment_element = (json_element_t *)json_data;					/* just to give it a value for the first loop */

	/* process each path segment in order, either as an Array or an Object */
	while ((segment_element) && (i < path_to_find->path_segments)) {
		search_list = next_search_list;
		if (path_to_find->path[i].value_type == JSON_array) {
			/* debugging only --> display_parsed_json_object(search_list, path_to_find->path[i].link.member_name, path_to_find->path[i].value_type, FALSE); */

			/* first test that this list_object is actually an array */
			if (_json_test_add_value_to_object(search_list, __func__, TRUE) != LIST_OK) {
				if (! suppress_errors) {
					fprintf(stderr, "ERROR  ==>  %s() segment #%u of path (%s) specified an Array, "\
									"but it is actually an Object.\n.", __func__, i+1, path_string);
				}
				segment_element = NULL;
				continue;
			}

			/* deal with an array for this segment */ 
			segment_element = list_get_index_ref(search_list, path_to_find->path[i].link.array_index, &last_found_ref);
			if ((! segment_element) && (! suppress_errors)) {
					fprintf(stderr, "ERROR  ==>  %s() segment #%u of path (%s) specified an Array "\
									"index (%u) that does not exist.\n.", __func__, i+1, 
									path_string, path_to_find->path[i].link.array_index);
			}
		} else {
			/* debugging only -->  display_parsed_json_object(search_list, path_to_find->path[i].link.member_name, path_to_find->path[i].value_type, FALSE); */

			/* Means that path_to_find->path[i].value_type == JSON_object */
			/* first test that this list_object is actually an object */
			if (_json_test_add_member_to_array(search_list, __func__, path_to_find->path[i].link.member_name, TRUE) != LIST_OK) {
				if (! suppress_errors) {
					fprintf(stderr, "ERROR  ==>  %s() segment #%u (%s) of path (%s) specified an Object, "\
									"but it is actually an Array.\n.", __func__, i+1, path_to_find->path[i].link.member_name, path_string);
				}
				segment_element = NULL;
				continue;
			}

			/* deal with an object for this segment */ 
			segment_element = list_search(	search_list, 
											path_to_find->path[i].link.member_name,
											ENTIRE_LIST,
											TAIL_OF_LIST,
											&last_found_ref,
											NULL);
			if (! segment_element) {
				/* we didn't find it, do we look further ? */
				if  ((strict_path) || (i > 0)) {
					/* no we don't, path failure */
					if (! suppress_errors) {
						fprintf(stderr, "ERROR  ==>  %s() segment #%u of path (%s) specified an Object "\
										"Member (%s) that does not exist.\n.", __func__, i+1, 
										path_string, path_to_find->path[i].link.member_name);
					}
					continue;
				} else {
					/* yes we can try a little harder, let's see what the wheel of fortune has to offer */
					/* Set up this search results list */
					if (! (results = (list_object *)_json_malloc(sizeof(list_object), __func__))) return NULL;
					list_create(results, LIST_COPY_DATA, json_member_find_size_function, LIST_SIZE_TINY);
					list_set_remove_function(results, json_member_list_remove_function);
					/* search 4 levels deep for a match, that is this level and 3 down */
					ret = _json_find_members_full_iterate(	results,
															NULL,
															search_list,
															4,
															path_to_find->path[i].link.member_name,
															FALSE,
															NULL,
															0,
															0,
															0,
															0,
															FALSE,
															invalid_empty);
	
					/* examine the results of the iteration search */
					if (ret) {
						fprintf(stderr, "ERROR  ==>  %s() *** internal error occurred *** (%d)\n.", __func__, ret);
					}
					if ((ret) || list_is_empty(results)) {
						if (! suppress_errors) {
							fprintf(stderr, "ERROR  ==>  %s() segment #%u of path (%s) specified "\
											"an Object Member (%s) that does not exist, even at "\
											"lower levels than the starting point provided.\n.",
											__func__, i+1, 
											path_string, 
											path_to_find->path[i].link.member_name);
						}
						list_erase(results);
						_json_free_(results);
						continue;
					}
					/* we have a result 
					
						Now we scroll through each fuzzy match, add the remainder of
						the callers path to it, and recursively call ourself to see if
						it is a full path match. 
						This call requires strict path adherence.						*/
					list_iteration_start(results, TAIL_OF_LIST);
					while ((! segment_element) && (list_iteration_has_more(results))) {
						temp_element = list_iteration_get_next(results, NULL);
						temp_element = json_path_locate(	temp_element->json_owner_object,
															path_to_find,
															TRUE,
															suppress_errors);
						if (temp_element) {
							/* we found a complete path match from this result, 
								exit the search with the result */
							segment_element = temp_element->json_member;
							last_found_ref = temp_element->reference;
							search_list = temp_element->json_owner_object;
							i = path_to_find->path_segments;				/* force a quick exit */
						}
					}
					list_iteration_stop(results);
					list_erase(results);
					_json_free_(results);

					if ((!segment_element) && (! suppress_errors)) {
						fprintf(stderr, "ERROR  ==>  %s() segment #%u of path (%s) specified "\
										"an Object Member (%s) that does not exist, even after "\
										"finding possible matches at lower levels than the "\
										"starting point provided.\n.", __func__, i+1, 
										path_string, path_to_find->path[i].link.member_name);
					}


				}		/*		else -> if  ((strict_path) || (i > 0)) 		*/
			}		/*		if (! segment_element)	*/
		}		/*		else -> if (path_to_find->path[i].value_type == JSON_array) {		*/

		if (segment_element) {
			/* if this element is an Object or an Array, we load it to go to the next level down */
			if ((segment_element->value_type == JSON_object) || 
					(segment_element->value_type == JSON_array)) {
				next_search_list = segment_element->value.object;
			}
			/* it is a value item, this could be the end result */
		}
		++i;
	}		/*		while ((segment_element) && (i < path_to_find->path_segments))		*/

	/* Path search is complete */

	/* Clean up our resources */
	_json_free_(path_string);

	/* prepare the reply data and exit */
	if ((segment_element) && (i >= path_to_find->path_segments)) {
		/* we found what we were looking for */
		path_to_find->json_owner_object = search_list;
		path_to_find->reference = last_found_ref;
		path_to_find->json_member = segment_element;
		path_to_find->value_type = segment_element->value_type;

		return path_to_find;	
	}
	
	return NULL;
}

/* Create a set of links describing a JSON data path, by parsing a character 
	array with JSON dotted notion in it. 
	
	This function will output an array of json_path_t elements in an otherwise
	empty json_member_list_t element. The json_member_list_t can be used as 
	input to other json_xxx functions that require and use JSON paths.
	
	Although the JSON text path may have an absolute start (a $.xxx in most
	implementations) this function will strip the extraneous header off,
	and provide the Member Name and Array path links as they are found.
	
	e.g.	$.bakery.product[19]."curry pan"
	
	will convert to	4 link segments		{JSON_object, "bakery"}, 
										{JSON_object, "product"}, 
										{JSON_array, 19}, 
										{JSON_object, "curry pan"},

	
	dotted_path		a nul terminated string that contains the textural JSON Dotted
					Path to parse and translate into internal segment format.
	
	returns		Pointer		a new json_member_list_t record that provides the list
							of links segments that represents the input JSON Path.
							It is the callers responsibility to manage this storage
							and free it when finished with it.
 				NULL		Invalid path was provided.									*/
json_member_list_t *json_parse_dotted_path(const char *dotted_path) {

	json_member_list_t *result = NULL;
	json_path_t *path_link;
	list_object segments_list;
	char *cursor = (char *)dotted_path;
	json_path_t new_segment;
	char *string_cursor;
	char char_storage[32];
	unsigned int i;
	
	boolean ending_quote = FALSE;
	boolean still_working = TRUE;

	if (! dotted_path) {
		fprintf(stderr, "ERROR  ==>  %s() was called with a NULL pointer.\n", __func__);
		return NULL;
	}

	list_create(&segments_list, LIST_COPY_DATA, json_path_size_function, LIST_SIZE_TINY);

	/* parse the input string and create the segment tokens */
	while ((*cursor != '\0') && (still_working)) {
		switch (*cursor) {
			case '\"':
				++cursor;
				new_segment.value_type = JSON_object;
				ending_quote = TRUE;
				break;
		
			case '.':
				++cursor;
				if (*cursor == '[')  {
					fprintf(stderr, "ERROR  ==>  %s() invalid JSON Dotted Syntax, Object "\
									"separator before array subscript in input path, "\
									"starting with '%s'.\n\n", __func__, (cursor-1));
					list_set_remove_function(&segments_list, json_path_list_remove_function);
					list_erase(&segments_list);
					return NULL;
				}
				if (*cursor == '\"') {
					++cursor;
					ending_quote = TRUE;
				}
				new_segment.value_type = JSON_object;		
				break;
		
			case '$':
				++cursor;
				if (*cursor != '.')  {
					fprintf(stderr, "ERROR  ==>  %s() invalid JSON Dotted Syntax, Absolute "\
									"Path notation must be followed by a period, "\
									"starting with '%s'.\n\n", __func__, (cursor-1));
					list_set_remove_function(&segments_list, json_path_list_remove_function);
					list_erase(&segments_list);
					return NULL;
				}
				/* Nothing to do, let it slide past this */
				new_segment.value_type = invalid_empty;
				break;
			case '[':
				++cursor;
				new_segment.value_type = JSON_array;		
				break;
		
			default:
				/* probably the first object member name, if not, oh well ! */
				new_segment.value_type = JSON_object;		
				break;
		}
		
		if (new_segment.value_type == JSON_object) {
			string_cursor = cursor;
			while ((*cursor != '\0') && (*cursor != '\"') && (*cursor != '.') && (*cursor != '[')) ++cursor;
			if (((ending_quote) && (*cursor == '.')) || ((! ending_quote) && (*cursor == '\"')))  {
				fprintf(stderr, "ERROR  ==>  %s() unterminated string in input path, "\
								"starting with '%s'.\n\n", __func__, string_cursor);
				list_set_remove_function(&segments_list, json_path_list_remove_function);
				list_erase(&segments_list);
				return NULL;
			}
			new_segment.link.member_name = _json_create_string_n_copy(string_cursor, (cursor - string_cursor));
			if (new_segment.link.member_name) {
				list_append(&segments_list, &new_segment);
			} else { 
				fprintf(stderr, "Warning  ==>  %s() Ignored invalid JSON Dotted Syntax, "\
								"Zero Length Object Member Name, starting with '%s'.\n\n",
								__func__, (cursor-1));
			}
			if (! ending_quote) --cursor;
			ending_quote = FALSE;
		}

		if (new_segment.value_type == JSON_array) {
			string_cursor = char_storage;
			while ((*cursor != '\0') && (*cursor != ']') && (string_cursor - char_storage < 32)) *string_cursor++ = *cursor++;
			if (*cursor != ']')  {
				fprintf(stderr, "ERROR  ==>  %s() unterminated array subscript in input path, "\
								"starting with '%s'.\n\n", __func__, string_cursor);
				list_set_remove_function(&segments_list, json_path_list_remove_function);
				list_erase(&segments_list);
				return NULL;
			}
			*string_cursor = '\0';
			sscanf(char_storage, "%u", &new_segment.link.array_index);
			list_append(&segments_list, &new_segment);
		}

		++cursor;
	}

	/* create the output record from the segment tokens */
	if (! list_is_empty(&segments_list)) {
		if (! (result = (json_member_list_t*)_json_malloc((sizeof(json_member_list_t) + (sizeof(json_path_t) * (list_size(&segments_list)))), __func__))) {
			list_set_remove_function(&segments_list, json_path_list_remove_function);
			list_erase(&segments_list);
			return NULL;
		}
		result->json_owner_object = NULL;
		result->reference = NULL;
		result->json_member = NULL;
		result->value_type = JSON_object;
		result->path_segments = list_size(&segments_list);
		i = 0;
		list_iteration_start(&segments_list, TAIL_OF_LIST);
		while (list_iteration_has_more(&segments_list)) {
			path_link = list_iteration_get_next(&segments_list, NULL);
			result->path[i].value_type = path_link->value_type;
			if (path_link->value_type == JSON_object) {
				result->path[i].link.member_name = path_link->link.member_name;
			} else {
				result->path[i].link.array_index = path_link->link.array_index;
			}
			++i;
		}
		list_iteration_stop(&segments_list);
	}

	list_erase(&segments_list);

	return result;
}

/* Create a character array describing a set of links in a JSON Data block
	for reporting or printing purposes. The output will be relative to the
	starting point where the input link list was created, this function
	cannot tell if this is absolute or relative.
	
	e.g. will convert 4 link segments		{JSON_object, "bakery"}, 
											{JSON_object, "product"}, 
											{JSON_array, 19}, 
											{JSON_object, "curry pan"},
	
	to the JSON dotted notation 
									bakery.product[19]."curry pan"
	
	some notations do not have a period after an array, some do, this does.
	
	path			an array of json_path_t elements that describe the 
					JSON path to generate.
	count			the number of elements in the path.
	
	returns		pointer to character array containing the text generated.
				Caller is responsible for proper management and disposal
				of storage related to this character array.								*/
char *json_generate_path_from_links(	json_path_t path[], 
										unsigned int count) {

#define CHUNK_SIZE 			128

	char temp_buffer[32];
	char *output_buffer = NULL;
	unsigned int current_output_offset = 0;
	unsigned int current_buffer_size = 0;
	unsigned int i = 0;
	unsigned int size_of_chunk;
	char *cursor = NULL;
	char *temp_ptr;
	boolean first_object = TRUE;
	boolean needs_quotes;
	
	if (count == 0) {
		fprintf(stderr, "ERROR  ==>  %s() was called with a zero segment count.\n", __func__);
		return NULL;
	}
	if (! path) {
		fprintf(stderr, "ERROR  ==>  %s() NULL Path pointer.\n", __func__);
		return NULL;
	}
	if (path[0].link.member_name == NULL) {
		fprintf(stderr, "ERROR  ==>  %s() NULL segment 0 path pointer.\n", __func__);
		return NULL;
	}
	
	while (i < count) {
		if (path[i].value_type == JSON_array) {
			size_of_chunk = 10;
		} else {
			if (path[i].link.member_name) size_of_chunk = strlen(path[i].link.member_name)+2;
			else {
				fprintf(stderr, "ERROR  ==>  %s() Object Member for segment %u has a NULL "\
								"Member Name.\n\n", __func__, i);
				_json_free_(output_buffer);

				return NULL;
			}
		}
			
		/* First do we have enough space to add this new output to the buffer ? */
		if ((current_output_offset + size_of_chunk + 2) > current_buffer_size) {
			/* Add another chunk of free space to keep writing the output data block */
			cursor = NULL;
			output_buffer = realloc(output_buffer, current_buffer_size + CHUNK_SIZE);
			if (! output_buffer) {
				fprintf(stderr, "CRITICAL ERROR  ==>  %s() realloc() failed requesting %u bytes.\n\n",
						__func__, (current_buffer_size + CHUNK_SIZE));
				current_buffer_size = 0;
				return NULL;
			} else {
				current_buffer_size += CHUNK_SIZE;
			}
		}
	
		if (output_buffer) {
			cursor = output_buffer + current_output_offset;
			if (path[i].value_type == JSON_array) {
				sprintf(temp_buffer, "[%u]", path[i].link.array_index);
				temp_ptr = temp_buffer;
				while (*temp_ptr != '\0')  { *cursor++ = *temp_ptr++; }
			} else {
				temp_ptr = path[i].link.member_name;
				if (strpbrk(path[i].link.member_name, "' .[]")) {
					needs_quotes = TRUE;
				} else {
					needs_quotes = FALSE;
				}
				if (first_object) first_object = FALSE;
				else *cursor++ = '.';
				if (needs_quotes) *cursor++ = '\"';
				while (*temp_ptr != '\0') { 
					*cursor++ = *temp_ptr++; 
				}
				if (needs_quotes) *cursor++ = '\"';
			}
			current_output_offset += (cursor - (output_buffer + current_output_offset));
		}
		++i;
	}
	
	if (cursor) {
		/* Shrink Wrap the result, and ship it out */
		*(output_buffer + current_output_offset) = '\0';
		output_buffer = realloc(output_buffer, current_output_offset+1);
	}

	return output_buffer;
}

/* Display to stdout text describing a set of links in a JSON data block. 

	The displayed text is started exactly where stdout is currently, no
	newlines etc are printed, no newline at the end. stdout is not flushed.
	It is up to the caller to ensure placement in the output stream is 
	correct, and stdout is flushed to ensure output is complete.

	path			an array of json_path_t elements that describe the 
					JSON path to generate.
	count			the number of elements in the path.
	
	returns		Nothing.																*/
void json_print_path_from_links(	json_path_t path[], 
									unsigned int count) {
	
	char *output_buffer = NULL;
	
	output_buffer = json_generate_path_from_links(path, count);
	fprintf(stdout, "%s", output_buffer);
	
	_json_free_(output_buffer);
	
	return;
}



/* This function will either display all the elements in a single list_object, 
	or will walk an entire hierarchy of list_objects displaying the elements
	as it goes through them.
	
	This function is really only called for internal and user code debugging 
	purposes. It also serves as example code.											*/
void display_parsed_json_object(	list_object * json_object, 
									const char *list_name, 
									const int list_type, 
									const boolean dig_down) {

	json_element_t *element;

	fprintf(stdout, "\nContents of %s JSON Data Block %s.\n", 
					(dig_down) ? "Hierarchical" : "Root Level",
					list_name);

	list_iteration_start(json_object, TAIL_OF_LIST);
	while (list_iteration_has_more(json_object)) {
		element = list_iteration_get_next(json_object, NULL);
		switch (element->value_type) {
			case JSON_object:
				fprintf(stdout, "    New JSON Object Structure %s.\n", element->member_name);
				if (dig_down) display_parsed_json_object(element->value.object, element->member_name, JSON_object, TRUE);
				break;
			case JSON_array:
				fprintf(stdout, "    New JSON Array Structure %s.\n", 
								(element->member_name != NULL) ? element->member_name : "");
				if (dig_down) display_parsed_json_object(element->value.object, element->member_name, JSON_array, TRUE);
				break;
			case JSON_string:
				fprintf(stdout, "    String Element \"%s%s%s\"\n", 
						(element->member_name != NULL) ? element->member_name : "",
						(list_type == JSON_object) ? " : " : "",
						element->value.string);
				break;
			case JSON_integer:
				fprintf(stdout, "    Number (Integer) Element %s%s%" PRId64 "\n", 
						(element->member_name != NULL) ? element->member_name : "",
						(list_type == JSON_object) ? " : " : "",
						element->value.integer);
				break;
			case JSON_double:
				fprintf(stdout, "    Number (Double) Element %s%s%Lf\n", 
						(element->member_name != NULL) ? element->member_name : "",
						(list_type == JSON_object) ? " : " : "",
						element->value.double_number);
				break;
			case JSON_float_str:
				fprintf(stdout, "    Number String Element %s%s%s\n", 
						(element->member_name != NULL) ? element->member_name : "",
						(list_type == JSON_object) ? " : " : "",
						element->value.string);
				break;
			case JSON_boolean:
				fprintf(stdout, "    Boolean Element %s%s%s\n", 
						(element->member_name != NULL) ? element->member_name : "",
						(list_type == JSON_object) ? " : " : "",
						(element->value.boolean_value) ? "true" : "false");
				break;
			case JSON_null:
				fprintf(stdout, "    NULL Element %s%s\"null\"\n",
						(element->member_name != NULL) ? element->member_name : "",
						(list_type == JSON_object) ? " : " : "");
				break;
		
			default:
				fprintf(stdout, "** Unknown element %s : \"%s\"\n", element->member_name, element->value.string);
		}
	}
	list_iteration_stop(json_object);
	fprintf(stdout, "\nEnd display of JSON Data Block %s.\n", list_name);

	return;
}







/*	=========================================================================

		Section for the INTERNAL ONLY -- JSON Support Functions.
		
		All functions below here service the external JSON Support Functions

	=========================================================================	*/



/* This function will perform the actual json_find_members_xxxx searches.
	
	The user callable functions above just set up the data for the search operation
	and call this function to do the work. This function is reentrant and will
	recursively call itself to traverse the input list_object hierarchy until
	it has completed its search.
	
	It is up to the caller to wash the resultant list_object and return to the
	external caller of their function.
	
	This function will perform a complete iterative search for JSON Object Members
	that match the specified search criteria. It can search for partial or full
	Object Member Name only, or using Object Members found by name match can further
	refine the search with value string, or value type or number value or number
	value range searches.

	The precedence order of the arguments when deciding the search type are:
		
		MEMBER_SEARCH_NAME_ONLY		All optional arguments either NULL or 0
		MEMBER_SEARCH_VALUE_TYPE	value_type has a non-zero value
		MEMBER_SEARCH_STRING_VALUE	string_value is non-NULL
		MEMBER_SEARCH_DOUBLE_VALUE	(maximum_double_value != 0 || minimum_double_value !=0) && partial_or_range == FALSE
		MEMBER_SEARCH_IOUBLE_RANGE	(maximum_double_value != 0 || minimum_double_value !=0) && partial_or_range == TRUE
		MEMBER_SEARCH_INTEGER_VALUE	partial_or_range == FALSE
		MEMBER_SEARCH_INTEGER_RANGE	last choice
		
		For number searches (specific or inclusive range) the double values
		take precedence over the integer values. So if either minimum_double_value
		or maximum_double_value is non-zero, the values of minimum_int_value
		and maximum_int_value are ignored. Under this precedence order you
		will be unable to search for an Object Member that also has a value of
		0 (zero) as under those conditions the search would default to 
		MEMBER_SEARCH_NAME_ONLY instead.
		
	This function was not designed to be accessed by external callers. it performs
	no argument validity checking, and assumes the caller knows how to set up
	the argument properly. Do not expose this function outside the realm of this
	code module. It is designed to be quick and efficient, and avoid the repetitive
	checking that would be required if exposed beyond this code module.
	
	For more complete descriptions of the arguments and their possible conditions
	please refer to the functions above 
		json_find_members_value_string()
		json_find_members_value_type()
		json_find_members_value_integer()
		json_find_members_value_double()
		
	This function iterates each list_object found in the JSON Data hierarchy in 
	2 passes to keep related Object Members together in the results. It would be
	faster and more efficient to process each layer in 1 pass, but that would
	create a results list where related Object Members found would be separated
	in the list by results of subordinate list_objects. It is also much easier to
	track and return the path of search matches using recursion.
	
	This function will find members of JSON Objects and apply the search criteria
	to the direct member values only. To illustrate exactly what this means, take
	the example searching for Member name "fred" and value 10. If this member is 
	found it will be returned as a match
		{ "fred" : 10, ... }
	as the 10 is the direct value of the Member name we matched against. The
	following example would also return a match
		{ 	"bob" : 52,
			"fred" : [ 20, 52, 10, "a string value" ],
			"amy" : [ 15, 25, 42 ], ... } 
	as the value 10 was found as a direct value of "fred" even though the value is
	part of an array, that array is semantically a direct value of the member name.
	In the following example there would be no match made
		{ 	"bob" : 52,
			"fred" : [ 20, [ 52, 10], "a string value" ],
			"amy" : [ 15, 25, 42 ], ... }
	as in this case the value 10 is not a direct value of the Member name, but a
	value of an array which is a direct value. This general purpose search tool
	is limited to finding same level matches of values, and only the caller is
	going to understand the implied semantics of various obscure JSON data 
	layouts. It would be unreasonable to expect a general purpose search tool to
	be able to devine the semantics of such layouts or structures.
	
	Search Depth. The search can be conducted on the entire tree below the starting
	point provided, or can be limited in scope to a specific number of levels. A level
	would be a new Object, or an Array. A search depth of 1 level means the level of
	Object Members or Array Elements of which the staring point is the parent. 1 is
	the minimum size allowed. 0 is used to terminate a recursive search, and -1 means
	no limit.  An illustration of this is:
		{ 
		  "bakery": 
			{
			  "product":
			   [
				 {	"sku" 12054, "name": "Curry Pan", "price": 2.95, "category": "pastries" },
				 {	"sku" 10061, "name": "Apple Danish", "price": 3.45, "category": "pastries" },
				 {	"sku" 12054, "name": "Onion Bread", "price": 6.95, "category": "breads" },
					...
				]
			}
			"deli":
			{
			  "product":
			   [
				 {	"sku" 2145, "name": "Shrimp Tempura", "price": 4.95, "category": "cooked" }
				 {	"sku" 2578, "name": "Squid Tempura", "price": 3.45, "category": "cooked" }
				 {	"sku" 3652, "name": "Squid Salad", "price": 6.95, "category": "fresh" }
					...
				]
			}
		}
	if you wanted to search for "deli" as a Member name from the above starting point
	a search depth of 1 would find the match. However if you wanted to find "Curry Pan"
	then you would need to provide a search depth of at least 4 to find it. The reason
	for search depth is to limit the search for Members or values that are repeated in
	areas outside the interest of the search, this is especially useful if the starting
	point is the root of a very large JSON Data Hierarchy, although in the illustration
	above it has more limited use. It is also important to remember that nested Arrays
	are themselves a new level so something like [ 44, [ 55, 66, 77, [909, 808. 707], 45], 6842 ]
	requires some consideration if you are limiting search depth and want specific
	matches. Not everyone who writes JSON data blocks seems to be giving attention
	to how nasty the nesting can get, ARIN for example produces very convoluted JSON
	as part of its RDAP responses, where APNIC produces much cleaner layouts.
	
	Search depth is typically used where a search / find is trying to follow a path
	but is unsure if the Member at the top is in the first Object or enclosed in an
	Array of an Array of Objects. To provide a more useful response and be more flexible
	providing a search depth of 3 will catch most matches at the top of the tree, without
	returning irrelevant matches from some obscure data deep in the tree with the same
	Member name.

	Unlike the JSON Validator, Parser or Generator, these support functions are
	forgiving of syntax and value errors, these will continue to process their
	functions while reporting on, and skipping, erroneous elements in the JSON
	data hierarchical list_objects.
	
	results			pointer to a field where the address of the return list_object
					will be written. The will list all instances of matches of the
					requested search criteria. NULL means no matches found. Each
					search result will contain a pointer to the parent list_object
					of that match, the Member name of that Object Member, and an
					array of json_path_t elements that will descrive the JSON path
					of that match.
	path_scratchpad	for recursive calls only, not for external callers, must be NULL
					used to keep track of the current position in the path while 
					searching. Used to construct a path if a match is found.
	json_data		list_object that describes the starting point of the search
	search_depth	how deep should the search go, from the starting point. 
					-1	No limit, search the entire tree under the starting point
					0	No search to be conducted (this is not used in calls, 
						but is used by recursion to end a search down a branch)
					>0	the number of levels, under the starting point, to search. 
	member_name		Same meaning as the calling function has with this argument.
	partial_name	Same meaning as the calling function has with this argument.
	string_value	Same meaning as the calling function has with this argument.
	minimum_int_value
					Same meaning as the calling function has with this argument.
	maximum_int_value
					Same meaning as the calling function has with this argument.
	minimum_double_value
					Same meaning as the calling function has with this argument.
	maximum_double_value
					Same meaning as the calling function has with this argument.
	partial_or_range
					Same meaning as the calling function has with this argument.
	value_type		Same meaning as the calling function has with this argument.
	
	returns 	found search JSON Object member elements in the list_object 
				results, the found members being described using the struct
				json_member_list_t as the element layout for the results list.			*/	
static int _json_find_members_full_iterate(	list_object *results,
											list_object *path_scratchpad,
											list_object *json_data, 
											const int search_depth,
											const char *member_name, 
											const boolean partial_name,
											const char *string_value,
											const json_integer_t minimum_int_value,
											const json_integer_t maximum_int_value,
											const json_double_t minimum_double_value,
											const json_double_t maximum_double_value,
											const boolean partial_or_range,
											const JSON_value_types value_type) {

	json_member_list_t *found_member;

	json_element_t *temp_element;
	element_reference *last_found_ref;
	json_element_t *array_value;
	json_path_t *path_link;
	
	json_path_t	new_link;
	
	element_search *previous_function;
	list_object *current_path_scratchpad = path_scratchpad;


	int member_search_type;
	int member_search_status;
	int array_index;
	int ret = 0;
	int i;
	int current_search_depth = search_depth;
	
	boolean subordinates_found = FALSE;
	boolean add_this_element = FALSE;
	boolean member_name_match;

	/* First, are we actually searching at all ? */
	if (search_depth == 0) {
		return 0;
	}
	if (search_depth < 0) { current_search_depth = -1; }	/* prevent the counter from dropping infinitely */

	/* Next if this is the initial function call (the one at the top) then the 
		patch scratchpad has not be allocated yet.										*/
	if (! current_path_scratchpad) {
		if (! (current_path_scratchpad = (list_object *)_json_malloc(sizeof(list_object), __func__))) return -1;
		list_create(current_path_scratchpad, LIST_COPY_DATA, json_path_size_function, LIST_SIZE_TINY);
	}

	/* what are we searching for ? */
	if ((! string_value) 
			&& (minimum_int_value == 0) 
			&& (minimum_int_value == 0) 
			&& (minimum_double_value == 0) 
			&& (minimum_double_value == 0) 
			&& (value_type == invalid_empty)) {
		member_search_type = MEMBER_SEARCH_NAME_ONLY;
	} else {
		if (value_type != invalid_empty) {
			member_search_type = MEMBER_SEARCH_VALUE_TYPE;
		} else {
			if (string_value) {
				member_search_type = MEMBER_SEARCH_STRING_VALUE;
			} else {
				if ((minimum_double_value == 0) && (maximum_double_value == 0)) {
					/* check the ints */				
					if (partial_or_range) {
						member_search_type = MEMBER_SEARCH_INTEGER_RANGE;			
					} else {
						member_search_type = MEMBER_SEARCH_INTEGER_VALUE;			
					}
				} else {
					/* it is going to be a non-integer search */
					if (partial_or_range) {
						member_search_type = MEMBER_SEARCH_DOUBLE_RANGE;			
					} else {
						member_search_type = MEMBER_SEARCH_DOUBLE_VALUE;			
					}
				}
			}
		}
	}

/* still need to make MEMBER_SEARCH_NUMBER_RANGE work */

	/* This is a 2 pass setup, however the second pass can be skipped if no 
		subordinate lists are found, or pass 1 skipped if this is not a JSON
		Object Member list.
			Pass 1 - we process all the elements and save any elements that
						meet the specified search criteria. We process any
						subordinate list_object members for specific search
						criteria matches, but do not recursively search them,
						yet. This pass is skipped if it is not a JSON Object
						Member list_object.
			Pass 2 - we again process all the elements, but this time we only
						locate any subordinate list_objects which are then
						recursively searched for any search criteria matches
						of their elements instead. This pass is skipped if
						no qualifying subordinate list_objects were found
						in pass 1.												*/
	for (member_search_status = MEMBER_SEARCH_PASS_1; member_search_status != MEMBER_SEARCH_COMLETE; 
			member_search_status = ((member_search_status == MEMBER_SEARCH_PASS_1) ? MEMBER_SEARCH_PASS_2 : MEMBER_SEARCH_COMLETE)) {

		array_index = 0;
		list_iteration_start(json_data, TAIL_OF_LIST);
		while (list_iteration_has_more(json_data)) {
			/* This is where the real work gets done */
			add_this_element = FALSE;
			temp_element = list_iteration_get_next(json_data, &last_found_ref);
			/* first check the JSON Object Member Name - we must always have a match to do anything else */
			if ((member_search_status == MEMBER_SEARCH_PASS_1) && (! temp_element->member_name)) {
				/* Ah!, we must be in a JSON Array list_object, n ot an Object list_object
					There will be no Member name matches here.								*/
				member_search_status = MEMBER_SEARCH_PASS_2;
				subordinates_found = TRUE;						/* to ensure we get a second pass at this */
			}
			if (member_search_status == MEMBER_SEARCH_PASS_1) {
				if ((temp_element->value_type == JSON_object) || (temp_element->value_type == JSON_array))	{
					subordinates_found = TRUE;					/* Signal we need a pass #2 */
				}
				if (partial_name) {
					member_name_match = (strstr(temp_element->member_name, member_name) != NULL);
				} else {
					member_name_match = (strcmp(temp_element->member_name, member_name) == 0);
				}
				if (member_name_match) {
					/* on pass 1 we only process an element if there is an Object Name 
						match, otherwise it is ignored, */
					switch (member_search_type) {
						case MEMBER_SEARCH_NAME_ONLY:
							add_this_element = TRUE;
							break;

						/* when we perform searches, we do NOT assume the default search
							functions are present, instead we save the existing one, then
							attach the one we need to the list, use it, and restore the
							original by re-attaching it.								*/
						case MEMBER_SEARCH_STRING_VALUE:
							if (temp_element->value_type == JSON_string) {
								if (partial_or_range) {
									add_this_element = (strstr(temp_element->value.string, string_value) != NULL);
								} else {
									add_this_element = (strcmp(temp_element->value.string, string_value) == 0);
								}
							}
							if ((temp_element->value_type == JSON_array) && (temp_element->value.object)) {
								previous_function = list_get_search_function(temp_element->value.object);
								if (partial_or_range) 
									list_set_search_function(temp_element->value.object, default_search_function_array_partial_string_value);
								else
									list_set_search_function(temp_element->value.object, default_search_function_array_string_value);
								add_this_element = (list_simple_search(temp_element->value.object, string_value) != NULL);
								list_set_search_function(temp_element->value.object, (element_search)previous_function);
							}
							break;

						case MEMBER_SEARCH_INTEGER_VALUE:
							if ((temp_element->value_type == JSON_integer) && 
									(temp_element->value.integer == minimum_int_value)) {
								add_this_element = TRUE;
							}
							if ((temp_element->value_type == JSON_array) && (temp_element->value.object)) {
								previous_function = list_get_search_function(temp_element->value.object);
								list_set_search_function(temp_element->value.object, default_search_function_array_integer_value);
								add_this_element = (list_simple_search(temp_element->value.object, &minimum_int_value) != NULL);
								list_set_search_function(temp_element->value.object, (element_search)previous_function);
							}							
							break;

						case MEMBER_SEARCH_DOUBLE_VALUE:
							if ((temp_element->value_type == JSON_double) && 
									(default_search_function_array_double_value)(temp_element, &minimum_double_value)) {
								add_this_element = TRUE;
							}
							if ((temp_element->value_type == JSON_array) && (temp_element->value.object)) {
								previous_function = list_get_search_function(temp_element->value.object);
								list_set_search_function(temp_element->value.object, default_search_function_array_double_value);
								add_this_element = (list_simple_search(temp_element->value.object, &minimum_double_value) != NULL);
								list_set_search_function(temp_element->value.object, (element_search)previous_function);
							}
							break;

						case MEMBER_SEARCH_INTEGER_RANGE:
							if ((temp_element->value_type == JSON_integer) &&
									((minimum_int_value <= temp_element->value.integer ) &&
									(temp_element->value.integer <= maximum_int_value))) {
								add_this_element = TRUE;
							}
							if ((temp_element->value_type == JSON_array) && (temp_element->value.object)) {
								list_iteration_start(temp_element->value.object, TAIL_OF_LIST);
								while ((list_iteration_has_more(temp_element->value.object)) && (! add_this_element)) {
									array_value = list_iteration_get_next(temp_element->value.object, NULL);
									if ((array_value->value_type == JSON_integer) &&
											((minimum_int_value <= array_value->value.integer ) ||
											(array_value->value.integer <= maximum_int_value))) {
										add_this_element = TRUE;
									}
								}
								list_iteration_stop(temp_element->value.object);
							}							
							break;
		
						case MEMBER_SEARCH_DOUBLE_RANGE:
							if (temp_element->value_type == JSON_double) {
								if (((minimum_double_value <= temp_element->value.double_number ) &&
											(temp_element->value.double_number <= maximum_double_value))||
										/* above covered the range, these below cover the edge cases (both with a fudge factor) */
										(default_search_function_array_double_value)(temp_element, &minimum_double_value) ||
										(default_search_function_array_double_value)(temp_element, &maximum_double_value)) {
									add_this_element = TRUE;
								}
							}
							if ((temp_element->value_type == JSON_array) && (temp_element->value.object)) {
								list_iteration_start(temp_element->value.object, TAIL_OF_LIST);
								while ((list_iteration_has_more(temp_element->value.object)) && (! add_this_element)) {
									array_value = list_iteration_get_next(temp_element->value.object, NULL);
									if (array_value->value_type == JSON_double) {
										if (((minimum_double_value <= array_value->value.double_number ) &&
													(array_value->value.double_number <= maximum_double_value))||
												/* above covered the range, these below cover the edge cases (both with a fudge factor) */
												(default_search_function_array_double_value)(array_value, &minimum_double_value) ||
												(default_search_function_array_double_value)(array_value, &maximum_double_value)) {
											add_this_element = TRUE;
										}
									}
								}
								list_iteration_stop(temp_element->value.object);
							}							
							break;

						case MEMBER_SEARCH_NUMBER_RANGE:
							/* I had considered adding a search that would search all numbers
								for a range match together (integer and non-integer) but decided
								nobody but myself would probably want it. If there is a request, 
								i will add it, here. If not, this is a blank space.	
							int (*element_compare)(const void *element_a, const void *element_b);
							element_compare = default_compare_function_array_number_values;				-- dbw	*/
							break;

						case MEMBER_SEARCH_VALUE_TYPE:
							add_this_element = (temp_element->value_type == value_type);
							break;
		
						default:
							/* should not be possible to get here */
							break;
					}
				}		/*	if (member_name_match)	*/
				if (add_this_element) {
					if (! (found_member = (json_member_list_t*)_json_malloc((sizeof(json_member_list_t) + (sizeof(json_path_t) * (1 + list_size(current_path_scratchpad)))), __func__))) return -1;
					found_member->json_owner_object = json_data;
					found_member->json_member = temp_element;
					found_member->reference = last_found_ref;
					found_member->value_type = temp_element->value_type;
					found_member->path_segments = list_size(current_path_scratchpad) + 1;
					i = 0;
					list_iteration_start(current_path_scratchpad, TAIL_OF_LIST);
					while (list_iteration_has_more(current_path_scratchpad)) {
						path_link = list_iteration_get_next(current_path_scratchpad, NULL);
						found_member->path[i].value_type = path_link->value_type;
						if (path_link->value_type == JSON_object) {
							found_member->path[i].link.member_name = _json_create_string_copy(path_link->link.member_name);
						} else {
							found_member->path[i].link.array_index = path_link->link.array_index;
						}
						++i;
					}
					list_iteration_stop(current_path_scratchpad);
					found_member->path[i].link.member_name = _json_create_string_copy(temp_element->member_name);
					found_member->path[i].value_type = JSON_object;
					list_append(results, found_member);
					_json_free_(found_member);
				}
			} else {		/*	if (member_search_status == MEMBER_SEARCH_PASS_1)	*/

				/* we are in pass 2, we only look for subordinate list_objects to search */
				if ((temp_element->value_type == JSON_object) || (temp_element->value_type == JSON_array)) {
					if (temp_element->value.object) {
					 if (! list_is_empty(temp_element->value.object)) {
							/* first record our path to this point... */
							if (temp_element->member_name) {
								new_link.value_type = JSON_object;
								new_link.link.member_name = temp_element->member_name;	/* correct, this is NOT a copy */							
							} else {
								new_link.value_type = JSON_array;
								new_link.link.array_index = array_index;
							}
							list_append(current_path_scratchpad, &new_link);			/* push link onto the stack */

							ret = _json_find_members_full_iterate(	results,
																	current_path_scratchpad,
																	temp_element->value.object, 
																	current_search_depth-1,
																	member_name, 
																	partial_name,
																	string_value,
																	minimum_int_value,
																	maximum_int_value,
																	minimum_double_value,
																	maximum_double_value,
																	partial_or_range,
																	value_type);
							if (ret) {
								fprintf(stderr, "ERROR  ==>  %s() *** internal error occurred *** (%d)\n.",
												__func__, ret);
								return ret;
							}

							list_fetch(current_path_scratchpad, TAIL_OF_LIST);			/* pop link off of the stack */
						}
					} else {
						fprintf(stderr, "Warning  ==>  %s() %s element found with NULL "\
										"value.object pointer.\n.", __func__,
										(temp_element->value_type == JSON_object) ? "JSON_object" : "JSON_array");
					}
				}
			}
			++array_index;											/* so we know where we are */
		}		/*	while (list_iteration_has_more(json_data)	*/

		list_iteration_stop(json_data);
		if (subordinates_found == FALSE) {
			member_search_status = MEMBER_SEARCH_PASS_2;
		}
	}		/*	for (member_search_status = MEMBER_SEARCH_PASS_1; ....	*/

	if (! path_scratchpad) {
		/* we are about to the return to the original caller, lets clean up */
		list_erase(current_path_scratchpad);
		_json_free_(current_path_scratchpad);
	}

	return 0;
}


/* Add a new element to a JSON List which can be either a JSON Object or a 
	JSON Array type list,  with a JSON value, defined by the list_object 
	provided.

	This function will add a new element to the provided list_object
	at the last position of that list. The calling function is responsible
	for proper argument validity checking, and formatting the input 
	arguments correctly. No checking is done here.
	
	list_to_add		the list_object of the JSON data hierarchy to add the new
					Member to.
	member_name		pointer to a nul terminated string to use as the JSON Object 
					Member Name, this argument is required both by this function, 
					and by JSON Standards RFC 8259.
	string_value	pointer the a nul terminated string to use a the Member Value
					or NULL, then one of the number arguments will be used instead.
	integer_number	used as the Member value only if a) string_value is NULL, and
					b) double_number == 0, otherwise it is ignored
	double_number	used as the Member value only if a) string_value is NULL, and
					b) this argument is non-zero, otherwise it is ignored
	boolean_value	TRUE or FALSE, to use as the Member Value
	new_list		the list_object of the new JSON Array or Object to add to
					the existing JSON list, as a new Member.
		
	returns		LIST_OK				success, new element added to the list
				LIST_ERR			on error	 										*/
static int _json_list_object_element_add(	list_object *list_to_add, 
											const char *member_name,
											const JSON_value_types value_type,
											const char *string_value,
											const json_integer_t integer_number,
											const json_double_t double_number,
											const boolean boolean_value,
											list_object *new_list) {

	json_element_t new_element;
	
	memset(&new_element, 0x00, sizeof(json_element_t));

	if (member_name) {
		new_element.member_name = _json_create_string_copy(member_name);
		new_element.free_string_ptr = TRUE;
	}

	new_element.value_type = value_type;
	switch (value_type) {
		case JSON_string:
			new_element.value.string = _json_create_string_copy(string_value);
			new_element.free_string_ptr = TRUE;
			break;
			
		case JSON_float_str:
			new_element.value.string = _json_create_string_copy(string_value);
			new_element.free_string_ptr = TRUE;
			break;
			
		case JSON_integer:
			new_element.value.integer = integer_number;
			break;
			
		case JSON_double:
			new_element.value.double_number = double_number;
			break;
			
		case JSON_boolean:
			new_element.value.boolean_value = boolean_value;
			break;
			
		case JSON_null:
			break;
			
		case JSON_array:
			new_element.value.object = new_list;
			break;
			
		case JSON_object:
			new_element.value.object = new_list;
			break;

		default:
			fprintf(stderr, "ERROR  ==>  %s() invalid JSON value type provided (%u).\n",
							__func__, value_type);
			return LIST_ERR;
	}
	
	return (list_append(list_to_add, &new_element));			/* add new Member to JSON Object */
}

/* Only called by a Object Member element add function to check if they are
	trying to add a Member to a JSON Array instead, which of course is invalid.			*/
static int _json_test_add_member_to_array(	list_object *list_to_add, 
											const char *calling_function,
											const char *member_name,
											const boolean suppress_errors) {

	json_element_t *test_element;

	if (! list_is_empty(list_to_add)) {
		test_element = list_get_index(list_to_add, 0);
		if ((test_element) && (! test_element->member_name)) {
			if (! suppress_errors) 
				fprintf(stderr, "ERROR  ==>  %s() attempting to add Member %s to JSON Array.\n",
								calling_function, member_name);
			return LIST_ERR;
		}
	}
	
	return LIST_OK;
}


/* Only called by a Array data element add function to check if they are
	trying to a data value to a JSON Object instead, which of course is invalid.		*/
static int _json_test_add_value_to_object(	list_object *list_to_add, 
											const char *calling_function,
											const boolean suppress_errors) {

	json_element_t *test_element;

	if (! list_is_empty(list_to_add)) {
		test_element = list_get_index(list_to_add, 0);
		if ((test_element) && (test_element->member_name)) {
			if (! suppress_errors) 
				fprintf(stderr, "ERROR  ==>  %s() attempting to add Array value element to "\
								"a JSON Object.\n", calling_function);
			return LIST_ERR;
		}
	}
	
	return LIST_OK;
}

char *_json_create_string_copy(const char *input_string) {
	char *new_string = NULL;

	if (input_string) {
		if (! (new_string = (char *)_json_malloc(strlen(input_string) +1, __func__))) return NULL;
		strcpy(new_string, input_string);
	}

	return new_string;
}
/* Same as _json_create_string_copy() except it will only copy the number of bytes
	indicated, rather than _json_create_string_copy() which copies until a nul byte		*/
char *_json_create_string_n_copy(const char *input_string, const size_t num_bytes) {
	char *new_string = NULL;
	int i = -1;
	
	if (num_bytes > 1024) {
		fprintf(stderr, "Warning  ==>  %s() excessive size requested (%lu bytes).\n", __func__, (num_bytes));
	}

	if ((input_string) && (num_bytes > 0)) {
		if (! (new_string = (char *)_json_malloc(num_bytes +1, __func__))) return NULL;
		while (++i < num_bytes) new_string[i] = input_string[i];
		new_string[num_bytes] = '\0';
	}

	return new_string;
}

/* JSON package internal allocator */
void *_json_malloc(const size_t size, const char *caller) {

	void *storage = NULL;

	storage = (void *)malloc(size);
	if (! storage) {
		/* oops */
		fprintf(stderr, "CRITICAL ERROR  ==>  %s() malloc() failed to allocate %lu bytes, with "\
						" the error %s.\n", caller, size, strerror(errno));
	}

	return storage;
}

/* JSON package internal de-allocator */
void *_json_free(	void *pointer,
					const char *caller) {

	if (pointer) {
		free(pointer);
	} else {
		fprintf(stderr, "Warning  ==>  %s() was attempting to free NULL pointer.\n", caller);
	}

	return NULL;
}


