/** =================================================
 **                License Statement
 ** =================================================
 * 
 ** Copyright (c) 2014-2022 David Winterburn <info@fwsentry.org>. All commercial
 *	rights reserved.
 * 
 ** This file contains Original Code and/or Modifications of Original Code, and is
 ** subject to the license terms below. If this file contains portions of other
 ** code, it is so noted at the end of this license, with copyright credit as
 ** required.
 * 
 ** Permission to use, copy, modify, and distribute this software for any purpose
 ** WITHOUT any fee is hereby granted as long as such purposes do not violate the
 ** terms of this license, and provided that the above copyright notice and this
 ** permission notice and license appear unmodified in ALL copies, distributed or
 ** stored. Permission is expressly denied and withheld for any commercial uses of
 ** this Original Code, either in whole or part, or for any purpose with a direct,
 ** or any indirect, fee. You may not include Original Code of the Author in any
 ** other work or package that will be sold, rented, or leased commercially. All
 ** commercial rights reserved, and all intellectual property and patent rights
 ** remain with the author.
 * 
 ** THE ORIGINAL CODE AND ALL SOFTWARE DISTRIBUTED UNDER THIS LICENSE IS PROVIDED
 ** ON AN "AS IS" BASIS WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESS OR IMPLIED,
 ** AND THE AUTHOR DISCLAIMS ALL SUCH WARRANTIES, OR LEGAL CLAIMS, WITH REGARD TO
 ** THIS SOFTWARE INCLUDING WITHOUT LIMITATION ANY WARRANTIES, IMPLIED OR OTHER,
 ** OF MERCHANTABILITY, PERFORMANCE AND FITNESS FOR A PARTICULAR PURPOSE, QUIET
 ** ENJOYMENT OR NON-INFRINGEMENT. UNDER NO CIRCUMSTANCES SHALL THE AUTHOR BE IN
 ** ANY WAY LIABLE FOR ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES, OR
 ** ANY DAMAGES WHATSOEVER, RESULTING FROM THE USE OF THIS SOFTWARE. SUCH DAMAGES
 ** MAY INCLUDE, BUT ARE NOT LIMITED TO, LOSS OF USE, EQUIPMENT, DATA, BILLABLE
 ** TIME, PERSONAL OR PHYSICAL INJURY, AND DIRECT OR INDIRECT PROFITS, WHETHER IN
 ** AN ACTION OF CONTRACT, NEGLIGENCE OR ANY OTHER TORTIOUS ACTION, ARISING OUT
 ** OF, OR IN CONNECTION WITH, THE USE OF, THE FITNESS OR PERFORMANCE OF THIS
 ** SOFTWARE, UNDER ANY CIRCUMSTANCES. THIS SOFTWARE WAS NOT DESIGNED TO BE USED
 ** IN ANY SITUATION WHERE HUMAN LIFE IS AT RISK, AND NO SUCH USE IS PROVIDED FOR
 ** OR PERMITTED.
 * 
 ** By using this software you agree not to violate any of the commercial use
 ** rights of the author with regard to the original code, and that any such
 ** violations of commercial use rights, either knowingly on unknowingly, will
 ** result in a) without notification from the author, the immediate and
 ** automatic termination of this license, and the subsequent requirement to
 ** remove and delete any copies of this software in your possession, b)
 ** forfeiture of all commercial use proceeds to the author and c) additional
 ** punitive damages, recovery and legal costs payable to the author. Under no
 ** circumstances whatsoever are others authorized to profit from the work(s) of
 ** the author without express written permission and consent.
 * 
 ** BY USING THIS SOFTWARE YOU AGREE THAT IT IS THE SOLE RESPONSIBILITY OF THE END
 ** USER TO DECIDE IF THIS SOFTWARE OR PACKAGE PERFORMS, OR IS FIT FOR, ANY
 ** INTENDED USAGE.
 * 
 ** This License and the rights granted hereunder will terminate; automatically
 ** without notice from the Author if You fail to comply with any term(s) of this
 ** License and fail to cure such breach within 30 days of becoming aware of such
 ** breach; or automatically without notice from the Author if You, at any time
 ** during the term of this License, commence any legal action against the Author
 ** or engage in any unauthorized commercial use activity with any of the Original
 ** Code. Upon termination of the license you are required to remove all copies of
 ** this software from all computer systems and data storage archives under your
 ** jurisdiction or control.
 * 
 ** This license is automatically extended, if in the future, the Author exercises
 ** any intellectual property, or patent, rights connected with any of the
 ** Original Code included in this package. ALL end users using this package
 ** within the terms of this license, will automatically licensed to use those
 ** exercised or covered claims free of charge as they apply to this software
 ** package.
 **/

/* 		FWSentry.
 * 
 * See http://info.fwsentry.org for more information on this package, or see
 * http://fwsentry.org/bugs/submission.html to report bugs or request new
 * features or improvements, or contact support@fwsentry.org. Bugs will be
 * addressed, new feature requests and program improvements will be considered,
 * at the sole discretion of the Author. Updates and new releases may be issued
 * in the future, if the Author considers it practical, enjoyable and desirable.
 */

/** 	Statement of Support.
 * 
 ** Bug reports must be accompanied with a reasonable description and reasonable
 ** documentation of the problem, and any code changes you may have made to the 
 ** affected module. If the problem turns out to be an issue with code not 
 ** supplied by the author, the author may, or may not, choose to fix it.
 * 
 ** If implemented, such requests or bug fixes may be added to a later release,
 ** package update, or support patch, at the sole discretion of the author, within
 ** time-frame's suitable to the author. All support that may optionally be
 ** provided by the author, including but not limited to, responding to requests
 ** for new attack signatures, bug reports, new feature requests, or general usage
 ** support; will be provided on an "as-is" basis, at the sole discretion of the
 ** author. Any and all support, or resulting code, documentation, or product
 ** changes, will be provided with the same license and legal disclaimers
 ** described in the original package (and above) and with absolutely no
 ** warranties, implied, stated or otherwise.
 **/ 

/*	JSON Parser version 1.0 -- Copyright (c) 2014-2022 David Winterburn <info@fwsentry.org>	*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <time.h>
#include <fcntl.h>
#include <pthread.h>
#include <signal.h>
#include <assert.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <math.h>
#include <float.h>

/* I have tried to make FWSentry suitable for wide variety of systems, not tying it to a C99 minimum
	standard. For now i am not going force users to use a C99 compiler and libraries, and so i define
	these here, so i can migrate my booleans to the C99 format, but keep the computability. I also
	personally prefer the TRUE and FALSE in upper case, it is easier quickly understand the code. */
#if __STDC_VERSION__ >- 199901L
#include <stdbool.h>
#endif		/* __STDC_VERSION__ >- 199901L */

#ifdef __bool_true_false_are_defined
#define TRUE										true
#define FALSE										false
#define boolean										_Bool

#else

/* for environments with no stdbool.h */
typedef	int	bool
#define	false										(bool)0
#define	true										(bool)1

#ifndef TRUE
#define TRUE										(bool)1
#endif		/* !TRUE */
#ifndef FALSE
#define FALSE										(bool)0
#endif		/* !FALSE */

#endif		/* __bool_true_false_are_defined */


#include "config.h"

#include "dlist/dlist.h"


/* global debugging variables created here */
#define INITIALIZE_DEBUGGING_FLAG
#include "json-parser.h"




#define JSON_ILLEGAL_COMMENT_1				'*'
#define JSON_ILLEGAL_COMMENT_2				'/'

#define MAX_NUMBER_TOKEN_LENGTH							40
#define MAX_NESTED_DEPTH								2048

#define PARSE_TOKEN_ERROR								-1
#define PARSE_TOKEN_NONE								0
#define PARSE_TOKEN_ARRAY_BEGIN							1
#define PARSE_TOKEN_OBJECT_BEGIN						2
#define PARSE_TOKEN_ARRAY_END							3
#define PARSE_TOKEN_OBJECT_END							4
#define PARSE_TOKEN_NAME_VALUE_SEPARATOR				5
#define PARSE_TOKEN_VALUE_SEPARATOR						6
#define PARSE_TOKEN_LITERAL_TRUE						7
#define PARSE_TOKEN_LITERAL_FALSE						8
#define PARSE_TOKEN_LITERAL_NULL						9
#define PARSE_TOKEN_STRING_VALUE						10
#define PARSE_TOKEN_NUMBER_VALUE						11

typedef struct {
	char buffer_token;
	int	parse_token;
} parse_tokens_t;
	
#define NUM_PARSE_TOKENS							22
parse_tokens_t parse_tokens[NUM_PARSE_TOKENS] = {
	{	'[',	PARSE_TOKEN_ARRAY_BEGIN				}, \
	{	'{',	PARSE_TOKEN_OBJECT_BEGIN			}, \
	{	']',	PARSE_TOKEN_ARRAY_END				}, \
	{	'}',	PARSE_TOKEN_OBJECT_END				}, \
	{	':',	PARSE_TOKEN_NAME_VALUE_SEPARATOR	}, \
	{	',',	PARSE_TOKEN_VALUE_SEPARATOR			}, \
	{	't',	PARSE_TOKEN_LITERAL_TRUE			}, \
	{	'f',	PARSE_TOKEN_LITERAL_FALSE			}, \
	{	'n',	PARSE_TOKEN_LITERAL_NULL			}, \
	{	'"',	PARSE_TOKEN_STRING_VALUE			}, \
	{	'0',	PARSE_TOKEN_NUMBER_VALUE			}, \
	{	'1',	PARSE_TOKEN_NUMBER_VALUE			}, \
	{	'2',	PARSE_TOKEN_NUMBER_VALUE			}, \
	{	'3',	PARSE_TOKEN_NUMBER_VALUE			}, \
	{	'4',	PARSE_TOKEN_NUMBER_VALUE			}, \
	{	'5',	PARSE_TOKEN_NUMBER_VALUE			}, \
	{	'6',	PARSE_TOKEN_NUMBER_VALUE			}, \
	{	'7',	PARSE_TOKEN_NUMBER_VALUE			}, \
	{	'8',	PARSE_TOKEN_NUMBER_VALUE			}, \
	{	'9',	PARSE_TOKEN_NUMBER_VALUE			}, \
	{	'-',	PARSE_TOKEN_NUMBER_VALUE			}, \
	{	' ',	PARSE_TOKEN_ERROR					}
};

#define PARSER_ARRAY_CLOSED							2
#define PARSER_WORKING								1
#define PARSER_SUCCESSFUL							0
#define PARSER_MISSING_SYNTAX						-1
#define PARSER_INVALID_SYNTAX						-2
#define PARSER_BAD_VALUE							-3
#define PARSER_MISSING_VALUE						-4



#define PARSER_INTERNAL_ERROR						-10

#define PARSER_TOKEN_LABEL_NULL						"null"
#define PARSER_TOKEN_LABEL_TRUE						"true"
#define PARSER_TOKEN_LABEL_FALSE					"false"

/*	Material specific to the JSON Generator
	=======================================		*/
#define ADD_NEW_LINE								TRUE
#define STAY_ON_SAME_LINE							FALSE
#define JSON_GENERATOR_CHUNK_SIZE					1024
#define SPACES_PER_INDENT							2
#define JSON_GENERATOR_EMPTY_OBJECT					"{ }"
#define JSON_GENERATOR_EMPTY_ARRAY					"[ ]"
#define JSON_GENERATOR_OBJECT_GRAMMAR				"\": "
#define JSON_GENERATOR_MULTI_VALUE_GRAMMAR			", "
#define JSON_GENERATOR_OBJECT_BEGIN					"{"
#define JSON_GENERATOR_OBJECT_END					"}"
#define JSON_GENERATOR_OBJECT_BEGIN_1_MEMBER		"{ "
#define JSON_GENERATOR_OBJECT_END_1_MEMBER			" }"
#define JSON_GENERATOR_ARRAY_BEGIN					"["
#define JSON_GENERATOR_ARRAY_END					"]"
#define JSON_GENERATOR_ARRAY_BEGIN_1_MEMBER			"[ "
#define JSON_GENERATOR_ARRAY_END_1_MEMBER			" ]"
#define JSON_GENERATOR_STRING_QUOTE					"\""
#define JSON_GENERATOR_NULL_STRING					""






/*	JSON data representation -
	========================

	Simply put, 
		a JSON Object is mapped to a subordinate D-List list_object
		a JSON Object name/value pair is an element within that list_object 
		a JSON array is a subordinate D-List list_object within an element 
			of the enclosing JSON object
		
		JSON value types -
			JSON_string is a pointer to the original string in the original JSON buffer,
				it is now nul terminated. The pointer points to the first char
				after the original double quote character, and the last character
				is the character before the ending double quote.
			JSON_array is a pointer to a new list_object which will contain the 
				values, one element per array value. If the array in the JSON code
				was a single entry then that single entry is a single element in
				the new subordinate list.
			JSON_object is a pointer to a new list_object which will contain the 
				name value pairs, one element per JSON object element. If the JSON 
				Object in the JSON data block was a single entry then that single
				entry is a single element in the new subordinate list. 
				If the JSON Object was empty " { } " then the subordinate list will
				have zero elements.
			All other JSON value types are converted and stored in the name/value, or
				value only, element, this includes numbers, boolean, etc.
		
	The D-List list_objects that now represent the JSON object, array and value 
	structures, can be used just like any other D-List list_object. They can be sorted
	and manipulated as needed. All data pointers within the lists point to portions
	of the original JSON string buffer that was parsed.
	
	DO NOT release the JSON buffer, until after the D-List lists that describe it have
	been erased. All D-List lists used for parsing JSON objects are built with the 
	COPY_DATA attribute, so all list storage is handled internally by the JSON parsing
	code, and the deletions use D-List functions to properly free all the allocated
	regions.
	
	This representation of a JSON object eliminates all white space, and compresses
	strings to remove escape characters, etc.
	
	jCard representations are handled in the same way, a vCardArray would be an
	array object, a new list_object, and each of the elements in that array would
	be elements in the new list. elements with nul parameters, would have a NULL
	entry in the parameters pointer, or there would be pointer to another list.

	So using this simple JSON data block, for example:
	{ "country" : "JP", "language" : "en", "names" : [ "Hayao", "Totoro", "Ponyo", "Nausica", "Kiki"
		], "films" : { "Ponyo" : 2008, "Princess Mononoke" : 1997, "Tonari no Totoro" : 1988, 
		"How Do You Live?" : "In Production"}}

	would be parsed into a set of D-Lists such as:
	
								value_type	member_name	: value
								----------	-----------	  -----
(p)-->	root_list->	element_1->	JSON_string	"country"	: "JP"
					element_2->	JSON_string	"language"	: "en"
					element_3->	JSON_array	"names"		: value.object --> array_list_1``-->(a)
					element_4->	JSON_object	"films"		: value.object --> object_list_2`-->(b)
				(end of root_list)

			 (a)-->	array_list_1->	element_1->	JSON_string	""	: "Hayao"
									element_2->	JSON_string	""	: "Totoro"
									element_3->	JSON_string	""	: "Ponyo"
									element_4->	JSON_string	""	: "Nausica"
									element_5->	JSON_string	""	: "Kiki"
								(end of array_list_1)

			 (b)-->	object_list_2->	element_1->	JSON_integer	"Ponyo"				: 2008
									element_2->	JSON_integer	"Princess Mononoke"	: 1997
									element_3->	JSON_integer	"Tonari no Totoro"	: 1988
									element_4->	JSON_string		"How Do You Live?"	: "In Production"
								(end of object_list_2)
								
				parser will return pointer(p) and json_data_type (JSON_object)

		for each element encountered in a list, the value_type of that element
		will tell you what type of element it is, and from there you know that
		it is either a value, in which case the value is there in the element,
		or it is a JSON array or JSON Object and the pointer to the relevant
		subordinate list is there in the element.
		
		So in the above example calling parse_json_data_block() with the above
		JSON data block would return a single pointer (p) to the root_list, which
		would have 4 elements in it, 2 of them with subordinate list pointers 
		in them, and a the json_data_type of JSON_object, so you knew the root
		list represented a JSON Object at the top level. In this case the 
		subordinate lists would have 5 and 4 elements in them. 
		
		These lists can then be retrieved, searched, sorted, and managed as you
		would any other D-List using the various D-List functions. It is easy to
		find out if a JSON array or JSON Object was empty by calling the function
		list_size() or list_is_empty() on that subordinate list_object. Or by
		using list iteration so any processing code is only used when there are
		elements to retrieve. Please refer to D-List documentation for more 
		information on how to use D-List list_objects and functions. However even
		though D-List provides a broad range of options to access and manage data
		elements, it is at heart very easy to use the core set of D-List functions.
		
		So with the above JSON sample data as input, this code snippet can find
		a particular film year,
			list_object *root_list;
			json_element_t *element_ptr, *film_element_ptr;
			int json_data_type;
			
			root_list  = parse_json_data_block(the_json_data, &json_data_type, my_search_function, NULL, FALSE);
			if (json_data_type == JSON_object) elememnt_ptr = list_simple_search(root_list, "films");
			if ((element_ptr) && (element_prt->value_type == JSON_object)) {
				film_element_ptr = list_simple_search(element_ptr->value.object, "Ponyo");
				if ((film_element_ptr) && (film_element_ptr->value_type == JSON_string)){
					fprintf(stdout, "The Movie %s was released in &ld.\n", 
							film_element_ptr->member_name, 
							film_element_ptr->value.integer);
				} else {
					fprintf(stdout, "Movie Ponyo not found in list\n");
				}
			} else {
				do something if the JSON object element name "films" is missing.
			}		
		
		The utility json-parser-test which is part of this JSON Parser package
		has a function to output a parsed set of list_objects of a JSON data 
		block to stdout, this is a good way to see how the JSON data block is 
		formatted into D-List list_objects and elements.

*/

/* No, i am not an anime fan, just trying to make the dry documentation fun. */


/* These values are setup by parse_json_data_block() and thereafter 
	only ever referenced or updated by parse_get_next_token()							*/
typedef struct {
	char *current_parse_cursor;							/* do not use or modify this in your code */
	char *parse_end_barrier;							/* do not use or modify this in your code */
	char *parser_last_token_returned;					/* used for error reporting only ... DO NOT TOUCH */
	long int current_parse_line_count;					/* used for error reporting only */
	int current_saved_parse_token;						/* Never use this..... ever. Used to save
															a structural token that was obliterated
															by a nul terminator. Or 0				*/
	/* These values are set by parse_json_data_block() and thereafter
		only ever referenced by parse_json_ojbect_token() and parse_json_aray_token()	*/
	functions_globals_t functions;

	/* These values are setup by parse_json_data_block() and thereafter 
		only ever referenced by parse_number_value()								*/
	boolean parser_keep_exponent_as_text;		/* set by the value of keep_exponent */
} parser_globals_t;

/* These values are used by the JSON Generator to keep track of the output buffer */
typedef struct {
	unsigned long int current_buffer_size;
	unsigned long int current_output_offset;
	unsigned int indent_level;
	char *output_buffer;
} generator_globals_t;




char json_whitespace_chars[4] = "\t\n\r ";
char json_value_end_token_chars[7] = "]},\t\n\r ";

char json_structural_tokens[6] = "[]{}:,";
char json_literal_tokens[3] = "tfn";
char json_number_tokens[15] = "0123456789+-eE.";

char json_non_integer_chars[] = "eE.+-";
char json_exponent_chars[] = "eE";

/* for hex conversions, in either direction. */
struct byte_hex_values_s {
	char hex_byte[2];
	unsigned char hex_value;
} byte_hex_values[32] = {
	{ "00", 0x00 },
	{ "01", 0x01 },
	{ "02", 0x02 },
	{ "03", 0x03 },
	{ "04", 0x04 },
	{ "05", 0x05 },
	{ "06", 0x06 },
	{ "07", 0x07 },
	{ "08", 0x08 },
	{ "09", 0x09 },
	{ "0A", 0x0A },
	{ "0B", 0x0B },
	{ "0C", 0x0C },
	{ "0D", 0x0D },
	{ "0E", 0x0E },
	{ "0F", 0x0F },
	{ "10", 0x10 },
	{ "11", 0x11 },
	{ "12", 0x12 },
	{ "13", 0x13 },
	{ "14", 0x14 },
	{ "15", 0x15 },
	{ "16", 0x16 },
	{ "17", 0x17 },
	{ "18", 0x18 },
	{ "19", 0x19 },
	{ "1A", 0x1A },
	{ "1B", 0x1B },
	{ "1C", 0x1C },
	{ "1D", 0x1D },
	{ "1E", 0x1E },
	{ "1F", 0x1F }
};



static int parse_json_element_token(	list_object *current_list_object,
										parser_globals_t *parser_data);

static int parse_json_array_token(	list_object *current_list_object,
									parser_globals_t *parser_data);

static int parse_value_token(	list_object *current_list_object, 
								json_element_t *new_element, 
								const boolean new_open_array,
								parser_globals_t *parser_data);
static int parse_get_next_token(char **next_token,
								parser_globals_t *parser_data);

static int parse_number_value(	char *token, 
								json_element_t *new_element,
								parser_globals_t *parser_data);

static inline boolean is_whitespace(const char *test_char);
static inline boolean is_valid_number_token(const char *test_char);
static inline boolean is_hex_digit(const char *test_char);
static inline unsigned char bytes_hex_value(const char *input); 

static inline boolean is_name_value_pair_valid(	boolean *obj_name_string, 
												boolean *colon_separator_found,
												boolean *value_found);
static inline boolean is_valid_value_end_token(const char *test_char);


static int json_generate_object(	list_object *object_data,
							generator_globals_t *generator_data);
static int json_generate_array(	list_object *array_data,
							generator_globals_t *generator_data);

static char *json_generate_value(	json_element_t *element, 
							char *temp_buffer);
static char *json_generate_write(	char *output_to_write, 
							const boolean new_line,
							generator_globals_t *generator_data);

#ifdef DBW_JSON_NOT_NEEDED_YET
static inline boolean is_valid_structural_token(const char *test_char);
static inline boolean is_valid_literal_token(const char *test_char);
#endif

/* do not want to expose these to the public via the header.
	Keep in sync with the originals in json-functions.c				*/
char *_json_create_string_copy(		const char *input_string);

char *_json_create_string_n_copy(	const char *input_string,
											const size_t num_bytes);

void *_json_malloc(	const size_t size, 
					const char *caller);

#define _json_free_(x)		x = _json_free(x, __func__)
void *_json_free(	void *pointer,
					const char *caller);



/*		Internal Only JSON Data Hierarchy D-List Functions		*/
static size_t json_parser_size_function(const void *element) { 
	if (element) {
		return sizeof(json_element_t); 
	} else {
		fprintf(stderr, "%s() called with NULL value    ==>  ERROR  <==\n", __func__);
		return 0;					/* Safety condition, but cannot happen unless someone has messed with the dlist.c code */
	}
}

static void json_parser_remove_function(const void *element) {
	json_element_t *extremis_elem = (json_element_t *)element;

	/* This remove function is added to all JSON lists by default,
		each element that is to be removed will be checked to see
		if it has a list_object pointer inside it. If it does, 
		that list will be erased before this element is erased.	
		This of course, creates a recursive erasure, which will 
		remove all the elements and list_objects underneath the
		list hierarchy the caller is trying to erase.				*/
	if (extremis_elem) {
		if ((extremis_elem->value_type == JSON_object) ||
				(extremis_elem->value_type == JSON_array)) {

			/* This code is NOT forgiving to developers who forget to remove dangling
				pointers to things they have already removed. 
				
				If there is a lingering pointer left in element->value.object, AND 
				D-List is operating in DEBUG mode (i.e. D-List was compiled with 
				DEBUG defined) then D-List will ouput this error to STDERR to warn you.
				
				==> Dlist Debug ERROR ==> list_erase() - called with pointer 0x100100130 to an uninitialized list object. You must call list_create() first.
				==> Dlist Debug ERROR ==> list_erase() - called with pointer 0x100100130 to invalid list_object, (error type 2).
		
				This will be followed by your code failing with a malloc() hard error.
				
				Better for you to find out now, during debug, than after you
				ship your code to someone else :-)										*/

			if (extremis_elem->value.object) {
				list_erase(extremis_elem->value.object);
				extremis_elem->value.object = NULL;
			}
		}
		/* If the user who created this element wants us to release the
			storage being used by the string, we do it now 					*/
		if ((extremis_elem->value_type == JSON_string) || 
				(extremis_elem->value_type == JSON_float_str)) {
			if (extremis_elem->free_string_ptr) {
				_json_free_(extremis_elem->value.string);
			}
		}
		/* If the user who created this Object Member element wants us to 
			release the storage being used by the Member Name string, we do it now 		*/
		if ((extremis_elem->free_string_ptr) && (extremis_elem->member_name)) {
			_json_free_(extremis_elem->member_name);
		}
	} else {
		fprintf(stderr, "%s() called with NULL value    ==>  ERROR  <==\n", __func__);
		return;						/* Safety condition, but cannot happen unless 
										someone has messed with the dlist.c code */
	}
		
	return;
}



/*		Caller available Default JSON Parser D-List Search Functions		*/
							
boolean default_search_function_member_name(const void *element, const void *key) {
	const char *key_value = (const char *)key;
	const json_element_t *object_element = (const json_element_t *)element;

	assert(object_element != NULL && object_element->member_name != NULL && key_value != NULL);

	if ((object_element) && (object_element->member_name) && (key_value)) {
		return (strcmp(object_element->member_name, key_value) == 0);
	} else {
		fprintf(stderr, "%s() called with NULL values    ==>  ERROR  <==\n", __func__);
		return FALSE;					/* This will cover when asserts have been compiled out */
	}
}
							
boolean default_search_function_partial_member_name(const void *element, const void *key) {
	const char *key_value = (const char *)key;
	const json_element_t *object_element = (const json_element_t *)element;

	assert(object_element != NULL && object_element->member_name != NULL && key_value != NULL);

	if ((object_element) && (object_element->member_name) && (key_value)) {
		return (strstr(object_element->member_name, key_value) != NULL);
	} else {
		fprintf(stderr, "%s() called with NULL values    ==>  ERROR  <==\n", __func__);
		return FALSE;					/* This will cover when asserts have been compiled out */
	}
}

boolean default_search_function_array_values(const void *element, const void *key) {
	const json_element_t *key_value = (const json_element_t *)key;
	const json_element_t *array_element = (const json_element_t *)element;

	assert(array_element != NULL && key_value != NULL);

	if ((! array_element) || (! key_value)) {
		fprintf(stderr, "%s() called with NULL values    ==>  ERROR  <==\n", __func__);
		return FALSE;					/* This will cover when asserts have been compiled out */
	}

	if (array_element->value_type != key_value->value_type) {
		return FALSE;
	}
	
	/* For each possible type of element, we set up the appropriate value equality check */
	if (array_element->value_type == JSON_object) {
		if (array_element->member_name) {
			/* we must be active inside a JSON Object list_object. */
			if (! key_value->member_name) {
				fprintf(stderr, "%s() called with NULL key object name    ==>  ERROR  <==\n", __func__);
				return FALSE;					/* This will cover when asserts have been compiled out */
			}
			return (default_search_function_member_name(array_element, key_value->member_name));
		} else {
			/* Nope, we must be active inside a JSON Array list_object, our default case */
			return TRUE;
		}
	}

	if (array_element->value_type == JSON_array)
		return TRUE;					/* the first array we find will result in a TRUE return */

	if (array_element->value_type == JSON_null)
		return TRUE;					/* null == null, whatever we do */

	if (array_element->value_type == JSON_boolean)
		return (array_element->value.boolean_value == key_value->value.boolean_value);

	if (array_element->value_type == JSON_integer)
		return (default_search_function_array_integer_value(array_element, &key_value->value.integer));

	if (array_element->value_type == JSON_double)
		return (default_search_function_array_double_value(array_element, &key_value->value.double_number));

	if (array_element->value_type == JSON_float_str)
		return (strcmp(array_element->value.string, key_value->value.string) == 0);
	
	/* Yes these are the same, but they are left split out in case the logic for one
		of them needs to change in the future											*/
	if (array_element->value_type == JSON_string)
		return (strcmp(array_element->value.string, key_value->value.string) == 0);

	/* safety in case the we were called with an erroneous element */
	fprintf(stderr, "%s() called with an invalid JSON_value_type    ==>  ERROR  <==\n", __func__);
	return FALSE;
}

boolean default_search_function_array_string_value(const void *element, const void *key) {
	const char *key_value = (const char *)key;
	const json_element_t *array_element = (const json_element_t *)element;

	assert(array_element != NULL && key_value != NULL);

	if ((! array_element) || (! key_value)) {
		fprintf(stderr, "%s() called with NULL values    ==>  ERROR  <==\n", __func__);
		return FALSE;					/* This will cover when asserts have been compiled out */
	}

	if (array_element->value_type == JSON_string) {
		if (! array_element->value.string) {
			fprintf(stderr, "%s() called with NULL element string value    ==>  ERROR  <==\n", __func__);
			return FALSE;
		}
		return (strcmp(array_element->value.string, key_value) == 0);
	} else {
		return FALSE;
	}
}

boolean default_search_function_array_partial_string_value(const void *element, const void *key) {
	const char *key_value = (const char *)key;
	const json_element_t *array_element = (const json_element_t *)element;

	assert(array_element != NULL && key_value != NULL);

	if ((! array_element) || (! key_value)) {
		fprintf(stderr, "%s() called with NULL values    ==>  ERROR  <==\n", __func__);
		return FALSE;					/* This will cover when asserts have been compiled out */
	}

	if (array_element->value_type == JSON_string) {
		if (! array_element->value.string) {
			fprintf(stderr, "%s() called with NULL element string value    ==>  ERROR  <==\n", __func__);
			return FALSE;
		}
		return (strstr(array_element->value.string, key_value) != NULL);
	} else {
		return FALSE;
	}
}

boolean default_search_function_array_integer_value(const void *element, const void *key) {
	const json_integer_t *key_value = (const json_integer_t *)key;
	const json_element_t *array_element = (const json_element_t *)element;

	assert(array_element != NULL && key_value != NULL);

	if ((! array_element) || (! key_value)) {
		fprintf(stderr, "%s() called with NULL values    ==>  ERROR  <==\n", __func__);
		return FALSE;					/* This will cover when asserts have been compiled out */
	}

	if (array_element->value_type == JSON_integer)
		return (array_element->value.integer == *key_value);
	else
		return FALSE;
}

boolean default_search_function_array_double_value(const void *element, const void *key) {
	const json_double_t *key_value = (const json_double_t *)key;
	const json_element_t *array_element = (const json_element_t *)element;

	assert(array_element != NULL && key_value != NULL);

	if ((! array_element) || (! key_value)) {
		fprintf(stderr, "%s() called with NULL values    ==>  ERROR  <==\n", __func__);
		return FALSE;					/* This will cover when asserts have been compiled out */
	}

	if (array_element->value_type == JSON_double)
		return (fabsl(array_element->value.double_number - *key_value) < LDBL_EPSILON);
	else
		return FALSE;
}

boolean default_search_function_array_precise_double_value(const void *element, const void *key) {
	const json_double_t *key_value = (const json_double_t *)key;
	const json_element_t *array_element = (const json_element_t *)element;

	assert(array_element != NULL && key_value != NULL);

	if ((! array_element) || (! key_value)) {
		fprintf(stderr, "%s() called with NULL values    ==>  ERROR  <==\n", __func__);
		return FALSE;					/* This will cover when asserts have been compiled out */
	}

	if (array_element->value_type == JSON_double)
		return (fabsl(array_element->value.double_number - *key_value) < LDBL_MIN);
	else
		return FALSE;
}

boolean default_search_function_array_sloppy_double_value(const void *element, const void *key) {
	const json_double_t *key_value = (const json_double_t *)key;
	const json_element_t *array_element = (const json_element_t *)element;

	assert(array_element != NULL && key_value != NULL);

	if ((! array_element) || (! key_value)) {
		fprintf(stderr, "%s() called with NULL values    ==>  ERROR  <==\n", __func__);
		return FALSE;					/* This will cover when asserts have been compiled out */
	}

	if (array_element->value_type == JSON_double)
		return (fabsl(array_element->value.double_number - *key_value) < 0.00001L);
	else
		return FALSE;
}

/*		Caller available Default JSON Parser D-List Compare Functions		*/

int default_compare_function_member_names(const void *element, const void *key) {
	const json_element_t *key_element = (const json_element_t *)key;
	const json_element_t *object_element = (const json_element_t *)element;

	assert(object_element != NULL && object_element->member_name != NULL && key_element != NULL && key_element->member_name != NULL);

	if ((! object_element->member_name) || (! key_element->member_name)) {
		fprintf(stderr, "%s() called with NULL object name values    ==>  ERROR  <==\n", __func__);
		return ELEMENTS_EQUAL;			/* This will cover when asserts have been compiled out */
	}

	if ((object_element->member_name) && (key_element->member_name))
		return (strcmp(object_element->member_name, key_element->member_name));
	else
		/* safety in case the we were called with an erroneous element */
		return ELEMENTS_EQUAL;	
}


int default_compare_function_array_values(const void *element, const void *key) {
	const json_element_t *key_element = (const json_element_t *)key;
	const json_element_t *array_element = (const json_element_t *)element;

	assert(array_element != NULL && key_element != NULL);
	
	/* first are these elements both numbers ? the same type or not.					*/
	if (((array_element->value_type == JSON_integer) || (array_element->value_type == JSON_double)) &&
			((key_element->value_type == JSON_integer) || (key_element->value_type == JSON_double))) {
		return (default_compare_function_array_number_values(element, key));
	}

	/* next are these elements both of the same type ?, if they are not, then
		just evaluate them based on their value types alone, and return the result.		*/
	if (array_element->value_type != key_element->value_type)
		return ((array_element->value_type > key_element->value_type) - (array_element->value_type < key_element->value_type));
	
	/* We do not evaluate JSON Objects or Arrays if they appear in a value list, 
		they are just grouped together as equal items, unsorted in a block
		at the front of the sort. The Objects first, then the Arrays.		 */
	if ((array_element->value_type == JSON_object) || (array_element->value_type == JSON_array))
		return ELEMENTS_EQUAL;

	/* Obviously there is no point in evaluating NULL values. They are just grouped
		together as equal items, unsorted in a block after Objects and then Arrays.		*/
	if (array_element->value_type == JSON_null)
		return ELEMENTS_EQUAL;

	if (array_element->value_type == JSON_boolean)
		return ((array_element->value.boolean_value > key_element->value.boolean_value) - (array_element->value.boolean_value < key_element->value.boolean_value));

	if ((array_element->value_type == JSON_float_str) || (array_element->value_type == JSON_string))
		return (strcmp(array_element->value.string, key_element->value.string));

	/* safety in case the we were called with an erroneous element */
	return ELEMENTS_EQUAL;
}


int default_compare_function_array_string_values(const void *element, const void *key) {
	const json_element_t *key_element = (const json_element_t *)key;
	const json_element_t *array_element = (const json_element_t *)element;

	assert(array_element != NULL && array_element->value.string != NULL && key_element != NULL && key_element->value.string != NULL);

	if ((array_element->value_type == JSON_string) && (key_element->value_type == JSON_string)) {
		if ((! array_element->value.string) || (! key_element->value.string)) {
			fprintf(stderr, "%s() called with NULL value strings    ==>  ERROR  <==\n", __func__);
			return ELEMENTS_EQUAL;					/* This will cover when asserts have been compiled out */
		}
		return (strcmp(array_element->value.string, key_element->value.string));
	} else {
		/* safety in case the user called us on a list with mixed value types */
		return ELEMENTS_EQUAL;
	}
}

int default_compare_function_array_integer_values(const void *element, const void *key) {
	const json_element_t *key_element = (const json_element_t *)key;
	const json_element_t *array_element = (const json_element_t *)element;

	assert(array_element != NULL && key_element != NULL);

	if ((array_element->value_type == JSON_integer) && (key_element->value_type == JSON_integer))
		return ((array_element->value.integer > key_element->value.integer) - (array_element->value.integer < key_element->value.integer));
	else
		/* safety in case the user called us on a list with mixed value types */
		return ELEMENTS_EQUAL;
}

int default_compare_function_array_double_values(const void *element, const void *key) {
	const json_element_t *key_element = (const json_element_t *)key;
	const json_element_t *array_element = (const json_element_t *)element;

	assert(array_element != NULL && key_element != NULL);

	/* Precision and rounding errors are not an issue here, even with those creeping
		errors, this is a sort, and the values will still compare properly with each
		other, even with a precision error 												*/
	if ((array_element->value_type == JSON_double) && (key_element->value_type == JSON_double))
		return ((array_element->value.double_number > key_element->value.double_number) - (array_element->value.double_number < key_element->value.double_number));
	else
		/* safety in case the user called us on a list with mixed value types */
		return ELEMENTS_EQUAL;
}

int default_compare_function_array_number_values(const void *element, const void *key) {
	const json_element_t *key_element = (const json_element_t *)key;
	const json_element_t *array_element = (const json_element_t *)element;

	json_double_t array_element_number = 0;
	json_double_t key_element_number = 0;
	
	assert(array_element != NULL && key_element != NULL);
	assert(array_element->value_type == JSON_integer || array_element->value_type == JSON_double);
	assert(key_element->value_type == JSON_integer || key_element->value_type == JSON_double);

	/* Precision and rounding errors are not an issue here, even with those creeping
		errors, this is a sort, and the values will still compare properly with each
		other, even with a precision error 												*/
	if (array_element->value_type == key_element->value_type) {
		if (array_element->value_type == JSON_integer) {
			return ((array_element->value.integer > key_element->value.integer) - (array_element->value.integer < key_element->value.integer));
		} else {
			return ((array_element->value.double_number > key_element->value.double_number) - (array_element->value.double_number < key_element->value.double_number));
		}
	}

	/* Ok, they are different number types, force them both to long double */
	if (array_element->value_type == JSON_integer) {
		array_element_number = array_element->value.integer;
		return ((array_element_number > key_element->value.double_number) - (array_element_number < key_element->value.double_number));
	} else {
		key_element_number = key_element->value.integer;
		return ((array_element->value.double_number > key_element_number) - (array_element->value.double_number < key_element_number));
	}

	/* safety in case something weird happened */
	return ELEMENTS_EQUAL;
}

/* Default set of search and compare functions used to create new list_objects	*/	
functions_globals_t functions_globals = {	default_search_function_member_name,
											default_compare_function_member_names,
											default_search_function_array_values,
											default_compare_function_array_values };


/* Call the JSON Parser and return a hierarchical list_object containing the
	parsed contents of the JSON data block requested. The original JSON data 
	block will be modified.
	
	The JSON data block can be parsed in 1 or 2 passes. If the JSON data 
	originates from an unknown or unreliable source, it would be wise to pass 
	it through the validator json_valid_data_block() first. Once identified as 
	valid it can be parsed with this function. Alternatively, if the JSON data 
	being parsed comes from a known good and reliable source (for example ARIN)
	then sending it to the parser in one pass is probably the right choice.
	
	This function parses the contents of the JSON data block while performing 
	syntax and data validation, building a set of D-List list_objects and 
	elements that contains all the JSON data with all the whitespace, syntax 
	and non-data removed. This function, like all those in the JSON Parser
	Package, is fully thread safe and MP safe.
	
	The original JSON data block provided by the caller is heavily modified as 
	part of the parsing process. If you want to retain the original JSON data 
	block intact, you must provide a copy of it to this function.
	
	The goal of the function is to parse the JSON data block creating as little 
	as possible in additional resources, and providing the data contents in a 
	form of D-List list_objects that is natively variable and non-determinate. 
	Rather than attempting to crow-bar variable data objects into some form of 
	static C language array. This makes the parsing code much simpler to write
	and maintain, while the results of the JSON data parsing activity are much 
	easier to handle, find, or modify by the caller.
	
	For example it is now very easy to rearrange the original JSON data 
	elements, or sort them by arbitrary keys etc, or perform arbitrary searches
	for data or keys.
	
	Upon successful return, a pointer to a list_object is returned. It is very 
	important to realize that the data values (such as strings) pointed to from
	within the elements of these hierarchical list_objects, are still inside the
	original JSON data block that was provided to this function. That original 
	JSON data block was extensively modified to allow the strings it contains to
	be nul terminated etc. and left in-situ. It is very important that the root
	list_object returned to the caller is properly disposed first, BEFORE any 
	attempt is made to release the original JSON data block. 
	
	So for example:
			big_lump = (char *)malloc(big_lump_to_fit_json_data_block);
			{... read file or grab data from socket to fill big_lump ...}
			if (valid_json_object(big_lump, NULL, NULL, FALSE) {
				json_object_ptr = parse_json_data_block(big_lump, &return_type, NULL, NULL, FALSE);
				if (json_object_ptr) {
					while (still_working) {
						do something useful with the returned list_objects
							...
					}
					list_erase(json_object_ptr);
					if (json_object_ptr) free(json_object_ptr);
				}
			}
			if (big_lump) free(big_lump);
	
	The root list_object returned is self cleaning, all data is managed 
	internally, and the single list_erase of the root object, will cause all
	the element records, arrays, and sub-level list_objects to be deallocated 
	and removed. No special JSON Parser calls are required to manage the root
	list_object, all data processing can be performed with the standard D-List
	function calls. If you wish to perform sorts, searches, etc. you will need 
	to add the appropriate D-List functions to the root list_object and/or the
	hierarchical list_objects.
	
	Additionally upon return, the output argument json_data_type is set to tell
	the caller what the top level list_object retuned describes. In JSON the
	root of a data block can only have one item. It can be a single value, or 
	an array or an object. Rather than have a list_object that only has a 
	single element, that itself is a list_object pointing to an array, etc; the 
	root level list_object alway describes the top level JSON data block, so if
	that is a JSON Array, then the elements of the root list_object will be the
	elements of that array. If it is a single JSON value there will be a single
	element describing that value.
	
	You may optionally select a replacement search or compare function for the 
	list_objects to be built during the parsing operation. If none are provided, 
	then in-built default functions are used. You can select from any of the
	other available built-in function, described below, or you may write your
	own to suit your specific data processing needs. If you chose to do replace
	the default search and compare functions after the lists are created, the 
	process to add them to every list_object in the data hierarchy would be very
	tedious. You can always override a specific function for a specific 
	list_object as needed, during later processing of the parsing results.
	
	Please do not use list_set_remove_function() to remove or replace the 
	original remove_function set by this parse_json_data_block() function call.
	It would lead to significant memory leaks, and potentially accessing 
	deallocated heap storage. Other than the remove function, you are free to 
	use or manipulate the returned list_objects as needed.
	
	The JSON Parser normally converts all value tokens found in the JSON data
	block. However the basic C language has limited ability to properly 
	represent high precision exponential numbers. As this code was designed 
	to be highly portable, only requiring a minimum of C99 to compile and run,
	it uses the base forms of the language for numbers. For some users either
	the limited precision of these forms, or their inability to hold huge
	numbers will be an issue. So there is an option to instruct the parser to
	instead provide these exponent numbers as a string value instead, and 
	allow the user to use whatever methods they want to convert or use those
	numbers. Integers and numbers with no exponent are always converted.
	Integers to C type long integer, and fractional numbers to C type
	long double. It is beyond the scope of this parser to accurately deal
	with very large numbers with the precision used by some professions.
	
	You can use the json_element_t typedef, to properly access the list element 
	data. The values of JSON_value_types are used to decode what type of data
	or value each list element contains.
	
	json_data_block	a nul terminated string containing the JSON data block
					to parse. The contents of this character buffer will 
					be extensively modified, and will be unusable by the
					caller upon return. However as noted above, do not 
					release or free this buffer, until after the returned
					list_object has been erased.
	json_data_type	Upon successful return, this argument is modified to
					reflect what the top level of the JSON data block
					represents, the 3 possible options are:
						JSON_object		the list_object returned is a 
										parsed JSON object
						JSON_array		the list_object returned is a 
										parsed JSON array
						JSON_value		the list_object returned contains 
										a single JSON value element, which
										is one of the type in the list 
										JSON_value_types
	object_search_function
					a D-List search function to add to each list_object that
					represents a JSON Object, that is created by the JSON 
					parsing activity. Or NULL if the default search function
					default_search_function_member_name() is to be added. 
	object_compare_function
					a D-List compare function to add to each list_object 
					represents a JSON Object, that is created by the JSON 
					parsing activity. Or NULL if the default compare function
					default_compare_function_member_names() is to be added.
	array_search_function
					a D-List search function to add to each list_object that
					represents a JSON Array, that is created by the JSON 
					parsing activity. Or NULL if the default search function
					default_search_function_array_values() is to be added. 
	array_compare_function
					a D-List compare function to add to each list_object 
					represents a JSON Array, that is created by the JSON 
					parsing activity. Or NULL if the default compare function
					default_compare_function_array_values() is to be added.
	keep_exponent	FALSE - Exponent numbers are converted to long double
					TRUE  - Exponent numbers are retained as text strings
					
	returns 		Upon successful parsing of the JSON data block, a pointer
					to a root list_object is returned, which is the top of
					the hierarchical JSON data block.
					If there was an error found in the JSON data block, a NULL
					pointer is returned.												*/
list_object *json_parse_data_block(	char *json_data_block, 
									unsigned int *json_data_type,
									element_search object_search_function,
									element_compare object_compare_function,
									element_search array_search_function,
									element_compare array_compare_function,
									const boolean keep_exponent) {

	parser_globals_t *parser_data;
	list_object *root_json_object = NULL;
	list_object *ret_json_object = NULL;
	json_element_t *json_list_element;
	json_element_t new_element;
	int parser_status = PARSER_WORKING;

	/* Only used to test for proper completion. This function does not process any tokens */
	char *token_value;				/* the contents of the token returned by the token grabber */
	int token_type;					/* the type of token returned by the token grabber */


	/* Quick argument check before we start */
	if (! json_data_block) {
		fprintf(stderr, "ERROR  ==>  %s() Argument #1 (json_data_block) is NULL.\n", __func__);
		return NULL;
	}
	if (! json_data_type) {
		fprintf(stderr, "ERROR  ==>  %s() Argument #2 (json_data_type) is NULL.\n", __func__);
		return NULL;
	}
	/* Set up the data for this JSON Parser run */
	if (! (parser_data = (parser_globals_t *)_json_malloc(sizeof(parser_globals_t), __func__))) return NULL;

	/* Set up the parsing run */
	parser_data->current_parse_cursor = json_data_block;
	parser_data->parse_end_barrier = json_data_block + strlen(json_data_block);
	parser_data->current_saved_parse_token = PARSE_TOKEN_NONE;
	parser_data->current_parse_line_count = 0;
	parser_data->parser_last_token_returned = NULL;
	parser_data->parser_keep_exponent_as_text = keep_exponent;

	memset(&new_element, 0x00, sizeof(json_element_t));

	if (object_search_function) {
		parser_data->functions.object_search_function = object_search_function;
	} else {
		parser_data->functions.object_search_function = functions_globals.object_search_function;
	}
	if (object_compare_function) {
		parser_data->functions.object_compare_function = object_compare_function;
	} else {
		parser_data->functions.object_compare_function = functions_globals.object_compare_function;
	}
	if (array_search_function) {
		parser_data->functions.array_search_function = array_search_function;
	} else {
		parser_data->functions.array_search_function = functions_globals.array_search_function;
	}
	if (array_compare_function) {
		parser_data->functions.array_compare_function = array_compare_function;
	} else {
		parser_data->functions.array_compare_function = functions_globals.array_compare_function;
	}
	
	/* First we create a root list_object to put the first element of the JSON data
		block in. There are 3 choices here, it could be a single value, an array of
		values, or a JSON object. This root list will hold whatever that top level
		element is.	For now we assume it is a JSON Object, we will fix it later
		if it ends up being something else												*/
	root_json_object = json_create_new_list_object(JSON_object, &parser_data->functions);
	if (root_json_object) {
		/* This function call causes all the work to be done in the parsing functions 
			below, and through recursive calls to those functions. The top level has
			nothing more to do than wait for the results.								*/
		parser_status = parse_value_token(root_json_object, &new_element, FALSE, parser_data);

		/* The parsing is now finished. 
			Just to verify that there are no invalid tokens left, we attempt to
			grab a new token to see what happens, there should be nothing left ...  
			if there is something, it is an error										*/
		if (parser_status == PARSER_SUCCESSFUL) {
			token_type = parse_get_next_token(&token_value, parser_data);
			if (token_type != PARSE_TOKEN_NONE) {
				fprintf(stderr, "Parsing ERROR  ==>  %s() Excessive Top Level "\
								"Tokens Present in JSON Data Block - %d '%s'.\n",
								__func__, token_type, token_value);
				parser_status = PARSER_INVALID_SYNTAX;
			}
		}
		/* if we returned from parsing with an error code, we report it here. */
		if (parser_status != PARSER_SUCCESSFUL) {
			fprintf(stderr, "Parsing ERROR  ==>  %s() JSON Data Block -> Line #%ld; at this "\
							"token:-\n==>%.80s\n\n", __func__,
							parser_data->current_parse_line_count,
							(parser_data->parser_last_token_returned));
			/* And dispose of any resources created by the parser */
			list_erase(root_json_object);
			_json_free_(root_json_object);
		}
	}

	/* just in case somebody kept a pointer, that they shouldn't have */
	parser_data->current_parse_cursor = NULL;
	parser_data->current_saved_parse_token = PARSE_TOKEN_NONE;
	/* flush */
	_json_free_(parser_data);

	/* Although strictly speaking this breaks with the list->list mapping model of
		JSON data, we flatten the top level of the data to reduce the amount of 
		redundant list opening and error checking required by the caller when 
		reading through the JSON data elements.
		
		If the JSON data was just a single value, it becomes the only element in the root list.
		If the JSON data was an array, the root list will contain an element for 
			each value in that array.
		If the JSON data was an Object, the root list will contain an element for
			each member name : value pair in the JSON Object.
			
		The underlying parser engine actually built a structure of list-> list -> list
		we flatten that out here if needed.												*/
	if (parser_status == PARSER_SUCCESSFUL) {
		if (list_size(root_json_object) > 0) {
			/* Prepare the list_object to return to the caller. If it is a value element  only
				we leave it alone, if it is an array or json object, we flatten the top level
				so as not to have a list pointing to a list pointing to a list...				*/
			json_list_element = list_get_index(root_json_object, 0);
			if ((json_list_element->value_type == JSON_object) || 
					(json_list_element->value_type == JSON_array)) {
				list_set_remove_function(root_json_object, NULL);		/* don't want to delete everything */
				ret_json_object = json_list_element->value.object;
				*json_data_type = json_list_element->value_type;
				list_erase(root_json_object);
				_json_free_(root_json_object);
			} else {
				*json_data_type = JSON_value;
				ret_json_object = root_json_object;
			}
		} else {
			/* we parsed an empty JSON data block, it is not an error, 
				but let the caller know, the list will have 0 elements in it. */
			ret_json_object = root_json_object;
		}

		/* either return the root object found, or nothing */
		return ret_json_object;
	} else {
		return NULL;
	}
}

/* Parse the contents of a JSON Object { "name" : value, "name" : value, .... }
	
	All tokens within this JSON Object are consumed and parsed. If another
	JSON Object is embedded in here as a Value item, this function would be
	called recursively to handle it.
	
	Only called by parse_value_token(), this function parses all tokens, 
	and returns a parser status code upon completion or error.
	
	current_list_object
					The list_object of the new list to add all the elements 
					of this new object to (name/value pairs). Upon entry this
					is an empty list, this function will fully populate this
					list with the contents of the JSON Object.
	element_search	The D-List search function passed on from the caller of the
					JSON Parser function.
	element_compare	The D-List compare function passed on from the caller of the
					JSON Parser function.
	
	returns		Status of the JSON Object parsing. PARSER_SUCCESSFUL if the 
				operation was completed correctly, or a parser error status
				if there was a problem with the syntax or values of the JSON
				Object being parsed.													*/
static int parse_json_element_token(	list_object *current_list_object,
										parser_globals_t *parser_data) {

	char *token_value;				/* the contents of the token returned by the token grabber */
	int token_type;					/* the type of token returned by the token grabber */
	
	json_element_t new_element;
	int parser_status = PARSER_WORKING;
	boolean first_name_value_pair = TRUE;


	/* parse a JSON object 
		{ } or { name : value } or { name : value, name : value, .... }
	
		before we were called the structural token to open the Object was 
		consumed. Before we return we will consume the closing Structural
		Token, the caller can continue parsing without inner knowledge of 
		the JSON Object we just parsed, just that it was parsed.						*/
	while (parser_status == PARSER_WORKING) {
		memset(&new_element, 0x00, sizeof(json_element_t));
		token_type = parse_get_next_token(&token_value, parser_data);
		switch (token_type) {
			case PARSE_TOKEN_OBJECT_END:
				parser_status = PARSER_SUCCESSFUL;
				continue;
				break;									/* not needed, but cosmetically good, 
															i like my code to be really clear for my intent */
			case PARSE_TOKEN_VALUE_SEPARATOR:
				if (first_name_value_pair) {
					parser_status = PARSER_INVALID_SYNTAX;
					continue;
				} else {
					token_type = parse_get_next_token(&token_value, parser_data);
					if (token_type != PARSE_TOKEN_STRING_VALUE) {
						/* By definition this token must be a Name String, or it's invalid */
						parser_status = PARSER_INVALID_SYNTAX;
						continue;
					}
				}
				/* Yes the is the MEANT to drop though to the next CASE: 
					If we instead went for another while loop to get the next token we
					would end up with all sorts of error checking code above, which is
					messy and unnecessary, this is much simpler, and easier to read 	*/
			
			case PARSE_TOKEN_STRING_VALUE: 
				new_element.member_name = token_value;
				new_element.free_string_ptr = FALSE;
				break;
			
			default:
				parser_status = PARSER_INVALID_SYNTAX;
				continue;
		}
		if (parse_get_next_token(&token_value, parser_data) != PARSE_TOKEN_NAME_VALUE_SEPARATOR) {
			parser_status = PARSER_MISSING_SYNTAX;
			continue;
		}
		/* The value token should be next, so grab and parse it */
		parser_status = parse_value_token(current_list_object, &new_element, FALSE, parser_data);
		if (parser_status == PARSER_SUCCESSFUL)	parser_status = PARSER_WORKING;
	
		/* if there were no errors, the new name/value element was added to
			this object list, so keep parsing. If there was an error 
			parser_status will stop the while() loop */
		first_name_value_pair = FALSE;
	
	} /* 	while (parser_status == PARSER_WORKING)		*/

	return parser_status;

}


/* Parse the contents of a JSON Array [ value, value, .... ]
	
	All tokens within this JSON Array are consumed and parsed. If another
	JSON Array is embedded in here as a Value item, this function would be
	called recursively to handle it.
	
	Only called by parse_value_token(), this function parses all tokens, 
	and returns a parser status code upon completion or error.
	
	current_list_object
					The list_object of the new list to add all the elements 
					of this new array to (value elements). Upon entry this
					is an empty list, this function will fully populate this
					list with the contents of the JSON Array.
	element_search	The D-List search function passed on from the caller of the
					JSON Parser function.
	element_compare	The D-List compare function passed on from the caller of the
					JSON Parser function.
	
	returns		Status of the JSON Array parsing. PARSER_SUCCESSFUL if the 
				operation was completed correctly, or a parser error status
				if there was a problem with the syntax or values of the JSON
				Array being parsed.													*/
static int parse_json_array_token(	list_object *current_list_object,
									parser_globals_t *parser_data) {


	char *token_value;				/* the contents of the token returned by the token grabber */
	int token_type;					/* the type of token returned by the token grabber */
	
	json_element_t new_element;
	int parser_status = PARSER_WORKING;
	boolean new_array = TRUE;


	/* parse a JSON array
		[ ] or [ value ] or [ value, value, .... ]
	
		before we were called the structural token to open the Array was 
		consumed. Before we return we will consume the closing Structural
		Token, the caller can continue parsing without inner knowledge of 
		the JSON Array we just parsed, just that it was parsed.						*/

	while (parser_status == PARSER_WORKING) {
		memset(&new_element, 0x00, sizeof(json_element_t));

		/* The value token should be next, so grab and parse it */
		parser_status = parse_value_token(current_list_object, &new_element, new_array, parser_data);
		new_array = FALSE;

		/* if there were no errors, the new value element was added to
			this object list, so keep parsing. If there was an error 
			parser_status will stop the while() loop */
		if (parser_status == PARSER_ARRAY_CLOSED) {
			/* this is to handle the case of an empty array, the close token was consumed already */
			parser_status = PARSER_SUCCESSFUL;
			continue;
		}
		if (parser_status == PARSER_SUCCESSFUL)	parser_status = PARSER_WORKING;
		else continue;

		/* In a JSON array the only valid token we can find here is either a
			close Array or a Value Separator, anything else is an error.			*/
		token_type = parse_get_next_token(&token_value, parser_data);
		switch (token_type) {
			case PARSE_TOKEN_ARRAY_END:
				parser_status = PARSER_SUCCESSFUL;
				continue;
				break;									/* not needed, but cosmetically good */

			case PARSE_TOKEN_VALUE_SEPARATOR:
				continue;
				break;

			default:
				parser_status = PARSER_INVALID_SYNTAX;
				continue;
		}
	
	} /* 	while (parser_status == PARSER_WORKING)		*/

	return parser_status;
}

/* Although this is the heart of the JSON Parser, the main hard work is actually done
	in the function parse_get_next_token(). This dispatches the results of getting the
	next token.																			*/
static int parse_value_token(	list_object *current_list_object, 
								json_element_t *new_element, 
								const boolean new_open_array,
								parser_globals_t *parser_data) {

	char *token_value;				/* the contents of the token returned by the token grabber */
	int token_type;					/* the type of token returned by the token grabber */
	
	int parser_status = PARSER_WORKING;

	/* the caller thinks the next token is a value token. */
	token_type = parse_get_next_token(&token_value, parser_data);
	switch (token_type) {
		case PARSE_TOKEN_ARRAY_BEGIN:
			new_element->value_type = JSON_array;
			new_element->value.object = json_create_new_list_object(JSON_array, &parser_data->functions);
			if (! new_element->value.object) {
				parser_status = PARSER_INTERNAL_ERROR;
			} else {
				list_append(current_list_object, new_element);
				parser_status = parse_json_array_token(new_element->value.object, parser_data);
			}
			break;

		case PARSE_TOKEN_OBJECT_BEGIN:
			new_element->value_type = JSON_object;
			new_element->value.object = json_create_new_list_object(JSON_object, &parser_data->functions);
			if (! new_element->value.object) {
				parser_status = PARSER_INTERNAL_ERROR;
			} else {
				list_append(current_list_object, new_element);
				parser_status = parse_json_element_token(new_element->value.object, parser_data);
			}
			break;

		case PARSE_TOKEN_ARRAY_END:
			/* special case for an empty array */
			if (new_open_array) parser_status = PARSER_ARRAY_CLOSED;
			else parser_status = PARSER_INVALID_SYNTAX;
			break;

		case PARSE_TOKEN_OBJECT_END:
			parser_status = PARSER_INVALID_SYNTAX;
			break;

		case PARSE_TOKEN_NAME_VALUE_SEPARATOR:
			parser_status = PARSER_INVALID_SYNTAX;
			break;

		case PARSE_TOKEN_VALUE_SEPARATOR:
			parser_status = PARSER_MISSING_VALUE;
			break;

		case PARSE_TOKEN_LITERAL_TRUE:
			new_element->value_type = JSON_boolean;
			new_element->value.boolean_value = TRUE; 
			break;

		case PARSE_TOKEN_LITERAL_FALSE:
			new_element->value_type = JSON_boolean;
			new_element->value.boolean_value = FALSE; 
			break;

		case PARSE_TOKEN_LITERAL_NULL:
			new_element->value_type = JSON_null;
			break;

		case PARSE_TOKEN_STRING_VALUE:
			new_element->value_type = JSON_string;
			new_element->value.string = token_value; 
			new_element->free_string_ptr = FALSE;
			break;

		case PARSE_TOKEN_NUMBER_VALUE:
			parser_status = parse_number_value(token_value, new_element, parser_data);
			if (parser_status == PARSER_SUCCESSFUL) parser_status = PARSER_WORKING;
			break;

		case PARSE_TOKEN_NONE:
			parser_status = PARSER_MISSING_VALUE;
			break;

		default:
			parser_status = PARSER_INVALID_SYNTAX;				/* This should never be able to happen */
			break;			
	}

	/* add the new element to this object list, and return. */
	if (parser_status == PARSER_WORKING) {
		list_append(current_list_object, new_element);
		parser_status = PARSER_SUCCESSFUL;
	}
	
	return parser_status;
}

/* Used by the parser to grab each token from the JSON data block, and keep
	track of where in the data block the parser engine is.
	
	This function uses global values initially set by the top level of the
	parser, but from then on only updated and used by this function.
	
	Any strings returned are nul terminated in-situ. This function modifies
	and erases portions of the original JSON data block, any token that may
	have been erased by these activities is preserved in global storage for
	later retrieval.
	
	Returns a pointer to the next token, and a return code stating what
	the type of token it is (JSON_string, JSON_array etc.)								*/
static int parse_get_next_token(char **next_token,
								parser_globals_t *parser_data) {

	char *cursor;
	char *string_cursor;							/* Used only for string tokens */
	char *hex_cursor;								/* Used only for \u escapes in case of nul bytes */
	unsigned char hex_byte_codes[4];
	int token_type = PARSE_TOKEN_NONE;
	
	int i, j;
	boolean match_found = FALSE;
	boolean syntax_error = FALSE;

	*next_token = NULL;							/* set a default response */

	/* Before we parse any further, we make sure that a token was not 
		set aside last time this function was called. This can happen
		if the JSON token last parsed, was terminated by a token with
		no whitespace between them, such as a number
			e.g. "label" : 2345, 
				or	[ "name", 3847.909]
		this would force the parser to write a nul terminator over the 
		ending structural token. If that happens the original token is 
		retained for the next pass through this function, so it is not lost.	*/
	if (parser_data->current_saved_parse_token > PARSE_TOKEN_NONE) {
		token_type = parser_data->current_saved_parse_token;
		parser_data->current_saved_parse_token = PARSE_TOKEN_NONE;
		return token_type;
	}
	
	cursor = parser_data->current_parse_cursor;
	/* scan the JSON buffer from the last point we left off, to find the next token */
	while ((! match_found) && (cursor < parser_data->parse_end_barrier)) {
		if (*cursor == '\n') {
			++parser_data->current_parse_line_count;						/* keep track of line numbers for reporting purposes */
			++cursor;
			continue;
		}
		/*	We are not inside a string, so whatever we find must be
				• a structural token
				• a literal name token
				• a number
				• or a whitespace
				anything else is a syntax error.								*/
		if (is_whitespace(cursor)) {
			++cursor;
			continue;			
		}
		/* we must have hit something interesting. */
		match_found = TRUE;
	}

	parser_data->parser_last_token_returned = cursor;				/* for error reporting only */

	/* Did we reach the End of the data block ? */
	if (cursor >= parser_data->parse_end_barrier) return PARSE_TOKEN_NONE;


	/* something was found, let's analyze it, first we search the token
		table to identify what this token is. Based on that we switch on
		the results of this search and process the token as needed.			*/
	for (i = 0, match_found = FALSE; ((! match_found) && (i <NUM_PARSE_TOKENS)); ++i) {
		if (*cursor == parse_tokens[i].buffer_token) match_found = TRUE;
	}
	token_type = parse_tokens[i-1].parse_token;
	if (match_found) {
		if (token_type == PARSE_TOKEN_STRING_VALUE) {
			/* Strings are the only complicated tokens to deal with.
				We must scan the string, validate the characters, replace escape
				sequences with the correct ASCII (unicode) character, and replace
				hex code sequences with actual hex. We compress the string by
				removing all escapes, and shifting the remaining chars to fill
				the space. We terminate the string with a nul 0x00, and remove
				the closing double quote, so it is not found again later.			*/
			++cursor;											/* skip the opening quote */
			*next_token = cursor;								/* the token ptr for the caller */
			string_cursor = cursor;
			while ((*cursor != 0x22) &&  (token_type == PARSE_TOKEN_STRING_VALUE) && (cursor < parser_data->parse_end_barrier)) {
				if (*cursor == 0x5C) {
					/* we found an escape character - a '\' */
					++cursor;									/* we need the next character */
					switch (*cursor) {
						case 0x5C:								/* a '\\' */
							*string_cursor = 0x5C;
							break;
						case 0x22:								/* a '\"' */
							*string_cursor = 0x22;
							break;
						case 0x2F:								/* a '\/' */
							*string_cursor = 0x2F;
							break;
						case 0x62:								/* a '\b' */
							*string_cursor = 0x08;
							break;
						case 0x66:								/* a '\f' */
							*string_cursor = 0x0C;
							break;
						case 0x6E:								/* a '\n' */
							*string_cursor = 0x0A;
							break;
						case 0x72:								/* a '\r' */
							*string_cursor = 0x0D;
							break;
						case 0x74:								/* a '\t' */
							*string_cursor = 0x09;
							break;
						case 0x75:								/* a '\u' */
							++cursor;							/* skip the 'u' */
							/* first verify the hex digits are legitimate */
							for (j = 0; ((! syntax_error) && (j < 4)); ++j) {
								if (is_hex_digit(cursor)) {
									hex_byte_codes[j] = bytes_hex_value(cursor);
									++cursor;
								} else { 
									syntax_error = TRUE;
								}
							}
							if (! syntax_error) {
								/* there is just no point in setting up a loop for this, it certainly looks 
									better, but is much slower and GCC would probably unroll it anyway */
								/* Even though they are valid in JSON text \u0000, nul bytes
									are NOT output to the parsed character array, for obvious reasons */
								j = 0;
								hex_cursor = string_cursor;
								hex_byte_codes[0] = (hex_byte_codes[0] << 4) | hex_byte_codes[1];
								if (hex_byte_codes[0]!= 0x00) { *string_cursor++ = hex_byte_codes[0]; ++j; }
								hex_byte_codes[1] = (hex_byte_codes[2] << 4) | hex_byte_codes[3];
								if (hex_byte_codes[1]!= 0x00) { *string_cursor = hex_byte_codes[1]; ++j; }
								
								if (j < 2) {					/* if j == 2 everything is fine */
									/* there must have been a nul byte in there, we must adjust the output pointer */
									if ((j == 0) || ((j == 1) && (string_cursor != hex_cursor))) {
										--string_cursor;
									} 
								}

								--cursor;						/* set it to the correct location */
								break;
							} else {
								token_type = PARSE_TOKEN_ERROR;
								continue;
								break;							/* not used, but to make my point, this token is foobar */
							}
							break;							/* not used, but for clarity */
						default:
							token_type = PARSE_TOKEN_ERROR;
							break;							/* not used, but to make my point, this token is foobar */

					}		/*	switch (*cursor)	*/
				} else {		/*	if (*cursor == 0x5C)	*/
					if (string_cursor != cursor) {
						/* shift this character to its correct position in the string */
						*string_cursor = *cursor;
					}
				}
				++cursor;
				++string_cursor;
			}		/*	while ((*cursor != 0x22) &&  (token_type == PARSE_TOKEN_STRING_VALUE) && (cursor < parse_end_barrier)) 	*/
			
			/* We hit the end of the string */
			if (*cursor != 0x22) {					/* if the cursor ran over, it points to a nul char */
				/* Oops, someone wrote an unterminated string that ran 
					to the end of the JSON data block. Tisk tisk ...  */
				token_type = PARSE_TOKEN_ERROR;
				*next_token = NULL;
			} else {
				*cursor = '\0';											/* remove the closing quote from the data block */
				if (string_cursor != cursor) *string_cursor = '\0';		/* terminate the string token for the caller */
			
			}
			/* we do not check string length, as zero length strings are valid in JSON 		*/


		
		}		/*	if (token_type == PARSE_TOKEN_STRING_VALUE)	*/

		if (token_type == PARSE_TOKEN_NUMBER_VALUE) {
			*next_token = cursor;
			while (is_valid_number_token(cursor)) ++cursor;
			if (is_whitespace(cursor)) {
				*cursor = '\0';
			} else {
				/* this must be a structure token, it cannot be a value, if the JSON
					has a syntax error, it will flag as an error on the next pass */

				for (i = 0, match_found = FALSE; ((! match_found) && (i <NUM_PARSE_TOKENS)); ++i) {
					if (*cursor == parse_tokens[i].buffer_token) match_found = TRUE;
				}
				parser_data->current_saved_parse_token = parse_tokens[i-1].parse_token;
				*cursor = '\0';
			}
		}		/*	if (token_type == PARSE_TOKEN_NUMBER_VALUE)	*/

		if (token_type == PARSE_TOKEN_LITERAL_TRUE) {
			if (strncmp(cursor, PARSER_TOKEN_LABEL_TRUE, 4) == 0) {
				cursor += 3;
			} else {
				token_type = PARSE_TOKEN_ERROR;
			}
		}

		if (token_type == PARSE_TOKEN_LITERAL_FALSE) {
			if (strncmp(cursor, PARSER_TOKEN_LABEL_FALSE, 5) == 0) {
				cursor += 4;
			} else {
				token_type = PARSE_TOKEN_ERROR;
			}
		}

		if (token_type == PARSE_TOKEN_LITERAL_NULL) {
			if (strncmp(cursor, PARSER_TOKEN_LABEL_NULL, 4) == 0) {
				cursor += 3;
			} else {
				token_type = PARSE_TOKEN_ERROR;
			}
		}
		/* If this is a structural token - there is nothing to do, but return its code */
		
		parser_data->current_parse_cursor = cursor + 1;				/* where we need to start next time */
	}

	if (json_parser_debugging) {
		fprintf(stdout, "%s() token type:%d token:%s next start:%.20s\n\n",
						__func__, token_type, *next_token, parser_data->current_parse_cursor);
	}

	return token_type;
}

/* Only called by parse_value_token(), placed here to keep the code in that
	function more readable and cleaner.
	
	It is important to know that this function does not attempt to reproduce
	very large numbers of high precision, such as may be used in some 
	professions. There is an option that can be provided to the JSON Parser
	to not attempt to convert these numbers and allow the user to process 
	them instead. Under those circumstances, such numbers are rendered as
	text strings in the JSON element, for the caller to use as is needed. 				*/
static int parse_number_value(	char *token, 
								json_element_t *new_element,
								parser_globals_t *parser_data) {

	char *scanner;
	json_integer_t input_integer;
	json_double_t input_double;
	int ret;
	
	/* First look for a leading zero, a pedantic JSON syntax requirement -
		if we see a leading 0, it must be followed by a period, or an immediate nul.	*/
	scanner = token;
	if (*token == '-') ++scanner;
	if ((*scanner == '0') && (strlen(scanner) > 1)) {
		++scanner;						/* skip the zero */
		if ((*scanner != '.') && (*scanner != 'e') && (*scanner != 'E')) {
			return PARSER_BAD_VALUE;
		}
	}

	/* Do we have an integer or a fractional / exponent number ? */
	if (*token == '-') {
		scanner = strpbrk((token+1), json_non_integer_chars);
	} else {
		scanner = strpbrk(token, json_non_integer_chars);
	}

	if (scanner) {
		/* Yes we have a non integer value. */
		if ((parser_data->parser_keep_exponent_as_text) && (strpbrk(token, json_exponent_chars) != NULL)) {
			/* The caller wants these left alone */
			new_element->value_type = JSON_float_str;
			new_element->value.string = token;
			new_element->free_string_ptr = FALSE;
			ret = 1;
		} else {
			/* either the user wants us to convert the number, or it is not an exponent number */
			new_element->value_type = JSON_double;
			ret = sscanf (token, "%Lf", &input_double);
			if (ret == 1) {
				new_element->value.double_number = input_double;
			} else {
				new_element->value.double_number = 0;
			}
		}
	} else {
		/* No, just a regular type of number, a good old integer. */
		new_element->value_type = JSON_integer;
		ret = sscanf (token, "%" SCNd64, &input_integer);
		if (ret == 1) {
			new_element->value.integer = input_integer;
		} else {
			new_element->value.integer = 0;
		}
	}

	return ((ret == 1) ? PARSER_SUCCESSFUL : PARSER_BAD_VALUE);
}


/* Checks a character to see if it is, or is not, whitespace. 

	Called by both the parser and the validator when needed.

	returns		TRUE	Character is whitespace
				FALSE	Is not															*/
static inline boolean is_whitespace(const char *test_char) {

	int i = 0;
	boolean char_is_whitespace = FALSE;
	
	while ((i < 4) && (! char_is_whitespace)) {
		if (*test_char == json_whitespace_chars[i++]) char_is_whitespace = TRUE;
	}

	return char_is_whitespace;
}

#ifdef DBW_JSON_NOT_NEEDED_YET
/* Not used, left in the code just in case it is needed later */
static inline boolean is_valid_structural_token(const char *test_char) {

	int i = 0;
	boolean char_is_valid_token = FALSE;
	
	while ((i < 5) && (! char_is_valid_token)) {
		if (*test_char == json_structural_tokens[i++]) char_is_valid_token = TRUE;
	}

	return char_is_valid_token;
}

/* Not used, left in the code just in case it is needed later */
static inline boolean is_valid_literal_token(const char *test_char) {

	int i = 0;
	boolean char_is_valid_token = FALSE;
	
	while ((i < 3) && (! char_is_valid_token)) {
		if (*test_char == json_literal_tokens[i++]) char_is_valid_token = TRUE;
	}

	return char_is_valid_token;
}
#endif		/*	#ifdef DBW_JSON_NOT_NEEDED_YET	*/

/* Called by the JSON Parser to validate number values */
static inline boolean is_valid_number_token(const char *test_char) {

	int i = 0;
	boolean char_is_valid_token = FALSE;
	
	while ((i < 15) && (! char_is_valid_token)) {
		if (*test_char == json_number_tokens[i++]) char_is_valid_token = TRUE;
	}

	return char_is_valid_token;
}

/* Called by both the parser and validator to test a character. */
static inline boolean is_hex_digit(const char *test_char) {

	if (*test_char < 0x30) 
		return FALSE;
	if (*test_char < 0x40) 
		return TRUE;
	if (*test_char == 0x40) 
		return FALSE;
	if (*test_char < 0x47) 
		return TRUE;
	if (*test_char < 0x61) 
		return FALSE;
	if (*test_char < 0x67) 
		return TRUE;
	return FALSE;
}

/* return the actual hex code for a character hex digit. 

	No validation is performed on the digit converted, if you
	pass a non-hex digit you will get garbage back.										*/
static inline unsigned char bytes_hex_value(const char *input) {

	
	if (*input < 0x40) {
		return *input - 0x30;
	}
	if (*input < 0x47) {
		return *input - 0x37;
	}
	if (*input < 0x67) {
		return *input - 0x57;
	}

	return 0xFF;
}




/*	=========================================================================

		Section for the JSON Data Validator.
		
		All functions below here are related to the validator only, they
		are not used by the parser.
		
		If you want to save footprint on a small or embedded system, and
		only use the parser, you can eliminated these functions below here.

	=========================================================================	*/

								
/* Call the JSON syntax validator and return a boolean value based on either
	successful or failed validation.
	
	The JSON data block is parsed in a linear pass, validating each character 
	of the provided JSON data block, all tokens, values, whitespace and syntax 
	are checked. This function, like all those in the JSON Parser Package, is 
	fully thread safe and MP safe.
	
	Checks are made for validity of syntax, contents, and the relative
	positioning of all tokens, structural, literal, object, array and value.
	
	The contents of the character buffer will be remain unmodified. Validation 
	only scans the JSON data block and makes no changes. If JSON errors are 
	found a pointer to the invalid token can optionally be returned.
	
	This JSON Validation will stop when it encounters a JSON syntax or value 
	error, it will not perform any further validation of the data block. 
	
	If the caller does not surpress warnings, information and the location of
	the JSON error are written to stderr. If the caller suppressed warnings and
	did not know why the JSON object was rejected by the validator, call this
	function again without suppressing warnings to see the specific error
	messages involved.
	
	json_object		a nul terminated string containing the JSON object to 
					validate. 
	object_error	if a syntax or value error is found in the JSON data 
					block, a pointer the portion of the JSON data block in 
					error is returned in this argument. This argument can 
					be NULL in which case no pointer is returned upon 
					encountering a JSON error.
	line_number		if a syntax or value error is found in the JSON data 
					block, the line number of the JSON token in error is 
					returned in this argument. This argument can be NULL
					in which case no line number is returned if a JSON 
					error was detected.
	suppress_warnings
					FALSE 	DO not suppress JSON error warnings written to
							stderr
					TRUE	Suppress possible error warnings, no output is
							written.
					
	returns 	TRUE 	- Successful validation of the JSON object syntax,
							grammar and values, no language errors.
					
				FALSE	- There was an error found in the JSON data block,
							if provided as arguments, a pointer is returned 
							in syntax_error and an integer in line_number
							corresponding to the found error. If warnings
							were not suppressed an message detailing the
							JSON error was written to stderr.							*/
boolean json_valid_data_block(	const char *json_object, 
								char **object_error, 
								unsigned long int *line_number,
								const boolean suppress_warnings) {
	
	char *array_scanner;
	char *cursor;
	char *barrier;
	
	long int line_count = 1;
	int j;
	int object_depth = 0;

	struct array_value_chk {
		int count;
		boolean obj_name_string;
		boolean colon_separator_found;
		boolean value_token_found;
	} array_depth[MAX_NESTED_DEPTH];

	boolean syntax_error = FALSE;
	boolean in_a_string = FALSE;
	
	boolean first_token;
	boolean token_separator;
	boolean array_token_found;

	boolean char_is_digit;

	/* This function was deliberately written in a different model than the Parser
		or the Generator. The author did not want the Validator to just a warmed
		over Parser that did no real work. Rather than using a recursive model,
		this scans the JSON data block pedantically byte by byte counting everything
		and every array and every object. It was designed to catch JSON errors in
		a different way, so a JSON data block passed through both that passed, must 
		be totally clean. Personally i much prefer the recursive code model, but
		this has its place as well.														*/

	/* lets get ready to rumble */
	srand(time(NULL));							/* seed random# gen */
	
	/* are we running in debug mode ? */
	json_parser_debugging = (getenv(ENVIRONMENT_VAR_DEBUGGING) != NULL);

	cursor = (char *)json_object;
	barrier = cursor + strlen(json_object);
	
	array_depth[object_depth].count = 0;
	array_depth[object_depth].obj_name_string = FALSE;
	array_depth[object_depth].colon_separator_found = FALSE;
	array_depth[object_depth].value_token_found = FALSE;

	while ((! syntax_error) && (cursor < barrier)) {
		if (*cursor == '\n') {
			++line_count;				/* keep track of line numbers for reporting purposes */
		}
		if (in_a_string) {
			switch (*cursor) {
				case 0x5C:										/* a '\' */
					++cursor;
					if ((*cursor == 0x5C) ||							/* a '\' */
							(*cursor == 0x22) ||						/* a '"' */
							(*cursor == 0x2F) ||						/* a '/' */
							(*cursor == 0x62) ||						/* a '\b' */
							(*cursor == 0x66) ||						/* a '\f' */
							(*cursor == 0x6E) ||						/* a '\n' */
							(*cursor == 0x72) ||						/* a '\r' */
							(*cursor == 0x74)) {						/* a '\t' */
						break;
					}
					if (*cursor == 0x75) {								/* a 'u' */
						for (j = 0, ++cursor; ((! syntax_error) && (j < 4)); ++j) {
							if (! is_hex_digit(cursor)) syntax_error = TRUE;
							else ++cursor;
						}
						--cursor;
						break;
					}
					syntax_error = TRUE;
					break;

				case 0x22:										/* a '"' */
					in_a_string = FALSE;
					break;

				default:
					break;
			}
		} else {
			/* If we are not inside a string, then whatever we find must be
					• a structural token
					• a literal name token
					• a number
					• or a whitespace
				anything else is a syntax error.								*/
			switch (*cursor) {
				/* Structural Tokens */
				case '{': 
					if (array_depth[object_depth].count > 0) {
						array_depth[object_depth].value_token_found = TRUE;
					}
					++object_depth; 
					if (object_depth > MAX_NESTED_DEPTH - 1) {
						if (! suppress_warnings) 
							fprintf(stderr, " %s() Too many nested JSON Objects, this syntax "\
											"validator has a maximum of %u.\n", __func__, 
											MAX_NESTED_DEPTH);
						syntax_error = TRUE;
					}
					array_depth[object_depth].count = 0;
					array_depth[object_depth].obj_name_string = FALSE;
					array_depth[object_depth].colon_separator_found = FALSE;
					array_depth[object_depth].value_token_found = FALSE;
					syntax_error = (! is_name_value_pair_valid(	&array_depth[object_depth-1].obj_name_string, 
																&array_depth[object_depth-1].colon_separator_found, 
																&array_depth[object_depth-1].value_token_found));
					break;

				case '}': 
					if (array_depth[object_depth].count != 0) {
						if (! suppress_warnings)
							fprintf(stderr, " %s() Object Structure Closed with unclosed Array.\n", __func__);
						syntax_error = TRUE;
						break;
					}
					--object_depth;
					if (object_depth < 0) {
						if (! suppress_warnings) 
							fprintf(stderr, " %s() Object Structure Close Token '}' "\
											"with no matching open Token.\n", __func__);
						syntax_error = TRUE;
					}
					break;

				case '[': 
					++array_depth[object_depth].count;
					if (array_depth[object_depth].count > MAX_NESTED_DEPTH - 1) {
						if (! suppress_warnings) 
							fprintf(stderr, " %s() Too many nested JSON Arrays, this "\
											"syntax validator has a maximum of %u.\n",
											__func__, MAX_NESTED_DEPTH);
						syntax_error = TRUE;
						break;
					}
					/* This is only to check that the value tokens within an array are
						properly separated by commas, not to validate the value tokens
						themselves. That is handled by the normal token code. This is
						code is an approximation, that works. The parser uses recursion
						and functions fully and correctly, this is simpler scan to 
						produce and true or false analysis.								*/
					for (array_scanner = cursor + 1, first_token = TRUE, token_separator = array_token_found =  FALSE; 
							((array_scanner < barrier) && 
							(! syntax_error) && 
							(*array_scanner != '[') &&
							(*array_scanner != '{') &&
							(*array_scanner != ']'))
							; ) {
						if (*array_scanner == '}') {
							syntax_error = TRUE;
							break;
						}
						if (is_whitespace(array_scanner)) {
							++array_scanner;
							continue;
						}
						if (*array_scanner == ',') {
							if (token_separator) {
								if (! suppress_warnings) 
									fprintf(stderr, " %s() Array Value Tokens separated "\
													"by multiple commas.\n", __func__);
								syntax_error = TRUE;
								continue;
							}
							if (! array_token_found) {
								if (! suppress_warnings) 
									fprintf(stderr, " %s() Comma in Array with no preceding "\
													"Value Token.\n", __func__);
								syntax_error = TRUE;
								continue;
							}
							token_separator = TRUE;
							array_token_found = FALSE;
							++array_scanner;
						} else {
							/* by definition anything in an array that is not a whitespace, 
								and not a comma, must be a token */
							array_token_found = TRUE;
							if (first_token) {
								first_token = FALSE;
							} else {
								if (token_separator) {
									token_separator = FALSE;
								} else {
									if (! suppress_warnings) 
										fprintf(stderr, " %s() Array Value Tokens "\
														"not separated by a comma.\n", __func__);
									syntax_error = TRUE;
									continue;
								}
							}
							/* skip the rest of the value */
							if (*array_scanner == '"') {
								++array_scanner;
								while (*array_scanner != '"') ++array_scanner;
								++array_scanner;
							} else {
								while ((*array_scanner != ',') && (! is_whitespace(array_scanner))) {
									++array_scanner;
								}
							}
						}
					}		/*	for (array_scanner = cursor + 1, first_token = TRUE, etc	*/
					if (syntax_error)
						break;
					
					syntax_error = (! is_name_value_pair_valid(	&array_depth[object_depth].obj_name_string, 
																&array_depth[object_depth].colon_separator_found, 
																&array_depth[object_depth].value_token_found));
					break;

				case ']': 
					--array_depth[object_depth].count;
					if (array_depth[object_depth].count < 0) {
						if (! suppress_warnings) 
							fprintf(stderr, " %s() Array Structure Close Token ']' "\
											"with no matching open Token.\n", __func__);
						syntax_error = TRUE;
					}
					break;

				case ':': 
					if (array_depth[object_depth].count > 0) {
						if (! suppress_warnings) fprintf(stderr, " %s() Colon separator inside an Array.\n", __func__);
						syntax_error = TRUE;
					}
					if (! array_depth[object_depth].obj_name_string) {
						if (! suppress_warnings) 
							fprintf(stderr, " %s() Colon separator found without a "\
											"leading Object Name.\n", __func__);
						syntax_error = TRUE;
					}
					if (array_depth[object_depth].colon_separator_found) {
						if (! suppress_warnings) 
							fprintf(stderr, " %s() Extra Colon separator found.\n", __func__);
						syntax_error = TRUE;
					}

					array_depth[object_depth].colon_separator_found = TRUE;
					break;

				case ',':
					if (! array_depth[object_depth].value_token_found) {
						if (! suppress_warnings) 
							fprintf(stderr, "%s() Extra Comma found, not separating "\
											"Value Tokens.\n", __func__);
						syntax_error = TRUE;
					}
					array_depth[object_depth].value_token_found = FALSE;
					break;
					
				/* Literal Name Tokens */

				case 't':
					if (strncmp(cursor, PARSER_TOKEN_LABEL_TRUE, 4) == 0) {
						cursor += 3;
						syntax_error = (! is_name_value_pair_valid(	&array_depth[object_depth].obj_name_string, 
																	&array_depth[object_depth].colon_separator_found, 
																	&array_depth[object_depth].value_token_found));
					} else {
						syntax_error =TRUE;
					}
					break;

				case 'f':
					if (strncmp(cursor, PARSER_TOKEN_LABEL_FALSE, 5) == 0) {
						cursor += 4;
						syntax_error = (! is_name_value_pair_valid(	&array_depth[object_depth].obj_name_string, 
																	&array_depth[object_depth].colon_separator_found, 
																	&array_depth[object_depth].value_token_found));
					} else {
						syntax_error =TRUE;
					}
					break;

				case 'n':
					if (strncmp(cursor, PARSER_TOKEN_LABEL_NULL, 4) == 0) {
						cursor += 3;
						syntax_error = (! is_name_value_pair_valid(	&array_depth[object_depth].obj_name_string, 
																	&array_depth[object_depth].colon_separator_found, 
																	&array_depth[object_depth].value_token_found));
					} else {
						syntax_error =TRUE;
					}
					break;

				/* String Token (starting tokens only, end token is handled by if (in_a_string) above */
				case '"': 
					in_a_string = TRUE;
					if ((array_depth[object_depth].count == 0) && 
							(array_depth[object_depth].obj_name_string)) {
						syntax_error = (! is_name_value_pair_valid(	&array_depth[object_depth].obj_name_string, 
																	&array_depth[object_depth].colon_separator_found, 
																	&array_depth[object_depth].value_token_found));
					} else {
						if ((array_depth[object_depth].count == 0) && 
								(! array_depth[object_depth].obj_name_string)) {
							array_depth[object_depth].obj_name_string = TRUE;
						}
					}
					if (array_depth[object_depth].count > 0) {
						array_depth[object_depth].value_token_found = TRUE;
					}
					break;

				/* Number Token */
				case '-': case '0': case '1': case '2': case '3': case '4': case '5':
				case '6': case '7': case '8': case '9':
					if (*cursor == '-') ++cursor;		/* skip over it, we might expect to see a minus in the front of a number */
					
					/* if we see a leading 0, it must be followed by a period, or an end token */
					if (*cursor == '0') {
						++cursor;						/* skip the zero */
						if ((*cursor != '.') && (! is_valid_value_end_token(cursor))) {
							if (! suppress_warnings) 
								fprintf(stderr, "%s() Leading Zero found in Value Token.\n", __func__);
							syntax_error = TRUE;
							break;
						}
					} else {
						/* digit found must be 1-9, so check the rest out */
						for (j = 1, char_is_digit = TRUE; 
								((j < MAX_NUMBER_TOKEN_LENGTH) && (char_is_digit)); ++j) {
							if (! isdigit(*cursor)) char_is_digit = FALSE;
							else ++cursor;
						}
						if (j > MAX_NUMBER_TOKEN_LENGTH-1) {
							if (! suppress_warnings) 
							fprintf(stderr, "%s() Excessive sized number found, it "\
											"exceeds what the C language can handle.\n", __func__);
							syntax_error = TRUE;
							break;
						}
						if ((*cursor != '.') && 
								(*cursor != 'e') && 
								(*cursor != 'E') && 
								(! is_valid_value_end_token(cursor))) {
							if (! suppress_warnings) 
								fprintf(stderr, "%s() Invalid Number containing non-digits, "\
												"found in Value Token.\n", __func__);
							syntax_error = TRUE;
							break;
						}
					}
					if (*cursor == '.') {
						/* deal with the fractional side of the number */
						++cursor;							/* skip over the period */
						for (j = 1, char_is_digit = TRUE; ((j < MAX_NUMBER_TOKEN_LENGTH) && (char_is_digit)); ++j) {
							if (! isdigit(*cursor)) char_is_digit = FALSE;
							else ++cursor;
						}
						if (j > MAX_NUMBER_TOKEN_LENGTH-1) {
							if (! suppress_warnings)
								fprintf(stderr, "%s() Excessive sized number found, it exceeds "\
												"what the language can handle.\n", __func__);
							syntax_error = TRUE;
							break;
						}
					}
					if ((*cursor == 'e') || (*cursor == 'E')) {
						++cursor;
						if ((*cursor == '+') || (*cursor == '-')) ++cursor;
						for (j = 1, char_is_digit = TRUE; ((j < MAX_NUMBER_TOKEN_LENGTH) && (char_is_digit)); ++j) {
							if (! isdigit(*cursor)) char_is_digit = FALSE;
							else ++cursor;
						}
						if (j > MAX_NUMBER_TOKEN_LENGTH-1) {
							if (! suppress_warnings) 
								fprintf(stderr, "%s() Excessive sized number found, it exceeds "\
												"what the C language can handle.\n", __func__);
							syntax_error = TRUE;
							break;
						}
					} 
					if (! is_valid_value_end_token(cursor)) {
						if (! suppress_warnings) 
							fprintf(stderr, "%s() Invalid Number containing non-digits, found "\
											"in Value Token.\n", __func__);
						syntax_error = TRUE;
						break;
					}
					syntax_error = (! is_name_value_pair_valid(	&array_depth[object_depth].obj_name_string, 
																&array_depth[object_depth].colon_separator_found, 
																&array_depth[object_depth].value_token_found));
					/* All numbers are terminated by whitespace, but we have consumed
						the char of that whitespace, need to rewind.					*/
					--cursor;
					
					break;


				/* Invalid Token */
				case '/':
					if ((*(cursor+1) == JSON_ILLEGAL_COMMENT_1) || 
							(*(cursor+1) == JSON_ILLEGAL_COMMENT_2)) {
						if (! suppress_warnings) 
							fprintf(stderr, "%s() Comments are not permitted by JSON Specification "\
											"RFC 8259. Please remove them and resubmit the "\
											"JSON data block for validation.\n", __func__);
						syntax_error = TRUE;
					}
				
				default:
					/* Whitespace check first */
					if (is_whitespace(cursor)) 
						break;
					
					if (! suppress_warnings) 
						fprintf(stderr, "%s() Invalid Object Token found.\n", __func__);
					syntax_error = TRUE;
					break;
			}
		}		/* if (in_a_string) else	*/
		++cursor;
	}
	
	/* Check the results and return to the caller */
	if (syntax_error) {
		if (! suppress_warnings) 
			fprintf(stderr, "%s() JSON Grammar Error located near Line #%ld-\n'%.80s' .\n\n",
							__func__, line_count, (cursor-2));
		if (object_error) *object_error = cursor;
		if (line_number) *line_number = line_count;
	} else {
		if (object_error) *object_error = NULL;
		if (line_number) *line_number = 0;
	}
	
	return (! syntax_error);
}



/* Test the boolean values presented, and issue errors or reset the values
	as required. This test is based on the premise that all Names and Values
	must be in pairs, must be separated by a colon, and can only occur in a
	JSON Object, not in an JSON Array.
	
	This function must only be called when a value item is found, that should
	have a matching Name and colon already parsed. It is to verify compliance
	(and reduce the occurrence of endless redundant code in the main validator 
	function).
	
	This function is only called by the validator. It is NOT used by the JSON
	parser.
	
	obj_name_string			Boolean Value based on if an Object Name has 
							been found yet for this Name/Value pair
	colon_separator_found	Boolean Value based on if a colon has been 
							found yet, that is supposed to separate the
							Name/Value pair
	value_found				Upon return, if syntax is VALID, set to TRUE
							if syntax is invalid, set to FALSE.
							This argument is present to prevent more repetitive
							code in the main syntax checker.
	returns					TRUE 	syntax is fine
							FALSE	syntax is in error.									*/
static inline boolean is_name_value_pair_valid(	boolean *obj_name_string, 
												boolean *colon_separator_found,
												boolean *value_found) {
	boolean syntax_error = FALSE;
	
	if ((obj_name_string) && (! colon_separator_found)) {
		fprintf(stderr, " %s() Name / Object Value Pair found, with no Colon separator.\n", __func__);
		syntax_error = TRUE;
	}
	if ((! obj_name_string) && (colon_separator_found)) {
		fprintf(stderr, " %s() A Colon and Object Value found, with no prior Name.\n", __func__);
		syntax_error = TRUE;
	}
	if ((obj_name_string) && (colon_separator_found)) {
		*obj_name_string = FALSE;
		*colon_separator_found = FALSE;
	}

	*value_found = !syntax_error;
	return (! syntax_error);
}


static inline boolean is_valid_value_end_token(const char *test_char) {

	int i = 0;
	boolean char_is_valid_end_token = FALSE;
	
	while ((i < 7) && (! char_is_valid_end_token)) {
		if (*test_char == json_value_end_token_chars[i++]) char_is_valid_end_token = TRUE;
	}

	return char_is_valid_end_token;
}




/*	=========================================================================

		Section for the JSON Data Generator.
		
		All functions below here are related to the Generator only, they
		are not used by the Parser or the Validator.
		
		If you want to save footprint on a small or embedded system, and
		only use the parser, you can eliminated these functions below here.

	=========================================================================	*/

								



/* Using a JSON Object data tree, represented by a hierarchical D-List list_object
	structure, create a textural version of that JSON data presented and return a 
	character array with that JSON data in it in strict JSON Grammar format.
	
	This function is essentially the reverse of parse_json_data_block(). It
	will walk through a D-List list_object structure containing the 
	representation of a JSON data block, and create a textural version of
	that JSON data ready to be stored or transmitted over a network.  This 
	function, like all those in the JSON Parser Package, is fully thread safe 
	and MP safe.
	
	The input data must be in the correct form and conform to JSON grammar.
	Using the D-List hierarchical structure, there is very little that can be
	in error in that structure, so the generation function produces few error
	messages. The only real data problems that can occur are simple mistakes
	such as providing inaccurate JSON value types, or forgetting to provide a
	name for JSON Object members. 
	
	All textural strings that are created by the JSON Generator assume the bytes
	are either ASCII or UTF-8. It does not enforce any encoding standards. All
	string bytes are passed through without change, except those that are 
	required by the relevant JSON standards to be escaped. This includes all 
	control characters which are either escaped with a '\' or by hex code.
	
	The JSON Generator will produce human readable JSON text, that is well 
	structured and easy to read. If you take an existing JSON data block,
	parse it with the JSON Parser, and then run that output list_object 
	through the JSON Generator, you will get back almost identical text
	with the exception of probably different formatting style, and most
	\uxxxx hex escapes in the original will be reproduced as UTF-8 instead.

	json_data		A pointer to a root list_object for the JSON data structure
					the caller wishes to use to generate a JSON Text Data Block.
	json_data_type	The type of data found at the top level, either JSON_object,
					JSON_array or JSON_value. This is the same kind of value 
					type that would be returned if a call was made to the
					function parse_json_data_block()
	json_text_buffer 
					A pointer to a location to write the pointer to the data block
					created by the JSON Generator.

	returns		 0	All JSON elements processed correctly, no syntax / grammar errors.
				 1	JSON Object was empty, but processed correctly.
				-1	A JSON grammar or syntax error was found below the top level
				-2	An error was found in the JSON syntax at the top level
				-3	An argument error occurred
				-4	An error occurred during memory allocation, there is no
					output buffer available.											*/
int json_generate_data_block(	list_object *json_data, 
								unsigned int json_data_type,
								char **json_text_buffer) {

	generator_globals_t *generator_data;
	char temp_buffer[MAX_NUMBER_TOKEN_LENGTH*2];
	char *value_buffer;
	int ret = 0;

	/* Note to users. Do not rely on this to generate exponent values. Please
		create them yourself and store in the list as a JSON_float_str, so you
		are happy with the results. It is beyond the scope of this package to
		deal with high precision numbers to the accuracy required by some users
		in either academic or high precision industry settings.
		
		This generator will convert integer and fractional numbers correctly, but
		exponent numbers are only handled using base C number forms.					*/
	
	assert(json_data);
	assert(json_data_type == JSON_object || json_data_type == JSON_array || json_data_type == JSON_value);
	assert(json_text_buffer);

	/* Set up the data for this JSON Generator run */
	if (! (generator_data = (generator_globals_t *)_json_malloc(sizeof(generator_globals_t), __func__))) return -4;
	generator_data->current_buffer_size = 0;
	generator_data->current_output_offset = 0;
	generator_data->output_buffer = NULL;
	generator_data->indent_level = 0;

	if (! json_data) {
		fprintf(stderr, "ERROR  ==>  %s() Argument #1 (json_data) is NULL.\n", __func__);
		return -3;
	}

	if (! json_text_buffer) {
		fprintf(stderr, "ERROR  ==>  %s() Argument #3 (json_text_buffer) is NULL.\n", __func__);
		return -3;
	}
	*json_text_buffer = NULL;

	if (list_is_empty(json_data)) {
		fprintf(stderr, "Warning  ==>  %s() JSON Data list_object is empty.\n", __func__);
		return 0;
	}

	/* This works in a similar recursive model to the Parser, we make one function
		call up here, and the rest is handled by recursive calling the functions below.	*/
	if (json_data_type == JSON_object) {
		ret = json_generate_object(json_data, generator_data); 
	}
	if (json_data_type == JSON_array) {
		ret = json_generate_array(json_data, generator_data); 
	}
	if (json_data_type == JSON_value) {
		if (list_size(json_data) == 1) {
			value_buffer = json_generate_value(list_get_index(json_data, 0), temp_buffer); 
			if (value_buffer) {
				generator_data->output_buffer = json_generate_write(value_buffer, ADD_NEW_LINE, generator_data);
				if (value_buffer != temp_buffer)  _json_free_(value_buffer); 		/* free the temp resource we were provided */
			} else {
				fprintf(stderr, "JSON Grammar ERROR  ==>  %s() invalid value at the top level.\n", __func__);
				ret = -2;
			}
		} else {
			fprintf(stderr, "JSON Grammar ERROR  ==>  %s() more than one value at the top level.\n", __func__);
			ret = -2;
		}
	}
		
	if (generator_data->output_buffer) {
		generator_data->output_buffer[generator_data->current_output_offset++] = '\0';
		/* shrink wrap */
		generator_data->output_buffer = realloc(generator_data->output_buffer, generator_data->current_output_offset);
		if (! generator_data->output_buffer) {
			fprintf(stderr, "ERROR  ==>  %s() realloc() failed while reducing buffer to %lu bytes.\n\n",
					__func__, (generator_data->current_output_offset));
			ret = -4;
		} else {
			*json_text_buffer = generator_data->output_buffer;
		}
	} else {
		fprintf(stderr, "ERROR  ==>  %s() a major error occurred during JSON data processing.\n", __func__);
		ret = -4;
	}

	if (ret < 0) {
		fprintf(stderr, "JSON Grammar ERROR  ==>  %s() while processing your JSON data elements.\n"\
						"The output buffer returned will end at the last place in your input data, that\n"\
						"the JSON grammar was known to be good.\n"\
						"Please check it and stderr for more information.", __func__);
	}

	if (ret > 0) ret = 0;						/* flush any minor warnings from below */
	_json_free_(generator_data);

	return ret;
}
/* Generate the output for a JSON Object list.
	
	Generates everything for the Object.
	
	object_data		The list_object that contains the elements of a JSON
					Object. All elements are processed.
	
	returns		0	All elements processed correctly, no syntax / grammar errors.
				1	JSON Object was empty, but processed correctly.
				-1	An error occurred, see output buffer last know good values			*/
static int json_generate_object(	list_object *object_data,
							generator_globals_t *generator_data) {

	json_element_t *object_element;

	char temp_buffer[MAX_NUMBER_TOKEN_LENGTH*2];
	char *value_buffer;
	int x = 0;
	int return_code = 0;
	boolean still_working = TRUE;

	if (list_is_empty(object_data)) {
		generator_data->output_buffer = json_generate_write(JSON_GENERATOR_EMPTY_OBJECT, STAY_ON_SAME_LINE, generator_data);
		if (generator_data->output_buffer)
			return 1;
		else
			return -1;
	}

	/* First start the JSON Object structure */
	if (list_size(object_data) > 1) {
		generator_data->output_buffer = json_generate_write(JSON_GENERATOR_NULL_STRING, ADD_NEW_LINE, generator_data);
		++generator_data->indent_level;
		if (generator_data->output_buffer) generator_data->output_buffer = json_generate_write(JSON_GENERATOR_OBJECT_BEGIN, ADD_NEW_LINE, generator_data);
	} else {
		++generator_data->indent_level;
		generator_data->output_buffer = json_generate_write(JSON_GENERATOR_OBJECT_BEGIN_1_MEMBER, STAY_ON_SAME_LINE, generator_data);
	}
	if (! generator_data->output_buffer) {
		fprintf(stderr, "JSON Generator ERROR  ==>  %s() internal problem(s) occurred.\n", __func__);
		return -1;
	}

	list_iteration_start(object_data, TAIL_OF_LIST);
	while (list_iteration_has_more(object_data) && (generator_data->output_buffer) && (still_working)) {
		object_element = list_iteration_get_next(object_data, NULL);
		if (! object_element->member_name) {
			fprintf(stderr, "JSON Grammar ERROR  ==>  %s() Member #%d of this JSON Object "\
							"has a NULL Member name pointer.\n", __func__, x);
			still_working = FALSE;
			continue;
		}
		generator_data->output_buffer = json_generate_write(JSON_GENERATOR_STRING_QUOTE, STAY_ON_SAME_LINE, generator_data);
		if (generator_data->output_buffer) generator_data->output_buffer = json_generate_write(object_element->member_name, STAY_ON_SAME_LINE, generator_data);
		if (generator_data->output_buffer) generator_data->output_buffer = json_generate_write(JSON_GENERATOR_OBJECT_GRAMMAR, STAY_ON_SAME_LINE, generator_data);
		if (generator_data->output_buffer) {
			/* Now deal with the JSON value in the element */
			if (object_element->value_type == JSON_object) {
				if (json_generate_object(object_element->value.object, generator_data) < 0) {
					still_working = FALSE;
					continue;
				}
			}
			if (object_element->value_type == JSON_array) {
				if (json_generate_array(object_element->value.object, generator_data) < 0) {
					still_working = FALSE;
					continue;
				}
			}
			if ((object_element->value_type != JSON_object) && (object_element->value_type != JSON_array)) {
				value_buffer = json_generate_value(object_element, temp_buffer); 
				if (value_buffer) {
					generator_data->output_buffer = json_generate_write(value_buffer, STAY_ON_SAME_LINE, generator_data);
					if (value_buffer != temp_buffer) _json_free_(value_buffer);		/* free the temp resource we were provided */
				} else {
					fprintf(stderr, "JSON Grammar ERROR  ==>  %s() invalid value for Object "\
									"Member %s.\n", __func__, object_element->member_name);
					still_working = FALSE;
					continue;
				}
			}
			if (list_iteration_has_more(object_data)) {
				if (generator_data->output_buffer) generator_data->output_buffer = json_generate_write(JSON_GENERATOR_MULTI_VALUE_GRAMMAR, ADD_NEW_LINE, generator_data);
			} else {
				--generator_data->indent_level;
				if (list_size(object_data) > 1) {
					generator_data->output_buffer = json_generate_write(JSON_GENERATOR_NULL_STRING, ADD_NEW_LINE, generator_data);
				}
			}
			
			
		}
		
		if (! generator_data->output_buffer) {
			fprintf(stderr, "JSON Generator ERROR  ==>  %s() output buffer was deallocated.\n", __func__);
			still_working = FALSE;
			continue;
		}
		++x;
	}
	list_iteration_stop(object_data);

	if (still_working) {
		/* Now finish the JSON Object structure */
		if (list_size(object_data) > 1) {
			generator_data->output_buffer = json_generate_write(JSON_GENERATOR_OBJECT_END, STAY_ON_SAME_LINE, generator_data);
		} else {
			generator_data->output_buffer = json_generate_write(JSON_GENERATOR_OBJECT_END_1_MEMBER, STAY_ON_SAME_LINE, generator_data);
		}
		if (! generator_data->output_buffer) {
			fprintf(stderr, "JSON Generator ERROR  ==>  %s() internal problem(s) occurred.\n", __func__);
			return -1;
		}
	} else {
		return_code = -1;
	}

	return return_code;
}


static int json_generate_array(	list_object *array_data,
							generator_globals_t *generator_data) {

	json_element_t *array_element;

	char temp_buffer[MAX_NUMBER_TOKEN_LENGTH*2];
	char *value_buffer;
	int x = 0;
	int return_code = 0;
	boolean still_working = TRUE;

	if (list_is_empty(array_data)) {
		generator_data->output_buffer = json_generate_write(JSON_GENERATOR_EMPTY_ARRAY, STAY_ON_SAME_LINE, generator_data);
		if (generator_data->output_buffer) return 1;
		else return -1;
	}

	/* First start the JSON Array structure */
	++generator_data->indent_level;
	if (list_size(array_data) > 1) generator_data->output_buffer = json_generate_write(JSON_GENERATOR_ARRAY_BEGIN, ADD_NEW_LINE, generator_data);
	else generator_data->output_buffer = json_generate_write(JSON_GENERATOR_ARRAY_BEGIN_1_MEMBER, STAY_ON_SAME_LINE, generator_data);

	if (! generator_data->output_buffer) {
		fprintf(stderr, "JSON Generator ERROR  ==>  %s() output buffer was deallocated.\n", __func__);
		return -1;
	}

	list_iteration_start(array_data, TAIL_OF_LIST);
	while (list_iteration_has_more(array_data) && (generator_data->output_buffer) && (still_working)) {
		array_element = list_iteration_get_next(array_data, NULL);
		if (array_element->member_name) {
			fprintf(stderr, "JSON Grammar Warning  ==>  %s() Value #%d of this JSON Array has a "\
							"non-NULL Member name pointer.\n", __func__, x);
			continue;
		}
		if (generator_data->output_buffer) {
			/* Now deal with the JSON value in the element */
			if (array_element->value_type == JSON_object) {
				if (json_generate_object(array_element->value.object, generator_data) < 0) {
					still_working = FALSE;
					continue;
				}
			}
			if (array_element->value_type == JSON_array) {
				if (json_generate_array(array_element->value.object, generator_data) < 0)  {
					still_working = FALSE;
					continue;
				}
			}
			if ((array_element->value_type != JSON_object) && (array_element->value_type != JSON_array)) {
				value_buffer = json_generate_value(array_element, temp_buffer); 
				if (value_buffer) {
					generator_data->output_buffer = json_generate_write(value_buffer, STAY_ON_SAME_LINE, generator_data);
					if (value_buffer != temp_buffer) _json_free_(value_buffer);		/* free the temp resource we were provided */
				} else {
					fprintf(stderr, "JSON Grammar ERROR  ==>  %s() invalid value inside an Array.\n", __func__);
					still_working = FALSE;
					continue;
				}
			}
			if (list_iteration_has_more(array_data)) {
				if (generator_data->output_buffer) generator_data->output_buffer = json_generate_write(JSON_GENERATOR_MULTI_VALUE_GRAMMAR, ADD_NEW_LINE, generator_data);
			} else {
				--generator_data->indent_level;
				if (list_size(array_data) > 1) {
					generator_data->output_buffer = json_generate_write(JSON_GENERATOR_NULL_STRING, ADD_NEW_LINE, generator_data);
				}
			}
		}
		
		if (! generator_data->output_buffer) {
			fprintf(stderr, "JSON Generator ERROR  ==>  %s() output buffer was deallocated.\n", __func__);
			still_working = FALSE;
			continue;
		}
		++x;
	}
	list_iteration_stop(array_data);

	if (still_working) {
		/* Now finish the JSON Object structure */
		if (list_size(array_data) > 1) {
			generator_data->output_buffer = json_generate_write(JSON_GENERATOR_ARRAY_END, STAY_ON_SAME_LINE, generator_data);
		} else {
			generator_data->output_buffer = json_generate_write(JSON_GENERATOR_ARRAY_END_1_MEMBER, STAY_ON_SAME_LINE, generator_data);
		}
		if (! generator_data->output_buffer) {
			fprintf(stderr, "JSON Generator ERROR  ==>  %s() internal problem(s) occurred.\n", __func__);
			return -1;
		}
	} else {
		return_code = -1;
	}

	return return_code;
}

/* Using a JSON data element as input, generate a textural value to
	output in a JSON data block to represent this value item.
	
	Upon return a pointer to a character array is returned. If this pointer
	is the same as the argument temp_buffer, no further action is needed.
	However if the address of the pointer is different from temp_buffer,
	then the caller must free() this memory block after they are finished 
	with it. It becomes the callers responsibility to properly dispose of it.
	
	element		json_element_t element of JSON data list to create a textural
				value for.
	temp_buffer	a character array provided by the caller to use as a scratchpad
				in creating the textural value. Must be a minimum size of at
				least MAX_NUMBER_TOKEN_LENGTH bytes.

	returns		pointer to textural item to add to JSON data block, or
				NULL, a value type or value content error occurred.						*/
	
static char *json_generate_value(json_element_t *element, char *temp_buffer) {

	char *string_buffer;
	char *return_ptr = NULL;

	char *string_cursor;
	char *cursor;



	*temp_buffer = '\0';								/* clean the scratch buffer */

	switch(element->value_type) {
		/* JSON_object and JSON_array types were already covered prior to call this function */
		case JSON_null:
			strcpy(temp_buffer, PARSER_TOKEN_LABEL_NULL);
			return_ptr = temp_buffer;
			break;

		case JSON_boolean:
			if (element->value.boolean_value) {
				strcpy(temp_buffer, PARSER_TOKEN_LABEL_TRUE);
			} else {
				strcpy(temp_buffer, PARSER_TOKEN_LABEL_FALSE);
			}
			return_ptr = temp_buffer;
			break;

		case JSON_integer:
			sprintf(temp_buffer, "%" PRId64, element->value.integer);
			return_ptr = temp_buffer;
			break;

		case JSON_double:
			sprintf(temp_buffer, "%Lf", element->value.double_number);
			return_ptr = temp_buffer;
			break;

		case JSON_float_str:
			string_buffer = _json_create_string_copy(element->value.string);
			return_ptr = string_buffer;
			break;

		case JSON_string:
			/* We will need a very large buffer in case the string is full of control
				characters etc. We pass UTF-8 as-is, but control chars must be escaped.
				a tab is 1 input byte but 2 output bytes in escaped form, or 6 in \u0011
				we could scan the string first, but why waste the time, heap is cheap and
				we will quickly return it back to the heap.									*/
			if (! (string_buffer = (char *)_json_malloc(strlen(element->value.string) * 6, __func__))) {
				return_ptr = NULL;
				break;		
			}

			/* Set up the string conversion */
			cursor = element->value.string;
			string_cursor = string_buffer;
			*string_cursor++ = '"';

			/* convert the string, byte by byte. UTF-8 and ASCII will both be properly 
				converted, the package spec limits guaranteed support to those encodings.	*/
			while (*cursor) {
				switch (*cursor) {
				
					case 0x5C:											/* a '\' */
						*string_cursor++ = '\\';
						*string_cursor++ = '\\';
						break;
						
					case 0x22:											/* a '"' */
						*string_cursor++ = '\\';
						*string_cursor++ = '"';
						break;
					case 0x08:											/* a '\b' */
						*string_cursor++ = '\\';
						*string_cursor++ = 'b';
						break;
					case 0x0C:											/* a '\f' */
						*string_cursor++ = '\\';
						*string_cursor++ = 'f';
						break;
					case 0x0A:											/* a '\n' */
						*string_cursor++ = '\\';
						*string_cursor++ = 'n';
						break;
					case 0x0D:											/* a '\r' */
						*string_cursor++ = '\\';
						*string_cursor++ = 'r';
						break;
					case 0x09:											/* a '\t' */
						*string_cursor++ = '\\';
						*string_cursor++ = 't';
						break;
					default:
						if ((unsigned char)*cursor < 0x20) {
							/* simple and fast */
							*string_cursor++ = '\\';
							*string_cursor++ = 'u';
							*string_cursor++ = '0';
							*string_cursor++ = '0';
							*string_cursor++ = byte_hex_values[(unsigned char)*cursor].hex_byte[0];
							*string_cursor++ = byte_hex_values[(unsigned char)*cursor].hex_byte[1];				
						} else {
							*string_cursor++ = *cursor;
						}
						break;
				}
				++cursor;
			}

			/* Finish the string off */
			*string_cursor++ = '"';
			*string_cursor++ = '\0';

			/* Shrink wrap the resultant string value for the caller, 
				return the excess ram back to the heap. If it fails, the
				program is going to fail anyway, we will return a NULL token.			*/
			string_buffer = (char *)realloc(string_buffer, (string_cursor - string_buffer));
			
			return_ptr = string_buffer;
			break;

		default:
			return_ptr = NULL;
			break;
		}

	return return_ptr;
}
/* Add contents of output_to_write to the JSON data block being created.

	This function maintains the output buffer, growing it as needed. It also
	takes care of adding in the whitespace between output segments that are
	presented to it, and also the indentation whitespace for each new line.
	Indentations are decided by the Object and Array processors, this function
	just adds them into the output as needed.
	
	This function always makes sure there are 2 spare bytes at the end of the
	output buffer.
	
	Due to the nature of having to reallocate memory when the output grows too
	large, it is required that callers must check the return pointer, every
	time. Sadly this creates tedious repetitive code, but there is no other
	safe option.
	
	output_to_write	
				nul terminated character array of text to add to the output
				buffer. Upon exit from this function, the output buffer is 
				NOT nul terminated. It is nul terminated when the JSON
				Generator has completed its output, until then it is an
				unusable item.
	new_line	when TRUE, this function will add a new line after the output
				is added to the buffer, and will write the current indentation
				in place prior to return.
				The output buffer is always left in a state where it is ready
				to receive the next chunk of data without any additional 
				processing.
				
	returns		pointer to the current output buffer. Do not assume it is the
				same as before, if the output buffer runs out of space, it
				reallocates the space, which can change the memory address it
				resides at.
				NULL there was a major error with the reallocation. You will
				need to EXIT rapidly.													*/
static char *json_generate_write(	char *output_to_write, 
							const boolean new_line,
							generator_globals_t *generator_data) {

	char *output_ptr;
	unsigned int i = 0;
	unsigned int j = 0;
	unsigned int z, size_of_output, size_of_new_chunk;

	if (! output_to_write) {
		/* This catches all instances of when the function json_generate_value() returned
			a NULL pointer in response to an error and the caller didn't check 				*/
		fprintf(stderr, "CRITICAL ERROR  ==>  %s() called with NULL text buffer, internal code failure.\n\n", __func__);
		return generator_data->output_buffer;
	}

	z = strlen(output_to_write);
	if (new_line) {
		/* the +1 is for the \n */
		size_of_output = z + (generator_data->indent_level * SPACES_PER_INDENT) + 1;
	} else {
		size_of_output = z;
	}
	
	if (size_of_output > JSON_GENERATOR_CHUNK_SIZE) {
		size_of_new_chunk = size_of_output + JSON_GENERATOR_CHUNK_SIZE;
	} else {
		size_of_new_chunk = JSON_GENERATOR_CHUNK_SIZE;
	}

	/* First do we have enough space to add this new output to the buffer ? */
	if ((generator_data->current_output_offset + size_of_output + 2) > generator_data->current_buffer_size) {
		/* Add another chunk of free space to keep writing the output data block */
		generator_data->output_buffer = realloc(generator_data->output_buffer, generator_data->current_buffer_size + size_of_new_chunk);
		if (! generator_data->output_buffer) {
			fprintf(stderr, "CRITICAL ERROR  ==>  %s() realloc() failed requesting %lu bytes.\n\n",
					__func__, (generator_data->current_buffer_size + size_of_new_chunk));
			generator_data->current_buffer_size = 0;
		} else {
			generator_data->current_buffer_size += size_of_new_chunk;
		}
	}
	
	if (generator_data->output_buffer) {
		output_ptr = generator_data->output_buffer + generator_data->current_output_offset;
		while (output_to_write[i] != '\0') {
			output_ptr[i] = output_to_write[i];
			++i;
		}
		if (new_line) {
			output_ptr[i++] = '\n';
			while (j++ < (generator_data->indent_level * SPACES_PER_INDENT)) output_ptr[i++] = ' ';
		}
		generator_data->current_output_offset += i;
	}

	return generator_data->output_buffer;
}


/* Create a new list_object to be used to represent either a JSON Object or 
	a JSON Array.
	
	Upon return the caller must check the result, if it is NULL no 
	list_object was created.
	
	list_type		Either JSON_object or JSON_array. Depending on what type of
					list_object you need to create. When created the correct
					default JSON D-List search and compare functions will be
					attached to the new list_object. 
	functions		NULL normal usage, create list_object with default search and
					compare functions.
					Pointer to array of functions, override the defaults with 
					these search and compare functions.
	
	returns		pointer to the list_object of the new list created.	
				or NULL if an internal error occurred.									*/
list_object *json_create_new_list_object(	const JSON_value_types list_type, 
											const functions_globals_t *functions) {

	list_object *new_list;
	
	assert(list_type == JSON_array || list_type == JSON_object);
	
	if ((list_type != JSON_array) && (list_type != JSON_object)) {
		fprintf(stderr, "ERROR  ==>  %s() Invalid List Type Provided in argument #1.\n", __func__);
		return NULL;
	};
	
	if (! (new_list = (list_object *)_json_malloc(sizeof(list_object), __func__))) return NULL;

	list_create(new_list, LIST_COPY_DATA, json_parser_size_function, LIST_SIZE_TINY);
	list_set_remove_function(new_list, json_parser_remove_function);

	if (list_type == JSON_object) {
		if (functions) {
			list_set_search_function(new_list, functions->object_search_function);
			list_set_compare_function(new_list, functions->object_compare_function);
		} else {
			list_set_search_function(new_list, functions_globals.object_search_function);
			list_set_compare_function(new_list, functions_globals.object_compare_function);
		}
	} else {
		if (functions) {
			list_set_search_function(new_list, functions->array_search_function);
			list_set_compare_function(new_list, functions->array_compare_function);
		} else {
			list_set_search_function(new_list, functions_globals.array_search_function);
			list_set_compare_function(new_list, functions_globals.array_compare_function);
		}
	}

	return new_list;
}

