/** =================================================
 **                License Statement
 ** =================================================
 * 
 ** Copyright (c) 2014,2015 David Winterburn <info@fwsentry.org>. All commercial rights reserved.
 * 
 ** This file contains Original Code and/or Modifications of Original Code, and is
 ** subject to the license terms below. If this file contains portions of other
 ** code, it is so noted at the end of this license, with copyright credit as
 ** required.
 * 
 ** The intent of the Author with regards to this software package and license, is
 ** to allow all end users (personal, corporate or other) to be able to freely use
 ** this software to protect their computers, data and assets; but nobody should
 ** be free to further profit from the Original Code or Original Work(s) of the
 ** Author, by incorporating any portion of Original Code in commercial software,
 ** packages or offerings, without explicit written authorization from the Author.
 ** The Author generally encourages Open Source works, but in this case wishes to
 ** ensure that all users may continue to use this package free of charge, as
 ** philosophically the Author is opposed to people and companies making profits
 ** from legitimate users trying to protect their valuable resources, due to the
 ** inherent security weakness in the design of the Internet and/or traditional
 ** UNIX based software, which have led to the current explosion of abuse of
 ** systems and servers, by criminals, hackers, bulk mailers etc.
 * 
 ** Permission to use, copy, modify, and distribute this software for any purpose
 ** WITHOUT any fee is hereby granted as long as such purposes do not violate the
 ** terms of this license, and provided that the above copyright notice and this
 ** permission notice and license appear unmodified in ALL copies, distributed or
 ** stored. Permission is expressly denied and withheld for any commercial uses of
 ** this Original Code, either in whole or part, or for any purpose with a direct,
 ** or any indirect, fee. You may not include Original Code of the Author in any
 ** other work or package that will be sold, rented, or leased commercially. All
 ** commercial rights reserved, and all intellectual property and patent rights
 ** remain with the author.
 * 
 ** THE ORIGINAL CODE AND ALL SOFTWARE DISTRIBUTED UNDER THIS LICENSE IS PROVIDED
 ** ON AN "AS IS" BASIS WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESS OR IMPLIED,
 ** AND THE AUTHOR DISCLAIMS ALL SUCH WARRANTIES, OR LEGAL CLAIMS, WITH REGARD TO
 ** THIS SOFTWARE INCLUDING WITHOUT LIMITATION ANY WARRANTIES, IMPLIED OR OTHER,
 ** OF MERCHANTABILITY, PERFORMANCE AND FITNESS FOR A PARTICULAR PURPOSE, QUIET
 ** ENJOYMENT OR NON-INFRINGEMENT. UNDER NO CIRCUMSTANCES SHALL THE AUTHOR BE IN
 ** ANY WAY LIABLE FOR ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES, OR
 ** ANY DAMAGES WHATSOEVER, RESULTING FROM THE USE OF THIS SOFTWARE. SUCH DAMAGES
 ** MAY INCLUDE, BUT ARE NOT LIMITED TO, LOSS OF USE, EQUIPMENT, DATA, BILLABLE
 ** TIME, PERSONAL OR PHYSICAL INJURY, AND DIRECT OR INDIRECT PROFITS, WHETHER IN
 ** AN ACTION OF CONTRACT, NEGLIGENCE OR ANY OTHER TORTIOUS ACTION, ARISING OUT
 ** OF, OR IN CONNECTION WITH, THE USE OF, THE FITNESS OR PERFORMANCE OF THIS
 ** SOFTWARE, UNDER ANY CIRCUMSTANCES. THIS SOFTWARE WAS NOT DESIGNED TO BE USED
 ** IN ANY SITUATION WHERE HUMAN LIFE IS AT RISK, AND NO SUCH USE IS PROVIDED FOR
 ** OR PERMITTED.
 * 
 ** By using this software you agree not to violate any of the commercial use
 ** rights of the author with regard to the original code, and that any such
 ** violations of commercial use rights, either knowingly on unknowingly, will
 ** result in a) without notification from the author, the immediate and
 ** automatic termination of this license, and the subsequent requirement to
 ** remove and delete any copies of this software in your possession, b)
 ** forfeiture of all commercial use proceeds to the author and c) additional
 ** punitive damages, recovery and legal costs payable to the author. Under no
 ** circumstances whatsoever are others authorized to profit from the work(s) of
 ** the author without express written permission and consent.
 * 
 ** BY USING THIS SOFTWARE YOU AGREE THAT IT IS THE SOLE RESPONSIBILITY OF THE END
 ** USER TO DECIDE IF THIS SOFTWARE OR PACKAGE PERFORMS, OR IS FIT FOR, ANY
 ** INTENDED USAGE.
 * 
 ** This License and the rights granted hereunder will terminate; automatically
 ** without notice from the Author if You fail to comply with any term(s) of this
 ** License and fail to cure such breach within 30 days of becoming aware of such
 ** breach; or automatically without notice from the Author if You, at any time
 ** during the term of this License, commence any legal action against the Author
 ** or engage in any unauthorized commercial use activity with any of the Original
 ** Code. Upon termination of the license you are required to remove all copies of
 ** this software from all computer systems and data storage archives under your
 ** jurisdiction or control.
 * 
 ** This license is automatically extended, if in the future, the Author exercises
 ** any intellectual property, or patent, rights connected with any of the
 ** Original Code included in this package. ALL end users using this package
 ** within the terms of this license, will automatically licensed to use those
 ** exercised or covered claims free of charge as they apply to this software
 ** package.
 **/

/* 		FWSentry.
 * 
 * See http://info.fwsentry.org for more information on this package, or see
 * http://fwsentry.org/bugs/submission.html to report bugs or request new
 * features or improvements, or contact support@fwsentry.org. Bugs will be
 * addressed, new feature requests and program improvements will be considered,
 * at the sole discretion of the Author. Updates and new releases may be issued
 * in the future, if the Author considers it practical, enjoyable and desirable.
 */

/** 	Statement of Support.
 * 
 ** Bug reports must be accompanied with a reasonable description and reasonable
 ** documentation of the problem, and any code changes you may have made to the 
 ** affected module. If the problem turns out to be an issue with code not 
 ** supplied by the author, the author may, or may not, choose to fix it.
 * 
 ** If implemented, such requests or bug fixes may be added to a later release,
 ** package update, or support patch, at the sole discretion of the author, within
 ** time-frame's suitable to the author. All support that may optionally be
 ** provided by the author, including but not limited to, responding to requests
 ** for new attack signatures, bug reports, new feature requests, or general usage
 ** support; will be provided on an "as-is" basis, at the sole discretion of the
 ** author. Any and all support, or resulting code, documentation, or product
 ** changes, will be provided with the same license and legal disclaimers
 ** described in the original package (and above) and with absolutely no
 ** warranties, implied, stated or otherwise.
 **/ 

/*	JSON Parser version 1.0 -- Copyright (c) 2014-2022 David Winterburn <info@fwsentry.org>	*/


#ifndef JSON_PARSER_H
#define JSON_PARSER_H

#ifdef __cplusplus
#include <string.h>
extern "C" {
#endif


/* inttypes.h support is broken MS Visual Studio compilers */
#if !defined(_WIN32) || !defined(_MSC_VER)
#include <inttypes.h>
#else
#include <basetsd.h>
typedef		UINT8		uint8_t;
typedef		INT8		int8_t;
typedef		UINT16		uint16_t;
typedef		INT16		int16_t;
typedef		ULONG32		uint32_t;
typedef		LONG32		int32_t;
typedef		UINT64		uint64_t;
typedef		INT64		int64_t;
#endif

/* Support for old compilers */
#ifdef __STDC_VERSION__
#	if __STDC_VERSION__ >= 199901L
	/* "inline" and "restrict" are already builtin keywords */
#else
	/* Stop compiler errors */
#define inline
#define restrict
#ifndef __func__
#define		__func__	"unknown_function"
#endif
#endif
#endif		/* (__STDC_VERSION__) */



#if __STDC_VERSION__ >- 199901L
#include <stdbool.h>
#endif		/* __STDC_VERSION__ >- 199901L */

#ifdef __bool_true_false_are_defined
#define TRUE										true
#define FALSE										false
#define boolean										_Bool

#else

/* for environments with no stdbool.h */
typedef	int	bool
#define	false										(bool)0
#define	true										(bool)1

#ifndef TRUE
#define TRUE										(bool)1
#endif		/* !TRUE */
#ifndef FALSE
#define FALSE										(bool)0
#endif		/* !FALSE */

#endif		/* __bool_true_false_are_defined */


/*	JSON data representation -
	========================

	Simply put, 

	a JSON Object is mapped to a subordinate D-List list_object
	a JSON Object Member name/value pair is an element within that list_object 
	a JSON Value is an element inside the parent list_object, which could be a
		JSON Object or JSON Array
	a JSON Array is a subordinate D-List list_object which has an element for
		each JSON Value within the JSON Array, inside the parent element which
		is enclosed in a JSON object or JSON Array
	
	JSON value types -
		JSON_string is a pointer to the original string in the original JSON buffer,
			it is now nul terminated and compacted to remve escape sequences etc. 
			The pointer points to the first char after the original double quote 
			character, and the last character is the character before the ending
			double quote.
		JSON_array is a pointer to a new list_object which will contain the 
			values, one element per array value. If the array in the JSON code
			was a single entry then that single entry is a single element in
			the new subordinate Array list_object.
		JSON_object is a pointer to a new list_object which will contain the 
			Object Member name:value pairs, one list_object element per JSON Object 
			Member. If this JSON Object in the JSON data block only enclosed a single
			entry then that single entry is a single element in the new subordinate 
			list_object. There are no shortcuts, the hierarchy is preserved even for
			single elements.
			If the JSON Object was empty " { } " then the subordinate list will
			have zero elements, but there will still be a list_object describing it.
		All other JSON value types are converted and stored directly in the list
			elements, be they JSON Object Members (name/value), or a value only
			element as would be included in an Array, this includes all value types
			numbers, boolean, null, etc.
		
	The D-List list_objects that now represent the JSON object, array and value 
	structures, can be used just like any other D-List list_object. They can be sorted
	and manipulated as needed. All data pointers within the lists point to portions
	of the original JSON string buffer that was parsed.
	
	DO NOT release the JSON buffer, until after the D-List lists that describe it have
	been erased. All D-List lists used for parsing JSON objects are built with the 
	COPY_DATA attribute, so all list storage is handled internally by the JSON parsing
	code, and the deletions use D-List functions to properly free all the allocated
	regions.
	
	This representation of a JSON object eliminates all white space, and compresses
	strings to remove escape characters, etc.
	
	jCard representations are handled in the same way, a vCardArray would be an
	array object, a new list_object, and each of the elements in that array would
	be elements in the new list. elements with nul parameters, would have a NULL
	entry in the parameters pointer, or there would be pointer to another list.

	Important. For every element in every list_object, root or subordinate, that is
	returned by the JSON Parser, you must always check the JSON_value_type in each
	element BEFORE attempting to access the element value. Inaccurate handling of 
	element values in your code will lead to code or data corruption. Please refer
	to the typedef json_element_t in json-parser.h for more information. Fields that
	are not used in an element will always be 0x00000000.								*/

/*	Example of Parser Usage
	=======================

	Using this simple JSON data block, for example:
		{	"country" : "JP", 
			"language" : "en", 
			"names" : [ "Hayao", "Totoro", "Ponyo", "Nausica", "Kiki" ], 
			"films" : { "Ponyo" : 2008,
						"Princess Mononoke" : 1997,
						"Tonari no Totoro" : 1988,
						"How Do You Live?" : "In Production"},
			"Ghibli" : [	{ "film" : "Spirited Away", "release" : 2001, "time" : 125},
							{ "film" : "Howl's Moving Castle", "release" : 2004, "time" : 119},
							{ "film" : "Grave of the Fireflies", "release" : 1988, "time" : 89},
							{ } ],
			"Studio" : "Kabushiki gaisha Sutajio Jiburi"
			"Formal" : "株式会社スタジオジブリ" 
		}

	would be parsed into a set of D-List list_objects such as:
	
									value_type	member_name	: value
									----------	-----------	  -----
	(p)-->	root_list->	element_1->	JSON_string	"country"	: "JP"

						element_2->	JSON_string	"language"	: "en"

						element_3->	JSON_array	"names"		: value.object --> array_list_a	 -->(a)

						element_4->	JSON_object	"films"		: value.object --> object_list_2 -->(b)

						element_5->	JSON_array	"Ghibli"	: value.object --> array_list_b	 -->(c)

						element_6->	JSON_string	"Studio"	: "Kabushiki gaisha Sutajio Jiburi"

						element_6->	JSON_string	"Formal"	: "株式会社スタジオジブリ"
					(end of root_list)

			 (a)-->	array_list_a->	element_1->	JSON_string	NULL : "Hayao"
									element_2->	JSON_string	NULL : "Totoro"
									element_3->	JSON_string	NULL : "Ponyo"
									element_4->	JSON_string	NULL : "Nausica"
									element_5->	JSON_string	NULL : "Kiki"
								(end of array_list_a)

			 (b)-->	object_list_2->	element_1->	JSON_integer	"Ponyo"				: 2008
									element_2->	JSON_integer	"Princess Mononoke"	: 1997
									element_3->	JSON_integer	"Tonari no Totoro"	: 1988
									element_4->	JSON_string		"How Do You Live?"	: "In Production"
								(end of object_list_2)

			 (c)-->	array_list_b->	element_1->	JSON_object	NULL : value.object --> object_list_3 -->(d)
									element_2->	JSON_object	NULL : value.object --> object_list_4 -->(e)
									element_3->	JSON_object	NULL : value.object --> object_list_5 -->(f)
									element_4->	JSON_object	NULL : value.object --> object_list_6 -->(g)
								(end of array_list_b)

			 (d)-->	object_list_3->	element_1->	JSON_string		"film"		: "Spirited Away"
									element_2->	JSON_integer	"release"	: 2001
									element_3->	JSON_integer	"time"		: 125
								(end of object_list_3)

			 (e)-->	object_list_4->	element_1->	JSON_string		"film"		: "Howl's Moving Castle"
									element_2->	JSON_integer	"release"	: 2004
									element_3->	JSON_integer	"time"		: 119
								(end of object_list_4)

			 (f)-->	object_list_5->	element_1->	JSON_string		"film"		: "Grave of the Fireflies"
									element_2->	JSON_integer	"release"	: 1988
									element_3->	JSON_integer	"time"		: 89
								(end of object_list_5)

			 (g)-->	object_list_6->	<no elements>
								(end of object_list_6)

	For each element encountered in any list_object, the value_type of that
	element will tell you what type of element it is, and from there you know
	that it is either a value, in which case the value is there in the element,
	or it is a JSON Array or JSON Object and the pointer to the relevant
	subordinate list_object is there in the element. 

	So in the above example calling json_parse_data_block() with the above
	JSON data block would return a single pointer (p) to the root_list, which
	would have 6 elements in it, 3 of them with subordinate list pointers 
	in them, and a the json_data_type of JSON_object, so you knew the root
	list represented a JSON Object at the top level. In this case the 
	subordinate lists would have 5, 4 and 4 elements in them respectively. 
	
	These lists can then be retrieved, searched, sorted, and managed as you
	would any other D-List using the various D-List functions. It is easy to
	find out if a JSON array or JSON Object was empty by calling the function
	list_size() or list_is_empty() on that subordinate list_object. Or by
	using list iteration so any processing code is only used when there are
	elements to retrieve. Please refer to D-List documentation for more 
	information on how to use D-List list_objects and functions. However even
	though D-List provides a broad range of options to access and manage data
	elements, it is at heart very easy to use the core set of D-List functions.
	
	So with the above JSON sample data as input, this code snippet can find
	a particular film year, using the JSON Parser default search functions

		list_object *root_list;
		json_member_list_t results;
		json_element_t *element_ptr, *film_element_ptr;
		unsigned int json_data_type;
		char *the_json_data;

		the_json_data = read_the_json_data(filename);
		
		root_list  = json_parse_data_block(the_json_data, &json_data_type, NULL, NULL, NULL, NULL, FALSE);
		if (json_data_type == JSON_object) element_ptr = list_simple_search(root_list, "films");
		if ((element_ptr) && (element_ptr->value_type == JSON_object)) {
			film_element_ptr = list_simple_search(element_ptr->value.object, "Ponyo");
			if ((film_element_ptr) && (film_element_ptr->value_type == JSON_string)){
				fprintf(stdout, "The Movie %s was released in %Ld.\n", 
						film_element_ptr->member_name, 
						film_element_ptr->value.integer);
			} else {
				fprintf(stdout, "Movie Ponyo not found in list\n");
			}
		} else {
			do something if the JSON object element name "films" is missing.
		}

	Another option to retrieve the specific film element is to use the JSON
	Support Functions provided in the Package. So for example:

		list_object *root_list;
		json_member_list_t results;
		unsigned int json_data_type;
		int ret;

		root_list  = json_parse_data_block(the_json_data, &json_data_type, NULL, NULL, NULL, NULL, FALSE);
		if (json_data_type == JSON_object) {
			ret = json_find_member_value(&results, root_list, "films", FALSE, "Ponyo", FALSE);
			if ((ret > 0) && (results.value_type == JSON_integer)){
				fprintf(stdout, "The Movie %s was released in %Ld.\n", 
						results.json_member->member_name, 
						results.json_member->value.integer);
			}
		} else {
			fprintf(stdout, "Movie Ponyo not found in list\n");
		}

	If you know the exact layout of a given JSON Data block it will be more 
	efficient for you to write code to navigate it directly, as in the first
	example above. However if you are unaware of its layout, it is easier to
	use the built JSON Support Functions which can take the hassle out of 
	finding arbitrary data in unknown or arbitrary locations within that 
	JSON data. 

	This example uses the function json_find_member_value() which will return
	a single instance of the data being searched for. However the function
	does return a count of the number of searches matches it actually found.
	As an alternative method you could call the broader function
	json_find_members_value_string() which will return a list of all the
	matches found based on the same type of (Member Name : String Value)
	evaluation of the entire JSON data block.

	The utility json-parser-test which is part of this JSON Parser package
	has a function to output a parsed set of list_objects of a JSON data 
	block to stdout, this is a good way to see how the JSON data block is 
	formatted into D-List list_objects and elements. 								*/
/* No, i am not an anime fan, just trying to make the dry documentation fun. */


/*	STANDARDS
	=========

	This implementation of this JSON Parser package is completely compliant with 
	RFC 8259 and ECMA-404. It does not impose any restrictions on what may or may 
	not be deemed fully interoperable between JSON peers sharing data, it does 
	conform to the specifications of RFC 8259, but does not impose restrictions
	based its guidelines of interoperability. 

	Based on RFC 8259's written document structure, the following notes should 
	help clarify any details.

	2. JSON Grammar 

	JSON grammar is strictly enforced by the validator and the parser. They
	enforce the same requirements as those imposed by RFC 8259 Section 10, and
	do not allow variations beyond those standards as MAY be permitted but not
	required or standard, by Section 9.

	3. Values

	All JSON values are kept intact, and in the order they were present in the
	original JSON data block. All literal values are converted to a JSON_value_type
	present in each D-List element that represents any data value or Object Member 
	name:value pair.

	All numbers are converted, integers to a 64 bit minimum size, and fractional
	numbers to a long double. Exponent numbers may be converted to long double or
	optionally may be left intact as text (string) representations for the caller
	to convert themselves later. This is an option available when calling the JSON
	Parser function to allow callers with high precision number requirements to 
	handle the conversions themselves to what ever standard of precision they wish.

	4. Objects

	Objects are represented as a D-List list_object, each member of the JSON Object
	being an element in that list. All JSON object members are converted in the 
	order they were presented in the original JSON data block. RFC 8259 states that
	member names should be unique. This implementation parses all object members,
	therefore it allows non-unique names in an object. If you use the D-List function
	list_simple_search() on such a list it will always return the last instance of
	a name:value member in that list. All the other list_search_xx() type functions
	in D-List can be used to find any or all of such duplicate members if needed.

	The JSON Support Functions provide a much richer set of tools to access JSON 
	Objects and data, these were built on top of the D-List functions to simplify
	JSON data usage for users. These functions can find all JSON Object Members with
	equal names, etc.

	However for proper interoperability the author strongly suggests that users of
	JSON stick to the recommendations of the RFC, and avoid duplicate member names,
	and use array values on a single instance of the member name instead. One area
	where non-unique member names is appropriate is where a JSON user wants to 
	incorporate something in the JSON grammer, that is not important to the
	consumer of that object, for example comments. Multiple object members with the
	name "comment" would be a sensible way of documenting a JSON data block without
	either violating the standards of JSON grammar, or confusing a parser or consumer 
	of that data.

	When using any element in an object list, or iterating through that list, do not
	make any assumptions about what type of value the element value is. ALWAYS check
	the element JSON_value_type first before accessing the value item.

	JSON Object Member names can be any string (by definition of the standards), so
	that also means they can be a zero length string  ( "" ) which may not be very
	useful to data blocks, but is part of the standard. A JSON Object Member D-List
	element will always have a pointer to the Member name, which will always be a
	valid string, even zero length.

	5. Arrays

	Arrays are represented by a subordinate D-List list_object. The elements of that
	list_object can be of any legitimate JSON value type. As per the RFC there is no
	requirement for the elements to be of the same type.

	Just as with JSON Objects, when using an array list, or iterating through that 
	list, do not make any assumptions about what type of value the element value is.
	ALWAYS check the element JSON_value_type first before accessing the value item.

	JSON Array values have no names, (by definition of the standards), so that also
	means a JSON array value D-List element will always have a NULL object name 
	pointer and therefore no Member name. The JSON Object Member name, if there
	was one, will be in the superior list element that originally pointed to this
	array list_object.

	If you code has an element from a JSON data block, but for some reason does not
	know where it came from (an Object or an Array), checking to see if the pointer
	to member_name is NULL or not will answer the question.

	6. Numbers, were covered above in 3. Values.

	7. Strings

	All strings in the JSON Validator, Parser and Generator eco-system are stored
	and transfered as nul terminated characters arrays. They are not kept with a
	specific size attached. This follows C language style semantics and formal
	UTF-8 encoding practices. It also means that it is not permitted in this JSON
	eco-system to embed nul (0x00) bytes in text values, as they will cause the 
	premature termination of such string values. The parser and validator will
	ignore nul bytes in the input values. The generator will terminate the 
	processing of a textural value when it encounters a nul byte.

	The JSON Parser preserves all string values as is. All string values are in 
	fact left in-situ in the original JSON data block that was presented for 
	parsing. This data block is extensively modified by the parser. It leaves 
	strings in place, but terminates them with a nul at the end of the string. 
	Additionally any escape sequences present in the string are converted to the 
	actual character in-situ, and the escape character is removed. So for 
	example "The test string\n\tNew line." would be converted to a character array
	of 27 characters, the \n would be changed to a 0x12 and the \t to 0x11, all 
	the following characters would be shifted to the left to fill the empty space
	left by the escape characters, and the string would be terminated at the end
	by 0x00. Therefore the char * pointer to that character array would act 
	correctly, and be treated as a string by all C library functions, and still 
	be in-situ in the original JSON data block. The design idea of this JSON 
	Parser was to reduce the heap overhead required to represent everything in 
	that JSON data block, and greatly simplify the removal of all the objects
	(text strings etc) after the user has finished with the JSON data.

	JSON escapes \u are similarly encoded in-situ removing the original hex escape 
	sequence. However it should be noted that if any byte in a \uxxxx sequence is
	nul (0x00), that byte will be ignored and not written into the output character
	string. So for example if the serializer of the JSON had escaped a tab control
	character 0x11 as \u0011 instead of \t (which is permitted by the standards) 
	the byte 0x00 will be ignored, and the byte 0x11 will be added to the string.
	If the original JSON serializer encoded \u0000 both bytes will be ignored by 
	this parser. Although it is valid in JSON grammar to pass such a value, in 
	any C type language, those bytes would terminate a character array and render
	the rest of the string useless. No syntax errors are thrown for nul bytes
	encoded this way, they are just silently ignored to preserve C language 
	semantics.

	If the caller of the JSON Parser wishes to preserve (not convert) all exponent
	numbers, they will be left in-situ as strings as well. However instead of being
	described as JSON_string types, they will be JSON_float_str so the caller knows
	the difference when they find the elements.

	8.1 Character Encoding

	This JSON Parser makes no attempt to enforce or even recognize the encoding used.
	It parses the characters in-situ and allows the caller to decide what data is or
	is not there. It neither removes byte order marks (0xFEFF) nor tags them as errors.
	It passes all text as is. Similarly unpaired UNICODE characters are passed as is
	by this parser. The consumer of these data items is responsible for safe handling 
	and usage of them. There is an underlying assumption that the data is in UTF-8
	form, but there is no such enforcement, and nothing changes or is damaged, if the
	data is not in UTF-8 format.

	8.3 String Comparison

	This JSON Parser provides a set of default search functions and compare 
	functions which act on strings as-is. If the caller wishes to transform 
	textural values into UNICODE first before such comparisons or compares, they 
	can provide their own search and compare functions to add to the lists by 
	default. These search or compare functions are very simple and easy to write.

	9. Parsers

	This JSON Parser accepts all forms of text that conform to the JSON grammar. This
	parser sets no limits on the size of texts it accepts. If it was able to fit in the
	original JSON data block, it is accepted and used in-situ.

	This JSON Validator does set a limit of 2048 nested levels of JSON Objects and
	JSON Arrays. 

	This JSON Parser sets no limits at all, except those that may be enforced by
	the host operating system, such as available heap space. The parser is 
	totally recursive and re-entrant, it will parse any valid JSON data block,
	mo matter how deeply it is nested.

	This JSON Validator and Parser set no limits on string length or contents.

	This JSON Generator imposes no limits, and acts as a mirror of the JSON
	Parser using the same recursive model.

	10. Generators

	This package includes a JSON Generator. It translates a valid D-List 
	list_object tree representation of JSON data into a textural JSON data 
	block, according to the standards of RFC 8259. The caller of the JSON 
	Generator function gets a nul terminated character array back, which they
	can write to a file, or send over a socket, or whatever they want to do 
	with it. The output of this JSON Generator is strictly compliant with the
	standards set forth by RFC 8259 and ECMA-404.

	Generator Notes: 
		Strings
	
		The JSON Generator assumes all input strings, either member names or
		values, are ASCII or UTF-8. There is no checking beyond nul bytes which
		have no place inside a UTF-8 string. No enforcement of FF, FE, C0 or C1.
		Both the parser and generator assume the callers will properly deal with
		byte encodings. It is the responsibility of the caller to enforce whatever
		encoding standards they wish for the text strings. All control characters
		(0x01 - 0x1F) are either directly escaped or rendered in \uxxxx form. nul
		0x00 is ignored as it will signify the end of the character array being 
		generated and any characters after the nul byte will be implicitly ignored.
		
		Style
		
		The human readable textural data output by the JSON Generator is in a 
		particular style, which is in-line with general conventions of both C
		and JSON. The author investigated a considerable number of JSON files 
		from many different sources, and settled on a format style that seems
		to fit within the general ways people write or generate JSON text data.
		
		Content
		
		If you use the JSON Parser to create D-List data from a JSON data file,
		and then use the JSON Generator to create another textural 
		representation of that original data, it will look very similar, with
		some style differences. The only substantial difference will be that
		any hex escape sequences will not be reproduced, but output instead as
		UTF-8 bytes, with the exception of any characters that are required by 
		the standards to be escaped.
		
		Exponent Numbers.
		
		The production of exponent numbers will depend on how the caller of the
		JSON Generator prepared those numbers prior to calling. If those exponent
		numbers were put in the D-List elements as textural strings, they are
		reproduced exactly. If they were put in as converted double float numbers
		the results are only going to be as good as the basic C libraries that 
		are available on your system, and the various precision and rounding 
		issues they may have. In general it is best to create these as textural
		floating point representations prior to calling the JSON Generator if
		accuracy and precision for exponent numbers is important to your needs.

	12. Security Considerations

	This parser uses the JSON data block provided by the caller. All other data
	fields and structures are provided by the D-List subsystem. If data leakage
	is considered an issue for the caller, they are responsible for managing the
	original JSON data block, and may take extra steps to scrub it after usage,
	or extra steps to allocate it in a random location if desired. The internal
	D-List data, the list_object, the elements, and the internal space management 
	data are released when the root list_object is erased by list_erase(). If
	extra security precautions are required by the caller, they may take 2 extra
	steps. First if they compile the D-List code that is to be used in this parser
	with the compile option DLIST_FLUSH_MEMORY, D-List will enforce a strict
	memory scrubbing system to purge all data before release. It makes D-List a
	little slower, but cleans everything. Another option for the caller is to
	provide their own remove_function to the root list_object and all the others.
	But be sure that your replacement also provides the same functions as the 
	in-built one does, otherwise there will be memory leaks galore.

	Extra Note
	----------

	It seems that the issue of whether or not comments					// blah blah 
	should be permitted in JSON data blocks is rather contentious among the user
	community. It does seem clear that there is no contention at all in the 
	standards community, IETF and ECMA both clearly state the JSON grammar does
	not include comments.

	This implementation of both the JSON Validator and JSON Parser sticks to the 
	standards, and will reject any JSON data block that contains comments, or 
	other non JSON grammar encoding. Although RFC 8259 section 9 states that 

		"A JSON parser MAY accept non-JSON forms or extensions."
	
	This implementation does not accept any non standard grammar forms or syntax.
	Not just for the obvious operability argument, but also because the next 
	section (10. Generators) states very clearly, that 
	
		"A JSON generator produces JSON text. The resulting text MUST 
		strictly conform to the JSON grammar."

	There would be no point in having a JSON Parser that works on a different 
	standard of grammar than that imposed on the JSON Generator of that grammar, 
	when both are in the same JSON Package.

	If at some point in the future the IETF and/or ECMA change the standard to 
	accept comments or update the available JSON grammar and syntax, this package
	will be updated accordingly.

	If you as a JSON user feel strongly about incorporating comments into the
	JSON grammar specification, i urge you to communicate with the IETF and ECMA
	about the issue. I also implore you to NOT just go and start changing software
	to accept what is not allowed by specification, just because you personally 
	don't like it. 

	Until then, just put your comments in a value field! It is not hard and 
	works within the existing standards. It also allows everyone to properly
	interoperate and exchange data, which was the whole point.
	
		{ 	"myobject" 	: "valuable data",
			"another" 	: 3.14159,					"//" : "was that pi ?",
			"comment"	: "simple enough isn't it",
			"more data"	: true,						"" : "Another style of comment",
			"label_b"	: "something important"		}
																						*/

/* The available JSON value types for elements in JSON block lists.
	
	VERY IMPORTANT: Do not alter the ordering of this value types list
					It will destroy the semantics of mixed value type
					evaluations.

	Note:	invalid_empty	is never used, it is a place holder for an 
							incomplete element
			JSON_value 		is never used in elements, but can be returned 
							by the function parse_json_data_block() to alert
							the caller that the parsed JSON block returned
							contains a value element (type described in that
							element), instead of a JSON array or JSON object.			*/
typedef unsigned int JSON_value_types;
#define invalid_empty			0
#define JSON_object				1
#define JSON_array				2
#define JSON_value				3				/* NOT a valid types for list elements */
#define JSON_null				4
#define JSON_boolean			5
#define JSON_integer			6
#define JSON_double				7
#define JSON_float_str			8
#define JSON_string				9

typedef int64_t json_integer_t;
typedef long double json_double_t;


typedef struct {
	JSON_value_types value_type;				/* the value type of this name/value pair (from JSON_value_types) */
	char *member_name;							/* Member name for JSON_object Member elements, NULL for all other value types */
	union {
		list_object *object;					/* ptr to list_object for a JSON_object, or JSON_array, or NULL */
		char *string;							/* For JSON_string or JSON_float_str if requested */
		json_integer_t integer;					/* For integer (JSON_integer) */
		json_double_t double_number;			/* For non integer (JSON_double) user must deal with the conversion issues */
		boolean boolean_value;					/* For JSON_boolean either TRUE or FALSE */
	} value;
	boolean free_string_ptr;					/* TRUE, remove function will free() the string and member_name pointer, FALSE it will not */
} json_element_t;


/* This typedef is used to describe JSON Object paths to either find data or
	express where data is in a JSON hierarchy. 
	
	So for example to express a JSON Object path of
		"bakery.product.[0].name" 
	would take an array of 4 json_path_t entries
		{JSON_object, "bakery"}, 
		{JSON_object, "product"}, 
		{JSON_array, 0}, 
		{JSON_object, "name"},
		{invalid_empty, NULL}				can be used to terminate, or a count can be provided */
typedef struct {
	JSON_value_types value_type;				/* the type of the link link in the JSON Object path, JSON_object or JSON_array */
	union {
		unsigned int array_index;
		char *member_name;						/* AKA keys in javascript land etc */
	} link;
} json_path_t;

/* the json_member_list_t struct is used by various JSON support functions
	that perform searches of hierarchical JSON data list_objects. If a list
	of members or array values is returned to a caller, this is the form of 
	each element returned. Additionally the support functions that accept
	JSON paths also use this structure to transfer data to and from the 
	callers.
	
	The pointer to the parent list_object of this element is required for
	quick access to other Members of the same JSON Object this element came
	from.

	The pointer to the element reference is required, in combination with 
	the parent list_object, to access or manage the element using D-List. 
	Without the element reference, you would have to search for the again.
	
	The pointer to the element itself is provided so you can access or update
	the JSON Member data value, or change it. However having the address of
	the element will not assist you when attempting to access or manage the
	element in D-List. D-List deliberately abstracts the address of an
	element away from managing the element. That is why you need the 
	list_object and the element reference. Please refer to the D_list 
	documentation if you need more information about abstract element
	references and using them to access and manipulate elements.
	
	The path and path segment count, replicate the JSON dotted syntax used
	to describe paths in JSON data blocks. These can also be used as input
	to the various json_path_xxx functions 										*/
typedef struct {
	list_object *json_owner_object;			/* parent list_object of this JSON Member element, use to locate element in D-List */
	element_reference *reference;			/* D-List element reference of this JSON Member element, use to manipulate element in D-List */
	json_element_t *json_member;			/* Pointer to the actual JSON Member Element, use to access the Member data */
	JSON_value_types value_type;			/* the Value type of the JSON Member value */
	unsigned int path_segments;				/* the number of elements in the path array (e.g. 4 in the example below)  */
	json_path_t path[];						/* the JSON path of the Member as recorded when found (e.g. bakery.product.[19]."curry pan") */
} json_member_list_t;


typedef struct {
	element_search object_search_function;
	element_compare object_compare_function;
	element_search array_search_function;
	element_compare array_compare_function;
} functions_globals_t;


/* the environment variables we can use */
#define ENVIRONMENT_VAR_DEBUGGING			"JSON_PARSER_DEBUG"

/* global debugging flags */

#if defined( INITIALIZE_DEBUGGING_FLAG )
	boolean json_parser_debugging = FALSE;
#else
	extern boolean json_parser_debugging;
#endif


/*	===================================================================

	JSON Conversion Functions for JSON Parser D-List hierarchical lists.

	===================================================================*/


/* Call the JSON Parser and return a hierarchical list_object containing the
	parsed contents of the JSON data block requested. The original JSON data 
	block will be modified.
	
	The JSON data block can be parsed in 1 or 2 passes. If the JSON data 
	originates from an unknown or unreliable source, it would be wise to pass 
	it through the validator json_valid_data_block() first. Once identified as 
	valid it can be parsed with this function. Alternatively, if the JSON data 
	being parsed comes from a known good and reliable source (for example ARIN)
	then sending it to the parser in one pass is probably the right choice.
	
	This function parses the contents of the JSON data block while performing 
	syntax and data validation, building a set of D-List list_objects and 
	elements that contains all the JSON data with all the whitespace, syntax 
	and non-data removed. This function, like all those in the JSON Parser
	Package, is fully thread safe and MP safe.
	
	The original JSON data block provided by the caller is heavily modified as 
	part of the parsing process. If you want to retain the original JSON data 
	block intact, you must provide a copy of it to this function.
	
	The goal of the function is to parse the JSON data block creating as little 
	as possible in additional resources, and providing the data contents in a 
	form of D-List list_objects that is natively variable and non-determinate. 
	Rather than attempting to crow-bar variable data objects into some form of 
	static C language array. This makes the parsing code much simpler to write
	and maintain, while the results of the JSON data parsing activity are much 
	easier to handle, find, or modify by the caller.
	
	For example it is now very easy to rearrange the original JSON data 
	elements, or sort them by arbitrary keys etc, or perform arbitrary searches
	for data or keys.
	
	Upon successful return, a pointer to a list_object is returned. It is very 
	important to realize that the data values (such as strings) pointed to from
	within the elements of these hierarchical list_objects, are still inside the
	original JSON data block that was provided to this function. That original 
	JSON data block was extensively modified to allow the strings it contains to
	be nul terminated etc. and left in-situ. It is very important that the root
	list_object returned to the caller is properly disposed first, BEFORE any 
	attempt is made to release the original JSON data block. 
	
	So for example:
			big_lump = (char *)malloc(big_lump_to_fit_json_data_block);
			{... read file or grab data from socket to fill big_lump ...}
			if (json_valid_data_block(big_lump, NULL, NULL, FALSE) {
				json_object_ptr = json_parse_data_block(big_lump, &return_type, NULL, NULL, FALSE);
				if (json_object_ptr) {
					while (still_working) {
						do something useful with the returned list_objects
							...
					}
					list_erase(json_object_ptr);
					if (json_object_ptr) free(json_object_ptr);
				}
			}
			if (big_lump) free(big_lump);
	
	The root list_object returned is self cleaning, all data is managed 
	internally, and the single list_erase of the root object, will cause all
	the element records, arrays, and sub-level list_objects to be deallocated 
	and removed. No special JSON Parser calls are required to manage the root
	list_object, all data processing can be performed with the standard D-List
	function calls. If you wish to perform sorts, searches, etc. the default
	JSON search and compare functions have already been attached to both the 
	root, and all the subordinate list_objects. The attached search and compare 
	functions have already been tailored to either JSON Object or JSON Array 
	type lists.
	
	Additionally upon return, the output argument json_data_type is set to tell
	the caller what the top level list_object retuned describes. In JSON the
	root of a data block can only have one item. It can be a single value, or 
	an array or an object. Rather than have a list_object that only has a 
	single element, that itself is a list_object pointing to an array, etc; the 
	root level list_object alway describes the top level JSON data block, so if
	that is a JSON Array, then the elements of the root list_object will be the
	elements of that array. If it is a single JSON value there will be a single
	element describing that value.
	
	You may optionally select a replacement search or compare function for the 
	list_objects to be built during the parsing operation. If none are provided, 
	then in-built default functions are used. You can select from any of the
	other available built-in function, described below, or you may write your
	own to suit your specific data processing needs. If you chose to replace
	the default search and compare functions after the lists are created, the 
	process to add them to every list_object in the data hierarchy would be
	very tedious. You can always override a specific search or compare function 
	for a specific list_object as needed, during later processing of the parsing
	results.
	
	Please do not use list_set_remove_function() to remove or replace the 
	original remove_function set by this json_parse_data_block() function call.
	It would lead to significant memory leaks, and potentially accessing 
	deallocated heap storage. Other than the remove function, you are free to 
	use or manipulate the returned list_objects as needed.
	
	The JSON Parser normally converts all value tokens found in the JSON data
	block. However the basic C language has limited ability to properly 
	represent high precision exponential numbers. As this code was designed 
	to be highly portable, only requiring a minimum of C99 to compile and run,
	it uses the base forms of the language for numbers. For some users either
	the limited precision of these forms, or their inability to hold huge
	numbers will be an issue. So there is an option to instruct the parser to
	instead provide these exponent numbers as a string value instead, and 
	allow the user to use whatever methods they want to convert or use those
	numbers. Integers and numbers with no exponent are always converted.
	Integers to C type long integer, and fractional numbers to C type
	long double. It is beyond the scope of this parser to accurately deal
	with very large numbers with the precision used by some professions.
	
	You can use the json_element_t typedef, to properly access the list element 
	data. The values of JSON_value_types are used to decode what type of data
	or value each list element contains.
	
	json_data_block	a nul terminated string containing the JSON data block
					to parse. The contents of this character buffer will 
					be extensively modified, and will be unusable by the
					caller upon return. However as noted above, do not 
					release or free this buffer, until after the returned
					list_object has been erased.
	json_data_type	Upon successful return, this argument is modified to
					reflect what the top level of the JSON data block
					represents, the 3 possible options are:
						JSON_object		the list_object returned is a 
										parsed JSON object
						JSON_array		the list_object returned is a 
										parsed JSON array
						JSON_value		the list_object returned contains 
										a single JSON value element, which
										is one of the type in the list 
										JSON_value_types
	object_search_function
					a D-List search function to add to each list_object that
					represents a JSON Object, that is created by the JSON 
					parsing activity. Or NULL if the default search function
					default_search_function_member_name() is to be added. 
	object_compare_function
					a D-List compare function to add to each list_object 
					represents a JSON Object, that is created by the JSON 
					parsing activity. Or NULL if the default compare function
					default_compare_function_member_names() is to be added.
	array_search_function
					a D-List search function to add to each list_object that
					represents a JSON Array, that is created by the JSON 
					parsing activity. Or NULL if the default search function
					default_search_function_array_values() is to be added. 
	array_compare_function
					a D-List compare function to add to each list_object 
					represents a JSON Array, that is created by the JSON 
					parsing activity. Or NULL if the default compare function
					default_compare_function_array_values() is to be added.
	keep_exponent	FALSE - Exponent numbers are converted to long double
					TRUE  - Exponent numbers are retained as text strings
					
	returns 		Upon successful parsing of the JSON data block, a pointer
					to a root list_object is returned, which is the top of
					the hierarchical JSON data block.
					If there was an error found in the JSON data block, a NULL
					pointer is returned.												*/
list_object *json_parse_data_block(	char *json_data_block, 
									JSON_value_types *json_data_type,
									element_search object_search_function,
									element_compare object_compare_function,
									element_search array_search_function,
									element_compare array_compare_function,
									const boolean keep_exponent);




/* Call the JSON syntax validator and return a boolean value based on either
	successful or failed validation.
	
	The JSON data block is parsed in a linear pass, validating each character 
	of the provided JSON data block, all tokens, values, whitespace and syntax 
	are checked. This function, like all those in the JSON Parser Package, is 
	fully thread safe and MP safe.
	
	Checks are made for validity of syntax, contents, and the relative
	positioning of all tokens, structural, literal, object, array and value.
	
	The contents of the character buffer will be remain unmodified. Validation 
	only scans the JSON data block and makes no changes. If JSON errors are 
	found a pointer to the invalid token can optionally be returned.
	
	This JSON Validation will stop when it encounters a JSON syntax or value 
	error, it will not perform any further validation of the data block. 
	
	If the caller does not suppress warnings, information and the location of
	the JSON error are written to stderr. If the caller suppressed warnings and
	did not know why the JSON object was rejected by the validator, call this
	function again without suppressing warnings to see the specific error
	messages involved.
	
	json_object		a nul terminated string containing the JSON object to 
					validate. 
	object_error	if a syntax or value error is found in the JSON data 
					block, a pointer the portion of the JSON data block in 
					error is returned in this argument. This argument can 
					be NULL in which case no pointer is returned upon 
					encountering a JSON error.
	line_number		if a syntax or value error is found in the JSON data 
					block, the line number of the JSON token in error is 
					returned in this argument. This argument can be NULL
					in which case no line number is returned if a JSON 
					error was detected.
	suppress_warnings
					FALSE 	DO not suppress JSON error warnings written to
							stderr
					TRUE	Suppress possible error warnings, no output is
							written.
					
	returns 	TRUE 	- Successful validation of the JSON object syntax,
							grammar and values, no language errors.
					
				FALSE	- There was an error found in the JSON data block,
							if provided as arguments, a pointer is returned 
							in syntax_error and an integer in line_number
							corresponding to the found error. If warnings
							were not suppressed an message detailing the
							JSON error was written to stderr.							*/
boolean json_valid_data_block(	const char *json_object, 
								char **object_error, 
								unsigned long int *line_number,
								const boolean suppress_warnings);

/* Using a JSON Object data tree, represented by a hierarchical D-List list_object
	structure, create a textural version of that JSON data presented and return a 
	character array with that JSON data in it, in strict JSON Grammar format.
	
	This function is essentially the reverse of json_parse_data_block(). It
	will walk through a D-List list_object structure containing the 
	representation of a JSON data block, and create a textural version of
	that JSON data ready to be stored or transmitted over a network.  This 
	function, like all those in the JSON Parser Package, is fully thread safe 
	and MP safe. 
	
	The input data must be in the correct form and conform to JSON grammar.
	Using the D-List hierarchical structure, there is very little that can be
	in error in that structure, so the generation function produces few error
	messages. The only real data problems that can occur are simple mistakes
	such as providing inaccurate JSON value types, or forgetting to provide a
	name for JSON Object members. 
	
	All textural strings that are created by the JSON Generator assume the bytes
	are either ASCII or UTF-8. It does not enforce any encoding standards. All
	string bytes are passed through without change, except those that are 
	required by the relevant JSON standards to be escaped. This includes all 
	control characters which are either escaped with a '\' or by hex code.
	
	The JSON Generator will produce human readable JSON text, that is well 
	structured and easy to read. If you take an existing JSON data block,
	parse it with the JSON Parser, and then run that output list_object 
	through the JSON Generator, you will get back almost identical text
	with the exception of probably different formatting style, and most
	\uxxxx hex escapes in the original will be reproduced as UTF-8 instead.

	json_data		A pointer to a root list_object for the JSON data structure
					the caller wishes to use to generate a JSON Text Data Block.
	json_data_type	The type of data found at the top level, either JSON_object,
					JSON_array or JSON_value. This is the same kind of value 
					type that would be returned if a call was made to the
					function parse_json_data_block()
	json_text_buffer 
					A pointer to a location to write the pointer to the data block
					created by the JSON Generator.

	returns		 0	All JSON elements processed correctly, no syntax / grammar errors.
				 1	JSON Object was empty, but processed correctly.
				-1	A JSON grammar or syntax error was found below the top level
				-2	An error was found in the JSON syntax at the top level
				-3	An argument error occurred
				-4	An error occurred during memory allocation, there is no
					output buffer available.											*/
int json_generate_data_block(	list_object *json_data, 
								JSON_value_types json_data_type,
								char **json_text_buffer);





/*	================================================================

	JSON D_list Functions for JSON Parser D-List hierarchical lists.

	================================================================*/


/*	Default D-List user functions for JSON Parser lists.
	====================================================*/
	
/* The following is the list of available search and compare functions that are
	built into JSON Parser. Callers may use any of these as they need, or use
	them as templates for writing functions that suit their specific data needs.

	default_search_function_member_name(const void *element, const void *key);	
	default_search_function_partial_member_name(const void *element, const void *key);

	default_search_function_array_values(const void *element, const void *key);

	default_search_function_array_string_value(const void *element, const void *key);
	default_search_function_partial_array_string_value(const void *element, const void *key);
	default_search_function_array_integer_value(const void *element, const void *key);
	default_search_function_array_double_value(const void *element, const void *key);
	default_search_function_precise_array_double_value(const void *element, const void *key);
	default_search_function_sloppy_array_double_value(const void *element, const void *key);

	default_compare_function_member_names(const void *element, const void *key);

	default_compare_function_array_values(const void *element, const void *key);

	default_compare_function_array_number_values(const void *element, const void *key);
	default_compare_function_array_string_values(const void *element, const void *key);
	default_compare_function_array_integer_values(const void *element, const void *key);
	default_compare_function_array_double_values(const void *element, const void *key);
	
	For more information about each of these functions, please see the prototype 
	information below.																	*/


/* This the default search function applied to all JSON Object lists, unless
	an alternative is provided by the caller of parse_json_data_block()
	
	This search function searches the elements of a specified JSON Object List
	looking for an exact match of object name to the key value string. It can
	only be used on a JSON Object list, not for an Array list.							*/
boolean default_search_function_member_name(const void *element, const void *key);	
/*	This search function searches the elements of a specified JSON Object List
	looking for a match of object name to the partial key value string. It can
	only be used on a JSON Object list, not for an Array list.							*/
boolean default_search_function_partial_member_name(const void *element, const void *key);

/* This the default search function applied to all JSON Array list_objects, unless
	an alternative is provided by the caller of parse_json_data_block(), or a
	different search function is provided for a specific list_object using the
	D-List function list_set_search_function().

	This default search function searches all the value elements of a specified 
	JSON Array List using a combination key of both a JSON_value_type and the 
	appropriate value in an input json_element_t structure.
	
	It will look for an exact match of element, of the same JSON_value_type and
	the same value instance. So a string will only be compared to a string value,
	a boolean only to another boolean, etc.
	
	Considerations for usage in JSON Arrays
	
		If searching for a JSON Object within an array, the search will return
		the first value found that is a JSON Object, with no other forms of 
		comparison performed. The input json_element_t structure must have a value
		type of JSON_object and a NULL member_name.
		
		If searching for a JSON Array as a value within an array, the search will
		return the first value found that is a JSON Array, with no other forms of 
		comparison performed. The input json_element_t structure must have a value
		type of JSON_array all other fields are ignored.
		
		If searching for a null value within an array, the search will return
		the first value found that is null. The input json_element_t structure must
		have a value type of JSON_null all other fields are ignored.
		
		If searching for a other value types within an array, the search will 
		return the first value found that matches the value type and value criteria
		provided in the input json_element_t structure, which must have a valid value
		type and a valid value field for that type. Strings, integers, float strings
		and booleans are compared for equality. 
		
		However floating point numbers are more difficult. It is the nature of 
		non-integer number representation in computers to be imprecise. Dealing
		with these numbers to the accuracy and precision demanded by some areas
		of expertise is beyond the scope of this package. This package does however
		provide some generally useful default tools, and allows users who need high
		precision to easily import/export these values for outside processing, and
		also provide their own search and compare functions to augment those 
		functions already provided. There are 3 floating point search functions
		provided in this package, each varying in its precision for matches. This
		general purpose default search function for JSON Array list_objects uses 
		the medium precision version of those included default search functions, 
		default_search_function_array_double_value()

		If a user of the JSON Parser has chosen to not convert exponent numbers,
		they will be checked for equality as textural items. This should always 
		yield accurate equality results, but it is possible that under some as yet 
		unknown circumstances, it may not.
		
	
	This general purpose, mixed value type search function can also be used on JSON
	Object lists. 

	Considerations for usage in JSON Objects

		If searching for a JSON Object within an Object, the search will be limited
		only to members that have the object name provided and a value of another 
		JSON Object. The input json_element_t structure must have a value type of 
		JSON_object and a character member_name to compare against. Setting the 
		search value_type to JSON_object will cause this search function to ignore
		all values in the search.
		
		Setting the value type to any other JSON_xxx value will cause the JSON Object
		members to be searched by value only, and the member name will be ignored.
		This search function cannot be used to find any random JSON Object member
		in a JSON Object. For that you will have to iterate through the list_object
		yourself to find what you are looking for.
			
	If you know that you will only be searching arrays for a specific type of 
	value (say text), then it is a lot more efficient to use a value type
	specific search function instead of this general purpose search function.
	
	This is the ONLY search function that takes a json_element_t structure as the
	search key. All the others use an independent data value instead.					*/
boolean default_search_function_array_values(const void *element, const void *key);
	
/*	This search function searches the elements of a specified JSON Array List
	looking for an exact match of element string value to the key value string. It 
	can be used on both JSON Object lists and Array lists. It will only attempt
	to find a match with elements that have the value_type of JSON_string. It will
	ignore any elements with a different value type.
	
	Unlike the general purpose search function default_search_function_array_values()
	these value type specific search functions use a simple non-structure key to
	perform the search, so using default_search_function_array_string_value() as your
	search function you could use list_simple_search(root_list, "search text");			*/
boolean default_search_function_array_string_value(const void *element, const void *key);

/*	This search function searches the elements of a specified JSON Array List
	looking for a match of element string value to the partial key value string.
	It can be used on both JSON Object lists and Array lists. It will only attempt
	to find a match with elements that have the value_type of JSON_string. It
	will ignore any elements with a different value type.
	
	Unlike the general purpose search function default_search_function_array_values()
	these value type specific search functions use a simple non-structure key to
	perform the search, so using default_search_function_partial_array_string_value() 
	as your search function you could use list_simple_search(root_list, "search text");	*/
boolean default_search_function_array_partial_string_value(const void *element, const void *key);

/*	This search function searches the elements of a specified JSON Array List
	looking for a match of element integer value to the key value integer.
	It can be used on both JSON Object lists and Array lists. It will only attempt
	to find a match with elements that have the value_type of JSON_integer. It
	will ignore any elements with a different value type.
	
	Unlike the general purpose search function default_search_function_array_values()
	these value type specific search functions use a simple non-structure key to
	perform the search, so using default_search_function_array_integer_value() as your
	search function you could use list_simple_search(root_list, 42);					*/
boolean default_search_function_array_integer_value(const void *element, const void *key);

/*	This search function searches the elements of a specified JSON Array List
	looking for a match of element number (non-integer) value to the key value number.
	It can be used on both JSON Object lists and Array lists. It will only attempt
	to find a match with elements that have the value_type of JSON_double. It
	will ignore any elements with a different value type.
	
	This is one of 3 available default search functions for floating point numbers.
	This is the default one used by default_search_function_array_values() and is the
	best choice for general purpose searches for non high precision work.
	
	Unlike the general purpose search function default_search_function_array_values()
	these value type specific search functions use a simple non-structure key to
	perform the search, so using default_search_function_array_double_value() as your
	search function you could use list_simple_search(root_list, 42.45);					*/
boolean default_search_function_array_double_value(const void *element, const void *key);

/*	This search function searches the elements of a specified JSON Array List
	looking for a match of element number (non-integer) value to the key value number.
	It can be used on both JSON Object lists and Array lists. It will only attempt
	to find a match with elements that have the value_type of JSON_double. It
	will ignore any elements with a different value type.
	
	This is one of 3 available default search functions for floating point numbers.
	This search function is the best choice for high precision searches when the
	caller has high confidence in the accuracy of the precision of the search elements
	and the key value.
	
	Unlike the general purpose search function default_search_function_array_values()
	these value type specific search functions use a simple non-structure key to
	perform the search, so using default_search_function_array_double_value() as your
	search function you could use list_simple_search(root_list, 42.45);					*/
boolean default_search_function_precise_array_double_value(const void *element, const void *key);

/*	This search function searches the elements of a specified JSON Array List
	looking for a match of element number (non-integer) value to the key value number.
	It can be used on both JSON Object lists and Array lists. It will only attempt
	to find a match with elements that have the value_type of JSON_double. It
	will ignore any elements with a different value type.
	
	This is one of 3 available default search functions for floating point numbers.
	This search function is the best choice for very low precision searches when the
	caller has poor confidence in the accuracy of the precision of the search elements
	and the key value. It enables search matches with a very high epsilon factor to
	be able find a broad match when precision has not been taken into consideration
	by the caller or the JSON data writer.
	
	Unlike the general purpose search function default_search_function_array_values()
	these value type specific search functions use a simple non-structure key to
	perform the search, so using default_search_function_array_double_value() as your
	search function you could use list_simple_search(root_list, 42.45);					*/
boolean default_search_function_sloppy_array_double_value(const void *element, const void *key);



/* This the default compare function applied to all JSON Object lists, unless
	an alternative is provided by the caller of parse_json_data_block()
	
	This compare function evaluates the names of the elements of a specified 
	JSON Object List.
	
	It should only be used on a JSON Object list, not for an Array list. If used 
	on an Array list in debug mode, it will raise an assert() exception. If run
	in production mode on an Array list, it will return an Elements Equal result.
	Either way the results are undefined.												*/
int default_compare_function_member_names(const void *element, const void *key);


/* This the default compare function applied to all JSON Array lists, unless
	an alternative is provided by the caller of parse_json_data_block()
	
	This general purpose compare function evaluates the elements of a specified
	JSON Array List. It can be used on both JSON Object lists and Array lists. 
	Only use it with JSON Object lists, if you wish to sort the list by value
	rather than Object Name.
	
	This compare function, unlike the others in the section below, can evaluate
	all the elements in an array, regardless of their value types.
	
	If two elements are of the same value type, they are compared with each other
	and the results returned. If the elements are of different value types, the
	evaluation is performed on the value type itself.
	
	This will yield a sorted list of the following superior orders, from least to
	most significant (assuming an ascending sort order).
	
		JSON Objects (grouped, but not sorted)
		JSON Arrays (grouped, but not sorted)
		JSON null values (grouped, but obviously not sorted)
		JSON boolean values (sorted)
		JSON number values (sorted, mixed types allowed- see note:a)
		JSON float string values (sorted - see note:b)
		JSON string values (sorted)
		
	This creates a well sorted list that makes logical sense in the context
	of the values presented. Literal values are sorted within the context
	of an alphanumeric sort, and given an appropriate value within it.
	
	Note a: JSON Numbers are represented in D-List list elements as either
			integer or long double variables. For to context of these evaluations
			the integers are converted to long doubles if the evaluation is 
			between elements of different value types. There is the possibility 
			of a loss of precision, depending on the C implementation and the 
			architecture of the system the code is running on. Any conversion
			issues would only affect the evaluations, not the actual data items,
			and would only be a tiny variance of any proper mathematical ordering.
	Note b: This JSON Parser allows users to prevent the conversion of exponent
			numbers from a JSON data block, so they may be accessed directly by the
			users instead, who may wish to use them with library implementations
			that allow for much higher precision than C99 C ever did. If these 
			exponent numbers were not converted they are left as exponent numbers
			in text form. They would be sorted as text fields, and placed before
			any actual text field, preserving some semblance of proper ordering. 
			The sorting performed will not take precision into account for these
			types of textural exponent numbers, so any such sorting for these 
			special value types, is approximate at best. Most users will never
			encounter these value types, or usage issues.								*/
int default_compare_function_array_values(const void *element, const void *key);



int default_compare_function_array_number_values(const void *element, const void *key);


/*	JSON Array List value specific compare functions. 
	
		You should ONLY use these compare functions below (that are named for a
		specific value type) if all the values represented in the list are of 
		the same value type, and also matches the type specified in the compare 
		function name.
		
		Compare functions are used by D-List to  perform sorts and other functions
		that evaluate the differences between elements. 
		
		Using any of these functions in the group below on mixed value type lists
		will yield unpredictable results. The code will not fail, each of these 
		compare functions has protection from such misuse, but the results will
		not be useful to the caller and are undefined.
		
		For mixed value lists, please use the default compare function, or the
		number function, above. Only use these if you are sure that is what you
		want, and to reduce overhead, and the generalized compare functions are 
		much more complicated to execute.												*/
		

/* This compare function evaluates the elements of a specified JSON Array List
	It can be used on both JSON Object lists and Array lists. It will only attempt
	to evaluate elements that have the value_type of JSON_string. It will return
	an "is equal" evaluation if the element is not a string.							*/
int default_compare_function_array_string_values(const void *element, const void *key);

/* This compare function evaluates the elements of a specified JSON Array List
	It can be used on both JSON Object lists and Array lists. It will only attempt
	to evaluate elements that have the value_type of JSON_integer. It will return
	an "is equal" evaluation if the element is not an integer.							*/
int default_compare_function_array_integer_values(const void *element, const void *key);

/* This compare function evaluates the elements of a specified JSON Array List
	It can be used on both JSON Object lists and Array lists. It will only attempt
	to evaluate elements that have the value_type of JSON_double. It will return
	an "is equal" evaluation if the element is not a double based number (i.e a
	fractional or exponent number, a non-integer).
		
	It is beyond the scope of this package to provide custom compare functions.
	If the caller needs to sort a list containing mixed number types, then they
	can either write a custom compare function, or copy the list elements to
	another list, converting the integers to long double, and then using this
	default compare function.															*/
int default_compare_function_array_double_values(const void *element, const void *key);









/*	=================================================================

	JSON Support Functions for JSON Parser D-List hierarchical lists.
	
	Please note:	All these functions are fully thread / MP safe.
	------------	However, it is still the callers responsibility
					to ensure that multiple threads do not attempt 
					to simultaneously modify and access the same data
					structures or list_object(s) used by these JSON
					functions. This should be done by using mutex 
					locks, semaphores, gates, barriers etc, or 
					whatever mechanisms you deem fit.

	=================================================================*/


/*	Special Note regarding the scope of these JSON Object Member search functions
	-----------------------------------------------------------------------------

	These JSON Object Member search functions will find members of JSON Objects
	and apply the search criteria to the direct member values only. To illustrate 
	exactly what this means, consider the following examples. 
	
	Searching for Member name "fred" and value 10. If this member is found it 
	would be returned as a match
	
		{ 	"bob" : 52,
			"fred" : 10,
			"amy" : [ 15, 25, 42 ], ... } 
	
	as the 10 is the direct value of the Member name we matched against. The
	following example would also return a match

		{ 	"bob" : 52,
			"fred" : [ 20, 52, 10, "a string value" ],
			"amy" : [ 15, 25, 42 ], ... } 

	as the value 10 was found as a direct value of "fred" even though the value is
	part of an array, that array is semantically a direct value of the member name.
	In the following example there would be no match made

		{ 	"bob" : 52,
			"fred" : [ 20, [ 52, 10], "a string value" ],
			"amy" : [ 15, 25, 42 ], ... }

	as in this case the value 10 is not a direct value of the Member name, but a
	value of an array which is itself a direct value. This general purpose search
	tool is limited to finding same level matches of values, and only the caller
	is going to understand the implied semantics of various obscure JSON data 
	layouts. It would be unreasonable to expect a general purpose search tool to
	be able to devine the semantics of such layouts or structures.
	
	Please also bear in mind that a match would be made in the following example

		{ 	"bob" : 52,
			"fred" : [ 20, [ { "fred": ["fred's cousin",52,10] } ], "a string" ],
			"amy" : [ 15, 25, 42 ], ... }

	but it would be a match on the Member of the JSON Object inside the array
	(fred's cousin) not the Member at the top level JSON Object.						*/



/* Using the provided search criteria, search the JSON data and locate all object
	members in the list hierarchy that match the criteria, returning them in a
	new list_object to the caller.
	
	This function can be called to search only JSON Object Member names, or it
	can search for Member names and a string value criteria as well. For example, 
	it can search the entire hierarchy for all JSON Object Members with the name 
	"handle" and return a list of all Object Members found. Alternatively it could
	search for all Object Members with the name "roles" and the value "technical",
	which would return all the Object Members named "roles" that also has a string
	value of "technical". The value of course could be a data value in an JSON 
	Array, depending on what the found Object Member had as a data value.
	
	This function may be called with an option to perform partial JSON Object Name
	and/or string value matches, so for the above example, a partial Object Name
	match for "role" would return all Object Members including, "role", roles" etc.

	Upon return if Object Members were found that matched the search criteria,
	a list containing elements of the type json_member_list_t would be provided
	one element in the list for each match found. When the caller has finished
	with the list, it must erased with list_erase() and then freed. 
	
	json_data		A D-List list_object that describes the JSON data to be searched.
					This can be a root list_object, or it can be any subordinate 
					list_object within that hierarchy. It can be an Array or an Object
					list. The search is confined to the list(s) covered by this top 
					level list_object.
	member_name		Pointer to a nul terminated string to use as a match against the
					JSON Object Member names in the JSON data list.
	partial_name	TRUE	perform the search using member_name as a partial name
							match,
					FALSE 	perform the search looking only for a complete match with
							member_name.
	string_value	Optional pointer to a nul terminated string to use as a match 
					against the JSON Object Member values in the JSON data list, that
					have already matched with member_name, and have the value type of
					JSON_string.
					NULL to perform a search for only JSON OObject Member names, and
					not involving any search of Object Member values. 
	partial_value	TRUE	perform the search using string_value as a partial string
							value match, only value if string_value is not NULL.
					FALSE 	perform the search looking only for a complete string 
							value match with string_value.
	
	returns		list_object	contains the JSON Object Members found that matched the
							search criteria. This list was created with LIST_COPY_DATA
							option, so element's storage will be managed as described
							by D-List depending on what functions you use to access or
							manage this list. list_erase() will release ay internal and
							element storage remaining when called.
				NULL		No JSON Object Members were found that matched the search 
							criteria. No list_object was created.						*/
list_object *json_find_members_value_string(	list_object *json_data, 
												const char *member_name, 
												const boolean partial_name,
												const char *string_value,
												const boolean partial_value);

/* Using the provided search criteria, search the JSON data and locate all object
	members in the list hierarchy that match the criteria, returning them in a
	new list_object to the caller.
	
	This function can only be called to search JSON Object Member names, and a 
	number value criteria as well. To search for JSON Object Member names alone 
	please refer to the function json_find_members_value_string() above.
	
	For example, to search for all Object Members in the list hierarchy for a 
	match with the partial name "PackageTags" and the number values between 0 
	and 100, could return such Members as 	"cachePackageTagsRefresh": 60,
	but not return		"cachePackageTagsStore": 200,
	
	This function may be called with an option to perform JSON Object Name 
	number value matches on a single number or an inclusive range of the
	minimum and maximum numbers provided.
	
	This function will only compare all JSON_integer value types. This function
	cannot evaluate numbers of different classes (non-integer and integer), if
	you require this or other more complicated searches of JSON data list 
	hierarchies, you will have to write your own code to iterate through those
	list_objects and perform whatever evaluations you need.
	
	JSON data elements with the value type JSON_float_str, may be evaluated by 
	string comparison using json_find_members_value_string() above.

	Upon return if Object Members were found that matched the search criteria,
	a list containing elements of the type json_member_list_t would be provided
	one element in the list for each match found. When the caller has finished
	with the list, it must erased with list_erase() and then freed. 
	
	json_data		A D-List list_object that describes the JSON data to be searched.
					This can be a root list_object, or it can be any subordinate 
					list_object within that hierarchy. It can be an Array or an Object
					list. The search is confined to the list(s) covered by this top 
					level list_object.
	member_name		Pointer to a nul terminated string to use as a match against the
					JSON Object Member names in the JSON data list.
	partial_name	TRUE	perform the search using member_name as a partial name
							match,
					FALSE 	perform the search looking only for a complete match with
							member_name.
	minimum_value	A number to be used either as a direct match against the JSON 
					Object Member number values in the JSON data list, or as a minimum
					for a range match. Only JSON Object Members that have already 
					matched with member_name, and have the value type of JSON_integer 
					will be evaluated against either the number or the range.
	maximum_value	Optional number to to be used as a maximum number for a range match.
					Only JSON Object Members that have already matched with member_name,
					and have the value type of JSON_integer will be evaluated against 
					the range provided.
	value_range		TRUE	perform the search using minimum_value and maximum_value as
							an inclusive range of numbers to value match.
					FALSE 	perform the search looking only for a match with the number
							in minimum_value.
	
	returns		list_object	contains the JSON Object Members found that matched the
							search criteria. This list was created with LIST_COPY_DATA
							option, so element's storage will be managed as described
							by D-List depending on what functions you use to access or
							manage this list. list_erase() will release ay internal and
							element storage remaining when called.
				NULL		No JSON Object Members were found that matched the search 
							criteria. No list_object was created.						*/
list_object *json_find_members_value_integer(	list_object *json_data, 
												char *member_name, 
												const boolean partial_name,
												const json_integer_t minimum_value,
												const json_integer_t maximum_value,
												const boolean value_range);

/* Using the provided search criteria, search the JSON data and locate all object
	members in the list hierarchy that match the criteria, returning them in a
	new list_object to the caller.
	
	This function can only be called to search JSON Object Member names, and a 
	number value criteria as well. To search for JSON Object Member names alone 
	please refer to the function json_find_members_value_string() above.
	
	For example, to search for all Object Members in the list hierarchy for a 
	match with the partial name "PackageTags" and the number values between 0 
	and 100, could return such Members as 	"cachePackageTagsRefresh": 52.5,
	but not return		"cachePackageTagsStore": 200,
	
	This function may be called with an option to perform JSON Object Name 
	number value matches on a single number or an inclusive range of the
	minimum and maximum numbers provided.
	
	This function will only compare all JSON_double value types. This function
	cannot evaluate numbers of different classes (non-integer and integer), if
	you require this or other more complicated searches of JSON data list 
	hierarchies, you will have to write your own code to iterate through those
	list_objects and perform whatever evaluations you need.
	
	JSON data elements with the value type JSON_float_str, may be evaluated by 
	string comparison using json_find_members_value_string() above.

	Upon return if Object Members were found that matched the search criteria,
	a list containing elements of the type json_member_list_t would be provided
	one element in the list for each match found. When the caller has finished
	with the list, it must erased with list_erase() and then freed. 
	
	json_data		A D-List list_object that describes the JSON data to be searched.
					This can be a root list_object, or it can be any subordinate 
					list_object within that hierarchy. It can be an Array or an Object
					list. The search is confined to the list(s) covered by this top 
					level list_object.
	member_name		Pointer to a nul terminated string to use as a match against the
					JSON Object Member names in the JSON data list.
	partial_name	TRUE	perform the search using member_name as a partial name
							match,
					FALSE 	perform the search looking only for a complete match with
							member_name.
	minimum_value	A number to be used either as a direct match against the JSON 
					Object Member number values in the JSON data list, or as a minimum
					for a range match. Only JSON Object Members that have already 
					matched with member_name, and have the value type of JSON_double 
					will be evaluated against either the number or the range.
	maximum_value	Optional number to to be used as a maximum number for a range match.
					Only JSON Object Members that have already matched with member_name,
					and have the value type of JSON_double will be  evaluated against
					the range provided.
	value_range		TRUE	perform the search using minimum_value and maximum_value as
							an inclusive range of numbers to value match.
					FALSE 	perform the search looking only for a match with the number
							in minimum_value.
	
	returns		list_object	contains the JSON Object Members found that matched the
							search criteria. This list was created with LIST_COPY_DATA
							option, so element's storage will be managed as described
							by D-List depending on what functions you use to access or
							manage this list. list_erase() will release ay internal and
							element storage remaining when called.
				NULL		No JSON Object Members were found that matched the search 
							criteria. No list_object was created.						*/
list_object *json_find_members_value_double(	list_object *json_data, 
												char *member_name, 
												const boolean partial_name,
												const json_double_t minimum_value,
												const json_double_t maximum_value,
												const boolean value_range);

/* Using the provided search criteria, search the JSON data and locate all object
	members in the list hierarchy that match the criteria, returning them in a
	new list_object to the caller.
	
	This function can only be called to search JSON Object Member names, and a 
	value type criteria as well. To search for JSON Object Member names alone 
	please refer to the function json_find_members_value_string() above.
	
	For example, to search for all Object Members in the list hierarchy for a 
	match with the name "init-param" and the value type JSON_object could return
	such Members as 	"init-param": {

	Upon return if Object Members were found that matched the search criteria,
	a list containing elements of the type json_member_list_t would be provided
	one element in the list for each match found. When the caller has finished
	with the list, it must erased with list_erase() and then freed. 
	
	json_data		A D-List list_object that describes the JSON data to be searched.
					This can be a root list_object, or it can be any subordinate 
					list_object within that hierarchy. It can be an Array or an Object
					list. The search is confined to the list(s) covered by this top 
					level list_object.
	member_name		Pointer to a nul terminated string to use as a match against the
					JSON Object Member names in the JSON data list.
	partial_name	TRUE	perform the search using member_name as a partial name
							match,
					FALSE 	perform the search looking only for a complete match with
							member_name.
	value_type		A JSON value type to be used as a direct match against the JSON 
					Object Member number value types in the JSON data list. Only JSON
					Object Members that have already matched with member_name, will be
					evaluated against supplied JSON value type. This can be ay valid
					JSON value type as defined in this header file above.
	
	returns		list_object	contains the JSON Object Members found that matched the
							search criteria. This list was created with LIST_COPY_DATA
							option, so element's storage will be managed as described
							by D-List depending on what functions you use to access or
							manage this list. list_erase() will release ay internal and
							element storage remaining when called.
				NULL		No JSON Object Members were found that matched the search 
							criteria. No list_object was created.						*/
list_object *json_find_members_value_type(	list_object *json_data, 
											const char *member_name, 
											const boolean partial_name,
											const JSON_value_types value_type);

/* Using the provided search criteria, search the JSON data and locate the first
	JSON object member in the list hierarchy that match the criteria, returning
	a pointer to it to the caller.
	
	This function can be called to search only JSON Object Member names, or it
	can search for Member names and a string value criteria as well. For example, 
	it can search the entire hierarchy for a JSON Object Members with the name 
	"handle" or it could search for an Object Members with the name "roles" and
	the value "technical", which would return the first Object Member named 
	"roles" that also has a string value of "technical". The value of course 
	could be a data value in an JSON Array, depending on what the found Object
	Member had as a data value.
	
	If you wish to get a list of all the JSON Object Members that matched the
	search criteria, you must use the function json_find_members_value_string()
	instead, which returns a list of Object Members instead.
	
	This function may be called with an option to perform partial JSON Object Name
	and/or string value matches, so for the above example, a partial Object Name
	match for "role" would return the first Object Member which could be "role",
	roles" etc.
	
	This function DOES NOT return the internal representation of the JSON Path
	with the match. This is because the json_member_list_t is supplied by the
	caller, and they could not have allocated it to cover unknown length of path.

	Upon return if an Object Members was found that matched the search criteria,
	a pointer to the element and a pointer to the owning list_object will be
	returned in a  json_member_list_t structure the caller already provided. 
	
	result			A json_member_list_t struct which will be used to write the 
					information about any JSON Object Member found that matched the
					search criteria. Only one Member will be returned, even if there
					are more that matched the criteria.
	json_data		A D-List list_object that describes the JSON data to be searched.
					This can be a root list_object, or it can be any subordinate 
					list_object within that hierarchy. It can be an Array or an Object
					list. The search is confined to the list(s) covered by this top 
					level list_object.
	member_name		Pointer to a nul terminated string to use as a match against the
					JSON Object Member names in the JSON data list.
	partial_name	TRUE	perform the search using member_name as a partial name
							match,
					FALSE 	perform the search looking only for a complete match with
							member_name.
	string_value	Optional pointer to a nul terminated string to use as a match 
					against the JSON Object Member values in the JSON data list, that
					have already matched with member_name, and have the value type of
					JSON_string.
					NULL to perform a search for only JSON OObject Member names, and
					not involving any search of Object Member values. 
	partial_value	TRUE	perform the search using string_value as a partial string
							value match, only value if string_value is not NULL.
					FALSE 	perform the search looking only for a complete string 
							value match with string_value.
	
	returns		0			No JSON Object Members were found that matched the
							search criteria.
				>0			The number of JSON Object Members found that matched the
							search criteria. Only the first match found was returned
							in the struct results, no JSON path is included. 	
				<0			An error occurred											*/
int json_find_member_value(	json_member_list_t *result,
							list_object *json_data, 
							const char *member_name, 
							const boolean partial_name,
							const char *string_value,
							const boolean partial_value);

/* Using the internal representation of a JSON dotted notation path, trace the
	provided path and return the list_object or element found.
	
	You can specify that the JSON path be followed precisely to find the JSON
	element, or you can ask for the path to be following in a more fuzzy method.
	For example if you are not sure if the top level JSON Object at the
	top of your path is actually at the top, or inside a top level array, then
	choosing a fuzzy approach to following the path will allow this function to
	try a few options to find the top segment of the specified path. After the
	first segment of the specified path, it is always followed strictly.
	
	Following paths with a fuzzy approach, is the default.
	
	If the JSON path you wish to locate is an absolute path, then you must also
	provide the root list_object for that JSON data block, or the search will 
	fail, as absolute paths must originate from the top of the JSON tree. This
	function makes no distinction between absolute and relative paths, but the
	starting point will always dictate how successful a search is going to be.
	
	json_data		A D-List list_object that describes the JSON data to be searched.
					This can be a root list_object, or it can be any subordinate 
					list_object within that hierarchy. It can be an Array or an Object
					list. The search is confined to the list(s) covered by this top 
					level list_object.
	path_to_find	a json_member_list_t struct that contains the path to follow,
					it is required that both .path_segments and .path[] fields be
					correctly filled in prior to calling this function. All the 
					other fields of the json_member_list_t struct are ignored upon
					entry, and upon return will be filled in with the information
					about the new element, and the parent list_object that new
					element now belongs to.
	strict_path		TRUE follow the provided path strictly, do not perform 
					additional searches if the top segment of the path is not
					found at the first level of the starting point.
	suppress_errors	TRUE do not output errors to stderr. FALSE output path and
					structure errors to stderr. Argument or coding errors will
					always be output to stderr.
	
	returns		Pointer to the updated json_member_list_t that was provided as an
				input parameter. If the element was not found, the path did not
				resolve to a findable element, then NULL is returned.					*/
json_member_list_t *json_path_locate(	list_object *json_data,
										json_member_list_t *path_to_find, 
										const boolean strict_path,
										const boolean suppress_errors);


/* Create a new list_object to be used to represent either a JSON Object or 
	a JSON Array.
	
	Upon return the caller must check the result, if it is NULL no 
	list_object was created.
	
	list_type		Either JSON_object or JSON_array. Depending on what type of
					list_object you need to create. When created the correct
					default JSON D-List search and compare functions will be
					attached to the new list_object. 
	functions		NULL normal usage, create list_object with default search and
					compare functions.
					Pointer to array of functions, override the defaults with 
					these search and compare functions.
	
	returns		pointer to the list_object of the new list created.	
				or NULL if an internal error occurred.									*/
list_object *json_create_new_list_object(	const JSON_value_types list_type, 
											const functions_globals_t *functions);

/* Given an unknown list_object, provide the JSON value type this list_object
	represents. The result can either be JSON_object, JSON_array or JSON_value.
	
	See the JSON Parser documentation for the meanings of these types and how
	to apply them to your work managing JSON data hierarchy structures.
	
	list		The list_object you need to find out the JSON type of
	
	returns		A valid JSON type, or invalid_empty if the list_object was not
				of a JSON_type															*/	
JSON_value_types json_report_list_object_type(list_object *list);

/* Add a new Member with a string value to a JSON Object, defined by
	the list_object provided.

	This function will add a new JSON Member to the provided list_object
	at the last position of that list. 
	
	If there are already elements in this list_object, then they are used to
	confirm that the list_object requested, is actually a JSON Object. If this
	is the first element to be added, then as no check can be made, it is 
	assumed the caller is providing a Member to a valid JSON list type. If the
	list type is incorrect, an error will be thrown up when the second element
	is added.
	
	list_to_add		the list_object of the JSON data hierarchy to add the new
					Member to.
	member_name		pointer to a nul terminated string to use as the JSON Object 
					Member Name, this argument is required both by this function, 
					and by JSON Standards RFC 8259.
	string_value	pointer the a nul terminated string to use a the Member Value
	
	returns		LIST_OK				success, new element added to the list
				LIST_ERR			on error	 										*/
int json_list_object_member_add_string(	list_object *list_to_add, 
										const char *member_name,
										const char *string_value);


/* Add a new Member with a number value to a JSON Object, defined by
	the list_object provided.

	This function will add a new JSON Member to the provided list_object
	at the last position of that list. 
	
	Only one value item can be provided, either an integer or a
	fractional number. The arguments take the following precedence,
		
		If double_number == 0, integer_number is used
		else double_number is used.
	
	which means if the new Member has a value of zero, it will always be added
	as an integer value.
	
	If there are already elements in this list_object, then they are used to
	confirm that the list_object requested, is actually a JSON Object. If this
	is the first element to be added, then as no check can be made, it is 
	assumed the caller is providing a Member to a valid JSON list type. If the
	list type is incorrect, an error will be thrown up when the second element
	is added.
	
	list_to_add		the list_object of the JSON data hierarchy to add the new
					Member to.
	member_name		pointer to a nul terminated string to use as the JSON Object 
					Member Name, this argument is required both by this function, 
					and by JSON Standards RFC 8259.
	integer_number	used as the Member value only if a) string_value is NULL, and
					b) double_number == 0, otherwise it is ignored
	double_number	used as the Member value only if a) string_value is NULL, and
					b) this argument is non-zero, otherwise it is ignored
	
	returns		LIST_OK				success, new element added to the list
				LIST_ERR			on error	 										*/
int json_list_object_member_add_number(	list_object *list_to_add, 
										const char *member_name,
										const json_integer_t integer_number,
										const json_double_t double_number);


/* Add a new Member with a exponential number represented as a string value 
	(instead of a binary floating number) to a JSON Object, defined by
	the list_object provided.

	This function will add a new JSON Member to the provided list_object
	at the last position of that list. 
	
	As the basic C language has limited ability to properly represent very high
	precision exponential numbers, and this JSON package was designed to be 
	highly portable, only requiring a minimum of C99 to compile and run, it uses
	the base forms of the language for numbers, and does not rely on any external
	or optional libraries to handle these numbers. For some users either the 
	limited precision of these forms, or their inability to hold huge numbers 
	will be an issue. So this JSON package provides an option for both the JSON
	Parser and the JSON Generator to handle these numbers in a textural form 
	instead. This functions allows you to add such a number form to a JSON data 
	structure, for later output into a JSON data block using the JSON Generator.
	
	If there are already elements in this list_object, then they are used to
	confirm that the list_object requested, is actually a JSON Object. If this
	is the first element to be added, then as no check can be made, it is 
	assumed the caller is providing a Member to a valid JSON list type. If the
	list type is incorrect, an error will be thrown up when the second element
	is added.
	
	list_to_add		the list_object of the JSON data hierarchy to add the new
					Member to.
	member_name		pointer to a nul terminated string to use as the JSON Object 
					Member Name, this argument is required both by this function, 
					and by JSON Standards RFC 8259.
	float_str		pointer the a nul terminated string to use as the Member Value
	
	returns		LIST_OK				success, new element added to the list
				LIST_ERR			on error	 										*/
int json_list_object_member_add_float_str(	list_object *list_to_add, 
											const char *member_name,
											const char *float_str);

/* Add a new Member with a boolean value to a JSON Object, defined by 
	the list_object provided.

	This function will add a new JSON Member to the provided list_object
	at the last position of that list. 
	
	This function only supports adding Members with the value type boolean. 
	
	If there are already elements in this list_object, then they are used to
	confirm that the list_object requested, is actually a JSON Object. If this
	is the first element to be added, then as no check can be made, it is 
	assumed the caller is providing a Member to a valid JSON list type. If the
	list type is incorrect, an error will be thrown up when the second element
	is added.
	
	list_to_add		the list_object of the JSON data hierarchy to add the new
					Member to.
	member_name		pointer to a nul terminated string to use as the JSON Object 
					Member Name, this argument is required both by this function, 
					and by JSON Standards RFC 8259.
	boolean_value	TRUE or FALSE, to use as the Member Value
	
	returns		LIST_OK				success, new element added to the list
				LIST_ERR			on error	 										*/
int json_list_object_member_add_boolean(	list_object *list_to_add, 
											const char *member_name,
											const boolean boolean_value);

/* Add a new Member with a null value to a JSON Object, defined by 
	the list_object provided.

	This function will add a new JSON Member to the provided list_object
	at the last position of that list. 
	
	This function only supports adding Members with the value type null.
	
	If there are already elements in this list_object, then they are used to
	confirm that the list_object requested, is actually a JSON Object. If this
	is the first element to be added, then as no check can be made, it is 
	assumed the caller is providing a Member to a valid JSON list type. If the
	list type is incorrect, an error will be thrown up when the second element
	is added.
	
	list_to_add		the list_object of the JSON data hierarchy to add the new
					Member to.
	member_name		pointer to a nul terminated string to use as the JSON Object 
					Member Name, this argument is required both by this function, 
					and by JSON Standards RFC 8259.
	
	returns		LIST_OK				success, new element added to the list
				LIST_ERR			on error	 										*/
int json_list_object_member_add_null(	list_object *list_to_add, 
										const char *member_name);

/* Add a new Member with a JSON Array or JSON Object value to an existing
	JSON Object, defined by the list_object provided.

	This function will add a new JSON Member to the provided list_object
	at the last position of that list, this Member will not have a data
	value, but will instead open either a new Array of values, or a new
	JSON Object, with the Member Name attached. 
	
	So either { "member_name" : { } }  or  { "member_name" : [ ] } 
	
	The caller must have previously created the new list_object to add as a
	Member to this JSON Object, and provide it in argument #3 (new_list).
	
	If there are already elements in this list_object, then they are used to
	confirm that the list_object requested, is actually a JSON Object. If this
	is the first element to be added, then as no check can be made, it is 
	assumed the caller is providing a Member to a valid JSON list type. If the
	list type is incorrect, an error will be thrown up when the second element
	is added.
	
	list_to_add		the list_object of the JSON data hierarchy to add the new
					Member to.
	member_name		pointer to a nul terminated string to use as the JSON Object 
					Member Name, this argument is required both by this function, 
					and by JSON Standards RFC 8259.
	new_list		the list_object of the new JSON Array or Object to add to
					the existing JSON Object as a new Member.
	list_type		must be either JSON_object or JSON_array.
	
	returns		LIST_OK				success, new element added to the list
				LIST_ERR			on error	 										*/
int json_list_object_member_add_list(	list_object *list_to_add, 
										const char *member_name,
										list_object *new_list,
										const JSON_value_types list_type);


/* Add a new String Value to a JSON Array, defined by the list_object provided.

	This function will add a new JSON Value to the provided list_object
	at the last position of that list. 
	
	If there are already elements in this list_object, then they are used to
	confirm that the list_object requested, is actually a JSON Array. If this
	is the first element to be added, then as no check can be made, it is 
	assumed the caller is providing a Member to a valid JSON list type. If the
	list type is incorrect, an error will be thrown up when the second element
	is added.
	
	list_to_add		the list_object of the JSON data hierarchy to add the new
					String Value to.
	string_value	pointer the a nul terminated string to use a the Array Value
	
	returns		LIST_OK				success, new element added to the list
				LIST_ERR			on error	 										*/
int json_list_object_array_value_add_string(	list_object *list_to_add, 
												const char *string_value);

/* Add a new Number Value to a JSON Array, defined by the list_object provided.

	This function will add a new Array Value to the provided list_object
	at the last position of that list. 
	
	One or more values can be provided, either an integer or a
	fractional number. The arguments take the following precedence,
		
		If double_number[0] == 0, integer_number is used
		else double_number is used.
	
	which means if the new Number Value has is zero, it will always be added
	as an integer value. Only the first entry of the input array is checked.
	
	If there are already elements in this list_object, then they are used to
	confirm that the list_object requested, is actually a JSON Array. If this
	is the first element to be added, then as no check can be made, it is 
	assumed the caller is providing a Value to a valid JSON list type. If the
	list type is incorrect, an error will be thrown up when the second element
	is added.
	
	list_to_add		the list_object of the JSON data hierarchy to add the new
					Number Value to.
	integer_number	used as the Value only if  double_number[0] == 0, 
					otherwise it is ignored
	double_number	used as the Value only if adouble_number[0] != 0,
					otherwise it is ignored
	item_count		The number of these items to add to the JSON Array. It is
					up to the caller to ensure the count and actual number of
					values in the input array are consistent.
	
	returns		LIST_OK				success, new element(s) added to the list
				LIST_ERR			on error	 										*/
int json_list_object_array_value_add_number(	list_object *list_to_add, 
												const json_integer_t *integer_number,
												const json_double_t *double_number,
												const unsigned int item_count);


/* Add a new value with a exponential number represented as a string, 
	(instead of a binary floating number) to a JSON Object, defined by
	the list_object provided.

	This function will add a new JSON Value to the provided list_object
	at the last position of that list. 
	
	As the basic C language has limited ability to properly represent very high
	precision exponential numbers, and this JSON package was designed to be 
	highly portable, only requiring a minimum of C99 to compile and run, it uses
	the base forms of the language for numbers, and does not rely on any external
	or optional libraries to handle these numbers. For some users either the 
	limited precision of these forms, or their inability to hold huge numbers 
	will be an issue. So this JSON package provides an option for both the JSON
	Parser and the JSON Generator to handle these numbers in a textural form 
	instead. This functions allows you to add such a number form to a JSON data 
	structure, for later output into a JSON data block using the JSON Generator.
	
	If there are already elements in this list_object, then they are used to
	confirm that the list_object requested, is actually a JSON Array. If this
	is the first element to be added, then as no check can be made, it is 
	assumed the caller is providing a Member to a valid JSON list type. If the
	list type is incorrect, an error will be thrown up when the second element
	is added.
	
	list_to_add		the list_object of the JSON data hierarchy to add the new
					Member to.
	float_str		pointer the a nul terminated string to use as the Array 
					Value
	
	returns		LIST_OK				success, new element added to the list
				LIST_ERR			on error	 										*/
int json_list_object_array_value_add_float_str(	list_object *list_to_add, 
												const char *float_str);

/* Add a new Boolean Value to a JSON Array, defined by the list_object provided.

	This function will add a new Value element to the provided list_object
	at the last position of that list. 
	
	This function only supports adding value of type boolean. 
	
	If there are already elements in this list_object, then they are used to
	confirm that the list_object requested, is actually a JSON Array. If this
	is the first element to be added, then as no check can be made, it is 
	assumed the caller is providing a Value to a valid JSON list type. If the
	list type is incorrect, an error will be thrown up when the second element
	is added.
	
	list_to_add		the list_object of the JSON data hierarchy to add the new
					value to.
	boolean_value	TRUE or FALSE, to use as the new Value
	
	returns		LIST_OK				success, new element added to the list
				LIST_ERR			on error	 										*/
int json_list_object_array_value_add_boolean(	list_object *list_to_add, 
												const boolean boolean_value);

/* Add a new null value to a JSON Array, defined by the list_object provided.

	This function will add a new value element to the provided list_object
	at the last position of that list. 
	
	This function only supports adding value type null.
	
	If there are already elements in this list_object, then they are used to
	confirm that the list_object requested, is actually a JSON Array. If this
	is the first element to be added, then as no check can be made, it is 
	assumed the caller is providing a value to a valid JSON list type. If the
	list type is incorrect, an error will be thrown up when the second element
	is added.
	
	list_to_add		the list_object of the JSON data hierarchy to add the new
					value to.
	
	returns		LIST_OK				success, new element added to the list
				LIST_ERR			on error	 										*/
int json_list_object_array_value_add_null(	list_object *list_to_add);

/* Add a new data value to an existing JSON Array, that has as a value type of 
	either a JSON Array or JSON Object, defined by the list_object provided.

	This function will add a new JSON value to the provided Array list_object
	at the last position of that list, this value will not have a data
	value, but will instead open either a new Array of values, or a new
	JSON Object. 
	
	So either [ ..., {  } ]  or  [ ..,  [ ] ] 
	
	The caller must have previously created the new list_object to add as the
	value element to this JSON Array, and provide it in argument #2 (new_list).
	
	If there are already elements in this list_object, then they are used to
	confirm that the list_object requested, is actually a JSON Array. If this
	is the first element to be added, then as no check can be made, it is 
	assumed the caller is providing a Member to a valid JSON list type. If the
	list type is incorrect, an error will be thrown up when the second element
	is added.
	
	list_to_add		the list_object of the JSON data hierarchy to add the new
					Member to.
	new_list		the list_object of the new JSON Array or Object to add to
					the existing JSON Object as a new Member.
	list_type		must be either JSON_object or JSON_array.
	
	returns		LIST_OK				success, new element added to the list
				LIST_ERR			on error	 										*/
int json_list_object_array_value_add_list(	list_object *list_to_add, 
											list_object *new_list,
											const JSON_value_types list_type);






/*	======================================================================

	JSON Support Functions for Reporting hierarchical lists and JSON data.

		create character strings from data for display, convert paths, etc

	=======================================================================*/



/* Convert a JSON dotted notation path to a list_object for that path.
	
	This function performs the work of converting a textural JSON dotted 
	notation path, into first a json_member_list_t set of path segments
	and then returning the specific list_object needed to access the original
	item the JSON dotted notation path was meant to describe.
	
	If the path is invalid, or describes a path that does not exist, various
	error messages will be output to sdterr, and a NULL pointer is returned 
	to the caller.
	
	The path is always followed strictly, if a segment of the path is not 
	found an error is produced and NULL returned. If you wish to have the path 
	converted by using the fuzzy search option available in the lower level 
	functions, you will have to write the calls to those functions yourself. It
	was felt that making the fuzzy search option available in a general purpose
	conversion tools was too error prone for users.
	
	If the JSON path you wish to locate is an absolute path, then you must also
	provide the root list_object for that JSON data block, or the search will 
	fail, as absolute paths must originate from the top of the JSON tree. This
	function makes no distinction between absolute and relative paths, but the
	starting point will always dictate how successful a search is going to be.

	The returned list object pointer can to be used as an argument in any of the 
	json_list_object_xxx() or json_find_xxx() functions, as well as any D-List 
	functions that you wish to use to access, manage, or update the data within
	that JSON Object or Array at the end of the original path.
	
	This function performs the same sequence a user would have to call, it 
	just removed the work from their code.
	
	json_data		A D-List list_object that describes the JSON data to be searched.
					This can be a root list_object, or it can be any subordinate 
					list_object within that hierarchy. It can be an Array or an Object
					list. The search is confined to the list(s) covered by this top 
					level list_object.
	path			pointer to a nul terminated character array that contains the
					JSON dotted notation path to be converted.
	returns		pointer to the list object that represents that JSON path object
				or NULL of there was an error, or the path was invalid.					*/
list_object *json_path_to_list_object(	list_object *json_data,
										const char *path);


/* Create a set of links describing a JSON data path, by parsing a character 
	array with JSON dotted notion in it. 
	
	This function will output an array of json_path_t elements in an otherwise
	empty json_member_list_t element. The json_member_list_t can be used as 
	input to other json_xxx functions that require and use JSON paths.
	
	Although the JSON text path may have an absolute start (a $.xxx in most
	implementations) this function will strip the extraneous header off,
	and provide the Member Name and Array path links as they are found.
	
	e.g.	$.bakery.product[19]."curry pan"
	
	will convert to	4 link segments		{JSON_object, "bakery"}, 
										{JSON_object, "product"}, 
										{JSON_array, 19}, 
										{JSON_object, "curry pan"},

	
	dotted_path		a nul terminated string that contains the textural JSON Dotted
					Path to parse and translate into internal segment format.
	
	returns		Pointer		a new json_member_list_t record that provides the list
							of links segments that represents the input JSON Path.
							It is the callers responsibility to manage this storage
							and free it when finished with it.
 				NULL		Invalid path was provided.									*/
json_member_list_t *json_parse_dotted_path(const char *dotted_path);

/* Create a character array describing a set of links in a JSON Data block
	for reporting or printing purposes.
	
	path			an array of json_path_t elements that describe the 
					JSON path to generate.
	count			the number of elements in the path.
	
	returns		pointer to character array containing the text generated.
				Caller is responsible for proper management and disposal
				of storage related to this character array.								*/
char *json_generate_path_from_links(	json_path_t path[],
										unsigned int count);

/* Display to stdout text describing a set of links in a JSON data block.

	path			an array of json_path_t elements that describe the 
					JSON path to generate.
	count			the number of elements in the path.
	
	returns		Nothing.																*/
void json_print_path_from_links(	json_path_t path[],
									unsigned int count);


/* This function will either display all the elements in a single list_object, 
	or will walk an entire hierarchy of list_objects displaying the elements
	as it goes through them.
	
	This function is really only called for internal and user code debugging 
	purposes. It also serves as example code.											*/
void display_parsed_json_object(	list_object* json_object, 
									const char *list_name, 
									const int list_type, 
									const boolean dig_down);




#ifdef __cplusplus
}
#endif		/* __cplusplus */

#endif		/* JSON_PARSER_H */
